const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const cors = require('cors');

let fetch = globalThis.fetch;
if (!fetch) {
  fetch = (...args) => import('node-fetch').then(({ default: nodeFetch }) => nodeFetch(...args));
}

const app = express();
const PORT = process.env.JOBCARD_PORT || 5005;

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [Job Card Reports] Using shared MongoDB connection');

// Job Card Schema
const jobCardSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  adminUid: { type: String, required: true },
  // Human-readable request number e.g. ZAM-KAN-JC-0001 (may include -OFFLINE on device)
  requestNumber: { type: String },
  title: { type: String, required: true },
  status: { 
    type: String, 
    enum: ['Hold', 'assigned', 'in_progress', 'complete', 'cancelled'], 
    default: 'Hold' 
  },
  priority: {
    type: String,
    enum: ['Low', 'Medium', 'High', 'Emergency'],
    default: 'Medium'
  },
  workOrderType: String,
  
  // Equipment information
  equipment: {
    equipmentName: String,
    mainCategory: String,
    model: String,
    vehicleNumber: String,
    meterReading: Number,
    status: String
  },
  
  // Assignment information
  assignTo: [{
    name: String,
    email: String,
    uid: String
  }],
  
  // Dates
  startDate: Date,
  dueDate: Date,
  completedDate: Date,
  
  // Description and notes
  description: String,
  notes: String,
  
  // Location information
  location: {
    country: String,
    project: String,
    coordinates: String,
    address: String
  },
  
  // Service information
  serviceScheduleId: String,
  serviceType: String,
  
  // File attachments
  attachments: [{
    filename: String,
    originalName: String,
    url: String,
    uploadedAt: Date
  }],
  
  // PDF and signature information
  pdfDownloadUrl: String,
  signatureDownloadUrl: String,
  
  // Tracking information
  tracking: {
    currentStep: String,
    progress: Number,
    lastUpdated: Date,
    history: [{
      step: String,
      timestamp: Date,
      updatedBy: String,
      notes: String
    }]
  },
  
  // Tasks/Items array for storing job card tasks - using Mixed type for flexibility
  items: { type: [mongoose.Schema.Types.Mixed], default: [] },
  
  // Checklist items (if any)
  checklistItems: { type: [mongoose.Schema.Types.Mixed], default: [] },
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  createdBy: String,
  lastModifiedBy: String
});

// Create indexes for better performance
jobCardSchema.index({ adminUid: 1, createdAt: -1 });
jobCardSchema.index({ status: 1 });
jobCardSchema.index({ priority: 1 });
jobCardSchema.index({ 'assignTo.uid': 1 });
jobCardSchema.index({ 'location.country': 1, 'location.project': 1 });
// Index requestNumber to allow fast sorting by last number
jobCardSchema.index({ requestNumber: 1 });

const JobCard = mongoose.model('JobCard', jobCardSchema, 'jobcards_v3');

// Drop and recreate collection route (for debugging)
app.post('/jobcards/drop-collection', async (req, res) => {
  try {
    await JobCard.collection.drop();
    console.log('✅ Job cards collection dropped successfully');
    res.json({ success: true, message: 'Collection dropped successfully' });
  } catch (error) {
    if (error.code === 26) {
      console.log('ℹ️ Collection does not exist, nothing to drop');
      res.json({ success: true, message: 'Collection does not exist' });
    } else {
      console.error('❌ Error dropping collection:', error);
      res.status(500).json({ success: false, error: 'Failed to drop collection: ' + error.message });
    }
  }
});

// Multer configuration for file uploads
const jobCardStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
      console.log(`📁 Created temp directory: ${tempDir}`);
    }
    
    console.log(`📁 Storing file temporarily in: ${tempDir}`);
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const fileExtension = path.extname(file.originalname);
    cb(null, file.fieldname + '-' + uniqueSuffix + fileExtension);
  }
});

const fileUpload = multer({ 
  storage: jobCardStorage,
  limits: { fileSize: 50 * 1024 * 1024 } // 50MB limit
});

// Routes

// Health check
app.get('/jobcards', (req, res) => {
  res.json({ 
    status: 'OK', 
    service: 'Job Card Handler',
    port: PORT,
    timestamp: new Date().toISOString()
  });
});

// Test database connection
app.get('/jobcards/test-db', async (req, res) => {
  try {
    console.log('🧪 Testing database connection...');
    console.log('📊 Connection state:', mongoose.connection.readyState);
    console.log('📊 Database name:', mongoose.connection.db ? mongoose.connection.db.databaseName : 'No database');
    
    // Test basic connection
    const collections = await mongoose.connection.db.listCollections().toArray();
    console.log('📊 Available collections:', collections.map(c => c.name));
    
    // Test jobcards collection specifically
    const jobCardsCollection = mongoose.connection.db.collection('jobcards');
    const count = await jobCardsCollection.countDocuments();
    const sample = await jobCardsCollection.findOne();
    
    console.log('📊 Job cards count:', count);
    console.log('📊 Sample document:', sample);
    
    res.json({
      success: true,
      connectionState: mongoose.connection.readyState,
      databaseName: mongoose.connection.db.databaseName,
      collections: collections.map(c => c.name),
      jobCardsCount: count,
      sampleDocument: sample
    });
  } catch (error) {
    console.error('❌ Database test error:', error);
    res.status(500).json({
      success: false,
      error: error.message,
      stack: error.stack
    });
  }
});

// Debug endpoint - MUST be before parameterized routes
app.get('/jobcards/debug', async (req, res) => {
  try {
    console.log('🔍 Debug endpoint called');
    console.log('📊 Mongoose connection state:', mongoose.connection.readyState);
    console.log('📊 Collection name:', JobCard.collection.name);
    
    const total = await JobCard.countDocuments({});
    const sample = await JobCard.findOne({});
    const all = await JobCard.find({}).limit(5);
    
    console.log('📊 Total documents found:', total);
    console.log('📊 Sample document:', sample);
    console.log('📊 All documents:', all.length);
    
    res.json({
      success: true,
      totalDocuments: total,
      sampleDocument: sample,
      allDocuments: all,
      collectionName: JobCard.collection.name,
      mongooseState: mongoose.connection.readyState,
      databaseName: mongoose.connection.db.databaseName
    });
  } catch (error) {
    console.error('❌ Debug endpoint error:', error);
    res.status(500).json({
      success: false,
      error: error.message,
      stack: error.stack
    });
  }
});

// Get all job cards from all users (for global view) - MUST be before parameterized routes
app.get('/jobcards/all', async (req, res) => {
  try {
    console.log('🔍 /jobcards/all endpoint called - START');
    const { page = 1, limit = 50 } = req.query;
    
    console.log('🔍 /jobcards/all endpoint called with params:', { page, limit });
    console.log('📊 Mongoose connection state:', mongoose.connection.readyState);
    console.log('📊 Collection name:', JobCard.collection.name);
    
    const skip = (page - 1) * limit;
    
    console.log('🔍 About to query JobCard.find({})');
    const jobCards = await JobCard.find({})
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    console.log('🔍 About to count documents');
    const total = await JobCard.countDocuments({});
    
    console.log('📊 Total job cards in collection:', total);
    console.log('📊 Job cards found:', jobCards.length);
    console.log('📊 Sample job card:', jobCards[0]);
    
    const response = {
      success: true,
      data: jobCards,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    };
    
    console.log('🔍 Sending response:', JSON.stringify(response, null, 2));
    res.json(response);
  } catch (error) {
    console.error('❌ Error fetching all job cards:', error);
    console.error('❌ Error stack:', error.stack);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch all job cards: ' + error.message 
    });
  }
});

// Get next incremental number based on existing request numbers in MongoDB
// Returns { lastNumber: 12, nextNumber: 13, lastRequestNumber: 'ZAM-KAN-JC-0012' }
app.get('/jobcards/next-number', async (req, res) => {
  try {
    // Optional filters if you need per-country/project numbering later
    const { country, project } = req.query;

    const match = {};
    if (country) match['location.country'] = country;
    if (project) match['location.project'] = project;

    const all = await JobCard.find(match)
      .select('requestNumber createdAt')
      .sort({ createdAt: -1 })
      .lean();

    let lastNumber = 0;
    let lastRequestNumber = null;

    for (const doc of all) {
      const rn = doc.requestNumber || '';
      // Remove any -OFFLINE suffix
      const cleaned = rn.replace('-OFFLINE', '');
      const m = cleaned.match(/-JC-(\d+)/);
      if (m) {
        const n = parseInt(m[1], 10);
        if (n > lastNumber) {
          lastNumber = n;
          lastRequestNumber = cleaned;
        }
      }
    }

    return res.json({
      success: true,
      lastNumber,
      nextNumber: lastNumber + 1,
      lastRequestNumber
    });
  } catch (error) {
    console.error('❌ Error computing next number:', error);
    return res.status(500).json({ success: false, error: 'Failed to compute next number: ' + error.message });
  }
});

// Get all job cards for a user
app.get('/jobcards/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    const { page = 1, limit = 50, status, priority, search } = req.query;
    
    const query = { adminUid };
    if (status) query.status = status;
    if (priority) query.priority = priority;
    
    // Add search functionality
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { 'equipment.equipmentName': { $regex: search, $options: 'i' } },
        { workOrderType: { $regex: search, $options: 'i' } },
        { 'assignTo.name': { $regex: search, $options: 'i' } }
      ];
    }
    
    const skip = (page - 1) * limit;
    
    const jobCards = await JobCard.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await JobCard.countDocuments(query);
    
    res.json({
      success: true,
      data: jobCards,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching job cards:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch job cards' 
    });
  }
});

// Get a specific job card by ID
app.get('/jobcards/report/:jobCardId', async (req, res) => {
  try {
    const { jobCardId } = req.params;
    
    const jobCard = await JobCard.findOne({ 
      _id: jobCardId 
    }).select('-__v');
    
    if (!jobCard) {
      return res.status(404).json({ 
        success: false, 
        error: 'Job card not found' 
      });
    }
    
    res.json({
      success: true,
      data: jobCard
    });
  } catch (error) {
    console.error('Error fetching job card:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch job card' 
    });
  }
});

// Helper function to send job card notifications
const sendJobCardNotifications = async (jobCardData, createdByName, creatorEmail) => {
  try {
    console.log('🔔 Sending job card notifications...');
    
    const pdfUrl =
      jobCardData.pdfDownloadUrl ||
      jobCardData.pdfUrl ||
      jobCardData.documentUrl ||
      '';
    const requestNumber = jobCardData.requestNumber || jobCardData.title || '';

    const notificationPayload = {
      jobCardData: {
        _id: jobCardData._id,
        id: jobCardData._id,
        title: jobCardData.title,
        equipmentName: jobCardData.equipment?.equipmentName || 'N/A',
        priority: jobCardData.priority || 'Low',
        workOrderType: jobCardData.workOrderType || 'N/A',
        country: jobCardData.location?.country || '',
        project: jobCardData.location?.project || '',
        description: jobCardData.description || '',
        assignedCount: jobCardData.assignTo?.length || 0,
        itemsCount: jobCardData.items?.length || 0,
        assignTo: jobCardData.assignTo || [],
        pdfUrl,
        requestNumber,
      },
      createdByName,
      creatorEmail,
    };
    
    const response = await fetch('https://api.titandrillingzm.com:6023/api/jobcard-notifications/new-jobcard', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(notificationPayload),
    });
    
    const result = await response.json();
    
    if (result.success) {
      console.log('✅ Job card notifications sent successfully!');
      console.log('📊 Recipients:', result.recipients);
    } else {
      console.error('❌ Failed to send notifications:', result.error);
    }
    
    return result;
  } catch (error) {
    console.error('❌ Error sending job card notifications:', error);
    return { success: false, error: error.message };
  }
};

// Create a new job card
app.post('/jobcards', async (req, res) => {
  try {
    const { adminUid, jobCardData } = req.body;
    
    console.log('📊 Raw request body:', JSON.stringify(req.body, null, 2));
    console.log('📊 Creating job card for adminUid:', adminUid);
    console.log('📊 Job card data received:', {
      title: jobCardData?.title,
      itemsCount: jobCardData?.items?.length || 0,
      hasEquipment: !!jobCardData?.equipment,
      hasAssignTo: !!jobCardData?.assignTo,
      itemsType: typeof jobCardData?.items,
      itemsIsArray: Array.isArray(jobCardData?.items)
    });
    
    if (!adminUid || !jobCardData) {
      console.log('❌ Missing required fields:', { adminUid: !!adminUid, jobCardData: !!jobCardData });
      return res.status(400).json({ 
        success: false, 
        error: 'adminUid and jobCardData are required' 
      });
    }
    
    const jobCardId = jobCardData._id || jobCardData.id || Date.now().toString();
    
    console.log('📊 Creating job card with data:', {
      _id: jobCardId,
      adminUid,
      title: jobCardData.title,
      status: jobCardData.status,
      itemsCount: jobCardData.items?.length || 0,
      itemsType: typeof jobCardData.items
    });
    
    console.log('📊 Schema validation - checking status:', jobCardData.status);
    console.log('📊 Valid status values:', ['Hold', 'assigned', 'in_progress', 'complete', 'cancelled']);

    // Handle items array - check if it's stringified
    if (jobCardData.items) {
      console.log('📊 Processing items array, type:', typeof jobCardData.items);
      
      if (typeof jobCardData.items === 'string') {
        console.log('⚠️ Items received as string, parsing...');
        try {
          jobCardData.items = JSON.parse(jobCardData.items);
          console.log('✅ Items parsed successfully, type after parse:', typeof jobCardData.items);
        } catch (parseError) {
          console.error('❌ Failed to parse items string:', parseError);
          jobCardData.items = [];
        }
      }
      
      if (Array.isArray(jobCardData.items)) {
        console.log('📊 Validating items array:', jobCardData.items.length, 'items');
        jobCardData.items.forEach((item, index) => {
          if (!item.id) {
            console.log(`⚠️ Item ${index} missing id, generating one`);
            item.id = `item_${Date.now()}_${index}`;
          }
          if (!item.status) {
            console.log(`⚠️ Item ${index} missing status, setting to Initiated`);
            item.status = 'Initiated';
          }
          if (!item.category) {
            console.log(`⚠️ Item ${index} missing category, setting to Task`);
            item.category = 'Task';
          }
        });
      } else {
        console.log('⚠️ Items is not an array, setting to empty array');
        jobCardData.items = [];
      }
    }

    const jobCard = new JobCard({
      _id: jobCardId,
      adminUid,
      ...jobCardData,
      createdBy: adminUid,
      lastModifiedBy: adminUid
    });
    
    console.log('📊 Attempting to save job card...');
    await jobCard.save();
    console.log('✅ Job card saved successfully');
    
    // Send notifications in background (fire-and-forget)
    setImmediate(async () => {
      try {
        // Get creator name and email from adminUid
        const GlobalUserSchema = new mongoose.Schema({}, { strict: false, collection: 'GlobalUsers' });
        let GlobalUser;
        try {
          GlobalUser = mongoose.model('GlobalUser');
        } catch (error) {
          GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
        }
        
        const creator = await GlobalUser.findOne({ _id: adminUid }).select('name email');
        const createdByName = creator?.name || 'Unknown User';
        const creatorEmail = creator?.email || '';
        
        console.log('📧 Sending job card notifications for creator:', createdByName);
        await sendJobCardNotifications(jobCard.toObject(), createdByName, creatorEmail);
      } catch (notifError) {
        console.error('❌ Error sending job card notifications:', notifError);
      }
    });
    
    res.status(201).json({
      success: true,
      data: jobCard,
      message: 'Job card created successfully'
    });
  } catch (error) {
    console.error('❌ Error creating job card:', error);
    console.error('❌ Error details:', {
      name: error.name,
      message: error.message,
      code: error.code,
      stack: error.stack
    });
    res.status(500).json({ 
      success: false, 
      error: 'Failed to create job card: ' + error.message 
    });
  }
});

// Update a job card (simplified - only requires jobCardId)
app.put('/jobcards/update/:jobCardId', async (req, res) => {
  try {
    const { jobCardId } = req.params;
    const updateData = req.body;
    
    console.log('🔧 Update request received:', { jobCardId, updateData });
    
    delete updateData._id;
    delete updateData.adminUid;
    
    updateData.updatedAt = new Date();
    
    console.log('🔧 Searching for job card with _id:', jobCardId);
    
    // First, let's check if the job card exists
    const existingJobCard = await JobCard.findOne({ _id: jobCardId });
    console.log('🔧 Existing job card found:', existingJobCard ? 'YES' : 'NO');
    if (existingJobCard) {
      console.log('🔧 Existing job card _id:', existingJobCard._id);
    }
    
    const jobCard = await JobCard.findOneAndUpdate(
      { _id: jobCardId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    console.log('🔧 Update result:', jobCard ? 'SUCCESS' : 'NOT FOUND');
    
    if (!jobCard) {
      return res.status(404).json({
        success: false,
        message: 'Job card not found'
      });
    }
    
    res.json({
      success: true,
      message: 'Job card updated successfully',
      data: jobCard
    });
  } catch (error) {
    console.error('❌ Error updating job card:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update job card',
      error: error.message
    });
  }
});

// Update a job card (original - requires adminUid and jobCardId)
app.put('/jobcards/:adminUid/:jobCardId', async (req, res) => {
  try {
    const { adminUid, jobCardId } = req.params;
    const updateData = req.body;
    
    delete updateData._id;
    delete updateData.adminUid;
    
    updateData.updatedAt = new Date();
    updateData.lastModifiedBy = adminUid;
    
    const jobCard = await JobCard.findOneAndUpdate(
      { _id: jobCardId, adminUid },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!jobCard) {
      return res.status(404).json({ 
        success: false, 
        error: 'Job card not found' 
      });
    }
    
    res.json({
      success: true,
      data: jobCard,
      message: 'Job card updated successfully'
    });
  } catch (error) {
    console.error('Error updating job card:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to update job card' 
    });
  }
});

// Delete a job card
app.delete('/jobcards/:adminUid/:jobCardId', async (req, res) => {
  try {
    const { adminUid, jobCardId } = req.params;
    
    const jobCard = await JobCard.findOneAndDelete({ 
      _id: jobCardId, 
      adminUid 
    });
    
    if (!jobCard) {
      return res.status(404).json({ 
        success: false, 
        error: 'Job card not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Job card deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting job card:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete job card' 
    });
  }
});

// Upload PDF for job card
app.post('/jobcards/upload-pdf/:adminUid', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 Job Card PDF Upload Request received:', {
      adminUid: req.params.adminUid,
      hasFile: !!req.file,
      fileInfo: req.file ? {
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size,
        path: req.file.path
      } : null
    });
    
    if (!req.file) {
      console.log('❌ No PDF file in request');
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const adminUid = req.params.adminUid;
    const fileName = `jobcard_${adminUid}_${Date.now()}.pdf`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'GlobalJobcardReports', adminUid);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    let finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    const fullDownloadUrl = `https://api.titandrillingzm.com:6005/jobcards/download-file/${adminUid}/${fileName}`;
    
    console.log('🔗 Generated full download URL:', fullDownloadUrl);
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: fullDownloadUrl,
      message: 'PDF uploaded successfully'
    });

  } catch (error) {
    console.error('Job card PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload PDF: ' + error.message
    });
  }
});

// Upload signature for job card
app.post('/jobcards/upload-signature/:adminUid', fileUpload.single('signature'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No signature image uploaded'
      });
    }

    const adminUid = req.params.adminUid;
    const fileName = `signature_${adminUid}_${Date.now()}.png`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'GlobalJobcardReports', adminUid);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    const fullDownloadUrl = `https://api.titandrillingzm.com:6005/jobcards/download-file/${adminUid}/${fileName}`;
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: fullDownloadUrl,
      message: 'Signature uploaded successfully'
    });

  } catch (error) {
    console.error('Job card signature upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload signature: ' + error.message
    });
  }
});

// Download file from job card storage
app.get('/jobcards/download-file/:adminUid/:fileName', (req, res) => {
  try {
    const { adminUid, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'GlobalJobcardReports', adminUid, fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'File not found'
      });
    }

    // Set appropriate headers
    const ext = path.extname(fileName).toLowerCase();
    let contentType = 'application/octet-stream';
    
    if (ext === '.pdf') {
      contentType = 'application/pdf';
    } else if (ext === '.png') {
      contentType = 'image/png';
    } else if (ext === '.jpg' || ext === '.jpeg') {
      contentType = 'image/jpeg';
    }
    
    res.set({
      'Content-Type': contentType,
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('File download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download file'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download file: ' + error.message
    });
  }
});

// Get job card statistics
app.get('/jobcards/:adminUid/stats', async (req, res) => {
  try {
    const { adminUid } = req.params;
    const { startDate, endDate } = req.query;
    
    const matchQuery = { adminUid };
    if (startDate || endDate) {
      matchQuery.createdAt = {};
      if (startDate) matchQuery.createdAt.$gte = new Date(startDate);
      if (endDate) matchQuery.createdAt.$lte = new Date(endDate);
    }
    
    const stats = await JobCard.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: null,
          totalJobCards: { $sum: 1 },
          completedJobCards: {
            $sum: { $cond: [{ $eq: ['$status', 'complete'] }, 1, 0] }
          },
          inProgressJobCards: {
            $sum: { $cond: [{ $eq: ['$status', 'in_progress'] }, 1, 0] }
          },
          assignedJobCards: {
            $sum: { $cond: [{ $eq: ['$status', 'assigned'] }, 1, 0] }
          },
          draftJobCards: {
            $sum: { $cond: [{ $eq: ['$status', 'Hold'] }, 1, 0] }
          }
        }
      }
    ]);
    
    const priorityStats = await JobCard.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: '$priority',
          count: { $sum: 1 }
        }
      },
      { $sort: { count: -1 } }
    ]);
    
    res.json({
      success: true,
      data: {
        overview: stats[0] || {
          totalJobCards: 0,
          completedJobCards: 0,
          inProgressJobCards: 0,
          assignedJobCards: 0,
          draftJobCards: 0
        },
        byPriority: priorityStats
      }
    });
  } catch (error) {
    console.error('Error fetching job card statistics:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch job card statistics' 
    });
  }
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Unhandled error:', error);
  res.status(500).json({ 
    success: false, 
    error: 'Internal server error' 
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({ 
    success: false, 
    error: 'Endpoint not found' 
  });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🔧 Job Card Handler server running on port ${PORT}`);
  console.log(`📊 Health check: http://localhost:${PORT}/jobcards`);
  console.log(`📊 External access: https://api.titandrillingzm.com:${PORT}/jobcards`);
  console.log(`🔢 Total count: https://api.titandrillingzm.com:${PORT}/jobcards/all`);
  console.log(`📁 PDF upload: https://api.titandrillingzm.com:${PORT}/jobcards/upload-pdf/:adminUid`);
  console.log(`📁 Signature upload: https://api.titandrillingzm.com:${PORT}/jobcards/upload-signature/:adminUid`);
  console.log(`📁 File download: https://api.titandrillingzm.com:${PORT}/jobcards/download-file/:adminUid/:fileName`);
  console.log(`🔍 Get job card by ID: https://api.titandrillingzm.com:${PORT}/jobcards/report/:jobCardId`);
  console.log(`📊 Statistics: https://api.titandrillingzm.com:${PORT}/jobcards/:adminUid/stats`);
});

module.exports = app;
