const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const cors = require('cors');

const app = express();
const PORT = process.env.HSE_INSPECTION_PORT || 5014;

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Using shared MongoDB connection from dbConnection.js
console.log('[HSE Inspection] Using shared MongoDB connection');

// HSE Inspection Report Schema
const hseInspectionSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  userId: { type: String, required: true },
  requestNumber: { type: String, required: true },
  status: { 
    type: String, 
    enum: ['draft', 'completed', 'archived'], 
    default: 'draft' 
  },
  
  // Step 1: HSE Information
  step1: {
    id: String,
    country: String,
    project: String,
    countryCode: String,
    projectCode: String,
    requestNumber: String,
    checklistTitle: String,
    checklistDocumentId: String,
    selectedChecklist: {
      document_id: String,
      document_name: String,
      version: String,
      version_date: String,
      approved_by: String,
      title: String,
      checklist: mongoose.Schema.Types.Mixed
    },
    selectedEquipment: {
      id: String,
      equipmentName: String,
      equipmentNumber: String,
      entryNumber: String,
      mainCategory: String,
      model: String,
      meterReading: Number,
      meterUnit: String,
      status: String,
      country: String,
      createdAt: Date
    },
    coordinates: String,
    gpsAddress: String,
    createdAt: Date
  },
  
  // Step 2: HSE Assessment
  step2: {
    id: String,
    checklistId: String,
    checklistName: String,
    sections: [{
      key: String,
      name: String,
      subItems: [{
        key: String,
        itemName: String,
        category: String,
        inputType: String,
        sortOrder: mongoose.Schema.Types.Mixed,
        isInspectionRequired: mongoose.Schema.Types.Mixed, // Allow string, boolean, or empty
        description: String,
        instruction: String,
        options: mongoose.Schema.Types.Mixed
      }]
    }],
    selections: mongoose.Schema.Types.Mixed,
    countInputs: mongoose.Schema.Types.Mixed,
    notes: mongoose.Schema.Types.Mixed,
    selectedFiles: mongoose.Schema.Types.Mixed,
    numericInputs: mongoose.Schema.Types.Mixed,
    completedAt: Date
  },
  
  // Step 3: HSE Review
  step3: {
    safetyOfficer: String,
    siteSupervisor: String,
    inspectionDate: Date,
    inspectionTime: Date,
    dynamicFields: [{
      id: Number,
      actionToBeTaken: String,
      date: Date,
      recommendation: String
    }],
    totalSafe: Number,
    totalUnsafe: Number,
    totalNA: Number,
    percentageSafe: Number,
    pdfDownloadUrl: String,
    createdAt: Date
  },
  
  // PDF and Files
  pdfDownloadUrl: String,
  attachments: [{
    fileName: String,
    fileUrl: String,
    uploadedAt: Date
  }],
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  createdBy: String,
  lastModifiedBy: String
}, { 
  strict: false,
  collection: 'globalhseinspections'
});

// Create indexes
hseInspectionSchema.index({ userId: 1, createdAt: -1 });
hseInspectionSchema.index({ requestNumber: 1 });
hseInspectionSchema.index({ 'step1.country': 1, 'step1.project': 1 });
hseInspectionSchema.index({ status: 1 });

// Pre-save middleware to sanitize data
hseInspectionSchema.pre('save', function(next) {
  // Sanitize step2 sections if they exist
  if (this.step2 && this.step2.sections) {
    this.step2.sections = this.step2.sections.map(section => {
      if (section.subItems) {
        section.subItems = section.subItems.map(item => {
          // Convert empty string to false for isInspectionRequired
          if (item.isInspectionRequired === '' || item.isInspectionRequired === null) {
            item.isInspectionRequired = false;
          } else if (item.isInspectionRequired === 'true' || item.isInspectionRequired === '1') {
            item.isInspectionRequired = true;
          } else if (item.isInspectionRequired === 'false' || item.isInspectionRequired === '0') {
            item.isInspectionRequired = false;
          }
          
          // Convert empty string to 0 for sortOrder
          if (item.sortOrder === '' || item.sortOrder === null) {
            item.sortOrder = 0;
          }
          
          return item;
        });
      }
      return section;
    });
  }
  next();
});

const HSEInspection = mongoose.model('HSEInspection', hseInspectionSchema);

// Report Number Counter Schema
const hseCounterSchema = new mongoose.Schema({
  _id: { type: String, default: 'hse_report_counter' },
  currentNumber: { type: Number, default: 0 }
}, { collection: 'hsecounters' });

const HSECounter = mongoose.model('HSECounter', hseCounterSchema);

// Multer configuration for file uploads
const hseStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
    }
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const fileExtension = path.extname(file.originalname);
    cb(null, file.fieldname + '-' + uniqueSuffix + fileExtension);
  }
});

const fileUpload = multer({ 
  storage: hseStorage,
  limits: { fileSize: 50 * 1024 * 1024 }
});

// Routes

// ========== ROUTES - ORDERED FROM MOST SPECIFIC TO LEAST SPECIFIC ==========

// Health check
app.get('/hse-inspections', (req, res) => {
  res.json({ 
    status: 'OK', 
    service: 'HSE Inspection Handler',
    port: PORT,
    timestamp: new Date().toISOString()
  });
});

// Get current counter (specific route)
app.get('/hse-inspections/counter/current', async (req, res) => {
  try {
    let counter = await HSECounter.findOne({ _id: 'hse_report_counter' });
    
    if (!counter) {
      counter = new HSECounter({ _id: 'hse_report_counter', currentNumber: 0 });
      await counter.save();
    }
    
    res.json({
      success: true,
      data: {
        currentNumber: counter.currentNumber,
        nextNumber: (counter.currentNumber + 1).toString().padStart(3, '0'),
        formattedNumber: (counter.currentNumber + 1).toString().padStart(3, '0')
      }
    });
  } catch (error) {
    console.error('❌ Error fetching counter:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch counter' 
    });
  }
});

// Increment counter (specific route)
app.post('/hse-inspections/counter/increment', async (req, res) => {
  try {
    let counter = await HSECounter.findOne({ _id: 'hse_report_counter' });
    
    if (!counter) {
      counter = new HSECounter({ _id: 'hse_report_counter', currentNumber: 1 });
    } else {
      counter.currentNumber += 1;
    }
    
    await counter.save();
    
    res.json({
      success: true,
      data: {
        currentNumber: counter.currentNumber,
        formattedNumber: counter.currentNumber.toString().padStart(3, '0')
      }
    });
  } catch (error) {
    console.error('❌ Error incrementing counter:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to increment counter' 
    });
  }
});

// Upload PDF (specific route)
app.post('/hse-inspections/upload-pdf/:userId', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 HSE PDF Upload Request received:', {
      userId: req.params.userId,
      hasFile: !!req.file
    });
    
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const userId = req.params.userId;
    const fileName = `hse_inspection_${userId}_${Date.now()}.pdf`;
    
    // Create user-specific directory: /root/node-mongo-api/uploads/documents/HSEinspectiondocx/{uid}/
    const userDir = path.join(__dirname, 'uploads', 'documents', 'HSEinspectiondocx', userId);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created HSE directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file:', moveError);
      // If move fails, try copying
      fs.copyFileSync(req.file.path, finalPath);
      fs.unlinkSync(req.file.path);
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `https://api.titandrillingzm.com:6014/hse-inspections/download-file/${userId}/${fileName}`,
      message: 'HSE PDF uploaded successfully'
    });

  } catch (error) {
    console.error('❌ HSE PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload PDF: ' + error.message
    });
  }
});

// Upload attachment (specific route)
app.post('/hse-inspections/upload-attachment/:userId', fileUpload.single('attachment'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No file uploaded'
      });
    }

    const userId = req.params.userId;
    const fileName = `attachment_${userId}_${Date.now()}${path.extname(req.file.originalname)}`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'HSEinspectiondocx', userId);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
    }
    
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
    } catch (moveError) {
      fs.copyFileSync(req.file.path, finalPath);
      fs.unlinkSync(req.file.path);
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `https://api.titandrillingzm.com:6014/hse-inspections/download-file/${userId}/${fileName}`,
      message: 'Attachment uploaded successfully'
    });

  } catch (error) {
    console.error('❌ Attachment upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload attachment: ' + error.message
    });
  }
});

// Download file (specific route)
app.get('/hse-inspections/download-file/:userId/:fileName', (req, res) => {
  try {
    const { userId, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'HSEinspectiondocx', userId, fileName);
    
    console.log('📥 Download request for:', filePath);
    
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'File not found'
      });
    }

    const ext = path.extname(fileName).toLowerCase();
    let contentType = 'application/octet-stream';
    
    if (ext === '.pdf') {
      contentType = 'application/pdf';
    } else if (ext === '.png') {
      contentType = 'image/png';
    } else if (ext === '.jpg' || ext === '.jpeg') {
      contentType = 'image/jpeg';
    }
    
    res.set({
      'Content-Type': contentType,
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('❌ File download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download file'
      });
    });

  } catch (error) {
    console.error('❌ Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download file: ' + error.message
    });
  }
});

// Get all HSE inspections from all users (specific route)
app.get('/hse-inspections/all', async (req, res) => {
  try {
    const { page = 1, limit = 100 } = req.query;
    
    console.log('📋 Fetching all HSE inspections');
    
    const skip = (page - 1) * limit;
    
    const reports = await HSEInspection.find({})
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await HSEInspection.countDocuments({});
    
    console.log(`✅ Found ${reports.length} total HSE inspections`);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('❌ Error fetching all HSE inspections:', error);
    res.json({ 
      success: false, 
      error: 'Failed to fetch all HSE inspections: ' + error.message 
    });
  }
});

// Get all HSE inspections for a user
app.get('/hse-inspections/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    const { page = 1, limit = 50, status, country, project } = req.query;
    
    console.log('📋 Fetching HSE inspections for user:', userId);
    
    const query = { userId };
    if (status) query.status = status;
    if (country) query['step1.country'] = country;
    if (project) query['step1.project'] = project;
    
    const skip = (page - 1) * limit;
    
    const reports = await HSEInspection.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await HSEInspection.countDocuments(query);
    
    console.log(`✅ Found ${reports.length} HSE inspections for user ${userId}`);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('❌ Error fetching HSE inspections:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch HSE inspections' 
    });
  }
});

// Get specific HSE inspection by ID
app.get('/hse-inspections/report/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const report = await HSEInspection.findOne({ _id: reportId }).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'HSE inspection not found' 
      });
    }
    
    res.json({
      success: true,
      data: report
    });
  } catch (error) {
    console.error('❌ Error fetching HSE inspection:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch HSE inspection' 
    });
  }
});

// Helper function to sanitize data before saving
const sanitizeReportData = (data) => {
  const sanitized = JSON.parse(JSON.stringify(data)); // Deep clone
  
  if (!sanitized.step1) {
    sanitized.step1 = {};
  }

  if (sanitized.step1.selectedChecklist) {
    const checklist = sanitized.step1.selectedChecklist;
    sanitized.step1.checklistTitle =
      sanitized.step1.checklistTitle ||
      checklist.document_name ||
      checklist.title ||
      checklist.document_id ||
      checklist.documentId ||
      '';
    sanitized.step1.checklistDocumentId =
      sanitized.step1.checklistDocumentId ||
      checklist.document_id ||
      checklist.documentId ||
      '';
  }

  // Sanitize step2 sections
  if (sanitized.step2 && sanitized.step2.sections) {
    sanitized.step2.sections = sanitized.step2.sections.map(section => {
      if (section.subItems) {
        section.subItems = section.subItems.map(item => {
          // Convert empty string to false for isInspectionRequired
          if (item.isInspectionRequired === '' || item.isInspectionRequired === null || item.isInspectionRequired === undefined) {
            item.isInspectionRequired = false;
          } else if (typeof item.isInspectionRequired === 'string') {
            item.isInspectionRequired = item.isInspectionRequired === 'true' || item.isInspectionRequired === '1';
          }
          
          // Convert sortOrder to number
          if (typeof item.sortOrder === 'string') {
            item.sortOrder = parseInt(item.sortOrder) || 0;
          } else if (item.sortOrder === null || item.sortOrder === undefined) {
            item.sortOrder = 0;
          }
          
          return item;
        });
      }
      return section;
    });
  }
  
  return sanitized;
};

// Create new HSE inspection
app.post('/hse-inspections', async (req, res) => {
  try {
    const { userId, reportData } = req.body;
    
    console.log('📝 Creating HSE inspection for user:', userId);
    console.log('📝 Report ID:', reportData._id || reportData.id);
    
    if (!userId || !reportData) {
      console.log('❌ Missing required fields');
      return res.status(400).json({ 
        success: false, 
        error: 'userId and reportData are required' 
      });
    }
    
    const reportId = reportData._id || reportData.id || Date.now().toString();
    
    console.log('📝 Creating HSE inspection with ID:', reportId);
    console.log('📝 Request number:', reportData.step1?.requestNumber || reportData.requestNumber);
    
    // Sanitize data before saving
    const sanitizedData = sanitizeReportData(reportData);
    
    const hseInspection = new HSEInspection({
      _id: reportId,
      userId,
      requestNumber: sanitizedData.step1?.requestNumber || sanitizedData.requestNumber || reportId,
      status: sanitizedData.status || 'draft',
      step1: sanitizedData.step1 || {},
      step2: sanitizedData.step2 || {},
      step3: sanitizedData.step3 || {},
      pdfDownloadUrl: sanitizedData.pdfDownloadUrl || '',
      attachments: sanitizedData.attachments || [],
      createdBy: userId,
      lastModifiedBy: userId
    });
    
    console.log('💾 Saving to MongoDB...');
    await hseInspection.save();
    
    console.log('✅ HSE inspection created successfully:', reportId);
    
    res.status(201).json({
      success: true,
      data: hseInspection,
      message: 'HSE inspection created successfully'
    });
  } catch (error) {
    console.error('❌ Error creating HSE inspection:', error);
    console.error('❌ Error details:', {
      name: error.name,
      message: error.message,
      stack: error.stack
    });
    res.status(500).json({ 
      success: false, 
      error: 'Failed to create HSE inspection: ' + error.message 
    });
  }
});

// Update HSE inspection
app.put('/hse-inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const updateData = req.body;
    
    delete updateData._id;
    delete updateData.userId;
    
    updateData.updatedAt = new Date();
    updateData.lastModifiedBy = userId;
    
    const report = await HSEInspection.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'HSE inspection not found' 
      });
    }
    
    console.log('✅ HSE inspection updated:', reportId);
    
    res.json({
      success: true,
      data: report,
      message: 'HSE inspection updated successfully'
    });
  } catch (error) {
    console.error('❌ Error updating HSE inspection:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to update HSE inspection' 
    });
  }
});

// Complete HSE inspection
app.put('/hse-inspections/:userId/:reportId/complete', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const { pdfDownloadUrl } = req.body;
    
    const updateData = {
      status: 'completed',
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    if (pdfDownloadUrl) {
      updateData.pdfDownloadUrl = pdfDownloadUrl;
      updateData['step3.pdfDownloadUrl'] = pdfDownloadUrl;
    }
    
    const report = await HSEInspection.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'HSE inspection not found' 
      });
    }
    
    console.log('✅ HSE inspection completed:', reportId);
    
    res.json({
      success: true,
      data: report,
      message: 'HSE inspection completed successfully'
    });
  } catch (error) {
    console.error('❌ Error completing HSE inspection:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to complete HSE inspection' 
    });
  }
});

// Delete HSE inspection
app.delete('/hse-inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    
    const report = await HSEInspection.findOneAndDelete({ 
      _id: reportId, 
      userId 
    });
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'HSE inspection not found' 
      });
    }
    
    console.log('✅ HSE inspection deleted:', reportId);
    
    res.json({
      success: true,
      message: 'HSE inspection deleted successfully'
    });
  } catch (error) {
    console.error('❌ Error deleting HSE inspection:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete HSE inspection' 
    });
  }
});

// Get HSE statistics
app.get('/hse-inspections/:userId/stats', async (req, res) => {
  try {
    const { userId } = req.params;
    const { country, project, startDate, endDate } = req.query;
    
    const matchQuery = { userId };
    if (country) matchQuery['step1.country'] = country;
    if (project) matchQuery['step1.project'] = project;
    if (startDate || endDate) {
      matchQuery.createdAt = {};
      if (startDate) matchQuery.createdAt.$gte = new Date(startDate);
      if (endDate) matchQuery.createdAt.$lte = new Date(endDate);
    }
    
    const stats = await HSEInspection.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: null,
          totalReports: { $sum: 1 },
          completedReports: {
            $sum: { $cond: [{ $eq: ['$status', 'completed'] }, 1, 0] }
          },
          draftReports: {
            $sum: { $cond: [{ $eq: ['$status', 'draft'] }, 1, 0] }
          },
          totalSafe: { $sum: '$step3.totalSafe' },
          totalUnsafe: { $sum: '$step3.totalUnsafe' },
          avgPercentageSafe: { $avg: '$step3.percentageSafe' }
        }
      }
    ]);
    
    res.json({
      success: true,
      data: stats[0] || {
        totalReports: 0,
        completedReports: 0,
        draftReports: 0,
        totalSafe: 0,
        totalUnsafe: 0,
        avgPercentageSafe: 0
      }
    });
  } catch (error) {
    console.error('❌ Error fetching HSE statistics:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch HSE statistics' 
    });
  }
});

// Search HSE inspections
app.get('/hse-inspections/:userId/search', async (req, res) => {
  try {
    const { userId } = req.params;
    const { q, page = 1, limit = 50 } = req.query;
    
    if (!q) {
      return res.status(400).json({ 
        success: false, 
        error: 'Search query is required' 
      });
    }
    
    const searchQuery = {
      userId,
      $or: [
        { requestNumber: { $regex: q, $options: 'i' } },
        { 'step1.country': { $regex: q, $options: 'i' } },
        { 'step1.project': { $regex: q, $options: 'i' } },
        { 'step1.selectedEquipment.equipmentName': { $regex: q, $options: 'i' } },
        { 'step1.selectedChecklist.document_id': { $regex: q, $options: 'i' } }
      ]
    };
    
    const skip = (page - 1) * limit;
    
    const reports = await HSEInspection.find(searchQuery)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await HSEInspection.countDocuments(searchQuery);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('❌ Error searching HSE inspections:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to search HSE inspections' 
    });
  }
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Unhandled error:', error);
  res.status(500).json({ 
    success: false, 
    error: 'Internal server error' 
  });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🏥 HSE Inspection Handler server running on port ${PORT}`);
  console.log(`📊 Health check: https://api.titandrillingzm.com:${PORT}/hse-inspections`);
  console.log(`📋 Get user inspections: https://api.titandrillingzm.com:${PORT}/hse-inspections/:userId`);
  console.log(`📋 Get all inspections: https://api.titandrillingzm.com:${PORT}/hse-inspections/all`);
  console.log(`📋 Get specific: https://api.titandrillingzm.com:${PORT}/hse-inspections/report/:reportId`);
  console.log(`📝 Create inspection: https://api.titandrillingzm.com:${PORT}/hse-inspections`);
  console.log(`📁 Upload PDF: https://api.titandrillingzm.com:${PORT}/hse-inspections/upload-pdf/:userId`);
  console.log(`📁 Upload attachment: https://api.titandrillingzm.com:${PORT}/hse-inspections/upload-attachment/:userId`);
  console.log(`📁 Download file: https://api.titandrillingzm.com:${PORT}/hse-inspections/download-file/:userId/:fileName`);
  console.log(`🔢 Get counter: https://api.titandrillingzm.com:${PORT}/hse-inspections/counter/current`);
  console.log(`🔢 Increment counter: https://api.titandrillingzm.com:${PORT}/hse-inspections/counter/increment`);
});

module.exports = app;

