const express = require('express');
const cors = require('cors');
const fs = require('fs/promises');
const path = require('path');
const bcrypt = require('bcryptjs');
const { mongoose } = require('./dbConnection');

require('dotenv').config();

const app = express();
const PORT = process.env.BULK_DATA_PORT || 5034;

app.use(cors());
app.use(express.json({ limit: '100mb' }));
app.use(express.urlencoded({ extended: true, limit: '100mb' }));

console.log('✅ [BulkDataUploader] Using shared MongoDB connection');

// ===== Models =====
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [BulkDataUploader] Reusing GlobalUser model');
} catch (error) {
  const GlobalUserSchema = new mongoose.Schema(
    {
      _id: { type: String, required: true },
      uid: { type: String, required: true },
      name: { type: String, required: true },
      email: { type: String, required: true },
      employeeNumber: { type: String },
      department: { type: String },
      countries: [String],
      modules: [Object],
      notifications: Object,
      loginStatus: { type: Boolean, default: false },
      lastLogin: { type: Date },
      authCreated: { type: Boolean, default: false },
      uploadTimestamp: { type: Date, default: Date.now },
      originalUserId: { type: String },
      password: { type: String },
      hashedPassword: { type: String },
      role: { type: String, default: 'TeamMember' },
      fcmToken: { type: String },
      fcmTokens: [
        {
          token: String,
          deviceId: String,
          platform: String,
          updatedAt: Date,
        },
      ],
      lastFCMUpdate: { type: Date },
    },
    {
      strict: false,
      collection: 'GlobalUsers',
    }
  );

  GlobalUserSchema.index({ email: 1 }, { unique: true });
  GlobalUserSchema.index({ employeeNumber: 1 });
  GlobalUserSchema.index({ email: 1, employeeNumber: 1 });

  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [BulkDataUploader] Created GlobalUser model');
}

const EquipmentSchema = new mongoose.Schema(
  {
    id: String,
    equipmentName: String,
    mainCategory: String,
    equipmentCategory: String,
    equipmentType: String,
    status: String,
    country: String,
    project: String,
    firebaseKey: String,
    make: String,
    model: String,
    year: String,
    serialNumber: String,
    location: String,
    description: String,
    specifications: mongoose.Schema.Types.Mixed,
    customName: String,
    databaseName: String,
    meterReading: mongoose.Schema.Types.Mixed,
    meterUnit: String,
    lastKnownCoordinates: String,
    gpsCoordinates: String,
    lastKnownLocation: String,
    gpsAddress: String,
    locationUpdatedAt: Date,
    lastInspectionCountry: String,
    lastInspectionProject: String,
    lastInspectionAt: Date,
    lastInspectionRequestNumber: String,
    uploadedAt: { type: Date, default: Date.now },
    uploadedFrom: { type: String, default: 'Bulk Import API' },
    mongoUploadVersion: { type: String, default: 'bulk-import-1.0.0' },
    firebaseData: mongoose.Schema.Types.Mixed,
  },
  {
    strict: false,
    timestamps: true,
  }
);

let Equipment;
try {
  Equipment = mongoose.model('Equipment');
  console.log('✅ [BulkDataUploader] Reusing Equipment model');
} catch (error) {
  Equipment = mongoose.model('Equipment', EquipmentSchema);
  console.log('✅ [BulkDataUploader] Created Equipment model');
}

// ===== Helpers =====
const parseDate = (value) => {
  if (!value) return null;
  const date = new Date(value);
  return Number.isNaN(date.getTime()) ? null : date;
};

const ensureArray = (value) => {
  if (!value) return [];
  return Array.isArray(value) ? value : [value].filter(Boolean);
};

const sanitizeNumber = (value) => {
  if (value === null || value === undefined || value === '') return null;
  const num = Number(value);
  return Number.isNaN(num) ? null : num;
};

const DEFAULT_DATA_DIR = path.resolve(__dirname, 'AAdataa');
const FALLBACK_DATA_DIR = path.resolve(
  __dirname,
  '../components/MainHome/AAdataa'
);

const buildDataPath = (envValue, fileName) => {
  if (envValue) return envValue;
  return path.join(DEFAULT_DATA_DIR, fileName);
};

const ADMIN_JSON_PATH = buildDataPath(
  process.env.ADMIN_EXPORT_JSON_PATH,
  'titan-drilling-1f8e9-default-rtdb-admins-export.json'
);

const EQUIPMENT_JSON_PATH = buildDataPath(
  process.env.EQUIPMENT_EXPORT_JSON_PATH,
  'titan-drilling-1f8e9-default-rtdb-GlobalEquipmentJsonData-export.json'
);

const resolveWithFallback = async (primaryPath, fallbackFileName) => {
  try {
    await fs.access(primaryPath);
    return primaryPath;
  } catch {
    const fallbackPath = path.join(FALLBACK_DATA_DIR, fallbackFileName);
    await fs.access(fallbackPath);
    return fallbackPath;
  }
};

const normalizeUserData = async (firebaseId, rawUser) => {
  const userId = (rawUser.uid || firebaseId || '').trim();
  const email = rawUser.email ? rawUser.email.toLowerCase().trim() : null;
  const employeeNumber = rawUser.employeeNumber
    ? String(rawUser.employeeNumber).trim()
    : null;

  if (!userId || !email) {
    throw new Error(
      `Missing required fields for user. FirebaseID: ${firebaseId}, UID: ${userId}, Email: ${email}`
    );
  }

  const password = employeeNumber || rawUser.password || 'password';
  const hashedPassword = await bcrypt.hash(password, 10);

  const sanitizedUser = { ...rawUser };

  const dateFields = [
    'lastLogin',
    'uploadTimestamp',
    'engagementDate',
    'titanInductionDate',
    'clientDrivingDate',
    'dateExamined',
    'expiryMedicals',
    'driverLicenseExpiry',
    'dateCompletedFirstAid',
    'expiryDateFirstAid',
    'createdAt',
    'updatedAt',
    'date_of_birth',
  ];

  dateFields.forEach((field) => {
    if (sanitizedUser[field]) {
      sanitizedUser[field] = parseDate(sanitizedUser[field]);
    }
  });

  const arrayFields = [
    'countries',
    'modules',
    'projects',
    'medicalDocs',
    'driverLicenseDocs',
    'defensiveDrivingDocs',
    'documents',
    'certifications',
  ];

  arrayFields.forEach((field) => {
    sanitizedUser[field] = ensureArray(sanitizedUser[field]);
  });

  return {
    cleanedData: {
      ...sanitizedUser,
      uid: userId,
      name: sanitizedUser.name || sanitizedUser.fullName || email.split('@')[0],
      email,
      employeeNumber,
      loginStatus: Boolean(sanitizedUser.loginStatus),
      authCreated: Boolean(sanitizedUser.authCreated),
      notifications: sanitizedUser.notifications || {},
      countries: sanitizedUser.countries,
      modules: sanitizedUser.modules,
      projects: sanitizedUser.projects || [],
      medicalDocs: sanitizedUser.medicalDocs || [],
      driverLicenseDocs: sanitizedUser.driverLicenseDocs || [],
      defensiveDrivingDocs: sanitizedUser.defensiveDrivingDocs || [],
      uploadTimestamp: new Date(),
      originalUserId: sanitizedUser.originalUserId || userId,
      password,
      hashedPassword,
      role: sanitizedUser.role || 'TeamMember',
    },
    userId,
    email,
    employeeNumber,
  };
};

const normalizeEquipmentData = (rawEquipment) => {
  const equipmentId =
    (rawEquipment.id || rawEquipment.equipmentNumber || rawEquipment.firebaseKey || '').toString().trim();

  if (!equipmentId) {
    throw new Error('Equipment item missing identifier');
  }

  const sanitizedEquipment = { ...rawEquipment };

  const dateFields = [
    'createdAt',
    'updatedAt',
    'uploadedAt',
    'lastInspectionAt',
    'locationUpdatedAt',
    'lastServiceDate',
    'purchaseDate',
    'warrantyExpDate',
  ];

  dateFields.forEach((field) => {
    if (sanitizedEquipment[field]) {
      sanitizedEquipment[field] = parseDate(sanitizedEquipment[field]);
    }
  });

  sanitizedEquipment.meterReading = sanitizeNumber(sanitizedEquipment.meterReading);

  return {
    cleanedData: {
      ...sanitizedEquipment,
      id: equipmentId,
      equipmentName: sanitizedEquipment.equipmentName || sanitizedEquipment.mainCategory || 'Equipment',
      equipmentNumber: sanitizedEquipment.equipmentNumber || equipmentId,
      mainCategory: sanitizedEquipment.mainCategory || sanitizedEquipment.equipmentCategory || '',
      uploadedAt: new Date(),
      uploadedFrom: 'Bulk Import API',
      mongoUploadVersion: 'bulk-import-1.0.0',
      databaseName: sanitizedEquipment.databaseName || 'titan_drilling',
    },
    equipmentId,
  };
};

// ===== Routes =====
app.get('/', async (req, res) => {
  let detectedAdminPath = ADMIN_JSON_PATH;
  let detectedEquipmentPath = EQUIPMENT_JSON_PATH;

  try {
    detectedAdminPath = await resolveWithFallback(
      ADMIN_JSON_PATH,
      'titan-drilling-1f8e9-default-rtdb-admins-export.json'
    );
  } catch (error) {
    detectedAdminPath = `${ADMIN_JSON_PATH} (missing)`;
  }

  try {
    detectedEquipmentPath = await resolveWithFallback(
      EQUIPMENT_JSON_PATH,
      'titan-drilling-1f8e9-default-rtdb-GlobalEquipmentJsonData-export.json'
    );
  } catch (error) {
    detectedEquipmentPath = `${EQUIPMENT_JSON_PATH} (missing)`;
  }

  res.json({
    status: 'Bulk Data Uploader running',
    port: PORT,
    adminJsonPath: detectedAdminPath,
    equipmentJsonPath: detectedEquipmentPath,
  });
});

const importUsers = async () => {
  const usersPath = await resolveWithFallback(
    ADMIN_JSON_PATH,
    'titan-drilling-1f8e9-default-rtdb-admins-export.json'
  );
  console.log(`📥 Reading users JSON from ${usersPath}`);
  const usersFileContent = await fs.readFile(usersPath, 'utf-8');
  const usersJson = JSON.parse(usersFileContent);
  const userEntries = Object.entries(usersJson);

  const summary = {
    users: {
      total: userEntries.length,
      created: 0,
      updated: 0,
      skipped: 0,
      errors: [],
    },
  };

  console.log(`👥 Processing ${userEntries.length} users...`);

  for (const [firebaseId, rawUser] of userEntries) {
    try {
      const { cleanedData, userId, email, employeeNumber } = await normalizeUserData(
        firebaseId,
        rawUser
      );

      const filters = [];
      if (userId) filters.push({ _id: userId });
      if (email) filters.push({ email });
      if (employeeNumber) filters.push({ employeeNumber });

      const existingUser = await GlobalUser.findOne(filters.length ? { $or: filters } : { _id: userId });

      if (existingUser) {
        await GlobalUser.findByIdAndUpdate(existingUser._id, cleanedData, {
          new: true,
          runValidators: false,
        });
        summary.users.updated += 1;
      } else {
        await GlobalUser.create({
          ...cleanedData,
          _id: userId,
        });
        summary.users.created += 1;
      }
    } catch (userError) {
      console.error(`❌ User import error (${firebaseId}):`, userError.message);
      summary.users.errors.push({
        firebaseId,
        message: userError.message,
      });
      summary.users.skipped += 1;
    }
  }

  return { summary, filePath: usersPath };
};

const importEquipment = async () => {
  const equipmentPath = await resolveWithFallback(
    EQUIPMENT_JSON_PATH,
    'titan-drilling-1f8e9-default-rtdb-GlobalEquipmentJsonData-export.json'
  );
  console.log(`📥 Reading equipment JSON from ${equipmentPath}`);
  const equipmentFileContent = await fs.readFile(equipmentPath, 'utf-8');
  const equipmentJson = JSON.parse(equipmentFileContent);

  if (!Array.isArray(equipmentJson)) {
    throw new Error('Equipment JSON must be an array');
  }

  const summary = {
    equipment: {
      total: equipmentJson.length,
      created: 0,
      updated: 0,
      skipped: 0,
      errors: [],
    },
  };

  console.log(`🛠️ Processing ${equipmentJson.length} equipment items...`);

  for (const rawEquipment of equipmentJson) {
    try {
      const { cleanedData, equipmentId } = normalizeEquipmentData(rawEquipment);
      const existingEquipment = await Equipment.findOne({ id: equipmentId });

      if (existingEquipment) {
        await Equipment.findByIdAndUpdate(existingEquipment._id, cleanedData, {
          new: true,
          runValidators: false,
        });
        summary.equipment.updated += 1;
      } else {
        await Equipment.create(cleanedData);
        summary.equipment.created += 1;
      }
    } catch (equipmentError) {
      console.error(`❌ Equipment import error:`, equipmentError.message);
      summary.equipment.errors.push({
        equipmentId: rawEquipment?.id || rawEquipment?.equipmentNumber || null,
        message: equipmentError.message,
      });
      summary.equipment.skipped += 1;
    }
  }

  return { summary, filePath: equipmentPath };
};

const combineSummaries = (base, addition) => {
  const combined = { ...base };

  Object.entries(addition).forEach(([key, value]) => {
    if (!combined[key]) {
      combined[key] = value;
      return;
    }

    combined[key] = {
      total: (combined[key].total || 0) + (value.total || 0),
      created: (combined[key].created || 0) + (value.created || 0),
      updated: (combined[key].updated || 0) + (value.updated || 0),
      skipped: (combined[key].skipped || 0) + (value.skipped || 0),
      errors: [...(combined[key].errors || []), ...(value.errors || [])],
    };
  });

  return combined;
};

app.post('/upload-all', async (req, res) => {
  const startedAt = Date.now();
  console.log('🚀 Bulk data upload requested (all)');

  const resultSummary = {};
  const filePaths = {};

  try {
    const usersResult = await importUsers();
    filePaths.users = usersResult.filePath;
    Object.assign(
      resultSummary,
      combineSummaries(resultSummary, usersResult.summary)
    );

    const equipmentResult = await importEquipment();
    filePaths.equipment = equipmentResult.filePath;
    Object.assign(
      resultSummary,
      combineSummaries(resultSummary, equipmentResult.summary)
    );

    const durationMs = Date.now() - startedAt;
    console.log(
      `✅ Bulk upload finished in ${(durationMs / 1000).toFixed(2)}s | Users created: ${
        resultSummary.users.created
      } updated: ${resultSummary.users.updated} | Equipment created: ${
        resultSummary.equipment.created
      } updated: ${resultSummary.equipment.updated}`
    );

    res.json({
      success: true,
      durationMs,
      summary: resultSummary,
      filePaths,
      timestamp: new Date().toISOString(),
    });
  } catch (error) {
    console.error('❌ Bulk upload failed:', error);
    res.status(500).json({
      success: false,
      error: error.message,
      summary: resultSummary,
      filePaths,
      timestamp: new Date().toISOString(),
    });
  }
});

app.post('/upload/users', async (req, res) => {
  const startedAt = Date.now();
  console.log('🚀 Users bulk upload requested');

  try {
    const { summary, filePath } = await importUsers();
    const durationMs = Date.now() - startedAt;

    res.json({
      success: true,
      durationMs,
      summary,
      filePath,
      timestamp: new Date().toISOString(),
    });
  } catch (error) {
    console.error('❌ Users bulk upload failed:', error);
    res.status(500).json({
      success: false,
      error: error.message,
      timestamp: new Date().toISOString(),
    });
  }
});

app.post('/upload/equipment', async (req, res) => {
  const startedAt = Date.now();
  console.log('🚀 Equipment bulk upload requested');

  try {
    const { summary, filePath } = await importEquipment();
    const durationMs = Date.now() - startedAt;

    res.json({
      success: true,
      durationMs,
      summary,
      filePath,
      timestamp: new Date().toISOString(),
    });
  } catch (error) {
    console.error('❌ Equipment bulk upload failed:', error);
    res.status(500).json({
      success: false,
      error: error.message,
      timestamp: new Date().toISOString(),
    });
  }
});

app.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 Bulk Data Uploader running on port ${PORT}`);
  console.log(`📡 Endpoint: https://api.titandrillingzm.com:${PORT}/upload-all`);
  console.log(`📁 Users JSON: ${ADMIN_JSON_PATH}`);
  console.log(`📁 Equipment JSON: ${EQUIPMENT_JSON_PATH}`);
});

module.exports = app;

