
import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Modal,
  ScrollView,
  Platform,
  Image,
  ActivityIndicator,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import DateTimePicker from '@react-native-community/datetimepicker';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import { Linking } from 'react-native';


const { width, height } = Dimensions.get('window');

const HSE41_42_step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [rig, setRig] = useState('');
  const [location, setLocation] = useState('');
  const [date, setDate] = useState(new Date());
  const [supervisor, setSupervisor] = useState('');
  const [selectedRole, setSelectedRole] = useState(null);
  const [showDatePicker, setShowDatePicker] = useState(false);

  const [supervisorSignature, setSupervisorSignature] = useState('');
  const [managerSignature, setManagerSignature] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);

  const roleOptions = [
    { label: 'Site Supervisor', value: 'supervisor' },
    { label: 'Site Manager', value: 'manager' },
    { label: 'Both', value: 'both' },
  ];

  const handleBackPress = () => {
    setActiveStep(2);
  };

  // Load equipment data from step 1
  useEffect(() => {
    const loadEquipmentData = async () => {
      try {
        const savedData = await AsyncStorage.getItem('hseStep1');
        if (savedData) {
          const data = JSON.parse(savedData);
          if (data.selectedEquipment) {
            setSelectedEquipment(data.selectedEquipment);
            // Auto-fill rig with equipment registration number
            if (data.selectedEquipment.regNum) {
              setRig(data.selectedEquipment.regNum);
            }
          }
        }
      } catch (error) {
        console.error('Error loading equipment data:', error);
      }
    };
    loadEquipmentData();
  }, []);

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setDate(selectedDate);
    }
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  const isValidSignatureUri = (uri) => uri && typeof uri === 'string' && uri.startsWith('data:image');

  const handleSignaturePress = (type) => {
    navigation.navigate('HSE41_42_signature', {
      role: type === 'supervisor' ? 'Supervisor' : 'Manager',
      onSave: (signature) => {
        if (type === 'supervisor') {
          setSupervisorSignature(signature);
        } else if (type === 'manager') {
          setManagerSignature(signature);
        }
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };



  const handleClearSignature = (type) => {
    if (type === 'supervisor') {
      setSupervisorSignature(null);
    } else if (type === 'manager') {
      setManagerSignature(null);
    }
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = (type) => {
    console.error(`${type} signature image failed to load`);
    Toast.show({ type: 'error', text1: 'Error', text2: `Failed to load ${type.toLowerCase()} signature. Please try again.`, position: 'bottom' });
    handleClearSignature(type);
  };

  const generatePDFContent = async () => {
    // Load data from AsyncStorage
    const savedHse1 = await AsyncStorage.getItem('hseStep1');
    const savedHse2 = await AsyncStorage.getItem('hseStep2');
    
    const step1Data = savedHse1 ? JSON.parse(savedHse1) : {};
    const step2Data = savedHse2 ? JSON.parse(savedHse2) : {};
    
    const checklistTitle = step1Data.selectedChecklist?.document_name || 'HSE Inspection Report';
    const equipmentName = step1Data.selectedEquipment?.equipmentName || 'N/A';
    const equipmentNumber = step1Data.selectedEquipment?.equipmentNumber || 'N/A';
    const country = step1Data.country || 'N/A';
    const project = step1Data.project || 'N/A';
    const coordinates = step1Data.coordinates || 'N/A';
    const gpsAddress = step1Data.gpsAddress || 'N/A';
    
    const assessmentData = step2Data.assessmentData || {};
    const sections = step2Data.sections || [];
    const notes = step2Data.notes || {};
    
    // Also try to load notes from the hseResults if available
    let additionalNotes = {};
    try {
      const savedHse1 = await AsyncStorage.getItem('hseStep1');
      if (savedHse1) {
        const step1Data = JSON.parse(savedHse1);
        const hseId = step1Data.id;
        if (hseId) {
          const savedNotes = await AsyncStorage.getItem(`hseNotes_${hseId}`);
          if (savedNotes) {
            additionalNotes = JSON.parse(savedNotes);
          }
        }
      }
    } catch (error) {
      console.error('Error loading additional notes:', error);
    }
    
    // Merge notes from both sources
    const allNotes = { ...notes, ...additionalNotes };
    
    // Calculate totals
    let totalYes = 0;
    let totalNo = 0;
    let totalNA = 0;
    
    // Generate checklist rows dynamically based on sections
    let checklistRows = '';
    let serialNumber = 1;
    
    if (sections && sections.length > 0) {
      sections.forEach((section) => {
        if (section.subItems && section.subItems.length > 0) {
          // Add section header
          checklistRows += `
            <tr>
              <td colspan="6" class="section-header">${section.name}</td>
            </tr>
          `;
          
          section.subItems.forEach((item) => {
            const selection = assessmentData[item.key] || '';
            const comment = allNotes[item.key] || '';
            
            // Determine which column to mark based on selection
            let notApplicable = '';
            let yes = '';
            let no = '';
            
            if (selection === 'Yes') {
              yes = '✓';
              totalYes++;
            } else if (selection === 'No') {
              no = '✗';
              totalNo++;
            } else if (selection === 'N/A' || selection === 'Not Applicable') {
              notApplicable = '✓';
              totalNA++;
            }
            
            checklistRows += `
              <tr>
                <td style="text-align: center;">${serialNumber}</td>
                <td>${item.itemName}</td>
                <td style="text-align: center;">${yes}</td>
                <td style="text-align: center;">${no}</td>
                <td style="text-align: center;">${notApplicable}</td>
                <td>${comment}</td>
              </tr>
            `;
            serialNumber++;
          });
        }
      });
    } else {
      // Fallback to old format if sections not available
      // Get all items from the checklist data
      const step1Data = savedHse1 ? JSON.parse(savedHse1) : {};
      const checklist = step1Data.selectedChecklist;
      
      if (checklist && checklist.checklist) {
        Object.entries(checklist.checklist).forEach(([sectionKey, sectionValue]) => {
          if (Array.isArray(sectionValue)) {
            // Add section header
            checklistRows += `
              <tr>
                <td colspan="6" class="section-header">${sectionKey}</td>
              </tr>
            `;
            
            sectionValue.forEach((item, index) => {
              if (typeof item === 'object' && item !== null) {
                const itemKey = Object.keys(item)[0];
                const itemName = itemKey;
                const key = `${checklist.document_id}_${sectionKey}_${index}`;
                const selection = assessmentData[key] || '';
                const comment = allNotes[key] || '';
                
                let notApplicable = '';
                let yes = '';
                let no = '';
                
                if (selection === 'Yes') {
                  yes = '✓';
                  totalYes++;
                } else if (selection === 'No') {
                  no = '✗';
                  totalNo++;
                } else if (selection === 'N/A' || selection === 'Not Applicable') {
                  notApplicable = '✓';
                  totalNA++;
                }
                
                checklistRows += `
                  <tr>
                    <td style="text-align: center;">${serialNumber}</td>
                    <td>${itemName}</td>
                    <td style="text-align: center;">${yes}</td>
                    <td style="text-align: center;">${no}</td>
                    <td style="text-align: center;">${notApplicable}</td>
                    <td>${comment}</td>
                  </tr>
                `;
                serialNumber++;
              }
            });
          }
        });
      }
    }

          return `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE Inspection Report</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 0;
              box-sizing: border-box;
            }
            .container {
              width: 100%;
              max-width: 800px;
              margin: 20px auto;
              border: 1px solid #000;
              box-sizing: border-box;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: center;
              background-color: #f0f0f0;
              padding: 10px;
              border-bottom: 1px solid #000;
            }
            .header .logo {
              width: 100px;
              height: 100px;
              
            }
            .header .details table {
              border-collapse: collapse;
              width: 100%;
            }
            .header .details td {
              padding: 2px 0;
              text-align: left;
            }
            .inspection-details {
              display: grid;
              grid-template-columns: 1fr 1fr;
              padding: 5px;
              border-bottom: 1px solid #000;
              gap: 2px;
            }
            .inspection-details div {
              padding: 2px;
            }
            .table {
              width: 100%;
              border-collapse: collapse;
              table-layout: fixed;
            }
            .table th,
            .table td {
              padding: 8px;
              border: 1px solid #000;
              text-align: left;
              word-wrap: break-word;
              overflow-wrap: break-word;
            }
            .table th {
              background-color: #d3d3d3;
            }
            .table .section-header {
              background-color: #a9a9a9;
              font-weight: bold;
            }
            @media (max-width: 600px) {
              .container {
                margin: 10px;
                border: none;
                width: 100%;
                box-sizing: border-box;
              }
              .header {
                flex-direction: column;
                align-items: flex-start;
                padding: 10px;
                text-align: left;
              }
              .header .logo {
                width: 100px;
                height: 100px;
                
              }
               
              .inspection-details {
                grid-template-columns: 1fr;
                padding: 5px;
              }
              .inspection-details div {
                width: 100%;
              }
              .table {
                width: 100%;
                display: block;
                overflow-x: hidden;
              }
              .table th,
              .table td {
                display: block;
                width: 100% !important;
                box-sizing: border-box;
                text-align: left;
              }
              .table tr {
                display: block;
                margin-bottom: 10px;
                border-bottom: 1px solid #000;
              }
              .table th {
                background-color: #d3d3d3;
                font-weight: bold;
              }
              .table .section-header {
                background-color: #a9a9a9;
                font-weight: bold;
                width: 100%;
              }
              .table td:empty {
                display: none;
              }
              .table th:empty {
                display: none;
              }
            }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <div class="Logo"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="200"></div>
              <div class="details">
                <table border="1">
                  <tr><td>Document name :</td><td>${checklistTitle}</td></tr>
                  <tr><td>Document number :</td><td>${step1Data.selectedChecklist?.document_id || 'HSE-043'}</td></tr>
                  <tr><td>version :</td><td>${step1Data.selectedChecklist?.version || '1'}</td></tr>
                  <tr><td>Revision date :</td><td>${step1Data.selectedChecklist?.version_date || new Date().toLocaleDateString()}</td></tr>
                  <tr><td>Approved by :</td><td>${step1Data.selectedChecklist?.approved_by || 'HSM'}</td></tr>
                </table>
              </div>
            </div>
            <div class="inspection-details">
              <div>Rig:</div><div>${rig}</div>
              <div>Location:</div><div>${location}</div>
              <div>Date:</div><div>${date.toLocaleDateString()}</div>
              <div>Supervisor:</div><div>${supervisor}</div>
            </div>
            <table class="table">
              <tr>
               <th>S. No
</th>
                <th>ITEMS TO BE INSPECTED
</th>
               
                <th>Yes</th>
                <th>No</th>
                <th>N/A</th>
                <th>REMARKS</th>
              </tr>
              ${checklistRows}
             
                          </table>
              
              <div style="margin-top: 30px; padding: 20px; border-top: 1px solid #000;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 20px;">
                  <div style="text-align: center; width: 45%;">
                    <div style="border-bottom: 1px solid #000; padding-bottom: 5px; margin-bottom: 10px;">
                      <strong>Site Supervisor</strong>
                    </div>
                    <div style="height: 60px; border: 1px solid #000; margin-bottom: 10px; display: flex; align-items: center; justify-content: center;">
                      ${supervisorSignature && isValidSignatureUri(supervisorSignature) ? 
                        `<img src="${supervisorSignature}" style="max-width: 100%; max-height: 100%;" alt="Supervisor Signature"/>` : 
                        'Signature'
                      }
                    </div>
                    <div style="font-weight: bold;">${supervisor}</div>
                    <div style="font-size: 12px; color: #666;">Date: ${date.toLocaleDateString()}</div>
                  </div>
                  
                  <div style="text-align: center; width: 45%;">
                    <div style="border-bottom: 1px solid #000; padding-bottom: 5px; margin-bottom: 10px;">
                      <strong>Site Manager</strong>
                    </div>
                    <div style="height: 60px; border: 1px solid #000; margin-bottom: 10px; display: flex; align-items: center; justify-content: center;">
                      ${managerSignature && isValidSignatureUri(managerSignature) ? 
                        `<img src="${managerSignature}" style="max-width: 100%; max-height: 100%;" alt="Manager Signature"/>` : 
                        'Signature'
                      }
                    </div>
                    <div style="font-weight: bold;">Manager Name</div>
                    <div style="font-size: 12px; color: #666;">Date: ${date.toLocaleDateString()}</div>
                  </div>
                </div>
              </div>
            </div>
          </body>
          </html>
        `;
  };

  const generateAndSavePDF = async () => {
    try {
      setIsLoading(true);
      
      // Check if required data exists
      const savedHse1 = await AsyncStorage.getItem('hseStep1');
      const savedHse2 = await AsyncStorage.getItem('hseStep2');
      
      if (!savedHse1 || !savedHse2) {
        throw new Error('Required inspection data is missing. Please complete steps 1 and 2 first.');
      }
      
      const htmlContent = await generatePDFContent();
      const options = {
        html: htmlContent,
        fileName: `HSE_Report_${Date.now()}`,
        directory: Platform.OS === 'android' ? 'Documents' : 'DocumentDirectory',
      };
      
      const result = await RNHTMLtoPDF.convert(options);
      if (!result || !result.filePath) {
        throw new Error('Failed to generate PDF file');
      }
      const filePath = result.filePath;
      console.log('PDF generated at:', filePath);

      if (isOnline) {
        // Upload to Firebase Storage
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (!userEmail) {
          throw new Error('User email not found. Please log in again.');
        }
        
        const userSnapshot = await database()
          .ref('/Globalusers/admins')
          .orderByChild('email')
          .equalTo(userEmail)
          .once('value');
        const userData = userSnapshot.val();
        if (!userData) {
          throw new Error('User not found in admin database.');
        }
        const userId = Object.keys(userData)[0];
        
        const storageRef = storage().ref(`HSE_Reports/${userId}/${Date.now()}.pdf`);
        await storageRef.putFile(filePath);
        const downloadUrl = await storageRef.getDownloadURL();
        
        if (!downloadUrl) {
          throw new Error('Failed to get PDF download URL');
        }
        
        // Save to Firebase Database
        const reportData = {
          rig,
          location,
          date: date.toISOString(),
          supervisor,
          selectedRole,
          supervisorSignature: isValidSignatureUri(supervisorSignature) ? supervisorSignature : null,
          managerSignature: isValidSignatureUri(managerSignature) ? managerSignature : null,
          pdfDownloadUrl: downloadUrl,
          createdAt: new Date().toISOString(),
          userId: userId,
        };
        
        const reportId = Date.now().toString();
        const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
        await dbRef.set(reportData);
        
        console.log('Report saved to database with ID:', reportId);
        
        setPdfDownloadUrl(downloadUrl);
        
        // Clean up local file
        await RNFS.unlink(filePath).catch(err => console.warn('Failed to delete local PDF:', err));
        
        return downloadUrl;
      } else {
        // Save locally for offline sync
        const offlineData = {
          rig,
          location,
          date: date.toISOString(),
          supervisor,
          selectedRole,
          supervisorSignature: isValidSignatureUri(supervisorSignature) ? supervisorSignature : null,
          managerSignature: isValidSignatureUri(managerSignature) ? managerSignature : null,
          pdfPath: filePath,
          createdAt: new Date().toISOString(),
        };
        
        const offlineReports = JSON.parse(await AsyncStorage.getItem('offlineHSEReports') || '[]');
        offlineReports.push(offlineData);
        await AsyncStorage.setItem('offlineHSEReports', JSON.stringify(offlineReports));
        
        return null;
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const handleGenerateReport = async () => {
    if (!rig.trim() || !location.trim() || !supervisor.trim() || !selectedRole) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please fill all required fields',
        position: 'bottom',
      });
      return;
    }

    if (selectedRole === 'supervisor' && !supervisorSignature) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please add supervisor signature',
        position: 'bottom',
      });
      return;
    }

    if (selectedRole === 'manager' && !managerSignature) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please add manager signature',
        position: 'bottom',
      });
      return;
    }

    if (selectedRole === 'both' && (!supervisorSignature || !managerSignature)) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please add both signatures',
        position: 'bottom',
      });
      return;
    }

    Alert.alert(
      'Generate Report',
      'Are you sure you want to generate the HSE inspection report?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Generate',
          style: 'default',
          onPress: async () => {
            try {
              setIsLoading(true);
              
              // Save all data to AsyncStorage
              const savedHse = await AsyncStorage.getItem('hseStep1');
              let hseData = savedHse ? JSON.parse(savedHse) : {};
              hseData.rig = rig.trim();
              hseData.location = location.trim();
              hseData.date = date.toISOString();
              hseData.supervisor = supervisor.trim();
              hseData.selectedRole = selectedRole;
              hseData.supervisorSignature = supervisorSignature;
              hseData.managerSignature = managerSignature;
              await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
              
              // Generate PDF and save to database
              const pdfUrl = await generateAndSavePDF();
              
              Toast.show({
                type: 'success',
                text1: 'Success',
                text2: 'HSE inspection report generated and saved successfully!',
                position: 'bottom',
              });
              
              // Show success modal with PDF view option
              if (pdfUrl) {
                setShowSuccessModal(true);
              } else {
                // Clear stored data after successful generation
                setTimeout(async () => {
                  try {
                    await AsyncStorage.removeItem('hseStep1');
                    await AsyncStorage.removeItem('hseStep2');
                    navigation.navigate('HomeScreen');
                  } catch (error) {
                    console.error('Error clearing data:', error);
                    navigation.navigate('HomeScreen');
                  }
                }, 2000);
              }
            } catch (error) {
              console.error('Error generating report:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: error.message || 'Failed to generate report.',
                position: 'bottom',
              });
            } finally {
              setIsLoading(false);
            }
          },
        },
      ]
    );
  };

  const handleViewPdf = () => {
    if (pdfDownloadUrl) {
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfDownloadUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
        }
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not available for viewing.',
        position: 'bottom',
      });
    }
  };

  const handleBackToHome = () => {
    setShowSuccessModal(false);
    // Clear stored data
    AsyncStorage.removeItem('hseStep1');
    AsyncStorage.removeItem('hseStep2');
    navigation.navigate('HomeScreen');
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        <View style={styles.headingContainer}>
          <MaterialIcons name="assignment" size={32} color="#015185" />
          <Text style={styles.headingText}>HSE Inspection Report</Text>
          <Text style={styles.subheadingText}>Enter inspection details and signatures</Text>
        </View>

        {/* Basic Information Section */}
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Rig *</Text>
          <TextInput
            style={styles.textInput}
            value={rig}
            onChangeText={setRig}
            placeholder="Enter registration number"
            placeholderTextColor="#888"
          />
        </View>

        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Location *</Text>
          <TextInput
            style={styles.textInput}
            value={location}
            onChangeText={setLocation}
            placeholder="Enter location"
            placeholderTextColor="#888"
          />
        </View>

        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Date *</Text>
          <TouchableOpacity style={styles.pickerButton} onPress={showDatePickerModal}>
            <Text style={styles.pickerButtonText}>{date.toLocaleDateString()}</Text>
            <MaterialIcons name="calendar-today" size={20} color="#015185" />
          </TouchableOpacity>
        </View>

        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Supervisor *</Text>
          <TextInput
            style={styles.textInput}
            value={supervisor}
            onChangeText={setSupervisor}
            placeholder="Enter supervisor name"
            placeholderTextColor="#888"
          />
        </View>

        {/* Role Selection Section */}
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Who are you? *</Text>
          <Dropdown
            style={styles.dropdown}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            data={roleOptions}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select your role"
            value={selectedRole}
            onChange={item => setSelectedRole(item.value)}
          />
        </View>

        {/* Signature Section */}
        {selectedRole && (
          <View style={styles.dynamicSection}>
            <Text style={styles.sectionTitle}>Signatures</Text>
            
            {(selectedRole === 'supervisor' || selectedRole === 'both') && (
              <View style={styles.dynamicFieldContainer}>
                <View style={styles.fieldHeader}>
                  <Text style={styles.fieldNumber}>Supervisor Signature</Text>
                </View>
                <View style={styles.inputContainer}>
                  <Text style={styles.inputLabel}>Supervisor Name</Text>
                  <TextInput
                    style={styles.textInput}
                    placeholder="Enter supervisor name"
                    placeholderTextColor="#888"
                    value={supervisor}
                    onChangeText={setSupervisor}
                  />
                </View>
                <View style={styles.signatureBoxImage}>
                  {isValidSignatureUri(supervisorSignature) ? (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('supervisor')}>
                      <Image
                        source={{ uri: supervisorSignature }}
                        style={styles.signatureImage}
                        resizeMode="contain"
                        onError={() => handleImageError('Supervisor')}
                      />
                    </TouchableOpacity>
                  ) : (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('supervisor')}>
                      <View style={styles.plusCircle}>
                        <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                      </View>
                    </TouchableOpacity>
                  )}
                </View>
              </View>
            )}

            {(selectedRole === 'manager' || selectedRole === 'both') && (
              <View style={styles.dynamicFieldContainer}>
                <View style={styles.fieldHeader}>
                  <Text style={styles.fieldNumber}>Manager Signature</Text>
                </View>
                <View style={styles.inputContainer}>
                  <Text style={styles.inputLabel}>Manager Name</Text>
                  <TextInput
                    style={styles.textInput}
                    placeholder="Enter manager name"
                    placeholderTextColor="#888"
                    value={supervisor}
                    onChangeText={setSupervisor}
                  />
                </View>
                <View style={styles.signatureBoxImage}>
                  {isValidSignatureUri(managerSignature) ? (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('manager')}>
                      <Image
                        source={{ uri: managerSignature }}
                        style={styles.signatureImage}
                        resizeMode="contain"
                        onError={() => handleImageError('Manager')}
                      />
                    </TouchableOpacity>
                  ) : (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('manager')}>
                      <View style={styles.plusCircle}>
                        <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                      </View>
                    </TouchableOpacity>
                  )}
                </View>
              </View>
            )}
          </View>
        )}

        <View style={styles.infoSection}>
          <MaterialIcons name="info" size={24} color="#015185" />
          <Text style={styles.infoText}>
            All assessment data has been collected. Fill in the details above and add your signature to generate the final report.
          </Text>
        </View>
      </ScrollView>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={date}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
        />
      )}



      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, isLoading && styles.disabledButton]}
          onPress={handleGenerateReport}
          disabled={isLoading}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              HSE Inspection Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Loading Modal */}
      {isLoading && (
        <Modal visible={isLoading} transparent={true} animationType="fade">
          <View style={styles.loadingModalOverlay}>
            <View style={styles.loadingModalContent}>
              <ActivityIndicator size="large" color="#015185" />
              <Text style={styles.loadingModalText}>Generating PDF...</Text>
            </View>
          </View>
        </Modal>
      )}

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingContainer: {
    alignItems: 'center',
    paddingVertical: 15,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '600',
    color: '#015185',
    marginTop: 5,
  },
  subheadingText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 5,
  },
  inputContainer: {
    marginBottom: width * 0.04,
  },
  inputLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
  },
  pickerButtonText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  dropdown: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    width: '100%',
    alignSelf: 'stretch',
  },
  placeholderStyle: {
    fontSize: 14,
    color: '#888',
  },
  dropdownText: {
    fontSize: 14,
    color: '#333',
  },
  dynamicSection: {
    marginTop: 20,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  dynamicFieldContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  fieldHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  fieldNumber: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
    marginVertical: width * 0.025,
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  infoSection: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f4fd',
    padding: 15,
    borderRadius: 8,
    marginTop: 20,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: '#015185',
  },
  infoText: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  loadingModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  loadingModalText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 10,
    fontWeight: '600',
  },
});

export default HSE41_42_step3;