<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class userPageController extends Controller
{
    protected $firebaseUrl;

    public function __construct()
    {
        // Updated URL to match your Firebase path
        $this->firebaseUrl = 'https://titan-drilling-1881a-default-rtdb.firebaseio.com/';
    }

    /**
     * Fetch users from Firebase and pass to Blade with pagination
     */
    public function index(Request $request)
    {
        $allUsers = $this->fetchUsersData();
        
        // Get current page from request, default to 1
        $currentPage = $request->get('page', 1);
        $perPage = 10;
        
        // Calculate pagination
        $totalUsers = count($allUsers);
        $totalPages = ceil($totalUsers / $perPage);
        $offset = ($currentPage - 1) * $perPage;
        
        // Get users for current page
        $users = array_slice($allUsers, $offset, $perPage);
        
        // Create pagination data
        $pagination = [
            'current_page' => $currentPage,
            'per_page' => $perPage,
            'total_users' => $totalUsers,
            'total_pages' => $totalPages,
            'has_previous' => $currentPage > 1,
            'has_next' => $currentPage < $totalPages,
            'previous_page' => $currentPage > 1 ? $currentPage - 1 : null,
            'next_page' => $currentPage < $totalPages ? $currentPage + 1 : null,
        ];
        
        return view('auth.users', compact('users', 'pagination'));
    }

    /**
     * Fetch users data from Firebase similar to inspection controller pattern
     */
    private function fetchUsersData()
    {
        try {
            Log::info('Starting to fetch users data from Firebase...');
            
            // Use the correct Firebase path for users
            $usersResponse = Http::get($this->firebaseUrl . 'Globalusers/admins.json');
            
            Log::info('Firebase response status: ' . $usersResponse->status());
            Log::info('Firebase response headers: ' . json_encode($usersResponse->headers()));
            
            if (!$usersResponse->successful()) {
                Log::error('Firebase request failed with status: ' . $usersResponse->status());
                Log::error('Firebase response body: ' . $usersResponse->body());
                return [];
            }
            
            $usersData = $usersResponse->json() ?? [];
            
            Log::info('Raw users data type: ' . gettype($usersData));
            Log::info('Raw users data count: ' . (is_array($usersData) ? count($usersData) : 'Not array'));
            
            if (!is_array($usersData)) {
                Log::error('Users data is not an array: ' . json_encode($usersData));
                return [];
            }

            if (empty($usersData)) {
                Log::warning('Users data is empty');
                return [];
            }

            $processedData = [];
            $processedCount = 0;
            $errorCount = 0;
            
            foreach ($usersData as $userId => $user) {
                try {
                    // Only process users with UUID format (contains letters and numbers, not just numbers)
                    if ($this->isUuidFormat($userId) && is_array($user)) {
                    $processedUser = [
                        'id'                 => $userId,
                            'fullName'           => $this->extractField($user, 'name', 'Unknown'),
                            'department'         => $this->extractField($user, 'department'),
                            'nrcId'              => $this->extractField($user, 'nrcIdNumber'),
                            'engagementDate'     => $this->formatDate($this->extractField($user, 'engagementDate')),
                            'countries'          => $this->extractField($user, 'countries', []),
                            'projects'           => $this->extractField($user, 'projects', []),
                            'modules'            => $this->extractField($user, 'modules', []),
                            'titanInductionDate' => $this->formatDate($this->extractField($user, 'titanInductionDate')),
                            'clientInductionDate'=> $this->formatDate($this->extractField($user, 'clientInduction')),
                            'medicalExamined'    => $this->formatDate($this->extractField($user, 'dateExamined')),
                            'medicalExpiry'      => $this->formatDate($this->extractField($user, 'expiryMedicals')),
                            'dlNumber'           => $this->extractField($user, 'driverLicenseNumber'),
                            'dlCode'             => $this->extractField($user, 'driverLicenseCode'),
                            'dlExpiry'           => $this->formatDate($this->extractField($user, 'expiryMedicals')),
                            'clientDrivingDate'  => $this->formatDate($this->extractField($user, 'clientDrivingDate')),
                            'faDateCompleted'    => $this->formatDate($this->extractField($user, 'dateCompletedFirstAid')),
                            'faLevel'            => $this->extractField($user, 'levelFirstAid'),
                            'faExpiryDate'       => $this->formatDate($this->extractField($user, 'expiryDateFirstAid')),
                    ];

                    $processedData[] = $processedUser;
                        $processedCount++;
                        
                        // Log first few users for debugging
                        if ($processedCount <= 3) {
                            Log::info('Processed UUID user ' . $processedCount . ': ' . json_encode($processedUser));
                            Log::info('Raw user data from Firebase: ' . json_encode($user));
                            
                            // Debug specific fields
                            Log::info('Countries field type: ' . gettype($user['countries'] ?? 'not set') . ' - Value: ' . json_encode($user['countries'] ?? 'not set'));
                            Log::info('Projects field type: ' . gettype($user['projects'] ?? 'not set') . ' - Value: ' . json_encode($user['projects'] ?? 'not set'));
                            Log::info('Modules field type: ' . gettype($user['modules'] ?? 'not set') . ' - Value: ' . json_encode($user['modules'] ?? 'not set'));
                            
                            // Debug processed fields
                            Log::info('Processed countries: ' . json_encode($processedUser['countries']));
                            Log::info('Processed projects: ' . json_encode($processedUser['projects']));
                            Log::info('Processed modules: ' . json_encode($processedUser['modules']));
                        }
                    } else {
                        if (!is_array($user)) {
                            Log::warning('User is not an array: ' . json_encode($user));
                        } else {
                            Log::info('Skipping non-UUID user: ' . $userId);
                        }
                        $errorCount++;
                    }
                } catch (\Exception $e) {
                    Log::error('Error processing user ' . $userId . ': ' . $e->getMessage());
                    $errorCount++;
                }
            }

            // Sort by full name alphabetically
            usort($processedData, function($a, $b) {
                return strcmp($a['fullName'], $b['fullName']);
            });

            Log::info('Successfully processed ' . $processedCount . ' users');
            if ($errorCount > 0) {
                Log::warning('Encountered ' . $errorCount . ' errors while processing users');
            }
            
            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching users data: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            // Return sample data for testing if Firebase fails
            return [
                [
                    'id' => 'test-user-1',
                    'fullName' => 'John Doe',
                    'department' => 'Engineering',
                    'nrcId' => '123456789',
                    'engagementDate' => '2024-01-15',
                    'countries' => ['Zambia', 'DRC'],
                    'projects' => ['IME', 'Kobold'],
                    'modules' => ['Inspection', 'Maintenance'],
                    'titanInductionDate' => '2024-01-20',
                    'clientInductionDate' => '2024-01-25',
                    'medicalExamined' => '2024-01-10',
                    'medicalExpiry' => '2025-01-10',
                    'dlNumber' => 'DL123456',
                    'dlCode' => 'B',
                    'dlExpiry' => '2025-12-31',
                    'clientDrivingDate' => '2024-02-01',
                    'faDateCompleted' => '2024-01-30',
                    'faLevel' => 'Level 1',
                    'faExpiryDate' => '2025-01-30',
                ],
                [
                    'id' => 'test-user-2',
                    'fullName' => 'Jane Smith',
                    'department' => 'Operations',
                    'nrcId' => '987654321',
                    'engagementDate' => '2024-02-01',
                    'countries' => ['Namibia'],
                    'projects' => ['Kalumbila'],
                    'modules' => ['Vehicles'],
                    'titanInductionDate' => '2024-02-05',
                    'clientInductionDate' => '2024-02-10',
                    'medicalExamined' => '2024-01-25',
                    'medicalExpiry' => '2025-01-25',
                    'dlNumber' => 'DL789012',
                    'dlCode' => 'C',
                    'dlExpiry' => '2026-06-30',
                    'clientDrivingDate' => '2024-02-15',
                    'faDateCompleted' => '2024-02-12',
                    'faLevel' => 'Level 2',
                    'faExpiryDate' => '2025-02-12',
                ]
            ];
        }
    }

    /**
     * Extract field value from user data with fallback
     */
    private function extractField($user, $field, $default = '')
    {
        if (isset($user[$field])) {
            $value = $user[$field];
            
            // For array fields (countries, projects, modules), always return array
            if (in_array($field, ['countries', 'projects', 'modules'])) {
                if (is_array($value)) {
                    // If it's already an array, flatten it to handle nested arrays
                    return $this->flattenArray($value);
                }
                
                if (is_object($value)) {
                    // If it's an object, convert to array
                    $arrayValue = (array) $value;
                    // Flatten nested arrays and filter out empty values
                    return $this->flattenArray($arrayValue);
                }
                
                if (is_string($value) && !empty(trim($value))) {
                    $trimmedValue = trim($value);
                    // If it contains commas, split into array
                    if (strpos($trimmedValue, ',') !== false) {
                        return array_map('trim', explode(',', $trimmedValue));
                    }
                    // If it's a single value, wrap in array
                    return [$trimmedValue];
                }
                
                // Return empty array for array fields if value is null/empty
                return [];
            }
            
            // For non-array fields, return string or default
            if (is_string($value) && !empty(trim($value))) {
                return trim($value);
            }
            
            if (is_array($value)) {
                // If it's an array but not an array field, convert to string
                return implode(', ', $value);
            }
            
            if (is_object($value)) {
                // If it's an object for non-array field, convert to string
                return implode(', ', (array) $value);
            }
        }
        
        // Return N/A for empty fields instead of empty string
        return $default === '' ? 'N/A' : $default;
    }

    /**
     * Check if user ID is in UUID format (contains letters and numbers, not just numbers)
     */
    private function isUuidFormat($userId)
    {
        // UUID format: contains both letters and numbers, not just numbers
        // Examples: "04VX1dmx2wVcojQ4kSNy7zZvt9p2", "-OUE2Gn_DBD_XXS54ARZ"
        // Not UUID: "10084258ZAM00808", "ZAMBIA_GROUP_001"
        
        // Check if it contains letters (not just numbers)
        if (preg_match('/[a-zA-Z]/', $userId)) {
            // Check if it's not just a numeric ID with some letters at the end
            // Skip patterns like "10084258ZAM00808" (numeric + letters + numeric)
            if (preg_match('/^\d+[A-Z]+\d+$/', $userId)) {
                return false; // This is a numeric ID with letters, not UUID
            }
            
            // Skip patterns like "ZAMBIA_GROUP_001" (all caps with underscores)
            if (preg_match('/^[A-Z_]+$/', $userId)) {
                return false; // This is a group name, not UUID
            }
            
            return true; // Contains letters and looks like UUID
        }
        
        return false; // No letters found, not UUID
    }

    /**
     * Format date to DD/MM/YYYY HH:MM format
     */
    private function formatDate($dateString)
    {
        if (empty($dateString)) {
            return 'N/A';
        }
        
        try {
            // Handle different date formats from Firebase
            $date = null;
            
            // Try to parse ISO 8601 format (2025-08-27T11:26:12.534Z)
            if (preg_match('/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}/', $dateString)) {
                $date = \DateTime::createFromFormat('Y-m-d\TH:i:s.u\Z', $dateString);
                if (!$date) {
                    $date = \DateTime::createFromFormat('Y-m-d\TH:i:s\Z', $dateString);
                }
            }
            // Try to parse YYYY-MM-DD format
            elseif (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateString)) {
                $date = \DateTime::createFromFormat('Y-m-d', $dateString);
            }
            // Try to parse DD/MM/YYYY format
            elseif (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $dateString)) {
                $date = \DateTime::createFromFormat('d/m/Y', $dateString);
            }
            // Try to parse MM/DD/YYYY format
            elseif (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $dateString)) {
                $date = \DateTime::createFromFormat('m/d/Y', $dateString);
            }
            
            if ($date) {
                return $date->format('d/m/Y g:i A');
            }
            
            // If all parsing fails, return original string
            return $dateString;
            
        } catch (\Exception $e) {
            Log::warning('Error formatting date: ' . $dateString . ' - ' . $e->getMessage());
            return $dateString;
        }
    }

    /**
     * Flatten nested arrays and filter out empty values
     */
    private function flattenArray($array)
    {
        $result = [];
        foreach ($array as $item) {
            if (is_array($item)) {
                // Recursively flatten nested arrays
                $flattened = $this->flattenArray($item);
                $result = array_merge($result, $flattened);
            } elseif (is_string($item) && !empty(trim($item))) {
                $result[] = trim($item);
            }
        }
        return $result;
    }
}
