<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class WorkorderController extends Controller
{
    private $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';

    public function index()
    {
        $workorderData = $this->fetchWorkorderData();
        
        return view('maintenance.Workorder', compact('workorderData'));
    }

    private function fetchWorkorderData()
    {
        try {
            Log::info('Fetching workorder data from Firebase', ['url' => $this->firebaseUrl]);
            
            $workorderResponse = Http::timeout(30)->get($this->firebaseUrl);
            
            if (!$workorderResponse->successful()) {
                Log::error('Failed to fetch workorder data from Firebase', [
                    'status' => $workorderResponse->status(),
                    'body' => $workorderResponse->body()
                ]);
                return [];
            }

            $workorderData = $workorderResponse->json() ?? [];
            
            if (empty($workorderData)) {
                Log::info('No workorder data found at the specified path');
                return [];
            }

            Log::info('Successfully fetched workorder data', ['count' => count($workorderData)]);

            // Process the data to extract required fields for multiple workorders
            $processedData = [];
            
            foreach ($workorderData as $workorderId => $workorder) {
                if (is_array($workorder)) {
                    $processedWorkorder = [
                        'id' => $workorderId,
                        'woNumber' => $workorder['requestNumber'] ?? $workorder['woNumber'] ?? 'N/A',
                        'createdDate' => $workorder['createdAt'] ?? $workorder['createdDate'] ?? 'N/A',
                        'vehicle' => isset($workorder['equipment']['equipmentName']) ? $workorder['equipment']['equipmentName'] : 
                                   (isset($workorder['vehicle']) ? $workorder['vehicle'] : 'N/A'),
                        'dueDate' => $workorder['dueDate'] ?? 'N/A',
                        'completedDate' => $workorder['completedDate'] ?? 'N/A',
                        'status' => $workorder['status'] ?? 'N/A',
                        'priority' => $workorder['priority'] ?? 'N/A',
                        'assigned' => isset($workorder['assignTo'][0]['fullName']) ? $workorder['assignTo'][0]['fullName'] : 
                                    (isset($workorder['assignedTo']) ? $workorder['assignedTo'] : 'N/A'),
                        'memo' => $workorder['memo'] ?? $workorder['description'] ?? 'N/A'
                    ];
                    
                    $processedData[] = $processedWorkorder;
                }
            }

            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching workorder data: ' . $e->getMessage());
            return [];
        }
    }

    public function testWorkorderData()
    {
        $data = $this->fetchWorkorderData();
        return response()->json([
            'success' => true,
            'count' => count($data),
            'data' => $data,
            'raw_firebase_data' => Http::get($this->firebaseUrl)->json()
        ]);
    }

    public function create()
    {
        // Generate request number
        $requestNumber = $this->generateRequestNumber();
        
        return view('maintenance.CreateNewWorkorderNew', compact('requestNumber'));
    }

    public function store(Request $request)
    {
        try {
            // Validate the request
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'country' => 'required|string',
                'project' => 'required|string',
                'equipment_id' => 'required|string',
                'priority' => 'nullable|string',
                'memo' => 'nullable|string',
                'start_date' => 'nullable|date',
                'due_date' => 'nullable|date',
                'work_order_type' => 'nullable|string',
                'estimated_cost' => 'nullable|numeric',
                'estimated_labor_time' => 'nullable|numeric',
                'link_inspection_id' => 'nullable|string',
                'linked_work_order_id' => 'nullable|string',
                'part_material_cost' => 'nullable|numeric',
                'total_labor_cost' => 'nullable|numeric',
                'additional_cost' => 'nullable|numeric',
                'tax_amount' => 'nullable|numeric',
                'tax_percentage' => 'nullable|numeric',
                'total_wo_cost' => 'nullable|numeric',
                'assign_to' => 'nullable|string',
                'items' => 'nullable|string'
            ]);

            // Generate request number
            $requestNumber = $this->generateRequestNumber($validated['country'], $validated['project']);

            // Prepare work order data
            $workOrderData = [
                'id' => time(),
                'title' => $validated['title'],
                'requestNumber' => $requestNumber,
                'country' => $validated['country'],
                'project' => $validated['project'],
                'equipment_id' => $validated['equipment_id'],
                'priority' => $validated['priority'] ?? 'Medium',
                'memo' => $validated['memo'],
                'start_date' => $validated['start_date'],
                'due_date' => $validated['due_date'],
                'work_order_type' => $validated['work_order_type'] ?? 'General',
                'estimated_cost' => $validated['estimated_cost'],
                'estimated_labor_time' => $validated['estimated_labor_time'],
                'link_inspection_id' => $validated['link_inspection_id'],
                'linked_work_order_id' => $validated['linked_work_order_id'],
                'part_material_cost' => $validated['part_material_cost'],
                'total_labor_cost' => $validated['total_labor_cost'],
                'additional_cost' => $validated['additional_cost'],
                'tax_amount' => $validated['tax_amount'],
                'tax_percentage' => $validated['tax_percentage'],
                'total_wo_cost' => $validated['total_wo_cost'],
                'assign_to' => json_decode($validated['assign_to'] ?? '[]', true),
                'items' => json_decode($validated['items'] ?? '[]', true),
                'status' => 'Initiated',
                'created_at' => now()->toISOString(),
                'created_by' => auth()->id()
            ];

            // Save to Firebase
            $this->saveToFirebase($workOrderData);

            // Generate PDF
            $pdfPath = $this->generatePDF($workOrderData);

            return response()->json([
                'success' => true,
                'message' => 'Work order created successfully',
                'workOrderId' => $workOrderData['id'],
                'requestNumber' => $requestNumber,
                'pdfPath' => $pdfPath
            ]);

        } catch (\Exception $e) {
            Log::error('Error creating work order: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating work order: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        // Get work order details
        $workOrder = $this->getWorkOrderById($id);
        
        if (!$workOrder) {
            abort(404, 'Work order not found');
        }

        return view('maintenance.WorkOrderDetails', compact('workOrder'));
    }

    public function pdf($id)
    {
        // Generate and return PDF
        $workOrder = $this->getWorkOrderById($id);
        
        if (!$workOrder) {
            abort(404, 'Work order not found');
        }

        $pdfPath = $this->generatePDF($workOrder);
        
        return response()->file($pdfPath);
    }

    // API endpoints for modals
    public function getEquipment()
    {
        try {
            // Fetch equipment from your data source
            $equipment = $this->fetchEquipmentData();
            return response()->json($equipment);
        } catch (\Exception $e) {
            Log::error('Error fetching equipment: ' . $e->getMessage());
            return response()->json([]);
        }
    }

    public function getUsers()
    {
        try {
            // Fetch users from your data source
            $users = $this->fetchUsersData();
            return response()->json($users);
        } catch (\Exception $e) {
            Log::error('Error fetching users: ' . $e->getMessage());
            return response()->json([]);
        }
    }

    public function getInspections()
    {
        try {
            // Fetch inspections from your data source
            $inspections = $this->fetchInspectionsData();
            return response()->json($inspections);
        } catch (\Exception $e) {
            Log::error('Error fetching inspections: ' . $e->getMessage());
            return response()->json([]);
        }
    }

    public function getWorkOrders()
    {
        try {
            // Fetch work orders from your data source
            $workOrders = $this->fetchWorkOrdersData();
            return response()->json($workOrders);
        } catch (\Exception $e) {
            Log::error('Error fetching work orders: ' . $e->getMessage());
            return response()->json([]);
        }
    }

    private function generateRequestNumber($country = null, $project = null)
    {
        if ($country && $project) {
            $countryCode = $country === 'Zambia' ? 'ZAM' : ($country === 'DRC' ? 'DRC' : 'XXX');
            
            // Handle special project codes
            $projectCode = $project === 'Koblenz' ? 'KOB-JC' : strtoupper(substr($project, 0, 3));
            
            // Get last request number and increment
            $lastNumber = $this->getLastRequestNumber();
            $nextNumber = $lastNumber + 1;
            
            return "{$countryCode}-{$projectCode}-{$nextNumber}";
        }
        
        return 'WO' . time();
    }

    private function getLastRequestNumber()
    {
        try {
            // Get last request number from Firebase (use same instance as mobile app)
            $lastRequestUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/lastRequestNumber.json';
            $response = Http::timeout(30)->get($lastRequestUrl);
            
            if ($response->successful()) {
                $lastNumberStr = $response->json();
                if ($lastNumberStr && is_string($lastNumberStr)) {
                    // Extract number from format like "ZAM-KOB-JC-1003" or "ZAM-KAL-1003"
                    preg_match('/(\d+)$/', $lastNumberStr, $matches);
                    if (isset($matches[1])) {
                        Log::info('Extracted last request number: ' . $matches[1] . ' from: ' . $lastNumberStr);
                        return (int)$matches[1];
                    }
                }
            }
        } catch (\Exception $e) {
            Log::warning('Could not fetch last request number from Firebase: ' . $e->getMessage());
        }
        
        // Fallback to 0 if no number found
        return 0;
    }

    private function saveToFirebase($workOrderData)
    {
        try {
            // Use the same Firebase instance as mobile app
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';
            
            $response = Http::timeout(30)->post($firebaseUrl, $workOrderData);
            
            if (!$response->successful()) {
                throw new \Exception('Failed to save to Firebase: ' . $response->body());
            }
            
            // Update lastRequestNumber in Firebase
            $this->updateLastRequestNumber($workOrderData['requestNumber']);
            
            Log::info('Work order saved to Firebase successfully');
        } catch (\Exception $e) {
            Log::error('Error saving to Firebase: ' . $e->getMessage());
            throw $e;
        }
    }

    private function updateLastRequestNumber($requestNumber)
    {
        try {
            // Use the same Firebase instance as mobile app
            $lastRequestUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/lastRequestNumber.json';
            $response = Http::timeout(30)->put($lastRequestUrl, $requestNumber);
            
            if (!$response->successful()) {
                Log::warning('Failed to update lastRequestNumber: ' . $response->body());
            } else {
                Log::info('Last request number updated successfully: ' . $requestNumber);
            }
        } catch (\Exception $e) {
            Log::error('Error updating lastRequestNumber: ' . $e->getMessage());
        }
    }

    private function generatePDF($workOrderData)
    {
        try {
            // Generate PDF using your preferred PDF library (e.g., DomPDF, TCPDF)
            // This is a placeholder - implement based on your PDF generation needs
            $pdfPath = storage_path('app/workorders/workorder_' . $workOrderData['id'] . '.pdf');
            
            // Ensure directory exists
            if (!file_exists(dirname($pdfPath))) {
                mkdir(dirname($pdfPath), 0755, true);
            }
            
            // Generate PDF content (implement your PDF generation logic here)
            $html = $this->generatePDFHtml($workOrderData);
            
            // Save PDF (implement your PDF saving logic here)
            file_put_contents($pdfPath, $html);
            
            return $pdfPath;
        } catch (\Exception $e) {
            Log::error('Error generating PDF: ' . $e->getMessage());
            throw $e;
        }
    }

    private function generatePDFHtml($workOrderData)
    {
        // Generate HTML for PDF (implement your PDF template here)
        return view('pdf.workorder', compact('workOrderData'))->render();
    }

    private function getWorkOrderById($id)
    {
        try {
            // Use the same Firebase instance as mobile app
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';
            
            $response = Http::timeout(30)->get($firebaseUrl);
            
            if (!$response->successful()) {
                return null;
            }
            
            $workOrders = $response->json() ?? [];
            return $workOrders[$id] ?? null;
        } catch (\Exception $e) {
            Log::error('Error fetching work order: ' . $e->getMessage());
            return null;
        }
    }

    private function fetchEquipmentData()
    {
        // Implement equipment data fetching
        return [
            [
                'id' => '1',
                'equipmentName' => 'Excavator CAT 320',
                'equipmentNumber' => 'EXC-001',
                'mainCategory' => 'Heavy Equipment'
            ],
            [
                'id' => '2',
                'equipmentName' => 'Bulldozer CAT D6',
                'equipmentNumber' => 'BUL-002',
                'mainCategory' => 'Heavy Equipment'
            ]
        ];
    }

    private function fetchUsersData()
    {
        // Implement users data fetching
        return [
            [
                'id' => '1',
                'name' => 'John Doe',
                'fullName' => 'John Doe',
                'employeeNumber' => 'EMP001',
                'role' => 'Technician'
            ],
            [
                'id' => '2',
                'name' => 'Jane Smith',
                'fullName' => 'Jane Smith',
                'employeeNumber' => 'EMP002',
                'role' => 'Supervisor'
            ]
        ];
    }

    private function fetchInspectionsData()
    {
        // Implement inspections data fetching
        return [
            [
                'id' => '1',
                'requestNumber' => 'INS-001',
                'equipment' => ['equipmentName' => 'Excavator CAT 320'],
                'date' => '2024-01-15',
                'faultyItems' => 2
            ]
        ];
    }

    private function fetchWorkOrdersData()
    {
        // Implement work orders data fetching
        return [
            [
                'id' => '1',
                'title' => 'Maintenance Work Order',
                'requestNumber' => 'WO-001',
                'equipment' => ['equipmentName' => 'Excavator CAT 320']
            ]
        ];
    }
} 