<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\FirebaseStorageService;

class NoticeController extends Controller
{
    protected $firebaseStorage;

    public function __construct(FirebaseStorageService $firebaseStorage)
    {
        $this->firebaseStorage = $firebaseStorage;
    }
    public function index()
    {
        try {
            // Fetch notices from Firebase
            $notices = $this->fetchNoticesFromFirebase();
            
            // Add serial numbers and format data
            foreach ($notices as $index => &$notice) {
                $notice['serial_number'] = $index + 1;
                $notice['time_ago'] = $this->timeAgo($notice['createdAt']);
                $notice['formatted_date'] = $this->formatDate($notice['createdAt']);
                
                // Ensure imageUrl key exists (set to null if not present)
                if (!isset($notice['imageUrl'])) {
                    $notice['imageUrl'] = null;
                }
                
                // Ensure other required keys exist
                if (!isset($notice['title'])) {
                    $notice['title'] = 'Untitled Notice';
                }
                if (!isset($notice['description'])) {
                    $notice['description'] = 'No description available';
                }
                if (!isset($notice['createdAt'])) {
                    $notice['createdAt'] = now()->toISOString();
                }
            }
            
            return view('notice.notice_board', compact('notices'));
        } catch (\Exception $e) {
            Log::error('Error fetching notices: ' . $e->getMessage());
            return view('notice.notice_board', ['notices' => []]);
        }
    }

    public function create()
    {
        return view('notice.add_notice');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120', // 5MB max
        ]);

        try {
            $noticeId = time() . '_' . uniqid();
            $userId = 'admin_user_id'; // This should come from authentication
            
            // Handle image upload if present
            $imageUrl = null;
            if ($request->hasFile('image')) {
                $imageUrl = $this->firebaseStorage->uploadImage(
                    $request->file('image'), 
                    "notices/{$userId}", 
                    $noticeId . '.' . $request->file('image')->getClientOriginalExtension()
                );
            }

            // Prepare notice data
            $noticeData = [
                'noticeId' => $noticeId,
                'title' => $request->title,
                'description' => $request->description,
                'imageUrl' => $imageUrl,
                'createdAt' => now()->toISOString(),
                'createdBy' => $userId,
            ];

            // Save to Firebase
            $this->saveNoticeToFirebase($noticeData, $userId);

            return redirect()->route('notice.index')->with('success', 'Notice created successfully!');
        } catch (\Exception $e) {
            Log::error('Error creating notice: ' . $e->getMessage());
            return back()->with('error', 'Failed to create notice: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        try {
            $notice = $this->fetchNoticeById($id);
            if (!$notice) {
                return redirect()->route('notice.index')->with('error', 'Notice not found');
            }
            
            // Ensure all required keys exist
            if (!isset($notice['imageUrl'])) {
                $notice['imageUrl'] = null;
            }
            if (!isset($notice['title'])) {
                $notice['title'] = 'Untitled Notice';
            }
            if (!isset($notice['description'])) {
                $notice['description'] = 'No description available';
            }
            if (!isset($notice['createdAt'])) {
                $notice['createdAt'] = now()->toISOString();
            }
            
            return view('notice.view_notice', compact('notice'));
        } catch (\Exception $e) {
            Log::error('Error fetching notice: ' . $e->getMessage());
            return redirect()->route('notice.index')->with('error', 'Failed to fetch notice');
        }
    }

    public function destroy($id)
    {
        try {
            $this->deleteNoticeFromFirebase($id);
            return redirect()->route('notice.index')->with('success', 'Notice deleted successfully!');
        } catch (\Exception $e) {
            Log::error('Error deleting notice: ' . $e->getMessage());
            return back()->with('error', 'Failed to delete notice: ' . $e->getMessage());
        }
    }

    private function fetchNoticesFromFirebase()
    {
        try {
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/users.json');
            
            if (!$response->successful()) {
                throw new \Exception('Failed to fetch notices from Firebase');
            }

            $usersData = $response->json() ?? [];
            $allNotices = [];

            // Collect all notices from all users
            foreach ($usersData as $userId => $userData) {
                if (isset($userData['notices'])) {
                    foreach ($userData['notices'] as $noticeId => $notice) {
                        $notice['createdBy'] = $userId;
                        $allNotices[] = $notice;
                    }
                }
            }

            // Sort by creation date (newest first)
            usort($allNotices, function($a, $b) {
                return strtotime($b['createdAt']) - strtotime($a['createdAt']);
            });

            return $allNotices;
        } catch (\Exception $e) {
            Log::error('Error fetching notices from Firebase: ' . $e->getMessage());
            return [];
        }
    }

    private function fetchNoticeById($id)
    {
        try {
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/users.json');
            
            if (!$response->successful()) {
                return null;
            }

            $usersData = $response->json() ?? [];

            // Search for notice in all users
            foreach ($usersData as $userId => $userData) {
                if (isset($userData['notices'][$id])) {
                    $notice = $userData['notices'][$id];
                    $notice['createdBy'] = $userId;
                    return $notice;
                }
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Error fetching notice by ID: ' . $e->getMessage());
            return null;
        }
    }

    private function saveNoticeToFirebase($noticeData, $userId)
    {
        try {
            $response = Http::put(
                "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/users/{$userId}/notices/{$noticeData['noticeId']}.json",
                $noticeData
            );

            if (!$response->successful()) {
                throw new \Exception('Failed to save notice to Firebase');
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Error saving notice to Firebase: ' . $e->getMessage());
            throw $e;
        }
    }

    private function deleteNoticeFromFirebase($noticeId)
    {
        try {
            // First, find which user owns this notice
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/users.json');
            
            if (!$response->successful()) {
                throw new \Exception('Failed to fetch users from Firebase');
            }

            $usersData = $response->json() ?? [];
            $found = false;

            foreach ($usersData as $userId => $userData) {
                if (isset($userData['notices'][$noticeId])) {
                    // Delete the notice
                    $deleteResponse = Http::delete(
                        "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/users/{$userId}/notices/{$noticeId}.json"
                    );

                    if (!$deleteResponse->successful()) {
                        throw new \Exception('Failed to delete notice from Firebase');
                    }

                    $found = true;
                    break;
                }
            }

            if (!$found) {
                throw new \Exception('Notice not found');
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Error deleting notice from Firebase: ' . $e->getMessage());
            throw $e;
        }
    }


    private function timeAgo($isoDate)
    {
        $now = now();
        $date = \Carbon\Carbon::parse($isoDate);
        $diffDays = $now->diffInDays($date);

        if ($diffDays === 0) {
            return 'Today';
        } elseif ($diffDays === 1) {
            return 'Yesterday';
        } else {
            return $diffDays . ' days ago';
        }
    }

    private function formatDate($isoDate)
    {
        $date = \Carbon\Carbon::parse($isoDate);
        return $date->format('M d, Y');
    }
}
