<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class DashboardController extends Controller
{
    // MongoDB API Endpoints
    private $mongoBaseUrl = 'http://168.231.113.219';
    private $inspectionPort = '5004';
    private $jobCardPort = '5005';
    private $maintenanceRequestPort = '5009';
    private $serviceSchedulePort = '5010';
    private $equipmentPort = '5001';
    
    public function index()
    {
        $dashboardData = $this->fetchMongoDBData();
        return view('dashboard.index', compact('dashboardData'));
    }
    
    public function enhanced()
    {
        $dashboardData = $this->fetchMongoDBData();
        return view('dashboard.enhanced', compact('dashboardData'));
    }
    
    public function testMongoDB()
    {
        try {
            $data = $this->fetchMongoDBData();
            return response()->json([
                'success' => true,
                'data' => $data,
                'message' => 'MongoDB connection successful'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'MongoDB connection failed'
            ], 500);
        }
    }
    
    public function testInspectionData()
    {
        try {
            $inspectionUrl = $this->mongoBaseUrl . ':' . $this->inspectionPort . '/inspections/all';
            $inspectionResponse = Http::get($inspectionUrl);
            $responseData = $inspectionResponse->json() ?? [];
            $inspectionData = $responseData['data'] ?? [];
            
            $sampleData = [];
            if (is_array($inspectionData) && !empty($inspectionData)) {
                $sampleData = array_slice($inspectionData, 0, 3, true);
            }
            
            return response()->json([
                'success' => true,
                'source' => 'MongoDB',
                'endpoint' => $inspectionUrl,
                'response_status' => $inspectionResponse->status(),
                'total_records' => is_array($inspectionData) ? count($inspectionData) : 'Not array',
                'sample_data' => $sampleData,
                'categorization_summary' => [
                    'total' => $this->countMongoInspections($inspectionData),
                    'good' => $this->countMongoGoodInspections($inspectionData),
                    'faulty' => $this->countMongoFaultyInspections($inspectionData)
                ],
                'message' => 'MongoDB inspection data retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Failed to retrieve MongoDB inspection data'
            ], 500);
        }
    }
    
    public function testServiceScheduleData()
    {
        try {
            $serviceScheduleUrl = $this->mongoBaseUrl . ':' . $this->serviceSchedulePort . '/service-schedules';
            $serviceScheduleResponse = Http::get($serviceScheduleUrl);
            $responseData = $serviceScheduleResponse->json() ?? [];
            $serviceScheduleData = $responseData['data'] ?? [];
            
            return response()->json([
                'success' => true,
                'source' => 'MongoDB',
                'endpoint' => $serviceScheduleUrl,
                'response_status' => $serviceScheduleResponse->status(),
                'total_records' => is_array($serviceScheduleData) ? count($serviceScheduleData) : 'Not array',
                'sample_keys' => is_array($serviceScheduleData) && !empty($serviceScheduleData) ? array_keys(array_slice($serviceScheduleData, 0, 3, true)) : [],
                'message' => 'MongoDB service schedule data retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Failed to retrieve MongoDB service schedule data'
            ], 500);
        }
    }
    
    public function testMaintenanceRequests()
    {
        try {
            $maintenanceUrl = $this->mongoBaseUrl . ':' . $this->maintenanceRequestPort . '/request-maintenance';
            $maintenanceResponse = Http::get($maintenanceUrl);
            $responseData = $maintenanceResponse->json() ?? [];
            $maintenanceData = $responseData['data'] ?? [];
            
            $requestedCount = $this->countMongoRequestedRequests($maintenanceData);
            $approvedCount = $this->countMongoAcceptedRequests($maintenanceData);
            
            return response()->json([
                'success' => true,
                'source' => 'MongoDB',
                'endpoint' => $maintenanceUrl,
                'response_status' => $maintenanceResponse->status(),
                'total_records' => is_array($maintenanceData) ? count($maintenanceData) : 'Not array',
                'requested_count' => $requestedCount,
                'approved_count' => $approvedCount,
                'message' => 'MongoDB maintenance request data retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Failed to retrieve MongoDB maintenance request data'
            ], 500);
        }
    }
    
    private function fetchMongoDBData()
    {
        try {
            // Fetch data from MongoDB API endpoints
            $inspectionUrl = $this->mongoBaseUrl . ':' . $this->inspectionPort . '/inspections/all';
            $jobCardUrl = $this->mongoBaseUrl . ':' . $this->jobCardPort . '/jobcards/all';
            $maintenanceUrl = $this->mongoBaseUrl . ':' . $this->maintenanceRequestPort . '/request-maintenance';
            $serviceScheduleUrl = $this->mongoBaseUrl . ':' . $this->serviceSchedulePort . '/service-schedules';
            $equipmentUrl = $this->mongoBaseUrl . ':' . $this->equipmentPort . '/EquipmentsDataJSON/GlobalEquipmentJsonData';
            
            // Make HTTP requests with timeout
            $inspectionResponse = Http::timeout(10)->get($inspectionUrl . '?limit=10000');
            $jobCardResponse = Http::timeout(10)->get($jobCardUrl);
            $maintenanceResponse = Http::timeout(10)->get($maintenanceUrl);
            $serviceScheduleResponse = Http::timeout(10)->get($serviceScheduleUrl);
            $equipmentResponse = Http::timeout(10)->get($equipmentUrl);
            
            // Also get inspection count separately as backup
            $inspectionCountUrl = $this->mongoBaseUrl . ':' . $this->inspectionPort . '/inspections/count';
            $inspectionCountResponse = Http::timeout(10)->get($inspectionCountUrl);
            
            // Extract data arrays from MongoDB responses
            $inspectionData = $inspectionResponse->successful() ? ($inspectionResponse->json()['data'] ?? []) : [];
            $jobCardData = $jobCardResponse->successful() ? ($jobCardResponse->json()['data'] ?? []) : [];
            $maintenanceData = $maintenanceResponse->successful() ? ($maintenanceResponse->json()['data'] ?? []) : [];
            $serviceScheduleData = $serviceScheduleResponse->successful() ? ($serviceScheduleResponse->json()['data'] ?? []) : [];
            $equipmentData = $equipmentResponse->successful() ? ($equipmentResponse->json()['data'] ?? []) : [];
            
            // Get inspection count from count endpoint if /all returns empty
            $inspectionCount = 0;
            if (empty($inspectionData) && $inspectionCountResponse->successful()) {
                $countData = $inspectionCountResponse->json();
                $inspectionCount = $countData['total'] ?? $countData['count'] ?? 0;
                Log::info('MongoDB: Using count endpoint, found ' . $inspectionCount . ' inspections');
            }
            
            // Log API responses for debugging
            Log::info('MongoDB API Responses:', [
                'inspectionUrl' => $inspectionUrl,
                'inspectionStatus' => $inspectionResponse->status(),
                'inspectionDataCount' => count($inspectionData),
                'inspectionCountBackup' => $inspectionCount,
                'jobCardCount' => count($jobCardData),
                'maintenanceCount' => count($maintenanceData),
                'scheduleCount' => count($serviceScheduleData),
                'equipmentCount' => count($equipmentData)
            ]);
            
            // Log responses for debugging
            Log::info('MongoDB Inspection Response Status: ' . $inspectionResponse->status());
            Log::info('MongoDB Inspection Data Count: ' . (is_array($inspectionData) ? count($inspectionData) : 'Not array'));
            Log::info('MongoDB Job Card Response Status: ' . $jobCardResponse->status());
            Log::info('MongoDB Job Card Data Count: ' . (is_array($jobCardData) ? count($jobCardData) : 'Not array'));
            Log::info('MongoDB Maintenance Response Status: ' . $maintenanceResponse->status());
            Log::info('MongoDB Maintenance Data Count: ' . (is_array($maintenanceData) ? count($maintenanceData) : 'Not array'));
            Log::info('MongoDB Service Schedule Response Status: ' . $serviceScheduleResponse->status());
            Log::info('MongoDB Service Schedule Data Count: ' . (is_array($serviceScheduleData) ? count($serviceScheduleData) : 'Not array'));
            Log::info('MongoDB Equipment Data Count: ' . (is_array($equipmentData) ? count($equipmentData) : 'Not array'));
            
            // Log sample inspection data structure
            if (is_array($inspectionData) && !empty($inspectionData)) {
                $sampleReport = reset($inspectionData);
                Log::info('MongoDB Sample Inspection Keys: ' . json_encode(array_keys($sampleReport)));
            }
            
            // Process the data to get counts
            $dashboardData = [
                'totalInspections' => !empty($inspectionData) ? $this->countMongoInspections($inspectionData) : $inspectionCount,
                'goodInspections' => $this->countMongoGoodInspections($inspectionData),
                'faultyInspections' => $this->countMongoFaultyInspections($inspectionData),
                'totalIssues' => $this->countMongoTotalIssues($inspectionData, $maintenanceData),
                'resolvedIssues' => $this->countMongoResolvedIssues($maintenanceData),
                'inProgressIssues' => $this->countMongoInProgressIssues($maintenanceData),
                'pendingIssues' => $this->countMongoPendingIssues($maintenanceData),
                'faultyVehicles' => $this->countMongoFaultyVehicles($inspectionData),
                'repairItems' => $this->countMongoRepairItems($inspectionData),
                'replaceItems' => $this->countMongoReplaceItems($inspectionData),
                'totalWorkOrders' => $this->countMongoTotalWorkOrders($jobCardData),
                'completedWorkOrders' => $this->countMongoCompletedWorkOrders($jobCardData),
                'initiatedWorkOrders' => $this->countMongoInitiatedWorkOrders($jobCardData),
                'inProgressWorkOrders' => $this->countMongoInProgressWorkOrders($jobCardData),
                'onHoldWorkOrders' => $this->countMongoOnHoldWorkOrders($jobCardData),
                'reworkWorkOrders' => $this->countMongoReworkWorkOrders($jobCardData),
                'postponedWorkOrders' => $this->countMongoPostponedWorkOrders($jobCardData),
                'dueSoonSchedules' => $this->countMongoDueSoonSchedules($serviceScheduleData),
                'overdueSchedules' => $this->countMongoOverdueSchedules($serviceScheduleData),
                'totalSchedules' => $this->countMongoTotalSchedules($serviceScheduleData),
                'requestedRequests' => $this->countMongoRequestedRequests($maintenanceData),
                'acceptedRequests' => $this->countMongoAcceptedRequests($maintenanceData),
                'resolvedRequests' => $this->countMongoResolvedRequests($maintenanceData),
                'totalVehicles' => $this->countMongoTotalVehicles($equipmentData),
                'activeVehicles' => $this->countMongoActiveVehicles($equipmentData),
                'maintenanceVehicles' => $this->countMongoMaintenanceVehicles($equipmentData),
                'breakdownVehicles' => $this->countMongoBreakdownVehicles($equipmentData),
            ];
            
            return $dashboardData;
            
        } catch (\Exception $e) {
            Log::error('MongoDB connection error: ' . $e->getMessage());
            Log::error('MongoDB error trace: ' . $e->getTraceAsString());
            return $this->getDefaultDashboardData();
        }
    }
    
    // ==================== MONGODB COUNTING METHODS ====================
    
    private function countMongoInspections($inspectionData)
    {
        return is_array($inspectionData) ? count($inspectionData) : 0;
    }
    
    private function countMongoGoodInspections($inspectionData)
    {
        $count = 0;
        if (is_array($inspectionData)) {
            foreach ($inspectionData as $report) {
                if (is_array($report)) {
                    // Check step3.overallCondition field
                    if (isset($report['step3']['overallCondition'])) {
                        $condition = strtolower($report['step3']['overallCondition']);
                        if (in_array($condition, [
                            'good condition', 'good', 'excellent', 'satisfactory', 
                            'passed', 'approved', 'acceptable', 'compliant'
                        ])) {
                            $count++;
                        }
                    }
                    // Also check for zero or low faulty items
                    elseif (isset($report['step3']['faultyItems']) && $report['step3']['faultyItems'] == 0) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoFaultyInspections($inspectionData)
    {
        $count = 0;
        if (is_array($inspectionData)) {
            foreach ($inspectionData as $report) {
                if (is_array($report)) {
                    // Check step3.faultyItems count
                    if (isset($report['step3']['faultyItems']) && $report['step3']['faultyItems'] > 0) {
                        $count++;
                    }
                    // Check step3.overallCondition
                    elseif (isset($report['step3']['overallCondition'])) {
                        $condition = strtolower($report['step3']['overallCondition']);
                        if (in_array($condition, [
                            'faulty condition', 'faulty', 'poor condition', 'poor', 
                            'failed', 'defective', 'unsafe', 'critical', 'needs repair',
                            'maintenance required', 'rejected', 'unsatisfactory'
                        ])) {
                            $count++;
                        }
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoTotalIssues($inspectionData, $maintenanceData)
    {
        // Count all faulty inspections + all maintenance requests
        $faultyCount = $this->countMongoFaultyInspections($inspectionData);
        $maintenanceCount = is_array($maintenanceData) ? count($maintenanceData) : 0;
        return $faultyCount + $maintenanceCount;
    }
    
    private function countMongoResolvedIssues($maintenanceData)
    {
        $count = 0;
        if (is_array($maintenanceData)) {
            foreach ($maintenanceData as $request) {
                if (is_array($request) && isset($request['status'])) {
                    $status = strtolower($request['status']);
                    if (in_array($status, ['resolved', 'completed', 'closed', 'done'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoInProgressIssues($maintenanceData)
    {
        $count = 0;
        if (is_array($maintenanceData)) {
            foreach ($maintenanceData as $request) {
                if (is_array($request) && isset($request['status'])) {
                    $status = strtolower($request['status']);
                    if (in_array($status, ['in progress', 'processing', 'working', 'active'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoPendingIssues($maintenanceData)
    {
        $count = 0;
        if (is_array($maintenanceData)) {
            foreach ($maintenanceData as $request) {
                if (is_array($request) && isset($request['status'])) {
                    $status = strtolower($request['status']);
                    if (in_array($status, ['pending', 'waiting', 'queued', 'submitted'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoFaultyVehicles($inspectionData)
    {
        $faultyVehicles = [];
        if (is_array($inspectionData)) {
            foreach ($inspectionData as $report) {
                if (is_array($report) && isset($report['step3']['faultyItems']) && $report['step3']['faultyItems'] > 0) {
                    // Get unique vehicle/equipment IDs
                    $vehicleId = $report['step1']['selectedEquipment']['id'] ?? null;
                    if ($vehicleId && !in_array($vehicleId, $faultyVehicles)) {
                        $faultyVehicles[] = $vehicleId;
                    }
                }
            }
        }
        return count($faultyVehicles);
    }
    
    private function countMongoRepairItems($inspectionData)
    {
        $count = 0;
        if (is_array($inspectionData)) {
            foreach ($inspectionData as $report) {
                if (is_array($report)) {
                    // Check if corrective action exists with repair status
                    if (isset($report['correctiveActionData']['subItemStatuses'])) {
                        $statuses = $report['correctiveActionData']['subItemStatuses'];
                        if (is_array($statuses)) {
                            foreach ($statuses as $status) {
                                if (isset($status) && strtolower($status) === 'repair') {
                                    $count++;
                                }
                            }
                        }
                    }
                    // Fallback to faulty items count
                    elseif (isset($report['step3']['faultyItems'])) {
                        $count += intval($report['step3']['faultyItems'] / 2); // Estimate: half need repair
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoReplaceItems($inspectionData)
    {
        $count = 0;
        if (is_array($inspectionData)) {
            foreach ($inspectionData as $report) {
                if (is_array($report)) {
                    // Check if corrective action exists with replace status
                    if (isset($report['correctiveActionData']['subItemStatuses'])) {
                        $statuses = $report['correctiveActionData']['subItemStatuses'];
                        if (is_array($statuses)) {
                            foreach ($statuses as $status) {
                                if (isset($status) && strtolower($status) === 'replace') {
                                    $count++;
                                }
                            }
                        }
                    }
                    // Fallback to faulty items count
                    elseif (isset($report['step3']['faultyItems'])) {
                        $count += intval($report['step3']['faultyItems'] / 2); // Estimate: half need replacement
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoTotalWorkOrders($jobCardData)
    {
        return is_array($jobCardData) ? count($jobCardData) : 0;
    }
    
    private function countMongoCompletedWorkOrders($jobCardData)
    {
        $count = 0;
        if (is_array($jobCardData)) {
            foreach ($jobCardData as $jobCard) {
                if (is_array($jobCard) && isset($jobCard['status'])) {
                    $status = strtolower($jobCard['status']);
                    if (in_array($status, ['completed', 'done', 'closed', 'finished'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoInitiatedWorkOrders($jobCardData)
    {
        $count = 0;
        if (is_array($jobCardData)) {
            foreach ($jobCardData as $jobCard) {
                if (is_array($jobCard) && isset($jobCard['status'])) {
                    $status = strtolower($jobCard['status']);
                    if (in_array($status, ['initiated', 'started', 'new', 'created'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoInProgressWorkOrders($jobCardData)
    {
        $count = 0;
        if (is_array($jobCardData)) {
            foreach ($jobCardData as $jobCard) {
                if (is_array($jobCard) && isset($jobCard['status'])) {
                    $status = strtolower($jobCard['status']);
                    if (in_array($status, ['in progress', 'processing', 'working', 'ongoing'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoOnHoldWorkOrders($jobCardData)
    {
        $count = 0;
        if (is_array($jobCardData)) {
            foreach ($jobCardData as $jobCard) {
                if (is_array($jobCard) && isset($jobCard['status'])) {
                    $status = strtolower($jobCard['status']);
                    if (in_array($status, ['on hold', 'paused', 'suspended', 'waiting'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoReworkWorkOrders($jobCardData)
    {
        $count = 0;
        if (is_array($jobCardData)) {
            foreach ($jobCardData as $jobCard) {
                if (is_array($jobCard) && isset($jobCard['status'])) {
                    $status = strtolower($jobCard['status']);
                    if (in_array($status, ['rework', 'redo', 'revision'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoPostponedWorkOrders($jobCardData)
    {
        $count = 0;
        if (is_array($jobCardData)) {
            foreach ($jobCardData as $jobCard) {
                if (is_array($jobCard) && isset($jobCard['status'])) {
                    $status = strtolower($jobCard['status']);
                    if (in_array($status, ['postponed', 'delayed', 'rescheduled'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoDueSoonSchedules($serviceScheduleData)
    {
        $count = 0;
        if (is_array($serviceScheduleData)) {
            $now = time();
            $sevenDaysFromNow = $now + (7 * 24 * 60 * 60);
            
            foreach ($serviceScheduleData as $schedule) {
                if (is_array($schedule) && isset($schedule['scheduledDate'])) {
                    $dueDate = strtotime($schedule['scheduledDate']);
                    if ($dueDate > $now && $dueDate <= $sevenDaysFromNow) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoOverdueSchedules($serviceScheduleData)
    {
        $count = 0;
        if (is_array($serviceScheduleData)) {
            $now = time();
            
            foreach ($serviceScheduleData as $schedule) {
                if (is_array($schedule) && isset($schedule['scheduledDate'])) {
                    $dueDate = strtotime($schedule['scheduledDate']);
                    if ($dueDate < $now) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoTotalSchedules($serviceScheduleData)
    {
        return is_array($serviceScheduleData) ? count($serviceScheduleData) : 0;
    }
    
    private function countMongoRequestedRequests($maintenanceData)
    {
        $count = 0;
        if (is_array($maintenanceData)) {
            foreach ($maintenanceData as $request) {
                if (is_array($request) && isset($request['status'])) {
                    $status = strtolower($request['status']);
                    if (in_array($status, ['requested', 'pending', 'submitted', 'new'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoAcceptedRequests($maintenanceData)
    {
        $count = 0;
        if (is_array($maintenanceData)) {
            foreach ($maintenanceData as $request) {
                if (is_array($request) && isset($request['status'])) {
                    $status = strtolower($request['status']);
                    if (in_array($status, ['accepted', 'approved', 'confirmed'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoResolvedRequests($maintenanceData)
    {
        $count = 0;
        if (is_array($maintenanceData)) {
            foreach ($maintenanceData as $request) {
                if (is_array($request) && isset($request['status'])) {
                    $status = strtolower($request['status']);
                    if (in_array($status, ['resolved', 'completed', 'done', 'closed'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoTotalVehicles($equipmentData)
    {
        return is_array($equipmentData) ? count($equipmentData) : 0;
    }
    
    private function countMongoActiveVehicles($equipmentData)
    {
        $count = 0;
        if (is_array($equipmentData)) {
            foreach ($equipmentData as $equipment) {
                if (is_array($equipment) && isset($equipment['status'])) {
                    $status = strtolower($equipment['status']);
                    if (in_array($status, ['active', 'operational', 'available', 'working'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoMaintenanceVehicles($equipmentData)
    {
        $count = 0;
        if (is_array($equipmentData)) {
            foreach ($equipmentData as $equipment) {
                if (is_array($equipment) && isset($equipment['status'])) {
                    $status = strtolower($equipment['status']);
                    if (in_array($status, ['maintenance', 'under maintenance', 'servicing', 'in maintenance'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function countMongoBreakdownVehicles($equipmentData)
    {
        $count = 0;
        if (is_array($equipmentData)) {
            foreach ($equipmentData as $equipment) {
                if (is_array($equipment) && isset($equipment['status'])) {
                    $status = strtolower($equipment['status']);
                    if (in_array($status, ['breakdown', 'broken', 'out of order', 'failed', 'faulty'])) {
                        $count++;
                    }
                }
            }
        }
        return $count;
    }
    
    private function getDefaultDashboardData()
    {
        return [
            'totalInspections' => 0,
            'goodInspections' => 0,
            'faultyInspections' => 0,
            'totalIssues' => 0,
            'resolvedIssues' => 0,
            'inProgressIssues' => 0,
            'pendingIssues' => 0,
            'faultyVehicles' => 0,
            'repairItems' => 0,
            'replaceItems' => 0,
            'totalWorkOrders' => 0,
            'completedWorkOrders' => 0,
            'initiatedWorkOrders' => 0,
            'inProgressWorkOrders' => 0,
            'onHoldWorkOrders' => 0,
            'reworkWorkOrders' => 0,
            'postponedWorkOrders' => 0,
            'dueSoonSchedules' => 0,
            'overdueSchedules' => 0,
            'totalSchedules' => 0,
            'requestedRequests' => 0,
            'acceptedRequests' => 0,
            'resolvedRequests' => 0,
            'totalVehicles' => 0,
            'activeVehicles' => 0,
            'maintenanceVehicles' => 0,
            'breakdownVehicles' => 0,
        ];
    }
}
