@extends('welcome')

@section('title', 'Notice Board')

@section('content')
<div class="notice-board-container">
    <div class="notice-board-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Notice Board</h1>
                <p>Manage driver inspection reminders and missed inspection reports</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('notice.create') }}" class="btn-create">
                    <i class="fas fa-plus"></i>
                    Create Notice
                </a>
            </div>
        </div>
    </div>
    
    <div class="notice-board-content">
        <div class="content-box">
            <!-- Search Section -->
            <div class="search-section">
                <div class="search-container">
                    <i class="fas fa-search search-icon"></i>
                    <input type="text" class="search-input" placeholder="Search notices..." id="searchInput">
                </div>
            </div>

            <!-- Notices List -->
            <div class="notices-list">
                @if(count($notices) > 0)
                    @php
                        $groupedNotices = [];
                        foreach ($notices as $notice) {
                            $date = \Carbon\Carbon::parse($notice['createdAt'])->format('M d, Y');
                            if (!isset($groupedNotices[$date])) {
                                $groupedNotices[$date] = [];
                            }
                            $groupedNotices[$date][] = $notice;
                        }
                    @endphp

                    @foreach($groupedNotices as $date => $dateNotices)
                        <div class="notice-group">
                            <div class="group-header">
                                <h3>{{ $date }}</h3>
                            </div>
                            
                            @foreach($dateNotices as $notice)
                                <div class="notice-card" data-notice-id="{{ $notice['noticeId'] }}">
                                    <div class="notice-header">
                                        <h4 class="notice-title">{{ $notice['title'] }}</h4>
                                        <div class="notice-actions">
                                            <a href="{{ route('notice.show', $notice['noticeId']) }}" class="btn-view">
                                                <i class="fas fa-eye"></i>
                                                View
                                            </a>
                                            <button class="btn-delete" onclick="deleteNotice('{{ $notice['noticeId'] }}')">
                                                <i class="fas fa-trash"></i>
                                                Delete
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div class="notice-content">
                                        <p class="notice-description">
                                            {{ Str::limit($notice['description'], 150) }}
                                        </p>
                                        
                                        @if(isset($notice['imageUrl']) && $notice['imageUrl'])
                                            <div class="notice-image">
                                                <img src="{{ $notice['imageUrl'] }}" alt="Notice Image" class="notice-img">
                                            </div>
                                        @endif
                                        
                                        <div class="notice-meta">
                                            <span class="notice-time">{{ $notice['time_ago'] }}</span>
                                            <span class="notice-author">By: Admin</span>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @endforeach
                @else
                    <div class="empty-state">
                        <div class="empty-content">
                            <i class="fas fa-exclamation-circle empty-icon"></i>
                            <h3>No Notices</h3>
                            <p>Create your first notice to get started.</p>
                            <a href="{{ route('notice.create') }}" class="btn-create-first">
                                <i class="fas fa-plus"></i>
                                Create Notice
                            </a>
                        </div>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal-overlay" id="deleteModal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Delete Notice</h3>
            <button class="modal-close" onclick="closeDeleteModal()">&times;</button>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to delete this notice? This action cannot be undone.</p>
        </div>
        <div class="modal-footer">
            <button class="btn-cancel" onclick="closeDeleteModal()">Cancel</button>
            <button class="btn-confirm-delete" onclick="confirmDelete()">Delete</button>
        </div>
    </div>
</div>

<style>
/* Notice Board Styles */
.notice-board-container {
    min-height: 100vh;
    background: #f8fafc;
    font-family: 'Inter', sans-serif;
}

.notice-board-header {
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    padding: 2rem 0;
    margin-bottom: 0;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.header-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 2rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.page-title h1 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #ffffff;
    margin: 0 0 0.5rem 0;
    letter-spacing: -0.025em;
}

.page-title p {
    font-size: 1.125rem;
    color: #cbd5e1;
    margin: 0;
    font-weight: 400;
}

.header-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.btn-create {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: #0055a5;
    color: #ffffff;
    border: none;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
}

.btn-create:hover {
    background: #004494;
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.notice-board-content {
    padding: 2rem;
}

.content-box {
    background: white;
    border-radius: 12px;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    overflow: hidden;
}

.search-section {
    padding: 1.5rem;
    border-bottom: 1px solid #e2e8f0;
    background: #f8fafc;
}

.search-container {
    position: relative;
    max-width: 400px;
}

.search-icon {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 0.75rem 1rem 0.75rem 3rem;
    border: 2px solid #e2e8f0;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 500;
    color: #1e293b;
    background: #ffffff;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.notices-list {
    padding: 1.5rem;
}

.notice-group {
    margin-bottom: 2rem;
}

.group-header {
    margin-bottom: 1rem;
    padding-bottom: 0.5rem;
    border-bottom: 2px solid #e2e8f0;
}

.group-header h3 {
    font-size: 1.125rem;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.notice-card {
    background: #ffffff;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 1rem;
    transition: all 0.2s ease;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
}

.notice-card:hover {
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    transform: translateY(-1px);
}

.notice-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1rem;
}

.notice-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1e293b;
    margin: 0;
    flex: 1;
}

.notice-actions {
    display: flex;
    gap: 0.5rem;
    align-items: center;
}

.btn-view, .btn-delete {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.5rem 1rem;
    border: none;
    border-radius: 6px;
    font-size: 0.875rem;
    font-weight: 500;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
}

.btn-view {
    background: #0055a5;
    color: #ffffff;
}

.btn-view:hover {
    background: #004494;
}

.btn-delete {
    background: #dc2626;
    color: #ffffff;
}

.btn-delete:hover {
    background: #b91c1c;
}

.notice-content {
    margin-top: 1rem;
}

.notice-description {
    color: #4b5563;
    line-height: 1.6;
    margin-bottom: 1rem;
}

.notice-image {
    margin: 1rem 0;
}

.notice-img {
    width: 100%;
    max-width: 300px;
    height: 200px;
    object-fit: cover;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.notice-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 1rem;
    border-top: 1px solid #f1f5f9;
    font-size: 0.875rem;
    color: #6b7280;
}

.notice-time {
    font-weight: 500;
}

.notice-author {
    font-style: italic;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-content {
    max-width: 400px;
    margin: 0 auto;
}

.empty-icon {
    font-size: 4rem;
    color: #cbd5e1;
    margin-bottom: 1.5rem;
}

.empty-content h3 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #374151;
    margin: 0 0 0.5rem 0;
}

.empty-content p {
    font-size: 1rem;
    color: #6b7280;
    margin: 0 0 2rem 0;
}

.btn-create-first {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: #0055a5;
    color: #ffffff;
    border: none;
    border-radius: 8px;
    font-size: 1rem;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.2s ease;
}

.btn-create-first:hover {
    background: #004494;
    transform: translateY(-1px);
}

/* Modal Styles */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 400px;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem;
    border-bottom: 1px solid #e2e8f0;
}

.modal-header h3 {
    margin: 0;
    font-size: 1.25rem;
    font-weight: 600;
    color: #1e293b;
}

.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #6b7280;
    cursor: pointer;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: all 0.2s ease;
}

.modal-close:hover {
    background: #f1f5f9;
    color: #374151;
}

.modal-body {
    padding: 1.5rem;
}

.modal-body p {
    margin: 0;
    color: #4b5563;
    line-height: 1.6;
}

.modal-footer {
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
    padding: 1.5rem;
    border-top: 1px solid #e2e8f0;
}

.btn-cancel, .btn-confirm-delete {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel {
    background: #f1f5f9;
    color: #374151;
}

.btn-cancel:hover {
    background: #e2e8f0;
}

.btn-confirm-delete {
    background: #dc2626;
    color: #ffffff;
}

.btn-confirm-delete:hover {
    background: #b91c1c;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .page-title h1 {
        font-size: 2rem;
    }
    
    .notice-board-content {
        padding: 1rem;
    }
    
    .notice-header {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .notice-actions {
        justify-content: center;
    }
    
    .notice-meta {
        flex-direction: column;
        gap: 0.5rem;
        align-items: flex-start;
    }
}
</style>

<script>
let currentNoticeId = null;

// Search functionality
document.getElementById('searchInput').addEventListener('input', function() {
    const searchTerm = this.value.toLowerCase();
    const noticeCards = document.querySelectorAll('.notice-card');
    
    noticeCards.forEach(card => {
        const title = card.querySelector('.notice-title').textContent.toLowerCase();
        const description = card.querySelector('.notice-description').textContent.toLowerCase();
        
        if (title.includes(searchTerm) || description.includes(searchTerm)) {
            card.style.display = 'block';
        } else {
            card.style.display = 'none';
        }
    });
});

// Delete notice functions
function deleteNotice(noticeId) {
    currentNoticeId = noticeId;
    document.getElementById('deleteModal').style.display = 'flex';
}

function closeDeleteModal() {
    document.getElementById('deleteModal').style.display = 'none';
    currentNoticeId = null;
}

function confirmDelete() {
    if (currentNoticeId) {
        // Create a form to submit DELETE request
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = `/notice/${currentNoticeId}`;
        
        // Add CSRF token
        const csrfToken = document.createElement('input');
        csrfToken.type = 'hidden';
        csrfToken.name = '_token';
        csrfToken.value = '{{ csrf_token() }}';
        form.appendChild(csrfToken);
        
        // Add method override for DELETE
        const methodField = document.createElement('input');
        methodField.type = 'hidden';
        methodField.name = '_method';
        methodField.value = 'DELETE';
        form.appendChild(methodField);
        
        document.body.appendChild(form);
        form.submit();
    }
}

// Close modal when clicking outside
document.getElementById('deleteModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeDeleteModal();
    }
});
</script>
@endsection
