@extends('welcome')

@section('title', 'Add Notice')

@section('content')
<div class="add-notice-container">
    <div class="add-notice-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Add Notice</h1>
                <p>Create a new notice for the notice board</p>
            </div>
        </div>
    </div>
    
    <div class="add-notice-content">
        <div class="content-box">
            <form action="{{ route('notice.store') }}" method="POST" enctype="multipart/form-data" id="noticeForm">
                @csrf
                
                <div class="form-section">
                    <div class="form-group">
                        <label for="title" class="form-label">Title *</label>
                        <input type="text" 
                               id="title" 
                               name="title" 
                               class="form-control @error('title') is-invalid @enderror" 
                               value="{{ old('title') }}" 
                               placeholder="Enter notice title"
                               required>
                        @error('title')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group">
                        <label for="description" class="form-label">Description *</label>
                        <textarea id="description" 
                                  name="description" 
                                  class="form-control @error('description') is-invalid @enderror" 
                                  rows="6" 
                                  placeholder="Write your notice description..."
                                  required>{{ old('description') }}</textarea>
                        @error('description')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-group">
                        <label for="image" class="form-label">Image (Optional)</label>
                        <div class="image-upload-container">
                            <input type="file" 
                                   id="image" 
                                   name="image" 
                                   class="form-control @error('image') is-invalid @enderror" 
                                   accept="image/*"
                                   onchange="previewImage(this)">
                            <div class="image-preview" id="imagePreview" style="display: none;">
                                <img id="previewImg" src="" alt="Preview">
                                <button type="button" class="remove-image" onclick="removeImage()">×</button>
                            </div>
                        </div>
                        @error('image')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <small class="form-text">Supported formats: JPEG, PNG, JPG, GIF. Max size: 5MB</small>
                    </div>
                </div>
                
                <div class="form-actions">
                    <a href="{{ route('notice.index') }}" class="btn-cancel">
                        <i class="fas fa-arrow-left"></i>
                        Back to Notices
                    </a>
                    <button type="submit" class="btn-submit">
                        <i class="fas fa-save"></i>
                        Create Notice
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Loading Modal -->
<div class="modal-overlay" id="loadingModal" style="display: none;">
    <div class="modal-content loading-modal">
        <div class="loading-content">
            <div class="loading-spinner"></div>
            <h3>Creating Notice</h3>
            <p>Please wait while we save your notice...</p>
        </div>
    </div>
</div>

<style>
/* Add Notice Styles */
.add-notice-container {
    min-height: 100vh;
    background: #f8fafc;
    font-family: 'Inter', sans-serif;
}

.add-notice-header {
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    padding: 2rem 0;
    margin-bottom: 0;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

.header-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 2rem;
}

.page-title h1 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #ffffff;
    margin: 0 0 0.5rem 0;
    letter-spacing: -0.025em;
}

.page-title p {
    font-size: 1.125rem;
    color: #cbd5e1;
    margin: 0;
    font-weight: 400;
}

.add-notice-content {
    padding: 2rem;
}

.content-box {
    background: white;
    border-radius: 12px;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    overflow: hidden;
    max-width: 800px;
    margin: 0 auto;
}

.form-section {
    padding: 2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-label {
    display: block;
    font-size: 0.875rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e2e8f0;
    border-radius: 8px;
    font-size: 0.875rem;
    color: #1e293b;
    background: #ffffff;
    transition: all 0.2s ease;
    font-family: inherit;
}

.form-control:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.form-control.is-invalid {
    border-color: #dc2626;
}

.invalid-feedback {
    display: block;
    color: #dc2626;
    font-size: 0.75rem;
    margin-top: 0.25rem;
}

.form-text {
    display: block;
    color: #6b7280;
    font-size: 0.75rem;
    margin-top: 0.25rem;
}

.image-upload-container {
    position: relative;
}

.image-preview {
    position: relative;
    margin-top: 1rem;
    display: inline-block;
}

.image-preview img {
    max-width: 200px;
    max-height: 200px;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.remove-image {
    position: absolute;
    top: -8px;
    right: -8px;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    background: #dc2626;
    color: white;
    border: none;
    font-size: 16px;
    font-weight: bold;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
}

.remove-image:hover {
    background: #b91c1c;
}

.form-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem 2rem;
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
}

.btn-cancel, .btn-submit {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-size: 0.875rem;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
}

.btn-cancel {
    background: #f1f5f9;
    color: #374151;
}

.btn-cancel:hover {
    background: #e2e8f0;
    color: #1e293b;
}

.btn-submit {
    background: #0055a5;
    color: #ffffff;
}

.btn-submit:hover {
    background: #004494;
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

/* Loading Modal */
.loading-modal {
    max-width: 400px;
    text-align: center;
}

.loading-content {
    padding: 2rem;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #f3f3f3;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 1rem;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-content h3 {
    font-size: 1.25rem;
    font-weight: 600;
    color: #1e293b;
    margin: 0 0 0.5rem 0;
}

.loading-content p {
    color: #6b7280;
    margin: 0;
}

/* Responsive Design */
@media (max-width: 768px) {
    .add-notice-content {
        padding: 1rem;
    }
    
    .form-section {
        padding: 1.5rem;
    }
    
    .form-actions {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .btn-cancel, .btn-submit {
        justify-content: center;
    }
}
</style>

<script>
// Image preview functionality
function previewImage(input) {
    const file = input.files[0];
    const preview = document.getElementById('imagePreview');
    const previewImg = document.getElementById('previewImg');
    
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            previewImg.src = e.target.result;
            preview.style.display = 'block';
        };
        reader.readAsDataURL(file);
    } else {
        preview.style.display = 'none';
    }
}

// Remove image functionality
function removeImage() {
    const input = document.getElementById('image');
    const preview = document.getElementById('imagePreview');
    
    input.value = '';
    preview.style.display = 'none';
}

// Form submission with loading modal
document.getElementById('noticeForm').addEventListener('submit', function(e) {
    // Show loading modal
    document.getElementById('loadingModal').style.display = 'flex';
    
    // The form will submit normally, and the loading modal will be hidden by page redirect
});

// Character count for description
document.getElementById('description').addEventListener('input', function() {
    const maxLength = 1000;
    const currentLength = this.value.length;
    const remaining = maxLength - currentLength;
    
    // You can add a character counter here if needed
    if (remaining < 0) {
        this.value = this.value.substring(0, maxLength);
    }
});

// Auto-resize textarea
document.getElementById('description').addEventListener('input', function() {
    this.style.height = 'auto';
    this.style.height = this.scrollHeight + 'px';
});
</script>
@endsection
