@extends('welcome')

@section('title', 'Operator Daily Reporting')

@section('content')
<div class="odr-container">
    <!-- Header Section -->
    <div class="odr-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Operator Daily Reporting</h1>
                <p>Track and manage daily operator reports and schedules</p>
            </div>
            <div class="header-actions">
                <div class="search-container">
                    <div class="search-input-wrapper">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search reports..." id="searchInput">
                    </div>
                </div>
                <div class="action-buttons">
                    <button class="btn-secondary" data-bs-toggle="modal" data-bs-target="#sorting">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M3 6h18M7 12h10M10 18h4"/>
                        </svg>
                        Sorting
                </button>
                    <button class="btn-primary">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                            <line x1="16" y1="13" x2="8" y2="13"></line>
                            <line x1="16" y1="17" x2="8" y2="17"></line>
                            <polyline points="10,9 9,9 8,9"></polyline>
                        </svg>
                        Excel Export
                </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Table Section -->
    <div class="table-section">
        <div class="table-container">
            <table class="odr-table" id="odrTable">
                <thead>
                    <tr>
                        <th>S.No</th>
                        <th>Action</th>
                        <th>Report Number</th>
                        <th>Start Date</th>
                        <th>Vehicle</th>
                        <th>Mon</th>
                        <th>Tue</th>
                        <th>Wed</th>
                        <th>Thu</th>
                        <th>Fri</th>
                        <th>Sat</th>
                        <th>Sun</th>
                        <th>Submitted Report</th>
                    </tr>
                </thead>
                <tbody id="odrTableBody">
                    <tr class="no-data-row">
                        <td colspan="13" class="no-data-cell">
                            <div class="no-data-content">
                                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="10"></circle>
                                    <line x1="12" y1="8" x2="12" y2="12"></line>
                                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                                </svg>
                                <h3>No Data Available</h3>
                                <p>No operator daily reports found. Start by creating a new report.</p>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
        <div class="table-footer">
            <div class="rows-per-page">
                <label for="rowsPerPageSelect">Rows per page</label>
                <select id="rowsPerPageSelect">
                    <option value="10" selected>10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                </select>
            </div>
            <div class="pagination-controls">
                <button type="button" class="pagination-btn" id="prevPageBtn" aria-label="Previous page">&laquo;</button>
                <span id="paginationInfo">0-0 of 0</span>
                <button type="button" class="pagination-btn" id="nextPageBtn" aria-label="Next page">&raquo;</button>
            </div>
        </div>
    </div>
</div>
<!-- Sorting Modal -->
<div class="modal fade" id="sorting" tabindex="-1" aria-labelledby="sortingModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="sortingModalLabel">Sorting Options</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="sortingForm">
                    <div class="sort-section">
                        <h6 class="section-title">Sort Order</h6>
                        <div class="radio-group">
                            <div class="radio-item">
                                <input type="radio" id="ascending" name="sortOrder" value="ascending" checked>
                                <label for="ascending">
                                    <span class="radio-custom"></span>
                                    Ascending
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="descending" name="sortOrder" value="descending">
                                <label for="descending">
                                    <span class="radio-custom"></span>
                                    Descending
                                </label>
                        </div>
                        </div>
                    </div>
                    
                    <div class="sort-section">
                        <h6 class="section-title">Sort By</h6>
                        <div class="radio-group">
                            <div class="radio-item">
                                <input type="radio" id="reportNumber" name="sortBy" value="reportNumber" checked>
                                <label for="reportNumber">
                                    <span class="radio-custom"></span>
                                    Report Number
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="startDate" name="sortBy" value="startDate">
                                <label for="startDate">
                                    <span class="radio-custom"></span>
                                    Start Date
                                </label>
                    </div>
                            <div class="radio-item">
                                <input type="radio" id="vehicleNumber" name="sortBy" value="vehicleNumber">
                                <label for="vehicleNumber">
                                    <span class="radio-custom"></span>
                                    Vehicle Number
                                </label>
                    </div>
                            <div class="radio-item">
                                <input type="radio" id="vehicleName" name="sortBy" value="vehicleName">
                                <label for="vehicleName">
                                    <span class="radio-custom"></span>
                                    Vehicle Name
                                </label>
                    </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-cancel" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn-reset" onclick="resetSorting()">Reset</button>
                <button type="button" class="btn-apply" onclick="applySorting()">Apply</button>
            </div>
        </div>
    </div>
</div>
<!-- Report Details Modal -->
<div class="modal fade" id="reportDetailsModal" tabindex="-1" aria-labelledby="reportDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered report-modal-dialog">
        <div class="modal-content report-modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="reportDetailsModalLabel">Report Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body report-modal-body">
                <div id="reportDetailsContent" class="report-details-content">
                    <p class="empty-state">Select a report to view the details.</p>
                </div>
            </div>
            <div class="modal-footer report-modal-footer">
                <button type="button" class="btn-cancel" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn-apply" id="viewPdfFromDetailsBtn" style="display: none;">View PDF</button>
            </div>
        </div>
    </div>
</div>
<!-- PDF Preview Modal -->
<div class="modal fade" id="pdfPreviewModal" tabindex="-1" aria-labelledby="pdfPreviewModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="pdfPreviewModalLabel">Daily Inspection Report</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body pdf-modal-body">
                <iframe id="pdfPreviewFrame" title="Daily inspection report" loading="lazy"></iframe>
            </div>
        </div>
    </div>
</div>
<script>
// Modernized client-side logic with pagination and modal handling
const searchInputEl = document.getElementById('searchInput');
const tableBodyEl = document.getElementById('odrTableBody');
const rowsPerPageSelect = document.getElementById('rowsPerPageSelect');
const paginationInfoEl = document.getElementById('paginationInfo');
const prevPageBtn = document.getElementById('prevPageBtn');
const nextPageBtn = document.getElementById('nextPageBtn');
const reportDetailsModalEl = document.getElementById('reportDetailsModal');
const reportDetailsContentEl = document.getElementById('reportDetailsContent');
const pdfPreviewModalEl = document.getElementById('pdfPreviewModal');
const pdfPreviewFrame = document.getElementById('pdfPreviewFrame');
const viewPdfFromDetailsBtn = document.getElementById('viewPdfFromDetailsBtn');

let currentSortBy = 'reportNumber';
let currentSortOrder = 'ascending';
let tableData = [];
let filteredData = [];
let currentPage = 1;
let rowsPerPage = parseInt(rowsPerPageSelect.value, 10) || 10;
let activePdfUrl = null;

searchInputEl.addEventListener('input', handleSearch);
rowsPerPageSelect.addEventListener('change', handleRowsPerPageChange);
prevPageBtn.addEventListener('click', () => changePage(-1));
nextPageBtn.addEventListener('click', () => changePage(1));

document.addEventListener('click', (event) => {
    const actionBtn = event.target.closest('[data-action]');
    if (!actionBtn) {
        return;
    }

    const reportId = actionBtn.getAttribute('data-report-id');
    const pdfUrl = actionBtn.getAttribute('data-pdf-url');
    const actionType = actionBtn.getAttribute('data-action');

    if (actionType === 'view-report') {
        openReportDetails(reportId);
    } else if (actionType === 'view-pdf') {
        openPdfModal(pdfUrl);
    }
});

reportDetailsModalEl?.addEventListener('hidden.bs.modal', () => {
    viewPdfFromDetailsBtn.style.display = 'none';
    viewPdfFromDetailsBtn.onclick = null;
});

pdfPreviewModalEl?.addEventListener('hidden.bs.modal', () => {
    pdfPreviewFrame.removeAttribute('src');
    activePdfUrl = null;
});

function handleSearch(event) {
    const term = event.target.value.trim().toLowerCase();
    if (!term) {
        filteredData = [...tableData];
    } else {
        filteredData = tableData.filter((row) => {
            const searchable = getSearchableText(row);
            return searchable.includes(term);
        });
    }
    currentPage = 1;
    applySortingToTable();
}

function handleRowsPerPageChange(event) {
    rowsPerPage = parseInt(event.target.value, 10) || 10;
    currentPage = 1;
    renderTableData();
}

function changePage(delta) {
    const totalPages = Math.max(1, Math.ceil(filteredData.length / rowsPerPage));
    const nextPage = currentPage + delta;
    if (nextPage < 1 || nextPage > totalPages) {
        return;
    }
    currentPage = nextPage;
    renderTableData();
}

function applySorting() {
    currentSortOrder = document.querySelector('input[name="sortOrder"]:checked').value;
    currentSortBy = document.querySelector('input[name="sortBy"]:checked').value;
    applySortingToTable();
    const modal = bootstrap.Modal.getInstance(document.getElementById('sorting'));
    modal.hide();
    showNotification(`Table sorted by ${getSortByLabel(currentSortBy)} (${currentSortOrder})`, 'success');
}

function resetSorting() {
    document.getElementById('ascending').checked = true;
    document.getElementById('reportNumber').checked = true;
    currentSortBy = 'reportNumber';
    currentSortOrder = 'ascending';
    applySortingToTable();
    showNotification('Sorting options reset to default', 'info');
}

function getSortByLabel(sortBy) {
    const labels = {
        reportNumber: 'Report Number',
        startDate: 'Start Date',
        vehicleNumber: 'Vehicle Number',
        vehicleName: 'Vehicle Name'
    };
    return labels[sortBy] || sortBy;
}

function applySortingToTable() {
    if (!filteredData.length) {
        renderTableData();
        return;
    }

    filteredData.sort((a, b) => {
        let aValue = getComparableValue(a, currentSortBy);
        let bValue = getComparableValue(b, currentSortBy);

        if (aValue > bValue) {
            return currentSortOrder === 'ascending' ? 1 : -1;
        }
        if (aValue < bValue) {
            return currentSortOrder === 'ascending' ? -1 : 1;
        }
        return 0;
    });

    renderTableData();
}

function getComparableValue(row, key) {
    if (key === 'startDate') {
        // Dates already formatted, attempt to parse as d/m/Y
        const parts = String(row.startDate || '').split('/');
        if (parts.length === 3) {
            return new Date(`${parts[2]}-${parts[1]}-${parts[0]}`).getTime() || 0;
        }
        return new Date(row.startDate || '').getTime() || 0;
    }
    const value = row[key] ?? '';
    return String(value).toLowerCase();
}

function renderTableData() {
    if (!filteredData.length) {
        tableBodyEl.innerHTML = `
            <tr class="no-data-row">
                <td colspan="13" class="no-data-cell">
                    <div class="no-data-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"></circle>
                            <line x1="12" y1="8" x2="12" y2="12"></line>
                            <line x1="12" y1="16" x2="12.01" y2="16"></line>
                        </svg>
                        <h3>No Data Available</h3>
                        <p>No operator daily reports found. Start by creating a new report.</p>
                    </div>
                </td>
            </tr>
        `;
        updatePaginationControls();
        return;
    }

    const totalPages = Math.max(1, Math.ceil(filteredData.length / rowsPerPage));
    if (currentPage > totalPages) {
        currentPage = totalPages;
    }

    const startIndex = (currentPage - 1) * rowsPerPage;
    const endIndex = startIndex + rowsPerPage;
    const pageItems = filteredData.slice(startIndex, endIndex);

    tableBodyEl.innerHTML = pageItems.map((row, index) => {
        const serial = startIndex + index + 1;
        return `
            <tr>
                <td>${serial}</td>
                <td>
                    <div class="dropdown action-dropdown">
                        <button class="btn-action-dropdown" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                            Actions
                        </button>
                        <ul class="dropdown-menu">
                            <li>
                                <button class="dropdown-item" type="button" data-action="view-report" data-report-id="${escapeAttribute(row.id)}">
                                    View Report
                                </button>
                            </li>
                            <li>
                                ${row.pdfUrl ? `
                                    <button class="dropdown-item" type="button" data-action="view-pdf" data-report-id="${escapeAttribute(row.id)}" data-pdf-url="${escapeAttribute(row.pdfUrl)}">
                                        View PDF
                                    </button>
                                ` : `
                                    <span class="dropdown-item disabled" aria-disabled="true">PDF Not Available</span>
                                `}
                            </li>
                        </ul>
                    </div>
                </td>
                <td>${escapeHtml(row.reportNumber || 'N/A')}</td>
                <td>${escapeHtml(row.startDate || 'N/A')}</td>
                <td>${escapeHtml(row.vehicleName || row.vehicleNumber || 'N/A')}</td>
                <td>${getDayStatus(row.mon)}</td>
                <td>${getDayStatus(row.tue)}</td>
                <td>${getDayStatus(row.wed)}</td>
                <td>${getDayStatus(row.thu)}</td>
                <td>${getDayStatus(row.fri)}</td>
                <td>${getDayStatus(row.sat)}</td>
                <td>${getDayStatus(row.sun)}</td>
                <td>${getSubmittedStatus(row.submitted)}</td>
            </tr>
        `;
    }).join('');

    updatePaginationControls();
}

function updatePaginationControls() {
    const totalItems = filteredData.length;
    const totalPages = Math.max(1, Math.ceil(totalItems / rowsPerPage));
    const start = totalItems ? (currentPage - 1) * rowsPerPage + 1 : 0;
    const end = Math.min(currentPage * rowsPerPage, totalItems);
    paginationInfoEl.textContent = `${start}-${end} of ${totalItems}`;
    prevPageBtn.disabled = currentPage <= 1;
    nextPageBtn.disabled = currentPage >= totalPages;
}

function getSearchableText(row) {
    return [
        row.reportNumber,
        row.vehicleName,
        row.vehicleNumber,
        row.startDate
    ].map((value) => String(value || '').toLowerCase()).join(' ');
}

function getDayStatus(day) {
    if (day === 'completed') {
        return '<span class="status-badge status-success">Done</span>';
    }
    if (day === 'pending') {
        return '<span class="status-badge status-warning">Pending</span>';
    }
    if (day === 'missed') {
        return '<span class="status-badge status-danger">Missed</span>';
    }
    return '<span class="status-badge status-secondary">-</span>';
}

function getSubmittedStatus(submitted) {
    if (submitted) {
        return '<span class="status-badge status-success">Submitted</span>';
    }
    return '<span class="status-badge status-warning">Pending</span>';
}

function escapeHtml(value) {
    return String(value ?? '')
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}

function escapeAttribute(value) {
    return String(value ?? '').replace(/"/g, '&quot;');
}

function openReportDetails(reportId) {
    const report = tableData.find((item) => String(item.id) === String(reportId));
    if (!report) {
        showNotification('Unable to load report details.', 'warning');
        return;
    }

    const detailRows = [
        { label: 'Report Number', value: report.reportNumber || 'N/A' },
        { label: 'Start Date', value: report.startDate || 'N/A' },
        { label: 'Vehicle', value: report.vehicleName || report.vehicleNumber || 'N/A' },
        { label: 'Created At', value: formatDateTime(report.created_at) },
        { label: 'Updated At', value: formatDateTime(report.updated_at) },
        { label: 'Submission', value: report.submitted ? 'Submitted' : 'Pending' }
    ];

    if (report.checklistTitle || (report.checklistDetails && Object.keys(report.checklistDetails).length)) {
        detailRows.push({
            label: 'Checklist',
            value: report.checklistTitle || (report.checklistDetails?.formTitle ?? report.checklistDetails?.title ?? 'N/A')
        });
    }

    const weeklyStatus = ['mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun'].map((key, index) => {
        const dayNames = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
        return `<div class="weekly-status-row"><span>${dayNames[index]}</span><span>${formatStatusText(report[key])}</span></div>`;
    }).join('');

    const checklistDetails = report.checklistDetails || {};
    const checklistItems = Array.isArray(checklistDetails.inspectionItems) ? checklistDetails.inspectionItems : [];
    const checklistDocument = checklistDetails.document || {};

    const checklistMetadata = [
        checklistDocument.number ? `Document No: ${escapeHtml(checklistDocument.number)}` : '',
        checklistDocument.version ? `Version: ${escapeHtml(checklistDocument.version)}` : '',
        checklistDocument.versionDate ? `Version Date: ${escapeHtml(checklistDocument.versionDate)}` : '',
        checklistDocument.approvedBy ? `Approved By: ${escapeHtml(checklistDocument.approvedBy)}` : ''
    ].filter(Boolean);

    const checklistMetadataMarkup = checklistMetadata.length
        ? `<div class="checklist-metadata">${checklistMetadata.map((item) => `<span>${item}</span>`).join('')}</div>`
        : '';

    const checklistItemsMarkup = checklistItems.length
        ? `<ul class="checklist-items">${checklistItems.map((item) => `<li>${escapeHtml(item)}</li>`).join('')}</ul>`
        : '<p class="empty-state">No checklist items recorded.</p>';

    const checklistSection = (report.checklistTitle || checklistItems.length || checklistMetadata.length)
        ? `
        <div class="checklist-details">
            <h6>Checklist Used</h6>
            ${report.checklistTitle ? `<p class="checklist-title">${escapeHtml(report.checklistTitle)}</p>` : ''}
            ${checklistMetadataMarkup}
            ${checklistItemsMarkup}
        </div>
        `
        : '';

    reportDetailsContentEl.innerHTML = `
        <div class="report-details-grid">
            ${detailRows.map((row) => `
                <div class="detail-item">
                    <span class="detail-label">${escapeHtml(row.label)}</span>
                    <span class="detail-value">${escapeHtml(row.value)}</span>
                </div>
            `).join('')}
        </div>
        <div class="weekly-status">
            <h6>Weekly Status</h6>
            ${weeklyStatus}
        </div>
        ${checklistSection}
    `;

    if (report.pdfUrl) {
        viewPdfFromDetailsBtn.style.display = 'inline-flex';
        viewPdfFromDetailsBtn.onclick = () => openPdfModal(report.pdfUrl);
    } else {
        viewPdfFromDetailsBtn.style.display = 'none';
        viewPdfFromDetailsBtn.onclick = null;
    }

    new bootstrap.Modal(reportDetailsModalEl).show();
}

function formatStatusText(status) {
    if (status === 'completed') {
        return 'Completed';
    }
    if (status === 'pending') {
        return 'Pending';
    }
    if (status === 'missed') {
        return 'Missed';
    }
    return 'Not Captured';
}

function formatDateTime(value) {
    if (!value) {
        return 'N/A';
    }
    try {
        const date = new Date(value);
        if (Number.isNaN(date.getTime())) {
            return value;
        }
        return date.toLocaleString('en-GB', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    } catch (error) {
        return value;
    }
}

function openPdfModal(pdfUrl) {
    if (!pdfUrl) {
        showNotification('PDF not available for this report.', 'warning');
        return;
    }
    activePdfUrl = pdfUrl;
    pdfPreviewFrame.src = pdfUrl;
    new bootstrap.Modal(pdfPreviewModalEl).show();
}

async function loadTableData() {
    try {
        const response = await fetch('/inspection/get-operator-daily-reports');
        const result = await response.json();

        if (!result.success) {
            throw new Error(result.message || 'Unable to load data');
        }

        tableData = Array.isArray(result.data) ? result.data : [];
        filteredData = [...tableData];

        if (tableData.length === 0) {
            showNotification('No daily reports with PDFs found.', 'info');
        } else {
            showNotification(`Loaded ${tableData.length} operator daily reports`, 'success');
        }

        applySortingToTable();
    } catch (error) {
        console.error('Error fetching operator daily reports:', error);
        tableData = [];
        filteredData = [];
        showNotification('Error loading daily reports from MongoDB.', 'error');
        renderTableData();
    }
}

document.querySelector('.btn-primary').addEventListener('click', () => {
    showNotification('Exporting data to Excel...', 'info');
});

function refreshData() {
    showNotification('Refreshing data...', 'info');
    loadTableData();
}

document.addEventListener('keydown', (event) => {
    if ((event.ctrlKey && event.key === 'r') || event.key === 'F5') {
        event.preventDefault();
        refreshData();
    }
});

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${escapeHtml(message)}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    document.body.appendChild(notification);
    setTimeout(() => notification.remove(), 5000);
}

document.addEventListener('DOMContentLoaded', () => {
    searchInputEl.focus();
    @if(isset($operatorReports) && !empty($operatorReports))
        tableData = @json($operatorReports);
        filteredData = [...tableData];
        applySortingToTable();
    @else
        loadTableData();
    @endif
});
</script>
<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* Main Container */
.odr-container {
    min-height: 100vh;
}

/* Header Section */
.odr-header {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    margin-bottom: 24px;
}

.header-content {
    padding: 32px;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 24px;
}

.page-title h1 {
    color: #0f172a;
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.page-title p {
    color: #64748b;
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

/* .sort-section{
    z-index: 211211121;
} */

.header-actions {
    display: flex;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

/* Search Container */
.search-container {
    position: relative;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 16px;
    z-index: 2;
    color: #64748b;
    width: 16px;
    height: 16px;
}

.search-input {
    padding: 12px 16px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    height: 44px;
    width: 300px;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-primary, .btn-secondary {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
}

.btn-primary {
    background: #0055a5;
    color: white;
}

.btn-primary:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
}

.btn-secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4b5563 0%, #374151 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: white;
}

/* Table Section */
.table-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    overflow: hidden;
}

.table-container {
    overflow-x: auto;
}

.odr-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.odr-table thead {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.odr-table th {
    padding: 16px 12px;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e2e8f0;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    white-space: nowrap;
}

.odr-table td {
    padding: 16px 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
    white-space: nowrap;
}

.odr-table tbody tr:hover {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

/* No Data Styling */
.no-data-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.no-data-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.no-data-content svg {
    color: #9ca3af;
}

.no-data-content h3 {
    color: #374151;
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.no-data-content p {
    color: #6b7280;
    font-weight: 400;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

/* Modal Styling */
.modal-content {
    border: 1px solid #e2e8f0;
    border-radius: 0;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    z-index: 1000222;
}

.modal-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.modal-title {
    color: #0f172a;
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.btn-close {
    position: relative;
    width: 32px;
    height: 32px;
    background: transparent;
    border: none;
    opacity: 1;
    transition: transform 0.15s ease;
}

.btn-close::before {
    content: '\00d7';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    font-size: 24px;
    color: #64748b;
    line-height: 1;
}

.btn-close:hover::before {
    color: #374151;
}

.btn-close:hover {
    transform: scale(1.1);
}

.modal-body {
    padding: 24px;
}

.report-modal-dialog {
    max-width: 900px;
}

.report-modal-content {
    max-height: 85vh;
    display: flex;
    flex-direction: column;
}

.report-modal-body {
    overflow-y: auto;
    padding: 24px;
}

.report-modal-footer {
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
    padding: 16px 24px;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
}

.sort-section {
    margin-bottom: 24px;
}

.section-title {
    color: #374151;
    font-weight: 600;
    font-size: 14px;
    margin-bottom: 12px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.radio-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.radio-item {
    display: flex;
    align-items: center;
    gap: 12px;
}

.radio-item input[type="radio"] {
    display: none;
}

.radio-item label {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    transition: all 0.2s ease;
}

.radio-custom {
    width: 18px;
    height: 18px;
    border: 2px solid #d1d5db;
    border-radius: 50%;
    position: relative;
    transition: all 0.2s ease;
}

.radio-item input[type="radio"]:checked + label .radio-custom {
    border-color: #3b82f6;
    background: #3b82f6;
}

.radio-item input[type="radio"]:checked + label .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 6px;
    height: 6px;
    background: white;
    border-radius: 50%;
}

.radio-item label:hover {
    color: #1e293b;
}

.radio-item label:hover .radio-custom {
    border-color: #3b82f6;
}

.modal-footer {
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
    padding: 16px 24px;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
}

.btn-cancel, .btn-reset, .btn-apply {
    padding: 10px 20px;
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
}

.btn-cancel:hover {
    background: #e5e7eb;
    color: #1f2937;
}

.btn-reset {
    background: #fef3c7;
    color: #92400e;
}

.btn-reset:hover {
    background: #fde68a;
    color: #78350f;
}

.btn-apply {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: white;
}

.btn-apply:hover {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .odr-container {
        padding: 20px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .search-input {
        width: 250px;
    }
    
    .table-section {
        margin-top: 20px;
    }
}

@media (max-width: 992px) {
    .odr-container {
        padding: 16px;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .header-actions {
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 16px;
    }
    
    .search-input {
        width: 100%;
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 12px;
    }
    
    .btn-primary, .btn-secondary {
        flex: 1;
        min-width: 140px;
        justify-content: center;
    }
    
    .table-section {
        margin-top: 16px;
    }
    
    .odr-table {
        font-size: 13px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 14px 10px;
    }
}

@media (max-width: 768px) {
    .odr-container {
        padding: 0;
    }
    
    .odr-header {
        margin: 0 16px 16px 16px;
    }
    
    .header-content {
        padding: 16px;
        gap: 16px;
    }
    
    .page-title h1 {
        font-size: 24px;
        margin-bottom: 6px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px 12px 44px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
        gap: 12px;
    }
    
    .btn-primary, .btn-secondary {
        flex: 1;
        justify-content: center;
        min-width: 140px;
        padding: 14px 20px;
        font-size: 14px;
        min-height: 48px;
    }
    
    .table-section {
        margin: 0 16px 16px 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .odr-table {
        font-size: 12px;
        min-width: 800px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 12px 8px;
        white-space: nowrap;
    }
    
    .odr-table th {
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.05em;
    }
    
    .odr-table td {
        font-size: 12px;
    }
    
    .action-buttons {
        gap: 6px;
    }
    
    .btn-action {
        width: 28px;
        height: 28px;
        min-width: 28px;
    }
    
    .btn-action svg {
        width: 14px;
        height: 14px;
    }
    
    .status-badge {
        padding: 3px 6px;
        font-size: 10px;
        min-width: 20px;
        height: 20px;
    }
    
    .no-data-cell {
        padding: 48px 16px !important;
    }
    
    .no-data-content h3 {
        font-size: 16px;
    }
    
    .no-data-content p {
        font-size: 13px;
    }
    
    .no-data-content svg {
        width: 40px;
        height: 40px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 10px;
        max-width: calc(100% - 20px);
    }
    
    .modal-content {
        border-radius: 8px;
    }
    
    .modal-header {
        padding: 16px 20px;
    }
    
    .modal-title {
        font-size: 16px;
    }
    
    .modal-body {
        padding: 20px;
    }
    
    .sort-section {
        margin-bottom: 20px;
    }
    
    .section-title {
        font-size: 13px;
        margin-bottom: 10px;
    }
    
    .radio-item {
        gap: 10px;
    }
    
    .radio-item label {
        font-size: 13px;
    }
    
    .radio-custom {
        width: 16px;
        height: 16px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 5px;
        height: 5px;
    }
    
    .modal-footer {
        padding: 16px 20px;
        flex-direction: column;
        gap: 12px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        width: 100%;
        padding: 12px 20px;
        font-size: 14px;
        min-height: 44px;
    }
}

@media (max-width: 576px) {
    .odr-container {
        padding: 0;
    }
    
    .odr-header {
        margin: 0 12px 12px 12px;
    }
    
    .header-content {
        padding: 12px;
        gap: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .header-actions {
        gap: 12px;
    }
    
    .search-input {
        padding: 10px 14px 10px 40px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .search-icon {
        width: 14px;
        height: 14px;
        left: 14px;
    }
    
    .action-buttons {
        gap: 8px;
    }
    
    .btn-primary, .btn-secondary {
        padding: 12px 16px;
        font-size: 13px;
        min-height: 44px;
        min-width: 120px;
    }
    
    .table-section {
        margin: 0 12px 12px 12px;
    }
    
    .odr-table {
        font-size: 11px;
        min-width: 700px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 10px 6px;
    }
    
    .odr-table th {
        font-size: 10px;
    }
    
    .odr-table td {
        font-size: 11px;
    }
    
    .btn-action {
        width: 24px;
        height: 24px;
        min-width: 24px;
    }
    
    .btn-action svg {
        width: 12px;
        height: 12px;
    }
    
    .status-badge {
        padding: 2px 4px;
        font-size: 9px;
        min-width: 18px;
        height: 18px;
    }
    
    .no-data-cell {
        padding: 40px 12px !important;
    }
    
    .no-data-content h3 {
        font-size: 15px;
    }
    
    .no-data-content p {
        font-size: 12px;
    }
    
    .no-data-content svg {
        width: 36px;
        height: 36px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 8px;
        max-width: calc(100% - 16px);
    }
    
    .modal-header {
        padding: 12px 16px;
    }
    
    .modal-title {
        font-size: 15px;
    }
    
    .modal-body {
        padding: 16px;
    }
    
    .sort-section {
        margin-bottom: 16px;
    }
    
    .section-title {
        font-size: 12px;
        margin-bottom: 8px;
    }
    
    .radio-item {
        gap: 8px;
    }
    
    .radio-item label {
        font-size: 12px;
    }
    
    .radio-custom {
        width: 14px;
        height: 14px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 4px;
        height: 4px;
    }
    
    .modal-footer {
        padding: 12px 16px;
        gap: 8px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        padding: 10px 16px;
        font-size: 13px;
        min-height: 40px;
    }
}

@media (max-width: 480px) {
    .odr-header {
        margin: 0 8px 8px 8px;
    }
    
    .header-content {
        padding: 10px;
        gap: 10px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .header-actions {
        gap: 10px;
    }
    
    .search-input {
        padding: 8px 12px 8px 36px;
        font-size: 16px;
        min-height: 40px;
    }
    
    .search-icon {
        width: 12px;
        height: 12px;
        left: 12px;
    }
    
    .action-buttons {
        gap: 6px;
    }
    
    .btn-primary, .btn-secondary {
        padding: 10px 14px;
        font-size: 12px;
        min-height: 40px;
        min-width: 100px;
    }
    
    .table-section {
        margin: 0 8px 8px 8px;
    }
    
    .odr-table {
        font-size: 10px;
        min-width: 600px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 8px 4px;
    }
    
    .odr-table th {
        font-size: 9px;
    }
    
    .odr-table td {
        font-size: 10px;
    }
    
    .btn-action {
        width: 22px;
        height: 22px;
        min-width: 22px;
    }
    
    .btn-action svg {
        width: 10px;
        height: 10px;
    }
    
    .status-badge {
        padding: 1px 3px;
        font-size: 8px;
        min-width: 16px;
        height: 16px;
    }
    
    .no-data-cell {
        padding: 32px 8px !important;
    }
    
    .no-data-content h3 {
        font-size: 14px;
    }
    
    .no-data-content p {
        font-size: 11px;
    }
    
    .no-data-content svg {
        width: 32px;
        height: 32px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 4px;
        max-width: calc(100% - 8px);
    }
    
    .modal-header {
        padding: 10px 12px;
    }
    
    .modal-title {
        font-size: 14px;
    }
    
    .modal-body {
        padding: 12px;
    }
    
    .sort-section {
        margin-bottom: 12px;
    }
    
    .section-title {
        font-size: 11px;
        margin-bottom: 6px;
    }
    
    .radio-item {
        gap: 6px;
    }
    
    .radio-item label {
        font-size: 11px;
    }
    
    .radio-custom {
        width: 12px;
        height: 12px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 3px;
        height: 3px;
    }
    
    .modal-footer {
        padding: 10px 12px;
        gap: 6px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 36px;
    }
}

@media (max-width: 360px) {
    .odr-header {
        margin: 0 6px 6px 6px;
    }
    
    .header-content {
        padding: 8px;
        gap: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .header-actions {
        gap: 8px;
    }
    
    .search-input {
        padding: 6px 10px 6px 32px;
        font-size: 16px;
        min-height: 36px;
    }
    
    .search-icon {
        width: 10px;
        height: 10px;
        left: 10px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-primary, .btn-secondary {
        padding: 8px 12px;
        font-size: 11px;
        min-height: 36px;
        min-width: 80px;
    }
    
    .table-section {
        margin: 0 6px 6px 6px;
    }
    
    .odr-table {
        font-size: 9px;
        min-width: 500px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 6px 3px;
    }
    
    .odr-table th {
        font-size: 8px;
    }
    
    .odr-table td {
        font-size: 9px;
    }
    
    .btn-action {
        width: 20px;
        height: 20px;
        min-width: 20px;
    }
    
    .btn-action svg {
        width: 8px;
        height: 8px;
    }
    
    .status-badge {
        padding: 1px 2px;
        font-size: 7px;
        min-width: 14px;
        height: 14px;
    }
    
    .no-data-cell {
        padding: 24px 6px !important;
    }
    
    .no-data-content h3 {
        font-size: 13px;
    }
    
    .no-data-content p {
        font-size: 10px;
    }
    
    .no-data-content svg {
        width: 28px;
        height: 28px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 2px;
        max-width: calc(100% - 4px);
    }
    
    .modal-header {
        padding: 8px 10px;
    }
    
    .modal-title {
        font-size: 13px;
    }
    
    .modal-body {
        padding: 10px;
    }
    
    .sort-section {
        margin-bottom: 10px;
    }
    
    .section-title {
        font-size: 10px;
        margin-bottom: 4px;
    }
    
    .radio-item {
        gap: 4px;
    }
    
    .radio-item label {
        font-size: 10px;
    }
    
    .radio-custom {
        width: 10px;
        height: 10px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 2px;
        height: 2px;
    }
    
    .modal-footer {
        padding: 8px 10px;
        gap: 4px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        padding: 6px 10px;
        font-size: 11px;
        min-height: 32px;
    }
}

/* Focus states for accessibility */
.btn-primary:focus,
.btn-secondary:focus,
.search-input:focus,
.btn-cancel:focus,
.btn-reset:focus,
.btn-apply:focus {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* Smooth animations */
* {
    transition: all 0.2s ease;
}

/* Icon improvements */
.btn-primary svg,
.btn-secondary svg {
    flex-shrink: 0;
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 4px 8px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    min-width: 24px;
    height: 24px;
}

.status-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.status-warning {
    background: #ffd205;
    color: #0f172a;
}

.status-danger {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.status-secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
}

/* ===== ACTION BUTTONS ===== */
.action-buttons {
    display: flex;
    gap: 8px;
    align-items: center;
}

.btn-action {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    border: none;
    border-radius: 0;
    cursor: pointer;
    transition: all 0.2s ease;
    /* box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05); */
}

.btn-view {
    background: transparent;
    color: black;
}

/* .btn-view:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
} */

.btn-edit {
    background: transparent;
    color: black;
}

/* .btn-edit:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
} */

/* Action dropdown */
.action-dropdown .btn-action-dropdown {
    background: #0055A5;
    color: white;
    border: none;
    border-radius: 0;
    padding: 8px 16px;
    font-size: 13px;
    font-weight: 500;
    cursor: pointer;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.action-dropdown .btn-action-dropdown:focus,
.action-dropdown .btn-action-dropdown:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.12);
}

.action-dropdown .dropdown-menu {
    border-radius: 0;
    padding: 0;
    min-width: 180px;
    border: 1px solid #e2e8f0;
}

.action-dropdown .dropdown-item {
    font-size: 13px;
    padding: 10px 16px;
}

.action-dropdown .dropdown-item:hover {
    background: #eef2ff;
    color: #1e3a8a;
}

.action-dropdown .dropdown-item.disabled {
    color: #94a3b8;
}

.table-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px;
    border-top: 1px solid #e2e8f0;
    background: #f8fafc;
    flex-wrap: wrap;
    gap: 12px;
}

.rows-per-page {
    display: flex;
    align-items: center;
    gap: 12px;
    font-size: 13px;
    color: #475569;
}

.rows-per-page select {
    border: 1px solid #cbd5f5;
    padding: 6px 12px;
    border-radius: 0;
    background: white;
    color: #0f172a;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 12px;
    font-size: 13px;
    color: #475569;
}

.pagination-btn {
    background: #0055A5;
    color: white;
    border: none;
    border-radius: 0;
    width: 36px;
    height: 32px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
    cursor: pointer;
    transition: opacity 0.2s ease;
}

.pagination-btn:disabled {
    opacity: 0.4;
    cursor: not-allowed;
}

.pdf-modal-body {
    height: 75vh;
    padding: 0;
}

.pdf-modal-body iframe {
    width: 100%;
    height: 100%;
    border: none;
}

.report-details-content {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.report-details-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 16px;
}

.detail-item {
    background: #f8fafc;
    padding: 16px;
    border: 1px solid #e2e8f0;
}

.detail-label {
    display: block;
    font-size: 12px;
    text-transform: uppercase;
    color: #64748b;
    margin-bottom: 6px;
    letter-spacing: 0.05em;
}

.detail-value {
    font-size: 14px;
    font-weight: 600;
    color: #0f172a;
}

.weekly-status {
    border: 1px solid #e2e8f0;
    padding: 16px;
    background: white;
}

.weekly-status h6 {
    font-size: 14px;
    font-weight: 600;
    margin-bottom: 12px;
    color: #0f172a;
}

.weekly-status-row {
    display: flex;
    justify-content: space-between;
    font-size: 13px;
    padding: 8px 0;
    border-bottom: 1px solid #f1f5f9;
    color: #334155;
}

.weekly-status-row:last-child {
    border-bottom: none;
}

.empty-state {
    margin: 0;
    color: #94a3b8;
    font-size: 14px;
    text-align: center;
}

.checklist-details {
    border: 1px solid #e2e8f0;
    padding: 16px;
    background: white;
    margin-top: 16px;
}

.checklist-details h6 {
    font-size: 14px;
    font-weight: 600;
    margin-bottom: 12px;
    color: #0f172a;
}

.checklist-title {
    font-size: 13px;
    font-weight: 600;
    color: #1e293b;
    margin-bottom: 10px;
}

.checklist-metadata {
    display: flex;
    flex-wrap: wrap;
    gap: 8px 16px;
    font-size: 12px;
    color: #475569;
    margin-bottom: 12px;
}

.checklist-metadata span {
    background: #f1f5f9;
    padding: 4px 8px;
    border: 1px solid #e2e8f0;
}

.checklist-items {
    padding-left: 20px;
    margin: 0;
    display: grid;
    gap: 6px;
    font-size: 13px;
    color: #334155;
}

.checklist-items li {
    list-style: disc;
}

@media (max-width: 768px) {
    .table-footer {
        flex-direction: column;
        align-items: stretch;
    }

    .pagination-controls {
        justify-content: space-between;
    }
}
</style>
@endsection