@extends('welcome')

@section('title', 'Vehicle Inspection')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #ffd205;
    --success-color: #00B894;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --sort-btn-color: #545d6b;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Jost', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

body.modal-open {
    overflow: hidden;
}

/* Main Container */
.inspection2-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.inspection2-header,
.header-gradient {
    background: var(--primary-color);
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-section {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin-bottom: 30px;
    padding: 30px;
}

.form-header {
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 1px solid var(--border-light);
}

.form-header h2 {
    font-size: 22px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 8px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.form-header h2 i {
    color: var(--primary-color);
}

.form-header p {
    font-size: 14px;
    color: var(--text-secondary);
    margin: 0;
}

/* Form Grid */
.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 25px;
    margin-bottom: 25px;
}

/* Form Groups */
.form-group {
    margin-bottom: 20px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.form-label.required::after {
    content: ' *';
    color: var(--danger-color);
    font-weight: bold;
}

.form-control {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-control:disabled,
.form-control:read-only {
    background: var(--bg-secondary);
    color: var(--text-secondary);
    cursor: not-allowed;
}

.form-control.is-invalid {
    border-color: var(--danger-color);
}

/* Meter Reading Group */
.meter-reading-group {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 10px;
}

.meter-info {
    margin-top: 8px;
    padding: 10px 12px;
    background: var(--bg-light);
    border-radius: var(--radius-sm);
    font-size: 13px;
}

.meter-label {
    color: var(--text-secondary);
    font-weight: 500;
}

.meter-value {
    color: var(--primary-color);
    font-weight: 600;
    margin-left: 5px;
}

/* Input with Button */
.input-with-button {
    display: flex;
    gap: 10px;
    align-items: stretch;
}

.input-with-button .form-control {
    flex: 1;
}

.btn-select,
.btn-clear {
    padding: 12px 16px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 44px;
}

.btn-select {
    background: var(--primary-color);
    color: white;
}

.btn-select:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.btn-clear {
    background: var(--danger-color);
    color: white;
}

.btn-clear:hover {
    background: #c82333;
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

/* GPS Section */
.gps-section {
    margin-top: 25px;
    padding-top: 25px;
    border-top: 1px solid var(--border-light);
}

.gps-controls {
    margin-bottom: 15px;
}

.btn-gps {
    background: var(--primary-color);
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.btn-gps:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.gps-details {
    background: var(--bg-light);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px;
}

.gps-info-grid {
    display: grid;
    gap: 12px;
}

.gps-info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.gps-label {
    font-weight: 500;
    color: var(--text-secondary);
    font-size: 13px;
}

.gps-value {
    color: var(--text-primary);
    font-size: 13px;
    font-weight: 500;
    text-align: right;
    word-break: break-all;
}

/* Form Actions */
.form-actions {
    display: flex;
    justify-content: space-between;
    gap: 15px;
    margin-top: 30px;
    padding-top: 20px;
    border-top: 1px solid var(--border-light);
}

.btn-next,
.btn-back {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 10px;
}

.btn-next {
    background: var(--primary-color);
    color: white;
    margin-left: auto;
}

.btn-next:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-back {
    background: var(--secondary-color);
    color: var(--text-primary);
}

.btn-back:hover {
    background: #e6bd04;
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}
/* Step 2: Inspection Checklist Styles */
.header-information-section {
    background: var(--bg-light);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 20px;
    margin-bottom: 25px;
}

.section-title-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.section-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.required-asterisk {
    color: var(--danger-color);
    font-weight: bold;
}

.btn-auto-fill {
    background: var(--primary-color);
    color: white;
    border: none;
    padding: 8px 16px;
    border-radius: var(--radius-md);
    font-size: 13px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 8px;
}

.btn-auto-fill:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.header-inputs-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
}

.input-group {
    margin-bottom: 0;
}

.input-label {
    display: block;
    font-size: 13px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.input-with-icon {
    position: relative;
}

.input-with-icon .form-input {
    width: 100%;
    padding-right: 40px;
}

.input-icon {
    position: absolute;
    right: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--text-secondary);
    pointer-events: none;
}

/* Inspection Sections */
.inspection-section {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    overflow: hidden;
    margin-bottom: 15px;
}

.section-header {
    background: var(--bg-light);
    padding: 15px 20px;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.section-header:hover {
    background: #e9ecef;
}

.section-header-content {
    display: flex;
    align-items: center;
    gap: 12px;
    flex: 1;
}

.section-toggle-icon {
    color: var(--primary-color);
    font-size: 16px;
    transition: transform 0.3s ease;
}

.section-header-text-container {
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-header-text {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
}

.section-item-count {
    font-size: 13px;
    color: var(--text-secondary);
    font-weight: 500;
}

.section-content {
    padding: 20px;
}

/* Inspection Items */
.inspection-item {
    padding: 15px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    margin-bottom: 15px;
    background: white;
    transition: var(--transition-base);
}

.inspection-item:hover {
    box-shadow: var(--shadow-sm);
    border-color: var(--primary-color);
}

.item-container {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.item-text {
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
}

.required-text {
    color: var(--danger-color);
    font-weight: 500;
    font-size: 12px;
}

.detail-text {
    font-size: 13px;
    color: var(--text-secondary);
    line-height: 1.5;
}

.item-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 15px;
    flex-wrap: wrap;
}

.options-container {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}

.option-button {
    padding: 8px 16px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    color: var(--text-primary);
    font-size: 13px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.option-button:hover {
    border-color: var(--primary-color);
    background: var(--bg-light);
}

.selected-option-button {
    border-color: var(--primary-color) !important;
    background: var(--primary-color) !important;
    color: white !important;
}

.ok-button {
    border-color: var(--success-color) !important;
    background: var(--success-color) !important;
    color: white !important;
}

.reject-button {
    border-color: var(--danger-color) !important;
    background: var(--danger-color) !important;
    color: white !important;
}

.na-button {
    border-color: var(--sort-btn-color) !important;
    background: var(--sort-btn-color) !important;
    color: white !important;
}

.icons-container {
    display: flex;
    gap: 8px;
}

.inspection-action-btn {
    padding: 8px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    color: var(--text-secondary);
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    justify-content: center;
}

.inspection-action-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
}

.note-display,
.image-display {
    margin-top: 10px;
    padding: 10px;
    background: var(--bg-light);
    border-radius: var(--radius-sm);
    font-size: 13px;
}

.inspection-image {
    max-width: 100%;
    max-height: 200px;
    border-radius: var(--radius-sm);
    margin-top: 8px;
}

/* Step 3: Summary Styles */
.faulty-items-container {
    background: var(--danger-color);
    color: white;
    padding: 15px 20px;
    border-radius: var(--radius-md);
    margin-bottom: 25px;
}

.faulty-items-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.faulty-items-label {
    font-size: 16px;
    font-weight: 600;
}

.faulty-items-count {
    font-size: 24px;
    font-weight: 700;
}

.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    padding: 30px;
}

.form-row {
    margin-bottom: 20px;
}

.select-with-icon {
    position: relative;
}

.form-select {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    background: white;
    cursor: pointer;
    transition: var(--transition-base);
}

.form-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.error-text {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 5px;
}

/* Status Options */
.status-options-container,
.safe-options-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 15px;
    margin-top: 10px;
}

.status-option-btn,
.safe-option-btn {
    padding: 15px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
}

.status-option-btn:hover,
.safe-option-btn:hover {
    border-color: var(--primary-color);
    transform: translateY(-2px);
    box-shadow: var(--shadow-sm);
}

.status-option-btn.selected,
.safe-option-btn.selected {
    border-color: var(--primary-color);
    background: var(--primary-color);
    color: white;
}

.status-icon,
.safe-icon {
    color: var(--primary-color);
}

.status-option-btn.selected .status-icon,
.safe-option-btn.selected .safe-icon {
    color: white;
}

/* Priority Container */
.priority-group {
    margin-top: 20px;
}

.priority-options-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 15px;
    margin-top: 10px;
}

.priority-item {
    display: flex;
}

.priority-btn {
    width: 100%;
    padding: 15px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
}

.priority-btn:hover {
    border-color: var(--primary-color);
    transform: translateY(-2px);
    box-shadow: var(--shadow-sm);
}

.priority-btn.selected {
    border-color: var(--primary-color);
    background: var(--primary-color);
    color: white;
}

.priority-icon {
    color: var(--primary-color);
}

.priority-btn.selected .priority-icon {
    color: white;
}

/* Signature Section */
.signature-section {
    margin-top: 30px;
    padding: 25px;
    background: var(--bg-light);
    border-radius: var(--radius-md);
}

.signature-section-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
}

.signature-type-btn {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    background: white;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 14px;
    font-weight: 500;
}

.signature-type-btn:hover {
    border-color: var(--primary-color);
    background: var(--bg-light);
}

.signature-field {
    margin-top: 20px;
    padding: 20px;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
}

.signature-field-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.signature-input {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    margin-bottom: 15px;
}

.signature-box {
    width: 100%;
    min-height: 150px;
    border: 2px dashed var(--border-color);
    border-radius: var(--radius-md);
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition-base);
    background: var(--bg-light);
}

.signature-box:hover {
    border-color: var(--primary-color);
    background: white;
}

.signature-box.has-signature {
    border-style: solid;
    border-color: var(--success-color);
}

.signature-preview {
    max-width: 100%;
    max-height: 140px;
}

.note-input-container {
    position: relative;
}

.voice-input-btn {
    position: absolute;
    right: 10px;
    top: 10px;
    background: var(--primary-color);
    color: white;
    border: none;
    width: 36px;
    height: 36px;
    border-radius: 50%;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    justify-content: center;
}

.voice-input-btn:hover {
    background: var(--primary-hover);
    transform: scale(1.1);
}


/* Auto-save indicator */
.auto-save-indicator {
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: var(--success-color);
    color: white;
    padding: 10px 20px;
    border-radius: var(--radius-lg);
    font-size: 13px;
    font-weight: 500;
    z-index: 1000;
    opacity: 0;
    transition: var(--transition-base);
    box-shadow: var(--shadow-md);
}

.auto-save-indicator.show {
    opacity: 1;
}

/* Modal Styles */
.custom-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: none;
    z-index: 10000;
    backdrop-filter: blur(4px);
    overflow-y: auto;
}

.custom-modal.show {
    display: block;
}

.custom-modal-content {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
    max-width: 600px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    animation: modalSlideIn 0.3s ease;
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    margin: 20px 0;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateX(-50%) scale(0.9);
    }
    to {
        opacity: 1;
        transform: translateX(-50%) scale(1);
    }
}

.custom-modal-header {
    background: var(--primary-color);
    padding: 20px 25px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid var(--border-light);
}

.custom-modal-header h3 {
    color: white;
    font-size: 20px;
    font-weight: 600;
    margin: 0;
}

.close-btn {
    background: none;
    border: none;
    color: white;
    font-size: 28px;
    cursor: pointer;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: var(--transition-base);
}

.close-btn:hover {
    background: rgba(255, 255, 255, 0.2);
}

.custom-modal-body {
    padding: 25px;
    overflow-y: auto;
    max-height: calc(80vh - 140px);
}

.modal-footer {
    padding: 20px 25px;
    border-top: 1px solid var(--border-light);
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}

/* Search Container */
.search-container {
    margin-bottom: 20px;
    display: flex;
    gap: 10px;
}

.search-input {
    flex: 1;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.search-btn {
    padding: 12px 20px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    justify-content: center;
}

.search-btn:hover {
    background: var(--primary-hover);
}

/* List Styles */
.equipment-list,
.checklist-list {
    max-height: 400px;
    overflow-y: auto;
}

.equipment-item,
.checklist-item {
    padding: 15px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    margin-bottom: 10px;
    cursor: pointer;
    transition: var(--transition-base);
    background: white;
}

.equipment-item:hover,
.checklist-item:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    transform: translateY(-2px);
    box-shadow: var(--shadow-sm);
}

.equipment-name,
.checklist-title {
    font-size: 15px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.equipment-details,
.checklist-description {
    font-size: 13px;
    color: var(--text-secondary);
    line-height: 1.5;
}

/* Success Modal */
.success-modal-content {
    max-width: 500px;
    text-align: center;
}

.success-container {
    padding: 40px 25px;
}

.success-icon {
    font-size: 64px;
    color: var(--success-color);
    margin-bottom: 20px;
}

.success-container h3 {
    font-size: 24px;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0 0 10px 0;
}

.success-container p {
    font-size: 16px;
    color: var(--text-secondary);
    margin: 0 0 30px 0;
}

.success-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
}

.success-buttons .btn {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 8px;
}

.success-buttons .btn-primary {
    background: var(--primary-color);
    color: white;
}

.success-buttons .btn-primary:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.success-buttons .btn-secondary {
    background: var(--secondary-color);
    color: var(--text-primary);
}

.success-buttons .btn-secondary:hover {
    background: #e6bd04;
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Loading Modal */
.loading-modal-content {
    max-width: 450px;
    text-align: center;
}

.loading-container {
    padding: 40px 25px;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 20px;
}

.loading-spinner {
    width: 60px;
    height: 60px;
    border: 4px solid var(--bg-secondary);
    border-top: 4px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-container h3 {
    color: var(--text-primary);
    font-size: 22px;
    font-weight: 600;
    margin: 0;
}

.loading-container p {
    color: var(--text-secondary);
    font-size: 14px;
    margin: 0;
}

.loading-steps {
    width: 100%;
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.loading-step {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 16px;
    background: var(--bg-light);
    border-radius: var(--radius-md);
    border-left: 4px solid var(--border-color);
    transition: var(--transition-base);
}

.loading-step.active {
    background: #e3f2fd;
    border-left-color: var(--primary-color);
}

.loading-step.completed {
    background: #e8f5e9;
    border-left-color: var(--success-color);
}

.loading-step i {
    color: var(--text-secondary);
    font-size: 18px;
}

.loading-step.active i {
    color: var(--primary-color);
}

.loading-step.completed i {
    color: var(--success-color);
}

.loading-step span {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
}

/* Signature Type Options */
.signature-type-option {
    padding: 15px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    margin-bottom: 10px;
    cursor: pointer;
    transition: var(--transition-base);
}

.signature-type-option:hover {
    border-color: var(--primary-color);
    background: var(--bg-light);
}

.signature-type-label {
    display: flex;
    justify-content: space-between;
    align-items: center;
    cursor: pointer;
}

.signature-type-label.selected {
    color: var(--primary-color);
}

.signature-type-text {
    font-size: 15px;
    font-weight: 500;
}

.signature-type-checkbox {
    width: 24px;
    height: 24px;
    border: 2px solid var(--border-color);
    border-radius: 4px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: transparent;
}

.signature-type-label.selected .signature-type-checkbox {
    border-color: var(--primary-color);
    background: var(--primary-color);
    color: white;
}

/* Signature Canvas */
.signature-canvas-container {
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    overflow: hidden;
    margin-bottom: 20px;
}

#signatureCanvas {
    width: 100%;
    height: 200px;
    cursor: crosshair;
    background: white;
}

.signature-controls {
    display: flex;
    gap: 10px;
    justify-content: flex-end;
}

/* Equipment Auto-fill Notification */
.equipment-autofill-notification {
    position: fixed;
    top: 80px;
    right: 20px;
    background: var(--success-color);
    color: white;
    padding: 15px 20px;
    border-radius: var(--radius-md);
    box-shadow: var(--shadow-md);
    max-width: 300px;
    z-index: 999;
    animation: slideInRight 0.3s ease;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

.notification-content {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.notification-content i {
    font-size: 20px;
}

.notification-content strong {
    font-weight: 600;
}

.notification-content small {
    font-size: 12px;
    opacity: 0.9;
}

/* Responsive Design */
@media (max-width: 768px) {
    .inspection2-header {
        padding: 15px 0;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 22px;
    }
    
    .create-btn {
        font-size: 14px;
        padding: 10px 16px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .form-header h2 {
        font-size: 20px;
    }
    
    .meter-reading-group {
        grid-template-columns: 1fr;
    }
    
    .gps-info-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }
    
    .gps-value {
        text-align: left;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .btn-next,
    .btn-back {
        width: 100%;
        justify-content: center;
        margin-left: 0 !important;
    }
    
    .header-inputs-grid {
        grid-template-columns: 1fr;
    }
    
    .status-options-container,
    .safe-options-container,
    .priority-options-container {
        grid-template-columns: 1fr;
    }
    
    .custom-modal-content {
        width: 95%;
        max-width: 95%;
    }
    
    .custom-modal-body {
        padding: 20px;
    }
    
    .success-buttons {
        flex-direction: column;
    }
    
    .success-buttons .btn {
        width: 100%;
    }
    
    .equipment-autofill-notification {
        right: 10px;
        left: 10px;
        max-width: none;
    }
}

@media (max-width: 480px) {
    .header-title {
        font-size: 18px;
    }
    
    .create-btn {
        font-size: 12px;
        padding: 8px 12px;
    }
    
    .form-section {
        padding: 15px;
    }
    
    .form-header h2 {
        font-size: 18px;
    }
    
    .btn-next,
    .btn-back {
        font-size: 14px;
        padding: 10px 20px;
    }
    
    .input-with-button {
        flex-direction: column;
    }
    
    .btn-select,
    .btn-clear {
        width: 100%;
    }
}

/* Utility Classes */
.hidden {
    display: none !important;
}

.text-center {
    text-align: center;
}

.text-success {
    color: var(--success-color);
}

.text-danger {
    color: var(--danger-color);
}

.text-warning {
    color: var(--warning-color);
}

.text-info {
    color: var(--info-color);
}

.mb-0 { margin-bottom: 0; }
.mb-1 { margin-bottom: 0.25rem; }
.mb-2 { margin-bottom: 0.5rem; }
.mb-3 { margin-bottom: 1rem; }
.mb-4 { margin-bottom: 1.5rem; }
.mb-5 { margin-bottom: 3rem; }

.mt-0 { margin-top: 0; }
.mt-1 { margin-top: 0.25rem; }
.mt-2 { margin-top: 0.5rem; }
.mt-3 { margin-top: 1rem; }
.mt-4 { margin-top: 1.5rem; }
.mt-5 { margin-top: 3rem; }

/* Empty section - Styles already defined above */
</style>
@endsection

@section('content')

<!-- Auto-save Indicator -->
<div class="auto-save-indicator" id="autoSaveIndicator">
    ✓ Auto-saved
</div>


<div class="inspection2-container">
    <!-- Header -->
    <div class="inspection2-header">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url('/inspection') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Vehicle Inspection</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="handleCreate()" style="display:none;" id="headerCreateBtn">
                    <i class="fas fa-save"></i>
                    Create Report
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Step 1 Form -->
    <div class="form-section" id="step1-form">
        <div class="form-header">
            <h2>1. Inspection Information</h2>
            <p>Fill in the basic information for your vehicle inspection</p>
        </div>
        
        <form id="inspection-form">
            @csrf
            <div class="form-grid">
                <!-- Left Column: Inspection Info -->
                <div class="form-column">
                    <div class="form-group">
                        <label for="reportNumber" class="form-label">Request Number</label>
                        <input type="text" class="form-control" id="reportNumber" name="reportNumber" placeholder="Select country and project first" readonly>
                    </div>
                    <div class="form-group">
                        <label for="country" class="form-label required">Country</label>
                        <select class="form-control" id="country" name="country" required onchange="updateProjects()">
                            <option value="">Select Country</option>
                            <option value="Zambia">Zambia</option>
                            <option value="DRC">DRC</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="project" class="form-label required">Project</label>
                        <select class="form-control" id="project" name="project" required disabled onchange="generateRequestNumber()">
                            <option value="">Select Project</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="inspector" class="form-label required">Inspector</label>
                        <input type="text" class="form-control" id="inspector" name="inspector" placeholder="Enter Inspector Name" required>
                    </div>
                </div>
                
                <!-- Right Column: Equipment and Checklist -->
                <div class="form-column">
                    <div class="form-group">
                        <label class="form-label required">Equipment</label>
                        <div class="input-with-button">
                            <input type="text" class="form-control" id="equipmentDisplay" name="equipmentDisplay" placeholder="Select Equipment" onclick="openEquipmentModal(event)" readonly required>
                            <button type="button" class="btn-select" onclick="openEquipmentModal(event)" title="Select Equipment">
                                <i class="fas fa-search"></i>
                            </button>
                            <button type="button" class="btn-clear" onclick="clearEquipmentAutoFill()" title="Clear Equipment" style="margin-left: 5px;">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        <input type="hidden" id="vehicleNumber" name="vehicleNumber" required>
                        <input type="hidden" id="vehicleName" name="vehicleName" required>
                        <input type="hidden" id="vinNumber" name="vinNumber" required>
                    </div>
                    <div class="form-group">
                        <label for="meterReading" class="form-label required">Meter Reading</label>
                        <div class="meter-reading-group">
                            <input type="text" class="form-control" id="meter-Reading-input" name="meterReading" placeholder="Enter new meter reading" required>
                            <select class="form-control" id="meterReading" name="meterUnit" required>
                                <option value="">Select Unit</option>
                                <option value="Miles">Miles</option>
                                <option value="Kilometers">Kilometers</option>
                                <option value="Hours">Hours</option>
                            </select>
                        </div>
                        <div class="meter-info">
                            <span class="meter-label">Current Reading:</span>
                            <span class="meter-value" id="currentMeterValue">Not selected</span>
                        </div>
                    </div>
                    <div class="form-group">
                        <label class="form-label required">Checklist</label>
                        <div class="input-with-button">
                            <input type="text" class="form-control" id="checklistDisplay" name="checklistDisplay" placeholder="Select Checklist" onclick="openChecklistModal(event)" readonly required>
                            <!-- <button type="button" class="btn-select" onclick="openChecklistModal(event)" title="Select Checklist">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M9 11H5a2 2 0 0 0-2 2v3c0 1.1.9 2 2 2h4m0-7v7m0-7h10a2 2 0 0 1 2 2v3c0 1.1-.9 2-2 2H9m0-7V9a2 2 0 0 1 2-2h2a2 2 0 0 1 2 2v2"/>
                                </svg>
                            </button> -->
                        </div>
                        <input type="hidden" id="inspectionChecklist" name="inspectionChecklist" required>
                    </div>
                </div>
            </div>
            
            <!-- GPS Location Section -->
            <div class="gps-section">
                <div class="form-group">
                    <label class="form-label">GPS Location</label>
                    <div class="gps-controls">
                        <button type="button" class="btn-gps" onclick="handleGetLocation()">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/>
                                <circle cx="12" cy="10" r="3"/>
                            </svg>
                            Get Location
                        </button>
                    </div>
                    <div class="gps-details" id="gpsDetailsBox" style="display: none;">
                        <div class="gps-info-grid">
                            <div class="gps-info-item">
                                <span class="gps-label">Coordinates:</span>
                                <span class="gps-value" id="coordinates">Not available</span>
                            </div>
                            <div class="gps-info-item">
                                <span class="gps-label">GPS Address:</span>
                                <span class="gps-value" id="gpsAddress">Not available</span>
                            </div>
                        </div>
                    </div>
                    <input type="hidden" id="coordinatesInput" name="coordinates">
                    <input type="hidden" id="gpsAddressInput" name="gpsAddress">
                </div>
            </div>
            
            <div class="form-actions">
                <button type="button" class="btn-next" onclick="goToStep2()">
                    <span>Next Step</span>
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M5 12h14M12 5l7 7-7 7"/>
                    </svg>
                </button>
            </div>
        </form>
    </div>

<!-- Step 2: Do Inspection -->
<div class="form-section" id="do-inspection-container" style="display: none;">
    <div class="form-header">
        <h2>2. Do Inspection</h2>
        <p>Complete the inspection checklist for your selected equipment</p>
        <div style="margin-top:10px; font-weight:600; color:#015185;">
            <span id="checklistTitleDisplay"></span>
        </div>
    </div>

    <!-- Header Information Form -->
    <div class="header-information-section mb-4">
        <div class="section-title-container">
            <h5 class="section-title">Header Information <span class="required-asterisk">*</span> Required</h5>
            <button type="button" class="btn-auto-fill" onclick="autoFillHeaderInfo()">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"/>
                    <path d="M21 3v5h-5"/>
                    <path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"/>
                    <path d="M3 21v-5h5"/>
                </svg>
                Auto Fill
            </button>
        </div>
        
        <div class="header-inputs-grid">
            <div class="input-group">
                <label for="modelInput" class="input-label">Model <span class="required-asterisk">*</span></label>
                <div class="input-with-icon">
                    <input type="text" class="form-input" id="modelInput" name="model" placeholder="Enter model" required>
                    <div class="input-icon">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                    </div>
                </div>
            </div>
            
            <div class="input-group">
                <label for="fleetNoInput" class="input-label">Fleet No <span class="required-asterisk">*</span></label>
                <div class="input-with-icon">
                    <input type="text" class="form-input" id="fleetNoInput" name="fleetNo" placeholder="Enter fleet number" value="LV1D" required>
                    <div class="input-icon">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                            <polyline points="3.27,6.96 12,12.01 20.73,6.96"/>
                            <line x1="12" y1="22.08" x2="12" y2="12"/>
                        </svg>
                    </div>
                </div>
            </div>
            
            <div class="input-group">
                <label for="shiftInput" class="input-label">Shift <span class="required-asterisk">*</span></label>
                <div class="input-with-icon">
                    <input type="text" class="form-input" id="shiftInput" name="shift" placeholder="Enter shift" required>
                    <div class="input-icon">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"/>
                            <polyline points="12,6 12,12 16,14"/>
                        </svg>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Dynamic header inputs from selected checklist will render here -->
    <div id="headerInputsContainer" style="margin-top:16px;"></div>

    <!-- Inspection Checklist -->
    <!-- Dynamic inspection sections from selected checklist will render here -->
    <div id="inspectionSectionsContainer" style="margin-top:20px;"></div>

    <div class="inspection-checklist-container" style="display:none;">
        <!-- Section A -->
        <div class="checklist-section">
            <div class="section-header" onclick="toggleSection('A')">
                <h4 class="section-title">A <span class="section-count">(0/6 selected)</span></h4>
                <svg class="section-arrow" id="arrow-A" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M6 9l6 6 6-6"/>
                </svg>
            </div>
            <div class="checklist-items" id="section-A">
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">1.</span>
                        <span class="item-name">DRIP TRAY / BUNDING</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('A1', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('A1', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('A1', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('A1')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('A1')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">2.</span>
                        <span class="item-name">5 METER AWAY FROM WATER SOURCE</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('A2', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('A2', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('A2', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('A2')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('A2')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">3.</span>
                        <span class="item-name">FIRE EXTINGUISHER & SIGNAGE</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('A3', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('A3', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('A3', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('A3')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('A3')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">4.</span>
                        <span class="item-name">ENGINE OIL LEVEL</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('A4', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('A4', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('A4', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('A4')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('A4')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">5.</span>
                        <span class="item-name">RADIATOR COOLANT (IF APPLICABLE)</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('A5', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('A5', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('A5', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('A5')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('A5')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">6.</span>
                        <span class="item-name">PRESSURE RELIEF VALVE</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('A6', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('A6', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('A6', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('A6')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('A6')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Section B -->
        <div class="checklist-section">
            <div class="section-header" onclick="toggleSection('B')">
                <h4 class="section-title">B <span class="section-count">(0/3 selected)</span></h4>
                <svg class="section-arrow" id="arrow-B" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M6 9l6 6 6-6"/>
                </svg>
            </div>
            <div class="checklist-items" id="section-B">
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">1.</span>
                        <span class="item-name">FUEL LEAKS</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('B1', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('B1', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('B1', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('B1')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('B1')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">2.</span>
                        <span class="item-name">ENGINE GUARD IN PLACE</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('B2', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('B2', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('B2', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('B2')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('B2')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">3.</span>
                        <span class="item-name">OIL LEAKS</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('B3', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('B3', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('B3', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('B3')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('B3')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Section C -->
        <div class="checklist-section">
            <div class="section-header" onclick="toggleSection('C')">
                <h4 class="section-title">C <span class="section-count">(0/13 selected)</span></h4>
                <svg class="section-arrow" id="arrow-C" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M6 9l6 6 6-6"/>
                </svg>
            </div>
            <div class="checklist-items" id="section-C">
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">1.</span>
                        <span class="item-name">V-BELTS</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C1', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C1', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C1', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C1')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C1')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">2.</span>
                        <span class="item-name">FUEL TANK CAP</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C2', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C2', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C2', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C2')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C2')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">3.</span>
                        <span class="item-name">EXHAUST BRACKET AND SILENCER</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C3', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C3', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C3', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C3')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C3')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">4.</span>
                        <span class="item-name">GAUGES WORKING</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C4', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C4', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C4', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C4')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C4')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">5.</span>
                        <span class="item-name">ENGINE MOUNTINGS</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C5', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C5', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C5', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C5')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C5')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">6.</span>
                        <span class="item-name">WORKING LIGHT</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C6', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C6', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C6', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C6')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C6')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">7.</span>
                        <span class="item-name">BATTERY WATER / TERMINALS & CLAMP</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C7', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C7', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C7', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C7')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C7')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">8.</span>
                        <span class="item-name">WATER LEAKS</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C8', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C8', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C8', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C8')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C8')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">9.</span>
                        <span class="item-name">PULLEY CONDITION</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C9', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C9', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C9', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C9')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C9')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">10.</span>
                        <span class="item-name">BELT CONDITION</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C10', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C10', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C10', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C10')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C10')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">11.</span>
                        <span class="item-name">MOUNTING BOLTS</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C11', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C11', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C11', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C11')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C11')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">12.</span>
                        <span class="item-name">CLUTCH & LEVER</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C12', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C12', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C12', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C12')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C12')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="checklist-item">
                    <div class="item-info">
                        <span class="item-number">13.</span>
                        <span class="item-name">ADAPTOR COUPLING</span>
                    </div>
                    <div class="item-actions">
                        <div class="status-buttons">
                            <button type="button" class="status-btn" data-status="ok" onclick="handleItemStatus('C13', 'ok')">Ok</button>
                            <button type="button" class="status-btn" data-status="reject" onclick="handleItemStatus('C13', 'reject')">Reject</button>
                            <button type="button" class="status-btn" data-status="na" onclick="handleItemStatus('C13', 'na')">N/A</button>
                        </div>
                        <div class="action-icons">
                            <button type="button" class="action-icon-btn" onclick="openNoteModal('C13')" title="Add Note">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                    <polyline points="14,2 14,8 20,8"/>
                                    <line x1="16" y1="13" x2="8" y2="13"/>
                                    <line x1="16" y1="17" x2="8" y2="17"/>
                                </svg>
                            </button>
                            <button type="button" class="action-icon-btn" onclick="openImageModal('C13')" title="Take Photo">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M23 19a2 2 0 0 1-2 2H3a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h4l2-3h6l2 3h4a2 2 0 0 1 2 2z"/>
                                    <circle cx="12" cy="13" r="4"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Navigation Buttons -->
    <div class="form-actions">
        <button type="button" class="btn-back" onclick="goToStep1()">
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M19 12H5M12 19l-7-7 7-7"/>
            </svg>
            <span>Back to Inspection Info</span>
        </button>
        <button type="button" class="btn-next" onclick="completeInspection()">
            <span>Inspection Complete</span>
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M5 12h14M12 5l7 7-7 7"/>
            </svg>
        </button>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="equipment-list" id="equipmentList">
                <!-- Equipment items will be loaded here -->
            </div>
        </div>
    </div>
</div>

<!-- Checklist Selection Modal -->
<div id="checklistModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Checklist</h3>
            <button type="button" class="close-btn" onclick="closeChecklistModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="checklistSearch" class="search-input" placeholder="Search checklists..." onkeyup="filterChecklist()">
                <button type="button" class="search-btn" onclick="filterChecklist()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="checklist-list" id="checklistList">
                <!-- Checklist items will be loaded here -->
            </div>
        </div>
    </div>
</div>


<!-- Step 3: Inspection Summary -->
<div id="Inspection-Summary-main" style="display: none;">

    <div class="form-header">
        <h2>3. Inspection Summary</h2>
    </div>
    
    <!-- Faulty Items Count -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="faulty-items-container">
                <div class="faulty-items-row">
                    <span class="faulty-items-label">Faulty Items:</span>
                    <span class="faulty-items-count" id="faultyItemsCount">0</span>
                </div>
            </div>
        </div>
    </div>

    <!-- Form Container with Padding -->
    <div class="form-container">
        <!-- First Row -->
        <div class="form-section">
            <!-- Overall Condition -->
            <div class="form-group">
                <div class="form-row">
                    <label for="overallCondition" class="form-label">Overall Condition: <span class="required-asterisk">*</span></label>
                    <div class="select-with-icon">
                       
                        <select class="form-select" id="overallCondition-select" name="overallCondition" required>
                            <option value="">Select</option>
                            <option value="Good Condition">Good Condition</option>
                            <option value="Minor Repair required">Minor Repair required</option>
                            <option value="Major Repair required">Major Repair required</option>
                            <option value="Cannot determine">Cannot determine</option>
                            <option value="Not Applicable">Not Applicable</option>
                        </select>
                    </div>
                </div>
                <div id="overallConditionError" class="error-text" style="display: none;">Overall Condition is required</div>
            </div>

            <!-- Equipment Status -->
            <div class="form-group">
                <div class="form-row">
                    <label class="form-label">Equipment Status: <span class="required-asterisk">*</span></label>
                    <div class="status-options-container">
                        <button type="button" class="status-option-btn" data-status="Available" onclick="handleStatusSelect('Available')">
                            <div class="status-icon">
                                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M9 12l2 2 4-4"/>
                                    <circle cx="12" cy="12" r="10"/>
                                </svg>
                            </div>
                            Available
                        </button>
                        <button type="button" class="status-option-btn" data-status="Maintenance Required" onclick="handleStatusSelect('Maintenance Required')">
                            <div class="status-icon">
                                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"/>
                                </svg>
                            </div>
                            Maintenance Required
                        </button>
                        <button type="button" class="status-option-btn" data-status="Breakdown" onclick="handleStatusSelect('Breakdown')">
                            <div class="status-icon">
                                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <circle cx="12" cy="12" r="10"/>
                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                </svg>
                            </div>
                            Breakdown
                        </button>
                    </div>
                </div>
                <div id="equipmentStatusError" class="error-text" style="display: none;">Equipment Status is required</div>
            </div>

            <!-- Priority (shown only when Maintenance Required or Breakdown) -->
            <div id="priorityContainer" class="form-group priority-group" style="display: none;">
                <div class="form-row">
                    <label class="form-label">Priority:</label>
                    <div class="priority-options-container">
                        <div class="priority-item">
                            <button type="button" class="priority-btn" data-priority="Low" onclick="handlePrioritySelect('Low')">
                                <div class="priority-icon">
                                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <circle cx="12" cy="12" r="10"/>
                                        <path d="M8 12h8"/>
                                    </svg>
                                </div>
                                <span>Low</span>
                            </button>
                        </div>
                        <div class="priority-item">
                            <button type="button" class="priority-btn" data-priority="Medium" onclick="handlePrioritySelect('Medium')">
                                <div class="priority-icon">
                                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M12 9v4"/>
                                        <path d="M12 17h.01"/>
                                        <circle cx="12" cy="12" r="10"/>
                                    </svg>
                                </div>
                                <span>Medium</span>
                            </button>
                        </div>
                        <div class="priority-item">
                            <button type="button" class="priority-btn" data-priority="High" onclick="handlePrioritySelect('High')">
                                <div class="priority-icon">
                                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                </div>
                                <span>High</span>
                            </button>
                        </div>
                        <div class="priority-item">
                            <button type="button" class="priority-btn" data-priority="Emergency" onclick="handlePrioritySelect('Emergency')">
                                <div class="priority-icon">
                                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                                        <path d="M12 9v4"/>
                                        <path d="M12 17h.01"/>
                                        <path d="M4 4l16 16"/>
                                    </svg>
                                </div>
                                <span>Emergency</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Equipment Safe to Use -->
            <div class="form-group">
                <div class="form-row">
                    <label class="form-label">Equipment Safe to Use: <span class="required-asterisk">*</span></label>
                    <div class="safe-options-container">
                        <button type="button" class="safe-option-btn" data-safe="Yes" onclick="handleSafeSelect('Yes')">
                            <div class="safe-icon">
                                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M9 12l2 2 4-4"/>
                                    <circle cx="12" cy="12" r="10"/>
                                </svg>
                            </div>
                            Yes
                        </button>
                        <button type="button" class="safe-option-btn" data-safe="No" onclick="handleSafeSelect('No')">
                            <div class="safe-icon">
                                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <circle cx="12" cy="12" r="10"/>
                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                </svg>
                            </div>
                            No
                        </button>
                    </div>
                </div>
                <div id="safeToUseError" class="error-text" style="display: none;">Equipment Safe to Use is required</div>
            </div>

            <!-- Additional Note -->
            <div class="form-group">
                <div class="form-row">
                    <label for="additionalNote" class="form-label">Additional Note:</label>
                    <div class="note-input-container">
                        <textarea class="form-control" id="additionalNote" name="additionalNote" rows="3" placeholder="Enter additional notes..."></textarea>
                        <button type="button" class="voice-input-btn" onclick="handleVoiceInput()" title="Voice Input">
                            <i class="fas fa-microphone"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Second Row - Signature Section -->
        <div class="form-section signature-section">
            <div class="signature-container">
                <h6 class="signature-section-title">Signature Information</h6>
                
                <!-- Signature Type Selection -->
                <div class="form-group">
                    <div class="form-row">
                        <label class="form-label">Signature Type: <span class="required-asterisk">*</span></label>
                        <button type="button" class="signature-type-btn" onclick="openSignatureTypeModal(event)">
                            <span id="selectedSignatureTypeText">Select Signature Type</span>
                            <i class="fas fa-chevron-down"></i>
                        </button>
                    </div>
                </div>

                <!-- Dynamic Signature Fields -->
                <div id="signatureFieldsContainer">
                    <!-- Signature fields will be dynamically loaded here -->
                </div>

                <!-- Signature Errors -->
                <div id="signatureErrorsContainer">
                    <!-- Signature errors will be displayed here -->
                </div>
            </div>
        </div>
    </div>

    <!-- Action Buttons -->
    <div class="form-actions">
        <button type="button" class="btn-back" onclick="goToStep2()">
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M19 12H5M12 19l-7-7 7-7"/>
            </svg>
            <span>Back to Do Inspection</span>
        </button>
        <button type="button" class="btn-next" id="createBtn" onclick="handleCreate()">
            <span>Generate Report</span>
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                <polyline points="14,2 14,8 20,8"/>
                <line x1="16" y1="13" x2="8" y2="13"/>
                <line x1="16" y1="17" x2="8" y2="17"/>
            </svg>
        </button>
    </div>
</div>

    </div><!-- End Main Content -->
</div><!-- End inspection2-container -->

<!-- Signature Type Selection Modal -->
<div id="signatureTypeModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Signature Type <span class="required-asterisk">*</span></h3>
            <button type="button" class="close-btn" onclick="closeSignatureTypeModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div id="signatureTypeOptions">
                <!-- Signature type options will be loaded here -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-primary" id="doneSignatureTypeBtn" onclick="confirmSignatureTypeSelection()" disabled>
                    Done (<span id="selectedCount">0</span> selected)
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Signature Modal -->
<div id="signatureModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3 id="signatureModalTitle">Signature</h3>
            <button type="button" class="close-btn" onclick="closeSignatureModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="signature-canvas-container">
                <canvas id="signatureCanvas" width="400" height="200"></canvas>
            </div>
            <div class="signature-controls">
                <button type="button" class="btn btn-secondary" onclick="clearSignature()">Clear</button>
                <button type="button" class="btn btn-primary" onclick="confirmSignature()">Confirm</button>
            </div>
        </div>
    </div>
</div>


<!-- Error Modal -->
<div id="errorModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header" style="background-color: #dc3545;">
            <h3>Error</h3>
        </div>
        <div class="custom-modal-body text-center">
            <i class="fas fa-exclamation-triangle text-danger" style="font-size: 48px; margin-bottom: 20px;"></i>
            <h5 id="errorModalTitle">An Error Occurred</h5>
            <p id="errorModalMessage">Please try again or contact support if the problem persists.</p>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeErrorModal()">Close</button>
        </div>
    </div>
</div>

<!-- Loading Modal -->
<div id="loadingModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content loading-modal-content">
        <div class="loading-container">
            <div class="loading-spinner"></div>
            <h3>Processing Report...</h3>
            <p>Please wait while we save your data and generate the PDF report</p>
            <div class="loading-steps">
                <div class="loading-step">
                    <i class="fas fa-database"></i>
                    <span>Saving to Database...</span>
                </div>
                <div class="loading-step">
                    <i class="fas fa-file-pdf"></i>
                    <span>Generating PDF...</span>
                </div>
                <div class="loading-step">
                    <i class="fas fa-cloud-upload-alt"></i>
                    <span>Uploading to Storage...</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content success-modal-content">
        <div class="success-container">
            <!-- Content will be dynamically updated by showReportSuccessModal() -->
            <div class="success-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3>Report Created Successfully!</h3>
            <p>Your inspection report has been generated and saved to the database.</p>
            <div class="success-buttons">
                <button type="button" class="btn btn-secondary" onclick="goBackToInspectionHome()" style="background: #ffd205; color: #212529; border: none; font-weight: 600;">
                    <i class="fas fa-arrow-left"></i> Back to Inspection Home
                </button>
                <button type="button" class="btn btn-primary" onclick="viewGeneratedPDF()" style="background: #0055a5; color: white; border: none; font-weight: 600;">
                    <i class="fas fa-file-pdf"></i> View PDF Report
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Sync Modal -->
<div id="syncModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-body text-center">
            <div class="sync-spinner">
                <i class="fas fa-sync-alt fa-spin" style="font-size: 48px; color: #0055A5;"></i>
            </div>
            <h5>Please Wait</h5>
            <p>Generating Report</p>
        </div>
    </div>
</div>

<!-- Voice Recognition Modal -->
<div id="voiceModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-body text-center">
            <i class="fas fa-microphone" style="font-size: 48px; color: #0055A5; margin-bottom: 20px;"></i>
            <h5>Listening...</h5>
            <button type="button" class="btn btn-danger" onclick="stopVoiceInput()">Stop</button>
        </div>
    </div>
</div>

<!-- Note Modal -->
<div id="noteModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Note</h3>
            <button type="button" class="close-btn" onclick="closeNoteModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="mb-3">
                <label for="noteText" class="form-label">Note Text:</label>
                <textarea class="form-control" id="noteText" rows="4" placeholder="Enter your note here..."></textarea>
            </div>
            <div class="d-flex justify-content-end gap-2">
                <button type="button" class="btn btn-secondary" onclick="closeNoteModal()">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveNote()">Save Note</button>
            </div>
        </div>
    </div>
</div>

<!-- Image Selection Modal -->
<div id="imageModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Image</h3>
            <button type="button" class="close-btn" onclick="closeImageModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="mb-3">
                <label class="form-label">Choose an option:</label>
                <div class="d-flex gap-2">
                    <button type="button" class="btn btn-outline-primary" onclick="takePicture()">
                        <i class="fas fa-camera"></i> Take Picture
                    </button>
                    <button type="button" class="btn btn-outline-secondary" onclick="chooseFromGallery()">
                        <i class="fas fa-images"></i> Choose from Gallery
                    </button>
                </div>
            </div>
            <input type="file" id="imageFileInput" accept="image/*" capture="environment" style="display: none;" onchange="handleImageFileSelect(event)">
            <div id="imagePreviewContainer" class="mb-3" style="display: none;">
                <label class="form-label">Image Preview:</label>
                <div class="text-center">
                    <img id="imagePreview" src="" alt="Preview" class="img-fluid" style="max-width: 100%; max-height: 200px; border: 1px solid #ddd; border-radius: 5px;">
                </div>
            </div>
            <div class="d-flex justify-content-end gap-2">
                <button type="button" class="btn btn-secondary" onclick="closeImageModal()">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveImage()" id="saveImageBtn" style="display: none;">Save Image</button>
            </div>
        </div>
    </div>
</div>

@endsection

  <script>
  // MongoDB-based Inspection System
  // All data comes from MongoDB APIs (no Firebase)
  
  // ===== MONGODB SAVE FUNCTIONS =====
  
  // Save inspection report to MongoDB
  async function saveInspectionToMongoDB(inspectionData) {
      try {
          console.log('📤 Starting MongoDB save process...');
          console.log('Inspection data:', inspectionData);
          
          const reportId = Date.now().toString();
          
          // Get current user ID from session
          const userId = '{{auth()->user()->email ?? "web_user"}}';
          
          // Prepare data structure for MongoDB
          const mongoData = {
              _id: reportId,
              userId: userId,
              requestNumber: inspectionData.requestNumber,
              status: 'completed',
              step1: {
                  id: reportId,
                  requestNumber: inspectionData.requestNumber,
                  country: inspectionData.country,
                  project: inspectionData.project,
                  inspector: inspectionData.inspector,
                  selectedEquipment: inspectionData.selectedEquipment,
                  meterReading: parseFloat(inspectionData.meterReading) || 0,
                  currentMeterReading: inspectionData.currentMeterReading || 0,
                  selectedChecklistId: inspectionData.selectedChecklistId,
                  checklistTitle: inspectionData.checklistTitle,
                  coordinates: inspectionData.coordinates || '',
                  gpsAddress: inspectionData.gpsAddress || '',
                  createdAt: new Date().toISOString()
              },
              step2: inspectionData.step2 || {},
              step3: inspectionData.step3 || {},
              createdAt: new Date().toISOString(),
              updatedAt: new Date().toISOString(),
              createdBy: userId,
              lastModifiedBy: userId
          };
          
          console.log('📦 MongoDB data structure:', mongoData);
          
          // Save to MongoDB via Laravel API
          const response = await fetch('/api/inspections/create', {
              method: 'POST',
              headers: {
                  'Content-Type': 'application/json',
                  'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
              },
              body: JSON.stringify(mongoData)
          });
          
          const result = await response.json();
          
          if (result.success) {
              console.log('✅ Report saved to MongoDB successfully:', reportId);
          return {
              success: true,
              reportId: reportId,
                  requestNumber: inspectionData.requestNumber,
                  data: result.data
              };
          } else {
              throw new Error(result.message || 'Failed to save to MongoDB');
          }
      } catch (error) {
          console.error('❌ Error saving to MongoDB:', error);
          throw new Error('Failed to save inspection data to MongoDB: ' + error.message);
      }
  }
  
  </script>
  
  <script>
// Add form field focus effects
document.querySelectorAll('.inspection2-main-form-control, .inspection2-main-form-select').forEach(field => {
    field.addEventListener('focus', function() {
        this.parentElement.classList.add('focused');
    });
    
    field.addEventListener('blur', function() {
        this.parentElement.classList.remove('focused');
    });
});

// Country and Project dependency
function updateProjects() {
    const country = document.getElementById('country').value;
    const projectSelect = document.getElementById('project');
    
    // Clear current projects
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    projectSelect.disabled = true;
    
    if (country === 'Zambia') {
        const projects = ['Kobold', 'Kalumbila', 'Kansanshi', 'Kimiteto', 'Mimosa'];
        projects.forEach(project => {
            const option = document.createElement('option');
            option.value = project;
            option.textContent = project;
            projectSelect.appendChild(option);
        });
        projectSelect.disabled = false;
    } else if (country === 'DRC') {
        const projects = ['Musompo', 'IME'];
        projects.forEach(project => {
            const option = document.createElement('option');
            option.value = project;
            option.textContent = project;
            projectSelect.appendChild(option);
        });
        projectSelect.disabled = false;
    }
    
    // Clear project selection when country changes
    projectSelect.value = '';
    
    // Generate request number
    generateRequestNumber();
}

// Generate request number in format: country-project-IR-number
async function generateRequestNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const reportNumberInput = document.getElementById('reportNumber');
    
    if (!country || !project) {
        reportNumberInput.value = '';
        reportNumberInput.placeholder = 'Select country and project first';
        return;
    }
    
    // Show loading state
    reportNumberInput.value = 'Generating...';
    reportNumberInput.placeholder = '';
    
    try {
        console.log('═══════════════════════════════════════════════════════');
        console.log('📋 GENERATING REQUEST NUMBER FROM MONGODB COUNT');
        console.log('═══════════════════════════════════════════════════════');
        
        // Generate country and project codes
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : country.substring(0, 3).toUpperCase();
        const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
        
        console.log('🏷️ Selected Country:', country);
        console.log('🏷️ Selected Project:', project);
        console.log('🏷️ Country Code:', countryCode);
        console.log('🏷️ Project Code:', projectCode);
        
        // Fetch ACTUAL count from MongoDB collection
        console.log('📊 Fetching ACTUAL count from MongoDB collection...');
        console.log('📊 API URL: https://api.titandrillingzm.com:6004/inspections/count');
        
        const response = await fetch('https://api.titandrillingzm.com:6004/inspections/count');
        const result = await response.json();
        
        console.log('📊 MongoDB API Response:', JSON.stringify(result, null, 2));
        
        let actualCount = 0;
        
        if (result && result.success === true) {
            if (typeof result.total === 'number') {
                actualCount = result.total;
                console.log('✅ ACTUAL Total Count from MongoDB:', actualCount);
            } else if (typeof result.count === 'number') {
                actualCount = result.count;
                console.log('✅ ACTUAL Total Count from MongoDB:', actualCount);
            } else {
                console.warn('⚠️ Response success but no valid count. Response:', result);
                actualCount = 0;
            }
        } else {
            console.error('❌ MongoDB API returned unsuccessful response:', result);
            actualCount = 0;
        }
        
        // New number = Actual MongoDB count + 1
        const newNumber = actualCount + 1;
        const newRequestNumber = `${countryCode}-${projectCode}-IR-${String(newNumber).padStart(3, '0')}`;
        
        console.log('═══════════════════════════════════════════════════════');
        console.log('✅ REQUEST NUMBER GENERATED');
        console.log('📊 Current Total in MongoDB:', actualCount);
        console.log('🔢 Next Number:', newNumber, '(Current Total + 1)');
        console.log('📋 Generated Request Number:', newRequestNumber);
        console.log('📝 Format: {COUNTRY}-{PROJECT}-IR-{NUMBER}');
        console.log('═══════════════════════════════════════════════════════');
        
        reportNumberInput.value = newRequestNumber;
        reportNumberInput.placeholder = '';
        
    } catch (error) {
        console.error('❌ Error generating request number:', error);
        console.error('❌ Error details:', error.message);
        console.error('❌ Error stack:', error.stack);
        
        // Fallback: Use country-project-IR-001
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : country.substring(0, 3).toUpperCase();
        const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
        const fallback = `${countryCode}-${projectCode}-IR-001`;
        
        console.log('⚠️ Using fallback:', fallback);
        reportNumberInput.value = fallback;
        reportNumberInput.placeholder = 'Error - using fallback';
    }
}

// GPS Location functionality
function handleGetLocation() {
    if (navigator.geolocation) {
        const gpsDetailsBox = document.getElementById('gpsDetailsBox');
        gpsDetailsBox.style.display = 'block';
        
        // Show loading state
        document.getElementById('coordinates').textContent = 'Getting location...';
        document.getElementById('gpsAddress').textContent = 'Getting address...';
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const latitude = position.coords.latitude;
                const longitude = position.coords.longitude;
                const coordinates = `${latitude}, ${longitude}`;
                
                // Update coordinates display
                document.getElementById('coordinates').textContent = coordinates;
                document.getElementById('coordinatesInput').value = coordinates;
                
                // Try to get address from coordinates
                fetchAddressFromCoordinates(latitude, longitude);
            },
            function(error) {
                console.error('Error getting location:', error);
                document.getElementById('coordinates').textContent = 'Location access denied';
                document.getElementById('gpsAddress').textContent = 'Address unavailable';
            },
            {
                enableHighAccuracy: true,
                timeout: 15000,
                maximumAge: 10000
            }
        );
    } else {
        alert('Geolocation is not supported by this browser.');
    }
}

async function fetchAddressFromCoordinates(latitude, longitude) {
    try {
        // Using a free reverse geocoding service (you may want to use Google Maps API with your own key)
        const response = await fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${latitude}&lon=${longitude}&zoom=18&addressdetails=1`);
        const data = await response.json();
        
        if (data.display_name) {
            const address = data.display_name;
            document.getElementById('gpsAddress').textContent = address;
            document.getElementById('gpsAddressInput').value = address;
        } else {
            document.getElementById('gpsAddress').textContent = 'Address not found';
        }
    } catch (error) {
        console.error('Error fetching address:', error);
        document.getElementById('gpsAddress').textContent = 'Address fetch failed';
    }
}

// Meter reading validation
function validateMeterReading() {
    const meterInput = document.getElementById('meter-Reading-input');
    const currentMeterValue = document.getElementById('currentMeterValue');
    
    if (!meterInput.value || !currentMeterValue || !currentMeterValue.textContent || currentMeterValue.textContent === 'Not selected') {
        return true; // Skip validation if no equipment selected or no current reading
    }
    
    const newReading = parseFloat(meterInput.value);
    const currentReadingText = currentMeterValue.textContent;
    const currentReading = parseFloat(currentReadingText.replace(/[^\d.]/g, ''));
    
    if (isNaN(newReading) || isNaN(currentReading)) {
        return true; // Skip validation if parsing fails
    }
    
    if (newReading <= currentReading) {
        // Show error message
        showMeterReadingError(`New meter reading (${newReading}) must be greater than current reading (${currentReading})`);
        return false;
    }
    
    // Clear any previous error
    clearMeterReadingError();
    return true;
}

function showMeterReadingError(message) {
    // Remove any existing error
    clearMeterReadingError();
    
    const meterInput = document.getElementById('meter-Reading-input');
    const errorDiv = document.createElement('div');
    errorDiv.className = 'meter-reading-error';
    errorDiv.style.color = '#dc3545';
    errorDiv.style.fontSize = '12px';
    errorDiv.style.marginTop = '5px';
    errorDiv.textContent = message;
    
    meterInput.parentElement.appendChild(errorDiv);
    meterInput.classList.add('is-invalid');
}

function clearMeterReadingError() {
    const existingError = document.querySelector('.meter-reading-error');
    if (existingError) {
        existingError.remove();
    }
    
    const meterInput = document.getElementById('meter-Reading-input');
    meterInput.classList.remove('is-invalid');
}

// Add meter reading validation on input change
document.addEventListener('DOMContentLoaded', async function() {
    console.log('Inspection2 page loaded');
    
    // Initialize page with URL-based step navigation
    await initializePage();
    
    // Sidebar toggle is now handled globally in welcome.blade.php
    // No need for page-specific sidebar toggle code
    
    const meterInput = document.getElementById('meter-Reading-input');
    if (meterInput) {
        meterInput.addEventListener('input', function() {
            if (this.value) {
                validateMeterReading();
            } else {
                clearMeterReadingError();
            }
        });
        
        meterInput.addEventListener('blur', function() {
            if (this.value) {
                validateMeterReading();
            }
        });
    }
});

// Do Inspection Functions

// Auto Fill Header Information
function autoFillHeaderInfo() {
    // Auto-fill with sample data
    document.getElementById('modelInput').value = 'CAT 320D';
    document.getElementById('fleetNoInput').value = 'LV1D';
    document.getElementById('shiftInput').value = 'Day Shift';
    
    // Show success message
    const button = document.querySelector('.btn-auto-fill');
    const originalText = button.innerHTML;
    button.innerHTML = `
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M9 12l2 2 4-4"/>
            <circle cx="12" cy="12" r="10"/>
        </svg>
        Auto Filled
    `;
    button.style.background = '#10b981';
    
    setTimeout(() => {
        button.innerHTML = originalText;
        button.style.background = '#3b82f6';
    }, 2000);
}

// Handle image upload and preview
document.addEventListener('DOMContentLoaded', function() {
    const imageUpload = document.getElementById('imageUpload');
    if (imageUpload) {
        imageUpload.addEventListener('change', function(event) {
            const file = event.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const imagePreview = document.getElementById('imagePreview');
                    if (imagePreview) {
                        imagePreview.src = e.target.result;
                    }
                };
                reader.readAsDataURL(file);
            }
        });
    }

    // Initialize do inspection expandable rows
    const expandableRows = document.querySelectorAll('.do-inspection-expandable-row');
    expandableRows.forEach(row => {
        row.addEventListener('click', function() {
            const icon = this.querySelector('.do-inspection-toggle-icon');
            if (icon) {
                const isExpanded = this.getAttribute('aria-expanded') === 'true';
                icon.classList.toggle('bi-chevron-down', !isExpanded);
                icon.classList.toggle('bi-chevron-up', isExpanded);
            }
        });
    });
    
    
    // Load initial data
    loadEquipmentData();
    loadChecklistData();
});

// Step 3: Inspection Summary Functions

// Global variables for Step 3
let step3Data = {
    overallCondition: '',
    equipmentStatus: '',
    safeToUse: '',
    additionalNote: '',
    signatureFields: [],
    selectedSignatureTypes: [],
    signatureValues: {},
    individualSignatures: {},
    individualSignatureStatus: {}
};

let currentSignatureField = null;
let currentSignatureType = null;
let signatureCanvas = null;
let isDrawing = false;
let lastX = 0;
let lastY = 0;

// Load faulty items count from Step 2
function loadFaultyItemsCount() {
    try {
        const selections = inspectionSelections || {};
        let faultyCount = 0;
        
        Object.values(selections).forEach(selection => {
            if (selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok') {
                faultyCount++;
            }
        });
        
        document.getElementById('faultyItemsCount').textContent = faultyCount;
        return faultyCount;
    } catch (error) {
        console.error('Error loading faulty items count:', error);
        return 0;
    }
}

// Update faulty count dynamically (can be called from anywhere)
function updateFaultyCount() {
    try {
        const selections = inspectionSelections || {};
        let faultyCount = 0;
        
        Object.values(selections).forEach(selection => {
            if (selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok') {
                faultyCount++;
            }
        });
        
        // Update the display if step 3 is visible
        const faultyCountElement = document.getElementById('faultyItemsCount');
        if (faultyCountElement) {
            faultyCountElement.textContent = faultyCount;
        }
        
        return faultyCount;
    } catch (error) {
        console.error('Error updating faulty items count:', error);
        return 0;
    }
}

// Handle status selection
function handleStatusSelect(status) {
    step3Data.equipmentStatus = status;
    
    // Update button states
    document.querySelectorAll('.status-option-btn').forEach(btn => {
        btn.classList.remove('selected');
        if (btn.dataset.status === status) {
            btn.classList.add('selected');
        }
    });
    
    // Show/hide priority container
    const priorityContainer = document.getElementById('priorityContainer');
    if (status === 'Maintenance Required' || status === 'Breakdown') {
        priorityContainer.style.display = 'flex';
    } else {
        priorityContainer.style.display = 'none';
        step3Data.priority = '';
        // Clear priority button selections
        document.querySelectorAll('.priority-btn').forEach(btn => {
            btn.classList.remove('selected');
        });
    }
    
    // Clear error
    document.getElementById('equipmentStatusError').style.display = 'none';
}

// Handle priority selection
function handlePrioritySelect(priority) {
    step3Data.priority = priority;
    
    // Update button states
    document.querySelectorAll('.priority-btn').forEach(btn => {
        btn.classList.remove('selected');
        if (btn.dataset.priority === priority) {
            btn.classList.add('selected');
        }
    });
}

// Handle safe selection
function handleSafeSelect(safe) {
    step3Data.safeToUse = safe;
    
    // Update button states
    document.querySelectorAll('.safe-option-btn').forEach(btn => {
        btn.classList.remove('selected');
        if (btn.dataset.safe === safe) {
            btn.classList.add('selected');
        }
    });
    
    // Clear error
    document.getElementById('safeToUseError').style.display = 'none';
}

// Handle overall condition change
document.addEventListener('DOMContentLoaded', function() {
    const overallConditionSelect = document.getElementById('overallCondition-select');
    if (overallConditionSelect) {
        overallConditionSelect.addEventListener('change', function() {
            step3Data.overallCondition = this.value;
            document.getElementById('overallConditionError').style.display = 'none';
        });
    }
    
    // Handle additional note change
    const additionalNoteTextarea = document.getElementById('additionalNote');
    if (additionalNoteTextarea) {
        additionalNoteTextarea.addEventListener('input', function() {
            step3Data.additionalNote = this.value;
        });
    }
});

// Voice input functionality
function handleVoiceInput() {
    if ('webkitSpeechRecognition' in window || 'SpeechRecognition' in window) {
        const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
        const recognition = new SpeechRecognition();
        
        recognition.continuous = false;
        recognition.interimResults = false;
        recognition.lang = 'en-US';
        
        // Show voice modal
        openModal('voiceModal');
        
        recognition.onstart = function() {
            console.log('Voice recognition started');
        };
        
        recognition.onresult = function(event) {
            const transcript = event.results[0][0].transcript;
            const additionalNoteTextarea = document.getElementById('additionalNote');
            if (additionalNoteTextarea) {
                additionalNoteTextarea.value = transcript;
                step3Data.additionalNote = transcript;
            }
        };
        
        recognition.onerror = function(event) {
            console.error('Voice recognition error:', event.error);
            alert('Voice recognition error: ' + event.error);
            closeVoiceModal();
        };
        
        recognition.onend = function() {
            closeVoiceModal();
        };
        
        recognition.start();
    } else {
        alert('Speech recognition is not supported in this browser.');
    }
}

function stopVoiceInput() {
    closeVoiceModal();
}

function closeVoiceModal() {
    closeModal('voiceModal');
}

// Signature functionality
function openSignatureTypeModal(event) {
    console.log('Opening signature type modal. Current selections:', step3Data.selectedSignatureTypes);
    
    // Load signature type options from checklist
    loadSignatureTypeOptions();
    
    // Show the modal
    openModal('signatureTypeModal', event);
    
    // Ensure the UI is properly updated with current selections
    setTimeout(() => {
        updateSignatureTypeSelectionUI();
        updateSignatureTypeText();
    }, 100);
}

function updateSignatureTypeText() {
    const selectedTextElement = document.getElementById('selectedSignatureTypeText');
    if (selectedTextElement && step3Data.selectedSignatureTypes && step3Data.selectedSignatureTypes.length > 0) {
        const selectedText = step3Data.selectedSignatureTypes.map(type => getSignatureTypeLabel(type)).join(' & ');
        selectedTextElement.textContent = selectedText;
        console.log('Updated signature type text to:', selectedText);
    } else if (selectedTextElement) {
        selectedTextElement.textContent = 'Select Signature Type';
        console.log('Reset signature type text to default');
    }
}

function closeSignatureTypeModal() {
    closeModal('signatureTypeModal');
}

function loadSignatureTypeOptions() {
    const container = document.getElementById('signatureTypeOptions');
    
    // Default signature types (you can load these from checklist data)
    const defaultSignatureTypes = [
        { key: 'driller', label: 'Driller Signature', required: true },
        { key: 'mechanic', label: 'Mechanic Signature', required: true },
        { key: 'supervisor', label: 'Supervisor Signature', required: true },
        { key: 'operator', label: 'Operator Signature', required: true },
        { key: 'inspector', label: 'Inspector Signature', required: true },
        { key: 'engineer', label: 'Engineer Signature', required: true },
        { key: 'technician', label: 'Technician Signature', required: true },
        { key: 'manager', label: 'Manager Signature', required: true }
    ];
    
    // Only recreate HTML if container is empty or doesn't exist
    if (!container || container.children.length === 0) {
        const html = defaultSignatureTypes.map(type => `
            <div class="signature-type-option" onclick="toggleSignatureTypeSelection('${type.key}')">
                <input type="checkbox" id="signature_${type.key}" data-key="${type.key}" style="display: none;">
                <label for="signature_${type.key}" class="signature-type-label">
                    <span class="signature-type-text">${type.label}</span>
                    <span class="signature-type-checkbox">
                        <i class="fas fa-check"></i>
                    </span>
                </label>
            </div>
        `).join('');
        
        container.innerHTML = html;
    }
    
    // Always sync checkboxes with current selection
    syncSignatureTypeCheckboxes();
}

function syncSignatureTypeCheckboxes() {
    console.log('Syncing checkboxes with selected types:', step3Data.selectedSignatureTypes);
    
    // Clear all checkboxes first
    document.querySelectorAll('#signatureTypeOptions input[type="checkbox"]').forEach(checkbox => {
        checkbox.checked = false;
        const label = checkbox.nextElementSibling;
        if (label) {
            label.classList.remove('selected');
        }
    });
    
    // Check the ones that are in the selected array
    if (step3Data.selectedSignatureTypes && step3Data.selectedSignatureTypes.length > 0) {
        step3Data.selectedSignatureTypes.forEach(key => {
            const checkbox = document.getElementById(`signature_${key}`);
            const label = checkbox?.nextElementSibling;
            
            if (checkbox && label) {
                checkbox.checked = true;
                label.classList.add('selected');
                console.log('Checked signature type:', key);
            } else {
                console.warn('Could not find checkbox or label for:', key);
            }
        });
    }
    
    console.log('Checkbox sync completed. Selected types:', step3Data.selectedSignatureTypes);
    updateSignatureTypeSelectionUI();
}

function toggleSignatureTypeSelection(key) {
    const checkbox = document.getElementById(`signature_${key}`);
    const label = checkbox.nextElementSibling;
    
    // Toggle the checkbox state
    checkbox.checked = !checkbox.checked;
    
    if (checkbox.checked) {
        // Add to selection
        label.classList.add('selected');
        if (!step3Data.selectedSignatureTypes.includes(key)) {
            step3Data.selectedSignatureTypes.push(key);
        }
    } else {
        // Remove from selection
        label.classList.remove('selected');
        step3Data.selectedSignatureTypes = step3Data.selectedSignatureTypes.filter(k => k !== key);
    }
    
    console.log('Toggled signature type:', key, 'checked:', checkbox.checked, 'selected types:', step3Data.selectedSignatureTypes);
    updateSignatureTypeSelectionUI();
}

function updateSignatureTypeSelectionUI() {
    const selectedCount = step3Data.selectedSignatureTypes.length;
    const doneBtn = document.getElementById('doneSignatureTypeBtn');
    const selectedCountSpan = document.getElementById('selectedCount');
    
    console.log('Updating signature selection UI - selected count:', selectedCount, 'selected types:', step3Data.selectedSignatureTypes);
    
    if (selectedCountSpan) {
        selectedCountSpan.textContent = selectedCount;
    }
    
    if (doneBtn) {
        doneBtn.disabled = selectedCount === 0;
    }
    
    // Verify the count matches the actual checked checkboxes
    const checkedCheckboxes = document.querySelectorAll('#signatureTypeOptions input[type="checkbox"]:checked');
    console.log('Actual checked checkboxes count:', checkedCheckboxes.length);
    
    if (checkedCheckboxes.length !== selectedCount) {
        console.warn('Count mismatch! Selected types:', selectedCount, 'Checked checkboxes:', checkedCheckboxes.length);
    }
}

function confirmSignatureTypeSelection() {
    if (step3Data.selectedSignatureTypes.length === 0) {
        alert('Please select at least one signature type.');
        return;
    }
    
    console.log('Confirming signature type selection:', step3Data.selectedSignatureTypes);
    
    // Clear existing signature fields first
    clearSignatureFields();
    
    // Generate signature fields based on selection
    generateSignatureFields();
    
    // Update the signature type text
    updateSignatureTypeText();
    
    closeSignatureTypeModal();
}

function clearSignatureFields() {
    const container = document.getElementById('signatureFieldsContainer');
    if (container) {
        container.innerHTML = '';
    }
    
    // Clear signature type text
    const selectedTextElement = document.getElementById('selectedSignatureTypeText');
    if (selectedTextElement) {
        selectedTextElement.textContent = 'Select Signature Type';
    }
    
    // Reset the generation flag
    signatureFieldsGenerated = false;
}

function resetSignatureTypeSelection() {
    // Clear selected signature types
    step3Data.selectedSignatureTypes = [];
    
    // Clear signature fields
    clearSignatureFields();
    
    // Reset signature type modal checkboxes
    document.querySelectorAll('#signatureTypeOptions input[type="checkbox"]').forEach(checkbox => {
        checkbox.checked = false;
        const label = checkbox.nextElementSibling;
        if (label) {
            label.classList.remove('selected');
        }
    });
    
    // Update signature type selection UI
    updateSignatureTypeSelectionUI();
}

function forceRegenerateSignatureFields() {
    console.log('Force regenerating signature fields...');
    clearSignatureFields();
    setTimeout(() => {
        generateSignatureFields();
    }, 100);
}


function fixSignatureTypeCount() {
    console.log('Fixing signature type count...');
    
    // Get all checked checkboxes
    const checkedCheckboxes = document.querySelectorAll('#signatureTypeOptions input[type="checkbox"]:checked');
    const checkedKeys = Array.from(checkedCheckboxes).map(cb => cb.dataset.key);
    
    console.log('Checked checkboxes keys:', checkedKeys);
    console.log('Current selectedSignatureTypes:', step3Data.selectedSignatureTypes);
    
    // Update the selectedSignatureTypes array to match the actual checked checkboxes
    step3Data.selectedSignatureTypes = checkedKeys;
    
    console.log('Updated selectedSignatureTypes:', step3Data.selectedSignatureTypes);
    
    // Update the UI
    updateSignatureTypeSelectionUI();
}

function restoreSignatureTypeSelections() {
    console.log('Restoring signature type selections from saved data...');
    
    // This function will be called when restoring data from localStorage
    if (step3Data.selectedSignatureTypes && step3Data.selectedSignatureTypes.length > 0) {
        console.log('Found saved signature types:', step3Data.selectedSignatureTypes);
        
        // Update the signature type text
        updateSignatureTypeText();
        
        // If modal is open, sync the checkboxes
        const modal = document.getElementById('signatureTypeModal');
        if (modal && modal.style.display === 'block') {
            syncSignatureTypeCheckboxes();
        }
    }
}

function generateSignatureFields() {
    const container = document.getElementById('signatureFieldsContainer');
    
    if (!container) {
        console.error('Signature fields container not found');
        return;
    }
    
    // Check if signature fields are already generated to prevent duplicates
    if (signatureFieldsGenerated && container.children.length > 0) {
        console.log('Signature fields already generated, skipping...');
        return;
    }
    
    // Clear existing content to prevent duplication
    container.innerHTML = '';
    
    // Check if we have selected signature types
    if (!step3Data.selectedSignatureTypes || step3Data.selectedSignatureTypes.length === 0) {
        console.log('No signature types selected');
        signatureFieldsGenerated = false;
        return;
    }
    
    console.log('Generating signature fields for:', step3Data.selectedSignatureTypes);
    
    // Build signature fields array (matches React Native structure)
    step3Data.signatureFields = step3Data.selectedSignatureTypes.map(type => ({
        key: type,
        label: getSignatureTypeLabel(type).toUpperCase(),
        nameKey: `${type}_name`,
        signatureKey: type,
        required: true,
        roleName: getSignatureTypeLabel(type),
        showInput: true,
        showSignature: true
    }));
    
    console.log('📝 Created signatureFields structure:', step3Data.signatureFields);
    
    const html = step3Data.selectedSignatureTypes.map(type => {
        const signatureType = getSignatureTypeLabel(type);
        return `
            <div class="signature-field">
                <label class="signature-field-label">${signatureType} Name:</label>
                <input type="text" class="signature-input" 
                       placeholder="Enter ${signatureType.toLowerCase()} name"
                       value="${step3Data.signatureValues[`${type}_name`] || ''}"
                       onchange="updateSignatureValue('${type}_name', this.value)">
                
                <label class="signature-field-label">${signatureType} Signature:</label>
                <div class="signature-box" onclick="openSignatureModal('${type}', event)" id="signature_box_${type}">
                    <div id="signature_content_${type}">
                        ${step3Data.individualSignatures[type] 
                            ? `<img src="${step3Data.individualSignatures[type]}" alt="Signature" class="signature-preview">
                               <p style="margin: 5px 0; color: #28a745; font-size: 12px;">Signature Done</p>
                               <button type="button" class="btn btn-sm btn-outline-secondary" onclick="openSignatureModal('${type}', event)">Upload again</button>`
                            : `<i class="fas fa-plus" style="font-size: 24px; color: #ccc;"></i>
                               <p style="margin: 5px 0; color: #666;">Tap to add signature</p>`
                        }
                    </div>
                </div>
            </div>
        `;
    }).join('');
    
    container.innerHTML = html;
    
    // Mark as generated
    signatureFieldsGenerated = true;
    
    // Update signature type text
    const selectedText = step3Data.selectedSignatureTypes.map(type => getSignatureTypeLabel(type)).join(' & ');
    const selectedTextElement = document.getElementById('selectedSignatureTypeText');
    if (selectedTextElement) {
        selectedTextElement.textContent = selectedText;
    }
    
    console.log('✅ Signature fields generated successfully with proper structure');
}

function getSignatureTypeLabel(key) {
    const labels = {
        'driller': 'Driller',
        'mechanic': 'Mechanic',
        'supervisor': 'Supervisor',
        'operator': 'Operator',
        'inspector': 'Inspector',
        'engineer': 'Engineer',
        'technician': 'Technician',
        'manager': 'Manager'
    };
    return labels[key] || key;
}

function updateSignatureValue(key, value) {
    step3Data.signatureValues[key] = value;
}

function openSignatureModal(type, event) {
    currentSignatureField = type;
    currentSignatureType = getSignatureTypeLabel(type);
    document.getElementById('signatureModalTitle').textContent = `${currentSignatureType} Signature`;
    openModal('signatureModal', event);
    
    // Initialize signature canvas
    initSignatureCanvas();
}

function closeSignatureModal() {
    closeModal('signatureModal');
    currentSignatureField = null;
    currentSignatureType = null;
}

function initSignatureCanvas() {
    const canvas = document.getElementById('signatureCanvas');
    const ctx = canvas.getContext('2d');
    
    // Set canvas size
    canvas.width = 400;
    canvas.height = 200;
    
    // Set canvas style
    ctx.strokeStyle = '#000';
    ctx.lineWidth = 2;
    ctx.lineCap = 'round';
    
    // Clear canvas
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    
    // Event listeners
    canvas.addEventListener('mousedown', startDrawing);
    canvas.addEventListener('mousemove', draw);
    canvas.addEventListener('mouseup', stopDrawing);
    canvas.addEventListener('mouseout', stopDrawing);
    
    // Touch events for mobile
    canvas.addEventListener('touchstart', handleTouch);
    canvas.addEventListener('touchmove', handleTouch);
    canvas.addEventListener('touchend', stopDrawing);
    
    signatureCanvas = { canvas, ctx };
}

function startDrawing(e) {
    isDrawing = true;
    const rect = signatureCanvas.canvas.getBoundingClientRect();
    lastX = e.clientX - rect.left;
    lastY = e.clientY - rect.top;
}

function draw(e) {
    if (!isDrawing) return;
    
    const rect = signatureCanvas.canvas.getBoundingClientRect();
    const currentX = e.clientX - rect.left;
    const currentY = e.clientY - rect.top;
    
    signatureCanvas.ctx.beginPath();
    signatureCanvas.ctx.moveTo(lastX, lastY);
    signatureCanvas.ctx.lineTo(currentX, currentY);
    signatureCanvas.ctx.stroke();
    
    lastX = currentX;
    lastY = currentY;
}

function stopDrawing() {
    isDrawing = false;
}

function handleTouch(e) {
    e.preventDefault();
    const touch = e.touches[0];
    const mouseEvent = new MouseEvent(e.type === 'touchstart' ? 'mousedown' : 
                                    e.type === 'touchmove' ? 'mousemove' : 'mouseup', {
        clientX: touch.clientX,
        clientY: touch.clientY
    });
    signatureCanvas.canvas.dispatchEvent(mouseEvent);
}

function clearSignature() {
    if (signatureCanvas) {
        signatureCanvas.ctx.clearRect(0, 0, signatureCanvas.canvas.width, signatureCanvas.canvas.height);
    }
}

function confirmSignature() {
    if (!currentSignatureField) return;
    
    // Convert canvas to data URL
    const signatureData = signatureCanvas.canvas.toDataURL();
    
    // Store signature
    step3Data.individualSignatures[currentSignatureField] = signatureData;
    step3Data.individualSignatureStatus[currentSignatureField] = true;
    
    // Update UI
    updateSignatureBox(currentSignatureField, signatureData);
    
    closeSignatureModal();
}

function updateSignatureBox(type, signatureData) {
    const signatureBox = document.getElementById(`signature_box_${type}`);
    const signatureContent = document.getElementById(`signature_content_${type}`);
    
    if (signatureBox && signatureContent) {
        signatureBox.classList.add('has-signature');
        signatureContent.innerHTML = `
            <img src="${signatureData}" alt="Signature" class="signature-preview">
            <p style="margin: 5px 0; color: #28a745; font-size: 12px;">Signature Done</p>
            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="openSignatureModal('${type}', event)">
                Upload again
            </button>
        `;
    }
}

// Validation functions
function validateStep3() {
    let isValid = true;
    
    // Validate overall condition
    if (!step3Data.overallCondition) {
        document.getElementById('overallConditionError').style.display = 'block';
        isValid = false;
    }
    
    // Validate equipment status
    if (!step3Data.equipmentStatus) {
        document.getElementById('equipmentStatusError').style.display = 'block';
        isValid = false;
    }
    
    // Validate safe to use
    if (!step3Data.safeToUse) {
        document.getElementById('safeToUseError').style.display = 'block';
        isValid = false;
    }
    
    // Validate signatures
    if (!validateSignatures()) {
        isValid = false;
    }
    
    return isValid;
}


function validateSignatures() {
    if (step3Data.selectedSignatureTypes.length === 0) {
        alert('Please select at least one signature type.');
        return false;
    }
    
    let isValid = true;
    
    step3Data.selectedSignatureTypes.forEach(type => {
        const nameKey = `${type}_name`;
        const hasName = step3Data.signatureValues[nameKey] && step3Data.signatureValues[nameKey].trim() !== '';
        const hasSignature = step3Data.individualSignatureStatus[type];
        
        if (!hasName || !hasSignature) {
            isValid = false;
        }
    });
    
    if (!isValid) {
        alert('Please complete all signature fields (name and signature).');
    }
    
    return isValid;
}

// Generate report
async function handleGenerateReport() {
    if (!validateStep3()) {
        return;
    }
    
    // Show sync modal
    openModal('syncModal');
    
    try {
        // Collect and log all form data from all three steps
        console.log('🚀 GENERATING REPORT - COLLECTING ALL DATA');
        console.log('📋 COLLECTING ALL FORM DATA FROM ALL THREE STEPS');
        
        // Step 1: Basic Information Form
        console.log('\n🔵 STEP 1 - BASIC INFORMATION FORM:');
        const step1FormData = {
            reportNumber: document.getElementById('reportNumber').value,
            country: document.getElementById('country').value,
            project: document.getElementById('project').value,
            inspector: document.getElementById('inspector').value,
            vehicleNumber: document.getElementById('vehicleNumber').value,
            vehicleName: document.getElementById('vehicleName').value,
            vinNumber: document.getElementById('vinNumber').value,
            meterReading: document.getElementById('meter-Reading-input').value,
            meterUnit: document.getElementById('meterReading').value,
            currentMeterReading: document.getElementById('currentMeterValue') ? document.getElementById('currentMeterValue').textContent : '',
            inspectionChecklist: document.getElementById('inspectionChecklist').value,
            checklistDisplay: document.getElementById('checklistDisplay').value,
            coordinates: document.getElementById('coordinatesInput').value,
            gpsAddress: document.getElementById('gpsAddressInput').value
        };
        console.log('Step 1 Form Fields:', step1FormData);
        
        // Step 2: Inspection Form (Dynamic Checklist)
        console.log('\n🟡 STEP 2 - INSPECTION FORM (DYNAMIC CHECKLIST):');
        console.log('Checklist Info:', {
            id: currentChecklist?.document_id || currentChecklist?.id || '',
            name: currentChecklist?.document_name || currentChecklist?.title || '',
            type: currentChecklist?.type || ''
        });
        console.log('Header Inputs:', headerInputValues);
        console.log('User Selections:', inspectionSelections);
        console.log('Notes:', inspectionNotes);
        console.log('Images:', Object.keys(inspectionImages).length + ' images uploaded');
        console.log('Faulty Items Count:', loadFaultyItemsCount());
        console.log('Total Sections:', currentSections.length);
        
        // Log each section with detailed item information
        currentSections.forEach((section, sectionIndex) => {
            console.log(`\n  📁 Section ${sectionIndex + 1}: ${section.name}`);
            if (section.subItems && section.subItems.length > 0) {
                section.subItems.forEach((item, itemIndex) => {
                    const selection = inspectionSelections[item.key] || 'Not Selected';
                    const note = inspectionNotes[item.key] || 'No Note';
                    const hasImage = !!inspectionImages[item.key];
                    
                    console.log(`    ${itemIndex + 1}. ${item.label || item.name || item.itemName}`);
                    console.log(`       Selection: ${selection}`);
                    console.log(`       Note: ${note}`);
                    console.log(`       Image: ${hasImage ? 'Yes' : 'No'}`);
                    console.log(`       Input Type: ${item.inputType || 'N/A'}`);
                    console.log(`       Required: ${item.isInspectionRequired || false}`);
                });
            }
        });
        
        // Step 3: Summary Form
        console.log('\n🟢 STEP 3 - SUMMARY FORM:');
        
        // Get form values directly from DOM elements as backup
        const overallConditionSelect = document.getElementById('overallCondition-select');
        const additionalNoteTextarea = document.getElementById('additionalNote');
        const selectedStatusBtn = document.querySelector('.status-option-btn.selected');
        const selectedSafeBtn = document.querySelector('.safe-option-btn.selected');
        
        const step3FormData = {
            overallCondition: step3Data.overallCondition || (overallConditionSelect ? overallConditionSelect.value : ''),
            equipmentStatus: step3Data.equipmentStatus || (selectedStatusBtn ? selectedStatusBtn.dataset.status : ''),
            safeToUse: step3Data.safeToUse || (selectedSafeBtn ? selectedSafeBtn.dataset.safe : ''),
            additionalNote: step3Data.additionalNote || (additionalNoteTextarea ? additionalNoteTextarea.value : ''),
            selectedSignatureTypes: step3Data.selectedSignatureTypes || [],
            signatureValues: step3Data.signatureValues || {},
            individualSignatures: step3Data.individualSignatures || {},
            individualSignatureStatus: step3Data.individualSignatureStatus || {},
            signatureFields: step3Data.signatureFields || [],
            faultyItems: loadFaultyItemsCount() || 0
        };
        console.log('Step 3 Form Fields:', step3FormData);
        console.log('Step 3 Individual Field Values:');
        console.log('  Overall Condition:', step3FormData.overallCondition);
        console.log('  Equipment Status:', step3FormData.equipmentStatus);
        console.log('  Safe to Use:', step3FormData.safeToUse);
        console.log('  Additional Note:', step3FormData.additionalNote);
        console.log('  Faulty Items:', step3FormData.faultyItems);
        
        // Log signature details
        if (step3Data.signatureValues && Object.keys(step3Data.signatureValues).length > 0) {
            console.log('Signature Details:');
            Object.entries(step3Data.signatureValues).forEach(([key, value]) => {
                console.log(`  ${key}: ${value ? (value.length > 50 ? value.substring(0, 50) + '...' : value) : 'Empty'}`);
            });
        }
        
        // Collect all step2 field data and log to console
        console.log('\n📋 COLLECTING STEP2 FIELD DATA FOR REPORT GENERATION');
        const step2Data = getInspectionItemsData();
        
        // Create final inspection data using the console logged form data
        const inspectionId = new Date().getTime().toString();
        
        const inspectionData = {
            id: inspectionId,
            step1: {
                checklistTitle: step1FormData.checklistDisplay || 'Inspection Checklist',
                coordinates: step1FormData.coordinates || '',
                country: step1FormData.country,
                createdAt: new Date().toISOString(),
                currentMeterReading: step1FormData.currentMeterReading || '',
                gpsAddress: step1FormData.gpsAddress || '',
                id: inspectionId,
                inspector: step1FormData.inspector,
                meterReading: step1FormData.meterReading,
                project: step1FormData.project,
                    requestNumber: newRequestNumber, // Use the incremented request number
                selectedChecklistId: step1FormData.inspectionChecklist,
                selectedEquipment: {
                    country: step1FormData.country,
                    createdAt: new Date().toISOString(),
                    dynamicField1: step1FormData.vehicleNumber,
                    dynamicField2: step1FormData.vinNumber,
                    dynamicField3: step1FormData.vinNumber,
                    engineNumber: '', // Add if available
                    equipmentCategory: 'Equipment', // Add if available
                    equipmentName: step1FormData.vehicleName,
                    equipmentNumber: step1FormData.vehicleNumber,
                    id: step1FormData.vehicleNumber,
                    mainCategory: 'Equipment', // Add if available
                    make: '', // Add if available
                    meterReading: parseInt(step1FormData.meterReading) || 0,
                    meterUnit: step1FormData.meterUnit,
                    model: step1FormData.vinNumber,
                    regNum: step1FormData.vinNumber,
                    status: 'Active', // Add if available
                    updatedAt: new Date().toISOString(),
                    vin: step1FormData.vinNumber
                }
            },
            step2: step2Data, // Use the formatted step2 data from getInspectionItemsData()
            step3: {
                additionalNote: step3FormData.additionalNote || '',
                createdAt: new Date().toISOString(),
                equipmentStatus: step3FormData.equipmentStatus || '',
                faultyItems: step2Data.faultyItemsCount || 0,
                individualSignatureStatus: step3FormData.individualSignatureStatus || {},
                individualSignatures: step3FormData.individualSignatures || {},
                overallCondition: step3FormData.overallCondition || '',
                pdfDownloadUrl: '', // Will be set after PDF generation
                priority: 'Medium', // Add if available
                safeToUse: step3FormData.safeToUse || '',
                selectedSignatureType: step3FormData.selectedSignatureType || '',
                selectedSignatureTypes: step3FormData.selectedSignatureTypes || [],
                signatureFields: step3FormData.signatureFields || [],
                signatureStatus: false, // Add if available
                signatureValues: step3FormData.signatureValues || {}
            }
        };
        
        console.log('\n=== FINAL COLLECTED INSPECTION DATA (USING CONSOLE LOGGED DATA) ===');
        console.log('Full Form Data:', inspectionData);
        console.log('Step 1 Data:', inspectionData.step1);
        console.log('Step 2 Data:', inspectionData.step2);
        console.log('Step 3 Data:', inspectionData.step3);
        console.log('Request Number:', inspectionData.step1?.requestNumber || inspectionData.id);
        console.log('=====================================');
        
        // Save to database
        const result = await saveInspectionToDatabase(inspectionData);
        
        if (result.success) {
            document.getElementById('syncModal').style.display = 'none';
            showSuccessModal(result.data);
        } else {
            throw new Error(result.message || 'Failed to save inspection data');
        }
    } catch (error) {
        console.error('Error generating report:', error);
        document.getElementById('syncModal').style.display = 'none';
        
        // Show error in a more user-friendly way
        showErrorModal('Error generating report: ' + error.message);
    } finally {
        // Reset button state
        generateBtn.disabled = false;
        generateText.textContent = originalText;
        generateBtn.innerHTML = '<i class="fas fa-file-pdf me-2"></i><span id="generateReportText">' + originalText + '</span>';
    }
}

// Collect data from all steps
function collectAllStepsData() {
    try {
        // Step 1: Inspection Info
        const step1Data = {
            reportNumber: document.getElementById('reportNumber').value,
            country: document.getElementById('country').value,
            project: document.getElementById('project').value,
            inspector: document.getElementById('inspector').value,
            equipment: {
                vehicleNumber: document.getElementById('vehicleNumber').value,
                vehicleName: document.getElementById('vehicleName').value,
                vinNumber: document.getElementById('vinNumber').value,
                meterReading: document.getElementById('meter-Reading-input').value,
                meterUnit: document.getElementById('meterReading').value,
                currentMeterReading: (document.getElementById('currentMeterValue') ? document.getElementById('currentMeterValue').textContent : '') || 'Not selected'
            },
            checklist: {
                id: document.getElementById('inspectionChecklist').value,
                display: document.getElementById('checklistDisplay').value
            },
            gpsLocation: {
                coordinates: document.getElementById('coordinatesInput').value || '',
                address: document.getElementById('gpsAddressInput').value || ''
            },
            timestamp: new Date().toISOString()
        };
        
        // Step 2: Do Inspection
        const step2Data = {
            checklist: currentChecklist || {},
            sections: currentSections || [],
            selections: inspectionSelections || {},
            headerInputs: headerInputValues || {},
            notes: inspectionNotes || {},
            images: inspectionImages || {},
            faultyItemsCount: loadFaultyItemsCount()
        };
        
        // Step 3: Inspection Summary
        const step3SummaryData = {
            overallCondition: step3Data.overallCondition || '',
            equipmentStatus: step3Data.equipmentStatus || '',
            safeToUse: step3Data.safeToUse || '',
            additionalNote: step3Data.additionalNote || '',
            signatureFields: step3Data.selectedSignatureTypes || [],
            signatureValues: step3Data.signatureValues || {},
            individualSignatures: step3Data.individualSignatures || {},
            individualSignatureStatus: step3Data.individualSignatureStatus || {}
        };
        
        // Validate required data
        if (!validateCollectedData(step1Data, step2Data, step3SummaryData)) {
            throw new Error('Required data is missing. Please complete all steps before generating the report.');
        }
        
        return {
            requestNumber: step1Data.reportNumber,
            step1: step1Data,
            step2: step2Data,
            step3: step3SummaryData,
            status: 'completed',
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString()
        };
    } catch (error) {
        console.error('Error collecting data:', error);
        throw error;
    }
}

// Validate collected data
function validateCollectedData(step1Data, step2Data, step3Data) {
    // Validate Step 1
    if (!step1Data.reportNumber || !step1Data.country || !step1Data.project || 
        !step1Data.inspector || !step1Data.equipment.vehicleNumber || 
        !step1Data.equipment.vehicleName || !step1Data.equipment.meterReading) {
        console.error('Step 1 validation failed:', step1Data);
        return false;
    }
    
    // Validate Step 2
    if (!step2Data.checklist || Object.keys(step2Data.selections).length === 0) {
        console.error('Step 2 validation failed:', step2Data);
        return false;
    }
    
    // Validate Step 3
    if (!step3Data.overallCondition || !step3Data.equipmentStatus || 
        !step3Data.safeToUse || step3Data.signatureFields.length === 0) {
        console.error('Step 3 validation failed:', step3Data);
        return false;
    }
    
    return true;
}


// Save inspection data to database
async function saveInspectionToDatabase(inspectionData) {
    try {
        const response = await fetch('/api/save-inspection-report', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('input[name="_token"]').value
            },
            body: JSON.stringify(inspectionData)
        });
        
        const result = await response.json();
        return result;
    } catch (error) {
        console.error('Error saving to database:', error);
        throw new Error('Failed to save inspection data to database');
    }
}

function showSuccessModal(savedData) {
    // Show notification statuses
    const notificationStatus = document.getElementById('notificationStatus');
    const emailStatus = document.getElementById('emailStatus');
    
    notificationStatus.innerHTML = `
        <i class="fas fa-bell text-success"></i>
        <span>Push notification sent successfully</span>
    `;
    notificationStatus.className = 'notification-status success';
    notificationStatus.style.display = 'flex';
    
    emailStatus.innerHTML = `
        <i class="fas fa-envelope text-success"></i>
        <span>Email notification sent successfully</span>
    `;
    emailStatus.className = 'email-status success';
    emailStatus.style.display = 'flex';
    
    // Store the saved data for PDF generation
    window.savedInspectionData = savedData;
    
    openModal('successModal');
}

function showErrorModal(message, title = 'An Error Occurred') {
    document.getElementById('errorModalTitle').textContent = title;
    document.getElementById('errorModalMessage').textContent = message;
    openModal('errorModal');
}

function closeErrorModal() {
    closeModal('errorModal');
}

function closeSuccessModal() {
    closeModal('successModal');
    // Reset form and go back to step 1
    resetStep3Form();
    goToStep1();
}

function viewPDF() {
    if (window.savedInspectionData) {
        // Generate PDF with the saved data
        generateInspectionPDF(window.savedInspectionData);
    } else {
        alert('No inspection data available for PDF generation.');
    }
}

// Generate inspection PDF with exact same format as React Native code
async function generateInspectionPDF(inspectionData) {
    try {
        console.log('=== GENERATING PDF WITH EXACT FORMAT ===');
        console.log('Inspection Data for PDF:', inspectionData);
        
        // Create HTML content with exact same format as React Native code
        const htmlContent = generatePDFHTML(inspectionData);
        
        // Log the generated HTML content
        console.log('=== GENERATED PDF HTML CONTENT ===');
        console.log('HTML Content Length:', htmlContent.length);
        console.log('HTML Content Preview:', htmlContent.substring(0, 500) + '...');
        
        // Create a Blob URL for the PDF content
        const blob = new Blob([htmlContent], { type: 'text/html' });
        const pdfUrl = URL.createObjectURL(blob);
        
        console.log('=== GENERATED PDF URL ===');
        console.log('PDF Blob URL:', pdfUrl);
        console.log('PDF Blob Size:', blob.size, 'bytes');
        
        // Create a new window/tab to display the PDF
        const printWindow = window.open(pdfUrl, '_blank');
        
        // Log the print window URL
        console.log('=== PDF WINDOW INFO ===');
        console.log('Print Window URL:', printWindow.location.href);
        console.log('Print Window Object:', printWindow);
        
        // Wait for content to load then print
        printWindow.onload = function() {
            console.log('=== PDF WINDOW LOADED ===');
            console.log('Final PDF Window URL:', printWindow.location.href);
            console.log('PDF Window Document Title:', printWindow.document.title);
            
            // Auto-print the PDF
            setTimeout(() => {
                console.log('=== TRIGGERING PDF PRINT ===');
                printWindow.print();
                // Don't close immediately, let user save if needed
                // printWindow.close();
            }, 500);
        };
        
        // Return the PDF URL for potential use
        return pdfUrl;
        
    } catch (error) {
        console.error('Error generating PDF:', error);
        alert('Error generating PDF: ' + error.message);
        return null;
    }
}

// Generate HTML content with exact same format as React Native PDF
function generatePDFHTML(inspectionData) {
    const step1 = inspectionData.step1 || {};
    const step2 = inspectionData.step2 || {};
    const step3 = inspectionData.step3 || {};
    
    // Helper functions to match React Native formatting
    const toTitleCase = (value) => {
        if (value === undefined || value === null) return '';
        return String(value)
            .replace(/_/g, ' ')
            .split(' ')
            .map((w) => (w ? w.charAt(0).toUpperCase() + w.slice(1).toLowerCase() : ''))
            .join(' ');
    };
    
    const toUpperUnderscore = (value) => {
        if (value === undefined || value === null) return '';
        return String(value).replace(/_/g, ' ').toUpperCase();
    };
    
    // Build checklist rows
    let checklistRows = '';
    const sections = step2.sections || [];
    const selections = step2.selections || {};
    const notes = step2.notes || {};
    
    // Include ALL items that have been inspected (have a selection)
    const inspectedItems = sections.flatMap((section) =>
        (section.subItems || []).filter((item) => 
            selections[item.key] !== undefined && selections[item.key] !== null
        )
    );
    
    if (inspectedItems.length === 0) {
        checklistRows += `
            <tr>
                <td colspan="6">No inspected items available</td>
            </tr>
        `;
    } else {
        // Sort items by classification (A, B, C) and then by their original order
        const sortedItems = inspectedItems.sort((a, b) => {
            const sectionA = sections.find(section => 
                section.subItems && section.subItems.some(subItem => subItem.key === a.key)
            );
            const sectionB = sections.find(section => 
                section.subItems && section.subItems.some(subItem => subItem.key === b.key)
            );
            
            const sectionNameA = sectionA?.name || a.category || 'C';
            const sectionNameB = sectionB?.name || b.category || 'C';
            
            const order = { 'A': 1, 'B': 2, 'C': 3 };
            const aOrder = order[sectionNameA] || 4;
            const bOrder = order[sectionNameB] || 4;
            
            if (aOrder !== bOrder) {
                return aOrder - bOrder;
            }
            return (a.sortOrder || 0) - (b.sortOrder || 0);
        });
        
        // Group items by section for better organization
        const itemsBySection = {};
        sortedItems.forEach(item => {
            const itemSection = sections.find(section => 
                section.subItems && section.subItems.some(subItem => subItem.key === item.key)
            );
            const sectionName = itemSection?.name || item.category || 'Unknown';
            
            if (!itemsBySection[sectionName]) {
                itemsBySection[sectionName] = [];
            }
            itemsBySection[sectionName].push(item);
        });
        
        // Generate rows with section headers for non-A/B/C sections
        Object.entries(itemsBySection).forEach(([sectionName, sectionItems]) => {
            const isABC = /^[ABC]$/.test(sectionName);
            
            if (!isABC) {
                // Add section header row for non-A/B/C sections
                checklistRows += `
                    <tr style="background-color: #f2f2f2; font-weight: bold;">
                        <td style="text-align: left; padding: 6px 8px; font-weight: bold;">${toTitleCase(sectionName)}</td>
                        <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                        <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                        <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                        <td style="text-align: center; padding: 6px 8px; font-weight: bold;"></td>
                        <td style="text-align: left; padding: 6px 8px; font-weight: bold;"></td>
                    </tr>
                `;
            }
            
            // Add items for this section (React Native line 1479-1501)
            sectionItems.forEach((item, index) => {
                const selection = selections[item.key] || 'N/A';
                const isOkay = selection === 'Okay' || selection === 'Ok';
                const isFaulty = selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok';
                const isNA = selection === 'N/A';
                const comment = (notes && notes[item.key]) ? notes[item.key] : '';
                
                // For A/B/C sections, show the class. For others, leave it empty
                const displayClass = isABC ? sectionName : '';
                
                console.log(`Item ${index + 1}: ${item.itemName || item.name || item.label} - Section: ${sectionName} - Classification: ${displayClass} - Selection: ${selection}`);
                
                checklistRows += `
                    <tr>
                        <td style="text-align: left; padding: 6px 8px;">${toUpperUnderscore(item.itemName || item.name || item.label || 'Unknown')}</td>
                        <td style="text-align: center; padding: 6px 8px;">${displayClass}</td>
                        <td style="text-align: center; padding: 6px 8px;">${isOkay ? '✓' : ''}</td>
                        <td style="text-align: center; padding: 6px 8px;">${isFaulty ? 'X' : ''}</td>
                        <td style="text-align: center; padding: 6px 8px;">${isNA ? '—' : ''}</td>
                        <td style="text-align: left; padding: 6px 8px;">${comment || ''}</td>
                    </tr>
                `;
            });
        });
    }
    
    // Build dynamic header info tables from Step 2 headerInputValues
    const headerPairs = Object.entries(step2.headerInputValues || {});
    const formattedHeaderPairs = headerPairs.map(([key, val]) => ({
        label: String(key).replace(/_/g, ' ').toUpperCase(),
        value: val ?? 'N/A',
    }));
    
    // Add Equipment Name and Equipment Model Number to header
    if (step1.selectedEquipment) {
        formattedHeaderPairs.push({
            label: 'EQUIPMENT NAME',
            value: step1.selectedEquipment.equipmentName || step1.selectedEquipment.vehicleNumber || 'N/A'
        });
        formattedHeaderPairs.push({
            label: 'EQUIPMENT MODEL NUMBER',
            value: step1.selectedEquipment.model || step1.selectedEquipment.vehicleNumber || 'N/A'
        });
    }
    
    const half = Math.ceil(formattedHeaderPairs.length / 2);
    const leftHeaders = formattedHeaderPairs.slice(0, half);
    const rightHeaders = formattedHeaderPairs.slice(half);
    
    const renderHeaderTable = (rows) => {
        if (rows.length === 0) return '<table></table>';
        return `
            <table>
                ${rows
                    .map(
                        (r) => `
                        <tr>
                            <td>${r.label}</td>
                            <td>${r.value}</td>
                        </tr>`
                    )
                    .join('')}
            </table>
        `;
    };
    
    const dynamicGeneralInfoHtml = formattedHeaderPairs.length > 0
        ? `
        <div class="general-info-section">
            <div class="info-box">
                ${renderHeaderTable(leftHeaders)}
            </div>
            <div class="info-box">
                ${renderHeaderTable(rightHeaders)}
            </div>
        </div>
    `
        : '';
    
    // Generate signature section - MATCHES React Native Summerystep3.js EXACTLY
    let signatureSectionHtml = '';
    
    // Debug: Log signature fields data (matches React Native line 1306-1337)
    console.log('=== PDF SIGNATURE FIELDS DEBUG ===');
    console.log('step3.signatureFields:', step3.signatureFields);
    console.log('step3.signatureValues:', step3.signatureValues);
    console.log('step3.individualSignatures:', step3.individualSignatures);
    console.log('step3.individualSignatureStatus:', step3.individualSignatureStatus);
    console.log('step3.selectedSignatureTypes:', step3.selectedSignatureTypes);
    console.log('step3.signatureData:', step3.signatureData);
    console.log('step3.signatureStatus:', step3.signatureStatus);
    
    // Generate main signature HTML (React Native line 1505-1521)
    let signatureHtml = 'Not Provided';
    
    // Check for main signature first
    if (step3.signatureStatus && step3.signatureData) {
        signatureHtml = `<img src="${step3.signatureData}" alt="Inspector's Signature" style="width: 120px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`;
        console.log('Using main signature from step3.signatureData');
    } 
    // Check for individual signatures if no main signature
    else if (step3.individualSignatures && Object.keys(step3.individualSignatures).length > 0) {
        const firstSignatureKey = Object.keys(step3.individualSignatures)[0];
        const firstSignature = step3.individualSignatures[firstSignatureKey];
        if (firstSignature && step3.individualSignatureStatus[firstSignatureKey]) {
            signatureHtml = `<img src="${firstSignature}" alt="Signature" style="width: 120px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`;
            console.log('Using first individual signature:', firstSignatureKey);
        }
    }
    
    console.log('Generated signatureHtml:', signatureHtml);
    
    // Build signature section (React Native line 1523-1722)
    console.log('Building signature section for PDF...');
    console.log('step3.signatureFields.length:', step3.signatureFields?.length || 0);
    
    if (step3.signatureFields && step3.signatureFields.length > 0) {
        // Use signature fields from step3 (React Native line 1532-1569)
        console.log('Using step3.signatureFields for PDF signature section');
        signatureSectionHtml = `
            <div class="signature-section">
                <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
        `;
        
        step3.signatureFields.forEach((field, index) => {
            console.log(`Processing signature field ${index}:`, field);
            const fieldValue = step3.signatureValues[field.nameKey] || step1.inspector || 'N/A';
            const signatureKey = field.signatureKey || field.key;
            const hasIndividualSignature = step3.individualSignatureStatus[signatureKey] && step3.individualSignatures[signatureKey];
            const signatureData = hasIndividualSignature ? step3.individualSignatures[signatureKey] : null;
            
            console.log(`Field ${field.label}: value=${fieldValue}, hasSignature=${hasIndividualSignature}, signatureKey=${signatureKey}`);
            
            signatureSectionHtml += `
                <tr>
                    <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">${field.label}:</td>
                    <td style="border: 1px solid #222; padding: 8px; width: 40%;">${fieldValue}</td>
                    <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                        ${hasIndividualSignature && signatureData
                            ? `<img src="${signatureData}" alt="Signature" style="width: 120px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`
                            : 'Not Provided'}
                    </td>
                </tr>
                <tr>
                    <td style="border: 1px solid #222; padding: 8px; font-weight: bold;"></td>
                    <td style="border: 1px solid #222; padding: 8px;"></td>
                </tr>
            `;
        });
        
        signatureSectionHtml += `
                </table>
            </div>
        `;
    } else {
        // Fallback to default signature section (React Native line 1657-1722)
        console.log('No signature fields, using fallback default signature section');
        signatureSectionHtml = `
            <div class="signature-section">
                <table style="width: 100%; border-collapse: collapse; margin-top: 20px;">
                    <tr>
                        <td style="border: 1px solid #222; padding: 8px; font-weight: bold; width: 30%;">DRILLER'S NAME:</td>
                        <td style="border: 1px solid #222; padding: 8px; width: 40%;">${step1.inspector || 'N/A'}</td>
                        <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                            ${signatureHtml}
                        </td>
                    </tr>
                    <tr>
                        <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">DRILLER'S SIGNATURE:</td>
                        <td style="border: 1px solid #222; padding: 8px;"></td>
                    </tr>
                    <tr>
                        <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">MECHANIC'S NAME:</td>
                        <td style="border: 1px solid #222; padding: 8px; width: 40%;">${step1.inspector || 'N/A'}</td>
                        <td style="border: 1px solid #222; padding: 8px; width: 30%; text-align: center;" rowspan="2">
                            ${signatureHtml}
                        </td>
                    </tr>
                    <tr>
                        <td style="border: 1px solid #222; padding: 8px; font-weight: bold;">MECHANIC'S SIGNATURE:</td>
                        <td style="border: 1px solid #222; padding: 8px;"></td>
                    </tr>
                </table>
            </div>
        `;
    }
    
    console.log('Final signatureSectionHtml generated:', signatureSectionHtml.substring(0, 200) + '...');
    
    // Get checklist metadata from currentChecklist or step1 (for dynamic document info)
    const checklistMeta = currentChecklist || step1.checklistMeta || {};
    
    return `
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8"/>
            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
            <title>${checklistMeta.document_name || checklistMeta.title || 'Daily DD Rig Pre-start Checklist'}</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                    padding: 0;
                    font-size: 12px;
                    line-height: 1.4;
                }
                .header-section {
                    display: flex;
                    justify-content: space-between;
                    align-items: flex-start;
                    margin-bottom: 20px;
                }
                .logo-section {
                    flex: 1;
                }
                .logo-section img {
                    width: 140px;
                    height: 50px;
                    object-fit: contain;
                }
                .document-info {
                    flex: 1;
                    margin-left: 20px;
                }
                .document-info table {
                    width: 100%;
                    border: 1px solid #222;
                    border-collapse: collapse;
                    font-size: 11px;
                }
                .document-info td {
                    border: 1px solid #222;
                    padding: 4px 6px;
                    text-align: left;
                }
                .document-info td:first-child {
                    font-weight: bold;
                    background-color: #f5f5f5;
                    width: 40%;
                }
                .title-section {
                    text-align: center;
                    margin: 20px 0;
                }
                .title-section h1 {
                    margin: 0;
                    font-size: 18px;
                    font-weight: bold;
                    color: #333;
                }
                .request-number {
                    font-size: 14px;
                    font-weight: bold;
                    margin-top: 5px;
                    color: #333;
                }
                .general-info-section {
                    display: flex;
                    justify-content: space-between;
                    margin: 20px 0;
                }
                .info-box {
                    width: 48%;
                }
                .info-box table {
                    width: 100%;
                    border: 1px solid #222;
                    border-collapse: collapse;
                    font-size: 11px;
                }
                .info-box td {
                    border: 1px solid #222;
                    padding: 4px 6px;
                    text-align: left;
                }
                .info-box td:first-child {
                    font-weight: bold;
                    background-color: #f5f5f5;
                    width: 40%;
                }
                .classification-rules {
                    margin: 20px 0;
                }
                .classification-rules table {
                    width: 100%;
                    border: 1px solid #222;
                    border-collapse: collapse;
                    font-size: 11px;
                }
                .classification-rules td {
                    border: 1px solid #222;
                    padding: 6px 8px;
                    text-align: left;
                    font-weight: bold;
                }
                .checklist-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin: 20px 0;
                    font-size: 11px;
                }
                .checklist-table th {
                    background-color: #f2f2f2;
                    border: 1px solid #222;
                    padding: 8px 6px;
                    text-align: center;
                    font-weight: bold;
                }
                .checklist-table td {
                    border: 1px solid #222;
                    padding: 6px 8px;
                }
                .checklist-table th:nth-child(1), .checklist-table td:nth-child(1) {
                    width: 35%;
                    text-align: left;
                }
                .checklist-table th:nth-child(2), .checklist-table td:nth-child(2) {
                    width: 10%;
                    text-align: center;
                }
                .checklist-table th:nth-child(3), .checklist-table td:nth-child(3),
                .checklist-table th:nth-child(4), .checklist-table td:nth-child(4),
                .checklist-table th:nth-child(5), .checklist-table td:nth-child(5) {
                    width: 12%;
                    text-align: center;
                }
                .checklist-table th:nth-child(6), .checklist-table td:nth-child(6) {
                    width: 19%;
                    text-align: left;
                }
                .remarks-row td {
                    text-align: left;
                    font-weight: bold;
                }
                .signature-section {
                    margin-top: 30px;
                }
                .signature-section table {
                    width: 100%;
                    border: none;
                    font-size: 12px;
                }
                .signature-section td {
                    border: none;
                    padding: 8px;
                    text-align: left;
                }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="header-section">
                <div class="logo-section">
                    <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
                </div>
                <div class="document-info">
                    <table>
                        <tr>
                            <td>Document Name</td>
                            <td>${checklistMeta.document_name || checklistMeta.title || 'HSE-FOR-8.1.1 DD Prestart Check'}</td>
                        </tr>
                        <tr>
                            <td>Document ID</td>
                            <td>${checklistMeta.document_id || checklistMeta.documentId || 'HSE-028'}</td>
                        </tr>
                        <tr>
                            <td>Version Date</td>
                            <td>${checklistMeta.version_date || checklistMeta.versionDate || '05/08/2021'}</td>
                        </tr>
                        <tr>
                            <td>Version</td>
                            <td>${checklistMeta.version || '2'}</td>
                        </tr>
                        <tr>
                            <td>Approved</td>
                            <td>${checklistMeta.approved_by || checklistMeta.approvedBy || 'HSM'}</td>
                        </tr>
                        <tr>
                            <td>NO</td>
                            <td>${step1.requestNumber || '0001'}</td>
                        </tr>
                    </table>
                </div>
            </div>

            <div class="title-section">
                <h1>${checklistMeta.Header_title || checklistMeta.header_title || checklistMeta.document_name || checklistMeta.title || 'CHECKLIST'}</h1>
            </div>

            ${dynamicGeneralInfoHtml || `
                <div class="general-info-section"> 
                    <div class="info-box"> 
                        <table>
                            <tr><td>DATE</td><td>${step2.headerInputValues?.date || new Date().toLocaleDateString()}</td></tr>
                            <tr><td>RIG</td><td>${step2.headerInputValues?.rig || step1.selectedEquipment?.equipmentName || step1.selectedEquipment?.rig || 'N/A'}</td></tr>
                            <tr><td>ODOMETER READING</td><td>${step2.headerInputValues?.odometer_reading || step2.headerInputValues?.meter_reading || step1.selectedEquipment?.meterReading || step1.meterReading || 'N/A'}</td></tr>
                            <tr><td>FUEL LEVEL</td><td>${step2.headerInputValues?.fuel_level || step1.selectedEquipment?.fuelLevel || 'N/A'}</td></tr>
                        </table>
                    </div>
                    <div class="info-box"> 
                        <table>
                            <tr><td>SITE / PROJECT</td><td>${step1.project || 'N/A'}</td></tr>
                            <tr><td>INSPECTOR</td><td>${step1.inspector || 'N/A'}</td></tr>
                            <tr><td>COMPRESSOR No</td><td>${step1.selectedEquipment?.equipmentName || 'N/A'}</td></tr>
                            <tr><td>HOURS</td><td>${step1.selectedEquipment?.meterReading || step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || 'Hours'}</td></tr>
                        </table>
                    </div>
                </div>
            `}

            <div class="classification-rules">
                <table>
                    <tr>
                        <td>1. ALL DEFECTS UNDER "A" CLASSIFICATION TO BE ATTENDED IMMEDIATELY</td>
                    </tr>
                    <tr>
                        <td>2. ALL DEFECTS UNDER "B" CLASSIFICATION TO BE ATTENDED WITHIN 48 HOURS</td>
                    </tr>
                    <tr>
                        <td>3. ALL DEFECTS UNDER "C" CLASSIFICATION TO BE ATTENDED TO AS SOON AS POSSIBLE</td>
                    </tr>
                </table>
            </div>

            <table class="checklist-table">
                <thead>
                    <tr>
                        <th>DESCRIPTION</th>
                        <th>CLASS</th>
                        <th>IN ORDER (✓)</th>
                        <th>DEFECTIVE (X)</th>
                        <th>N/A</th>
                        <th>COMMENT</th>
                    </tr>
                </thead>
                <tbody>
                    ${checklistRows}
                    <tr class="remarks-row">
                        <td>REMARKS</td>
                        <td colspan="5">${step3.additionalNote || 'N/A'}</td>
                    </tr>
                </tbody>
            </table>

            ${signatureSectionHtml}
        </body>
        </html>
    `;
}

function resetStep3Form() {
    step3Data = {
        overallCondition: '',
        equipmentStatus: '',
        safeToUse: '',
        additionalNote: '',
        signatureFields: [],
        selectedSignatureTypes: [],
        signatureValues: {},
        individualSignatures: {},
        individualSignatureStatus: {}
    };
    
    // Reset UI
    document.getElementById('overallCondition-select').value = '';
    document.getElementById('additionalNote').value = '';
    document.getElementById('signatureFieldsContainer').innerHTML = '';
    document.getElementById('selectedSignatureTypeText').textContent = 'Select Signature Type';
    
    // Clear button selections
    document.querySelectorAll('.status-option-btn, .priority-btn, .safe-option-btn').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    // Hide priority container
    document.getElementById('priorityContainer').style.display = 'none';
    
    // Clear errors
    document.querySelectorAll('.error-text').forEach(error => {
        error.style.display = 'none';
    });
}

// URL and Data Persistence Functions
function updateURL(step) {
    const url = new URL(window.location);
    url.searchParams.set('step', step);
    window.history.pushState({ step: step }, '', url);
}

function getCurrentStepFromURL() {
    const urlParams = new URLSearchParams(window.location.search);
    return parseInt(urlParams.get('step')) || 1;
}

function saveFormData() {
    // Use the new session system
    const success = saveToSession();
    
    if (success) {
        console.log('Form data saved to session');
    } else {
        console.error('Failed to save form data to session');
    }
}

function showAutoSaveIndicator() {
    const indicator = document.getElementById('autoSaveIndicator');
    if (indicator) {
        indicator.classList.add('show');
        setTimeout(() => {
            indicator.classList.remove('show');
        }, 2000);
    }
}


// Function to clear saved data (useful for testing)
function clearSavedData() {
    localStorage.removeItem('inspectionFormData');
    console.log('Saved data cleared');
}

// Function to get current faulty countimage.png
function getFaultyCount() {
    try {
        const selections = inspectionSelections || {};
        let faultyCount = 0;
        
        Object.values(selections).forEach(selection => {
            if (selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok') {
                faultyCount++;
            }
        });
        
        return faultyCount;
    } catch (error) {
        console.error('Error getting faulty count:', error);
        return 0;
    }
}


function loadFormData() {
    // Use the new session system
    const sessionData = loadFromSession();
    if (sessionData) {
        console.log('Loading session data:', sessionData);
        
        // Restore from session
        const restored = restoreFromSession();
        if (restored) {
            return sessionData.currentStep || 1;
        }
    }
    
    console.log('No session data found, starting fresh');
    return 1;
}

function collectStep1Data() {
    const data = {};
    const inputs = document.querySelectorAll('#step1-form input, #step1-form select, #step1-form textarea');
    inputs.forEach(input => {
        if (input.name) {
            if (input.type === 'checkbox' || input.type === 'radio') {
                data[input.name] = input.checked ? input.value : '';
            } else {
                data[input.name] = input.value;
            }
        }
    });
    return data;
}

function collectStep2Data() {
    const data = {};
    const inputs = document.querySelectorAll('#do-inspection-container input, #do-inspection-container select, #do-inspection-container textarea');
    inputs.forEach(input => {
        if (input.name) {
            if (input.type === 'checkbox' || input.type === 'radio') {
                data[input.name] = input.checked ? input.value : '';
            } else {
                data[input.name] = input.value;
            }
        }
    });
    
    // Also collect inspection selections
    data.inspectionSelections = inspectionSelections || {};
    data.checklistSelections = checklistSelections || {};
    
    return data;
}

function collectStep3Data() {
    const data = {};
    const inputs = document.querySelectorAll('#Inspection-Summary-main input, #Inspection-Summary-main select, #Inspection-Summary-main textarea');
    inputs.forEach(input => {
        if (input.name) {
            if (input.type === 'checkbox' || input.type === 'radio') {
                data[input.name] = input.checked ? input.value : '';
            } else {
                data[input.name] = input.value;
            }
        }
    });
    
    // Also collect step 3 specific data
    data.step3Data = window.step3Data || {};
    data.faultyCount = getFaultyCount();
    
    console.log('Collected Step 3 data:', data);
    return data;
}

function restoreStep1Data(data) {
    if (!data) return;
    
    console.log('Restoring Step 1 data:', data);
    
    Object.keys(data).forEach(key => {
        const input = document.querySelector(`[name="${key}"]`);
        if (input) {
            if (input.type === 'checkbox' || input.type === 'radio') {
                input.checked = data[key] === input.value;
            } else {
                input.value = data[key];
            }
        }
    });
}

function restoreStep2Data(data) {
    if (!data) return;
    
    console.log('Restoring Step 2 data:', data);
    
    Object.keys(data).forEach(key => {
        if (key === 'inspectionSelections' || key === 'checklistSelections') {
            window[key] = data[key];
            return;
        }
        
        const input = document.querySelector(`[name="${key}"]`);
        if (input) {
            if (input.type === 'checkbox' || input.type === 'radio') {
                input.checked = data[key] === input.value;
            } else {
                input.value = data[key];
            }
        }
    });
}

function restoreStep3Data(data) {
    if (!data) return;
    
    console.log('Restoring Step 3 data:', data);
    
    Object.keys(data).forEach(key => {
        if (key === 'step3Data') {
            // Restore step 3 specific data
            window.step3Data = data[key];
            return;
        }
        
        if (key === 'faultyCount') {
            // Update faulty count display
            const faultyCountElement = document.getElementById('faultyCount');
            if (faultyCountElement) {
                faultyCountElement.textContent = data[key];
            }
            return;
        }
        
        const input = document.querySelector(`[name="${key}"]`);
        if (input) {
            if (input.type === 'checkbox' || input.type === 'radio') {
                input.checked = data[key] === input.value;
            } else {
                input.value = data[key];
            }
        }
    });
    
    // Restore step 3 specific state
    if (data.step3Data) {
        // Restore any step 3 specific variables or state
        Object.keys(data.step3Data).forEach(key => {
            if (window[key] !== undefined) {
                window[key] = data.step3Data[key];
            }
        });
    }
    
    // Restore signature types and regenerate fields if needed
    if (data.selectedSignatureTypes && Array.isArray(data.selectedSignatureTypes)) {
        step3Data.selectedSignatureTypes = data.selectedSignatureTypes;
        
        // Restore signature type selections
        restoreSignatureTypeSelections();
        
        // Clear existing signature fields to prevent duplication
        clearSignatureFields();
        
        // Regenerate signature fields if there are selected types
        if (step3Data.selectedSignatureTypes.length > 0) {
            setTimeout(() => {
                generateSignatureFields();
            }, 100);
        }
    }
}

function navigateToStep(step) {
    console.log('Navigating to step:', step);
    
    // Hide all steps first
    document.getElementById('step1-form').style.display = 'none';
    document.getElementById('do-inspection-container').style.display = 'none';
    document.getElementById('Inspection-Summary-main').style.display = 'none';
    
    // Show the correct step
    switch(step) {
        case 1:
            document.getElementById('step1-form').style.display = 'block';
            updateURL(1);
            break;
        case 2:
            document.getElementById('do-inspection-container').style.display = 'block';
            updateURL(2);
            // Load checklist data for Step 2
            loadChecklistDataForStep2();
            // Update faulty count
            updateFaultyCount();
            break;
        case 3:
            document.getElementById('Inspection-Summary-main').style.display = 'block';
            updateURL(3);
            // Load faulty items count and initialize step 3
            updateFaultyCount();
            // Add delay to ensure DOM is ready for step 3 initialization
            setTimeout(() => {
                initializeStep3();
            }, 50);
            break;
        default:
            document.getElementById('step1-form').style.display = 'block';
            updateURL(1);
    }
    
    // Save current state
    saveFormData();
}

// Initialize page with URL-based step navigation
async function initializePage() {
    const currentStep = getCurrentStepFromURL();
    
    console.log('Initializing page - URL step:', currentStep);
    
    // Request number generated automatically when country and project are selected
    
    // Load saved data first
    const savedStep = loadFormData();
    
    // Use URL step if available, otherwise use saved step
    const stepToShow = currentStep || savedStep || 1;
    
    console.log('Step to show:', stepToShow);
    
    // Add a small delay to ensure DOM is fully loaded
    setTimeout(() => {
        // Navigate to the appropriate step
        navigateToStep(stepToShow);
        
    }, 100);
    
    
    // Event listeners for country and project handled by updateProjects() and generateRequestNumber()
    // No additional listeners needed - request number auto-generates on selection
    
    // Add auto-save functionality
    setInterval(saveFormData, 30000); // Auto-save every 30 seconds
    
    // Add form input listeners for immediate auto-save
    const formInputs = document.querySelectorAll('input, select, textarea');
    formInputs.forEach(input => {
        input.addEventListener('change', function() {
            // Debounce auto-save to avoid too frequent saves
            clearTimeout(window.autoSaveTimeout);
            window.autoSaveTimeout = setTimeout(saveFormData, 1000);
        });
    });
    
    // Listen for browser back/forward buttons
    window.addEventListener('popstate', function(event) {
        const step = event.state ? event.state.step : getCurrentStepFromURL();
        navigateToStep(step);
    });
}

// Form Navigation Functions
function goToStep1() {
    document.getElementById('step1-form').style.display = 'block';
    document.getElementById('do-inspection-container').style.display = 'none';
    document.getElementById('Inspection-Summary-main').style.display = 'none';
    updateURL(1);
    saveFormData();
}

function goToStep2() {
    // Validate step 1 form if coming from step 1
    if (document.getElementById('step1-form').style.display !== 'none') {
        if (!validateStep1()) {
            return;
        }
    }
    
    document.getElementById('step1-form').style.display = 'none';
    document.getElementById('do-inspection-container').style.display = 'block';
    document.getElementById('Inspection-Summary-main').style.display = 'none';
    updateURL(2);
    saveFormData();
    
    // Load checklist data for Step 2
    loadChecklistDataForStep2();
    
    // Update faulty count in case user made changes in step 3 and is going back
    updateFaultyCount();
}


// Checklist functionality
let checklistSelections = {};

function handleItemStatus(itemId, status) {
    // Update the selection
    checklistSelections[itemId] = status;
    
    // Update button states for this item
    const itemButtons = document.querySelectorAll(`[onclick*="handleItemStatus('${itemId}'"]`);
    itemButtons.forEach(btn => {
        btn.classList.remove('selected');
        if (btn.dataset.status === status) {
            btn.classList.add('selected');
        }
    });
    
    // Update section counts
    updateSectionCounts();
}

function updateSectionCounts() {
    // Count selections for Section A
    const sectionAItems = ['A1', 'A2', 'A3', 'A4', 'A5', 'A6'];
    const sectionASelected = sectionAItems.filter(item => checklistSelections[item]).length;
    document.querySelector('.checklist-section:nth-child(1) .section-count').textContent = `(${sectionASelected}/6 selected)`;
    
    // Count selections for Section B
    const sectionBItems = ['B1', 'B2', 'B3'];
    const sectionBSelected = sectionBItems.filter(item => checklistSelections[item]).length;
    document.querySelector('.checklist-section:nth-child(2) .section-count').textContent = `(${sectionBSelected}/3 selected)`;
    
    // Count selections for Section C
    const sectionCItems = ['C1', 'C2', 'C3', 'C4', 'C5', 'C6', 'C7', 'C8', 'C9', 'C10', 'C11', 'C12', 'C13'];
    const sectionCSelected = sectionCItems.filter(item => checklistSelections[item]).length;
    document.querySelector('.checklist-section:nth-child(3) .section-count').textContent = `(${sectionCSelected}/13 selected)`;
}

function toggleSection(sectionId) {
    const section = document.getElementById(`section-${sectionId}`);
    const arrow = document.getElementById(`arrow-${sectionId}`);
    
    if (section.style.display === 'none') {
        section.style.display = 'block';
        arrow.style.transform = 'rotate(180deg)';
    } else {
        section.style.display = 'none';
        arrow.style.transform = 'rotate(0deg)';
    }
}


function goToStep3() {
    // Validate step 2 if coming from step 2
    if (document.getElementById('do-inspection-container').style.display !== 'none') {
        if (!validateStep2()) {
            return;
        }
    }
    
    document.getElementById('step1-form').style.display = 'none';
    document.getElementById('do-inspection-container').style.display = 'none';
    document.getElementById('Inspection-Summary-main').style.display = 'block';
    updateURL(3);
    saveFormData();
    
    // Load faulty items count and initialize step 3
    updateFaultyCount(); // Use the dynamic update function
    initializeStep3();
}

function validateStep2() {
    // Check if at least one inspection item has been completed (from dynamic or static lists)
    const dynamicSelections = inspectionSelections || {};
    const staticSelections = checklistSelections || {};
    const hasSelections = (Object.keys(dynamicSelections).length > 0) || (Object.keys(staticSelections).length > 0);
    
    if (!hasSelections) {
        alert('Please complete at least one item from A or B or C.');
        return false;
    }
    
    // Check if header inputs are filled (if any exist)
    if (Object.keys(headerInputs).length > 0) {
        const headerInputKeys = Object.keys(headerInputs);
        const hasHeaderInputs = headerInputKeys.some(key => 
            headerInputValues[key] && headerInputValues[key].trim() !== ''
        );
        
        if (!hasHeaderInputs) {
            alert('Please fill in the required header information before proceeding.');
            return false;
        }
    }
    
    return true;
}

function initializeStep3() {
    // Reset step 3 data
    resetStep3Form();
    
    // Load faulty items count
    updateFaultyCount(); // Use the dynamic update function
    
    // Initialize signature canvas if needed
    if (signatureCanvas) {
        signatureCanvas.ctx.clearRect(0, 0, signatureCanvas.canvas.width, signatureCanvas.canvas.height);
    }
}

function validateStep1() {
    const requiredFields = ['country', 'project', 'inspector', 'vehicleNumber', 'vehicleName', 'vinNumber', 'meter-Reading-input', 'meterReading', 'inspectionChecklist'];
    let isValid = true;
    
    // Clear previous errors
    requiredFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            field.classList.remove('is-invalid');
        }
    });
    
    // Validate required fields
    requiredFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (!field.value.trim()) {
            field.classList.add('is-invalid');
            isValid = false;
        }
    });
    
    // Validate meter reading specifically
    if (!validateMeterReading()) {
        isValid = false;
    }
    
    if (!isValid) {
        alert('Please fill in all required fields and ensure the new meter reading is greater than the current reading.');
    }
    
    return isValid;
}


// Global variables for data
let equipmentData = [];
let checklistData = [];

// Modal functions
// Standardized Modal functions with click positioning
function openModal(modalId, clickEvent = null) {
    const modal = document.getElementById(modalId);
    if (modal) {
        // Lock body scrolling
        document.body.classList.add('modal-open');
        
        // Add show class for consistent styling
        modal.classList.add('show');
        modal.style.display = 'block';
        
        // Position modal based on click or center of viewport
        const modalContent = modal.querySelector('.custom-modal-content, .success-modal-content, .loading-modal-content');
        if (modalContent) {
            let topPosition = 0;
            
            if (clickEvent && clickEvent.clientY) {
                // Get click position relative to viewport
                const clickY = clickEvent.clientY + window.scrollY;
                const viewportHeight = window.innerHeight;
                const modalHeight = modalContent.offsetHeight || 400; // fallback height
                
                // Calculate top position centered on click point
                topPosition = clickY - (modalHeight / 2);
                
                // Ensure modal doesn't go off-screen at top
                if (topPosition < window.scrollY + 20) {
                    topPosition = window.scrollY + 20;
                }
                
                // Ensure modal doesn't go off-screen at bottom
                if (topPosition + modalHeight > window.scrollY + viewportHeight - 20) {
                    topPosition = window.scrollY + viewportHeight - modalHeight - 20;
                }
                
                // If modal is still too tall, center it in viewport
                if (topPosition < window.scrollY + 20) {
                    topPosition = window.scrollY + Math.max(20, (viewportHeight - modalHeight) / 2);
                }
            } else {
                // Center vertically in viewport if no click event
                const viewportHeight = window.innerHeight;
                const modalHeight = modalContent.offsetHeight || 400;
                topPosition = window.scrollY + Math.max(20, (viewportHeight - modalHeight) / 2);
            }
            
            modalContent.style.top = topPosition + 'px';
        }
        
        // Add click outside to close
        modal.onclick = function(event) {
            if (event.target === modal) {
                closeModal(modalId);
            }
        };
        
        // Add escape key to close
        const handleEscape = function(event) {
            if (event.key === 'Escape') {
                closeModal(modalId);
                document.removeEventListener('keydown', handleEscape);
            }
        };
        document.addEventListener('keydown', handleEscape);
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        // Unlock body scrolling
        document.body.classList.remove('modal-open');
        
        // Remove show class and hide modal
        modal.classList.remove('show');
        modal.style.display = 'none';
    }
}

// Legacy function for backward compatibility
function centerModal(modalId) {
    openModal(modalId);
}

function openEquipmentModal(event) {
    openModal('equipmentModal', event);
    loadEquipmentData();
}

function closeEquipmentModal() {
    closeModal('equipmentModal');
}

function openChecklistModal(event) {
    openModal('checklistModal', event);
    loadChecklistData();
}

function closeChecklistModal() {
    closeModal('checklistModal');
}

// Load equipment data from MongoDB
async function loadEquipmentData() {
    try {
        console.log('📥 Loading equipment from MongoDB...');
        // Show loading state
        const equipmentList = document.getElementById('equipmentList');
        equipmentList.innerHTML = '<div class="text-center p-3"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Loading equipment from MongoDB...</p></div>';
        
        // Fetch from MongoDB API via Laravel endpoint
        const response = await fetch('/api/equipment/all?limit=1000');
        const data = await response.json();
        
        console.log('✅ Equipment data received:', data);
        
        if (data.success && data.data) {
            equipmentData = data.data || [];
            console.log(`📊 Loaded ${equipmentData.length} equipment items`);
            renderEquipmentList(equipmentData);
        } else {
            console.error('❌ Failed to load equipment:', data.message);
            equipmentList.innerHTML = '<p class="text-center text-muted p-3">Failed to load equipment data from MongoDB</p>';
        }
    } catch (error) {
        console.error('❌ Error loading equipment:', error);
        const equipmentList = document.getElementById('equipmentList');
        equipmentList.innerHTML = '<p class="text-center text-muted p-3">Error loading equipment data. Please try again.</p>';
    }
}

// Load checklist data from MongoDB (global + custom checklists)
async function loadChecklistData() {
    try {
        console.log('📥 Loading checklists from MongoDB...');
        // Show loading state
        const checklistList = document.getElementById('checklistList');
        checklistList.innerHTML = '<div class="text-center p-3"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Loading checklists from MongoDB...</p></div>';
        
        // Fetch global checklists and custom checklists in parallel
        const [globalResponse, customResponse] = await Promise.all([
            fetch('https://api.titandrillingzm.com:6004/checklists?limit=1000'),
            fetch('https://api.titandrillingzm.com:6004/checklists/custom/all?limit=1000')
        ]);
        
        const globalData = await globalResponse.json();
        const customData = await customResponse.json();
        
        console.log('✅ Global checklists:', globalData);
        console.log('✅ Custom checklists:', customData);
        
        let allChecklists = [];
        
        // Process global checklists
        if (globalData.success && globalData.data) {
            const globalChecklists = globalData.data.map(doc => ({
                id: doc.documentId || doc._id,
                title: doc.documentName || doc.headerTitle || 'Untitled',
                description: doc.description || '',
                type: 'global',
                isCustom: false,
                version: doc.version || '',
                checklist: doc.checklist || {},
                header_inputs: doc.header_inputs || {},
                footer_inputs: doc.footer_inputs || {}
            }));
            allChecklists = [...allChecklists, ...globalChecklists];
            console.log(`📋 Loaded ${globalChecklists.length} global checklists`);
        }
        
        // Process custom checklists
        if (customData.success && customData.data) {
            const customChecklists = customData.data.map(doc => ({
                id: doc._id || doc.documentId,
                title: doc.title || 'Untitled Custom Checklist',
                description: doc.description || 'Custom checklist',
                type: 'custom',
                isCustom: true,
                items: doc.items || [],
                createdBy: doc.createdBy || 'User'
            }));
            allChecklists = [...allChecklists, ...customChecklists];
            console.log(`📋 Loaded ${customChecklists.length} custom checklists`);
        }
        
        checklistData = allChecklists;
        console.log(`📊 Total checklists loaded: ${checklistData.length}`);
        renderChecklistList(checklistData);
        
    } catch (error) {
        console.error('❌ Error loading checklists:', error);
        const checklistList = document.getElementById('checklistList');
        checklistList.innerHTML = '<p class="text-center text-muted p-3">Error loading checklist data. Please try again.</p>';
    }
}

// Render equipment list
function renderEquipmentList(equipment) {
    const container = document.getElementById('equipmentList');
    
    if (!equipment || equipment.length === 0) {
        container.innerHTML = '<p class="text-center text-muted p-3">No equipment found</p>';
        return;
    }
    
     const html = equipment.map(item => {
         // Use model as heading if equipment name is not available
         const displayName = item.equipmentName || item.vehicleNumber || item.model || 'Unknown Equipment';
         const category = item.mainCategory || item.equipmentCategory || 'N/A';
         const model = item.model || 'N/A';
         
         return `
         <div class="equipment-item" onclick="selectEquipment('${item.id || item._id || 'unknown'}', '${displayName}', '${model}', '${category}')">
             <div class="equipment-name">${displayName}</div>
             <div class="equipment-details">
                 <strong>Category:</strong> ${category}<br>
                 <strong>Model:</strong> ${model}<br>
                 <strong>Status:</strong> ${item.status || 'Available'}<br>
                 <strong>Meter Reading:</strong> ${item.meterReading || '0'} ${item.meterUnit || 'KM'}
             </div>
         </div>
         `;
     }).join('');
    
    container.innerHTML = html;
}

// Render checklist list (with custom checklist badges)
function renderChecklistList(checklists) {
    const container = document.getElementById('checklistList');
    
    if (!checklists || checklists.length === 0) {
        container.innerHTML = '<p class="text-center text-muted p-3">No checklists found</p>';
        return;
    }
    
    const html = checklists.map(item => {
        const isCustom = item.isCustom || item.type === 'custom';
        const customBadge = isCustom ? '<span style="background: #ff9800; color: white; padding: 2px 8px; border-radius: 12px; font-size: 10px; font-weight: bold; margin-left: 8px;">CUSTOM</span>' : '';
        
        return `
            <div class="checklist-item" onclick="selectChecklist('${item.id}', '${(item.title || 'Untitled').replace(/'/g, "\\'")}', ${isCustom})">
                <div class="checklist-title">
                    ${item.title || item.document_name || 'Untitled Checklist'}
                    ${customBadge}
                </div>
            <div class="checklist-description">
                    ${isCustom ? 
                        `<strong>Type:</strong> Custom<br><strong>Created By:</strong> ${item.createdBy || 'User'}` : 
                        `<strong>ID:</strong> ${item.id}<br><strong>Version:</strong> ${item.version || 'N/A'}`
                    }
            </div>
        </div>
        `;
    }).join('');
    
    container.innerHTML = html;
}

// Select equipment
function selectEquipment(id, name, model, category) {
    console.log('Selecting equipment:', { id, name, model, category });
    
    // Update equipment display and hidden fields
    document.getElementById('equipmentDisplay').value = `${name} - ${model} (${category})`;
    document.getElementById('vehicleNumber').value = id;
    document.getElementById('vehicleName').value = name;
    document.getElementById('vinNumber').value = model; // Using model as VIN for now
    
    // Clear any validation errors on equipment-related fields
    clearMeterReadingError();
    
    // Clear any validation errors on equipment fields
    const equipmentFields = ['equipmentDisplay', 'vehicleNumber', 'vehicleName', 'vinNumber'];
    equipmentFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            field.classList.remove('is-invalid');
            // Trigger change event to notify any validation listeners
            field.dispatchEvent(new Event('change', { bubbles: true }));
        }
    });
    
    console.log('Equipment fields updated:', {
        equipmentDisplay: document.getElementById('equipmentDisplay').value,
        vehicleNumber: document.getElementById('vehicleNumber').value,
        vehicleName: document.getElementById('vehicleName').value,
        vinNumber: document.getElementById('vinNumber').value
    });
    
    // Find the selected equipment data to get meter reading
    const selectedEquipment = equipmentData.find(item => 
        (item.id || item._id) === id || 
        (item.equipmentName || item.vehicleNumber) === name
    );
    
    console.log('Selected equipment data:', selectedEquipment);
    
    // Declare variables outside the if block
    let currentMeterReading = 0;
    let meterUnit = 'KM';
    let newReading = 0;
    
    if (selectedEquipment) {
        // Current meter reading and unit
        currentMeterReading = selectedEquipment.meterReading || 0;
        meterUnit = selectedEquipment.meterUnit || 'KM';
        
        // Normalize meter unit to match select options
        if (meterUnit === 'KM' || meterUnit === 'km' || meterUnit === 'Kilometers') {
            meterUnit = 'KM';
        } else if (meterUnit === 'Miles' || meterUnit === 'miles') {
            meterUnit = 'Miles';
        } else if (meterUnit === 'Hours' || meterUnit === 'hours' || meterUnit === 'Hrs') {
            meterUnit = 'Hours';
        }
        
        console.log('Meter reading data:', { currentMeterReading, meterUnit });
        
        // Update the meter unit select
        const meterUnitSelect = document.getElementById('meterReading');
        if (meterUnitSelect) {
            // Clear previous selection
            meterUnitSelect.selectedIndex = 0;
            
            // Find and select the matching option
            for (let i = 0; i < meterUnitSelect.options.length; i++) {
                const option = meterUnitSelect.options[i];
                if (option.value === meterUnit) {
                    option.selected = true;
                    console.log('Selected meter unit:', meterUnit);
                    break;
                }
            }
        }
        
        // Update the current meter reading display
        const currentMeterDisplay = document.getElementById('currentMeterValue');
        if (currentMeterDisplay) {
            // Format number to avoid scientific notation
            const reading = parseFloat(currentMeterReading);
            const formattedReading = reading > 999999 ? 
                reading.toFixed(0).replace(/\B(?=(\d{3})+(?!\d))/g, ',') : 
                reading.toString();
            currentMeterDisplay.textContent = `${formattedReading} ${meterUnit}`;
            console.log('Updated current meter display:', `${formattedReading} ${meterUnit}`);
        }
        
        // Update the meter reading input with current reading value
        const meterInput = document.getElementById('meter-Reading-input');
        if (meterInput) {
            // Auto-fill with current reading + 1 to pass validation (without commas)
            newReading = parseFloat(currentMeterReading) + 1;
            const rawReading = newReading.toString();
            // Format current reading for display (avoid scientific notation)
            const currentReading = parseFloat(currentMeterReading);
            const formattedCurrentReading = currentReading > 999999 ? 
                currentReading.toFixed(0).replace(/\B(?=(\d{3})+(?!\d))/g, ',') : 
                currentReading.toString();
            meterInput.value = rawReading;
            meterInput.placeholder = `Enter new reading (current: ${formattedCurrentReading} ${meterUnit})`;
            console.log('Auto-filled meter input with new reading:', rawReading);
        }
        
        // Show success notification
        console.log('Showing equipment auto-fill notification...');
        showEquipmentAutoFillNotification(name, model, currentMeterReading, meterUnit, newReading);
        console.log('Equipment auto-fill notification shown');
        
        // Trigger form validation to clear any validation errors
        setTimeout(() => {
            validateEquipmentFields();
            // Also trigger a general form validation check
            checkStep1Validation();
        }, 100);
        
        // Save the auto-filled data to session
        saveToSession();
        
    } else {
        console.warn('Equipment data not found for:', { id, name });
        
        // Reset meter reading display if no equipment data found
        const currentMeterDisplay = document.getElementById('currentMeterValue');
        if (currentMeterDisplay) {
            currentMeterDisplay.textContent = 'Not selected';
        }
        
        const meterInput = document.getElementById('meter-Reading-input');
        if (meterInput) {
            meterInput.value = '';
            meterInput.placeholder = 'Enter new meter reading';
        }
    }
    
    // Close the modal immediately
    console.log('Closing equipment modal...');
    closeEquipmentModal();
    console.log('Equipment modal closed');
    
    // Also ensure modal is closed after a short delay as fallback
    setTimeout(() => {
        const modal = document.getElementById('equipmentModal');
        if (modal && modal.style.display !== 'none') {
            console.log('Fallback: Force closing equipment modal...');
            modal.style.display = 'none';
        }
    }, 200);
}

// Show equipment auto-fill notification
function showEquipmentAutoFillNotification(name, model, currentReading, unit, newReading = null) {
    console.log('showEquipmentAutoFillNotification called with:', { name, model, currentReading, unit, newReading });
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = 'equipment-autofill-notification';
    
    // Format readings to avoid scientific notation
    const formatReading = (reading) => {
        const num = parseFloat(reading);
        return num > 999999 ? 
            num.toFixed(0).replace(/\B(?=(\d{3})+(?!\d))/g, ',') : 
            num.toString();
    };
    
    let readingText = `Current reading: ${formatReading(currentReading)} ${unit}`;
    if (newReading) {
        readingText += ` → New reading: ${formatReading(newReading)} ${unit}`;
    }
    
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-check-circle text-success"></i>
            <span>Auto-filled data for <strong>${name}</strong> (${model})</span>
            <small class="d-block">${readingText}</small>
            <button type="button" class="btn-close" onclick="this.parentElement.parentElement.remove()">&times;</button>
        </div>
    `;
    
    // Add styles
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        background: #d4edda;
        border: 1px solid #c3e6cb;
        border-radius: 8px;
        padding: 12px 16px;
        z-index: 1000;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        animation: slideInRight 0.3s ease-out;
        max-width: 300px;
    `;
    
    // Add to page
    console.log('Adding notification to DOM...');
    document.body.appendChild(notification);
    console.log('Notification added to DOM');
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            console.log('Auto-removing notification...');
            notification.remove();
        }
    }, 5000);
}

// Clear equipment auto-fill data
function clearEquipmentAutoFill() {
    console.log('Clearing equipment auto-fill data');
    
    // Clear equipment display and hidden fields
    document.getElementById('equipmentDisplay').value = '';
    document.getElementById('vehicleNumber').value = '';
    document.getElementById('vehicleName').value = '';
    document.getElementById('vinNumber').value = '';
    
    // Reset meter reading display
    const currentMeterDisplay = document.getElementById('currentMeterValue');
    if (currentMeterDisplay) {
        currentMeterDisplay.textContent = 'Not selected';
    }
    
    // Reset meter unit select
    const meterUnitSelect = document.getElementById('meterReading');
    if (meterUnitSelect) {
        meterUnitSelect.selectedIndex = 0;
    }
    
    // Reset meter reading input
    const meterInput = document.getElementById('meter-Reading-input');
    if (meterInput) {
        meterInput.value = '';
        meterInput.placeholder = 'Enter new meter reading';
        console.log('Reset meter reading input');
    }
    
    // Clear any meter reading errors
    clearMeterReadingError();
    
    // Save cleared data to session
    saveToSession();
    
    console.log('Equipment auto-fill data cleared');
}

// Validate equipment fields after selection
function validateEquipmentFields() {
    console.log('Validating equipment fields...');
    
    const equipmentFields = [
        'equipmentDisplay',
        'vehicleNumber', 
        'vehicleName', 
        'vinNumber',
        'meterReading'
    ];
    
    let allValid = true;
    
    equipmentFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            const hasValue = field.value && field.value.trim() !== '';
            field.classList.toggle('is-invalid', !hasValue);
            
            if (!hasValue) {
                allValid = false;
            }
            
            console.log(`Field ${fieldId}:`, {
                value: field.value,
                hasValue: hasValue,
                isInvalid: field.classList.contains('is-invalid')
            });
        }
    });
    
    console.log('Equipment validation result:', allValid);
    return allValid;
}

// Check Step 1 validation status
function checkStep1Validation() {
    console.log('Checking Step 1 validation status...');
    
    const requiredFields = [
        'country',
        'project', 
        'inspector',
        'equipmentDisplay',
        'vehicleNumber',
        'vehicleName',
        'vinNumber',
        'meterReading',
        'inspectionChecklist'
    ];
    
    let allValid = true;
    const invalidFields = [];
    
    requiredFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            const hasValue = field.value && field.value.trim() !== '';
            field.classList.toggle('is-invalid', !hasValue);
            
            if (!hasValue) {
                allValid = false;
                invalidFields.push(fieldId);
            }
        }
    });
    
    console.log('Step 1 validation result:', {
        allValid: allValid,
        invalidFields: invalidFields
    });
    
    return allValid;
}

// Enhanced equipment selection with better data handling
function selectEquipmentFromModal(equipmentItem) {
    console.log('Selecting equipment from modal:', equipmentItem);
    
    const id = equipmentItem.id || equipmentItem._id || 'unknown';
    const name = equipmentItem.equipmentName || equipmentItem.vehicleNumber || 'Unknown Equipment';
    const model = equipmentItem.model || 'N/A';
    const category = equipmentItem.mainCategory || equipmentItem.equipmentCategory || 'N/A';
    
    // Call the main selectEquipment function
    selectEquipment(id, name, model, category);
}

// Select checklist (with custom checklist handling)
function selectChecklist(id, title, isCustom = false) {
    console.log('📋 Selecting checklist:', { id, title, isCustom });
    
    // Find the full checklist object from checklistData
    const selectedChecklistObj = checklistData.find(c => c.id === id);
    
    if (selectedChecklistObj) {
        console.log('✅ Found checklist object:', selectedChecklistObj);
        
        // Store the full checklist object for later use
        window.selectedChecklistData = selectedChecklistObj;
        
        // Update display
        const displayTitle = isCustom ? `${title} (CUSTOM)` : title;
        document.getElementById('checklistDisplay').value = displayTitle;
    document.getElementById('inspectionChecklist').value = id;
        
    closeChecklistModal();
        
        console.log('✅ Checklist selected and stored');
    } else {
        console.error('❌ Checklist not found in checklistData');
        alert('Error: Could not load checklist data');
    }
}

// Filter functions
 function filterEquipment() {
     const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
     const filtered = equipmentData.filter(item => {
         // Use the same display logic for filtering
         const displayName = item.equipmentName || item.vehicleNumber || item.model || 'Unknown Equipment';
         const category = item.mainCategory || item.equipmentCategory || 'N/A';
         const model = item.model || 'N/A';
         
         return displayName.toLowerCase().includes(searchTerm) ||
                category.toLowerCase().includes(searchTerm) ||
                model.toLowerCase().includes(searchTerm);
     });
     renderEquipmentList(filtered);
 }

function filterChecklist() {
    const searchTerm = document.getElementById('checklistSearch').value.toLowerCase();
    const filtered = checklistData.filter(item => 
        (item.document_name || item.title || '').toLowerCase().includes(searchTerm) ||
        (item.document_id || item.id || '').toLowerCase().includes(searchTerm)
    );
    renderChecklistList(filtered);
}

// Close modals when clicking outside
window.onclick = function(event) {
    const equipmentModal = document.getElementById('equipmentModal');
    const checklistModal = document.getElementById('checklistModal');
    const noteModal = document.getElementById('noteModal');
    const imageModal = document.getElementById('imageModal');
    const errorModal = document.getElementById('errorModal');
    const loadingModal = document.getElementById('loadingModal');
    const successModal = document.getElementById('successModal');
    
    if (event.target === equipmentModal) {
        closeEquipmentModal();
    }
    if (event.target === checklistModal) {
        closeChecklistModal();
    }
    if (event.target === noteModal) {
        closeNoteModal();
    }
    if (event.target === imageModal) {
        closeImageModal();
    }
    if (event.target === errorModal) {
        closeErrorModal();
    }
    if (event.target === loadingModal) {
        // Don't allow closing loading modal by clicking outside
        return;
    }
    if (event.target === successModal) {
        // Don't allow closing success modal by clicking outside
        return;
    }
}

// Form submission handling
document.addEventListener('DOMContentLoaded', function() {
    const inspectionForm = document.getElementById('inspection-form');
    const submitInspectionBtn = document.getElementById('submitInspectionBtn');

    if (inspectionForm) {
        inspectionForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Validate all required fields
            if (!validateInspectionForm()) {
                return;
            }
            
            // Disable submit button to prevent double submission
            submitInspectionBtn.disabled = true;
            submitInspectionBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>Generating Report...';
            
            // Get form data
            const formData = new FormData(this);
            
            // Convert FormData to JSON
            const jsonData = {};
            formData.forEach((value, key) => {
                if (value !== '') {
                    jsonData[key] = value;
                }
            });

            // Add additional data that might not be in the form
            jsonData.reportNumber = document.getElementById('reportNumber').value;
            jsonData.coordinates = document.getElementById('coordinatesInput').value || '';
            jsonData.gpsAddress = document.getElementById('gpsAddressInput').value || '';
            jsonData.currentMeterReading = (document.getElementById('currentMeterValue') ? document.getElementById('currentMeterValue').textContent : '');

            // Add inspection items data (if any)
            jsonData.inspectionItems = getInspectionItemsData();

            // Send AJAX request
            fetch('/inspection/store-inspection', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('input[name="_token"]').value
                },
                body: JSON.stringify(jsonData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show success message
                    alert('Inspection report generated successfully! Index: ' + data.index);
                    
                    // Reset form
                    inspectionForm.reset();
                    
                    // Redirect to inspection list
                    window.location.href = '/inspection';
                } else {
                    // Show error message
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while generating the inspection report. Please try again.');
            })
            .finally(() => {
                // Re-enable submit button
                submitInspectionBtn.disabled = false;
                submitInspectionBtn.innerHTML = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24"><g fill="none" stroke="#fff" stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"><path d="M20 12V5.749a.6.6 0 0 0-.176-.425l-3.148-3.148A.6.6 0 0 0 16.252 2H4.6a.6.6 0 0 0-.6.6v18.8a.6.6 0 0 0 .6.6H11M8 10h8M8 6h4m-4 8h3m9.5 6.5L22 22" /><path d="M15 18a3 3 0 1 0 6 0a3 3 0 0 0-6 0m1-16v3.4a.6.6 0 0 0 .6.6H20" /></g></svg>Generate Report';
            });
        });
    }
});

function getInspectionItemsData() {
    console.log('=== COLLECTING ALL STEP2 FIELD DATA ===');
    
    // Collect all data from step2
    const step2Data = {
        // Checklist information
        checklist: {
            id: currentChecklist?.document_id || currentChecklist?.id || '',
            name: currentChecklist?.document_name || currentChecklist?.title || '',
            type: currentChecklist?.type || ''
        },
        
        // Sections and items
        sections: currentSections || [],
        
        // User selections for each item
        selections: inspectionSelections || {},
        
        // Header input values
        headerInputs: headerInputValues || {},
        
        // Notes for each item
        notes: inspectionNotes || {},
        
        // Images for each item
        images: inspectionImages || {},
        
        // Calculated values
        faultyItemsCount: loadFaultyItemsCount(),
        
        // Timestamp
        timestamp: new Date().toISOString()
    };
    
    // Log detailed breakdown
    console.log('📋 Checklist Info:', step2Data.checklist);
    console.log('📝 Header Inputs:', step2Data.headerInputs);
    console.log('✅ User Selections:', step2Data.selections);
    console.log('📄 Notes:', step2Data.notes);
    console.log('🖼️ Images:', Object.keys(step2Data.images).length + ' images');
    console.log('❌ Faulty Items Count:', step2Data.faultyItemsCount);
    console.log('📊 Total Sections:', step2Data.sections.length);
    
    // Log each section with its items
    step2Data.sections.forEach((section, sectionIndex) => {
        console.log(`\n📁 Section ${sectionIndex + 1}: ${section.name}`);
        console.log(`   Items: ${section.subItems?.length || 0}`);
        
        if (section.subItems) {
            section.subItems.forEach((item, itemIndex) => {
                const selection = step2Data.selections[item.key] || 'Not Selected';
                const note = step2Data.notes[item.key] || 'No Note';
                const hasImage = !!step2Data.images[item.key];
                
                console.log(`   ${itemIndex + 1}. ${item.label || item.name}`);
                console.log(`      Selection: ${selection}`);
                console.log(`      Note: ${note}`);
                console.log(`      Image: ${hasImage ? 'Yes' : 'No'}`);
            });
        }
    });
    
    // Log header inputs in detail
    console.log('\n📝 Header Input Details:');
    Object.entries(step2Data.headerInputs).forEach(([key, value]) => {
        const label = headerInputs[key] || key;
        console.log(`   ${label}: ${value}`);
    });
    
    console.log('\n=== STEP2 DATA COLLECTION COMPLETE ===');
    console.log('Full Step2 Data Object:', step2Data);
    
    return step2Data;
}

function validateInspectionForm() {
    const requiredFields = [
        'country', 'project', 'inspector', 'vehicleNumber', 'vehicleName', 
        'vinNumber', 'meterReading', 'meterUnit', 'inspectionChecklist'
    ];
    
    let isValid = true;
    
    requiredFields.forEach(fieldName => {
        const field = document.querySelector(`[name="${fieldName}"]`);
        if (field && !field.value.trim()) {
            field.classList.add('is-invalid');
            isValid = false;
        } else if (field) {
            field.classList.remove('is-invalid');
        }
    });
    
    // Validate meter reading specifically
    if (!validateMeterReading()) {
        isValid = false;
    }
    
    if (!isValid) {
        alert('Please fill in all required fields and ensure the new meter reading is greater than the current reading.');
    }
    
    return isValid;
}

// Add Bootstrap JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize any Bootstrap components if needed
    console.log('Bootstrap initialized');
});

// Global variables for Step 2
let currentChecklist = null;
let currentSections = [];
let expandedSections = {};
let inspectionSelections = {};
let headerInputs = {};
let headerInputValues = {};
let headerInputErrors = {};
let showHeaderErrors = false;
let currentNoteItem = null;
let currentImageItem = null;
let inspectionNotes = {};
let inspectionImages = {};

// Flag to prevent duplicate signature field generation
let signatureFieldsGenerated = false;

// Session management for inspection data
const SESSION_KEY = 'inspection_form_session';
const SESSION_EXPIRY = 24 * 60 * 60 * 1000; // 24 hours in milliseconds

// Session Management Functions
function saveToSession() {
    try {
        const sessionData = {
            timestamp: Date.now(),
            step1: collectStep1Data(),
            step2: collectStep2Data(),
            step3: collectStep3Data(),
            currentStep: getCurrentStepFromURL(),
            inspectionSelections: window.inspectionSelections || {},
            checklistSelections: window.checklistSelections || {},
            step3Data: window.step3Data || {},
            currentChecklist: currentChecklist,
            currentSections: currentSections,
            inspectionNotes: inspectionNotes,
            inspectionImages: inspectionImages,
            headerInputValues: headerInputValues,
            expandedSections: expandedSections
        };

        localStorage.setItem(SESSION_KEY, JSON.stringify(sessionData));
        console.log('Data saved to session:', sessionData);
        
        // Show session save indicator
        showSessionSaveIndicator();
        
        return true;
    } catch (error) {
        console.error('Error saving to session:', error);
        return false;
    }
}

function loadFromSession() {
    try {
        const sessionData = localStorage.getItem(SESSION_KEY);
        if (!sessionData) {
            console.log('No session data found');
            return null;
        }

        const data = JSON.parse(sessionData);
        
        // Check if session has expired
        if (Date.now() - data.timestamp > SESSION_EXPIRY) {
            console.log('Session expired, clearing data');
            clearSession();
            return null;
        }

        console.log('Loading session data:', data);
        return data;
    } catch (error) {
        console.error('Error loading from session:', error);
        return null;
    }
}

function clearSession() {
    try {
        localStorage.removeItem(SESSION_KEY);
        console.log('Session cleared');
        
        // Reset all form data
        resetAllFormData();
        
        // Show session clear indicator
        showSessionClearIndicator();
        
        return true;
    } catch (error) {
        console.error('Error clearing session:', error);
        return false;
    }
}

function restoreFromSession() {
    const sessionData = loadFromSession();
    if (!sessionData) {
        console.log('No session data to restore');
        return false;
    }

    try {
        console.log('Restoring from session...');

        // Restore global variables
        if (sessionData.inspectionSelections) {
            window.inspectionSelections = sessionData.inspectionSelections;
        }
        if (sessionData.checklistSelections) {
            window.checklistSelections = sessionData.checklistSelections;
        }
        if (sessionData.step3Data) {
            window.step3Data = sessionData.step3Data;
        }
        if (sessionData.currentChecklist) {
            currentChecklist = sessionData.currentChecklist;
        }
        if (sessionData.currentSections) {
            currentSections = sessionData.currentSections;
        }
        if (sessionData.inspectionNotes) {
            inspectionNotes = sessionData.inspectionNotes;
        }
        if (sessionData.inspectionImages) {
            inspectionImages = sessionData.inspectionImages;
        }
        if (sessionData.headerInputValues) {
            headerInputValues = sessionData.headerInputValues;
        }
        if (sessionData.expandedSections) {
            expandedSections = sessionData.expandedSections;
        }

        // Restore form data for each step
        if (sessionData.step1) {
            restoreStep1Data(sessionData.step1);
        }
        if (sessionData.step2) {
            restoreStep2Data(sessionData.step2);
        }
        if (sessionData.step3) {
            restoreStep3Data(sessionData.step3);
        }

        // Restore signature type selections
        if (sessionData.step3Data && sessionData.step3Data.selectedSignatureTypes) {
            restoreSignatureTypeSelections();
        }

        console.log('Session data restored successfully');
        return true;
    } catch (error) {
        console.error('Error restoring from session:', error);
        return false;
    }
}

function resetAllFormData() {
    // Reset step 1 data
    document.getElementById('country').value = '';
    document.getElementById('project').value = '';
    document.getElementById('inspector').value = '';
    document.getElementById('equipmentDisplay').value = '';
    document.getElementById('vehicleNumber').value = '';
    document.getElementById('vehicleName').value = '';
    document.getElementById('vinNumber').value = '';
    document.getElementById('meter-Reading-input').value = '';
    document.getElementById('meterReading').value = '';
    document.getElementById('inspectionChecklist').value = '';
    document.getElementById('checklistDisplay').value = '';
    document.getElementById('coordinatesInput').value = '';
    document.getElementById('gpsAddressInput').value = '';

    // Reset step 2 data
    window.inspectionSelections = {};
    window.checklistSelections = {};
    inspectionNotes = {};
    inspectionImages = {};
    headerInputValues = {};
    expandedSections = {};
    currentChecklist = null;
    currentSections = [];

    // Reset step 3 data
    if (window.step3Data) {
        window.step3Data = {
            overallCondition: '',
            equipmentStatus: '',
            safeToUse: '',
            additionalNote: '',
            signatureFields: [],
            selectedSignatureTypes: [],
            signatureValues: {},
            individualSignatures: {},
            individualSignatureStatus: {}
        };
    }

    // Clear signature fields
    clearSignatureFields();
    
    // Reset signature type text
    const selectedTextElement = document.getElementById('selectedSignatureTypeText');
    if (selectedTextElement) {
        selectedTextElement.textContent = 'Select Signature Type';
    }

    console.log('All form data reset');
}

function showSessionSaveIndicator() {
    const indicator = document.getElementById('autoSaveIndicator');
    if (indicator) {
        indicator.textContent = '✓ Session saved';
        indicator.classList.add('show');
        setTimeout(() => {
            indicator.classList.remove('show');
            indicator.textContent = '✓ Auto-saved';
        }, 2000);
    }
}

function showSessionClearIndicator() {
    const indicator = document.getElementById('autoSaveIndicator');
    if (indicator) {
        indicator.textContent = '✓ Session cleared';
        indicator.classList.add('show');
        setTimeout(() => {
            indicator.classList.remove('show');
            indicator.textContent = '✓ Auto-saved';
        }, 2000);
    }
}


// Load checklist data for Step 2 (SUPPORTS Global + Custom Checklists)
async function loadChecklistDataForStep2() {
    try {
        const selectedChecklistId = document.getElementById('inspectionChecklist').value;
        if (!selectedChecklistId) {
            console.warn('No checklist selected');
            return;
        }

        console.log('📋 Loading checklist for Step 2, ID:', selectedChecklistId);
        
        // Use the stored checklist data from selection
        let checklist = window.selectedChecklistData;
        
        // If not found in window, fetch from API
        if (!checklist) {
            console.log('📥 Fetching checklist from API...');
            const response = await fetch('/api/fetch-checklists-from-firebase');
            const data = await response.json();
            
            if (data.success) {
                checklist = data.checklists.find(c => c.document_id === selectedChecklistId || c.id === selectedChecklistId);
            }
        }
        
        if (checklist) {
            currentChecklist = checklist;
            const checklistTitle = checklist.document_name || checklist.title || 'Untitled Checklist';
            document.getElementById('checklistTitleDisplay').textContent = checklistTitle;
            
            console.log('✅ Checklist loaded:', checklistTitle);
            console.log('📊 Checklist type:', checklist.type || checklist.isCustom ? 'custom' : 'global');
            
            // Load header inputs
            if (checklist.header_inputs) {
                headerInputs = checklist.header_inputs;
                renderHeaderInputs();
                autoFillHeaderInputs();
            }
            
            // Load sections - HANDLE BOTH GLOBAL AND CUSTOM CHECKLISTS
            if (checklist.checklist) {
                // Global checklist format: { A: [...], B: [...], C: [...] }
                console.log('📋 Loading GLOBAL checklist sections');
                currentSections = processChecklistSections(checklist.checklist);
                renderInspectionSections();
            } else if (checklist.items && Array.isArray(checklist.items)) {
                // Custom checklist format: { items: [...] }
                console.log('📋 Loading CUSTOM checklist items');
                currentSections = processCustomChecklistItems(checklist.items);
                renderInspectionSections();
            } else {
                console.error('❌ Checklist has no valid data structure');
                console.log('Checklist object:', checklist);
            }
            
            console.log('✅ Step 2 sections loaded:', currentSections.length, 'sections');
        } else {
            console.error('❌ Checklist not found for ID:', selectedChecklistId);
        }
    } catch (error) {
        console.error('Error loading checklist data for Step 2:', error);
    }
}

// Process custom checklist items (for custom checklists)
function processCustomChecklistItems(items) {
    console.log('📋 Processing custom checklist items:', items);
    
    if (!items || !Array.isArray(items)) {
        console.error('❌ Invalid custom checklist items');
        return [];
    }
    
    // Group items into a single section for custom checklists
    const subItems = items.map((item, index) => ({
        key: `custom_item_${index}`,
        itemName: item.itemName || item.name || item.label || `Item ${index + 1}`,
        category: 'Custom',
        inputType: item.inputType || 'Okay/Faulty/N/A',
        sortOrder: index,
        isInspectionRequired: item.isRequired || item.isInspectionRequired || false,
        description: item.description || '',
        instruction: item.instruction || '',
        label: item.label || item.itemName || item.name
    }));
    
    console.log('✅ Processed custom items:', subItems.length, 'items');
    
    return [{
        key: 'custom_section',
        name: 'Checklist Items',
        subItems: subItems
    }];
}

// Process checklist sections from the checklist data (for global checklists)
function processChecklistSections(checklistData) {
    const sections = [];
    const sectionEntries = Object.entries(checklistData);
    
    sectionEntries.forEach(([sectionKey, sectionValue], idx) => {
        let subItems = [];
        
        // Handle array of strings
        if (Array.isArray(sectionValue) && sectionValue.length > 0 && typeof sectionValue[0] === 'string') {
            subItems = sectionValue.map((item, i) => ({
                key: `${sectionKey}_${i}`,
                itemName: item,
                category: sectionKey,
                inputType: 'Okay/Faulty/N/A',
                sortOrder: i,
                isInspectionRequired: false,
                description: '',
                instruction: ''
            }));
        }
        // Handle array of objects
        else if (Array.isArray(sectionValue) && sectionValue.length > 0 && typeof sectionValue[0] === 'object') {
            subItems = sectionValue.map((item, i) => ({
                key: `${sectionKey}_${i}`,
                itemName: item.itemName || item,
                category: sectionKey,
                inputType: item.inputType || 'Okay/Faulty/N/A',
                sortOrder: i,
                isInspectionRequired: item.isInspectionRequired || false,
                description: item.description || '',
                instruction: item.instruction || ''
            }));
        }
        
        if (subItems.length > 0) {
            sections.push({
                key: sectionKey,
                name: sectionKey,
                subItems: subItems
            });
        }
    });
    
    // Sort sections: A, B, C first, then the rest
    const sectionOrder = ['A', 'B', 'C'];
    const orderedSections = [];
    sectionOrder.forEach((key) => {
        const idx = sections.findIndex((s) => s.name === key);
        if (idx !== -1) {
            orderedSections.push(sections[idx]);
            sections.splice(idx, 1);
        }
    });
    
    return [...orderedSections, ...sections];
}

// Render header inputs
function renderHeaderInputs() {
    const container = document.getElementById('headerInputsContainer');
    if (!headerInputs || Object.keys(headerInputs).length === 0) {
        container.innerHTML = '<p class="empty-section-text">No header inputs found for this checklist</p>';
        return;
    }
    
    const html = Object.entries(headerInputs).map(([key, label]) => {
        const isDateField = isDateFieldKeyOrLabel(key, label);
        const isProjectSite = isProjectOrSiteField(key, label);
        
        // Skip date fields as they are auto-filled
        if (isDateField) {
            return '';
        }
        
        return `
            <div class="input-field-container">
                <label class="input-label">
                    ${formatHeaderLabel(label || key)} <span class="required-text">*</span>
                </label>
                <input type="text" 
                       class="form-control header-input-field ${isProjectSite ? 'disabled-input' : ''}" 
                       id="header_${key}" 
                       data-key="${key}"
                       value="${headerInputValues[key] || ''}"
                       placeholder="Enter ${formatHeaderLabel(label || key).toLowerCase()}"
                       ${isProjectSite ? 'readonly' : ''}
                       onchange="handleHeaderInputChange('${key}', this.value)">
                <div class="error-text" id="header_error_${key}" style="display: none;">This field is required</div>
            </div>
        `;
    }).join('');
    
    container.innerHTML = html;
}

// Render inspection sections
function renderInspectionSections() {
    const container = document.getElementById('inspectionSectionsContainer');
    
    if (!currentSections || currentSections.length === 0) {
        container.innerHTML = '<p class="empty-section-text">No sections available</p>';
        return;
    }
    
    const html = currentSections.map((section) => {
        const sectionItems = section.subItems || [];
        const selectedItemsCount = sectionItems.filter(item => inspectionSelections[item.key]).length;
        const totalItemsCount = sectionItems.length;
        
        return `
            <div class="inspection-section mb-3">
                <div class="section-header" onclick="toggleSection('${section.key}')">
                    <div class="section-header-content">
                        <i class="fas fa-chevron-${expandedSections[section.key] ? 'up' : 'down'} section-toggle-icon"></i>
                        <div class="section-header-text-container">
                            <span class="section-header-text">${formatSectionTitle(section.name)}</span>
                            <span class="section-item-count">(${selectedItemsCount}/${totalItemsCount} selected)</span>
                        </div>
                    </div>
                </div>
                <div class="section-content" id="section_${section.key}" style="display: ${expandedSections[section.key] ? 'block' : 'none'}">
                    ${section.subItems.map((item, index) => renderInspectionItem(item, index)).join('')}
                </div>
            </div>
        `;
    }).join('');
    
    container.innerHTML = html;
    
    // Update faulty count after rendering sections
    updateFaultyCount();
}

// Render individual inspection item
function renderInspectionItem(item, index) {
    const selectedOption = inspectionSelections[item.key] || '';
    const options = getOptionsForInputType(item.inputType);
    const itemNote = inspectionNotes[item.key] || '';
    const selectedFileUri = inspectionImages[item.key] || '';
    
    return `
        <div class="inspection-item mb-3">
            <div class="item-container">
                <div class="item-text">
                    ${index + 1}. ${formatItemNameUpper(item.itemName)}
                    ${item.isInspectionRequired ? '<span class="required-text"> (Required)</span>' : ''}
                </div>
                ${item.description ? `<div class="detail-text">Description: ${item.description}</div>` : ''}
                ${item.instruction ? `<div class="detail-text">Instruction: ${item.instruction}</div>` : ''}
                <div class="item-row">
                    <div class="options-container">
                        ${options.map(option => `
                            <button type="button" 
                                    class="btn option-button ${selectedOption === option ? 'selected-option-button' : ''} 
                                           ${getOptionButtonClass(option, selectedOption === option)}"
                                    onclick="handleSelectOption('${item.key}', '${option}')">
                                ${option}
                            </button>
                        `).join('')}
                    </div>
                    <div class="icons-container">
                        <button type="button" class="btn btn-outline-secondary btn-sm inspection-action-btn" onclick="handleNotePress('${item.key}')" title="Add Note">
                            <i class="fas fa-sticky-note"></i>
                        </button>
                        <button type="button" class="btn btn-outline-secondary btn-sm inspection-action-btn" onclick="handleFilePick('${item.key}')" title="Add Image">
                            <i class="fas fa-camera"></i>
                        </button>
                    </div>
                </div>
                ${itemNote ? `<div class="note-display"><strong>Note:</strong> ${itemNote}</div>` : ''}
                ${selectedFileUri ? `<div class="image-display"><img src="${selectedFileUri}" alt="Inspection Image" class="inspection-image"></div>` : ''}
            </div>
        </div>
    `;
}

// Helper functions
function isDateFieldKeyOrLabel(key, label) {
    const lowerKey = (key || '').toLowerCase();
    const lowerLabel = (label || '').toLowerCase();
    return lowerKey.includes('date') || lowerLabel.includes('date');
}

function isProjectOrSiteField(key, label) {
    const lowerKey = (key || '').toLowerCase();
    const lowerLabel = (label || '').toLowerCase();
    return (
        lowerKey.includes('project') ||
        lowerLabel.includes('project') ||
        lowerKey.includes('site') ||
        lowerLabel.includes('site') ||
        lowerKey.includes('location') ||
        lowerLabel.includes('location')
    );
}

function formatHeaderLabel(label) {
    if (!label) return '';
    return label
        .replace(/_/g, ' ')
        .split(' ')
        .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
        .join(' ');
}

function formatSectionTitle(title) {
    if (!title) return '';
    return title
        .toString()
        .replace(/_/g, ' ')
        .split(' ')
        .map(word => word ? word.charAt(0).toUpperCase() + word.slice(1).toLowerCase() : '')
        .join(' ');
}

function formatItemNameUpper(name) {
    if (!name && name !== 0) return '';
    return String(name).replace(/_/g, ' ').toUpperCase();
}

function getOptionsForInputType(inputType) {
    switch (inputType) {
        case 'Okay/Faulty/N/A':
            return ['Okay', 'Faulty', 'N/A'];
        case 'Ok/Reject/N/A':
            return ['Ok', 'Reject', 'N/A'];
        case 'Ok/Not Ok':
            return ['Ok', 'Not Ok'];
        default:
            if (typeof inputType === 'string' && inputType.includes('/')) {
                const options = inputType.split('/').map((opt) => opt.trim()).filter((opt) => opt);
                if (options.length > 0) {
                    return options;
                }
            }
            return ['Okay', 'Faulty', 'N/A'];
    }
}

function getOptionButtonClass(option, isSelected) {
    if (!isSelected) return '';
    
    if (option === 'Okay' || option === 'Ok') return 'ok-button';
    if (option === 'Faulty' || option === 'Reject' || option === 'Not Ok') return 'reject-button';
    if (option === 'N/A') return 'na-button';
    return 'custom-option-button';
}

// Event handlers
function toggleSection(sectionKey) {
    expandedSections[sectionKey] = !expandedSections[sectionKey];
    const sectionContent = document.getElementById(`section_${sectionKey}`);
    const toggleIcon = document.querySelector(`#section_${sectionKey}`).previousElementSibling.querySelector('.section-toggle-icon');
    
    if (expandedSections[sectionKey]) {
        sectionContent.style.display = 'block';
        toggleIcon.className = 'fas fa-chevron-up section-toggle-icon';
    } else {
        sectionContent.style.display = 'none';
        toggleIcon.className = 'fas fa-chevron-down section-toggle-icon';
    }
}

function handleSelectOption(itemKey, option) {
    inspectionSelections[itemKey] = option;
    renderInspectionSections(); // Re-render to update counts
    updateFaultyCount(); // Update faulty count in real-time
}

function handleHeaderInputChange(key, value) {
    headerInputValues[key] = value;
    
    // Validate in real-time
    const isEmpty = !value || value.trim() === '';
    headerInputErrors[key] = isEmpty;
    
    const errorElement = document.getElementById(`header_error_${key}`);
    if (errorElement) {
        errorElement.style.display = isEmpty ? 'block' : 'none';
    }
    
    const inputElement = document.getElementById(`header_${key}`);
    if (inputElement) {
        inputElement.classList.toggle('is-invalid', isEmpty);
    }
}

function handleNotePress(itemKey) {
    const selection = (inspectionSelections && inspectionSelections[itemKey]) || (checklistSelections && checklistSelections[itemKey]);
    if (!selection) {
        alert('Please make a selection first');
        return;
    }

    currentNoteItem = itemKey;
    const existingNote = inspectionNotes[itemKey] || '';
    document.getElementById('noteText').value = existingNote;
    openModal('noteModal');
}

function handleFilePick(itemKey) {
    const selection = (inspectionSelections && inspectionSelections[itemKey]) || (checklistSelections && checklistSelections[itemKey]);
    if (!selection) {
        alert('Please make a selection first');
        return;
    }

    currentImageItem = itemKey;
    openModal('imageModal');
}

// Note modal functions
function closeNoteModal() {
    closeModal('noteModal');
    currentNoteItem = null;
}

function saveNote() {
    if (!currentNoteItem) return;
    
    const noteText = document.getElementById('noteText').value.trim();
    if (noteText) {
        inspectionNotes[currentNoteItem] = noteText;
        renderInspectionSections(); // Re-render to show the note
    }
    
    closeNoteModal();
}

// Wrapper functions used by static checklist icons in Step 2
function openNoteModal(itemKey) {
    handleNotePress(itemKey);
}

function openImageModal(itemKey) {
    handleFilePick(itemKey);
}

// Image modal functions
function closeImageModal() {
    closeModal('imageModal');
    currentImageItem = null;
    document.getElementById('imagePreviewContainer').style.display = 'none';
    document.getElementById('saveImageBtn').style.display = 'none';
}

function takePicture() {
    // For web, we'll simulate taking a picture by opening the file input
    // In a real mobile app, this would use the camera
    document.getElementById('imageFileInput').click();
}

function chooseFromGallery() {
    document.getElementById('imageFileInput').click();
}

function handleImageFileSelect(event) {
    const file = event.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('imagePreview').src = e.target.result;
            document.getElementById('imagePreviewContainer').style.display = 'block';
            document.getElementById('saveImageBtn').style.display = 'inline-block';
        };
        reader.readAsDataURL(file);
    }
}

function saveImage() {
    if (!currentImageItem) return;
    
    const imageSrc = document.getElementById('imagePreview').src;
    if (imageSrc) {
        inspectionImages[currentImageItem] = imageSrc;
        renderInspectionSections(); // Re-render to show the image
    }
    
    closeImageModal();
}

// Auto-fill header inputs with equipment data
function autoFillHeaderInputs() {
    try {
        // Get equipment data from Step 1
        const selectedEquipment = {
            model: document.getElementById('vinNumber').value,
            equipmentName: document.getElementById('vehicleName').value,
            vehicleNumber: document.getElementById('vehicleNumber').value,
            meterReading: document.getElementById('meter-Reading-input').value
        };
        
        const project = document.getElementById('project').value;
        
        // Auto-fill based on field names
        Object.keys(headerInputs).forEach(key => {
            const lowerKey = key.toLowerCase();
            const lowerLabel = (headerInputs[key] || '').toLowerCase();
            
            // Auto-fill model
            if (lowerKey.includes('model') || lowerLabel.includes('model')) {
                headerInputValues[key] = selectedEquipment.model || '';
            }
            
            // Auto-fill equipment name
            if (lowerKey.includes('equipment') || lowerLabel.includes('equipment')) {
                headerInputValues[key] = selectedEquipment.equipmentName || '';
            }
            
            // Auto-fill fleet number
            if (lowerKey.includes('fleet') || lowerLabel.includes('fleet')) {
                headerInputValues[key] = selectedEquipment.vehicleNumber || '';
            }
            
            // Auto-fill meter reading
            if (lowerKey.includes('meter') || lowerKey.includes('odometer') || lowerLabel.includes('meter') || lowerLabel.includes('odometer')) {
                headerInputValues[key] = selectedEquipment.meterReading || '';
            }
            
            // Auto-fill project/site/location
            if (isProjectOrSiteField(key, headerInputs[key])) {
                headerInputValues[key] = project || '';
            }
        });
        
        // Set today's date for date fields
        const today = new Date().toLocaleDateString();
        Object.keys(headerInputs).forEach(key => {
            const label = headerInputs[key] || '';
            if (isDateFieldKeyOrLabel(key, label)) {
                headerInputValues[key] = today;
            }
        });
        
        renderHeaderInputs();
    } catch (error) {
        console.error('Error auto-filling header inputs:', error);
    }
}

// Validate header inputs
function validateHeaderInputs() {
    const headerInputKeys = Object.keys(headerInputs);
    let hasErrors = false;
    
    headerInputKeys.forEach(key => {
        const isEmpty = !headerInputValues[key] || headerInputValues[key].trim() === '';
        headerInputErrors[key] = isEmpty;
        if (isEmpty) hasErrors = true;
        
        const errorElement = document.getElementById(`header_error_${key}`);
        const inputElement = document.getElementById(`header_${key}`);
        
        if (errorElement) {
            errorElement.style.display = isEmpty ? 'block' : 'none';
        }
        if (inputElement) {
            inputElement.classList.toggle('is-invalid', isEmpty);
        }
    });
    
    if (hasErrors) {
        alert('Please fill in all header information fields before proceeding.');
        return false;
    }
    
    return true;
}

// Complete inspection and go to Step 3
function completeInspection() {
    if (!validateHeaderInputs()) {
        return;
    }
    
    // Save inspection data
    const inspectionData = {
        checklist: currentChecklist,
        sections: currentSections,
        selections: inspectionSelections,
        headerInputs: headerInputValues,
        notes: inspectionNotes,
        images: inspectionImages,
        timestamp: new Date().toISOString()
    };
    
    // Store in localStorage for Step 3
    localStorage.setItem('inspectionStep2Data', JSON.stringify(inspectionData));
    
    // Navigate to Step 3 using the new navigation function
    navigateToStep(3);
}

// Step 3 Functions - Now handled by navigateToStep(3)

function initializeStep3() {
    console.log('Initializing Step 3...');
    
    try {
        // Initialize status option handlers
        if (typeof initializeStatusOptions === 'function') {
            initializeStatusOptions();
        }
        if (typeof initializeSafetyOptions === 'function') {
            initializeSafetyOptions();
        }
        if (typeof initializePriorityOptions === 'function') {
            initializePriorityOptions();
        }
        
        // Update faulty count display
        updateFaultyCount();
        
        // Initialize any other step 3 specific functionality
        if (typeof resetStep3Form === 'function') {
            resetStep3Form();
        }
        
        // Ensure signature fields are properly initialized
        if (step3Data.selectedSignatureTypes && step3Data.selectedSignatureTypes.length > 0) {
            // Only generate if not already generated
            if (!signatureFieldsGenerated) {
                clearSignatureFields();
                setTimeout(() => {
                    generateSignatureFields();
                }, 100);
            }
        }
        
        console.log('Step 3 initialized successfully');
    } catch (error) {
        console.error('Error initializing Step 3:', error);
    }
}

function initializeStatusOptions() {
    const statusOptions = document.querySelectorAll('.status-option-btn');
    statusOptions.forEach(option => {
        option.addEventListener('click', function() {
            // Remove selected class from all options
            statusOptions.forEach(opt => opt.classList.remove('selected'));
            // Add selected class to clicked option
            this.classList.add('selected');
            
            // Show/hide priority section based on selection
            const priorityContainer = document.getElementById('priorityContainer');
            const selectedStatus = this.dataset.status;
            
            if (selectedStatus === 'Maintenance Required' || selectedStatus === 'Breakdown') {
                priorityContainer.style.display = 'block';
            } else {
                priorityContainer.style.display = 'none';
            }
        });
    });
}

function initializeSafetyOptions() {
    const safetyOptions = document.querySelectorAll('.safe-option-btn');
    safetyOptions.forEach(option => {
        option.addEventListener('click', function() {
            // Remove selected class from all options
            safetyOptions.forEach(opt => opt.classList.remove('selected'));
            // Add selected class to clicked option
            this.classList.add('selected');
        });
    });
}

function initializePriorityOptions() {
    const priorityOptions = document.querySelectorAll('.priority-btn');
    priorityOptions.forEach(option => {
        option.addEventListener('click', function() {
            // Remove selected class from all options
            priorityOptions.forEach(opt => opt.classList.remove('selected'));
            // Add selected class to clicked option
            this.classList.add('selected');
        });
    });
}

function handleStatusSelect(status) {
    // Find the status option and trigger click
    const statusOption = document.querySelector(`[data-status="${status}"]`);
    if (statusOption) {
        statusOption.click();
    }
}

function handlePrioritySelect(priority) {
    // Find the priority option and trigger click
    const priorityOption = document.querySelector(`[data-priority="${priority}"]`);
    if (priorityOption) {
        priorityOption.click();
    }
}

function handleSafeSelect(safe) {
    // Find the safety option and trigger click
    const safetyOption = document.querySelector(`[data-safe="${safe}"]`);
    if (safetyOption) {
        safetyOption.click();
    }
}


function showError(elementId, message) {
    const errorElement = document.getElementById(elementId);
    if (errorElement) {
        errorElement.textContent = message;
        errorElement.style.display = 'block';
    }
}

function clearAllErrors() {
    const errorElements = document.querySelectorAll('.error-text');
    errorElements.forEach(element => {
        element.style.display = 'none';
    });
}

    // Create button function - NOW CALLS NEW MongoDB FLOW (matches React Native)
    async function handleCreate() {
        try {
            console.log('=== CREATE BUTTON CLICKED - USING NEW MONGODB FLOW ===');
            
            // Store current form data in sessionStorage for handleGenerateReport to use
            const step1FormData = {
                requestNumber: document.getElementById('reportNumber')?.value || '',
                country: document.getElementById('country')?.value || '',
                project: document.getElementById('project')?.value || '',
                inspector: document.getElementById('inspector')?.value || '',
                selectedEquipment: {
                    vehicleNumber: document.getElementById('vehicleNumber')?.value || '',
                    equipmentName: document.getElementById('vehicleName')?.value || '',
                    model: document.getElementById('vinNumber')?.value || '',
                    meterReading: document.getElementById('meter-Reading-input')?.value || '0',
                    status: 'Available'
                },
                meterReading: document.getElementById('meter-Reading-input')?.value || '0',
                currentMeterReading: document.getElementById('currentMeterValue')?.textContent || 'Not selected',
                selectedChecklistId: document.getElementById('inspectionChecklist')?.value || '',
                checklistTitle: document.getElementById('checklistDisplay')?.value || '',
                coordinates: document.getElementById('coordinatesInput')?.value || '',
                gpsAddress: document.getElementById('gpsAddressInput')?.value || '',
                createdAt: new Date().toISOString()
            };
            
            const step2FormData = {
                sections: currentSections || [],
                selections: inspectionSelections || {},
                selectedFiles: inspectionImages || {},
                images: inspectionImages || {}, // Add for compatibility
                numericInputs: {},
                notes: inspectionNotes || {},
                checklistTitle: currentChecklist?.document_name || currentChecklist?.title || '',
                headerInputValues: headerInputValues || {},
                footerInputValues: {},
                faultyItems: loadFaultyItemsCount() || 0,
                timestamp: new Date().toISOString()
            };
            
            // Collect Step 3 data INCLUDING signatures
            const step3FormData = {
                faultyItems: loadFaultyItemsCount() || 0,
                overallCondition: document.getElementById('overallCondition-select')?.value || '',
                equipmentStatus: document.querySelector('.status-option-btn.selected')?.dataset.status || '',
                priority: document.querySelector('.priority-btn.selected')?.dataset.priority || '',
                safeToUse: document.querySelector('.safe-option-btn.selected')?.dataset.safe || '',
                additionalNote: document.getElementById('additionalNote')?.value || '',
                signatureStatus: step3Data.signatureStatus || false,
                signatureData: step3Data.signatureData || null,
                signatureFields: step3Data.signatureFields || [],
                signatureValues: step3Data.signatureValues || {},
                selectedSignatureType: step3Data.selectedSignatureType || '',
                selectedSignatureTypes: step3Data.selectedSignatureTypes || [],
                individualSignatures: step3Data.individualSignatures || {},
                individualSignatureStatus: step3Data.individualSignatureStatus || {}
            };
            
            console.log('📊 Step 2 Data to save:', step2FormData);
            console.log('📊 Step 3 Data to save:', step3FormData);
            console.log('🖊️ Signature Fields:', step3FormData.signatureFields);
            console.log('🖊️ Individual Signatures:', step3FormData.individualSignatures);
            
            // Save to sessionStorage
            sessionStorage.setItem('inspectionStep1', JSON.stringify(step1FormData));
            sessionStorage.setItem('inspectionStep2', JSON.stringify(step2FormData));
            sessionStorage.setItem('inspectionStep3', JSON.stringify(step3FormData));
            
            console.log('📊 Data saved to sessionStorage, calling handleGenerateReport...');
            
            // Call the new MongoDB flow (matches React Native exactly)
            await handleGenerateReport();
            
        } catch (error) {
            console.error('❌ Error in handleCreate:', error);
            alert(`Error: ${error.message}`);
        }
    }

// Old Firebase version removed - using MongoDB-only version above

// Loading Modal Functions
function showLoadingModal() {
    const loadingModal = document.getElementById('loadingModal');
    if (loadingModal) {
        openModal('loadingModal');
        // Reset loading steps
        updateLoadingStep(0);
    }
}

function hideLoadingModal() {
    closeModal('loadingModal');
}

// Update loading step progress
function updateLoadingStep(stepIndex) {
    const steps = document.querySelectorAll('.loading-step');
    steps.forEach((step, index) => {
        step.classList.remove('active', 'completed');
        if (index < stepIndex) {
            step.classList.add('completed');
        } else if (index === stepIndex) {
            step.classList.add('active');
        }
    });
}

// Success Modal Functions
function showSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        openModal('successModal');
    }
}

function hideSuccessModal() {
    closeModal('successModal');
}

// Navigation Functions (Legacy - redirects to new functions)
function goBackToReports() {
    goBackToInspectionHome();
}

function viewPDF() {
    viewGeneratedPDF();
}

// PDF URLs now stored directly in MongoDB inspection documents - no separate storage needed


// Request Number Management - MongoDB Only (No Firebase)

// Request number generation now handled by generateRequestNumber() function only

async function fetchAndIncrementRequestNumber() {
    try {
        console.log('═══════════════════════════════════════════════════════');
        console.log('📋 FINAL REQUEST NUMBER GENERATION (CREATE REPORT)');
        console.log('═══════════════════════════════════════════════════════');
        
        // Get country and project from form
        const country = document.getElementById('country').value;
        const project = document.getElementById('project').value;
        
        console.log('📍 Selected Country:', country);
        console.log('📍 Selected Project:', project);
        
        if (!country || !project) {
            console.error('❌ Country or project not selected');
            return `ZAM-KAN-IR-001`;
        }
        
        // Convert to 3-letter codes
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : country.substring(0, 3).toUpperCase();
        const projectCode = project.substring(0, 3).toUpperCase();
        
        console.log('🏷️ Country Code:', countryCode);
        console.log('🏷️ Project Code:', projectCode);
        
        // Fetch ACTUAL count from MongoDB
        console.log('📊 Fetching ACTUAL count from MongoDB for final number...');
        console.log('📊 API URL: https://api.titandrillingzm.com:6004/inspections/count');
        
        const response = await fetch('https://api.titandrillingzm.com:6004/inspections/count');
        const result = await response.json();
        
        console.log('📊 MongoDB Count API Response:', JSON.stringify(result, null, 2));
        
        let actualCount = 0;
        
        if (result && result.success === true) {
            if (typeof result.total === 'number') {
                actualCount = result.total;
                console.log('✅ ACTUAL MongoDB Count:', actualCount);
            } else if (typeof result.count === 'number') {
                actualCount = result.count;
                console.log('✅ ACTUAL MongoDB Count:', actualCount);
            } else {
                console.warn('⚠️ Response success but no valid count');
                actualCount = 0;
            }
        } else {
            console.error('❌ MongoDB API unsuccessful');
            actualCount = 0;
        }
        
        // Next number = Current count + 1
        const nextNumber = actualCount + 1;
        const finalRequestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}`;
        
        console.log('═══════════════════════════════════════════════════════');
        console.log('✅ FINAL REQUEST NUMBER FOR NEW REPORT');
        console.log('═══════════════════════════════════════════════════════');
        console.log('📊 Current MongoDB Total:', actualCount, 'reports');
        console.log('🔢 Next Report Number:', nextNumber);
        console.log('📋 Final Request Number:', finalRequestNumber);
        console.log('📝 This will be report #' + nextNumber + ' in MongoDB');
        console.log('═══════════════════════════════════════════════════════');
        
        return finalRequestNumber;
        
    } catch (error) {
        console.error('❌ Error fetching/incrementing request number:', error);
        console.error('❌ Error details:', error.message);
        console.error('❌ Stack:', error.stack);
        
        // Fallback based on current selections
        const country = document.getElementById('country').value || 'ZAM';
        const project = document.getElementById('project').value || 'KAN';
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : country.substring(0, 3).toUpperCase();
        const projectCode = project.substring(0, 3).toUpperCase();
        const fallback = `${countryCode}-${projectCode}-IR-001`;
        
        console.log('⚠️ Using fallback request number:', fallback);
        return fallback;
    }
}

// Firebase functions removed - using MongoDB only

// Function to collect all inspection data in the exact format as reference JSON
function collectInspectionData() {
    // Get all form data
    const formData = new FormData(document.querySelector('form'));
    const data = Object.fromEntries(formData.entries());
    
    // Generate unique ID and request number
    const uniqueId = Date.now().toString();
    const requestNumber = getActualValue('requestNumber') || `REQ-${uniqueId}`;
    
    // Structure data to match the reference JSON format exactly
    const inspectionData = {
        id: uniqueId,
        
        // Step 1: Header Information
        step1: {
            checklistTitle: getActualValue('checklistTitle') || 'Inspection Checklist',
            coordinates: getActualValue('coordinates') || '',
            country: getActualValue('country') || '',
            createdAt: new Date().toISOString(),
            currentMeterReading: getActualValue('currentMeterReading') || '',
            gpsAddress: getActualValue('gpsAddress') || '',
            id: uniqueId,
            inspector: getActualValue('inspector') || '',
            meterReading: getActualValue('meterReading') || '',
            project: getActualValue('project') || '',
            requestNumber: requestNumber,
            selectedChecklistId: getActualValue('selectedChecklistId') || 'CHECKLIST-001',
            selectedEquipment: {
                country: getActualValue('country') || '',
                createdAt: new Date().toISOString(),
                dynamicField1: getActualValue('dynamicField1') || '',
                dynamicField2: getActualValue('dynamicField2') || '',
                dynamicField3: getActualValue('dynamicField3') || '',
                engineNumber: getActualValue('engineNumber') || '',
                equipmentCategory: getActualValue('equipmentCategory') || 'Vehicle',
                equipmentName: getActualValue('equipmentName') || '',
                equipmentNumber: getActualValue('equipmentNumber') || '',
                id: `equipment-${uniqueId}`,
                mainCategory: getActualValue('mainCategory') || 'Vehicles',
                make: getActualValue('make') || '',
                meterReading: parseInt(getActualValue('meterReading')) || 0,
                meterUnit: getActualValue('meterUnit') || 'Miles',
                model: getActualValue('model') || '',
                regNum: getActualValue('regNum') || '',
                status: getActualEquipmentStatus(),
                updatedAt: new Date().toISOString(),
                vin: getActualValue('vin') || ''
            }
        },
        
        // Step 2: Checklist Data
        step2: {
            checklistTitle: getActualValue('checklistTitle') || 'Inspection Checklist',
            faultyItems: countFaultyItems(),
            headerInputValues: {
                date: getActualValue('date') || new Date().toISOString().split('T')[0],
                hour_reading: getActualValue('hour_reading') || '',
                pump_number: getActualValue('pump_number') || '',
                site_location: getActualValue('site_location') || getActualValue('project') || ''
            },
            inspectionId: uniqueId,
            numericInputs: collectActualNumericInputs(),
            sections: collectActualChecklistSections(),
            selections: collectActualSelections(),
            timestamp: new Date().toISOString()
        },
        
        // Step 3: Summary Data
        step3: {
            additionalNote: getActualValue('additionalNote') || '',
            createdAt: new Date().toISOString(),
            equipmentStatus: getActualEquipmentStatus(),
            faultyItems: countFaultyItems(),
            individualSignatureStatus: getActualSignatureStatus(),
            individualSignatures: getActualIndividualSignatures(),
            overallCondition: getActualOverallCondition(),
            pdfDownloadUrl: '', // Will be set after PDF generation
            priority: getActualValue('priority') || 'Medium',
            safeToUse: getActualSafeToUseStatus(),
            selectedSignatureType: getActualValue('selectedSignatureType') || '',
            selectedSignatureTypes: getActualSelectedSignatureTypes(),
            signatureFields: getActualSignatureFields(),
            signatureStatus: getActualOverallSignatureStatus(),
            signatureValues: getActualSignatureValues()
        }
    };
    
    return inspectionData;
}

// Helper function to get actual value from any form element
function getActualValue(fieldName) {
    // Try by ID first
    let element = document.getElementById(fieldName);
    if (element && element.value !== undefined && element.value !== '') {
        return element.value;
    }
    
    // Try by name attribute
    element = document.querySelector(`[name="${fieldName}"]`);
    if (element && element.value !== undefined && element.value !== '') {
        return element.value;
    }
    
    // Try by data attribute
    element = document.querySelector(`[data-field="${fieldName}"]`);
    if (element && element.value !== undefined && element.value !== '') {
        return element.value;
    }
    
    // Try by class name
    element = document.querySelector(`.${fieldName}`);
    if (element && element.value !== undefined && element.value !== '') {
        return element.value;
    }
    
    return '';
}

// Helper function to get actual equipment status
function getActualEquipmentStatus() {
    const statusButton = document.querySelector('.status-option-btn.selected');
    if (statusButton && statusButton.dataset.status) {
        return statusButton.dataset.status;
    }
    
    // Try other status selectors
    const statusSelectors = [
        'input[name="equipmentStatus"]:checked',
        'select[name="equipmentStatus"]',
        'input[name="status"]:checked',
        'select[name="status"]',
        '.equipment-status.selected',
        '.status-selected'
    ];
    
    for (let selector of statusSelectors) {
        const element = document.querySelector(selector);
        if (element) {
            return element.value || element.dataset.status || element.textContent.trim();
        }
    }
    
    return 'Operational';
}

// Helper function to get actual overall condition
function getActualOverallCondition() {
    const conditionSelect = document.getElementById('overallCondition-select');
    if (conditionSelect && conditionSelect.value) {
        return conditionSelect.value;
    }
    
    // Try other condition selectors
    const conditionSelectors = [
        'select[name="overallCondition"]',
        'input[name="overallCondition"]:checked',
        'select[name="condition"]',
        'input[name="condition"]:checked',
        '.overall-condition.selected',
        '.condition-selected'
    ];
    
    for (let selector of conditionSelectors) {
        const element = document.querySelector(selector);
        if (element) {
            return element.value || element.dataset.condition || element.textContent.trim();
        }
    }
    
    return 'Good Condition';
}

// Helper function to get actual safe to use status
function getActualSafeToUseStatus() {
    const safeButton = document.querySelector('.safe-option-btn.selected');
    if (safeButton && safeButton.dataset.safe) {
        return safeButton.dataset.safe === 'Safe' ? 'Yes' : 'No';
    }
    
    // Try other safe/unsafe selectors
    const safeSelectors = [
        'input[name="safeToUse"]:checked',
        'select[name="safeToUse"]',
        'input[name="safe"]:checked',
        'select[name="safe"]',
        '.safe-option.selected',
        '.safety-selected'
    ];
    
    for (let selector of safeSelectors) {
        const element = document.querySelector(selector);
        if (element) {
            const value = element.value || element.dataset.safe || element.textContent.trim();
            return value === 'Safe' || value === 'Yes' ? 'Yes' : 'No';
        }
    }
    
    return 'No';
}

// Helper function to get actual signature status
function getActualSignatureStatus() {
    const status = {};
    
    // Look for all signature fields
    const signatureInputs = document.querySelectorAll('input[name*="signature"], textarea[name*="signature"], canvas[data-signature]');
    signatureInputs.forEach(input => {
        let fieldName = input.name;
        if (!fieldName && input.dataset.signature) {
            fieldName = input.dataset.signature;
        }
        
        if (fieldName) {
            let hasValue = false;
            if (input.type === 'text' || input.type === 'textarea') {
                hasValue = input.value && input.value.trim() !== '';
            } else if (input.tagName === 'CANVAS') {
                hasValue = !input.toDataURL().includes('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==');
            }
            
            status[fieldName] = hasValue;
        }
    });
    
    return status;
}

// Helper function to get actual individual signatures
function getActualIndividualSignatures() {
    const signatures = {};
    
    // Look for all signature fields
    const signatureInputs = document.querySelectorAll('input[name*="signature"], textarea[name*="signature"], canvas[data-signature]');
    signatureInputs.forEach(input => {
        let fieldName = input.name;
        if (!fieldName && input.dataset.signature) {
            fieldName = input.dataset.signature;
        }
        
        if (fieldName) {
            if (input.type === 'text' || input.type === 'textarea') {
                signatures[fieldName] = input.value || '';
            } else if (input.tagName === 'CANVAS') {
                signatures[fieldName] = input.toDataURL();
            }
        }
    });
    
    return signatures;
}

// Helper function to get actual selected signature types
function getActualSelectedSignatureTypes() {
    const types = [];
    
    // Look for checked signature type checkboxes
    const signatureTypeCheckboxes = document.querySelectorAll('input[name*="signatureType"]:checked, input[name*="signature_type"]:checked');
    signatureTypeCheckboxes.forEach(checkbox => {
        types.push(checkbox.value);
    });
    
    // If no checkboxes found, look for signature fields that exist
    if (types.length === 0) {
        const signatureFields = document.querySelectorAll('input[name*="signature"], canvas[data-signature]');
        if (signatureFields.length > 0) {
            types.push('mechanic'); // Default
        }
    }
    
    return types;
}

// Helper function to get actual signature fields
function getActualSignatureFields() {
    const fields = [];
    
    // Look for signature field configurations
    const signatureConfigs = document.querySelectorAll('[data-signature-field]');
    signatureConfigs.forEach(config => {
        const field = {
            key: config.dataset.key || 'signature',
            label: config.dataset.label || 'Signature',
            nameKey: config.dataset.nameKey || 'signature_name',
            originalData: {
                conf: {
                    nameKey: config.dataset.nameKey || 'signature_name',
                    signatureKey: config.dataset.signatureKey || 'signature'
                },
                groupKey: config.dataset.groupKey || 'signature'
            },
            required: config.dataset.required === 'true',
            roleName: config.dataset.roleName || 'User',
            showInput: config.dataset.showInput !== 'false',
            showSignature: config.dataset.showSignature !== 'false',
            signatureKey: config.dataset.signatureKey || 'signature'
        };
        fields.push(field);
    });
    
    // If no configured fields found, create default based on actual signature fields
    if (fields.length === 0) {
        const signatureFields = document.querySelectorAll('input[name*="signature"], canvas[data-signature]');
        if (signatureFields.length > 0) {
            fields.push({
                key: 'mechanic',
                label: 'Mechanic',
                nameKey: 'mechanic_name',
                originalData: {
                    conf: {
                        nameKey: 'mechanic_name',
                        signatureKey: 'mechanic_signature'
                    },
                    groupKey: 'mechanic'
                },
                required: true,
                roleName: 'Mechanic',
                showInput: true,
                showSignature: true,
                signatureKey: 'mechanic_signature'
            });
        }
    }
    
    return fields;
}

// Helper function to get actual overall signature status
function getActualOverallSignatureStatus() {
    const signatureInputs = document.querySelectorAll('input[name*="signature"], canvas[data-signature]');
    let hasSignature = false;
    
    signatureInputs.forEach(input => {
        if (input.type === 'text' || input.type === 'textarea') {
            if (input.value && input.value.trim()) {
                hasSignature = true;
            }
        } else if (input.tagName === 'CANVAS') {
            if (!input.toDataURL().includes('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==')) {
                hasSignature = true;
            }
        }
    });
    
    return hasSignature;
}

// Helper function to get actual signature values
function getActualSignatureValues() {
    const values = {};
    
    // Look for signature name fields
    const nameInputs = document.querySelectorAll('input[name*="name"]');
    nameInputs.forEach(input => {
        if (input.name.includes('signature') || input.name.includes('mechanic')) {
            values[input.name] = input.value || '';
        }
    });
    
    // Look for signature fields
    const signatureInputs = document.querySelectorAll('input[name*="signature"], textarea[name*="signature"], canvas[data-signature]');
    signatureInputs.forEach(input => {
        let fieldName = input.name;
        if (!fieldName && input.dataset.signature) {
            fieldName = input.dataset.signature;
        }
        
        if (fieldName) {
            if (input.type === 'text' || input.type === 'textarea') {
                values[fieldName] = input.value || '';
            } else if (input.tagName === 'CANVAS') {
                values[fieldName] = input.toDataURL();
            }
        }
    });
    
    return values;
}

// Helper function to count faulty items
function countFaultyItems() {
    const faultyItems = document.querySelectorAll('input[type="radio"][value="Reject"]:checked, input[type="radio"][value="Faulty"]:checked');
    return faultyItems.length;
}

// Helper function to collect actual numeric inputs
function collectActualNumericInputs() {
    const numericInputs = {};
    const numericFields = document.querySelectorAll('input[type="number"], input[type="text"][data-numeric="true"], input[type="text"][name*="numeric"]');
    numericFields.forEach(field => {
        if (field.value && field.value.trim() !== '') {
            numericInputs[field.name || field.id] = field.value;
        }
    });
    return numericInputs;
}

// Helper function to collect actual checklist sections
function collectActualChecklistSections() {
    const sections = [];
    
    // Look for common checklist section patterns
    const sectionSelectors = [
        '[data-section]',
        '.checklist-section',
        '.inspection-section',
        '.form-section',
        'fieldset',
        '.step-content',
        '.checklist-item-group',
        '.inspection-group'
    ];
    
    let sectionElements = [];
    sectionSelectors.forEach(selector => {
        const elements = document.querySelectorAll(selector);
        sectionElements = sectionElements.concat(Array.from(elements));
    });
    
    // Remove duplicates
    sectionElements = [...new Set(sectionElements)];
    
    sectionElements.forEach(section => {
        const sectionData = {
            key: section.getAttribute('data-section-key') || section.id || `section_${sections.length}`,
            name: section.getAttribute('data-section-name') || section.querySelector('legend')?.textContent?.trim() || `Section ${sections.length + 1}`,
            subItems: []
        };
        
        // Look for checklist items within this section
        const itemSelectors = [
            '[data-item]',
            '.checklist-item',
            '.inspection-item',
            '.form-group',
            '.input-group',
            'tr[data-item]',
            '.item-row',
            '.checklist-row',
            '.inspection-row'
        ];
        
        let subItems = [];
        itemSelectors.forEach(selector => {
            const items = section.querySelectorAll(selector);
            subItems = subItems.concat(Array.from(items));
        });
        
        // Remove duplicates
        subItems = [...new Set(subItems)];
        
        subItems.forEach(item => {
            // Get the actual selected value for this item
            const selectedValue = getSelectedValueForItem(item);
            
            const subItem = {
                category: item.getAttribute('data-category') || extractCategoryFromElement(item) || 'A',
                description: item.getAttribute('data-description') || '',
                inputType: item.getAttribute('data-input-type') || 'Ok/Reject/N/A',
                instruction: item.getAttribute('data-instruction') || '',
                isInspectionRequired: item.getAttribute('data-required') === 'true',
                itemName: item.getAttribute('data-item-name') || extractItemName(item),
                key: item.getAttribute('data-item-key') || item.id || `item_${subItems.indexOf(item)}`,
                sortOrder: parseInt(item.getAttribute('data-sort-order')) || subItems.indexOf(item) + 1,
                selectedValue: selectedValue, // Add the actual selected value
                isSelected: !!selectedValue
            };
            sectionData.subItems.push(subItem);
        });
        
        if (sectionData.subItems.length > 0) {
            sections.push(sectionData);
        }
    });
    
    return sections;
}

// Helper function to get selected value for a checklist item
function getSelectedValueForItem(item) {
    // Look for radio buttons within this item
    const radioButtons = item.querySelectorAll('input[type="radio"]:checked');
    if (radioButtons.length > 0) {
        return radioButtons[0].value;
    }
    
    // Look for checkboxes
    const checkboxes = item.querySelectorAll('input[type="checkbox"]:checked');
    if (checkboxes.length > 0) {
        return checkboxes[0].value || 'checked';
    }
    
    // Look for select dropdowns
    const selects = item.querySelectorAll('select');
    if (selects.length > 0 && selects[0].value) {
        return selects[0].value;
    }
    
    // Look for text inputs
    const textInputs = item.querySelectorAll('input[type="text"], input[type="number"], textarea');
    if (textInputs.length > 0 && textInputs[0].value) {
        return textInputs[0].value;
    }
    
    return null;
}

// Helper function to extract category from element
function extractCategoryFromElement(item) {
    // Look for category indicators in class names or data attributes
    const categoryMatch = item.className.match(/category-([A-Z])/i) || 
                         item.getAttribute('data-category') ||
                         item.closest('[data-category]')?.getAttribute('data-category');
    return categoryMatch || 'A';
}

// Helper function to extract item name
function extractItemName(item) {
    // Try to get item name from various sources
    const label = item.querySelector('label');
    if (label) {
        return label.textContent.trim();
    }
    
    const th = item.querySelector('th');
    if (th) {
        return th.textContent.trim();
    }
    
    const strong = item.querySelector('strong');
    if (strong) {
        return strong.textContent.trim();
    }
    
    // Get text content and clean it up
    const text = item.textContent.trim();
    if (text && text.length < 100) {
        return text;
    }
    
    return `Item ${item.id || 'Unknown'}`;
}

// Helper function to collect actual selections
function collectActualSelections() {
    const selections = {};
    
    // Collect radio button selections (Ok/Reject/N/A, etc.)
    const radioButtons = document.querySelectorAll('input[type="radio"]:checked');
    radioButtons.forEach(radio => {
        if (radio.name) {
            selections[radio.name] = radio.value;
        }
    });
    
    // Collect checkbox selections
    const checkboxes = document.querySelectorAll('input[type="checkbox"]:checked');
    checkboxes.forEach(checkbox => {
        if (checkbox.name) {
            selections[checkbox.name] = checkbox.value || 'checked';
        }
    });
    
    // Collect select dropdown selections
    const selects = document.querySelectorAll('select');
    selects.forEach(select => {
        if (select.value && select.name) {
            selections[select.name] = select.value;
        }
    });
    
    // Collect specific inspection checklist selections
    const inspectionRadios = document.querySelectorAll('input[type="radio"][name*="HSE-"], input[type="radio"][name*="inspection"], input[type="radio"][name*="checklist"]');
    inspectionRadios.forEach(radio => {
        if (radio.checked && radio.name) {
            selections[radio.name] = radio.value;
        }
    });
    
    // Collect status button selections
    const statusButtons = document.querySelectorAll('.status-option-btn.selected, .safe-option-btn.selected');
    statusButtons.forEach(button => {
        const status = button.dataset.status || button.dataset.safe;
        const name = button.dataset.name || button.className;
        if (status) {
            selections[name] = status;
        }
    });
    
    // Collect all form inputs with specific patterns
    const allInputs = document.querySelectorAll('input, select, textarea');
    allInputs.forEach(input => {
        if (input.name && input.value && !selections[input.name]) {
            // Skip if already collected
            if (input.type === 'radio' && !input.checked) return;
            if (input.type === 'checkbox' && !input.checked) return;
            
            selections[input.name] = input.value;
        }
    });
    
    return selections;
}

/**
 * Safely parse a value to a number
 * Handles strings like "Not selected", "22586 Miles", etc.
 */
function parseNumberSafely(value) {
    // If already a number, return it
    if (typeof value === 'number' && !isNaN(value)) {
        return value;
    }
    
    // If null, undefined, or "Not selected", return 0
    if (!value || value === 'Not selected' || value === 'N/A' || value === '') {
        return 0;
    }
    
    // If string, extract numbers only
    if (typeof value === 'string') {
        // Remove any text (Miles, Km, Hours, etc.) and keep only numbers and decimal point
        const cleaned = value.replace(/[^\d.]/g, '');
        const parsed = parseFloat(cleaned);
        
        // Return the parsed number or 0 if invalid
        return isNaN(parsed) ? 0 : parsed;
    }
    
    // Fallback
    return 0;
}

/**
 * Save inspection report to MongoDB with correct structure
 */
async function saveInspectionToMongoDB(inspectionData) {
    try {
        console.log('💾 Saving inspection to MongoDB...');
        console.log('📊 Inspection data:', inspectionData);
        
        // Get user ID from session or localStorage
        const userId = localStorage.getItem('userId') || 'test_proadmin_amit_001';
        
        // ALWAYS generate proper IDs (never undefined)
        const requestNumber = inspectionData.step1?.requestNumber || 
                             inspectionData.requestNumber || 
                             `INS-${Date.now()}`;
        
        const reportId = inspectionData._id || 
                        inspectionData.id || 
                        requestNumber || 
                        `report_${Date.now()}`;
        
        console.log('✅ Generated IDs:', { reportId, requestNumber, userId });
        
        // Structure data according to MongoDB schema
        const reportData = {
            _id: reportId,
            requestNumber: requestNumber,
            status: inspectionData.status || 'completed',
            
            // Step 1: Inspection Info
            step1: {
                id: inspectionData.step1?.id || reportId,
                requestNumber: requestNumber,
                country: inspectionData.step1?.country || '',
                project: inspectionData.step1?.project || '',
                inspector: inspectionData.step1?.inspector || '',
                selectedEquipment: inspectionData.step1?.selectedEquipment || {},
                meterReading: parseNumberSafely(inspectionData.step1?.meterReading),
                currentMeterReading: parseNumberSafely(inspectionData.step1?.currentMeterReading),
                selectedChecklistId: inspectionData.step1?.selectedChecklistId || '',
                checklistTitle: inspectionData.step1?.checklistTitle || '',
                coordinates: inspectionData.step1?.coordinates || '',
                gpsAddress: inspectionData.step1?.gpsAddress || '',
                createdAt: inspectionData.step1?.createdAt || new Date().toISOString()
            },
            
            // Step 2: Do Inspection
            step2: {
                selections: inspectionData.step2?.selections || {},
                selectedFiles: inspectionData.step2?.selectedFiles || {},
                numericInputs: inspectionData.step2?.numericInputs || {},
                notes: inspectionData.step2?.notes || {},
                sections: inspectionData.step2?.sections || [],
                checklistTitle: inspectionData.step2?.checklistTitle || inspectionData.step1?.checklistTitle || '',
                headerInputValues: inspectionData.step2?.headerInputValues || {},
                footerInputValues: inspectionData.step2?.footerInputValues || {},
                faultyItems: parseNumberSafely(inspectionData.step2?.faultyItems),
                timestamp: inspectionData.step2?.timestamp || new Date().toISOString()
            },
            
            // Step 3: Summary
            step3: {
                faultyItems: parseNumberSafely(inspectionData.step3?.faultyItems),
                overallCondition: inspectionData.step3?.overallCondition || '',
                equipmentStatus: inspectionData.step3?.equipmentStatus || '',
                priority: inspectionData.step3?.priority || '',
                safeToUse: inspectionData.step3?.safeToUse || '',
                additionalNote: inspectionData.step3?.additionalNote || '',
                signatureStatus: inspectionData.step3?.signatureStatus || '',
                signatureData: inspectionData.step3?.signatureData || {},
                signatureDownloadUrl: inspectionData.step3?.signatureDownloadUrl || '',
                signatureFields: inspectionData.step3?.signatureFields || [],
                signatureValues: inspectionData.step3?.signatureValues || {},
                selectedSignatureType: inspectionData.step3?.selectedSignatureType || '',
                selectedSignatureTypes: inspectionData.step3?.selectedSignatureTypes || [],
                individualSignatures: inspectionData.step3?.individualSignatures || {},
                individualSignatureStatus: inspectionData.step3?.individualSignatureStatus || {},
                pdfDownloadUrl: inspectionData.step3?.pdfDownloadUrl || '',
                createdAt: inspectionData.step3?.createdAt || new Date().toISOString()
            },
            
            createdAt: inspectionData.createdAt || new Date().toISOString(),
            updatedAt: new Date().toISOString()
        };
        
        console.log('📤 Sending to MongoDB API:', reportData);
        
        // Send to Laravel API which forwards to MongoDB
        const response = await fetch('/api/inspections/create', {
            method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify({
                userId: userId,
                reportData: reportData
            })
        });
        
        const result = await response.json();
        
        console.log('📊 Full API Response:', result);
        console.log('📊 Response status:', response.status);
        console.log('📊 Response ok:', response.ok);
        
        if (result.success) {
            console.log('✅ Inspection saved to MongoDB successfully!');
            console.log('📊 Saved Report ID:', result.reportId || result.data?._id);
            return result;
        } else {
            console.error('❌ Failed to save inspection!');
            console.error('❌ Error message:', result.message);
            console.error('❌ Error details:', result.error);
            console.error('❌ Full response:', result);
            
            // Show detailed error in alert
            const errorDetails = result.error || result.message || 'Unknown error';
            throw new Error(`Failed to save inspection: ${errorDetails}`);
        }
        
    } catch (error) {
        console.error('❌ Error saving inspection to MongoDB:', error);
        throw error;
    }
}

/**
 * Upload PDF to MongoDB server and return the FULL download URL with IP and port
 */
async function uploadPdfToMongoDB(pdfBlob, userId, requestNumber) {
    try {
        console.log('📤 Uploading PDF to MongoDB server...');
        
        // MongoDB server base URL with IP and port
        const MONGODB_SERVER_URL = 'https://api.titandrillingzm.com:6004';
        
        const formData = new FormData();
        formData.append('pdf', pdfBlob, `${requestNumber}.pdf`);
        
        const response = await fetch(`${MONGODB_SERVER_URL}/inspections/upload-pdf/${userId}`, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Construct FULL URL with IP and port
            const relativePath = result.downloadUrl; // e.g., /inspections/download-file/${userId}/${fileName}
            const fullDownloadUrl = `${MONGODB_SERVER_URL}${relativePath}`;
            
            console.log('✅ PDF uploaded successfully!');
            console.log('📎 Relative URL:', relativePath);
            console.log('📎 FULL Download URL:', fullDownloadUrl);
            
            return fullDownloadUrl; // Returns: https://api.titandrillingzm.com:6004/inspections/download-file/${userId}/${fileName}
        } else {
            console.error('❌ Failed to upload PDF:', result.error);
            throw new Error(result.error || 'Failed to upload PDF');
        }
        
    } catch (error) {
        console.error('❌ Error uploading PDF:', error);
        throw error;
    }
}

/**
 * Generate Report - MATCHES React Native Summerystep3.js (line 2182-2623)
 * This is the main function that orchestrates PDF generation and MongoDB saving
 */
async function handleGenerateReport() {
    try {
        console.log('🚀 Starting report generation (matching React Native flow)...');
        
        //  Show loading
        showLoadingModal('Generating Report...');
        
        // Get user ID
        const userId = '{{ auth()->user()->email ?? "test_proadmin_amit_001" }}';
        
        // Collect all data from sessionStorage (React Native uses AsyncStorage)
        const step1Data = JSON.parse(sessionStorage.getItem('inspectionStep1') || '{}');
        const step2Results = JSON.parse(sessionStorage.getItem('inspectionStep2') || '{}');
        const step3Saved = JSON.parse(sessionStorage.getItem('inspectionStep3') || '{}');
        
        // Generate report ID
        const reportId = Date.now().toString();
        
        // Get request number from step1
        const requestNumber = step1Data.requestNumber || `INS-${reportId}`;
        
        console.log('📝 Report Info:', { reportId, requestNumber, userId });
        console.log('📊 Step 1 Data:', step1Data);
        console.log('📊 Step 2 Results:', step2Results);
        console.log('📊 Step 3 Saved:', step3Saved);
        
        // Calculate faulty items (React Native line 2248-2257)
        const sections = step2Results.sections || [];
        const selections = step2Results.selections || {};
        const faultyCount = sections.flatMap((section) =>
            (section.subItems || []).filter((item) => 
                selections[item.key] === 'Faulty' || 
                selections[item.key] === 'Reject' || 
                selections[item.key] === 'Not Ok'
            )
        ).length;
        
        console.log('📊 Sections count:', sections.length);
        console.log('📊 Selections count:', Object.keys(selections).length);
        console.log('📊 Faulty count:', faultyCount);
        
        // Use Step 3 data from sessionStorage (already includes signatures)
        const step3DataCollected = {
            ...step3Saved,
            faultyItems: faultyCount,
            createdAt: new Date().toISOString()
        };
        
        console.log('📊 Final Step 3 Data:', step3DataCollected);
        console.log('📝 Signature Fields:', step3DataCollected.signatureFields);
        console.log('📝 Signature Values:', step3DataCollected.signatureValues);
        console.log('🖊️ Individual Signatures:', step3DataCollected.individualSignatures);
        console.log('🖊️ Individual Signature Status:', step3DataCollected.individualSignatureStatus);
        
        // Get checklist metadata for PDF generation
        const checklistMeta = currentChecklist || window.selectedChecklistData || {};
        
        console.log('📋 Checklist Metadata for PDF:', checklistMeta);
        
        // Structure report data EXACTLY like React Native (line 2280-2328)
        const reportData = {
            _id: reportId,
            userId: userId,
            requestNumber: requestNumber,
            status: 'completed',
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
            
            step1: {
                ...step1Data,
                id: reportId,
                requestNumber: requestNumber,
                meterReading: parseNumberSafely(step1Data.meterReading),
                currentMeterReading: parseNumberSafely(step1Data.currentMeterReading),
                checklistMeta: checklistMeta // Add checklist metadata for PDF generation
            },
            
            step2: {
                sections: step2Results.sections || [],
                selections: step2Results.selections || {},
                selectedFiles: step2Results.selectedFiles || {},
                images: step2Results.images || step2Results.selectedFiles || {}, // For PDF compatibility
                numericInputs: step2Results.numericInputs || {},
                notes: step2Results.notes || {},
                checklistTitle: step2Results.checklistTitle || '',
                headerInputValues: step2Results.headerInputValues || {},
                footerInputValues: step2Results.footerInputValues || {},
                faultyItems: faultyCount,
                timestamp: new Date().toISOString()
            },
            
            step3: step3DataCollected,
            
            createdBy: userId,
            lastModifiedBy: userId
        };
        
        console.log('📦 Structured report data (matches React Native):', reportData);
        
        // STEP 1: Generate PDF using Laravel backend (React Native line 2366)
        console.log('📄 Step 1: Generating PDF...');
        const pdfBlob = await generatePdfBlob(reportData);
        
        if (!pdfBlob) {
            throw new Error('PDF generation failed');
        }
        
        console.log('✅ PDF generated successfully');
        
        // STEP 2: Upload PDF to MongoDB (React Native line 2371)
        console.log('📤 Step 2: Uploading PDF to MongoDB...');
        const pdfDownloadUrl = await uploadPdfToMongoDB(pdfBlob, userId, requestNumber);
        
        console.log('✅ PDF uploaded, URL:', pdfDownloadUrl);
        
        // STEP 3: Add PDF URL to report data (React Native line 2375)
        reportData.step3.pdfDownloadUrl = pdfDownloadUrl;
        
        // STEP 4: Save report to MongoDB (React Native line 2441)
        console.log('💾 Step 3: Saving inspection to MongoDB...');
        const saveResult = await saveInspectionToMongoDB(reportData);
        
        if (!saveResult.success) {
            throw new Error(saveResult.message || 'Failed to save inspection');
        }
        
        console.log('✅ All steps complete! Report saved successfully.');
        
        // STEP 5: Send notifications and emails (React Native line 2555-2563)
        console.log('📧 Step 4: Sending notifications and emails...');
        try {
            await sendNotificationsAndEmails(reportData, pdfDownloadUrl);
            console.log('✅ Notifications and emails sent successfully');
        } catch (notifError) {
            console.error('⚠️ Failed to send notifications/emails:', notifError);
            // Don't fail the whole process if notifications fail
        }
        
        // Hide loading modal
        hideLoadingModal();
        
        // Store PDF URL and report data globally
        window.generatedPDFUrl = pdfDownloadUrl;
        window.generatedRequestNumber = requestNumber;
        window.generatedFaultyCount = faultyCount;
        window.generatedStatus = step3DataCollected.equipmentStatus;
        
        // Show success modal instead of alert
        showReportSuccessModal(requestNumber, faultyCount, step3DataCollected.equipmentStatus, pdfDownloadUrl);
        
    } catch (error) {
        console.error('❌ Error generating report:', error);
        hideLoadingModal();
        alert(`Error creating report: ${error.message}`);
    }
}

/**
 * Send notifications and emails after report creation
 */
async function sendNotificationsAndEmails(reportData, pdfUrl) {
    try {
        console.log('📧 Sending notifications and emails...');
        
        const createdByName = reportData.step1?.inspector || 'Inspector';
        const inspectorEmail = '{{ auth()->user()->email ?? "" }}';
        
        const notificationData = {
            inspectionData: reportData,
            createdByName: createdByName,
            inspectorEmail: inspectorEmail,
            pdfUrl: pdfUrl
        };
        
        console.log('📧 Notification data:', notificationData);
        
        // Send to notification service (running on port 5019)
        const response = await fetch('https://api.titandrillingzm.com:6019/api/inspection-notifications/new-inspection', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(notificationData)
        });
        
        const result = await response.json();
        
        if (result.success) {
            console.log('✅ Notifications and emails sent:', result);
        } else {
            console.warn('⚠️ Notification service returned error:', result);
        }
        
        return result;
        
    } catch (error) {
        console.error('❌ Error sending notifications:', error);
        throw error;
    }
}

/**
 * Show success modal after report creation
 */
function showReportSuccessModal(requestNumber, faultyCount, status, pdfUrl) {
    console.log('🎉 Showing success modal...');
    
    // Update success modal content
    const successModal = document.getElementById('successModal');
    if (successModal) {
        // Update modal content with report details
        const successContainer = successModal.querySelector('.success-container');
        if (successContainer) {
            successContainer.innerHTML = `
                <div class="success-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <h3>Report Created Successfully!</h3>
                <p style="margin-bottom: 20px;">Your inspection report has been generated and saved to the database.</p>
                
                <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 25px; text-align: left;">
                    <div style="display: grid; gap: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="color: #6c757d; font-weight: 500;">Report Number:</span>
                            <span style="color: #0055a5; font-weight: 700; font-family: 'Courier New', monospace;">${requestNumber}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="color: #6c757d; font-weight: 500;">Faulty Items:</span>
                            <span style="color: ${faultyCount > 0 ? '#dc3545' : '#00B894'}; font-weight: 700;">${faultyCount}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="color: #6c757d; font-weight: 500;">Equipment Status:</span>
                            <span style="color: #212529; font-weight: 600;">${status}</span>
                        </div>
                    </div>
                </div>
                
                <div style="background: #d4edda; border: 1px solid #c3e6cb; border-radius: 6px; padding: 12px; margin-bottom: 25px;">
                    <div style="display: flex; align-items: center; gap: 10px; color: #155724;">
                        <i class="fas fa-check-circle"></i>
                        <span style="font-size: 14px; font-weight: 500;">Report saved to database</span>
                    </div>
                    <div style="display: flex; align-items: center; gap: 10px; color: #155724; margin-top: 8px;">
                        <i class="fas fa-file-pdf"></i>
                        <span style="font-size: 14px; font-weight: 500;">PDF generated and uploaded</span>
                    </div>
                    <div style="display: flex; align-items: center; gap: 10px; color: #155724; margin-top: 8px;">
                        <i class="fas fa-envelope"></i>
                        <span style="font-size: 14px; font-weight: 500;">Email notifications sent</span>
                    </div>
                </div>
                
                <div class="success-buttons">
                    <button type="button" class="btn btn-secondary" onclick="goBackToInspectionHome()" style="background: #ffd205; color: #212529; border: none; font-weight: 600;">
                        <i class="fas fa-arrow-left"></i> Back to Inspection Home
                    </button>
                    <button type="button" class="btn btn-primary" onclick="viewGeneratedPDF()" style="background: #0055a5; color: white; border: none; font-weight: 600;">
                        <i class="fas fa-file-pdf"></i> View PDF Report
                    </button>
                </div>
            `;
        }
        
        // Show the modal
        openModal('successModal');
    }
}

/**
 * Go back to inspection home page
 */
function goBackToInspectionHome() {
    console.log('🏠 Navigating back to inspection home...');
    
    // Clear all session data
    clearSession();
    sessionStorage.clear();
    
    // Navigate to inspection reports page
    window.location.href = '/inspection/inspection-report';
}

/**
 * View the generated PDF
 */
function viewGeneratedPDF() {
    if (window.generatedPDFUrl) {
        console.log('📄 Opening PDF:', window.generatedPDFUrl);
        window.open(window.generatedPDFUrl, '_blank');
    } else {
        alert('PDF URL not available');
    }
}

/**
 * Generate PDF as Blob using HTML content (MATCHES React Native)
 */
async function generatePdfBlob(reportData) {
    try {
        console.log('📄 Generating PDF HTML content...');
        console.log('📊 Report Data for PDF:', reportData);
        
        // Generate HTML using the same function as React Native
        const htmlContent = generatePDFHTML(reportData);
        
        console.log('✅ HTML Content Generated');
        console.log('📏 HTML Length:', htmlContent.length);
        console.log('📄 HTML Preview:', htmlContent.substring(0, 500) + '...');
        
        // Send HTML to Laravel backend to convert to PDF
        const response = await fetch('/api/generate-inspection-pdf', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify({
                html: htmlContent,
                reportData: reportData,
                requestNumber: reportData.requestNumber
            })
        });
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ PDF API Error Response:', errorText);
            throw new Error('PDF generation API failed: ' + errorText);
        }
        
        const pdfBlob = await response.blob();
        console.log('✅ PDF Blob generated, size:', pdfBlob.size);
        
        return pdfBlob;
        
    } catch (error) {
        console.error('❌ PDF generation failed:', error);
        throw error;
    }
}

// Helper functions for loading modal
function showLoadingModal(message) {
    const modal = document.getElementById('loadingModal') || createLoadingModal();
    const messageEl = modal.querySelector('.loading-message');
    if (messageEl) messageEl.textContent = message;
    modal.style.display = 'flex';
}

function hideLoadingModal() {
    const modal = document.getElementById('loadingModal');
    if (modal) modal.style.display = 'none';
}

function createLoadingModal() {
    const modal = document.createElement('div');
    modal.id = 'loadingModal';
    modal.innerHTML = `
        <div style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); display: flex; align-items: center; justify-content: center; z-index: 10000;">
            <div style="background: white; padding: 30px; border-radius: 10px; text-align: center;">
                <div class="spinner-border text-primary" role="status"></div>
                <p class="loading-message" style="margin-top: 15px; font-size: 16px; color: #333;">Processing...</p>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
    return modal;
}

// Make handleGenerateReport globally accessible
window.handleGenerateReport = handleGenerateReport;

// MongoDB save function is now properly defined above
// All inspection data saves directly to MongoDB via Laravel API
// PDF files are uploaded to MongoDB server and URLs are stored in the inspection report
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>