@extends('welcome')

@section('title', 'Back to Incident Report')

@section('content')
<div class="incident-report-container" role="main" aria-label="Incident Report Form">
    <!-- Header Section -->
    <header class="incident-header" role="banner">
        <div class="header-content">
            <nav class="back-section" aria-label="Navigation">
                <a href="#" class="back-link" aria-label="Go back to incident report list" title="Back to Incident Report">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                        <path d="M19 12H5M12 19l-7-7 7-7"/>
                    </svg>
                    Back to Incident Report
                </a>
            </nav>
            <div class="page-title">
                <h1>Create Incident</h1>
                <p>Report and document vehicle incidents with detailed information</p>
            </div>
        </div>
    </header>
    <!-- Form Section -->
    <section class="form-section" aria-labelledby="form-title">
        <div class="form-header">
            <h2 id="form-title">Incident Information</h2>
            <p>Fill in the basic details about the incident</p>
        </div>
        
        <form class="incident-form" role="form" aria-label="Incident Report Form" novalidate>
            <div class="form-grid">
                <!-- Row 1 -->
                <div class="form-group">
                    <label for="incidentNumber" class="form-label required">Incident Number</label>
                    <input type="text" class="form-control" id="incidentNumber" name="incident-number" placeholder="Enter incident number" required aria-required="true" aria-describedby="incidentNumber-help">
                    <div id="incidentNumber-help" class="sr-only">Enter a unique incident number for tracking purposes</div>
                </div>
                <div class="form-group">
                    <label for="vehicleNumber" class="form-label required">Vehicle Number</label>
                    <select class="form-control" id="vehicleNumber" name="vehicle-number" required aria-required="true" aria-describedby="vehicleNumber-help">
                        <option value="">Select vehicle from inventory list</option>
                    </select>
                    <div id="vehicleNumber-help" class="sr-only">Choose the vehicle involved in the incident from the dropdown list</div>
                </div>
                
                <!-- Row 2 -->
                <div class="form-group">
                    <label for="incidentDate" class="form-label required">Incident Date</label>
                    <input type="date" class="form-control" id="incidentDate" name="incident-date" required>
                </div>
                <div class="form-group">
                    <label for="vehicleName" class="form-label required">Vehicle Name</label>
                    <input type="text" class="form-control" id="vehicleName" name="vehicle-name" placeholder="Enter vehicle name" required>
                </div>
                
                <!-- Row 3 -->
                <div class="form-group">
                    <label for="incidentTime" class="form-label required">Incident Time</label>
                    <input type="time" class="form-control" id="incidentTime" name="incident-time" required>
                </div>
                <div class="form-group">
                    <label for="vehicleCategory" class="form-label">Vehicle Category</label>
                    <input type="text" class="form-control" id="vehicleCategory" name="vehicle-category" placeholder="Enter vehicle category">
                </div>
                
                <!-- Row 4 -->
                <div class="form-group">
                    <label for="location" class="form-label required">Location</label>
                    <input type="text" class="form-control" id="location" name="location" placeholder="Enter incident location" required>
                </div>
                <div class="form-group">
                    <label for="vinNumber" class="form-label">VIN Number</label>
                    <input type="text" class="form-control" id="vinNumber" name="vin-number" placeholder="Enter VIN number">
                </div>
                
                <!-- Row 5 -->
                <div class="form-group">
                    <label for="typesOfIncident" class="form-label required">Types of Incident</label>
                    <select class="form-control" id="typesOfIncident" name="types-of-incident" required>
                        <option value="">Select type</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="operatorName" class="form-label">Operator Name</label>
                    <input type="text" class="form-control" id="operatorName" name="operator-name" placeholder="Enter operator name">
                </div>
                
                <!-- Row 6 -->
                <div class="form-group">
                    <label for="incidentArea" class="form-label">Incident Area</label>
                    <select class="form-control" id="incidentArea" name="incident-area">
                        <option value="">Select area</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="operatorCondition" class="form-label required">Operator Condition</label>
                    <select class="form-control" id="operatorCondition" name="operator-condition" required>
                        <option value="">Select condition</option>
                    </select>
                </div>
                
                <!-- Row 7 -->
                <div class="form-group">
                    <label for="incidentSeverity" class="form-label required">Incident Severity Rating</label>
                    <select class="form-control" id="incidentSeverity" name="incident-severity" required>
                        <option value="">Select rating</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="vehicleActivity" class="form-label required">Vehicle Activity at Time of Incident</label>
                    <select class="form-control" id="vehicleActivity" name="vehicle-activity" required>
                        <option value="">Select activity</option>
                    </select>
                </div>
            </div>
        </form>
    </div>
    <!-- Image Upload Section -->
    <section class="image-section" aria-labelledby="image-title">
        <div class="section-header">
            <h3 id="image-title">Incident Images</h3>
            <p>Upload photos documenting the incident</p>
        </div>
        
        <div class="image-grid">
            <div class="image-upload-item">
                <div class="image-preview" id="preview1">
                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                        <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                        <circle cx="8.5" cy="8.5" r="1.5"/>
                        <polyline points="21,15 16,10 5,21"/>
                    </svg>
                    <span class="preview-text">No image selected</span>
                </div>
                <label class="image-label">Incident Image 1</label>
                <button class="btn-upload" onclick="document.getElementById('fileInput1').click()">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="7,10 12,15 17,10"/>
                        <line x1="12" y1="15" x2="12" y2="3"/>
                    </svg>
                    Upload Image
                </button>
                <input type="file" id="fileInput1" style="display: none;" accept="image/*" onchange="handleImageUpload(this, 'preview1')">
            </div>
            
            <div class="image-upload-item">
                <div class="image-preview" id="preview2">
                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                        <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                        <circle cx="8.5" cy="8.5" r="1.5"/>
                        <polyline points="21,15 16,10 5,21"/>
                    </svg>
                    <span class="preview-text">No image selected</span>
                </div>
                <label class="image-label">Incident Image 2</label>
                <button class="btn-upload" onclick="document.getElementById('fileInput2').click()">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="7,10 12,15 17,10"/>
                        <line x1="12" y1="15" x2="12" y2="3"/>
                    </svg>
                    Upload Image
                </button>
                <input type="file" id="fileInput2" style="display: none;" accept="image/*" onchange="handleImageUpload(this, 'preview2')">
            </div>
            
            <div class="image-upload-item">
                <div class="image-preview" id="preview3">
                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                        <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                        <circle cx="8.5" cy="8.5" r="1.5"/>
                        <polyline points="21,15 16,10 5,21"/>
                    </svg>
                    <span class="preview-text">No image selected</span>
                </div>
                <label class="image-label">Incident Image 3</label>
                <button class="btn-upload" onclick="document.getElementById('fileInput3').click()">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="7,10 12,15 17,10"/>
                        <line x1="12" y1="15" x2="12" y2="3"/>
                    </svg>
                    Upload Image
                </button>
                <input type="file" id="fileInput3" style="display: none;" accept="image/*" onchange="handleImageUpload(this, 'preview3')">
            </div>
            
            <div class="image-upload-item">
                <div class="image-preview" id="preview4">
                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                        <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                        <circle cx="8.5" cy="8.5" r="1.5"/>
                        <polyline points="21,15 16,10 5,21"/>
                    </svg>
                    <span class="preview-text">No image selected</span>
                </div>
                <label class="image-label">Incident Image 4</label>
                <button class="btn-upload" onclick="document.getElementById('fileInput4').click()">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="7,10 12,15 17,10"/>
                        <line x1="12" y1="15" x2="12" y2="3"/>
                    </svg>
                    Upload Image
                </button>
                <input type="file" id="fileInput4" style="display: none;" accept="image/*" onchange="handleImageUpload(this, 'preview4')">
            </div>
        </div>
        
        <div class="additional-uploads">
            <button class="btn-upload-more" onclick="document.getElementById('fileInputMore').click()">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                    <polyline points="7,10 12,15 17,10"/>
                    <line x1="12" y1="15" x2="12" y2="3"/>
                </svg>
                Upload Additional Images
            </button>
            <input type="file" id="fileInputMore" style="display: none;" accept="image/*" multiple onchange="handleMultipleUpload(this)">
        </div>
    </div>
    <!-- Description Section -->
    <section class="description-section" aria-labelledby="description-title">
        <div class="section-header">
            <h3 id="description-title">Incident Details</h3>
            <p>Provide detailed information about the incident</p>
        </div>
        
        <div class="description-grid">
            <div class="form-group">
                <label for="incidentDescription" class="form-label">How incident occurred and what were the causes?</label>
                <textarea class="form-control" id="incidentDescription" name="incident-description" rows="4" placeholder="Describe how the incident occurred and identify the root causes..."></textarea>
            </div>
            <div class="form-group">
                <label for="additionalComment" class="form-label">Additional Comment</label>
                <textarea class="form-control" id="additionalComment" name="additional-comment" rows="4" placeholder="Add any additional comments or observations..."></textarea>
            </div>
            <div class="form-group">
                <label for="equipmentDamage" class="form-label">What broke on the equipment, list all damages?</label>
                <textarea class="form-control" id="equipmentDamage" name="equipment-damage" rows="4" placeholder="List all equipment damages and broken components..."></textarea>
            </div>
            <div class="form-group">
                <label for="reportedBy" class="form-label">Reported By</label>
                <textarea class="form-control" id="reportedBy" name="reported-by" rows="4" placeholder="Enter reporter information..."></textarea>
            </div>
        </div>
    </div>
    
    <!-- Submit Section -->
    <div class="submit-section">
        <button class="btn-submit" onclick="saveIncident()" type="button" aria-label="Save incident details and submit report">
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                <path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/>
                <polyline points="17,21 17,13 7,13 7,21"/>
                <polyline points="7,3 7,8 15,8"/>
            </svg>
            <span>Save Incident Details</span>
        </button>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
(function() {
    'use strict';
    
    // Configuration object
    const CONFIG = {
        selectors: {
            form: '.incident-form',
            requiredInputs: 'input[required], select[required]',
            notification: '.notification'
        },
        messages: {
            success: 'Incident details saved successfully!',
            error: 'Please fill in all required fields',
            imageSelected: 'additional image(s) selected'
        },
        timing: {
            notificationDelay: 100,
            notificationDuration: 3000,
            animationDuration: 300,
            saveDelay: 1500
        },
        styles: {
            success: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
            error: 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
            info: 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)',
            errorBorder: '#ef4444',
            normalBorder: '#d1d5db'
        }
    };
    
    // DOM elements cache
    let elements = {};
    
    // Initialize DOM elements
    function initElements() {
        elements.form = document.querySelector(CONFIG.selectors.form);
        elements.requiredInputs = document.querySelectorAll(CONFIG.selectors.requiredInputs);
    }
    
    // Handle single image upload
    function handleImageUpload(input, previewId) {
        const file = input.files[0];
        const preview = document.getElementById(previewId);
        
        if (file && preview) {
            const reader = new FileReader();
            reader.onload = function(e) {
                preview.innerHTML = `<img src="${e.target.result}" alt="Uploaded image" style="width: 100%; height: 100%; object-fit: cover;">`;
            };
            reader.readAsDataURL(file);
        }
    }
    
    // Handle multiple image upload
    function handleMultipleUpload(input) {
        const files = input.files;
        if (files.length > 0) {
            const message = `${files.length} ${CONFIG.messages.imageSelected}`;
            showNotification(message, 'success');
        }
    }
    
    // Validate form data
    function validateForm(formData) {
        const requiredFields = ['incidentNumber', 'vehicleNumber', 'incidentDate', 'vehicleName', 'incidentTime', 'location', 'typesOfIncident', 'operatorCondition', 'incidentSeverity', 'vehicleActivity'];
        return requiredFields.filter(field => !formData[field] || formData[field].trim() === '');
    }
    
    // Get form data
    function getFormData() {
        return {
            incidentNumber: document.getElementById('incidentNumber')?.value || '',
            vehicleNumber: document.getElementById('vehicleNumber')?.value || '',
            incidentDate: document.getElementById('incidentDate')?.value || '',
            vehicleName: document.getElementById('vehicleName')?.value || '',
            incidentTime: document.getElementById('incidentTime')?.value || '',
            vehicleCategory: document.getElementById('vehicleCategory')?.value || '',
            location: document.getElementById('location')?.value || '',
            vinNumber: document.getElementById('vinNumber')?.value || '',
            typesOfIncident: document.getElementById('typesOfIncident')?.value || '',
            operatorName: document.getElementById('operatorName')?.value || '',
            incidentArea: document.getElementById('incidentArea')?.value || '',
            operatorCondition: document.getElementById('operatorCondition')?.value || '',
            incidentSeverity: document.getElementById('incidentSeverity')?.value || '',
            vehicleActivity: document.getElementById('vehicleActivity')?.value || '',
            incidentDescription: document.getElementById('incidentDescription')?.value || '',
            additionalComment: document.getElementById('additionalComment')?.value || '',
            equipmentDamage: document.getElementById('equipmentDamage')?.value || '',
            reportedBy: document.getElementById('reportedBy')?.value || ''
        };
    }
    
    // Save incident function
    function saveIncident() {
        const formData = getFormData();
        const missingFields = validateForm(formData);
        
        if (missingFields.length > 0) {
            showNotification(CONFIG.messages.error, 'error');
            return;
        }

        // Save incident data (you can implement your save logic here)
        console.log('Saving incident:', formData);
        
        // Show success message
        showNotification(CONFIG.messages.success, 'success');
        
        // Reset form or redirect
        setTimeout(() => {
            // You can redirect to incident list or reset form
            console.log('Incident saved');
        }, CONFIG.timing.saveDelay);
    }
    
    // Show notification
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.textContent = message;
        
        // Style the notification
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 0;
            color: white;
            font-weight: 500;
            z-index: 10000;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
        `;
        
        // Set background color based on type
        notification.style.background = CONFIG.styles[type] || CONFIG.styles.info;
        
        // Add to page
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, CONFIG.timing.notificationDelay);
        
        // Remove after duration
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (document.body.contains(notification)) {
                    document.body.removeChild(notification);
                }
            }, CONFIG.timing.animationDuration);
        }, CONFIG.timing.notificationDuration);
    }
    
    // Initialize form validation
    function initFormValidation() {
        elements.requiredInputs.forEach(input => {
            input.addEventListener('blur', function() {
                if (!this.value || this.value.trim() === '') {
                    this.style.borderColor = CONFIG.styles.errorBorder;
                } else {
                    this.style.borderColor = CONFIG.styles.normalBorder;
                }
            });
        });
    }
    
    // Initialize the application
    function init() {
        initElements();
        initFormValidation();
    }
    
    // Make functions globally accessible
    window.handleImageUpload = handleImageUpload;
    window.handleMultipleUpload = handleMultipleUpload;
    window.saveIncident = saveIncident;
    window.showNotification = showNotification;
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
</script>
<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: rgba(255, 255, 255, 0.2);
    --success-color: #10b981;
    --success-hover: #059669;
    --error-color: #ef4444;
    --error-hover: #dc2626;
    --text-primary: #0f172a;
    --text-secondary: #374151;
    --text-muted: #64748b;
    --text-light: #6b7280;
    --bg-primary: #f8fafc;
    --bg-white: #ffffff;
    --bg-hover: #f9fafb;
    --border-color: #e2e8f0;
    --border-light: #d1d5db;
    --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    --shadow-md: 0 2px 4px rgba(59, 130, 246, 0.2);
    --shadow-lg: 0 4px 8px rgba(59, 130, 246, 0.3);
    --radius-sm: 0;
    --radius-md: 0;
    --radius-lg: 0;
    --transition-base: all 0.2s ease;
    --transition-slow: all 0.3s ease;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
}

/* Main Container */
.incident-report-container {
    min-height: 100vh;
}

/* Header Section */
.incident-header {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.header-content {
    padding: 32px;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 24px;
    flex-direction: row-reverse;
}

.back-section {
    flex-shrink: 0;
}

.back-link {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    color: var(--primary-color);
    text-decoration: none;
    font-weight: 500;
    font-size: 14px;
    transition: var(--transition-base);
}

.back-link:hover {
    color: var(--text-primary);
    transform: translateX(-2px);
}

.back-link svg {
    flex-shrink: 0;
}

.page-title h1 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.page-title p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

/* Form Section */
.form-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.form-header {
    padding: 32px 32px 24px 32px;
    border-bottom: 1px solid var(--border-color);
}

.form-header h2 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 24px;
    line-height: 1.2;
    margin: 0 0 8px 0;
}

.form-header p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

/* Form Grid */
.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 24px;
    padding: 32px;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.form-label {
    color: var(--text-secondary);
    font-weight: 500;
    font-size: 14px;
    margin: 0;
}

.form-label.required::after {
    content: " *";
    color: var(--error-color);
}

.form-control {
    padding: 12px 16px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-secondary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.form-control::placeholder {
    color: #9ca3af;
}

/* Image Section */
.image-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.section-header {
    padding: 32px 32px 24px 32px;
    border-bottom: 1px solid var(--border-color);
}

.section-header h3 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 20px;
    line-height: 1.2;
    margin: 0 0 8px 0;
}

.section-header p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 14px;
    line-height: 1.5;
    margin: 0;
}

.image-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 24px;
    padding: 32px;
}

.image-upload-item {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.image-preview {
    width: 100%;
    height: 150px;
    background: var(--bg-hover);
    border: 2px dashed var(--border-light);
    border-radius: var(--radius-sm);
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 8px;
    transition: var(--transition-base);
}

.image-preview:hover {
    border-color: var(--primary-color);
    background: var(--bg-primary);
}

.image-preview svg {
    color: #9ca3af;
}

.preview-text {
    color: var(--text-light);
    font-size: 12px;
    font-weight: 500;
}

.image-label {
    color: var(--text-secondary);
    font-weight: 500;
    font-size: 14px;
    text-align: center;
}

.btn-upload {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid var(--primary-color);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--primary-color);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-upload:hover {
    background: var(--primary-color);
    color: var(--bg-white);
}

.additional-uploads {
    padding: 0 32px 32px 32px;
    display: flex;
    justify-content: center;
}

.btn-upload-more {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    border: 1px solid var(--text-light);
    border-radius: var(--radius-sm);
    background: var(--bg-white);
    color: var(--text-light);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-upload-more:hover {
    background: var(--text-light);
    color: var(--bg-white);
}

/* Description Section */
.description-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.description-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 24px;
    padding: 32px;
}

.description-grid .form-group {
    grid-column: span 1;
}

.description-grid textarea.form-control {
    min-height: 120px;
    resize: vertical;
}

/* Submit Section */
.submit-section {
    display: flex;
    justify-content: center;
    padding: 24px;
}

.btn-submit {
    display: inline-flex;
    align-items: center;
    gap: 12px;
    padding: 16px 32px;
    border: none;
    border-radius: var(--radius-sm);
    background: var(--primary-color);
    color: var(--bg-white);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-md);
}

.btn-submit:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-lg);
    color: var(--text-primary);
}

.btn-submit svg {
    flex-shrink: 0;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .incident-report-container {
        padding: 20px;
    }
    
    .incident-header {
        margin-bottom: 20px;
    }
    
    .header-content {
        padding: 28px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .form-header {
        padding: 28px 28px 20px 28px;
    }
    
    .form-grid {
        gap: 20px;
        padding: 28px;
    }
    
    .image-grid {
        gap: 20px;
        padding: 28px;
    }
    
    .description-grid {
        gap: 20px;
        padding: 28px;
    }
}

@media (max-width: 992px) {
    .incident-report-container {
        padding: 16px;
    }
    
    .incident-header {
        margin-bottom: 18px;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
        padding: 24px;
    }
    
    .back-section {
        order: 1;
    }
    
    .page-title {
        order: 2;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .form-header {
        padding: 24px 24px 18px 24px;
    }
    
    .form-header h2 {
        font-size: 20px;
    }
    
    .form-header p {
        font-size: 14px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 18px;
        padding: 24px;
    }
    
    .form-group {
        gap: 6px;
    }
    
    .form-control {
        padding: 12px 14px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .image-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 16px;
        padding: 24px;
    }
    
    .image-preview {
        height: 140px;
    }
    
    .description-grid {
        grid-template-columns: 1fr;
        gap: 18px;
        padding: 24px;
    }
    
    .description-grid textarea.form-control {
        min-height: 100px;
    }
    
    .btn-submit {
        padding: 14px 28px;
        font-size: 15px;
    }
}

@media (max-width: 768px) {
    .incident-report-container {
        padding: 0;
    }
    
    .incident-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 16px;
        gap: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 6px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .back-link {
        font-size: 13px;
        padding: 8px 0;
    }
    
    .form-section {
        margin-bottom: 16px;
    }
    
    .form-header {
        padding: 20px 16px 16px 16px;
    }
    
    .form-header h2 {
        font-size: 18px;
        margin-bottom: 6px;
    }
    
    .form-header p {
        font-size: 13px;
    }
    
    .form-grid {
        padding: 20px 16px;
        gap: 16px;
    }
    
    .form-group {
        gap: 6px;
    }
    
    .form-label {
        font-size: 13px;
        font-weight: 600;
    }
    
    .form-control {
        padding: 12px 14px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .image-section {
        margin-bottom: 16px;
    }
    
    .section-header {
        padding: 20px 16px 16px 16px;
    }
    
    .section-header h3 {
        font-size: 16px;
        margin-bottom: 6px;
    }
    
    .section-header p {
        font-size: 12px;
    }
    
    .image-grid {
        grid-template-columns: 1fr;
        gap: 16px;
        padding: 20px 16px;
    }
    
    .image-preview {
        height: 120px;
    }
    
    .image-label {
        font-size: 13px;
    }
    
    .btn-upload {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
    }
    
    .additional-uploads {
        padding: 0 16px 20px 16px;
    }
    
    .btn-upload-more {
        padding: 12px 20px;
        font-size: 13px;
        min-height: 44px;
    }
    
    .description-section {
        margin-bottom: 16px;
    }
    
    .description-grid {
        padding: 20px 16px;
        gap: 16px;
    }
    
    .description-grid textarea.form-control {
        min-height: 100px;
        font-size: 16px;
    }
    
    .submit-section {
        padding: 16px;
    }
    
    .btn-submit {
        padding: 14px 24px;
        font-size: 15px;
        min-height: 48px;
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .incident-report-container {
        padding: 0;
    }
    
    .incident-header {
        margin-bottom: 12px;
    }
    
    .header-content {
        padding: 12px;
        gap: 12px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .back-link {
        font-size: 12px;
        padding: 6px 0;
    }
    
    .form-section {
        margin-bottom: 12px;
    }
    
    .form-header {
        padding: 16px 12px 12px 12px;
    }
    
    .form-header h2 {
        font-size: 16px;
        margin-bottom: 4px;
    }
    
    .form-header p {
        font-size: 12px;
    }
    
    .form-grid {
        padding: 16px 12px;
        gap: 14px;
    }
    
    .form-group {
        gap: 5px;
    }
    
    .form-label {
        font-size: 12px;
        font-weight: 600;
    }
    
    .form-control {
        padding: 12px 14px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .image-section {
        margin-bottom: 12px;
    }
    
    .section-header {
        padding: 16px 12px 12px 12px;
    }
    
    .section-header h3 {
        font-size: 15px;
        margin-bottom: 4px;
    }
    
    .section-header p {
        font-size: 11px;
    }
    
    .image-grid {
        padding: 16px 12px;
        gap: 14px;
    }
    
    .image-preview {
        height: 100px;
    }
    
    .image-label {
        font-size: 12px;
    }
    
    .btn-upload {
        padding: 10px 12px;
        font-size: 12px;
        min-height: 44px;
    }
    
    .additional-uploads {
        padding: 0 12px 16px 12px;
    }
    
    .btn-upload-more {
        padding: 10px 16px;
        font-size: 12px;
        min-height: 44px;
    }
    
    .description-section {
        margin-bottom: 12px;
    }
    
    .description-grid {
        padding: 16px 12px;
        gap: 14px;
    }
    
    .description-grid textarea.form-control {
        min-height: 90px;
        font-size: 16px;
    }
    
    .submit-section {
        padding: 12px;
    }
    
    .btn-submit {
        padding: 12px 20px;
        font-size: 14px;
        min-height: 48px;
        width: 100%;
        justify-content: center;
    }
}

/* Focus states for accessibility */
.btn-upload:focus,
.btn-upload-more:focus,
.btn-submit:focus,
.form-control:focus {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* Smooth animations */
* {
    transition: all 0.2s ease;
}

/* Screen reader only text */
.sr-only {
    position: absolute;
    width: 1px;
    height: 1px;
    padding: 0;
    margin: -1px;
    overflow: hidden;
    clip: rect(0, 0, 0, 0);
    white-space: nowrap;
    border: 0;
}

/* Icon improvements */
.back-link svg,
.btn-upload svg,
.btn-upload-more svg,
.btn-submit svg,
.image-preview svg {
    flex-shrink: 0;
}

/* ===== ENHANCED RESPONSIVE DESIGN ===== */
@media (max-width: 1400px) {
    .incident-report-container {
        padding: 18px;
    }
    
    .incident-header {
        margin-bottom: 20px;
    }
    
    .header-content {
        padding: 28px;
    }
}

@media (max-width: 1200px) {
    .incident-report-container {
        padding: 16px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

@media (max-width: 992px) {
    .incident-report-container {
        padding: 14px;
    }
    
    .incident-header {
        margin-bottom: 16px;
    }
    
    .header-content {
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .form-section {
        padding: 16px;
    }
    
    .form-header h2 {
        font-size: 24px;
    }
    
    .form-header p {
        font-size: 14px;
    }
    
    .form-grid {
        gap: 14px;
    }
    
    .form-group {
        width: 100%;
    }
}

@media (max-width: 768px) {
    .incident-report-container {
        padding: 12px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .form-section {
        padding: 12px;
    }
    
    .form-header h2 {
        font-size: 20px;
    }
    
    .form-header p {
        font-size: 13px;
    }
    
    .form-grid {
        gap: 12px;
    }
    
    .form-control,
    .form-select,
    .form-textarea {
        padding: 8px 12px;
        font-size: 14px;
    }
    
    .btn-upload,
    .btn-upload-more,
    .btn-submit {
        padding: 10px 16px;
        font-size: 14px;
    }
}

@media (max-width: 576px) {
    .incident-report-container {
        padding: 8px;
    }
    
    .header-content {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .form-section {
        padding: 10px;
    }
    
    .form-header h2 {
        font-size: 18px;
    }
    
    .form-header p {
        font-size: 12px;
    }
    
    .form-grid {
        gap: 10px;
    }
    
    .form-control,
    .form-select,
    .form-textarea {
        padding: 6px 10px;
        font-size: 13px;
    }
    
    .btn-upload,
    .btn-upload-more,
    .btn-submit {
        padding: 8px 12px;
        font-size: 13px;
    }
    
    .image-upload-section {
        padding: 12px;
    }
    
    .image-upload-box {
        padding: 16px;
    }
}

@media (max-width: 480px) {
    .incident-report-container {
        padding: 0;
    }
    
    .incident-header {
        margin-bottom: 8px;
    }
    
    .header-content {
        padding: 10px;
        gap: 10px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .back-link {
        font-size: 11px;
        padding: 4px 0;
    }
    
    .form-section {
        margin-bottom: 8px;
    }
    
    .form-header {
        padding: 12px 10px 10px 10px;
    }
    
    .form-header h2 {
        font-size: 14px;
        margin-bottom: 3px;
    }
    
    .form-header p {
        font-size: 11px;
    }
    
    .form-grid {
        padding: 12px 10px;
        gap: 12px;
    }
    
    .form-group {
        gap: 4px;
    }
    
    .form-label {
        font-size: 11px;
        font-weight: 600;
    }
    
    .form-control {
        padding: 10px 12px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .image-section {
        margin-bottom: 8px;
    }
    
    .section-header {
        padding: 12px 10px 10px 10px;
    }
    
    .section-header h3 {
        font-size: 14px;
        margin-bottom: 3px;
    }
    
    .section-header p {
        font-size: 10px;
    }
    
    .image-grid {
        padding: 12px 10px;
        gap: 12px;
    }
    
    .image-preview {
        height: 80px;
    }
    
    .image-label {
        font-size: 11px;
    }
    
    .btn-upload {
        padding: 8px 10px;
        font-size: 11px;
        min-height: 40px;
    }
    
    .additional-uploads {
        padding: 0 10px 12px 10px;
    }
    
    .btn-upload-more {
        padding: 8px 12px;
        font-size: 11px;
        min-height: 40px;
    }
    
    .description-section {
        margin-bottom: 8px;
    }
    
    .description-grid {
        padding: 12px 10px;
        gap: 12px;
    }
    
    .description-grid textarea.form-control {
        min-height: 80px;
        font-size: 16px;
    }
    
    .submit-section {
        padding: 10px;
    }
    
    .btn-submit {
        padding: 10px 16px;
        font-size: 13px;
        min-height: 44px;
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 360px) {
    .incident-report-container {
        padding: 0;
    }
    
    .incident-header {
        margin-bottom: 6px;
    }
    
    .header-content {
        padding: 8px;
        gap: 8px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .back-link {
        font-size: 10px;
        padding: 3px 0;
    }
    
    .form-section {
        margin-bottom: 6px;
    }
    
    .form-header {
        padding: 10px 8px 8px 8px;
    }
    
    .form-header h2 {
        font-size: 13px;
        margin-bottom: 2px;
    }
    
    .form-header p {
        font-size: 10px;
    }
    
    .form-grid {
        padding: 10px 8px;
        gap: 10px;
    }
    
    .form-group {
        gap: 3px;
    }
    
    .form-label {
        font-size: 10px;
        font-weight: 600;
    }
    
    .form-control {
        padding: 8px 10px;
        font-size: 16px;
        min-height: 40px;
    }
    
    .image-section {
        margin-bottom: 6px;
    }
    
    .section-header {
        padding: 10px 8px 8px 8px;
    }
    
    .section-header h3 {
        font-size: 13px;
        margin-bottom: 2px;
    }
    
    .section-header p {
        font-size: 9px;
    }
    
    .image-grid {
        padding: 10px 8px;
        gap: 10px;
    }
    
    .image-preview {
        height: 70px;
    }
    
    .image-label {
        font-size: 10px;
    }
    
    .btn-upload {
        padding: 6px 8px;
        font-size: 10px;
        min-height: 36px;
    }
    
    .additional-uploads {
        padding: 0 8px 10px 8px;
    }
    
    .btn-upload-more {
        padding: 6px 10px;
        font-size: 10px;
        min-height: 36px;
    }
    
    .description-section {
        margin-bottom: 6px;
    }
    
    .description-grid {
        padding: 10px 8px;
        gap: 10px;
    }
    
    .description-grid textarea.form-control {
        min-height: 70px;
        font-size: 16px;
    }
    
    .submit-section {
        padding: 8px;
    }
    
    .btn-submit {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 40px;
        width: 100%;
        justify-content: center;
    }
}
</style>

@endsection