@extends('welcome')

@section('title', 'HSE Inspection Reports')

@section('styles')
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" integrity="sha512-dQ39zqK1p5MnZ3V4n5nRkubgOZpO2wK0uX0hUxX67+xoycGgSoxuV+/Nx8IJ3mVXkB0Uqz3BSPBZz1Yz4Z6kNA==" crossorigin="anonymous" referrerpolicy="no-referrer" />
<style>
    :root {
        --hse-primary: #015185;
        --hse-primary-dark: #023f64;
        --hse-accent: #0f8a5f;
        --hse-danger: #c0392b;
        --hse-warning: #f39c12;
        --hse-bg: #f5f7fb;
        --hse-border: #e0e7ef;
        --hse-text: #0f172a;
        --hse-muted: #64748b;
        --hse-card-bg: #ffffff;
        --hse-shadow: 0 12px 30px rgba(15, 81, 133, 0.08);
    }

    body.modal-open {
        overflow: hidden;
    }

    .hse-screen {
        min-height: 100vh;
        padding: 32px;
        background: var(--hse-bg);
    }

    @media (max-width: 1024px) {
        .hse-screen {
            padding: 20px;
        }
    }

    .hse-header {
        background: linear-gradient(135deg, var(--hse-primary), var(--hse-primary-dark));
        border-radius: 20px;
        padding: 28px;
        color: #fff;
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        gap: 24px;
        flex-wrap: wrap;
        box-shadow: var(--hse-shadow);
    }

    .hse-header-left {
        display: flex;
        align-items: flex-start;
        gap: 16px;
        flex: 1 1 60%;
        min-width: 280px;
    }

    .hse-back-button {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 44px;
        height: 44px;
        border-radius: 50%;
        border: 1px solid rgba(255,255,255,0.4);
        color: #fff;
        text-decoration: none;
        font-size: 18px;
        transition: opacity 0.2s ease, transform 0.2s ease;
    }

    .hse-back-button:hover {
        opacity: 0.85;
        transform: translateX(-2px);
    }

    .hse-header-text h1 {
        margin: 4px 0;
    }

    .hse-breadcrumb {
        color: rgba(255,255,255,0.75);
        margin: 0;
        font-size: 14px;
    }

    .hse-breadcrumb a {
        color: rgba(255,255,255,0.85);
        text-decoration: none;
    }

    .hse-breadcrumb span {
        margin: 0 6px;
    }

    .hse-header-right {
        display: flex;
        flex-direction: column;
        gap: 12px;
        align-items: stretch;
        flex: 1 1 35%;
        min-width: 260px;
    }

    .hse-header-search {
        position: relative;
        display: flex;
        align-items: center;
    }

    .hse-header-search i {
        position: absolute;
        left: 14px;
        color: var(--hse-muted);
    }

    .hse-header-search input {
        width: 100%;
        padding: 12px 16px 12px 42px;
        border-radius: 999px;
        border: none;
        outline: none;
        font-size: 14px;
        color: var(--hse-text);
        background: rgba(255,255,255,0.95);
        box-shadow: 0 10px 25px rgba(0,0,0,0.12);
    }

    .hse-header-buttons {
        display: flex;
        gap: 10px;
        justify-content: flex-end;
        flex-wrap: wrap;
    }

    .sr-only {
        position: absolute;
        width: 1px;
        height: 1px;
        padding: 0;
        margin: -1px;
        overflow: hidden;
        clip: rect(0,0,0,0);
        white-space: nowrap;
        border: 0;
    }

    .hse-header h1 {
        font-size: 28px;
        margin-bottom: 10px;
    }

    .hse-header p {
        margin: 0;
        color: rgba(255, 255, 255, 0.85);
    }

    .hse-header-actions {
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .hse-controls {
        margin-top: 20px;
        margin-bottom: 12px;
        display: flex;
        flex-wrap: wrap;
        gap: 12px;
        align-items: center;
        background: #fff;
        padding: 16px 20px;
        border-radius: 14px;
        border: 1px solid var(--hse-border);
        box-shadow: var(--hse-shadow);
    }

    .hse-search {
        flex: 1;
        min-width: 240px;
        position: relative;
    }

    .hse-search input {
        width: 100%;
        padding: 12px 16px 12px 42px;
        border-radius: 12px;
        border: 1px solid var(--hse-border);
        font-size: 15px;
        color: var(--hse-text);
        background: #fff;
        transition: border-color 0.2s ease;
    }

    .hse-search input::placeholder {
        color: var(--hse-muted);
    }

    .hse-search input:focus {
        outline: none;
        border-color: var(--hse-primary);
        box-shadow: 0 0 0 3px rgba(1, 81, 133, 0.08);
    }

    .hse-search i {
        position: absolute;
        top: 50%;
        left: 14px;
        transform: translateY(-50%);
        color: var(--hse-muted);
    }

    .hse-filter-group {
        display: flex;
        align-items: center;
        gap: 10px;
        flex-wrap: wrap;
    }

    .hse-filter-group select {
        border-radius: 10px;
        border: 1px solid var(--hse-border);
        padding: 10px 14px;
        background: #fff;
        color: var(--hse-text);
        min-width: 170px;
    }

    .hse-btn {
        border: none;
        border-radius: 10px;
        padding: 10px 18px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.15s ease, box-shadow 0.15s ease, background 0.2s ease;
        display: inline-flex;
        align-items: center;
        gap: 8px;
    }

    .hse-btn i {
        font-size: 14px;
    }

    .hse-btn.primary {
        background: var(--hse-primary);
        color: #fff;
        box-shadow: 0 8px 18px rgba(1, 81, 133, 0.2);
    }

    .hse-btn.primary:hover {
        background: var(--hse-primary-dark);
    }

    .hse-btn.ghost {
        background: rgba(1, 81, 133, 0.08);
        color: var(--hse-primary);
    }

    .hse-btn.light {
        background: #fff;
        border: 1px solid var(--hse-border);
        color: var(--hse-text);
    }

    .hse-btn.danger {
        background: rgba(192, 57, 43, 0.1);
        color: var(--hse-danger);
    }

    .hse-btn:disabled {
        opacity: 0.6;
        cursor: not-allowed;
        box-shadow: none;
    }

    .hse-card-container {
        background: #fff;
        border-radius: 18px;
        border: 1px solid var(--hse-border);
        box-shadow: var(--hse-shadow);
        margin-top: 24px;
    }

    .hse-card-section {
        padding: 24px;
    }

    .hse-summary-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        gap: 12px;
        margin-bottom: 18px;
    }

    .hse-summary-subtitle {
        color: var(--hse-muted);
        margin: 4px 0 0 0;
        font-size: 14px;
    }

    .hse-summary-timestamp {
        color: var(--hse-muted);
        font-size: 13px;
    }

    .hse-summary-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(210px, 1fr));
        gap: 18px;
    }

    .hse-summary-card {
        background: var(--hse-card-bg);
        border-radius: 16px;
        padding: 20px;
        border: 1px solid var(--hse-border);
        box-shadow: var(--hse-shadow);
    }

    .hse-summary-label {
        font-size: 14px;
        color: var(--hse-muted);
        text-transform: uppercase;
        letter-spacing: 0.08em;
    }

    .hse-summary-value {
        font-size: 32px;
        font-weight: 700;
        margin: 12px 0 6px;
        color: var(--hse-text);
    }

    .hse-summary-meta {
        font-size: 13px;
        color: var(--hse-muted);
    }

    .hse-table-card {
        margin-top: 28px;
        background: var(--hse-card-bg);
        border-radius: 18px;
        border: 1px solid var(--hse-border);
        padding: 0;
        box-shadow: var(--hse-shadow);
        overflow: hidden;
    }

    .hse-table-card > .hse-table-toolbar {
        padding: 24px;
    }

    .hse-table-toolbar {
        display: flex;
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 12px;
        margin-bottom: 16px;
    }

    .hse-table-meta {
        color: var(--hse-muted);
        font-size: 14px;
    }

    .hse-table-wrapper {
        overflow-x: auto;
        border-radius: 12px;
        border: 1px solid var(--hse-border);
        margin: 0 24px 24px;
    }

    table.hse-table {
        width: 100%;
        border-collapse: collapse;
        min-width: 900px;
    }

    .hse-table thead {
        background: #f8fafc;
    }

    .hse-table th {
        text-align: left;
        padding: 12px 16px;
        font-size: 13px;
        text-transform: uppercase;
        letter-spacing: 0.05em;
        color: var(--hse-muted);
        font-weight: 600;
    }

    .hse-table tbody tr:nth-child(every) {
        background: #fff;
    }

    .hse-table td {
        padding: 14px 16px;
        border-top: 1px solid var(--hse-border);
        font-size: 14px;
        color: var(--hse-text);
    }

    .hse-table tbody tr:hover {
        background: rgba(1, 81, 133, 0.03);
    }

    .hse-dropdown {
        position: relative;
        display: inline-block;
    }

    .hse-action-btn {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 8px 14px;
        border-radius: 999px;
        border: none;
        background: var(--hse-primary);
        color: #fff;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: background 0.2s ease, transform 0.2s ease;
        box-shadow: 0 10px 20px rgba(1, 81, 133, 0.2);
    }

    .hse-action-btn.active {
        background: var(--hse-primary-dark);
    }

    .hse-action-btn i {
        font-size: 14px;
    }

    .hse-dropdown-menu {
        position: absolute;
        right: 0;
        top: calc(100% + 10px);
        min-width: 200px;
        background: #fff;
        border: 1px solid var(--hse-border);
        border-radius: 12px;
        box-shadow: 0 20px 40px rgba(15, 23, 42, 0.15);
        padding: 8px;
        opacity: 0;
        pointer-events: none;
        transform: translateY(-5px);
        transition: opacity 0.15s ease, transform 0.15s ease;
        z-index: 30;
    }

    .hse-dropdown-menu.show {
        opacity: 1;
        pointer-events: auto;
        transform: translateY(0);
        position: relative;
    }

    .hse-dropdown-item {
        width: 100%;
        border: none;
        background: none;
        display: flex;
        align-items: center;
        gap: 8px;
        padding: 10px 12px;
        border-radius: 8px;
        font-size: 13px;
        font-weight: 500;
        color: var(--hse-text);
        cursor: pointer;
        transition: background 0.15s ease, color 0.15s ease;
        text-align: left;
    }

    .hse-dropdown-item:hover:not(:disabled) {
        background: rgba(1, 81, 133, 0.08);
        color: var(--hse-primary);
    }

    .hse-dropdown-item.danger {
        color: var(--hse-danger);
    }

    .hse-dropdown-item.danger:hover:not(:disabled) {
        background: rgba(192, 57, 43, 0.12);
        color: var(--hse-danger);
    }

    .hse-dropdown-item:disabled {
        opacity: 0.45;
        cursor: not-allowed;
    }

    .hse-dropdown-divider {
        height: 1px;
        width: 100%;
        background: var(--hse-border);
        margin: 6px 0;
    }

    .status-pill {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        border-radius: 999px;
        padding: 4px 12px;
        font-size: 12px;
        font-weight: 600;
        text-transform: capitalize;
    }

    .status-pill.completed {
        background: rgba(15, 154, 95, 0.1);
        color: var(--hse-accent);
    }

    .status-pill.draft {
        background: rgba(243, 156, 18, 0.12);
        color: var(--hse-warning);
    }

    .status-pill.archived {
        background: rgba(15, 23, 42, 0.08);
        color: var(--hse-text);
    }

    .status-pill.pending {
        background: rgba(192, 57, 43, 0.12);
        color: var(--hse-danger);
    }

.corrective-cell {
    display: flex;
    flex-direction: column;
    gap: 6px;
}

.corrective-meta-line {
    display: flex;
    flex-wrap: wrap;
    gap: 10px;
    font-size: 12px;
    color: var(--hse-muted);
    align-items: center;
}

.corrective-number {
    font-weight: 600;
    color: var(--hse-text);
    background: rgba(1, 81, 133, 0.08);
    padding: 2px 8px;
    border-radius: 999px;
}

.corrective-number.muted {
    color: var(--hse-muted);
    background: rgba(100, 116, 139, 0.1);
}

.corrective-unsafe {
    display: inline-flex;
    align-items: center;
    gap: 4px;
}

    .hse-pagination {
        margin-top: 20px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-wrap: wrap;
        gap: 12px;
    }

    .hse-pagination .page-controls {
        display: flex;
        gap: 8px;
    }

    .page-btn {
        border: 1px solid var(--hse-border);
        background: #fff;
        border-radius: 8px;
        padding: 8px 12px;
        min-width: 40px;
        text-align: center;
        cursor: pointer;
        font-weight: 600;
        color: var(--hse-text);
    }

    .page-btn.active {
        background: var(--hse-primary);
        color: #fff;
        border-color: var(--hse-primary);
    }

    .page-btn:disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }

    .hse-empty-state,
    .hse-error-state {
        text-align: center;
        padding: 40px 20px;
        color: var(--hse-muted);
    }

    .hse-empty-state i,
    .hse-error-state i {
        font-size: 42px;
        margin-bottom: 12px;
        color: var(--hse-primary);
    }

    .hse-inline-loader {
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 24px;
        gap: 12px;
        color: var(--hse-primary);
    }

    .spinner {
        width: 28px;
        height: 28px;
        border: 3px solid rgba(1, 81, 133, 0.15);
        border-top-color: var(--hse-primary);
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        to {
            transform: rotate(360deg);
        }
    }

    .hse-modal {
        position: fixed;
        inset: 0;
        background: rgba(15, 23, 42, 0.55);
        display: none;
        align-items: center;
        justify-content: center;
        padding: 20px;
        z-index: 1050;
    }

    .hse-modal.active {
        display: flex;
    }

    .hse-modal-content {
        background: #fff;
        border-radius: 18px;
        width: min(100%, 960px);
        max-height: 90vh;
        display: flex;
        flex-direction: column;
        box-shadow: 0 30px 60px rgba(15, 23, 42, 0.25);
    }

    .hse-modal-header,
    .hse-modal-footer {
        padding: 20px 24px;
        border-bottom: 1px solid var(--hse-border);
        display: flex;
        justify-content: space-between;
        align-items: center;
        gap: 12px;
    }

    .hse-modal-header h3 {
        margin: 0;
        font-size: 20px;
        color: var(--hse-text);
    }

    .hse-modal-body {
        padding: 20px 24px;
        overflow-y: auto;
    }

    .modal-close-btn {
        border: none;
        background: transparent;
        font-size: 26px;
        color: var(--hse-muted);
        cursor: pointer;
    }

    .details-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
        gap: 14px;
    }

    .details-card {
        border: 1px solid var(--hse-border);
        border-radius: 12px;
        padding: 14px;
        background: #f8fafc;
    }

    .details-card span {
        display: block;
    }

    .details-label {
        font-size: 12px;
        text-transform: uppercase;
        color: var(--hse-muted);
        letter-spacing: 0.08em;
    }

    .details-value {
        font-size: 15px;
        font-weight: 600;
        color: var(--hse-text);
        margin-top: 4px;
    }

    .details-section {
        margin-top: 18px;
    }

    .details-section h4 {
        margin-bottom: 12px;
        color: var(--hse-text);
    }

    .details-badges {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
    }

    .details-badge {
        background: rgba(1, 81, 133, 0.1);
        color: var(--hse-primary);
        border-radius: 999px;
        padding: 6px 14px;
        font-size: 13px;
        font-weight: 600;
    }

    .safety-summary {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }

    .safety-chip {
        flex: 1;
        min-width: 140px;
        border-radius: 12px;
        padding: 12px;
        color: #fff;
    }

    .safety-chip.safe {
        background: linear-gradient(135deg, #0f9d58, #0b7c44);
    }

    .safety-chip.unsafe {
        background: linear-gradient(135deg, #d93025, #b3241b);
    }

    .safety-chip.na {
        background: linear-gradient(135deg, #a0aec0, #718096);
    }

    .safety-chip strong {
        font-size: 20px;
        display: block;
    }

    .attachments-list {
        list-style: none;
        padding-left: 0;
        margin: 0;
        display: flex;
        flex-direction: column;
        gap: 8px;
    }

    .attachments-list a {
        color: var(--hse-primary);
        text-decoration: none;
        font-weight: 600;
    }

    .attachments-list a:hover {
        text-decoration: underline;
    }

    .pdf-modal iframe {
        width: 100%;
        height: 70vh;
        border: none;
        border-radius: 0 0 18px 18px;
    }

    .hse-toast-container {
        position: fixed;
        top: 24px;
        right: 24px;
        display: flex;
        flex-direction: column;
        gap: 10px;
        z-index: 1100;
    }

    .hse-toast {
        min-width: 260px;
        max-width: 360px;
        padding: 14px 16px;
        border-radius: 12px;
        color: #fff;
        font-weight: 500;
        display: flex;
        align-items: center;
        gap: 10px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        transition: opacity 0.3s ease;
    }

    .hse-toast.success {
        background: linear-gradient(135deg, #0f9d58, #0b7c44);
    }

    .hse-toast.error {
        background: linear-gradient(135deg, #d93025, #b3241b);
    }

    .hse-toast.info {
        background: linear-gradient(135deg, #2563eb, #1d4ed8);
    }

    .hse-global-loader {
        position: fixed;
        inset: 0;
        background: rgba(15, 23, 42, 0.65);
        display: flex;
        align-items: center;
        justify-content: center;
        flex-direction: column;
        gap: 14px;
        color: #fff;
        z-index: 1200;
    }
</style>
@endsection

@section('content')
<div class="hse-screen" role="main" aria-label="HSE inspection reports">
<div class="hse-header">
    <div class="hse-header-left">
        <a href="{{ route('inspection.inspection_report') }}" class="hse-back-button" aria-label="Back to Inspection Reports">
            <i class="fas fa-arrow-left"></i>
        </a>
        <div class="hse-header-text">
            <p class="hse-breadcrumb">
                <a href="{{ route('inspection.inspection_report') }}">Inspection Reports</a>
                <span>/</span>
                HSE Inspection Reports
            </p>
            <h1>HSE Inspection Reports</h1>
            <p>Review and manage every HSE inspection, mirror the native experience, and keep corrective actions on track.</p>
        </div>
    </div>
    <div class="hse-header-right">
        <div class="hse-header-search">
            <i class="fas fa-search" aria-hidden="true"></i>
            <input type="text" id="hseSearchInput" placeholder="Search by request #, project, equipment, document ID, location..." aria-label="Search HSE reports">
        </div>
        <div class="hse-header-buttons">
            <button type="button" class="hse-btn light" id="hseRefreshBtn">
                <i class="fas fa-sync-alt"></i>
                Refresh
            </button>
            <button type="button" class="hse-btn primary" onclick="window.location.href='{{ route('inspection.inspection2') }}'">
                <i class="fas fa-plus"></i>
                New HSE Inspection
            </button>
        </div>
    </div>
</div>

<div class="hse-controls">
    <div class="hse-filter-group">
        <label class="sr-only" for="hseStatusFilter">Filter by status</label>
        <select id="hseStatusFilter" aria-label="Filter by status">
            <option value="all">All statuses</option>
            <option value="completed">Completed</option>
            <option value="draft">Draft</option>
            <option value="archived">Archived</option>
        </select>
        <label class="sr-only" for="hsePageSize">Rows per page</label>
        <select id="hsePageSize" aria-label="Rows per page">
            <option value="10">10 rows</option>
            <option value="20">20 rows</option>
            <option value="50">50 rows</option>
        </select>
        <button type="button" class="hse-btn light" id="hseClearFiltersBtn">
            <i class="fas fa-eraser"></i> Clear Filters
        </button>
    </div>
</div>


    <div class="hse-table-card" aria-live="polite">
        <div class="hse-table-toolbar">
            <div>
                <h2 style="margin:0; font-size:20px; color:var(--hse-text);">Reports Table</h2>
                <p class="hse-table-meta" id="hseTableMeta">Loading reports...</p>
            </div>
            <div style="color: var(--hse-muted); font-size: 13px;">
                Data source: {{ route('inspection.hse_reports.data') }}
            </div>
        </div>

        <div class="hse-table-wrapper">
            <table class="hse-table" aria-describedby="hseTableMeta">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Actions</th>
                        <th>Request #</th>
                        <th>Document ID</th>
                        <th>Country</th>
                        <th>Project</th>
                        <th>Equipment</th>
                        <th>Status</th>
                        <th>Corrective</th>
                        <th>Created</th>
                    </tr>
                </thead>
                <tbody id="hseTableBody">
                    <!-- Rows injected via JS -->
                </tbody>
            </table>
        </div>

        <div id="hseTableLoader" class="hse-inline-loader" hidden>
            <div class="spinner" role="status" aria-label="Loading"></div>
            <span>Fetching HSE inspections...</span>
        </div>

        <div id="hseEmptyState" class="hse-empty-state" hidden>
            <i class="fas fa-file-alt" aria-hidden="true"></i>
            <p>No HSE reports found.</p>
        </div>

        <div id="hseErrorState" class="hse-error-state" hidden>
            <i class="fas fa-triangle-exclamation" aria-hidden="true"></i>
            <p id="hseErrorMessage">Unable to fetch reports. Please try again.</p>
        </div>

        <div class="hse-pagination" id="hsePagination">
            <!-- Pagination controls injected dynamically -->
        </div>
    </div>
</div>

<!-- Details Modal -->
<div id="hseDetailsModal" class="hse-modal" role="dialog" aria-modal="true" aria-labelledby="hseDetailsTitle">
    <div class="hse-modal-content">
        <div class="hse-modal-header">
            <h3 id="hseDetailsTitle">HSE Report Details</h3>
            <button type="button" class="modal-close-btn" data-modal-close aria-label="Close details modal">&times;</button>
        </div>
        <div class="hse-modal-body" id="hseDetailsContent">
            <!-- Details injected via JS -->
        </div>
        <div class="hse-modal-footer">
            <button type="button" class="hse-btn ghost" id="hseDetailsHsePdfBtn">
                <i class="fas fa-file-pdf"></i> HSE PDF
            </button>
            <button type="button" class="hse-btn ghost" id="hseDetailsCorrectivePdfBtn">
                <i class="fas fa-screwdriver-wrench"></i> Corrective PDF
            </button>
            <button type="button" class="hse-btn primary" data-modal-close>Close</button>
        </div>
    </div>
</div>

<!-- PDF Modal -->
<div id="hsePdfModal" class="hse-modal" role="dialog" aria-modal="true" aria-labelledby="hsePdfTitle">
    <div class="hse-modal-content pdf-modal">
        <div class="hse-modal-header">
            <h3 id="hsePdfTitle">PDF Preview</h3>
            <button type="button" class="modal-close-btn" data-modal-close aria-label="Close PDF modal">&times;</button>
        </div>
        <div>
            <iframe id="hsePdfIframe" title="HSE PDF preview"></iframe>
        </div>
        <div class="hse-modal-footer">
            <a id="hsePdfDownload" class="hse-btn primary" href="#" target="_blank" rel="noopener">
                <i class="fas fa-download"></i> Download
            </a>
            <button type="button" class="hse-btn light" data-modal-close>Close</button>
        </div>
    </div>
</div>

<div id="hseToastContainer" class="hse-toast-container" aria-live="polite"></div>
<div id="hseGlobalLoader" class="hse-global-loader" hidden>
    <div class="spinner" role="status" aria-label="Processing"></div>
    <p id="hseGlobalLoaderText">Processing...</p>
</div>
@endsection

@section('scripts')
<script>
(function() {
    const REPORT_LIMIT = 500;
    const ROUTES = {
        reports: "{{ route('inspection.hse_reports.data') }}",
        corrective: "{{ url('/inspection/hse-reports/corrective') }}",
        delete: "{{ url('/inspection/hse-reports') }}"
    };
    const CSRF_TOKEN = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';

    const SAFE_VALUES = new Set(['safe', 'ok', 'okay', 'pass', 'operational', 'satisfactory']);
    const UNSAFE_VALUES = new Set(['unsafe', 'faulty', 'reject', 'not ok', 'not okay', 'fail', 'failures', 'unsatisfactory']);
    const NA_VALUES = new Set(['n/a', 'not applicable', 'na']);

    const state = {
        allReports: [],
        filteredReports: [],
        correctiveCache: new Map(),
        searchTerm: '',
        statusFilter: 'all',
        pageSize: 10,
        currentPage: 1,
        isLoading: false,
        activeReportId: null
    };

    const dom = {
        searchInput: document.getElementById('hseSearchInput'),
        statusFilter: document.getElementById('hseStatusFilter'),
        pageSize: document.getElementById('hsePageSize'),
        clearFilters: document.getElementById('hseClearFiltersBtn'),
        refreshBtn: document.getElementById('hseRefreshBtn'),
        tableBody: document.getElementById('hseTableBody'),
        tableLoader: document.getElementById('hseTableLoader'),
        emptyState: document.getElementById('hseEmptyState'),
        errorState: document.getElementById('hseErrorState'),
        errorMessage: document.getElementById('hseErrorMessage'),
        pagination: document.getElementById('hsePagination'),
        tableMeta: document.getElementById('hseTableMeta'),
        totalCount: document.getElementById('hseTotalCount'),
        totalMeta: document.getElementById('hseTotalMeta'),
        completedCount: document.getElementById('hseCompletedCount'),
        draftCount: document.getElementById('hseDraftCount'),
        attentionCount: document.getElementById('hseAttentionCount'),
        toastContainer: document.getElementById('hseToastContainer'),
        globalLoader: document.getElementById('hseGlobalLoader'),
        globalLoaderText: document.getElementById('hseGlobalLoaderText'),
        detailsModal: document.getElementById('hseDetailsModal'),
        detailsContent: document.getElementById('hseDetailsContent'),
        detailsHsePdfBtn: document.getElementById('hseDetailsHsePdfBtn'),
        detailsCorrectivePdfBtn: document.getElementById('hseDetailsCorrectivePdfBtn'),
        pdfModal: document.getElementById('hsePdfModal'),
        pdfIframe: document.getElementById('hsePdfIframe'),
        pdfDownload: document.getElementById('hsePdfDownload'),
        pdfTitle: document.getElementById('hsePdfTitle')
    };

    let openDropdownMenu = null;
    let openDropdownToggle = null;

    document.addEventListener('DOMContentLoaded', init);

    function init() {
        attachEvents();
        loadReports();
    }

    function attachEvents() {
        dom.searchInput.addEventListener('input', (event) => {
            state.searchTerm = (event.target.value || '').trim().toLowerCase();
            applyFilters();
        });

        dom.statusFilter.addEventListener('change', (event) => {
            state.statusFilter = event.target.value;
            applyFilters();
        });

        dom.pageSize.addEventListener('change', (event) => {
            state.pageSize = Number(event.target.value) || 10;
            state.currentPage = 1;
            renderTable();
        });

        dom.clearFilters.addEventListener('click', () => {
            state.searchTerm = '';
            state.statusFilter = 'all';
            dom.searchInput.value = '';
            dom.statusFilter.value = 'all';
            applyFilters();
        });

        dom.refreshBtn.addEventListener('click', () => {
            loadReports();
        });

        dom.tableBody.addEventListener('click', handleTableAction);
        document.addEventListener('click', handleGlobalClicks);
        document.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                closeAllDropdowns();
            }
        });

        document.querySelectorAll('[data-modal-close]').forEach((btn) => {
            btn.addEventListener('click', () => closeModal(btn.closest('.hse-modal')));
        });

        dom.detailsModal.addEventListener('click', (event) => {
            if (event.target === dom.detailsModal) {
                closeModal(dom.detailsModal);
            }
        });

        dom.pdfModal.addEventListener('click', (event) => {
            if (event.target === dom.pdfModal) {
                closeModal(dom.pdfModal);
            }
        });

        dom.detailsHsePdfBtn.addEventListener('click', () => {
            if (state.activeReportId) {
                viewHsePdf(state.activeReportId);
            }
        });

        dom.detailsCorrectivePdfBtn.addEventListener('click', () => {
            if (state.activeReportId) {
                viewCorrectivePdf(state.activeReportId);
            }
        });
    }

    async function loadReports() {
        if (state.isLoading) return;
        state.isLoading = true;
        toggleTableLoader(true);
        toggleError(false);

        try {
            const response = await fetch(`${ROUTES.reports}?page=1&limit=${REPORT_LIMIT}`, {
                method: 'GET',
                headers: { 'Accept': 'application/json' }
            });

            if (!response.ok) {
                throw new Error(`API responded with status ${response.status}`);
            }

            const payload = await response.json();
            const reports = Array.isArray(payload.data) ? payload.data : [];

            state.allReports = reports.map(normalizeReport);
            state.correctiveCache.clear();
            applyFilters(true);
            updateSummaryCards();
            showToast('success', `Loaded ${state.allReports.length} HSE reports`);
            hydrateCorrectiveData(state.allReports).catch((error) => {
                console.warn('Corrective hydration skipped:', error);
            });
        } catch (error) {
            console.error('Failed to load HSE reports:', error);
            toggleError(true, error.message || 'Unable to fetch reports');
            state.allReports = [];
            state.filteredReports = [];
            renderTable();
            updateSummaryCards();
        } finally {
            toggleTableLoader(false);
            state.isLoading = false;
        }
    }

    function normalizeReport(report) {
        const clone = { ...report };
        clone._id = clone._id || clone.id || clone.requestNumber || clone.step1?.requestNumber || '';
        clone.requestNumber = clone.requestNumber || clone.step1?.requestNumber || clone._id;
        clone.status = (clone.status || 'draft').toLowerCase();
        clone.createdAt = clone.createdAt || clone.step1?.createdAt || clone.updatedAt;
        clone.correctiveAction =
            clone.correctiveAction ||
            clone.correctiveActionData ||
            clone.correctiveActionInfo ||
            clone.meta?.correctiveAction ||
            null;
        return clone;
    }

    function applyFilters(skipReset = false) {
        const term = state.searchTerm;
        const status = state.statusFilter;

        state.filteredReports = state.allReports.filter((report) => {
            const matchesStatus = status === 'all' || (report.status || '').toLowerCase() === status;
            if (!matchesStatus) return false;
            if (!term) return true;

            const haystack = [
                pickFirst(report.requestNumber, report._id),
                pickFirst(report.step1?.selectedChecklist?.document_id),
                pickFirst(report.step1?.country, report.country),
                pickFirst(report.step1?.project, report.project),
                pickFirst(report.step1?.selectedEquipment?.equipmentName, report.step1?.selectedEquipment?.equipmentNumber),
                pickFirst(report.userId),
            ]
                .filter(Boolean)
                .map((value) => value.toLowerCase());

            return haystack.some((value) => value.includes(term));
        });

        if (!skipReset) {
            state.currentPage = 1;
        }

        renderTable();
        updateSummaryCards();
    }

    function renderTable() {
        const total = state.filteredReports.length;
        const allTotal = state.allReports.length;

        if (!total) {
            dom.tableBody.innerHTML = '';
            dom.emptyState.hidden = false;
            dom.tableMeta.textContent = allTotal
                ? 'No reports match the applied filters.'
                : 'No reports available.';
            dom.pagination.innerHTML = '';
            return;
        }

        dom.emptyState.hidden = true;

        const totalPages = Math.max(1, Math.ceil(total / state.pageSize));
        state.currentPage = Math.min(state.currentPage, totalPages);
        const startIndex = (state.currentPage - 1) * state.pageSize;
        const endIndex = Math.min(startIndex + state.pageSize, total);
        const pageItems = state.filteredReports.slice(startIndex, endIndex);

        dom.tableMeta.textContent = `Showing ${startIndex + 1}-${endIndex} of ${total} filtered reports (from ${allTotal} total)`;

        dom.tableBody.innerHTML = pageItems
            .map((report, index) => buildTableRow(report, state.filteredReports.length - (startIndex + index)))
            .join('');

        renderPagination(totalPages, total);
    }

    function buildTableRow(report, serialNumber) {
        const id = escapeHtml(report._id || '');
        const requestNumber = escapeHtml(pickFirst(report.requestNumber, report._id, 'N/A'));
        const documentId = escapeHtml(
            pickFirst(
                report.step1?.selectedChecklist?.document_id,
                report.selectedChecklist?.document_id,
                'N/A'
            )
        );
        const country = escapeHtml(pickFirst(report.step1?.country, report.country, '—'));
        const project = escapeHtml(pickFirst(report.step1?.project, report.project, '—'));
        const equipment = escapeHtml(
            pickFirst(
                report.step1?.selectedEquipment?.equipmentName,
                report.step1?.selectedEquipment?.equipmentNumber,
                'N/A'
            )
        );
        const createdAt = formatDate(report.createdAt);
        const embeddedCorrective = getEmbeddedCorrective(report);
        const correctiveStatus = getCorrectiveStatus(embeddedCorrective, report.status);
        const statusClass = correctiveStatus.className;
        const statusLabel = correctiveStatus.label;
        const dropdownId = `hse-dropdown-${id}`;
        const hasHsePdf = Boolean(report.pdfDownloadUrl);
        const hasCorrectivePdf = Boolean(embeddedCorrective?.pdfDownloadUrl);

        return `
            <tr>
                <td>${serialNumber}</td>
                <td>
                    <div class="hse-dropdown">
                        <button class="hse-action-btn" type="button" data-dropdown-toggle="${dropdownId}">
                            <i class="fas fa-ellipsis-v" aria-hidden="true"></i>
                            Actions
                        </button>
                        <div class="hse-dropdown-menu" id="${dropdownId}">
                            <button type="button" class="hse-dropdown-item" data-action="details" data-id="${id}">
                                <i class="fas fa-eye"></i>
                                View Details
                            </button>
                            <button type="button" class="hse-dropdown-item" data-action="view-hse-pdf" data-id="${id}" ${hasHsePdf ? '' : 'disabled'}>
                                <i class="fas fa-file-pdf"></i>
                                View HSE PDF
                            </button>
                            <button type="button" class="hse-dropdown-item" data-action="view-corrective-pdf" data-id="${id}" ${hasCorrectivePdf ? '' : 'disabled'}>
                                <i class="fas fa-screwdriver-wrench"></i>
                                Corrective PDF
                            </button>
                            <div class="hse-dropdown-divider"></div>
                            <button type="button" class="hse-dropdown-item danger" data-action="delete" data-id="${id}">
                                <i class="fas fa-trash"></i>
                                Delete Report
                            </button>
                        </div>
                    </div>
                </td>
                <td>${requestNumber}</td>
                <td>${documentId}</td>
                <td>${country}</td>
                <td>${project}</td>
                <td>${equipment}</td>
                <td>
                    <span class="status-pill ${statusClass}">
                        <i class="fas fa-circle"></i> ${statusLabel}
                    </span>
                </td>
                <td>
                    ${buildCorrectiveCell(embeddedCorrective, correctiveStatus, report)}
                </td>
                <td>${createdAt}</td>
            </tr>
        `;
    }

    function renderPagination(totalPages, totalItems) {
        if (totalPages <= 1) {
            dom.pagination.innerHTML = '';
            return;
        }

        const infoText = `Page ${state.currentPage} of ${totalPages}`;
        const prevDisabled = state.currentPage === 1 ? 'disabled' : '';
        const nextDisabled = state.currentPage === totalPages ? 'disabled' : '';

        dom.pagination.innerHTML = `
            <div class="hse-table-meta">${infoText}</div>
            <div class="page-controls">
                <button class="page-btn" ${prevDisabled} data-page="${state.currentPage - 1}"><i class="fas fa-chevron-left"></i></button>
                ${buildPageButtons(totalPages)}
                <button class="page-btn" ${nextDisabled} data-page="${state.currentPage + 1}"><i class="fas fa-chevron-right"></i></button>
            </div>
        `;

        dom.pagination.querySelectorAll('.page-btn').forEach((btn) => {
            btn.addEventListener('click', () => {
                const targetPage = Number(btn.dataset.page);
                if (targetPage && targetPage >= 1 && targetPage <= totalPages) {
                    state.currentPage = targetPage;
                    renderTable();
                }
            });
        });
    }

    function buildPageButtons(totalPages) {
        let buttons = '';
        const maxButtons = 5;
        let start = Math.max(1, state.currentPage - Math.floor(maxButtons / 2));
        let end = Math.min(totalPages, start + maxButtons - 1);
        if (end - start < maxButtons - 1) {
            start = Math.max(1, end - maxButtons + 1);
        }
        for (let page = start; page <= end; page++) {
            const activeClass = page === state.currentPage ? 'active' : '';
            buttons += `<button class="page-btn ${activeClass}" data-page="${page}">${page}</button>`;
        }
        return buttons;
    }

    function updateSummaryCards() {
        if (!dom.totalCount || !dom.completedCount || !dom.draftCount || !dom.attentionCount || !dom.totalMeta) {
            return;
        }
        const total = state.allReports.length;
        const completed = state.allReports.filter((report) => getCorrectiveStatus(getEmbeddedCorrective(report), report.status).key === 'completed').length;
        const drafts = state.allReports.filter((report) => {
            const status = getCorrectiveStatus(getEmbeddedCorrective(report), report.status).key;
            return status === 'draft' || status === 'in_progress';
        }).length;
        const attention = state.allReports.filter((report) => {
            const unsafe = getUnsafeItemCount(report);
            if (unsafe === 0) return false;
            const correctiveStatus = getCorrectiveStatus(getEmbeddedCorrective(report)).key;
            return correctiveStatus !== 'completed';
        }).length;

        dom.totalCount.textContent = total;
        dom.totalMeta.textContent = total ? `Last refresh ${new Date().toLocaleString()}` : 'Awaiting data...';
        dom.completedCount.textContent = completed;
        dom.draftCount.textContent = drafts;
        dom.attentionCount.textContent = attention;
    }

    function toggleTableLoader(isLoading) {
        dom.tableLoader.hidden = !isLoading;
    }

    function toggleError(show, message = '') {
        dom.errorState.hidden = !show;
        if (show) {
            dom.errorMessage.textContent = message || 'Something went wrong.';
        }
    }

    function toggleGlobalLoader(show, message = 'Processing...') {
        dom.globalLoader.hidden = !show;
        dom.globalLoaderText.textContent = message;
        document.body.classList.toggle('modal-open', show);
    }

    function showToast(type, message) {
        if (!dom.toastContainer) return;
        const toast = document.createElement('div');
        toast.className = `hse-toast ${type}`;
        toast.innerHTML = `<i class="fas fa-info-circle" aria-hidden="true"></i><span>${message}</span>`;
        dom.toastContainer.appendChild(toast);
        setTimeout(() => {
            toast.classList.add('fade-out');
            toast.addEventListener('transitionend', () => toast.remove());
            toast.style.opacity = '0';
        }, 3500);
    }

    async function handleTableAction(event) {
        closeAllDropdowns();
        const button = event.target.closest('[data-action]');
        if (!button) return;
        const action = button.dataset.action;
        const reportId = button.dataset.id;

        if (!reportId) {
            showToast('error', 'Unable to determine report ID.');
            return;
        }

        switch (action) {
            case 'details':
                await openDetailsModal(reportId);
                break;
            case 'view-hse-pdf':
                viewHsePdf(reportId);
                break;
            case 'view-corrective-pdf':
                await viewCorrectivePdf(reportId);
                break;
            case 'delete':
                await deleteReport(reportId);
                break;
            default:
                break;
        }
    }

    function handleGlobalClicks(event) {
        const toggle = event.target.closest('[data-dropdown-toggle]');
        if (toggle) {
            const menuId = toggle.getAttribute('data-dropdown-toggle');
            const menu = document.getElementById(menuId);
            if (!menu) {
                return;
            }
            const isOpen = menu.classList.contains('show');
            closeAllDropdowns();
            if (!isOpen) {
                menu.classList.add('show');
                toggle.classList.add('active');
                openDropdownMenu = menu;
                openDropdownToggle = toggle;
            }
            event.preventDefault();
            event.stopPropagation();
            return;
        }

        if (!event.target.closest('.hse-dropdown')) {
            closeAllDropdowns();
        }
    }

    function closeAllDropdowns() {
        if (openDropdownMenu) {
            openDropdownMenu.classList.remove('show');
            openDropdownMenu = null;
        }
        if (openDropdownToggle) {
            openDropdownToggle.classList.remove('active');
            openDropdownToggle = null;
        }
    }

    async function openDetailsModal(reportId) {
        const report = findReportById(reportId);
        if (!report) {
            showToast('error', 'Report not found.');
            return;
        }

        state.activeReportId = reportId;
        dom.detailsHsePdfBtn.disabled = !report.pdfDownloadUrl;
        dom.detailsCorrectivePdfBtn.disabled = false;

        const summary = deriveSafetySummary(report);
        const corrective = await ensureCorrectiveAction(reportId, { silent: true }) || getEmbeddedCorrective(report);
        dom.detailsContent.innerHTML = buildDetailsMarkup(report, summary, corrective);
        openModal(dom.detailsModal);
    }

    function buildDetailsMarkup(report, summary, corrective) {
        const attachments = Array.isArray(report.attachments) ? report.attachments : [];
        const notes = report.step2?.notes || report.step3?.notes || report.notes;
        const equipment = report.step1?.selectedEquipment || {};
        const correctiveStatus = getCorrectiveStatus(corrective, report.status);

        const attachmentsHtml = attachments.length
            ? attachments
                  .map((file, index) => {
                      const href = sanitizeUrl(file.fileUrl || file.downloadUrl || file.url);
                      if (!href) return `<li><span>${escapeHtml(file.fileName || `Attachment ${index + 1}`)}</span></li>`;
                      return `<li><a href="${href}" target="_blank" rel="noopener">${escapeHtml(file.fileName || `Attachment ${index + 1}`)}</a></li>`;
                  })
                  .join('')
            : '<li>No attachments uploaded.</li>';

        return `
            <div class="details-section">
                <h4>General Information</h4>
                <div class="details-grid">
                    ${detailsCard('Request Number', pickFirst(report.requestNumber, report._id, 'N/A'))}
                    ${detailsCard('Document ID', pickFirst(report.step1?.selectedChecklist?.document_id, 'N/A'))}
                    ${detailsCard('Country', pickFirst(report.step1?.country, '—'))}
                    ${detailsCard('Project', pickFirst(report.step1?.project, '—'))}
                    ${detailsCard('Equipment', pickFirst(equipment.equipmentName, equipment.equipmentNumber, '—'))}
                    ${detailsCard('Status', formatStatus(report.status))}
                    ${detailsCard('Created At', formatDate(report.createdAt), true)}
                    ${detailsCard('Updated At', formatDate(report.updatedAt), true)}
                </div>
            </div>

            <div class="details-section">
                <h4>Safety Summary</h4>
                <div class="safety-summary">
                    ${safetyChip('safe', 'Safe', summary.safe)}
                    ${safetyChip('unsafe', 'Unsafe', summary.unsafe)}
                    ${safetyChip('na', 'N/A', summary.na)}
                </div>
                <div class="details-badges" style="margin-top:12px;">
                    <span class="details-badge">Total items: ${summary.total}</span>
                    <span class="details-badge">Safe %: ${summary.percentSafe}%</span>
                </div>
            </div>

            <div class="details-section">
                <h4>Corrective Action</h4>
                <div class="details-badges" style="margin-bottom:12px;">
                    <span class="details-badge ${correctiveStatus.className}">${correctiveStatus.label}</span>
                    <span class="details-badge">Unsafe items: ${getUnsafeItemCount(report)}</span>
                </div>
                <p style="color: var(--hse-muted); font-size: 14px;">
                    ${corrective ? escapeHtml(corrective.notes || 'Latest corrective action available.') : 'No corrective action has been submitted for this report.'}
                </p>
            </div>

            <div class="details-section">
                <h4>Attachments</h4>
                <ul class="attachments-list">
                    ${attachmentsHtml}
                </ul>
            </div>

            ${
                notes
                    ? `<div class="details-section">
                        <h4>Inspector Notes</h4>
                        <p style="white-space: pre-wrap; color: var(--hse-text);">${escapeHtml(notes)}</p>
                    </div>`
                    : ''
            }
        `;
    }

    function detailsCard(label, value, allowEmpty = false) {
        const safeValue = allowEmpty ? value : value || '—';
        return `
            <div class="details-card">
                <span class="details-label">${escapeHtml(label)}</span>
                <span class="details-value">${escapeHtml(safeValue)}</span>
            </div>
        `;
    }

    function safetyChip(type, label, value) {
        return `
            <div class="safety-chip ${type}">
                <span>${label}</span>
                <strong>${value}</strong>
            </div>
        `;
    }

    function openModal(modal) {
        if (!modal) return;
        modal.classList.add('active');
        modal.setAttribute('aria-hidden', 'false');
        document.body.classList.add('modal-open');
    }

    function closeModal(modal) {
        if (!modal) return;
        modal.classList.remove('active');
        modal.setAttribute('aria-hidden', 'true');
        if (modal === dom.pdfModal) {
            dom.pdfIframe.src = '';
        }
        if (!document.querySelector('.hse-modal.active')) {
            document.body.classList.remove('modal-open');
        }
    }

    function viewHsePdf(reportId) {
        const report = findReportById(reportId);
        if (!report || !report.pdfDownloadUrl) {
            showToast('info', 'PDF not available for this report.');
            return;
        }
        openPdfModal(report.pdfDownloadUrl, `HSE Report • ${pickFirst(report.requestNumber, report._id)}`);
    }

    async function viewCorrectivePdf(reportId) {
        const corrective = await ensureCorrectiveAction(reportId);
        if (corrective && corrective.pdfDownloadUrl) {
            openPdfModal(corrective.pdfDownloadUrl, `Corrective Action • ${reportId}`);
            return;
        }
        showToast('info', 'Corrective action PDF not available.');
    }

    function openPdfModal(url, title) {
        const proxyUrl = buildPdfProxyUrl(url);
        if (!proxyUrl) {
            showToast('error', 'Invalid PDF URL.');
            return;
        }
        dom.pdfIframe.src = proxyUrl;
        dom.pdfDownload.href = proxyUrl;
        dom.pdfTitle.textContent = title || 'PDF Preview';
        openModal(dom.pdfModal);
    }

    function buildPdfProxyUrl(originalUrl) {
        if (!originalUrl) return null;
        const safeUrl = originalUrl.trim();

        if (!safeUrl) {
            return null;
        }

        if (safeUrl.startsWith('/api/inspections/view-pdf')) {
            return safeUrl;
        }

        return `/api/inspections/view-pdf?path=${encodeURIComponent(safeUrl)}`;
    }

    function formatFileNameFragment(value, fallback = 'document') {
        if (!value) return fallback;
        const cleaned = String(value).trim().replace(/[^a-zA-Z0-9_-]+/g, '_');
        return cleaned || fallback;
    }

    async function ensureCorrectiveAction(reportId, options = {}) {
        const { silent = false } = options;
        if (state.correctiveCache.has(reportId)) {
            return state.correctiveCache.get(reportId);
        }

        try {
            const response = await fetch(`${ROUTES.corrective}/${encodeURIComponent(reportId)}`, {
                headers: { 'Accept': 'application/json' }
            });
            if (!response.ok) {
                throw new Error(`Corrective action API responded with ${response.status}`);
            }
            const payload = await response.json();
            if (!payload.success || !payload.data) {
                throw new Error(payload.error || 'Corrective action not found');
            }
            state.correctiveCache.set(reportId, payload.data);
            const targetReport = findReportById(reportId);
            if (targetReport) {
                targetReport.correctiveAction = payload.data;
                targetReport.correctiveActionData = payload.data;
            }
            return payload.data;
        } catch (error) {
            console.warn('Corrective action fetch failed:', error);
            if (!silent) {
                showToast('info', 'Corrective action data not found.');
            }
            return null;
        }
    }

    async function hydrateCorrectiveData(reports = []) {
        if (!Array.isArray(reports) || reports.length === 0) {
            return;
        }
        const candidates = reports.filter((report) => {
            if (!report?._id) return false;
            const embedded = getEmbeddedCorrective(report);
            return !embedded || !embedded.pdfDownloadUrl;
        });
        if (!candidates.length) {
            return;
        }
        let updated = false;
        for (const report of candidates) {
            const corrective = await ensureCorrectiveAction(report._id, { silent: true });
            if (corrective) {
                report.correctiveAction = corrective;
                report.correctiveActionData = corrective;
                updated = true;
            }
        }
        if (updated) {
            applyFilters(true);
            updateSummaryCards();
            showToast('info', 'Corrective action PDFs synced from backend.');
        }
    }

    async function deleteReport(reportId) {
        const report = findReportById(reportId);
        if (!report) {
            showToast('error', 'Report not found.');
            return;
        }

        const confirmed = window.confirm(`Delete report ${report.requestNumber || reportId}? This cannot be undone.`);
        if (!confirmed) return;

        toggleGlobalLoader(true, 'Deleting HSE report...');

        try {
            const response = await fetch(`${ROUTES.delete}/${encodeURIComponent(report.userId)}/${encodeURIComponent(reportId)}`, {
                method: 'DELETE',
                headers: {
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': CSRF_TOKEN
                }
            });

            if (!response.ok) {
                throw new Error(`Delete API responded with ${response.status}`);
            }

            const payload = await response.json();
            if (!payload.success) {
                throw new Error(payload.error || 'Failed to delete report');
            }

            state.allReports = state.allReports.filter((item) => (item._id || item.id) !== reportId);
            applyFilters(true);
            updateSummaryCards();
            showToast('success', 'Report deleted successfully.');
        } catch (error) {
            console.error('Delete failed:', error);
            showToast('error', error.message || 'Unable to delete report.');
        } finally {
            toggleGlobalLoader(false);
        }
    }

    function deriveSafetySummary(report) {
        const summary = { safe: 0, unsafe: 0, na: 0, total: 0, percentSafe: 0 };
        const step3 = report.step3 || {};

        const hasStep3Totals =
            typeof step3.totalSafe === 'number' ||
            typeof step3.totalUnsafe === 'number' ||
            typeof step3.totalNA === 'number';

        if (hasStep3Totals) {
            summary.safe = coerceNumber(step3.totalSafe);
            summary.unsafe = coerceNumber(step3.totalUnsafe);
            summary.na = coerceNumber(step3.totalNA);
        }

        if (!hasStep3Totals || summary.safe + summary.unsafe + summary.na === 0) {
            const sections = Array.isArray(report.step2?.sections) ? report.step2.sections : [];
            const selections = report.step2?.selections || {};
            const countInputs = report.step2?.countInputs || {};

            sections.forEach((section) => {
                (section?.subItems || []).forEach((item) => {
                    if (!item?.key) return;
                    const key = item.key;
                    const inputType = normalizeValue(item.inputType);
                    const selection = normalizeValue(selections[key]);
                    const counts = countInputs[key] || {};

                    if (inputType.includes('safe/unsafe')) {
                        summary.safe += coerceNumber(counts.safe);
                        summary.unsafe += coerceNumber(counts.unsafe);
                        if (normalizeValue(counts.notApplicable) === 'true') {
                            summary.na += 1;
                        }
                        return;
                    }

                    if (!selection) return;
                    if (SAFE_VALUES.has(selection)) summary.safe += 1;
                    else if (UNSAFE_VALUES.has(selection)) summary.unsafe += 1;
                    else if (NA_VALUES.has(selection)) summary.na += 1;
                });
            });
        }

        summary.total = summary.safe + summary.unsafe + summary.na;
        summary.percentSafe = summary.total > 0 ? Math.round((summary.safe / summary.total) * 100) : 0;

        return summary;
    }

    function getUnsafeItemCount(report) {
        const sections = Array.isArray(report.step2?.sections) ? report.step2.sections : [];
        const selections = report.step2?.selections || {};
        const countInputs = report.step2?.countInputs || {};
        const counted = new Set();

        sections.forEach((section) => {
            (section?.subItems || []).forEach((item) => {
                if (!item?.key || counted.has(item.key)) return;
                const selection = normalizeValue(selections[item.key]);
                const unsafeCount = coerceNumber(countInputs[item.key]?.unsafe);
                if (unsafeCount > 0 || UNSAFE_VALUES.has(selection)) {
                    counted.add(item.key);
                }
            });
        });

        return counted.size;
    }

    function getCorrectiveStatus(correctiveAction, fallbackStatus = 'draft') {
        if (correctiveAction?.pdfDownloadUrl) {
            return mapStatusDescriptor('completed');
        }
        if (!correctiveAction) {
            return mapStatusDescriptor(fallbackStatus || 'draft', true);
        }

        const statuses = Object.values(correctiveAction.subItemStatuses || {});
        if (statuses.length) {
            const isComplete = statuses.every((status) => {
                const normalized = normalizeValue(status);
                return normalized === 'fixed' || normalized === 'no need' || normalized === 'no_need';
            });
            if (isComplete) {
                return { key: 'completed', label: 'Completed', className: 'completed' };
            }
            return mapStatusDescriptor('in_progress');
        }

        return mapStatusDescriptor(correctiveAction.status || fallbackStatus || 'draft');
    }

    function mapStatusDescriptor(status = 'draft', allowEmpty = false) {
        const normalized = normalizeValue(status);
        switch (normalized) {
            case 'completed':
                return { key: 'completed', label: 'Completed', className: 'completed' };
            case 'in_progress':
            case 'progress':
            case 'active':
                return { key: 'in_progress', label: 'In Progress', className: 'draft' };
            case 'pending':
            case 'none':
                return { key: 'pending', label: 'Pending', className: 'pending' };
            case 'draft':
                return { key: 'draft', label: 'Draft', className: 'draft' };
            default:
                return allowEmpty
                    ? { key: 'none', label: 'Not Submitted', className: 'pending' }
                    : { key: 'draft', label: 'Draft', className: 'draft' };
        }
    }

    function buildCorrectiveCell(correctiveAction, statusInfo, report) {
        const number = correctiveAction?.correctiveActionNumber
            ? escapeHtml(correctiveAction.correctiveActionNumber)
            : `CA-${escapeHtml(pickFirst(report.requestNumber, report._id, 'N/A'))}`;
        const unsafe = correctiveAction?.unsafeSummary?.totalUnsafeItems ?? getUnsafeItemCount(report);
        return `
            <div class="corrective-cell">
                <span class="status-pill ${statusInfo.className}">
                    <i class="fas fa-screwdriver-wrench"></i> ${statusInfo.label}
                </span>
                <div class="corrective-meta-line">
                    <span class="corrective-number ${correctiveAction?.correctiveActionNumber ? '' : 'muted'}">${number}</span>
                    <span class="corrective-unsafe"><i class="fas fa-exclamation-circle"></i> Unsafe: ${unsafe}</span>
                </div>
            </div>
        `;
    }

    function getStatusClass(status) {
        return mapStatusDescriptor(status).className;
    }

    function formatStatus(status) {
        return mapStatusDescriptor(status).label;
    }

    function formatDate(value) {
        if (!value) return '—';
        const date = new Date(value);
        if (isNaN(date)) return '—';
        return date.toLocaleString('en-GB', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    function pickFirst(...values) {
        for (const value of values) {
            if (value === null || value === undefined) continue;
            const text = typeof value === 'string' || typeof value === 'number' ? String(value).trim() : '';
            if (text) return text;
        }
        return '';
    }

    function escapeHtml(value) {
        const text = value === undefined || value === null ? '' : String(value);
        return text
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    function sanitizeUrl(url) {
        if (!url || typeof url !== 'string') return null;
        if (url.startsWith('http://') || url.startsWith('https://') || url.startsWith('/')) {
            return url;
        }
        return null;
    }

    function normalizeValue(value) {
        if (value === null || value === undefined) return '';
        return String(value).trim().toLowerCase();
    }

    function coerceNumber(value) {
        const num = Number(value);
        return Number.isFinite(num) ? num : 0;
    }

    function findReportById(reportId) {
        return state.allReports.find((report) => (report._id || report.id) === reportId);
    }

    function getEmbeddedCorrective(report = {}) {
        return (
            report.correctiveAction ||
            report.correctiveActionData ||
            report.correctiveActionInfo ||
            report.meta?.correctiveAction ||
            null
        );
    }
})();
</script>
@endsection

