@extends('welcome')

@section('title', 'Funeral Grant Applications')

@section('styles')
<meta name="csrf-token" content="{{ csrf_token() }}">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f3f4f6;
    --border-color: #e5e7eb;
    --shadow-md: 0 20px 25px -5px rgba(0,0,0,0.1);
    --shadow-sm: 0 5px 10px rgba(0,0,0,0.05);
    --radius-lg: 18px;
}

body {
    font-family: 'Inter', sans-serif;
    background: var(--bg-light);
    color: var(--text-primary);
}

.applications-container {
    min-height: 100vh;
    background: linear-gradient(180deg, rgba(0,85,165,0.05) 0%, rgba(0,85,165,0.01) 100%);
}

.header-gradient {
    background: linear-gradient(135deg, var(--primary-color), var(--primary-hover));
    padding: 24px 0;
    color: white;
    box-shadow: var(--shadow-md);
}

.header-content {
    width: 100%;
    padding: 0 24px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 16px;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 18px;
}

.header-title {
    font-size: 26px;
    font-weight: 600;
    margin: 0;
}

.back-button {
    width: 44px;
    height: 44px;
    border-radius: 14px;
    border: 1px solid rgba(255,255,255,0.4);
    background: rgba(255,255,255,0.15);
    color: white;
    display: grid;
    place-items: center;
    font-size: 18px;
    transition: all 0.2s ease;
}

.back-button:hover {
    color: white;
    text-decoration: none;
    transform: translateY(-1px);
    background: rgba(255,255,255,0.25);
}

.header-right .create-btn {
    background: white;
    color: var(--primary-color);
    padding: 12px 24px;
    border-radius: 14px;
    font-weight: 600;
    border: none;
    display: inline-flex;
    align-items: center;
    gap: 10px;
    box-shadow: var(--shadow-sm);
    transition: all 0.2s ease;
}

.header-right .create-btn:hover {
    text-decoration: none;
    transform: translateY(-2px);
}

.main-content {
    padding: 30px 24px;
    max-width: 1200px;
    margin: 0 auto;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
    gap: 16px;
    margin-bottom: 30px;
}

.stat-card {
    background: white;
    padding: 20px;
    border-radius: var(--radius-lg);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
}

.stat-card h4 {
    margin: 0;
    font-size: 14px;
    color: var(--text-secondary);
    font-weight: 500;
}

.stat-card .stat-value {
    font-size: 32px;
    font-weight: 700;
    margin-top: 8px;
}

.applications-panel {
    background: white;
    border-radius: var(--radius-lg);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-md);
    padding: 0 0 20px 0;
}

.panel-header {
    padding: 24px 28px;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    align-items: center;
    gap: 14px;
}

.panel-header h2 {
    margin: 0;
    font-size: 20px;
    font-weight: 600;
}

.applications-list {
    padding: 24px 28px;
    display: flex;
    flex-direction: column;
    gap: 18px;
}

.application-card {
    border: 1px solid var(--border-color);
    border-radius: 16px;
    padding: 20px;
    background: #fff;
    box-shadow: var(--shadow-sm);
    transition: all 0.2s ease;
    cursor: pointer;
}

.application-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
}

.applicant-name {
    font-size: 18px;
    font-weight: 600;
    margin: 0;
}

.employee-no {
    margin: 2px 0 0 0;
    font-size: 14px;
    color: var(--text-secondary);
}

.status {
    padding: 6px 14px;
    border-radius: 100px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.pending { background: #fff4d7; color: #9c6b00; }
.status.approved { background: #d9f7e7; color: #0f8a46; }
.status.rejected { background: #ffe1e1; color: #b82020; }

.card-body {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
    gap: 12px;
    margin-bottom: 14px;
}

.detail-block span {
    display: block;
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.6px;
}

.detail-block strong {
    font-size: 15px;
    color: var(--text-primary);
    font-weight: 600;
}

.btn-menu {
    width: 38px;
    height: 38px;
    border-radius: 12px;
    border: 1px solid var(--border-color);
    background: #f8fafc;
    display: grid;
    place-items: center;
    color: #6b7280;
    transition: all 0.2s ease;
}

.btn-menu:hover {
    background: #e5e7eb;
    color: var(--text-primary);
}

.loading-state,
.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: var(--text-secondary);
}

.loading-spinner {
    width: 42px;
    height: 42px;
    border-radius: 50%;
    border: 4px solid #e5e7eb;
    border-top-color: var(--primary-color);
    margin: 0 auto 16px auto;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.action-menu {
    position: fixed;
    background: white;
    border-radius: 14px;
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-md);
    overflow: hidden;
    z-index: 5000;
    width: 190px;
}

.action-menu-item {
    padding: 12px 16px;
    font-size: 14px;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 10px;
    cursor: pointer;
}

.action-menu-item:hover {
    background: var(--bg-light);
}

.action-menu-item-danger {
    color: var(--danger-color);
}

.confirm-dialog {
    position: fixed;
    inset: 0;
    background: rgba(15,23,42,0.55);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9000;
}

.confirm-dialog-content {
    background: #fff;
    padding: 24px;
    border-radius: 16px;
    width: min(420px, 90%);
    box-shadow: var(--shadow-md);
}

.confirm-dialog-buttons {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    margin-top: 20px;
}

.btn {
    border: none;
    border-radius: 10px;
    padding: 10px 18px;
    font-weight: 600;
    cursor: pointer;
}

.btn-confirm {
    background: var(--primary-color);
    color: white;
}

.btn-danger {
    background: var(--danger-color);
    color: white;
}

.btn-cancel {
    background: var(--bg-light);
    color: var(--text-primary);
}

.app-notification {
    position: fixed;
    top: 90px;
    right: 20px;
    padding: 12px 20px;
    border-radius: 12px;
    color: white;
    font-weight: 600;
    z-index: 9999;
    opacity: 0;
    transform: translateX(120%);
    transition: all 0.3s ease;
}

.app-notification.show {
    opacity: 1;
    transform: translateX(0);
}

.app-notification-success {
    background: linear-gradient(135deg, #16a34a, #22c55e);
}

.app-notification-error {
    background: linear-gradient(135deg, #dc2626, #ef4444);
}

.app-notification-warning {
    background: linear-gradient(135deg, #d97706, #fbbf24);
}

.app-notification-info {
    background: linear-gradient(135deg, #2563eb, #3b82f6);
}

@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
    }

    .applications-list {
        padding: 20px;
    }

    .card-body {
        grid-template-columns: 1fr;
    }
}
</style>
@endsection

@section('content')
<div class="applications-container">
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('hr.admin.dashboard') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <div>
                    <h1 class="header-title">Funeral Grant Applications</h1>
                    <p style="margin: 4px 0 0; opacity: 0.8;">Monitor payouts and approvals across teams</p>
                </div>
            </div>
            <div class="header-right">
                <a href="{{ route('hr.create.application') }}" class="create-btn">
                    <i class="fas fa-plus"></i>
                    New Funeral Grant
                </a>
            </div>
        </div>
    </div>

    <div class="main-content">
        <div class="stats-grid">
            <div class="stat-card">
                <h4>Total</h4>
                <div class="stat-value" id="funeralTotalCount">0</div>
            </div>
            <div class="stat-card">
                <h4>Pending</h4>
                <div class="stat-value" id="funeralPendingCount">0</div>
            </div>
            <div class="stat-card">
                <h4>Approved</h4>
                <div class="stat-value" id="funeralApprovedCount">0</div>
            </div>
            <div class="stat-card">
                <h4>Rejected</h4>
                <div class="stat-value" id="funeralRejectedCount">0</div>
            </div>
        </div>

        <div class="applications-panel">
            <div class="panel-header">
                <i class="fas fa-heartbeat" style="font-size: 22px; color: var(--primary-color);"></i>
                <h2>Recent Applications</h2>
            </div>
            <div class="applications-list" id="applicationsList">
                <div class="loading-state" id="loadingState">
                    <div class="loading-spinner"></div>
                    <p>Loading funeral applications...</p>
                </div>
                <div class="empty-state" id="emptyState" style="display: none;">
                    <i class="fas fa-inbox" style="font-size: 42px;"></i>
                    <p style="margin-top: 12px;">No funeral grant applications available.</p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
const CSRF_TOKEN = document.querySelector('meta[name="csrf-token"]')?.content || '';
const FUNERAL_STORAGE_KEY = 'funeralApplicationsCount';
const FUNERAL_SESSION_KEY = 'funeralApplications';
const FUNERAL_TYPE = 'funeral_grant_form';
const HR_API_BASE_URL = @json(rtrim(env('MONGODB_HR_API_BASE_URL', 'https://api.titandrillingzm.com:6013'), '/'));
const HR_REMOTE_UID_RAW = @json(config('services.hr_module.admin_uid'));
const HR_REMOTE_UID = encodeURIComponent(HR_REMOTE_UID_RAW);

document.addEventListener('DOMContentLoaded', () => {
    loadFuneralApplications();
});

async function fetchFuneralApplicationsData() {
    const endpoints = [
        {
            url: `/api/hr-applications/${FUNERAL_TYPE}`,
            options: { credentials: 'same-origin' }
        },
        {
            url: `${HR_API_BASE_URL}/hr-applications/${FUNERAL_TYPE}/${HR_REMOTE_UID}`,
            options: { credentials: 'omit' }
        }
    ];

    let lastError = null;

    for (const endpoint of endpoints) {
        try {
            const response = await fetch(endpoint.url, endpoint.options);
            if (!response.ok) {
                console.warn(`[funeral] ${endpoint.url} responded with ${response.status}`);
                lastError = new Error(`HTTP ${response.status}`);
                continue;
            }

            const payload = await response.json();
            const data = normalizeApplications(payload);
            if (Array.isArray(data)) {
                console.log(`[funeral] Loaded data from ${endpoint.url}`);
                return data;
            }

            console.warn('[funeral] Unexpected payload shape from', endpoint.url, payload);
            lastError = new Error('Unexpected response shape');
        } catch (error) {
            console.error('[funeral] Fetch error from', endpoint.url, error);
            lastError = error;
        }
    }

    throw lastError || new Error('Unable to load funeral applications');
}

async function loadFuneralApplications() {
    ensureStatePlaceholders();
    const loadingState = document.getElementById('loadingState');
    const emptyState = document.getElementById('emptyState');
    const applicationsList = document.getElementById('applicationsList');

    if (loadingState) loadingState.style.display = 'block';
    if (emptyState) emptyState.style.display = 'none';

    try {
        const applications = await fetchFuneralApplicationsData();

        if (loadingState) loadingState.style.display = 'none';

        if (!applications.length) {
            if (emptyState) emptyState.style.display = 'block';
            updateSummaryCounts([]);
            applicationsList?.querySelectorAll('.application-card').forEach(card => card.remove());
            return;
        }

        if (emptyState) emptyState.style.display = 'none';
        updateSummaryCounts(applications);
        displayApplications(applications);
    } catch (error) {
        console.error('Error loading funeral applications:', error);
        if (loadingState) loadingState.style.display = 'none';
        if (emptyState) emptyState.style.display = 'block';
        showNotification('Failed to load funeral applications. Please try again.', 'error');
    }
}

function ensureStatePlaceholders() {
    const container = document.getElementById('applicationsList');
    if (!container) return;

    if (!document.getElementById('loadingState')) {
        const loading = document.createElement('div');
        loading.id = 'loadingState';
        loading.className = 'loading-state';
        loading.innerHTML = `
            <div class="loading-spinner"></div>
            <p>Loading funeral applications...</p>
        `;
        container.prepend(loading);
    }

    if (!document.getElementById('emptyState')) {
        const empty = document.createElement('div');
        empty.id = 'emptyState';
        empty.className = 'empty-state';
        empty.style.display = 'none';
        empty.innerHTML = `
            <i class="fas fa-inbox" style="font-size: 42px;"></i>
            <p style="margin-top: 12px;">No funeral grant applications available.</p>
        `;
        container.appendChild(empty);
    }
}

function normalizeApplications(payload) {
    if (!payload) return [];
    if (Array.isArray(payload.data)) return payload.data;
    if (Array.isArray(payload)) return payload;
    if (Array.isArray(payload.applications)) return payload.applications;
    if (payload.data && typeof payload.data === 'object') return Object.values(payload.data);
    return [];
}

function updateSummaryCounts(applications) {
    const normalizeStatus = (status) => (status || 'pending').toLowerCase();
    const total = applications.length;
    const pending = applications.filter(app => {
        const status = normalizeStatus(app.status);
        return status === 'pending' || status === '';
    }).length;
    const approved = applications.filter(app => normalizeStatus(app.status) === 'approved').length;
    const rejected = applications.filter(app => normalizeStatus(app.status) === 'rejected').length;

    document.getElementById('funeralTotalCount').textContent = total;
    document.getElementById('funeralPendingCount').textContent = pending;
    document.getElementById('funeralApprovedCount').textContent = approved;
    document.getElementById('funeralRejectedCount').textContent = rejected;

    const counts = {
        total,
        pending,
        approved,
        rejected,
        lastUpdated: new Date().toISOString()
    };

    try {
        localStorage.setItem(FUNERAL_STORAGE_KEY, JSON.stringify(counts));
        if (window.opener && !window.opener.closed) {
            window.opener.postMessage({
                type: 'funeralApplicationsUpdate',
                counts
            }, '*');
        }
    } catch (error) {
        console.warn('Unable to persist funeral counts:', error);
    }
}

function displayApplications(applications) {
    const applicationsList = document.getElementById('applicationsList');
    if (!applicationsList) return;

    applicationsList.querySelectorAll('.application-card').forEach(card => card.remove());

    const mappedApplications = {};
    applications.forEach(app => {
        const key = app.id || app._id || app.applicationId || `funeral_${Math.random().toString(36).slice(2)}`;
        mappedApplications[key] = { ...app };
    });
    sessionStorage.setItem(FUNERAL_SESSION_KEY, JSON.stringify(mappedApplications));

    const sorted = Object.values(mappedApplications).sort((a, b) => {
        const dateA = new Date(a.submittedAt || a.submissionDate || a.createdAt || 0);
        const dateB = new Date(b.submittedAt || b.submissionDate || b.createdAt || 0);
        return dateB - dateA;
    });

    sorted.forEach(application => {
        const card = createApplicationCard(application);
        applicationsList.appendChild(card);
    });

    addCardEventListeners();
}

function createApplicationCard(application) {
    const card = document.createElement('div');
    card.className = 'application-card';
    card.dataset.applicationId = application.id || application._id || application.applicationId || '';

    const status = (application.status || 'Pending').toLowerCase();
    const statusClass = ['approved', 'rejected', 'pending'].includes(status) ? status : 'pending';
    const statusLabel = status[0]?.toUpperCase() + status.slice(1) || 'Pending';

    const relationship = formatRelationship(application.selectedFuneralRelationshipType || application.relationshipType);
    const totalGrant = formatCurrency(calculateTotalGrant(application));
    const coverage = formatDateRange(application.funeralFromDate, application.funeralToDate);
    const dateOfDeath = formatDate(application.deceasedDate);

    card.innerHTML = `
        <div class="card-header">
            <div>
                <h3 class="applicant-name">${application.fullName || application.applicantName || 'N/A'}</h3>
                <p class="employee-no">
                    Employee No: ${application.employeeNo || application.employeeNumber || 'N/A'}
                    • ${application.department || 'No Dept'}
                </p>
            </div>
            <span class="status ${statusClass}">${statusLabel}</span>
        </div>
        <div class="card-body">
            <div class="detail-block">
                <span>Relationship</span>
                <strong>${relationship}</strong>
            </div>
            <div class="detail-block">
                <span>Deceased</span>
                <strong>${application.deceasedName || 'N/A'}</strong>
            </div>
            <div class="detail-block">
                <span>Date of Death</span>
                <strong>${dateOfDeath}</strong>
            </div>
            <div class="detail-block">
                <span>Coverage</span>
                <strong>${coverage}</strong>
            </div>
            <div class="detail-block">
                <span>Total Grant</span>
                <strong>${totalGrant}</strong>
            </div>
        </div>
        <div style="display: flex; justify-content: flex-end;">
            <button class="btn-menu" title="Actions">
                <i class="fas fa-ellipsis-h"></i>
            </button>
        </div>
    `;

    return card;
}

function formatRelationship(value) {
    const map = {
        'spouse_child': 'Spouse & Biological Child',
        'mother_father': 'Mother & Father',
        'both': 'All Immediate Family'
    };
    return map[value] || (value ? value.replace(/_/g, ' ') : 'N/A');
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    if (isNaN(date)) return 'N/A';
    return date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
}

function formatDateRange(from, to) {
    if (!from && !to) return 'N/A';
    if (!from) return `Until ${formatDate(to)}`;
    if (!to) return formatDate(from);
    return `${formatDate(from)} - ${formatDate(to)}`;
}

function calculateTotalGrant(application) {
    const sumRows = (rows = []) => rows.reduce((sum, row) => sum + (Number(row.amount) || 0), 0);
    const spouseTotal = Number(application.spouseChildTotal) || sumRows(application.funeralSpouseChildData || []);
    const motherTotal = Number(application.motherFatherTotal) || sumRows(application.funeralMotherFatherData || []);
    return spouseTotal + motherTotal;
}

function formatCurrency(value) {
    const amount = Number(value) || 0;
    return `K ${amount.toLocaleString('en-ZM', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
}

let currentActionMenu = null;

function addCardEventListeners() {
    document.querySelectorAll('.application-card').forEach(card => {
        card.addEventListener('click', function(e) {
            if (e.target.closest('.btn-menu')) return;
            const applicationId = this.dataset.applicationId;
            window.location.href = `/application-detail?id=${applicationId}&type=${FUNERAL_TYPE}`;
        });
    });

    document.querySelectorAll('.btn-menu').forEach(button => {
        button.addEventListener('click', function(e) {
            e.stopPropagation();
            const card = this.closest('.application-card');
            showActionMenu(this, card.dataset.applicationId);
        });
    });
}

function showActionMenu(button, applicationId) {
    if (currentActionMenu) {
        currentActionMenu.remove();
        currentActionMenu = null;
    }

    const menu = document.createElement('div');
    menu.className = 'action-menu';
    menu.innerHTML = `
        <div class="action-menu-item" data-action="view"><i class="fas fa-eye"></i> View Details</div>
        <div class="action-menu-item" data-action="approve"><i class="fas fa-check"></i> Approve</div>
        <div class="action-menu-item" data-action="reject"><i class="fas fa-times"></i> Reject</div>
        <div class="action-menu-item action-menu-item-danger" data-action="delete"><i class="fas fa-trash"></i> Delete</div>
    `;

    const rect = button.getBoundingClientRect();
    menu.style.top = `${rect.bottom + 6}px`;
    menu.style.left = `${rect.left - 150}px`;

    document.body.appendChild(menu);
    currentActionMenu = menu;

    menu.querySelectorAll('.action-menu-item').forEach(item => {
        item.addEventListener('click', () => {
            handleMenuAction(item.dataset.action, applicationId);
            menu.remove();
            currentActionMenu = null;
        });
    });

    setTimeout(() => {
        document.addEventListener('click', function handleOutsideClick(event) {
            if (menu && !menu.contains(event.target)) {
                menu.remove();
                currentActionMenu = null;
                document.removeEventListener('click', handleOutsideClick);
            }
        });
    }, 10);
}

function handleMenuAction(action, applicationId) {
    switch (action) {
        case 'view':
            window.location.href = `/application-detail?id=${applicationId}&type=${FUNERAL_TYPE}`;
            break;
        case 'approve':
            showConfirmDialog('Approve Application', 'Approve this funeral grant application?', () => {
                updateApplicationStatus(applicationId, 'approved');
            });
            break;
        case 'reject':
            showConfirmDialog('Reject Application', 'Reject this funeral grant application?', () => {
                updateApplicationStatus(applicationId, 'rejected');
            });
            break;
        case 'delete':
            showConfirmDialog('Delete Application', 'This action cannot be undone. Continue?', () => {
                deleteApplication(applicationId);
            }, true);
            break;
    }
}

async function updateApplicationStatus(applicationId, status) {
    try {
        const response = await fetch(`/api/hr-applications/${applicationId}/status`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': CSRF_TOKEN
            },
            body: JSON.stringify({
                status,
                comments: '',
                approverName: 'HR Admin',
                approverRole: 'ProAdmin',
                reason: `${status} via portal`,
                timestamp: new Date().toISOString()
            })
        });

        if (!response.ok) {
            throw new Error('Failed to update status');
        }

        const result = await response.json();
        if (!result.success) {
            throw new Error(result.error || 'Update failed');
        }

        showNotification(`Application ${status}`, 'success');
        loadFuneralApplications();
    } catch (error) {
        console.error('Status update error:', error);
        showNotification('Unable to update status', 'error');
    }
}

async function deleteApplication(applicationId) {
    try {
        const response = await fetch(`/api/hr-applications/${applicationId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': CSRF_TOKEN
            }
        });

        if (!response.ok) {
            throw new Error('Failed to delete application');
        }

        const result = await response.json();
        if (!result.success) {
            throw new Error(result.error || 'Delete failed');
        }

        showNotification('Application deleted', 'success');
        loadFuneralApplications();
    } catch (error) {
        console.error('Delete error:', error);
        showNotification('Unable to delete application', 'error');
    }
}

function showConfirmDialog(title, message, onConfirm, isDanger = false) {
    const dialog = document.createElement('div');
    dialog.className = 'confirm-dialog';
    dialog.innerHTML = `
        <div class="confirm-dialog-content">
            <h3 style="margin-top: 0;">${title}</h3>
            <p style="margin-bottom: 0;">${message}</p>
            <div class="confirm-dialog-buttons">
                <button class="btn btn-cancel">Cancel</button>
                <button class="btn btn-confirm ${isDanger ? 'btn-danger' : ''}">${isDanger ? 'Delete' : 'Confirm'}</button>
            </div>
        </div>
    `;

    document.body.appendChild(dialog);

    dialog.querySelector('.btn-cancel').addEventListener('click', () => dialog.remove());
    dialog.querySelector('.btn-confirm').addEventListener('click', () => {
        onConfirm();
        dialog.remove();
    });

    dialog.addEventListener('click', (event) => {
        if (event.target === dialog) {
            dialog.remove();
        }
    });
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `app-notification app-notification-${type}`;
    notification.textContent = message;
    document.body.appendChild(notification);

    requestAnimationFrame(() => notification.classList.add('show'));

    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => notification.remove(), 300);
    }, 3200);
}
</script>
@endsection