@extends('welcome')

@section('title', 'Users')

@section('content')
<!-- SweetAlert2 -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<div class="users-container" role="main">
    <!-- Header Section - matching Create New Workorder style -->
    <header class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <div>
                    <h1 class="header-title">Users Information</h1>
                    <p class="header-subtitle">Manage and track all users</p>
                </div>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <div class="search-input-wrapper">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search users..." id="userSearchInput" aria-label="Search users" autocomplete="off">
                    </div>
                </div>
                <div class="action-buttons">
                    <button class="btn-primary" type="button" id="addUserBtn" aria-label="Add new user">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <line x1="12" y1="5" x2="12" y2="19"></line>
                            <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                        Add User
                    </button>
                    <button class="btn-secondary" type="button" id="sortUsersBtn" aria-label="Open sorting options">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <path d="M3 6h18M7 12h10M10 18h4"/>
                        </svg>
                        Sorting
                    </button>
                    <button class="btn-success" type="button" id="exportUsersBtn" aria-label="Export users to Excel">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                            <line x1="16" y1="13" x2="8" y2="13"></line>
                            <line x1="16" y1="17" x2="8" y2="17"></line>
                        </svg>
                        Excel Export
                    </button>
                </div>
            </div>
        </div>
    </header>
    
    <!-- Table Section -->
    <section class="table-section" role="region" aria-label="Users table">
        <div class="table-container">
            <table class="users-table" id="usersTable" role="table" aria-label="Users">
                <thead>
                    <tr role="row">
                        <th scope="col">Action</th>
                        <th scope="col">Full Name</th>
                        <th scope="col">Department</th>
                        <th scope="col">NRC / ID Number</th>
                        <th scope="col">Engagement Date</th>
                        <th scope="col">Countries</th>
                        <th scope="col">Projects</th>
                        <th scope="col">Modules</th>
                        <th scope="col">Titan Induction Date</th>
                        <th scope="col">Client Induction Date</th>
                        <th scope="col">Medical Examined</th>
                        <th scope="col">Medical Expiry</th>
                        <th scope="col">Driver's License Number</th>
                        <th scope="col">Driver's License Code</th>
                        <th scope="col">Driver's License Expiry</th>
                        <th scope="col">Client Driving Date</th>
                        <th scope="col">First Aid Date Completed</th>
                        <th scope="col">First Aid Level</th>
                        <th scope="col">First Aid Expiry Date</th>
                    </tr>
                </thead>
                <tbody id="usersTableBody" role="rowgroup">
                    <tr class="loading-row">
                        <td colspan="19" class="loading-cell">
                            <div class="loading-content">
                                <div class="spinner"></div>
                                <h3>Loading Users...</h3>
                                <p>Please wait while we fetch user data from MongoDB.</p>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination Section (Dynamic) -->
        <div class="pagination-section" id="paginationSection" style="display: none;">
            <div class="pagination-info">
                <span class="pagination-text" id="paginationInfo">
                    Showing 0 to 0 of 0 users
                </span>
            </div>
            
            <div class="pagination-controls" id="paginationControls">
                <!-- Pagination controls will be dynamically generated by JavaScript -->
                </div>
            </div>
    </section>
</div>


<style>
/* CSS Variables for consistent theming - matching Create New Workorder */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.users-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles - matching Create New Workorder */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    margin-bottom: 30px;
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-subtitle {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

/* Search and Action Styles */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 16px;
    z-index: 2;
    color: #64748b;
    width: 16px;
    height: 16px;
}

.search-input {
    padding: 12px 16px 12px 44px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    height: 44px;
    width: 300px;
    font-size: 14px;
    color: #374151;
    background: white;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.action-buttons {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-primary, .btn-secondary, .btn-success, .btn-info {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.btn-primary {
    background: white;
    color: var(--primary-color);
}

.btn-primary:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-secondary {
    background: var(--secondary-color);
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-success {
    background: var(--success-color);
    color: white;
}

.btn-success:hover {
    background: #218838;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-info {
    background: var(--info-color);
    color: white;
}

.btn-info:hover {
    background: #138496;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

/* Table Section */
.table-section {
    background: var(--bg-primary);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin: 0 20px 30px 20px;
}

.table-container {
    overflow-x: auto;
}

.users-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.users-table thead {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.users-table th {
    padding: 16px 12px;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e2e8f0;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    white-space: nowrap;
}

.users-table td {
    padding: 16px 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
    white-space: nowrap;
}

/* Form layout classes */
.form-row {
    display: flex;
    flex-wrap: wrap;
    margin: -8px;
}

.form-col-half {
    flex: 0 0 50%;
    padding: 8px;
    box-sizing: border-box;
}

.form-col-full {
    flex: 0 0 100%;
    padding: 8px;
    box-sizing: border-box;
}

.form-col-third {
    flex: 0 0 33.333333%;
    padding: 8px;
    box-sizing: border-box;
}

.form-row-end {
    align-items: flex-end;
}

/* Hidden input class */
.hidden-input {
    display: none;
}


/* Alert classes */
.alert {
    position: relative;
    padding: 0.75rem 1.25rem;
    margin-bottom: 1rem;
    border: 1px solid transparent;
    border-radius: 0.25rem;
}

.alert-success {
    color: #0f5132;
    background-color: #d1e7dd;
    border-color: #badbcc;
}

.alert-danger {
    color: #842029;
    background-color: #f8d7da;
    border-color: #f5c2c7;
}

.alert-warning {
    color: #664d03;
    background-color: #fff3cd;
    border-color: #ffecb5;
}

.alert-info {
    color: #055160;
    background-color: #d1ecf1;
    border-color: #b8daff;
}

.alert-dismissible {
    padding-right: 4rem;
}

.alert-dismissible .btn-close {
    position: absolute;
    top: 0;
    right: 0;
    z-index: 2;
    padding: 1.25rem 1rem;
}

.fade {
    transition: opacity 0.15s linear;
}

.show {
    display: block !important;
}

.position-fixed {
    position: fixed !important;
}
.incident-table thead{background:linear-gradient(135deg,#f8fafc 0%,#f1f5f9 100%)}
.incident-table th{padding:16px 12px;text-align:left;font-weight:600;color:#374151;border-bottom:1px solid #e2e8f0;font-size:13px;text-transform:uppercase;letter-spacing:.05em;white-space:nowrap}
.incident-table td{padding:16px 12px;border-bottom:1px solid #f1f5f9;vertical-align:middle;white-space:nowrap}
.loading-cell{padding:64px 24px !important;text-align:center}
.loading-content{display:flex;flex-direction:column;align-items:center;gap:16px}
.spinner{width:32px;height:32px;border:3px solid #e2e8f0;border-top:3px solid #0055a5;border-radius:50%;animation:spin 1s linear infinite}
@keyframes spin{0%{transform:rotate(0)}100%{transform:rotate(360deg)}}

.loading-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.loading-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}
/* Additional styles for action buttons in table */
.btn-small { padding: 4px 8px; font-size: 12px; margin-right: 4px; border: none; border-radius: 4px; cursor: pointer; }
.btn-small.danger { background: #ef4444; color: white; }

/* User info styling */
.user-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.user-name-text {
    font-weight: 600;
    color: var(--text-primary);
    font-size: 14px;
}

.user-id {
    font-size: 11px;
    color: var(--text-muted);
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
}

/* Action Button */
.dropdown .action-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 12px;
    background: var(--success-color);
    color: white;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.dropdown .action-btn:hover {
    background: var(--success-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
    color: white !important;
}

/* Action Cell */
td.action-cell {
    position: relative;
}

/* Dropdown Container */
.dropdown {
    position: relative;
}

/* Dropdown Menu */
.dropdown-menu {
    position: absolute;
    top: 100%;
    left: 0;
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    border-radius: 8px;
    padding: 8px 0;
    min-width: 200px;
    z-index: 1000;
    list-style: none;
    margin: 5px 0 0 0;
    display: none;
    opacity: 0;
    transform: translateY(-10px);
    transition: all 0.2s ease;
}

.dropdown-menu.show {
    display: block;
    opacity: 1;
    transform: translateY(0);
    background: white;
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    color: var(--text-secondary);
    text-decoration: none;
    font-size: 13px;
    font-weight: 400;
    transition: var(--transition-base);
    border-radius: var(--radius-sm);
}

.dropdown-item:hover {
    background: linear-gradient(135deg, var(--bg-primary) 0%, var(--bg-secondary) 100%);
    color: var(--text-primary);
    transform: translateX(4px);
}

.dropdown-item.danger {
    color: #dc2626;
}

.dropdown-item.danger:hover {
    background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);
    color: #b91c1c;
}

.dropdown-divider {
    height: 1px;
    background: var(--border-color);
    margin: 8px 0;
}

/* No Data Styling */
.no-data-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.no-data-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.no-data-content svg {
    color: var(--text-light);
}

.no-data-content h3 {
    color: var(--text-secondary);
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.no-data-content p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

/* Pagination Styles */
.pagination-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-top: none;
    padding: 20px 24px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 16px;
}

.pagination-info {
    display: flex;
    align-items: center;
}

.pagination-text {
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 400;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 8px;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 12px;
    background: var(--primary-color);
    color: white;
    text-decoration: none;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    transition: var(--transition-base);
    border: none;
    cursor: pointer;
}

.pagination-btn:hover {
    background: var(--primary-hover);
    color: white;
    text-decoration: none;
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.pagination-btn.pagination-disabled {
    background: #e5e7eb;
    color: #9ca3af;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.pagination-btn.pagination-disabled:hover {
    background: #e5e7eb;
    color: #9ca3af;
    transform: none;
    box-shadow: none;
}

.pagination-numbers {
    display: flex;
    align-items: center;
    gap: 4px;
    margin: 0 12px;
}

.pagination-number {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 36px;
    height: 36px;
    padding: 0 8px;
    background: var(--bg-white);
    color: var(--text-secondary);
    text-decoration: none;
    border: 1px solid var(--border-light);
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    transition: var(--transition-base);
}

.pagination-number:hover {
    background: var(--bg-primary);
    color: var(--text-primary);
    border-color: var(--secondary-color);
    text-decoration: none;
}

.pagination-number.pagination-current {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.pagination-number.pagination-current:hover {
    background: var(--primary-hover);
    color: white;
    border-color: var(--primary-hover);
}

.pagination-ellipsis {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 36px;
    height: 36px;
    color: var(--text-muted);
    font-size: 14px;
    font-weight: 500;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
        padding: 0 15px;
    }
    
    .header-left {
        text-align: center;
    }
    
    .header-right {
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .search-input {
        width: 100%;
    }
    
    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .table-section {
        margin: 0 15px 30px 15px;
    }
    
    .pagination-section {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
        padding: 16px;
    }
    
    .pagination-info {
        justify-content: center;
    }
    
    .pagination-controls {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .pagination-numbers {
        margin: 0 8px;
    }
    
    .pagination-number {
        min-width: 32px;
        height: 32px;
        font-size: 13px;
    }
    
    .pagination-btn {
        padding: 6px 10px;
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .header-title {
        font-size: 24px;
    }
    
    .header-subtitle {
        font-size: 14px;
    }
    
    .table-section {
        margin: 0 10px 20px 10px;
    }
    
    .pagination-section {
        padding: 12px;
    }
    
    .pagination-numbers {
        gap: 2px;
        margin: 0 4px;
    }
    
/* SweetAlert2 Custom Styles */
.swal-title-custom {
    font-family: 'Jost', sans-serif !important;
}

.user-view-modal {
    font-family: 'Jost', sans-serif !important;
}

.swal-delete-btn {
    font-family: 'Jost', sans-serif !important;
    font-weight: 600 !important;
    padding: 12px 24px !important;
    border-radius: 8px !important;
    font-size: 15px !important;
}

.swal-cancel-btn {
    font-family: 'Jost', sans-serif !important;
    font-weight: 600 !important;
    padding: 12px 24px !important;
    border-radius: 8px !important;
    font-size: 15px !important;
    }
    
    .pagination-number {
        min-width: 28px;
        height: 28px;
        font-size: 12px;
    }
    
    .pagination-btn {
        padding: 5px 8px;
        font-size: 12px;
    }
    
    .pagination-text {
        font-size: 13px;
    }
}
</style>

<script>
(function(){
    'use strict';
    const CONFIG = {
        selectors: {
            search: '#userSearchInput',
            body: '#usersTableBody',
            addBtn: '#addUserBtn'
        },
        messages: {
            loading: 'Loading users...',
            noData: 'No users found.'
        }
    };
    let allUsers = [];
    let filtered = [];
    let currentPage = 1;
    let totalPages = 0;
    let totalUsers = 0;
    let perPage = 20;

    function el(q) { return document.querySelector(q); }

    function init() {
        const s = el(CONFIG.selectors.search);
        if (s) { s.addEventListener('input', () => filter(s.value)); }

        const addBtn = el(CONFIG.selectors.addBtn);
        if (addBtn) { addBtn.addEventListener('click', () => window.location.href = '{{ route("add-user") }}'); }

        // Load users from MongoDB
        loadUsersFromMongoDB(currentPage);
    }

    // Load users from MongoDB with Pagination
    async function loadUsersFromMongoDB(page = 1) {
        try {
            console.log(`🔄 Loading users from MongoDB (Page ${page})...`);
            
            const response = await fetch(`/api/users/all?page=${page}&limit=${perPage}`);
            const result = await response.json();
            
            console.log('MongoDB API response:', result);
            
            if (result.success && result.data) {
                allUsers = [];
                
                // Process MongoDB user data
                result.data.forEach(userData => {
                        const mappedUser = {
                        id: userData._id || userData.id,
                        fullName: userData.name || 'Unknown User',
                            department: userData.department || '',
                        nrcId: userData.nrcIdNumber || '',
                            engagementDate: userData.engagementDate || '',
                            countries: userData.countries || [],
                            projects: userData.projects || [],
                            modules: userData.modules || [],
                            titanInductionDate: userData.titanInductionDate || '',
                            clientInductionDate: userData.clientInduction || '',
                            medicalExamined: userData.dateExamined || '',
                            medicalExpiry: userData.expiryMedicals || '',
                            dlNumber: userData.driverLicenseNumber || '',
                            dlCode: userData.driverLicenseCode || '',
                            dlExpiry: userData.driverLicenseExpiry || '',
                            clientDrivingDate: userData.clientDrivingDate || '',
                            faDateCompleted: userData.dateCompletedFirstAid || '',
                            faLevel: userData.levelFirstAid || '',
                            faExpiryDate: userData.expiryDateFirstAid || '',
                            email: userData.email || '',
                            mobile: userData.mobile || '',
                        employeeNumber: userData.employeeNumber || '',
                            role: userData.role || '',
                            isBlocked: userData.isBlocked || false,
                            loginStatus: userData.loginStatus || false,
                        password: userData.password || ''
                        };
                        
                        allUsers.push(mappedUser);
                });
                
                // Update pagination state
                if (result.pagination) {
                    currentPage = result.pagination.currentPage;
                    totalPages = result.pagination.totalPages;
                    totalUsers = result.pagination.totalUsers;
                }
                
                console.log(`✅ Loaded ${allUsers.length} users (Page ${currentPage} of ${totalPages})`);
                
                filtered = [...allUsers];
                render();
                renderPagination();
                
                showNotification(`Loaded ${allUsers.length} users from page ${currentPage}`, 'success');
                } else {
                console.log('No users found in MongoDB');
                displayEmptyState();
                hidePagination();
            }
        } catch (error) {
            console.error('❌ Error loading users from MongoDB:', error);
            showNotification('Failed to load users: ' + error.message, 'error');
            displayEmptyState();
            hidePagination();
        }
    }
    
    // Render Pagination Controls
    function renderPagination() {
        const paginationSection = document.getElementById('paginationSection');
        const paginationInfo = document.getElementById('paginationInfo');
        const paginationControls = document.getElementById('paginationControls');
        
        if (!paginationSection || !paginationInfo || !paginationControls) return;
        
        if (totalPages <= 1) {
            paginationSection.style.display = 'none';
            return;
        }
        
        paginationSection.style.display = 'flex';
        
        // Update pagination info
        const startUser = ((currentPage - 1) * perPage) + 1;
        const endUser = Math.min(currentPage * perPage, totalUsers);
        paginationInfo.textContent = `Showing ${startUser} to ${endUser} of ${totalUsers} users`;
        
        // Generate pagination controls
        let paginationHTML = '';
        
        // Previous button
        if (currentPage > 1) {
            paginationHTML += `
                <button class="pagination-btn pagination-prev" onclick="goToPage(${currentPage - 1})">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="15,18 9,12 15,6"></polyline>
                    </svg>
                    Previous
                </button>
            `;
        } else {
            paginationHTML += `
                <span class="pagination-btn pagination-disabled">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="15,18 9,12 15,6"></polyline>
                    </svg>
                    Previous
                </span>
            `;
        }
        
        // Page numbers
        paginationHTML += '<div class="pagination-numbers">';
        
        const startPage = Math.max(1, currentPage - 2);
        const endPage = Math.min(totalPages, currentPage + 2);
        
        if (startPage > 1) {
            paginationHTML += `<button class="pagination-number" onclick="goToPage(1)">1</button>`;
            if (startPage > 2) {
                paginationHTML += `<span class="pagination-ellipsis">...</span>`;
            }
        }
        
        for (let i = startPage; i <= endPage; i++) {
            if (i === currentPage) {
                paginationHTML += `<span class="pagination-number pagination-current">${i}</span>`;
            } else {
                paginationHTML += `<button class="pagination-number" onclick="goToPage(${i})">${i}</button>`;
            }
        }
        
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                paginationHTML += `<span class="pagination-ellipsis">...</span>`;
            }
            paginationHTML += `<button class="pagination-number" onclick="goToPage(${totalPages})">${totalPages}</button>`;
        }
        
        paginationHTML += '</div>';
        
        // Next button
        if (currentPage < totalPages) {
            paginationHTML += `
                <button class="pagination-btn pagination-next" onclick="goToPage(${currentPage + 1})">
                    Next
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="9,18 15,12 9,6"></polyline>
                    </svg>
                </button>
            `;
        } else {
            paginationHTML += `
                <span class="pagination-btn pagination-disabled">
                    Next
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="9,18 15,12 9,6"></polyline>
                    </svg>
                </span>
            `;
        }
        
        paginationControls.innerHTML = paginationHTML;
    }
    
    // Go to specific page
    function goToPage(page) {
        if (page < 1 || page > totalPages || page === currentPage) return;
        loadUsersFromMongoDB(page);
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }
    
    // Expose goToPage to global scope for onclick handlers
    window.goToPage = goToPage;
    
    // Hide pagination
    function hidePagination() {
        const paginationSection = document.getElementById('paginationSection');
        if (paginationSection) {
            paginationSection.style.display = 'none';
        }
    }

    function render() {
        const b = el(CONFIG.selectors.body);
        if (!b) return;
        if (!filtered.length) {
            displayEmptyState();
            return;
        }
        
        b.innerHTML = filtered.map(u => {
            // Format arrays for display
            const formatArray = (arr) => {
                if (!Array.isArray(arr)) return arr || 'N/A';
                return arr.length > 0 ? arr.join(', ') : 'N/A';
            };
            
            // Format modules for MongoDB structure (objects with module and permissions)
            const formatModules = (modules) => {
                if (!modules) return 'N/A';
                if (Array.isArray(modules)) {
                    const moduleNames = [];
                    modules.forEach(module => {
                        if (typeof module === 'object' && module.module) {
                            // MongoDB format: {module: "name", permissions: []}
                            moduleNames.push(module.module);
                        } else if (Array.isArray(module)) {
                            moduleNames.push(...module.filter(m => m && typeof m === 'string'));
                        } else if (typeof module === 'string' && module.trim()) {
                            moduleNames.push(module.trim());
                        }
                    });
                    return moduleNames.length > 0 ? moduleNames.join(', ') : 'N/A';
                }
                return modules;
            };
            
            return `
                <tr class="table-row" role="row">
                    <td class="action-cell" role="cell">
                        <div class="dropdown">
                            <button class="action-btn" type="button" id="actionDropdown${u.id}" aria-expanded="false" aria-label="Actions for user ${u.id}">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                    <circle cx="12" cy="12" r="3"></circle>
                                    <path d="M12 1v6m0 6v6m11-7h-6m-6 0H1"></path>
                                </svg>
                                Actions
                            </button>
                            <ul class="dropdown-menu" aria-labelledby="actionDropdown${u.id}" role="menu">
                                <li role="none">
                                    <a class="dropdown-item" href="#" onclick="event.preventDefault(); viewUser('${u.id}');" role="menuitem" aria-label="View user ${u.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                            <circle cx="12" cy="12" r="3"></circle>
                                        </svg>
                                        View Details
                                    </a>
                                </li>
                                <li role="none">
                                    <a class="dropdown-item" href="#" onclick="event.preventDefault(); editUser('${u.id}');" role="menuitem" aria-label="Edit user ${u.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                                            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                                        </svg>
                                        Edit User
                                    </a>
                                </li>
                                <li class="dropdown-divider"></li>
                                <li role="none">
                                    <a class="dropdown-item danger" href="#" onclick="event.preventDefault(); deleteUser('${u.id}');" role="menuitem" aria-label="Delete user ${u.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <polyline points="3,6 5,6 21,6"></polyline>
                                            <path d="M19,6v14a2,2,0,0,1-2,2H7a2,2,0,0,1-2-2V6M8,6V4a2,2,0,0,1,2-2h4a2,2,0,0,1,2,2V6"></path>
                                        </svg>
                                        Delete User
                                    </a>
                                </li>
                            </ul>
                        </div>
                    </td>
                    <td class="user-name" role="cell">
                        <div class="user-info">
                            <span class="user-name-text">${u.fullName || 'Unknown'}</span>
                            <span class="user-id">${u.id || 'N/A'}</span>
                        </div>
                    </td>
                    <td role="cell">${u.department || ''}</td>
                    <td role="cell">${u.nrcId || ''}</td>
                    <td role="cell">${u.engagementDate || ''}</td>
                    <td role="cell">${formatArray(u.countries)}</td>
                    <td role="cell">${formatArray(u.projects)}</td>
                    <td role="cell">${formatModules(u.modules)}</td>
                    <td role="cell">${u.titanInductionDate || ''}</td>
                    <td role="cell">${u.clientInductionDate || ''}</td>
                    <td role="cell">${u.medicalExamined || ''}</td>
                    <td role="cell">${u.medicalExpiry || ''}</td>
                    <td role="cell">${u.dlNumber || ''}</td>
                    <td role="cell">${u.dlCode || ''}</td>
                    <td role="cell">${u.dlExpiry || ''}</td>
                    <td role="cell">${u.clientDrivingDate || ''}</td>
                    <td role="cell">${u.faDateCompleted || ''}</td>
                    <td role="cell">${u.faLevel || ''}</td>
                    <td role="cell">${u.faExpiryDate || ''}</td>
                </tr>
            `;
        }).join('');
    }

    // Display empty state
    function displayEmptyState() {
        const b = el(CONFIG.selectors.body);
        if (!b) return;
        
        b.innerHTML = `
            <tr class="no-data-row">
                <td colspan="19" class="no-data-cell">
                    <div class="no-data-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"></circle>
                            <line x1="12" y1="8" x2="12" y2="12"></line>
                            <line x1="12" y1="16" x2="12.01" y2="16"></line>
                        </svg>
                        <h3>No Users Found</h3>
                        <p>No users found in the system. Start by adding a new user.</p>
                    </div>
                </td>
            </tr>
        `;
    }

    function filter(term) {
        term = (term || '').toLowerCase();
        filtered = !term ? [...allUsers] : allUsers.filter(u => 
            (u.fullName || '').toLowerCase().includes(term) ||
            (u.department || '').toLowerCase().includes(term) ||
            (u.nrcId || '').toLowerCase().includes(term) ||
            (u.engagementDate || '').toLowerCase().includes(term) ||
            (Array.isArray(u.countries) ? u.countries.join(', ') : (u.countries || '')).toLowerCase().includes(term) ||
            (Array.isArray(u.projects) ? u.projects.join(', ') : (u.projects || '')).toLowerCase().includes(term) ||
            (Array.isArray(u.modules) ? u.modules.join(', ') : (u.modules || '')).toLowerCase().includes(term) ||
            (u.titanInductionDate || '').toLowerCase().includes(term) ||
            (u.clientInductionDate || '').toLowerCase().includes(term) ||
            (u.medicalExamined || '').toLowerCase().includes(term) ||
            (u.medicalExpiry || '').toLowerCase().includes(term) ||
            (u.dlNumber || '').toLowerCase().includes(term) ||
            (u.dlCode || '').toLowerCase().includes(term) ||
            (u.dlExpiry || '').toLowerCase().includes(term) ||
            (u.clientDrivingDate || '').toLowerCase().includes(term) ||
            (u.faDateCompleted || '').toLowerCase().includes(term) ||
            (u.faLevel || '').toLowerCase().includes(term) ||
            (u.faExpiryDate || '').toLowerCase().includes(term)
        );
        render();
    }


    // User action functions
    function editUser(userId) {
        console.log('Editing user:', userId);
        // Redirect to add-user page with edit mode and user ID
        window.location.href = `/add-user?edit=true&userId=${userId}`;
    }

    async function viewUser(userId) {
        try {
        console.log('Viewing user:', userId);
            
            // Show loading with SweetAlert2
            Swal.fire({
                title: 'Loading User Details...',
                html: '<div style="padding: 20px;"><div class="spinner"></div></div>',
                allowOutsideClick: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            // Fetch user details from MongoDB
            const response = await fetch(`/api/users/get/${userId}`);
            const result = await response.json();
            
            if (result.success && result.data) {
                const user = result.data;
                
                // Format modules with permissions
                let modulesHTML = '<div style="max-height: 250px; overflow-y: auto; margin-top: 10px;">';
                if (user.modules && Array.isArray(user.modules) && user.modules.length > 0) {
                    modulesHTML += '<div style="display: grid; gap: 12px;">';
                    user.modules.forEach(mod => {
                        if (mod.module) {
                            const perms = mod.permissions ? mod.permissions.map(p => 
                                `<span style="background: #e0f2fe; color: #0369a1; padding: 4px 10px; border-radius: 12px; font-size: 11px; margin-right: 5px; display: inline-block; margin-bottom: 4px; font-weight: 500;">${p}</span>`
                            ).join('') : '<span style="color: #9ca3af;">No permissions</span>';
                            modulesHTML += `
                                <div style="background: #f8fafc; padding: 12px; border-left: 4px solid #0055a5; border-radius: 6px;">
                                    <div style="font-weight: 600; color: #0055a5; margin-bottom: 8px; font-size: 14px;">${mod.module}</div>
                                    <div style="line-height: 1.8;">${perms}</div>
                                </div>
                            `;
                        }
                    });
                    modulesHTML += '</div>';
                } else {
                    modulesHTML += '<p style="color: #9ca3af; text-align: center; padding: 20px;">No modules assigned</p>';
                }
                modulesHTML += '</div>';
                
                // Format countries
                let countriesHTML = '';
                if (user.countries && Array.isArray(user.countries) && user.countries.length > 0) {
                    countriesHTML = user.countries.map(c => 
                        `<span style="background: linear-gradient(135deg, #0055a5, #0066cc); color: white; padding: 6px 14px; border-radius: 20px; font-size: 13px; margin-right: 6px; display: inline-block; margin-bottom: 6px; font-weight: 500; box-shadow: 0 2px 4px rgba(0,85,165,0.2);">
                            <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 4px;"><rect x="1" y="4" width="22" height="16" rx="2" ry="2"></rect><line x1="1" y1="10" x2="23" y2="10"></line></svg>
                            ${c}
                        </span>`
                    ).join('');
                } else {
                    countriesHTML = '<span style="color: #9ca3af;">No countries assigned</span>';
                }
                
                // Format projects
                let projectsHTML = '';
                if (user.projects && Array.isArray(user.projects) && user.projects.length > 0) {
                    projectsHTML = user.projects.map(p => 
                        `<span style="background: linear-gradient(135deg, #ffd205, #ffdd33); color: #1a202c; padding: 6px 14px; border-radius: 20px; font-size: 13px; margin-right: 6px; display: inline-block; margin-bottom: 6px; font-weight: 500; box-shadow: 0 2px 4px rgba(255,210,5,0.2);">
                            <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 4px;"><rect x="2" y="7" width="20" height="14" rx="2" ry="2"></rect><path d="M16 21V5a2 2 0 0 0-2-2h-4a2 2 0 0 0-2 2v16"></path></svg>
                            ${p}
                        </span>`
                    ).join('');
                } else {
                    projectsHTML = '<span style="color: #9ca3af;">No projects assigned</span>';
                }
                
                // Create modal content
                const modalContent = `
                    <div style="text-align: left; font-family: 'Jost', sans-serif;">
                        <!-- User Avatar & Name -->
                        <div style="text-align: center; padding: 20px; background: linear-gradient(135deg, #0055a5, #0066cc); border-radius: 12px; margin-bottom: 20px; color: white;">
                            <div style="width: 80px; height: 80px; background: white; border-radius: 50%; margin: 0 auto 15px; display: flex; align-items: center; justify-content: center; font-size: 36px; color: #0055a5;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path><circle cx="12" cy="7" r="4"></circle></svg>
                            </div>
                            <h2 style="margin: 0 0 5px 0; font-size: 24px; font-weight: 600;">${user.name || 'Unknown User'}</h2>
                            <p style="margin: 0; font-size: 14px; opacity: 0.9;">${user.email || ''}</p>
                        </div>
                        
                        <!-- Basic Information -->
                        <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 12px; margin-bottom: 20px;">
                            <div style="padding: 12px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
                                <div style="color: #6b7280; font-size: 11px; margin-bottom: 4px; text-transform: uppercase; letter-spacing: 0.5px;">Role</div>
                                <div style="font-weight: 600; color: #0055a5; font-size: 15px;">${user.role || 'N/A'}</div>
                            </div>
                            <div style="padding: 12px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
                                <div style="color: #6b7280; font-size: 11px; margin-bottom: 4px; text-transform: uppercase; letter-spacing: 0.5px;">Employee #</div>
                                <div style="font-weight: 600; color: #1a202c; font-size: 15px;">${user.employeeNumber || 'N/A'}</div>
                            </div>
                            <div style="padding: 12px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
                                <div style="color: #6b7280; font-size: 11px; margin-bottom: 4px; text-transform: uppercase; letter-spacing: 0.5px;">Department</div>
                                <div style="font-weight: 600; color: #1a202c; font-size: 15px;">${user.department || 'N/A'}</div>
                            </div>
                            <div style="padding: 12px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
                                <div style="color: #6b7280; font-size: 11px; margin-bottom: 4px; text-transform: uppercase; letter-spacing: 0.5px;">Mobile</div>
                                <div style="font-weight: 600; color: #1a202c; font-size: 15px;">${user.mobile || 'N/A'}</div>
                            </div>
                            <div style="padding: 12px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0; grid-column: 1 / -1;">
                                <div style="color: #6b7280; font-size: 11px; margin-bottom: 4px; text-transform: uppercase; letter-spacing: 0.5px;">NRC/ID Number</div>
                                <div style="font-weight: 600; color: #1a202c; font-size: 15px;">${user.nrcIdNumber || 'N/A'}</div>
                            </div>
                            <div style="padding: 15px; background: linear-gradient(135deg, #fef3c7, #fde68a); border-radius: 12px; border: 3px solid #fcd34d; grid-column: 1 / -1; text-align: center;">
                                <div style="color: #92400e; font-size: 13px; margin-bottom: 8px; font-weight: 700; text-transform: uppercase; letter-spacing: 1px;">🔐 Login Password</div>
                                <div style="font-weight: 700; color: #78350f; font-size: 20px; font-family: 'Courier New', monospace; background: white; padding: 12px; border-radius: 8px; letter-spacing: 2px;">${user.password || 'Not Set'}</div>
                            </div>
                        </div>
                        
                        <!-- Countries -->
                        <div style="background: white; padding: 18px; border-radius: 12px; margin-bottom: 20px; border: 1px solid #e2e8f0;">
                            <h3 style="color: #0055a5; font-size: 15px; margin: 0 0 12px 0; display: flex; align-items: center; gap: 8px;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="1" y="4" width="22" height="16" rx="2" ry="2"></rect><line x1="1" y1="10" x2="23" y2="10"></line></svg>
                                Assigned Countries
                            </h3>
                            <div style="line-height: 2;">${countriesHTML}</div>
                        </div>
                        
                        <!-- Projects -->
                        <div style="background: white; padding: 18px; border-radius: 12px; margin-bottom: 20px; border: 1px solid #e2e8f0;">
                            <h3 style="color: #0055a5; font-size: 15px; margin: 0 0 12px 0; display: flex; align-items: center; gap: 8px;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="2" y="7" width="20" height="14" rx="2" ry="2"></rect><path d="M16 21V5a2 2 0 0 0-2-2h-4a2 2 0 0 0-2 2v16"></path></svg>
                                Assigned Projects
                            </h3>
                            <div style="line-height: 2;">${projectsHTML}</div>
                        </div>
                        
                        <!-- Modules & Permissions -->
                        <div style="background: white; padding: 18px; border-radius: 12px; border: 1px solid #e2e8f0;">
                            <h3 style="color: #0055a5; font-size: 15px; margin: 0 0 12px 0; display: flex; align-items: center; gap: 8px;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="11" width="18" height="11" rx="2" ry="2"></rect><path d="M7 11V7a5 5 0 0 1 10 0v4"></path></svg>
                                Modules & Permissions
                            </h3>
                            ${modulesHTML}
                        </div>
                    </div>
                `;
                
                // Show beautiful modal with SweetAlert2
                Swal.fire({
                    title: `<div style="display: flex; align-items: center; gap: 10px; justify-content: center;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#0055a5" stroke-width="2"><path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path><circle cx="12" cy="7" r="4"></circle></svg>
                                User Profile
                            </div>`,
                    html: modalContent,
                    width: '900px',
                    showCloseButton: true,
                    confirmButtonText: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 5px;"><polyline points="20 6 9 17 4 12"></polyline></svg> Close',
                    confirmButtonColor: '#0055a5',
                    customClass: {
                        popup: 'user-view-modal',
                        title: 'swal-title-custom'
                    }
                });
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'Failed to load user details',
                    confirmButtonColor: '#0055a5'
                });
            }
        } catch (error) {
            console.error('Error viewing user:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'An error occurred while loading user details',
                confirmButtonColor: '#0055a5'
            });
        }
    }

    function resetPassword(userId) {
        console.log('Resetting password for user:', userId);
        if (confirm('Are you sure you want to reset the password for this user?')) {
            showNotification(`Password reset for user ${userId}`, 'success');
            // TODO: Implement password reset functionality
        }
    }

    async function deleteUser(userId) {
        try {
            console.log('Attempting to delete user:', userId);
            
            // Find user name for confirmation
            const user = allUsers.find(u => u.id === userId);
            const userName = user ? user.fullName : 'this user';
            const userEmail = user ? user.email : '';
            
            // Show beautiful confirmation dialog with SweetAlert2
            const result = await Swal.fire({
                title: '<div style="color: #dc2626; font-weight: 700;">⚠️ Delete User?</div>',
                html: `
                    <div style="text-align: left; font-family: 'Jost', sans-serif; padding: 10px;">
                        <div style="background: #fef2f2; border: 2px solid #fecaca; border-radius: 12px; padding: 20px; margin-bottom: 20px;">
                            <div style="text-align: center;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#dc2626" stroke-width="2" style="margin-bottom: 10px;">
                                    <circle cx="12" cy="12" r="10"></circle>
                                    <line x1="12" y1="8" x2="12" y2="12"></line>
                                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                                </svg>
                                <h3 style="color: #dc2626; margin: 0 0 10px 0; font-size: 18px;">You are about to delete:</h3>
                                <div style="font-size: 20px; font-weight: 700; color: #1a202c; margin-bottom: 5px;">${userName}</div>
                                <div style="color: #6b7280; font-size: 14px;">${userEmail}</div>
                            </div>
                        </div>
                        <div style="background: #fff7ed; border-left: 4px solid #f59e0b; padding: 15px; border-radius: 8px;">
                            <p style="margin: 0; color: #92400e; font-weight: 600; font-size: 14px;">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 5px;"><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path><line x1="12" y1="9" x2="12" y2="13"></line><line x1="12" y1="17" x2="12.01" y2="17"></line></svg>
                                Warning: This action cannot be undone!
                            </p>
                            <ul style="margin: 10px 0 0 25px; padding: 0; color: #78350f; font-size: 13px;">
                                <li>All user data will be permanently removed</li>
                                <li>User will lose access to all modules</li>
                                <li>This cannot be reversed</li>
                            </ul>
                        </div>
                    </div>
                `,
                icon: null,
                showCancelButton: true,
                confirmButtonColor: '#dc2626',
                cancelButtonColor: '#6b7280',
                confirmButtonText: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 5px;"><polyline points="3,6 5,6 21,6"></polyline><path d="M19,6v14a2,2,0,0,1-2,2H7a2,2,0,0,1-2-2V6M8,6V4a2,2,0,0,1,2-2h4a2,2,0,0,1,2,2V6"></path></svg> Yes, Delete User',
                cancelButtonText: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 5px;"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg> Cancel',
                width: '600px',
                customClass: {
                    confirmButton: 'swal-delete-btn',
                    cancelButton: 'swal-cancel-btn'
                },
                focusCancel: true
            });
            
            if (!result.isConfirmed) {
                console.log('Delete cancelled by user');
                return;
            }
            
            // Show deleting progress
            Swal.fire({
                title: 'Deleting User...',
                html: `<div style="padding: 20px;">
                    <div class="spinner"></div>
                    <p style="margin-top: 15px; color: #6b7280;">Please wait while we remove ${userName} from the system...</p>
                </div>`,
                allowOutsideClick: false,
                showConfirmButton: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            // Call MongoDB API to delete user
            const response = await fetch(`/api/users/delete/${userId}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });
            
            const deleteResult = await response.json();
            
            if (deleteResult.success) {
                // Show success message
                await Swal.fire({
                    icon: 'success',
                    title: '<div style="color: #16a34a;">User Deleted Successfully!</div>',
                    html: `<p style="font-size: 16px; color: #1a202c;">${userName} has been removed from the system.</p>`,
                    confirmButtonColor: '#0055a5',
                    confirmButtonText: 'OK',
                    timer: 3000
                });
                
                // Reload current page to refresh the list
                loadUsersFromMongoDB(currentPage);
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Delete Failed',
                    text: deleteResult.message || 'Failed to delete user',
                    confirmButtonColor: '#0055a5'
                });
            }
            
        } catch (error) {
            console.error('Error deleting user:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'An error occurred while deleting the user: ' + error.message,
                confirmButtonColor: '#0055a5'
            });
        }
    }

    // Show notification function
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
        notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
        notification.innerHTML = `
            ${message}
            <button type="button" class="btn-close"></button>
        `;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }

    // Initialize dropdown functionality with event delegation
    function initializeDropdowns() {
        // Use event delegation for dynamically created dropdowns
        document.addEventListener('click', function(e) {
            // Toggle dropdown when action button is clicked
            if (e.target.closest('.action-btn')) {
                e.preventDefault();
                e.stopPropagation();
                
                const button = e.target.closest('.action-btn');
                const dropdown = button.parentElement.querySelector('.dropdown-menu');
                
                // Close all other dropdowns
                document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                    if (menu !== dropdown) {
                        menu.classList.remove('show');
                    }
                });
                
                // Toggle current dropdown
                if (dropdown) {
                    dropdown.classList.toggle('show');
                }
            } else {
                // Close all dropdowns when clicking outside
                if (!e.target.closest('.dropdown-menu')) {
                    document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                        dropdown.classList.remove('show');
                    });
                }
            }
        });

        // Prevent dropdown from closing when clicking inside menu
        document.addEventListener('click', function(e) {
            if (e.target.closest('.dropdown-menu')) {
                // Only close if clicking on a dropdown item
                if (e.target.closest('.dropdown-item')) {
                    setTimeout(() => {
                        document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                            dropdown.classList.remove('show');
                        });
                    }, 100);
                }
            }
        });
    }

    // Make functions globally accessible
    window.editUser = editUser;
    window.viewUser = viewUser;
    window.resetPassword = resetPassword;
    window.deleteUser = deleteUser;
    window.showNotification = showNotification;

    document.addEventListener('DOMContentLoaded', function() {
        init();
        initializeDropdowns();
    });
})();
</script>
@endsection