@extends('welcome')

@section('title', 'Inspections with Job Cards')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="inspections-opened-container">
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Inspections...</div>
        </div>
    </div>
    
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('dashboard.team_kpi.home') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Inspections with Job Cards</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn">
                    <i class="fas fa-file-excel"></i>
                    Export
                </button>
                <button type="button" class="create-btn" id="refresh-btn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="inspections-opened-content">
        <div class="content-box">
            <!-- Date Range Picker -->
            <div class="date-picker-container">
                <div class="date-inputs">
                    <div class="date-input-group">
                        <label>From Date:</label>
                        <input type="date" id="from-date" class="date-input">
                    </div>
                    <div class="date-input-group">
                        <label>To Date:</label>
                        <input type="date" id="to-date" class="date-input">
                    </div>
                    <button class="btn-clear-dates" id="clear-dates">Clear</button>
                </div>
            </div>
            
            <!-- Tabs for With/Without Job Card -->
            <div class="tab-selector">
                <button class="tab-btn active" data-tab="withJobCard" id="with-jobcard-btn">
                    <i class="fas fa-link"></i>
                    With Job Card
                </button>
                <button class="tab-btn" data-tab="withoutJobCard" id="without-jobcard-btn">
                    <i class="fas fa-unlink"></i>
                    Without Job Card
                </button>
            </div>
            
            <!-- Summary Stats -->
            <div class="summary-header">
                <div class="summary-stat">
                    <i class="fas fa-clipboard-list"></i>
                    <div>
                        <div class="stat-label">Total Inspections</div>
                        <div class="stat-value" id="total-inspections">0</div>
                    </div>
                </div>
                <div class="summary-stat">
                    <i class="fas fa-link"></i>
                    <div>
                        <div class="stat-label">With Job Cards</div>
                        <div class="stat-value" id="with-jobcard-count">0</div>
                    </div>
                </div>
                <div class="summary-stat">
                    <i class="fas fa-unlink"></i>
                    <div>
                        <div class="stat-label">Without Job Cards</div>
                        <div class="stat-value" id="without-jobcard-count">0</div>
                    </div>
                </div>
            </div>
            
            <!-- Inspections List -->
            <div id="inspections-list" class="inspections-list">
                <div class="loading-state">
                    <i class="fas fa-spinner fa-spin"></i>
                    <span>Loading inspections...</span>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
// MongoDB Backend API Configuration
const API_BASE_URL = {
    inspections: 'https://api.titandrillingzm.com:6004',
    jobCards: 'https://api.titandrillingzm.com:6005'
};

// Date formatting helper - Day/Month/Year with time
function formatDate(dateValue) {
    if (!dateValue) return 'N/A';
    try {
        const date = dateValue instanceof Date ? dateValue : new Date(dateValue);
        if (isNaN(date.getTime())) return 'N/A';
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        const seconds = String(date.getSeconds()).padStart(2, '0');
        return `${day}/${month}/${year} ${hours}:${minutes}:${seconds}`;
    } catch (e) {
        return 'N/A';
    }
}

// Date formatting helper - Day/Month/Year only
function formatDateOnly(dateValue) {
    if (!dateValue) return 'N/A';
    try {
        const date = dateValue instanceof Date ? dateValue : new Date(dateValue);
        if (isNaN(date.getTime())) return 'N/A';
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        return `${day}/${month}/${year}`;
    } catch (e) {
        return 'N/A';
    }
}

let allInspections = [];
let currentTab = 'withJobCard';
let fromDate = null;
let toDate = null;

function showLoadingState(isLoading) {
    document.getElementById('loading-overlay').style.display = isLoading ? 'flex' : 'none';
}

// Parse date helper for MongoDB dates
function parseDate(dateValue) {
    if (!dateValue) return null;
    if (dateValue instanceof Date) return dateValue;
    if (typeof dateValue === 'string') return new Date(dateValue);
    if (dateValue.$date) return new Date(dateValue.$date);
    return null;
}

async function fetchInspections() {
    showLoadingState(true);
    try {
        // 1. Fetch all inspections from MongoDB
        const inspResponse = await fetch(`${API_BASE_URL.inspections}/inspections/all?limit=10000`);
        if (!inspResponse.ok) throw new Error(`Inspections API error: ${inspResponse.status}`);
        const inspResult = await inspResponse.json();
        const inspections = inspResult.data || (Array.isArray(inspResult) ? inspResult : []);
        
        // 2. Fetch all job cards from MongoDB
        const jobResponse = await fetch(`${API_BASE_URL.jobCards}/jobcards/all?limit=10000`);
        if (!jobResponse.ok) throw new Error(`Job Cards API error: ${jobResponse.status}`);
        const jobResult = await jobResponse.json();
        const jobCards = jobResult.data || (Array.isArray(jobResult) ? jobResult : []);
        
        console.log('Fetched inspections:', inspections.length);
        console.log('Fetched job cards:', jobCards.length);
        
        // 3. Build inspection to job card map
        const inspectionToJobCard = {};
        jobCards.forEach(jobCard => {
            const linkId = jobCard.inspectionId || jobCard.linkedInspectionId || jobCard.inspectionRef || jobCard.inspection_id || 
                          (jobCard.linkInspection && jobCard.linkInspection.id);
            if (linkId) {
                inspectionToJobCard[linkId] = {
                    jobCardNumber: jobCard.jobCardNumber || jobCard.requestNumber || '-',
                    jobCardId: jobCard._id || jobCard.id,
                    pdfDownloadUrl: jobCard.pdfDownloadUrl || null,
                    jobCardStatus: jobCard.status || '-',
                    jobCardStartDate: parseDate(jobCard.startDate),
                    jobCardDueDate: parseDate(jobCard.dueDate || jobCard.due),
                    adminUid: jobCard.adminUid || jobCard.userId || 'unknown'
                };
            }
        });
        
        // 4. Map all inspections (both with and without job cards)
        const inspectionsList = [];
        inspections.forEach(inspection => {
            const inspectionId = inspection._id || inspection.id;
            const jobCard = inspectionToJobCard[inspectionId];
            const createdAt = parseDate(inspection.step3?.createdAt || inspection.step1?.createdAt || inspection.createdAt);
            const equipment = inspection.step1?.selectedEquipment?.equipmentName || 
                            inspection.step1?.selectedEquipment?.equipmentNumber || 
                            inspection.equipmentName || '-';
            
            inspectionsList.push({
                id: inspectionId,
                adminId: inspection.userId || inspection.adminUid || 'unknown',
                date: createdAt,
                equipment: equipment,
                status: inspection.step3?.status || 'Closed',
                inspectionNumber: inspection.step1?.inspectionNumber || inspection.requestNumber || inspectionId,
                jobCardNumber: jobCard ? jobCard.jobCardNumber : 'No Job Card',
                jobCardId: jobCard ? jobCard.jobCardId : null,
                jobCardPdf: jobCard ? jobCard.pdfDownloadUrl : null,
                jobCardStatus: jobCard ? jobCard.jobCardStatus : '-',
                jobCardStartDate: jobCard ? jobCard.jobCardStartDate : null,
                jobCardDueDate: jobCard ? jobCard.jobCardDueDate : null,
                project: inspection.step1?.project || inspection.project || '-',
                country: inspection.step1?.country || inspection.country || '-',
                details: inspection,
            });
        });
        
        // Sort by date descending
        allInspections = inspectionsList.sort((a, b) => {
            const dateA = a.date ? new Date(a.date).getTime() : 0;
            const dateB = b.date ? new Date(b.date).getTime() : 0;
            return dateB - dateA;
        });
        
        console.log('Fetched inspections:', allInspections.length);
        
        renderInspectionsList();
        updateSummary();
        
        showLoadingState(false);
        showNotification(`Loaded ${allInspections.length} inspections`, 'success');
    } catch (error) {
        console.error('Error:', error);
        showNotification('Failed to load data. Please check if backend servers are running.', 'error');
        showLoadingState(false);
        document.getElementById('inspections-list').innerHTML = '<div class="no-data">Error loading data</div>';
    }
}

function filterInspections() {
    // Filter out completed/success/rejected job cards
    let filtered = allInspections.filter(item => {
        // Date filtering
        if (item.date) {
            const date = new Date(item.date);
            
            if (fromDate) {
                const from = new Date(fromDate);
                from.setHours(0, 0, 0, 0);
                const itemDate = new Date(date);
                itemDate.setHours(0, 0, 0, 0);
                if (itemDate < from) return false;
            }
            
            if (toDate) {
                const to = new Date(toDate);
                to.setHours(23, 59, 59, 999);
                if (date > to) return false;
            }
        } else {
            // No date, skip this item
            return false;
        }
        
        // Exclude completed/success/rejected job cards
        const status = (item.jobCardStatus || '').toLowerCase();
        if (status === 'complete' || status === 'completed' || status === 'success' || status === 'rejected') return false;
        
        return true;
    });
    
    // Apply tab filter
    if (currentTab === 'withJobCard') {
        filtered = filtered.filter(item => {
            if (!item.jobCardId) return false;
            
            // If job card due date is present and valid, only show if today or future
            if (item.jobCardDueDate && item.jobCardDueDate !== '-') {
                const due = new Date(item.jobCardDueDate);
                const now = new Date();
                due.setHours(0, 0, 0, 0);
                now.setHours(0, 0, 0, 0);
                if (due < now) return false;
            }
            return true;
        });
    } else {
        filtered = filtered.filter(item => !item.jobCardId);
    }
    
    return filtered;
}

function renderInspectionsList() {
    const container = document.getElementById('inspections-list');
    const inspectionsToShow = filterInspections();
    
    if (inspectionsToShow.length === 0) {
        container.innerHTML = '<div class="no-data">No inspections found for the selected criteria</div>';
        return;
    }
    
    const html = inspectionsToShow.map((item, idx) => `
        <div class="inspection-card">
            <div class="card-header">
                <span class="card-number">#${idx + 1}</span>
                <span class="inspection-number">${item.inspectionNumber || item.id}</span>
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-calendar"></i> Date:</strong> ${item.date ? formatDate(item.date) : '-'}
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-cog"></i> Equipment:</strong> ${item.equipment}
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-project-diagram"></i> Project:</strong> ${item.project}
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-globe"></i> Country:</strong> ${item.country}
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-file-alt"></i> Job Card Number:</strong> ${item.jobCardNumber}
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-info-circle"></i> Job Card Status:</strong> 
                <span class="status-badge" style="background-color: ${getStatusColor(item.jobCardStatus)}; color: white; padding: 4px 10px; border-radius: 12px">${item.jobCardStatus || '-'}</span>
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-play-circle"></i> Start Date:</strong> ${item.jobCardStartDate && item.jobCardStartDate !== '-' ? formatDate(item.jobCardStartDate) : '-'}
            </div>
            <div class="inspection-row">
                <strong><i class="fas fa-flag-checkered"></i> Due Date:</strong> ${item.jobCardDueDate && item.jobCardDueDate !== '-' ? formatDate(item.jobCardDueDate) : '-'}
            </div>
            ${item.jobCardId ? `
                <button class="btn-view-jobcard" onclick="handleViewJobCard('${item.jobCardId}')">
                    <i class="fas fa-eye"></i> View Job Card
                </button>
            ` : ''}
        </div>
    `).join('');
    
    container.innerHTML = html;
}

function getStatusColor(status) {
    if (typeof status !== 'string') return '#888';
    
    const s = status.toLowerCase();
    if (s === 'completed' || s === 'complete' || s === 'success') return '#2ecc40';
    if (s === 'rejected') return '#d81b60';
    if (s === 'processing' || s === 'in progress') return '#ff9800';
    if (s === 'initialized' || s === 'on hold') return '#fbc02d';
    return '#888';
}

function updateSummary() {
    // Base filtered inspections (date range + excluding completed/rejected)
    const baseFiltered = allInspections.filter(item => {
        // Date filtering
        if (item.date) {
            const date = new Date(item.date);
            
            if (fromDate) {
                const from = new Date(fromDate);
                from.setHours(0, 0, 0, 0);
                const itemDate = new Date(date);
                itemDate.setHours(0, 0, 0, 0);
                if (itemDate < from) return false;
            }
            
            if (toDate) {
                const to = new Date(toDate);
                to.setHours(23, 59, 59, 999);
                if (date > to) return false;
            }
        } else {
            return false;
        }
        
        // Exclude completed/success/rejected
        const status = (item.jobCardStatus || '').toLowerCase();
        if (status === 'complete' || status === 'completed' || status === 'success' || status === 'rejected') return false;
        
        return true;
    });
    
    // Count with job card (and due date is today or future)
    const withJobCard = baseFiltered.filter(item => {
        if (!item.jobCardId) return false;
        
        if (item.jobCardDueDate && item.jobCardDueDate !== '-') {
            const due = new Date(item.jobCardDueDate);
            const now = new Date();
            due.setHours(0, 0, 0, 0);
            now.setHours(0, 0, 0, 0);
            if (due < now) return false;
        }
        return true;
    });
    
    // Count without job card
    const withoutJobCard = baseFiltered.filter(item => !item.jobCardId);
    
    const totalFiltered = withJobCard.length + withoutJobCard.length;
    
    document.getElementById('total-inspections').textContent = totalFiltered;
    document.getElementById('with-jobcard-count').textContent = withJobCard.length;
    document.getElementById('without-jobcard-count').textContent = withoutJobCard.length;
}

function handleViewJobCard(jobCardId) {
    if (!jobCardId) return;
    showNotification(`Job Card ID: ${jobCardId}`, 'info');
}

function exportToExcel() {
    const inspectionsToShow = filterInspections();
    
    if (inspectionsToShow.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const excelData = inspectionsToShow.map((item, index) => ({
        'S.No': index + 1,
        'Inspection Number': item.inspectionNumber || item.id,
        'Inspection Date': item.date ? formatDate(item.date) : '-',
        'Equipment': item.equipment,
        'Project': item.project,
        'Country': item.country,
        'Inspection Status': item.status || '-',
        'Job Card Number': item.jobCardNumber,
        'Job Card Status': item.jobCardStatus || '-',
        'Job Card Start Date': item.jobCardStartDate && item.jobCardStartDate !== '-' ? formatDate(item.jobCardStartDate) : '-',
        'Job Card Due Date': item.jobCardDueDate && item.jobCardDueDate !== '-' ? formatDate(item.jobCardDueDate) : '-',
    }));
    
    const ws = XLSX.utils.json_to_sheet(excelData);
    ws['!cols'] = [
        { wch: 6 }, { wch: 20 }, { wch: 20 }, { wch: 25 }, { wch: 20 },
        { wch: 15 }, { wch: 18 }, { wch: 20 }, { wch: 18 }, { wch: 20 }, { wch: 20 }
    ];
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Inspections');
    
    // Add summary sheet
    const withJobCardCount = inspectionsToShow.filter(i => i.jobCardId).length;
    const withoutJobCardCount = inspectionsToShow.filter(i => !i.jobCardId).length;
    
    const summary = [
        { 'Metric': 'Total Inspections', 'Value': inspectionsToShow.length },
        { 'Metric': 'With Job Cards', 'Value': withJobCardCount },
        { 'Metric': 'Without Job Cards', 'Value': withoutJobCardCount },
        { 'Metric': 'Current Tab', 'Value': currentTab === 'withJobCard' ? 'With Job Card' : 'Without Job Card' },
        { 'Metric': 'Date Range From', 'Value': fromDate ? formatDateOnly(fromDate) : 'Not Set' },
        { 'Metric': 'Date Range To', 'Value': toDate ? formatDateOnly(toDate) : 'Not Set' },
        { 'Metric': 'Export Date', 'Value': formatDate(new Date()) }
    ];
    const wsSummary = XLSX.utils.json_to_sheet(summary);
    wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    const timestamp = new Date().toISOString().split('T')[0];
    const tabName = currentTab === 'withJobCard' ? 'WithJobCard' : 'WithoutJobCard';
    XLSX.writeFile(wb, `Inspections_${tabName}_${timestamp}.xlsx`);
    showNotification('Data exported successfully', 'success');
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `<div class="notification-content"><span>${message}</span><button class="notification-close">&times;</button></div>`;
    document.body.appendChild(notification);
    setTimeout(() => notification.remove(), 5000);
    notification.querySelector('.notification-close').addEventListener('click', () => notification.remove());
}

document.addEventListener('DOMContentLoaded', () => {
    // Set default date range (last 7 days)
    const today = new Date();
    today.setHours(23, 59, 59, 999);
    const oneWeekAgo = new Date(today);
    oneWeekAgo.setDate(today.getDate() - 7);
    oneWeekAgo.setHours(0, 0, 0, 0);
    
    fromDate = oneWeekAgo;
    toDate = today;
    
    document.getElementById('from-date').value = oneWeekAgo.toISOString().split('T')[0];
    document.getElementById('to-date').value = new Date().toISOString().split('T')[0];
    
    fetchInspections();
    
    // Tab switching
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const tab = this.getAttribute('data-tab');
            currentTab = tab;
            
            document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            renderInspectionsList();
            updateSummary();
        });
    });
    
    // Date range
    document.getElementById('from-date').addEventListener('change', function() {
        if (this.value) {
            fromDate = new Date(this.value);
            fromDate.setHours(0, 0, 0, 0);
            document.getElementById('to-date').min = this.value;
        } else {
            fromDate = null;
        }
        renderInspectionsList();
        updateSummary();
    });
    
    document.getElementById('to-date').addEventListener('change', function() {
        if (this.value) {
            toDate = new Date(this.value);
            toDate.setHours(23, 59, 59, 999);
        } else {
            toDate = null;
        }
        renderInspectionsList();
        updateSummary();
    });
    
    document.getElementById('clear-dates').addEventListener('click', () => {
        document.getElementById('from-date').value = '';
        document.getElementById('to-date').value = '';
        fromDate = null;
        toDate = null;
        renderInspectionsList();
        updateSummary();
        showNotification('Date range cleared', 'info');
    });
    
    // Refresh
    document.getElementById('refresh-btn').addEventListener('click', () => {
        showNotification('Refreshing data...', 'info');
        fetchInspections();
    });
    
    // Export
    document.getElementById('export-excel-btn').addEventListener('click', exportToExcel);
});
</script>

<style>
@import url('https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap');
:root{--primary-color:#0055a5;--primary-hover:#003f7d;--text-primary:#212529;--text-secondary:#6c757d;--bg-light:#f8fafc;--bg-white:#ffffff;--border-light:#e2e8f0;--shadow-sm:0 .125rem .25rem rgba(0,0,0,.075);--shadow-md:0 .5rem 1rem rgba(0,0,0,.15);--shadow-lg:0 1rem 3rem rgba(0,0,0,.175);--radius-md:.375rem;--radius-lg:.5rem;--transition-base:all .15s ease-in-out;--font-family:'Jost',-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif}
.inspections-opened-container{min-height:100vh;background:var(--bg-light);font-family:var(--font-family)}
.header-gradient{background:#0055a5;padding:20px 0;box-shadow:0 4px 12px rgba(0,85,165,.3)}
.header-content{width:100%;padding:0 20px;display:flex;align-items:center;justify-content:space-between}
.header-left{display:flex;align-items:center;gap:15px}
.back-button{color:white;font-size:20px;text-decoration:none;padding:8px;border-radius:50%;transition:var(--transition-base)}
.back-button:hover{background-color:rgba(255,255,255,.1)}
.header-title{color:white;font-size:28px;font-weight:700;margin:0}
.header-right{display:flex;gap:15px}
.create-btn{background:white;color:var(--primary-color);border:none;padding:12px 24px;border-radius:var(--radius-md);font-size:16px;font-weight:600;cursor:pointer;transition:var(--transition-base);box-shadow:var(--shadow-sm);display:inline-flex;align-items:center;gap:8px}
.create-btn:hover{background:rgba(255,255,255,.9);transform:translateY(-2px);box-shadow:var(--shadow-md)}
.inspections-opened-content{padding:30px 20px}
.content-box{background:white;padding:30px;border-radius:var(--radius-lg);box-shadow:var(--shadow-md);border:1px solid var(--border-light)}
.date-picker-container{margin-bottom:20px;display:flex;justify-content:center}
.date-inputs{display:flex;gap:15px;align-items:end;flex-wrap:wrap;justify-content:center}
.date-input-group{display:flex;flex-direction:column;gap:8px}
.date-input-group label{font-size:14px;font-weight:500;color:var(--text-primary)}
.date-input{padding:12px 16px;border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-family:var(--font-family);min-width:180px}
.date-input:focus{outline:none;border-color:var(--primary-color);box-shadow:0 0 0 3px rgba(0,85,165,.1)}
.btn-clear-dates{padding:12px 24px;background:white;color:var(--text-primary);border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base)}
.btn-clear-dates:hover{background:var(--bg-light);border-color:var(--primary-color);color:var(--primary-color)}
.tab-selector{display:flex;justify-content:center;gap:15px;margin-bottom:25px;background:#F0F4F8;border-radius:24px;padding:4px;max-width:500px;margin-left:auto;margin-right:auto}
.tab-btn{flex:1;padding:12px 24px;background:transparent;border:none;border-radius:20px;font-size:16px;font-weight:600;color:#015185;cursor:pointer;transition:var(--transition-base);display:flex;align-items:center;justify-content:center;gap:8px}
.tab-btn.active{background:#0055a5;color:white;box-shadow:0 2px 4px rgba(0,85,165,.3)}
.tab-btn:hover:not(.active){background:rgba(0,85,165,.1)}
.summary-header{display:grid;grid-template-columns:repeat(auto-fit,minmax(250px,1fr));gap:20px;margin-bottom:30px}
.summary-stat{display:flex;align-items:center;gap:15px;background:linear-gradient(135deg,#f8fafc 0%,#e3f2fd 100%);padding:20px;border-radius:var(--radius-lg);border:1px solid var(--border-light)}
.summary-stat i{font-size:32px;color:var(--primary-color)}
.stat-label{font-size:12px;color:var(--text-secondary);font-weight:500;text-transform:uppercase;margin-bottom:4px}
.stat-value{font-size:28px;color:var(--primary-color);font-weight:700}
.inspections-list{display:grid;gap:16px}
.inspection-card{background:#F5F7FA;border-radius:8px;padding:16px;border:1px solid #E0E0E0;box-shadow:var(--shadow-sm);transition:var(--transition-base)}
.inspection-card:hover{box-shadow:var(--shadow-md);transform:translateY(-2px)}
.card-header{display:flex;justify-content:space-between;align-items:center;margin-bottom:12px;padding-bottom:10px;border-bottom:2px solid #E0E0E0}
.card-number{font-size:16px;font-weight:700;color:#025383;background:white;padding:4px 12px;border-radius:12px}
.inspection-number{font-size:14px;font-weight:600;color:#2267A2}
.inspection-row{margin-bottom:8px;font-size:15px;color:#015185;display:flex;align-items:center;gap:8px}
.inspection-row strong{color:#2267A2;font-weight:700;display:flex;align-items:center;gap:6px}
.inspection-row i{font-size:14px;color:#0055a5}
.status-badge{font-weight:600;font-size:14px;padding:4px 12px;border-radius:12px;background:#f0f0f0}
.btn-view-jobcard{margin-top:12px;padding:8px 16px;background:#015185;color:white;border:none;border-radius:6px;font-weight:700;cursor:pointer;transition:var(--transition-base);display:inline-flex;align-items:center;gap:8px}
.btn-view-jobcard:hover{background:#003f7d;transform:translateY(-2px)}
.loading-state,.no-data{text-align:center;padding:60px;color:var(--text-secondary);font-size:18px}
.loading-state i{margin-right:10px;font-size:24px}
@keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}
.sync-modal{position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.5);display:flex;align-items:center;justify-content:center;z-index:10001}
.sync-content{background:white;border-radius:var(--radius-lg);padding:40px;text-align:center;max-width:300px;box-shadow:var(--shadow-lg)}
.sync-icon{width:40px;height:40px;margin:0 auto 20px;font-size:40px;animation:spin 1s linear infinite}
.sync-text{font-size:18px;font-weight:600;margin-bottom:10px}
.sync-subtext{font-size:14px;color:var(--text-secondary)}
.notification{position:fixed;top:20px;right:20px;z-index:10000;padding:16px;border-radius:8px;box-shadow:0 4px 12px rgba(0,0,0,.15);font-family:var(--font-family);max-width:400px;background:white;animation:slideIn .3s}
.notification-success{border-left:4px solid #28a745}
.notification-error{border-left:4px solid #dc3545}
.notification-info{border-left:4px solid #17a2b8}
.notification-content{display:flex;justify-content:space-between;align-items:center}
.notification-close{background:none;border:none;font-size:18px;color:#999;cursor:pointer}
@keyframes slideIn{from{transform:translateX(100%);opacity:0}to{transform:translateX(0);opacity:1}}
@media (max-width:768px){.header-title{font-size:20px}.summary-header{grid-template-columns:1fr}.header-right{flex-direction:column;width:100%}.create-btn{width:100%;justify-content:center}.date-inputs{flex-direction:column;align-items:stretch}.btn-clear-dates,.date-input{width:100%}.tab-selector{flex-direction:column;max-width:100%}}
@media (max-width:480px){.header-content{flex-direction:column;gap:15px;align-items:flex-start}.header-right{width:100%}.create-btn{width:100%}}
</style>
@endsection
