import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  Alert,
  ScrollView,
  Easing,
  SafeAreaView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { getFilteredTeamMembers } from '../MainHome/Utils/teamMemberUtils';
import JobCardAPI from '../ConfigMongo/JobCardMongoConfig.js';
import { MongoAPI } from '../ConfigMongo/MongoDBConfig';
import OfflineWorkOrderHelper from './OfflineWorkOrderHelper';
import RNFS from 'react-native-fs';

// Import tab components
import AllWotab from './AllWotab';
import AssignedWotab from './AssignedWotab';
import CompleteWotab from './CompleteWotab';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.58;

const WorkOrdertab = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [activeTab, setActiveTab] = useState('All');
  const [userModalVisible, setUserModalVisible] = useState(false);
  const [userDetails, setUserDetails] = useState(null);
  const [userCountries, setUserCountries] = useState([]);
  const [userProjects, setUserProjects] = useState([]);
  const [jobcardPermissions, setJobcardPermissions] = useState([]);
  const [canCreateJobCard, setCanCreateJobCard] = useState(false);
  const [userUid, setUserUid] = useState(null);
  const [refreshKey, setRefreshKey] = useState(0);
  
  // Offline state tracking
  const [isOnline, setIsOnline] = useState(true);
  const [offlineWorkOrdersCount, setOfflineWorkOrdersCount] = useState(0);
  const [onlineWorkOrdersCount, setOnlineWorkOrdersCount] = useState(0);
  const [localStorageSize, setLocalStorageSize] = useState('0 KB');
  const [isSyncing, setIsSyncing] = useState(false);
  const isSyncingOfflineWorkOrders = useRef(false);

  // Animation setup for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  
  // Show toast notification
  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };
  
  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      const online = netInfo.isConnected && netInfo.isInternetReachable !== false;
      setIsOnline(online);
      return online;
    } catch (error) {
      console.error('Error checking network status:', error);
      setIsOnline(false);
      return false;
    }
  };
  
  // Upload PDF to MongoDB
  const uploadPDFToMongoDB = async (pdfPath, adminUid) => {
    try {
      console.log('📁 Uploading PDF to MongoDB...');
      console.log('📁 File path:', pdfPath);
      console.log('👤 Admin UID:', adminUid);
      
      if (!pdfPath || !adminUid) {
        throw new Error('Missing required parameters: pdfPath or adminUid');
      }
      
      // Check if file exists
      const fileExists = await RNFS.exists(pdfPath);
      console.log('📁 File exists:', fileExists);
      
      if (!fileExists) {
        throw new Error('PDF file does not exist at path: ' + pdfPath);
      }
      
      // Get file stats for validation
      const fileStats = await RNFS.stat(pdfPath);
      console.log('📁 File size:', fileStats.size, 'bytes');
      
      if (fileStats.size === 0) {
        throw new Error('PDF file is empty');
      }
      
      // Create FormData for file upload
      const formData = new FormData();
      formData.append('pdf', {
        uri: `file://${pdfPath}`,
        type: 'application/pdf',
        name: `jobcard_${adminUid}_${Date.now()}.pdf`
      });
      
      console.log('📤 Uploading PDF to MongoDB...');
      
      const response = await JobCardAPI.uploadJobCardPdf(adminUid, formData);
      console.log('📊 Upload response:', JSON.stringify(response, null, 2));
      
      if (response && response.success) {
        const downloadUrl = response.downloadUrl || response.fileDownloadUrl || response.url;
        if (!downloadUrl) {
          throw new Error('Upload successful but no download URL returned');
        }
        console.log('✅ PDF uploaded successfully, URL:', downloadUrl);
        return downloadUrl;
      } else {
        const errorMsg = response?.error || response?.message || 'Failed to upload PDF';
        throw new Error(errorMsg);
      }
    } catch (error) {
      console.error('❌ Error uploading PDF to MongoDB:', error);
      console.error('❌ Error details:', error.message);
      throw error;
    }
  };
  
  // Update work order counts (offline vs online)
  const updateWorkOrderCounts = async () => {
    try {
      // Get offline pending sync work orders (created offline, not yet in MongoDB)
      const offlinePendingWorkOrders = await OfflineWorkOrderHelper.getOfflineWorkOrders();
      const pendingCount = offlinePendingWorkOrders.length;
      
      // Get cached work orders from MongoDB (synced and stored offline)
      const cachedWorkOrdersRaw = await AsyncStorage.getItem('globalWorkOrders');
      const cachedWorkOrders = cachedWorkOrdersRaw ? JSON.parse(cachedWorkOrdersRaw) : [];
      const cachedCount = cachedWorkOrders.length;
      
      // TOTAL OFFLINE STORED = Pending + Cached (ALL work orders available offline)
      const totalOfflineCount = pendingCount + cachedCount;
      setOfflineWorkOrdersCount(totalOfflineCount);
      
      // Online in DB count = Just the cached work orders (synced to MongoDB)
      setOnlineWorkOrdersCount(cachedCount);
      
      // Calculate local storage size
      try {
        let totalSize = 0;
        const keys = await AsyncStorage.getAllKeys();
        const workOrderKeys = keys.filter(key => 
          key.includes('workOrder') || 
          key.includes('jobcard') || 
          key.includes('offline')
        );
        
        for (const key of workOrderKeys) {
          const value = await AsyncStorage.getItem(key);
          if (value) {
            totalSize += value.length;
          }
        }
        
        // Convert to KB/MB
        const sizeInKB = (totalSize / 1024).toFixed(2);
        const sizeInMB = (totalSize / (1024 * 1024)).toFixed(2);
        const displaySize = totalSize > 1024 * 1024 ? `${sizeInMB} MB` : `${sizeInKB} KB`;
        setLocalStorageSize(displaySize);
      } catch (storageError) {
        // Silent fail
      }
      
    } catch (error) {
      console.error('❌ Count update error:', error.message);
    }
  };
  
  // Sync offline work orders to MongoDB
  const syncOfflineWorkOrders = async () => {
    if (!(await checkNetworkStatus())) return { success: true, synced: 0, failed: 0 };
    
    // Prevent concurrent syncs
    if (isSyncingOfflineWorkOrders.current) {
      console.log('⚠️ Offline sync already in progress, skipping...');
      return { success: true, synced: 0, failed: 0 };
    }

    try {
      isSyncingOfflineWorkOrders.current = true;
      
      if (!userUid) return { success: false, synced: 0, failed: 0 };
      
      console.log('🔒 Starting offline work orders sync (locked)...');

      // Track the last assigned number to ensure sequential numbering for batch syncs
      let lastAssignedNumber = null;
      
      // Sync offline work orders using OfflineWorkOrderHelper
      const syncResult = await OfflineWorkOrderHelper.syncOfflineWorkOrders(async (workOrder) => {
        try {
          console.log('📤 Syncing offline work order:', workOrder.requestNumber);
          
          // Generate dynamic request number based on CURRENT MongoDB count (refetch each time)
          let newRequestNumber = workOrder.requestNumber;
          
          // CRITICAL: Remove "-OFFLINE" suffix if present
          if (newRequestNumber && newRequestNumber.includes('-OFFLINE')) {
            newRequestNumber = newRequestNumber.replace('-OFFLINE', '');
            console.log('🧹 Removed -OFFLINE suffix from:', workOrder.requestNumber, '→', newRequestNumber);
          }
          
          try {
            console.log('🔢 Generating new request number from LIVE MongoDB count...');
            
            // CRITICAL: Refetch from MongoDB to get the LATEST count including just-synced work orders
            const response = await JobCardAPI.getAllJobCards();
            
            let nextNumber = 1; // Default if no work orders exist
            
            if (response.success && response.data && response.data.length > 0) {
              // Sort by createdAt to get the most recent
              const sortedWorkOrders = response.data.sort((a, b) => 
                new Date(b.createdAt) - new Date(a.createdAt)
              );
              const lastNumber = sortedWorkOrders[0].requestNumber;
              
              console.log('📊 Last request number in MongoDB:', lastNumber);
              
              // Extract number from last work order
              const match = lastNumber.match(/-JC-(\d+)/);
              if (match) {
                nextNumber = parseInt(match[1], 10) + 1;
              }
            } else {
              console.log('ℹ️ No existing work orders in MongoDB, starting from 1');
            }
            
            // If we've assigned a number in this batch, ensure we don't reuse it
            if (lastAssignedNumber !== null && nextNumber <= lastAssignedNumber) {
              nextNumber = lastAssignedNumber + 1;
              console.log('🔄 Adjusted next number to avoid duplicate:', nextNumber);
            }
            
            // Store this number as the last assigned
            lastAssignedNumber = nextNumber;
            
            // Generate clean request number with location codes
            const country = workOrder.location?.country || 'Unknown';
            const project = workOrder.location?.project || 'Unknown';
            const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
            const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
            
            newRequestNumber = `${countryCode}-${projectCode}-JC-${String(nextNumber).padStart(4, '0')}`;
            
            console.log('✅ Generated DYNAMIC request number:', newRequestNumber);
            console.log('   📍 Old (offline):', workOrder.requestNumber);
            console.log('   🆕 New (MongoDB):', newRequestNumber);
            console.log('   🔢 Sequential #:', nextNumber);
            
            // Update work order with new clean request number
            workOrder.requestNumber = newRequestNumber;
            
            // Update title to include new request number
            const titleParts = workOrder.title.split(' - ');
            if (titleParts.length > 1) {
              workOrder.title = `${newRequestNumber} - ${titleParts.slice(1).join(' - ')}`;
            } else {
              workOrder.title = `${newRequestNumber} - ${workOrder.title}`;
            }
            
          } catch (numberError) {
            console.error('❌ Failed to generate dynamic request number:', numberError.message);
            // Use cleaned version (without -OFFLINE) if generation fails
            workOrder.requestNumber = newRequestNumber;
          }
          
          // Check if work order already exists in MongoDB to prevent duplicates
          try {
            const existingWorkOrder = await JobCardAPI.getJobCard(workOrder._id);
            if (existingWorkOrder.success && existingWorkOrder.data) {
              console.log('⚠️ Work order already exists in MongoDB, skipping:', workOrder._id);
              return { success: true, skipped: true, message: 'Work order already synced' };
            }
          } catch (checkError) {
            // Work order doesn't exist, proceed with creation
            console.log('✅ Work order not found in MongoDB, proceeding with creation...');
          }
          
          // UPLOAD PDF TO MONGODB (if local PDF exists) - Multiple strategies
          let pdfDownloadUrl = '';
          let pdfLocalPath = null;
          
          try {
            console.log('🔍 Searching for PDF using multiple strategies...');
            
            // STRATEGY 1: Check if PDF path is stored in work order data itself
            if (workOrder.pdfLocalPath) {
              pdfLocalPath = workOrder.pdfLocalPath;
              console.log('✅ Strategy 1: Found PDF path in work order data:', pdfLocalPath);
            }
            
            // STRATEGY 2: Look for PDF with OLD request number (with -OFFLINE)
            if (!pdfLocalPath) {
              const oldRequestNumber = workOrder.requestNumber.includes('-OFFLINE') 
                ? workOrder.requestNumber 
                : `${workOrder.requestNumber}-OFFLINE`;
              const pdfKey = `workorder_pdf_local_${oldRequestNumber}`;
              pdfLocalPath = await AsyncStorage.getItem(pdfKey);
              if (pdfLocalPath) {
                console.log('✅ Strategy 2: Found PDF with OLD request number key:', pdfKey);
              }
            }
            
            // STRATEGY 3: Look for PDF without -OFFLINE suffix
            if (!pdfLocalPath) {
              const cleanRequestNumber = workOrder.requestNumber.replace('-OFFLINE', '');
              const pdfKey = `workorder_pdf_local_${cleanRequestNumber}`;
              pdfLocalPath = await AsyncStorage.getItem(pdfKey);
              if (pdfLocalPath) {
                console.log('✅ Strategy 3: Found PDF with clean request number key:', pdfKey);
              }
            }
            
            // STRATEGY 4: Look for PDF with work order ID
            if (!pdfLocalPath) {
              const pdfKey = `workorder_pdf_local_${workOrder._id}`;
              pdfLocalPath = await AsyncStorage.getItem(pdfKey);
              if (pdfLocalPath) {
                console.log('✅ Strategy 4: Found PDF with work order ID key:', pdfKey);
              }
            }
            
            // If PDF path found, verify file exists and upload
            if (pdfLocalPath) {
              console.log('📄 Verifying PDF file exists:', pdfLocalPath);
              const fileExists = await RNFS.exists(pdfLocalPath);
              console.log('📄 File exists:', fileExists);
              
              if (fileExists) {
                console.log('📤 Uploading PDF to MongoDB...');
                console.log('   From:', pdfLocalPath);
                console.log('   User:', userUid);
                
                pdfDownloadUrl = await uploadPDFToMongoDB(pdfLocalPath, userUid);
                
                if (pdfDownloadUrl && pdfDownloadUrl.startsWith('http')) {
                  console.log('✅ PDF uploaded successfully!');
                  console.log('   URL:', pdfDownloadUrl);
                  
                  // Update work order with PDF URL
                  workOrder.pdfDownloadUrl = pdfDownloadUrl;
                  
                  // Store PDF with NEW request number for future access
                  const newPdfKey = `workorder_pdf_local_${newRequestNumber}`;
                  await AsyncStorage.setItem(newPdfKey, pdfLocalPath);
                  console.log('✅ Stored PDF with new request number key:', newPdfKey);
                } else {
                  console.error('❌ Invalid PDF URL returned:', pdfDownloadUrl);
                  workOrder.pdfDownloadUrl = '';
                }
              } else {
                console.warn('⚠️ PDF file not found at path:', pdfLocalPath);
                workOrder.pdfDownloadUrl = '';
              }
            } else {
              console.warn('⚠️ No local PDF path found using any strategy');
              console.warn('   Tried:');
              console.warn('   - workOrder.pdfLocalPath');
              console.warn('   - workorder_pdf_local_${requestNumber}-OFFLINE');
              console.warn('   - workorder_pdf_local_${requestNumber}');
              console.warn('   - workorder_pdf_local_${workOrder._id}');
              workOrder.pdfDownloadUrl = '';
            }
          } catch (pdfUploadError) {
            console.error('❌ PDF upload failed during sync:', pdfUploadError.message);
            console.error('❌ PDF upload error stack:', pdfUploadError.stack);
            // Continue without PDF URL - work order will save without it
            workOrder.pdfDownloadUrl = '';
          }
          
          // Create the work order in MongoDB with updated request number and PDF URL
          console.log('💾 Saving work order to MongoDB...');
          console.log('   Request number:', newRequestNumber);
          console.log('   PDF URL:', workOrder.pdfDownloadUrl || 'None');
          
          const createResponse = await JobCardAPI.createJobCard(userUid, workOrder);
          
          if (createResponse.success) {
            console.log('✅ Work order created in MongoDB:', newRequestNumber);
            console.log('   💾 Saved to database successfully');
            console.log('   📄 PDF URL saved:', workOrder.pdfDownloadUrl || 'None');
            return { success: true };
          } else {
            throw new Error('Failed to create work order in MongoDB');
          }
        } catch (error) {
          console.error('❌ Sync error for work order:', error.message);
          return { success: false, error: error.message };
        }
      });
      
      if (syncResult.synced > 0) {
        showToast('success', 'Sync Complete', `${syncResult.synced} offline work orders synced to MongoDB`);
        console.log('📊 Sync Summary:', {
          synced: syncResult.synced,
          failed: syncResult.failed,
          total: syncResult.synced + syncResult.failed
        });
        
        // Refresh the tab content
        setRefreshKey(prev => prev + 1);
        await updateWorkOrderCounts();
      }
      
      if (syncResult.failed > 0) {
        showToast('warning', 'Partial Sync', `${syncResult.failed} work orders failed to sync`);
      }
      
      console.log('🔓 Offline sync completed (unlocked)');
      return syncResult;
    } catch (error) {
      console.error('❌ Sync offline error:', error.message);
      return { success: false, synced: 0, failed: 0 };
    } finally {
      isSyncingOfflineWorkOrders.current = false; // Always unlock
    }
  };
  
  // Cloud sync button handler
  const handleCloudSync = async () => {
    if (isSyncing) return;
    
    try {
      setIsSyncing(true);
      setSyncModalVisible(true);
      
      // Check if online
      const onlineStatus = await checkNetworkStatus();
      
      if (!onlineStatus) {
        showToast('error', 'Offline', 'Cannot sync while offline');
        setIsSyncing(false);
        setSyncModalVisible(false);
        return;
      }

      if (!userUid) {
        showToast('error', 'Error', 'No user is logged in');
        setIsSyncing(false);
        setSyncModalVisible(false);
        return;
      }

      console.log('🔄 Starting cloud sync for work orders...');
      
      // Sync offline work orders
      const offlineSyncResult = await syncOfflineWorkOrders();
      
      if (offlineSyncResult && offlineSyncResult.synced > 0) {
        console.log(`✅ Synced ${offlineSyncResult.synced} offline work orders to MongoDB`);
      }
      
      // Reload fresh work orders from MongoDB
      setRefreshKey(prev => prev + 1);
      
      showToast('success', 'Synced', `Work orders synced successfully`);
      
    } catch (error) {
      console.error('❌ Error during cloud sync:', error);
      showToast('error', 'Sync Failed', 'Could not sync from MongoDB');
    } finally {
      setIsSyncing(false);
      setSyncModalVisible(false);
    }
  };

  // Fetch user data and permissions on mount
  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        
        setUserUid(userUid);
        
        // Check network status first
        const online = await checkNetworkStatus();
        
        if (online) {
          // Fetch user data from MongoDB
          const response = await MongoAPI.getUserById(userUid);
          if (response.success && response.data) {
            const userData = response.data;
            setUserCountries(Array.isArray(userData.countries) ? userData.countries : []);
            setUserProjects(Array.isArray(userData.projects) ? userData.projects : []);
            const permissions = JobCardAPI.getUserJobCardPermissions(userData);
            setJobcardPermissions(permissions);
            setCanCreateJobCard(JobCardAPI.canCreateJobCard(permissions));
            
            // Cache user data for offline access
            await AsyncStorage.setItem('workOrderModuleUserInfo', JSON.stringify({
              countries: userData.countries || [],
              projects: userData.projects || [],
              jobcardPermissions: permissions,
              canCreate: JobCardAPI.canCreateJobCard(permissions),
            }));
          }
        } else {
          // Load from cache if offline
          const cachedUserInfo = await AsyncStorage.getItem('workOrderModuleUserInfo');
          if (cachedUserInfo) {
            const userInfo = JSON.parse(cachedUserInfo);
            setUserCountries(userInfo.countries || []);
            setUserProjects(userInfo.projects || []);
            setJobcardPermissions(userInfo.jobcardPermissions || []);
            setCanCreateJobCard(userInfo.canCreate || false);
            console.log('✅ Loaded user info from cache (offline)');
          }
        }
        
        // Update counts
        await updateWorkOrderCounts();
      } catch (err) {
        console.error('Error fetching user data:', err);
      }
    };
    fetchUserData();
  }, []);
  
  // Network status monitoring
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      const currentOnlineStatus = state.isConnected && state.isInternetReachable !== false;
      const wasOffline = !isOnline;
      const isNowOnline = currentOnlineStatus;
      
      setIsOnline(currentOnlineStatus);
      
      // Show toast notifications for status changes
      if (wasOffline && isNowOnline) {
        console.log('🌐 Connection restored');
        showToast('success', 'Online', 'Connection restored - Use cloud sync to upload work orders');
        
        // Reload data from MongoDB
        setTimeout(() => {
          setRefreshKey(prev => prev + 1);
        }, 1000);
      } else if (!wasOffline && !isNowOnline) {
        console.log('📴 Connection lost - Switching to offline mode');
        showToast('info', 'Offline', 'No internet connection - Using cached data');
      }
    });

    return () => {
      unsubscribe();
    };
  }, [isOnline]);

  // Refresh data when screen is focused (increment refreshKey)
  useFocusEffect(
    React.useCallback(() => {
      setRefreshKey((prev) => prev + 1);
      updateWorkOrderCounts(); // Update counts on screen focus
    }, [])
  );

  const handleSearch = (text) => {
    setSearchQuery(text);
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
        easing: Easing.linear,
      })
    ).start();

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setRefreshKey(prev => prev + 1);
      updateWorkOrderCounts();
    }, 2000);
  };

  const handleTabPress = (tabName) => {
    setActiveTab(tabName);
  };

  const handleCreateJobCard = () => {
    if (navigation && navigation.navigate) {
      navigation.navigate('CreateWorkOrder');
    } else {
      Alert.alert('Navigation error', 'Cannot open Create Job Card screen.');
    }
  };

  // Handler for lock icon click - show permissions
  const handleLockIconPress = async () => {
    try {
      // Get user UID from AsyncStorage
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        return;
      }

      // Fetch user data from MongoDB
      const response = await MongoAPI.getUserById(userUid);
      if (!response.success || !response.data) {
        Alert.alert('Error', 'User data not found in database.');
        return;
      }

      const userData = response.data;

      // Extract jobcard module permissions
      const jobcardPermissions = JobCardAPI.getUserJobCardPermissions(userData);

      // Prepare user details for modal
      const details = {
        name: userData.name || 'N/A',
        email: userData.email || 'N/A',
        employeeNumber: userData.employeeNumber || 'N/A',
        countries: Array.isArray(userData.countries)
          ? userData.countries.join(', ')
          : 'None',
        projects: Array.isArray(userData.projects)
          ? userData.projects.join(', ')
          : 'None',
        jobcardPermissions: Array.isArray(jobcardPermissions)
          ? jobcardPermissions.join(', ')
          : 'None',
      };

      setUserDetails(details);
      setUserModalVisible(true); // Show modal
    } catch (err) {
      console.error('Error fetching user details:', err);
      Alert.alert('Error', err.message || 'Failed to fetch user details.');
    }
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'All':
        return (
          <AllWotab
            refreshKey={refreshKey}
            searchQuery={searchQuery}
            onRefresh={handleRefresh}
            userCountries={userCountries}
            userProjects={userProjects}
            jobcardPermissions={jobcardPermissions}
            userUid={userUid}
          />
        );
      case 'Assigned':
        return <AssignedWotab />;
      case 'Completed':
        return <CompleteWotab />;
      default:
        return (
          <AllWotab
            searchQuery={searchQuery}
            onRefresh={handleRefresh}
            userCountries={userCountries}
            userProjects={userProjects}
          />
        );
    }
  };

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <SafeAreaView style={styles.safeAreaHeader}>
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <TouchableOpacity 
                style={styles.backButton}
                onPress={() => navigation.goBack()}
              >
                <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>Job Card</Text>
            </View>
            <View style={styles.headerRight}>
              {!isOnline && (
                <View style={styles.offlineIndicator}>
                  <MaterialIcons name="wifi-off" size={width * 0.05} color="#ffd205" />
                </View>
              )}
              <TouchableOpacity 
                style={[styles.refreshButton, isSyncing && styles.syncingButton]} 
                onPress={handleCloudSync}
                disabled={isSyncing}
              >
                <View style={styles.iconBackground}>
                  <Image
                    source={require('../../Images/adminhome/equp/cloud_1.png')}
                    style={[styles.refreshIcon, isSyncing && styles.syncingIcon]}
                  />
                </View>
              </TouchableOpacity>
              <TouchableOpacity style={styles.lockButton} onPress={handleLockIconPress}>
                <View style={styles.iconBackground}>
                  <MaterialIcons 
                    name="lock" 
                    size={width * 0.055} 
                    color="#fff" 
                  />
                </View>
              </TouchableOpacity>
            </View>
          </View>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <MaterialIcons name="search" size={width * 0.05} color="#0055A5" style={styles.searchIconLeft} />
            <TextInput
              style={styles.searchInput}
              placeholder="Search job cards..."
              placeholderTextColor="#999"
              value={searchQuery}
              onChangeText={handleSearch}
            />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>

          {/* Tabs */}
          <View style={styles.tabContainer}>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'All' && styles.activeTab]}
              onPress={() => handleTabPress('All')}
            >
              <Text style={[styles.tabText, activeTab === 'All' && styles.activeTabText]}>All</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'Assigned' && styles.activeTab]}
              onPress={() => handleTabPress('Assigned')}
            >
              <Text style={[styles.tabText, activeTab === 'Assigned' && styles.activeTabText]}>
                Assigned
              </Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'Completed' && styles.activeTab]}
              onPress={() => handleTabPress('Completed')}
            >
              <Text style={[styles.tabText, activeTab === 'Completed' && styles.activeTabText]}>
                Completed
              </Text>
            </TouchableOpacity>
          </View>
        </SafeAreaView>
      </LinearGradient>

      {/* Main Content */}
      <View style={styles.mainContent}>{renderTabContent()}</View>

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient 
          colors={['#0055A5', '#003875']} 
          start={{ x: 0, y: 0 }}
          end={{ x: 1, y: 0 }}
          style={styles.buttonGradient}
        >
          <TouchableOpacity
            style={[styles.startButton, !canCreateJobCard && { opacity: 0.5 }]}
            onPress={handleCreateJobCard}
            disabled={!canCreateJobCard}
          >
            <Ionicons name="add-circle" size={width * 0.06} color="#ffd205" />
            <Text style={styles.buttonText}>Create Job Card</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* User Details Modal */}
      <Modal
        visible={userModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setUserModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.userModalContent}>
            <Text style={styles.modalTitle}>System Information</Text>
            
            {/* Status Counter Widgets in Modal */}
            <View style={styles.modalStatusCounterContainer}>
              {/* Offline Stored Widget */}
              <View style={[styles.modalStatusCounterCard, styles.modalOfflineCard]}>
                <MaterialIcons name="wifi-off" size={width * 0.06} color="#ffd205" />
                <Text style={styles.modalStatusCounterLabel}>Offline</Text>
                <Text style={styles.modalStatusCounterValue}>{offlineWorkOrdersCount}</Text>
              </View>
              
              {/* Online in DB Widget */}
              <View style={[styles.modalStatusCounterCard, styles.modalOnlineCard]}>
                <MaterialIcons name="cloud-done" size={width * 0.06} color="#4CAF50" />
                <Text style={styles.modalStatusCounterLabel}>Online</Text>
                <Text style={styles.modalStatusCounterValue}>{onlineWorkOrdersCount}</Text>
              </View>
              
              {/* Storage Widget */}
              <View style={[styles.modalStatusCounterCard, styles.modalStorageCard]}>
                <MaterialIcons name="storage" size={width * 0.06} color="#00BCD4" />
                <Text style={styles.modalStatusCounterLabel}>Storage</Text>
                <Text style={[styles.modalStatusCounterValue, { fontSize: width * 0.032 }]}>{localStorageSize}</Text>
              </View>
            </View>

            {userDetails ? (
              <ScrollView style={styles.modalScroll}>
                <Text style={styles.modalSectionTitle}>User Details</Text>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Name:</Text>
                  <Text style={styles.userInfoValue}>{userDetails.name}</Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Email:</Text>
                  <Text style={styles.userInfoValue}>{userDetails.email}</Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Countries:</Text>
                  <Text style={styles.userInfoValue}>
                    {userDetails.countries && userDetails.countries !== 'None' 
                      ? userDetails.countries 
                      : 'N/A'}
                  </Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Projects:</Text>
                  <Text style={styles.userInfoValue}>
                    {userDetails.projects && userDetails.projects !== 'None' 
                      ? userDetails.projects 
                      : 'N/A'}
                  </Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Job Card Permissions:</Text>
                  <Text style={styles.userInfoValue}>
                    {userDetails.jobcardPermissions && userDetails.jobcardPermissions !== 'None' 
                      ? userDetails.jobcardPermissions 
                      : 'None'}
                  </Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Employee Number:</Text>
                  <Text style={styles.userInfoValue}>{userDetails.employeeNumber || 'N/A'}</Text>
                </View>
              </ScrollView>
            ) : (
              <Text style={styles.noDataText}>No user data available.</Text>
            )}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setUserModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#F5F7FA',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  safeAreaHeader: {
    flex: 1,
    width: '100%',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: width * 0.05,
    marginTop: width * 0.03,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.055,
    fontWeight: 'bold',
    marginLeft: width * 0.03,
    letterSpacing: 0.5,
    textShadowColor: 'rgba(0, 0, 0, 0.3)',
    textShadowOffset: { width: 0, height: 2 },
    textShadowRadius: 4,
  },
  backButton: {
    padding: width * 0.01,
  },
  refreshButton: {
    marginLeft: width * 0.02,
  },
  refreshIcon: {
    width: width * 0.05,
    height: width * 0.05,
    resizeMode: 'contain',
  },
  lockButton: {
    marginLeft: width * 0.02,
  },
  iconBackground: {
    backgroundColor: 'rgba(255, 255, 255, 0.15)',
    borderRadius: width * 0.05,
    padding: width * 0.02,
    justifyContent: 'center',
    alignItems: 'center',
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.03,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.03,
    paddingHorizontal: width * 0.04,
    marginRight: width * 0.025,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 3,
  },
  searchInput: {
    flex: 1,
    height: width * 0.11,
    color: '#333',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginLeft: width * 0.02,
  },
  searchIconLeft: {
    marginRight: width * 0.01,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    padding: width * 0.028,
    borderRadius: width * 0.02,
  },
  tabContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: width * 0.04,
    paddingTop: width * 0.03,
    borderTopWidth: 1,
    borderTopColor: 'rgba(255,255,255,0.15)',
  },
  tabButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: width * 0.035,
    marginHorizontal: width * 0.01,
  },
  activeTab: {
    borderBottomWidth: 4,
    borderBottomColor: '#ffd205',
    shadowColor: '#ffd205',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.5,
    shadowRadius: 4,
  },
  tabText: {
    color: 'rgba(255,255,255,0.6)',
    fontSize: width * 0.04,
    fontWeight: '700',
    letterSpacing: 0.5,
  },
  activeTabText: {
    color: '#fff',
    fontSize: width * 0.042,
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  userModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '90%',
    maxHeight: '85%',
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#0055A5',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  modalSectionTitle: {
    fontSize: width * 0.042,
    fontWeight: 'bold',
    color: '#0055A5',
    marginBottom: width * 0.025,
    marginTop: width * 0.02,
    paddingBottom: width * 0.015,
    borderBottomWidth: 2,
    borderBottomColor: '#0055A5',
  },
  modalScroll: {
    maxHeight: width * 0.6,
    width: '100%',
  },
  modalStatusCounterContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: width * 0.04,
    gap: width * 0.02,
  },
  modalStatusCounterCard: {
    flex: 1,
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.04,
    paddingHorizontal: width * 0.02,
    borderRadius: width * 0.025,
    minHeight: width * 0.22,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 3,
    elevation: 3,
  },
  modalOfflineCard: {
    backgroundColor: '#FFF9E6',
    borderWidth: 2,
    borderColor: '#ffd205',
  },
  modalOnlineCard: {
    backgroundColor: '#E8F5E9',
    borderWidth: 2,
    borderColor: '#4CAF50',
  },
  modalStorageCard: {
    backgroundColor: '#E0F7FA',
    borderWidth: 2,
    borderColor: '#00BCD4',
  },
  modalStatusCounterLabel: {
    fontSize: width * 0.032,
    color: '#333',
    fontWeight: '700',
    marginTop: width * 0.015,
    marginBottom: width * 0.008,
    textAlign: 'center',
    textTransform: 'uppercase',
    letterSpacing: 0.5,
  },
  modalStatusCounterValue: {
    fontSize: width * 0.055,
    fontWeight: 'bold',
    color: '#0055A5',
    textAlign: 'center',
  },
  detailRow: {
    flexDirection: 'row',
    marginBottom: 10,
    flexWrap: 'wrap',
  },
  detailLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    width: '40%',
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#333',
    width: '60%',
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    marginBottom: 15,
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.015,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
    width: '60%',
  },
  offlineIndicator: {
    marginRight: width * 0.02,
    padding: width * 0.01,
  },
  syncingButton: {
    opacity: 0.6,
  },
  syncingIcon: {
    opacity: 0.6,
  },
});

export default WorkOrdertab;