import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  Animated,
  Modal,
  Image,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const AllWotab = ({ searchQuery, onRefresh }) => {
  const navigation = useNavigation();
  const [workOrders, setWorkOrders] = useState([]);
  const [filteredWorkOrders, setFilteredWorkOrders] = useState([]);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(null);

  // Animation for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Animation for dropdown
  const fadeAnim = useRef(new Animated.Value(0)).current;

  // Show toast notifications
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Load work orders from Firebase
  const loadWorkOrders = async () => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const workOrdersRef = database().ref(`/workOrders/${adminUid}`);
      workOrdersRef.on('value', (snapshot) => {
        const data = snapshot.val();
        let workOrderList = [];
        if (data) {
          workOrderList = Object.keys(data)
            .filter((key) => key !== 'lastRequestNumber')
            .map((key) => ({
              id: key,
              ...data[key],
            }));
          workOrderList.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        }
        setWorkOrders(workOrderList);
        handleSearch(searchQuery, workOrderList);
      });

      return () => workOrdersRef.off('value');
    } catch (error) {
      console.error('Error loading work orders from Firebase:', error);
      showToast('error', 'Error', 'Failed to load work orders.');
    }
  };

  useEffect(() => {
    loadWorkOrders();
  }, []);

  // Handle search
  const handleSearch = (query, workOrdersList = workOrders) => {
    if (query) {
      const filtered = workOrdersList.filter(
        (wo) =>
          (wo.title || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.equipment?.equipmentName || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.priority || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.assignTo?.some((user) => user.name.toLowerCase().includes(query.toLowerCase())) || false) ||
          (wo.workOrderType || '').toLowerCase().includes(query.toLowerCase())
      );
      setFilteredWorkOrders(filtered);
    } else {
      setFilteredWorkOrders(workOrdersList);
    }
  };

  useEffect(() => {
    handleSearch(searchQuery);
  }, [searchQuery]);

  // Handle refresh
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    loadWorkOrders();

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      if (onRefresh) onRefresh();
    }, 2000);
  };

  // Delete work order from Firebase
  const deleteWorkOrder = async (workOrderId) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const workOrderRef = database().ref(`/workOrders/${adminUid}/${workOrderId}`);
      await workOrderRef.remove();

      const updatedWorkOrders = workOrders.filter((wo) => wo.id !== workOrderId);
      setWorkOrders(updatedWorkOrders);
      handleSearch(searchQuery, updatedWorkOrders);
      setDropdownVisible(null);
      showToast('success', 'Success', 'Work Order deleted successfully!');
    } catch (error) {
      console.error('Error deleting work order from Firebase:', error);
      showToast('error', 'Error', 'Failed to delete work order.');
    }
  };

  // View PDF
  const handleViewPDF = async (workOrderId) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const workOrderRef = database().ref(`/workOrders/${adminUid}/${workOrderId}`);
      const snapshot = await workOrderRef.once('value');
      const workOrderData = snapshot.val();

      if (!workOrderData) {
        showToast('error', 'Error', 'Work Order not found in the database.');
        return;
      }

      const pdfUrl = workOrderData.pdfDownloadUrl;
      if (!pdfUrl) {
        showToast('error', 'Error', 'PDF URL not available for this Work Order.');
        return;
      }

      setDropdownVisible(null);
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => {
          if (onRefresh) onRefresh();
          navigation.goBack();
        },
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      showToast('error', 'Error', 'Could not fetch the PDF URL. Please try again.');
    }
  };

  // Get priority color
  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'Low':
        return '#2ecc71';
      case 'Medium':
        return '#f1c40f';
      case 'High':
        return '#e67e22';
      case 'Emergency':
        return '#e74c3c';
      default:
        return '#666';
    }
  };

  // Format date
  const formatDate = (date) => {
    if (!date) return 'N/A';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Toggle dropdown with animation
  const toggleDropdown = (id) => {
    if (dropdownVisible === id) {
      Animated.timing(fadeAnim, {
        toValue: 0,
        duration: 200,
        useNativeDriver: true,
      }).start(() => setDropdownVisible(null));
    } else {
      setDropdownVisible(id);
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 200,
        useNativeDriver: true,
      }).start();
    }
  };

  // Render work order item
  const renderWorkOrder = ({ item }) => (
    <View style={styles.workOrderRow}>
      {dropdownVisible === item.id && (
        <TouchableOpacity
          style={styles.dropdownOverlay}
          activeOpacity={1} // Prevent opacity change on press
          onPress={() => setDropdownVisible(null)} // Close dropdown on overlay press
        />
      )}
      <TouchableOpacity
        style={styles.workOrderContent}
        onPress={() => {
          // Optionally navigate to a Work Order details screen
          // navigation.navigate('WorkOrderDetails', { workOrder: item });
        }}
      >
        <View style={styles.workOrderHeader}>
          <Text style={styles.workOrderTitle} numberOfLines={1} ellipsizeMode="tail">
            {item.title || 'Untitled Work Order'}
          </Text>
          <View style={[styles.priorityBadge, { backgroundColor: getPriorityColor(item.priority) }]}>
            <Text style={styles.priorityText}>{item.priority || 'N/A'}</Text>
          </View>
        </View>
        <View style={styles.workOrderInfo}>
          <View style={styles.detailRow}>
            <MaterialIcons name="build" size={18} color="#015185" />
            <Text style={styles.detailLabel}>Equipment:</Text>
            <Text style={styles.detailValue} numberOfLines={1} ellipsizeMode="tail">
              {item.equipment?.equipmentName || 'No Equipment'}
            </Text>
          </View>
          <View style={styles.detailRow}>
            <MaterialIcons name="category" size={18} color="#015185" />
            <Text style={styles.detailLabel}>Type:</Text>
            <Text style={styles.detailValue}>{item.workOrderType || 'N/A'}</Text>
          </View>
          <View style={styles.detailRow}>
            <MaterialIcons name="calendar-today" size={18} color="#015185" />
            <Text style={styles.detailLabel}>Start Date:</Text>
            <Text style={styles.detailValue}>{formatDate(item.startDate)}</Text>
          </View>
          <View style={styles.detailRow}>
            <MaterialIcons name="event" size={18} color="#015185" />
            <Text style={styles.detailLabel}>Due Date:</Text>
            <Text style={styles.detailValue}>{formatDate(item.dueDate)}</Text>
          </View>
          <View style={styles.detailRow}>
            <MaterialIcons name="people" size={18} color="#015185" />
            <Text style={styles.detailLabel}>Assigned To:</Text>
            <Text style={styles.detailValue} numberOfLines={1} ellipsizeMode="tail">
              {item.assignTo?.map((user) => user.name).join(', ') || 'Unassigned'}
            </Text>
            {/* Three-dot button */}
            <View style={styles.threeDotsContainer}>
              <TouchableOpacity
                style={styles.threeDotsButton}
                onPress={() => toggleDropdown(item.id)}
              >
                <MaterialIcons name="more-vert" size={20} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>
        </View>
        {/* Dropdown menu */}
        {dropdownVisible === item.id && (
          <Animated.View style={[styles.dropdownMenu, { opacity: fadeAnim }]}>
            <TouchableOpacity
              style={styles.dropdownItem}
              onPress={() => {
                setDropdownVisible(null);
                navigation.navigate('UpdateCreateWorkOrder', { workOrder: item });
              }}
            >
              <View style={styles.dropdownItemContent}>
                <MaterialIcons name="edit" size={16} color="#333" style={styles.dropdownIcon} />
                <Text style={styles.dropdownText}>Edit</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.dropdownItem}
              onPress={() => handleViewPDF(item.id)}
            >
              <View style={styles.dropdownItemContent}>
                <MaterialIcons name="picture-as-pdf" size={16} color="#333" style={styles.dropdownIcon} />
                <Text style={styles.dropdownText}>View PDF</Text>
              </View>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.dropdownItem}
              onPress={() => deleteWorkOrder(item.id)}
            >
              <View style={styles.dropdownItemContent}>
                <MaterialIcons name="delete" size={16} color="#d32f2f" style={styles.dropdownIcon} />
                <Text style={[styles.dropdownText, { color: '#d32f2f' }]}>Delete</Text>
              </View>
            </TouchableOpacity>
          </Animated.View>
        )}
      </TouchableOpacity>
    </View>
  );

  return (
    <View style={styles.container}>
      <FlatList
        data={filteredWorkOrders}
        renderItem={renderWorkOrder}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.listContent}
        style={styles.flatList}
        ListEmptyComponent={
          <Text style={styles.emptyText}>
            {searchQuery ? 'No matching work orders found' : 'No work orders available'}
          </Text>
        }
        refreshing={isRefreshing}
        onRefresh={handleRefresh}
      />

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  flatList: {
    flex: 1,
  },
  listContent: {
    padding: 16,
    paddingBottom: 100,
  },
  workOrderRow: {
    backgroundColor: '#fff',
    borderRadius: 8, // Increased for better appearance
    marginBottom: 12,
    padding: 12,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    elevation: 2, // Slightly increased for better shadow
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    position: 'relative',
  },
  workOrderContent: {
    flex: 1,
  },
  workOrderHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 12,
  },
  workOrderTitle: {
    fontSize: 16,
    fontWeight: '600', // Bolder for emphasis
    color: '#015185',
    flex: 1,
    marginRight: 8,
  },
  priorityBadge: {
    borderRadius: 4, // Smoother corners
    paddingVertical: 4,
    paddingHorizontal: 8,
  },
  priorityText: {
    fontSize: 12,
    color: '#fff',
    fontWeight: '600',
  },
  workOrderInfo: {
    flex: 1,
  },
  detailRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
    position: 'relative',
    flexWrap: 'wrap', // Allow wrapping for long content
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    marginLeft: 8,
    width: 90,
    fontWeight: '500',
  },
  detailValue: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    fontWeight: '400',
  },
  threeDotsContainer: {
    marginLeft: 8,
    position: 'absolute',
    right: 0,
    top: 0,
    zIndex: 1000, // Increased zIndex
  },
  threeDotsButton: {
    padding: 4,
    justifyContent: 'center',
    alignItems: 'center',
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#f0f0f0', // Light background for the button
  },
  dropdownMenu: {
    position: 'absolute',
    top: 32, // Adjusted to position below the three dots
    right: 0,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#d0d0d0',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    zIndex: 1001, // Ensure dropdown is above the container and other elements
    minWidth: 140,
    paddingVertical: 4,
  },
  dropdownItem: {
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  dropdownItemContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  dropdownIcon: {
    marginRight: 8,
  },
  dropdownText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  emptyText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 24,
  },
  syncModalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dropdownOverlay: {
    flex: 1,
    // Use a background color to catch taps outside the dropdown
    backgroundColor: 'rgba(0,0,0,0.01)', // Adjust transparency as needed
  },
  dropdownMenuPortal: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#d0d0d0',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    minWidth: 140,
    paddingVertical: 4,
    // zIndex needs to be very high to appear above everything
    zIndex: 9999,
  },
});

export default AllWotab;