import React, { useState, useRef, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  TextInput,
  ScrollView,
  KeyboardAvoidingView,
  Keyboard,
  Platform,
  Image,
  ActivityIndicator,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import LinearGradient from 'react-native-linear-gradient';
import DateTimePicker from '@react-native-community/datetimepicker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');

const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  showSelectButton = false,
  onSelectPress,
  noBackground = false,
  noInput = false,
  onFocus,
  onLayout,
  onPress,
  style,
  inputStyle,
  accessibilityLabel,
  accessibilityRole = 'text',
  ...props
}) => (
  <View style={[inputStyles.container, noInput && { marginBottom: width * 0.02 }, style]} onLayout={onLayout}>
    {noInput ? (
      <View style={inputStyles.noInputContainer}>
        <Text style={inputStyles.label}>
          {label}
          {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
        </Text>
        <TouchableOpacity
          style={inputStyles.selectButton}
          onPress={onSelectPress}
          accessibilityLabel={`Select ${label}`}
          accessibilityRole="button"
        >
          <Text style={inputStyles.selectText}>Select</Text>
          <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
        </TouchableOpacity>
      </View>
    ) : (
      <>
        <Text style={inputStyles.label}>
          {label}
          {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
        </Text>
        <TouchableOpacity
          style={inputStyles.inputContainer}
          onPress={!editable ? onPress : undefined}
          activeOpacity={!editable ? 0.7 : 1}
        >
          <TextInput
            style={[
              inputStyles.input,
              showError ? inputStyles.inputError : null,
              !editable && inputStyles.disabledInput,
              noBackground && inputStyles.noBackground,
              inputStyle,
            ]}
            value={value}
            onChangeText={onChangeText}
            editable={editable}
            placeholderTextColor="#888"
            onFocus={onFocus}
            accessibilityLabel={accessibilityLabel || label}
            accessibilityRole={accessibilityRole}
            {...props}
          />
          {icon && (
            <TouchableOpacity
              style={inputStyles.inputIcon}
              onPress={onIconPress || onPress}
              accessibilityLabel={`Select ${label}`}
              accessibilityRole="button"
            >
              {icon}
            </TouchableOpacity>
          )}
          {showSelectButton && (
            <TouchableOpacity
              style={inputStyles.selectButton}
              onPress={onSelectPress}
              accessibilityLabel={`Select ${label}`}
              accessibilityRole="button"
            >
              <Text style={inputStyles.selectText}>Select</Text>
              <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
            </TouchableOpacity>
          )}
        </TouchableOpacity>
        {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
      </>
    )}
  </View>
);

// Helper function to show toast messages
const showToast = (type, title, message) => {
  Toast.show({
    type: type,
    text1: title,
    text2: message,
    position: 'bottom',
  });
};

const TeanHrApplicationApply = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const scrollViewRef = useRef(null);
  const [fullName, setFullName] = useState('');
  const [employeeNo, setEmployeeNo] = useState('');
  const [country, setCountry] = useState([]);
  const [project, setProject] = useState([]);
  const [department, setDepartment] = useState('');
  const [nrcNo, setNrcNo] = useState('');
  const [manNo, setManNo] = useState('');
  const [applicationType, setApplicationType] = useState(null);
  const [applicationTypeOpen, setApplicationTypeOpen] = useState(false);
  const [isKeyboardVisible, setIsKeyboardVisible] = useState(false);
  const [keyboardHeight, setKeyboardHeight] = useState(0);

  // State variables for Leave Application fields
  const [leaveStartDate, setLeaveStartDate] = useState('');
  const [leaveEndDate, setLeaveEndDate] = useState('');
  const [reportDate, setReportDate] = useState('');
  const [reasonOfLeave, setReasonOfLeave] = useState('');
  const [addressWhileOnLeave, setAddressWhileOnLeave] = useState('');
  const [otherContactName, setOtherContactName] = useState('');
  const [otherContactCell, setOtherContactCell] = useState('');
  const [signatureDate, setSignatureDate] = useState('');
  const [leaveSignature, setLeaveSignature] = useState(null);
  const [leaveSignatureError, setLeaveSignatureError] = useState(false);
  const [leaveSignatureLoading, setLeaveSignatureLoading] = useState(false);

  // State variables for Funeral Grant Form fields
  const [funeralSpouseChildData, setFuneralSpouseChildData] = useState([
    { id: '1', date: '', description: 'STANDARD COFFIN', amount: '' },
    { id: '2', date: '', description: 'CASH', amount: '' },
    { id: '3', date: '', description: 'TRANSPORT', amount: '' },
  ]);
  const [funeralMotherFatherData, setFuneralMotherFatherData] = useState([
    { id: '1', date: '', description: 'CASH', amount: '' },
    { id: '2', date: '', description: 'TRANSPORT', amount: '' },
  ]);
  const [deceasedName, setDeceasedName] = useState('');
  const [deceasedRelationship, setDeceasedRelationship] = useState('');
  const [deceasedDate, setDeceasedDate] = useState('');
  const [funeralEmployeeSignatureDate, setFuneralEmployeeSignatureDate] = useState('');
  const [funeralFromDate, setFuneralFromDate] = useState('');
  const [funeralToDate, setFuneralToDate] = useState('');
  const [selectedFuneralRelationshipType, setSelectedFuneralRelationshipType] = useState(null);
  const [funeralRelationshipTypeOpen, setFuneralRelationshipTypeOpen] = useState(false);
  const [funeralSignature, setFuneralSignature] = useState(null);
  const [funeralSignatureError, setFuneralSignatureError] = useState(false);
  const [funeralSignatureLoading, setFuneralSignatureLoading] = useState(false);

  // State variables for Salary Advance Form fields
  const [salaryAdvanceDepartment, setSalaryAdvanceDepartment] = useState('');
  const [salaryAdvanceAmount, setSalaryAdvanceAmount] = useState('');
  const [selectedSites, setSelectedSites] = useState([]);
  const [salaryAdvanceOtherSite, setSalaryAdvanceOtherSite] = useState('');
  const [selectedRecoveryPeriod, setSelectedRecoveryPeriod] = useState(null);
  const [selectedReasons, setSelectedReasons] = useState([]);
  const [salaryAdvanceOtherReason, setSalaryAdvanceOtherReason] = useState('');
  const [salaryAdvanceSignatureDate, setSalaryAdvanceSignatureDate] = useState('');
  const [salaryAdvanceSignature, setSalaryAdvanceSignature] = useState(null);
  const [salaryAdvanceSignatureError, setSalaryAdvanceSignatureError] = useState(false);
  const [salaryAdvanceSignatureLoading, setSalaryAdvanceSignatureLoading] = useState(false);

  // State variables for Application for Appeal fields
  const [appealFormDate, setAppealFormDate] = useState('');
  const [appealManNo, setAppealManNo] = useState('');
  const [appealName, setAppealName] = useState('');
  const [appealSanction, setAppealSanction] = useState('');
  const [appealOffence, setAppealOffence] = useState('');
  const [appealOffenceDate, setAppealOffenceDate] = useState('');
  const [appealGrounds, setAppealGrounds] = useState('');
  const [appellantSignature, setAppellantSignature] = useState(null);
  const [appealDateOfAppeal, setAppealDateOfAppeal] = useState('');
  const [appellantSignatureError, setAppellantSignatureError] = useState(false);
  const [appellantSignatureLoading, setAppellantSignatureLoading] = useState(false);

  // State for DatePicker
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [datePickerValue, setDatePickerValue] = useState(new Date());
  const [currentEditingDateField, setCurrentEditingDateField] = useState(null);
  const [currentTableId, setCurrentTableId] = useState(null);
  const [datePickerMaxDate, setDatePickerMaxDate] = useState(null);

  const applicationTypes = [
    { label: 'Leave Application', value: 'leave_application' },
    { label: 'Funeral Grant Form', value: 'funeral_grant_form' },
    { label: 'Salary Advance Application', value: 'salary_advance_application' },
    { label: 'Application for Appeal', value: 'application_for_appeal' },
  ];

  const funeralRelationshipOptions = [
    { label: 'SPOUSE AND BIOLOGICAL CHILD', value: 'spouse_child' },
    { label: 'MOTHER AND FATHER', value: 'mother_father' },
    { label: 'BOTH', value: 'both' },
  ];

  const sitesOptions = [
    { label: 'KALUMBILA', value: 'kalumbila' },
    { label: 'KANSANSHI', value: 'kansanshi' },
    { label: 'KIMITETO', value: 'kimteto' },
    { label: 'OTHERS', value: 'others' },
  ];

  const reasonsOptions = [
    { label: 'Medical Assistance', value: 'medical_assistance' },
    { label: 'Funeral', value: 'funeral' },
    { label: 'School Fees', value: 'school_fees' },
    { label: 'Tools Recovery', value: 'tools_recovery' },
    { label: 'Others', value: 'others' },
  ];

  const recoveryPeriodOptions = [
    { label: '1 Instalment', value: '1_instalment' },
    { label: '2 Instalments', value: '2_instalments' },
    { label: '3 Instalments', value: '3_instalments' },
  ];

  const [layoutPositions, setLayoutPositions] = useState({});

  // Fetch user data from Firebase on component mount
  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const teamMemberId = await AsyncStorage.getItem('teamMemberId');
        if (teamMemberId) {
          const userRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}`);
          const snapshot = await userRef.once('value');
          const userData = snapshot.val();
          if (userData) {
            setFullName(userData.fullName || '');
            setEmployeeNo(userData.employeeNumber || '');
            setCountry(userData.country || []);
            setProject(userData.project || []);
            setDepartment(userData.department || '');
            setNrcNo(userData.nrcIdNumber || '');
            setManNo(userData.manNo || '');
          } else {
            showToast('warning', 'Warning', 'User data not found in database.');
          }
        } else {
          showToast('error', 'Error', 'Team Member ID not found in storage.');
        }
      } catch (error) {
        console.error('Error fetching user data:', error);
        showToast('error', 'Error', 'Failed to load user data.');
      }
    };

    fetchUserData();
  }, []);

  // Keyboard handling for visibility and height
  useEffect(() => {
    const keyboardDidShowListener = Keyboard.addListener('keyboardDidShow', (e) => {
      setIsKeyboardVisible(true);
      setKeyboardHeight(e.endCoordinates.height);
    });
    const keyboardDidHideListener = Keyboard.addListener('keyboardDidHide', () => {
      setIsKeyboardVisible(false);
      setKeyboardHeight(0);
    });

    return () => {
      keyboardDidShowListener.remove();
      keyboardDidHideListener.remove();
    };
  }, []);

  const handleInputFocus = useCallback((field) => () => {
    if (scrollViewRef.current && layoutPositions[field]) {
      const offset = layoutPositions[field] - (height * 0.1 + keyboardHeight);
      scrollViewRef.current.scrollTo({ y: offset, animated: true });
    }
  }, [layoutPositions, keyboardHeight]);

  const handleLayout = (field) => (event) => {
    const layout = event.nativeEvent.layout;
    setLayoutPositions((prev) => ({ ...prev, [field]: layout.y }));
  };

  const formatDate = (date) => {
    if (!date) return '';
    const d = new Date(date);
    const year = d.getFullYear();
    const month = String(d.getMonth() + 1).padStart(2, '0');
    const day = String(d.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  const onDateChange = (event, selectedDate) => {
    if (event.type === 'dismissed') {
      setShowDatePicker(false);
      return;
    }

    const currentDate = selectedDate || datePickerValue;
    setShowDatePicker(false);
    setDatePickerValue(currentDate);

    if (currentTableId) {
      if (currentTableId.startsWith('funeralSpouseChild')) {
        const updatedData = funeralSpouseChildData.map((item) =>
          item.id === currentTableId.split('-')[1] ? { ...item, date: formatDate(currentDate) } : item
        );
        setFuneralSpouseChildData(updatedData);
      } else if (currentTableId.startsWith('funeralMotherFather')) {
        const updatedData = funeralMotherFatherData.map((item) =>
          item.id === currentTableId.split('-')[1] ? { ...item, date: formatDate(currentDate) } : item
        );
        setFuneralMotherFatherData(updatedData);
      }
      setCurrentTableId(null);
    } else {
      switch (currentEditingDateField) {
        case 'leaveStartDate':
          setLeaveStartDate(formatDate(currentDate));
          break;
        case 'leaveEndDate':
          if (leaveStartDate && new Date(currentDate) < new Date(leaveStartDate)) {
            showToast('error', 'Error', 'End date must be after start date.');
            return;
          }
          setLeaveEndDate(formatDate(currentDate));
          break;
        case 'reportDate':
          if (leaveEndDate && new Date(currentDate) <= new Date(leaveEndDate)) {
            showToast('error', 'Error', 'Report date must be after end date.');
            return;
          }
          setReportDate(formatDate(currentDate));
          break;
        case 'signatureDate':
          setSignatureDate(formatDate(currentDate));
          break;
        case 'deceasedDate':
          setDeceasedDate(formatDate(currentDate));
          break;
        case 'funeralEmployeeSignatureDate':
          setFuneralEmployeeSignatureDate(formatDate(currentDate));
          break;
        case 'funeralFromDate':
          setFuneralFromDate(formatDate(currentDate));
          break;
        case 'funeralToDate':
          if (funeralFromDate && new Date(currentDate) < new Date(funeralFromDate)) {
            showToast('error', 'Error', 'To date must be after from date.');
            return;
          }
          setFuneralToDate(formatDate(currentDate));
          break;
        case 'salaryAdvanceSignatureDate':
          setSalaryAdvanceSignatureDate(formatDate(currentDate));
          break;
        case 'appealFormDate':
          setAppealFormDate(formatDate(currentDate));
          break;
        case 'appealOffenceDate':
          setAppealOffenceDate(formatDate(currentDate));
          break;
        case 'appealDateOfAppeal':
          setAppealDateOfAppeal(formatDate(currentDate));
          break;
        default:
          break;
      }
    }
  };

  const showMode = (currentMode, field, tableId = null) => {
    setShowDatePicker(true);
    setCurrentEditingDateField(field);
    setCurrentTableId(tableId);

    let maxDate = null;
    if (
      [
        'leaveEndDate',
        'reportDate',
        'funeralToDate',
        'appealDateOfAppeal',
        'signatureDate',
        'funeralEmployeeSignatureDate',
        'salaryAdvanceSignatureDate',
        'appealFormDate',
      ].includes(field)
    ) {
      maxDate = null; // Allow future dates
    } else {
      maxDate = new Date(); // Restrict to today or past
    }
    setDatePickerMaxDate(maxDate);

    let initialDate = new Date();
    if (tableId) {
      if (tableId.startsWith('funeralSpouseChild')) {
        const item = funeralSpouseChildData.find((d) => d.id === tableId.split('-')[1]);
        initialDate = item && item.date ? new Date(item.date) : new Date();
      } else if (tableId.startsWith('funeralMotherFather')) {
        const item = funeralMotherFatherData.find((d) => d.id === tableId.split('-')[1]);
        initialDate = item && item.date ? new Date(item.date) : new Date();
      }
    } else {
      switch (field) {
        case 'leaveStartDate':
          initialDate = leaveStartDate ? new Date(leaveStartDate) : new Date();
          break;
        case 'leaveEndDate':
          initialDate = leaveEndDate ? new Date(leaveEndDate) : new Date();
          break;
        case 'reportDate':
          initialDate = reportDate ? new Date(reportDate) : new Date();
          break;
        case 'signatureDate':
          initialDate = signatureDate ? new Date(signatureDate) : new Date();
          break;
        case 'deceasedDate':
          initialDate = deceasedDate ? new Date(deceasedDate) : new Date();
          break;
        case 'funeralEmployeeSignatureDate':
          initialDate = funeralEmployeeSignatureDate ? new Date(funeralEmployeeSignatureDate) : new Date();
          break;
        case 'funeralFromDate':
          initialDate = funeralFromDate ? new Date(funeralFromDate) : new Date();
          break;
        case 'funeralToDate':
          initialDate = funeralToDate ? new Date(funeralToDate) : new Date();
          break;
        case 'salaryAdvanceSignatureDate':
          initialDate = salaryAdvanceSignatureDate ? new Date(salaryAdvanceSignatureDate) : new Date();
          break;
        case 'appealFormDate':
          initialDate = appealFormDate ? new Date(appealFormDate) : new Date();
          break;
        case 'appealOffenceDate':
          initialDate = appealOffenceDate ? new Date(appealOffenceDate) : new Date();
          break;
        case 'appealDateOfAppeal':
          initialDate = appealDateOfAppeal ? new Date(appealDateOfAppeal) : new Date();
          break;
      }
    }
    setDatePickerValue(initialDate);
  };

  const showDatePickerDialog = (field, tableId = null) => {
    showMode('date', field, tableId);
  };

  const handleFuneralAmountChange = (tableType, id, value) => {
    const cleanedValue = value.replace(/[^0-9.]/g, '');
    const parts = cleanedValue.split('.');
    if (parts.length > 2) return;
    if (parts[1] && parts[1].length > 2) return;

    if (tableType === 'spouseChild') {
      const updatedData = funeralSpouseChildData.map((item) =>
        item.id === id ? { ...item, amount: cleanedValue } : item
      );
      setFuneralSpouseChildData(updatedData);
    } else if (tableType === 'motherFather') {
      const updatedData = funeralMotherFatherData.map((item) =>
        item.id === id ? { ...item, amount: cleanedValue } : item
      );
      setFuneralMotherFatherData(updatedData);
    }
  };

  const calculateTotal = (data) => {
    return data.reduce((sum, item) => sum + (parseFloat(item.amount) || 0), 0).toFixed(2);
  };

  const toggleSiteSelection = (value) => {
    setSelectedSites((prev) => {
      if (prev.includes(value)) {
        return prev.filter((item) => item !== value);
      } else {
        return [...prev, value];
      }
    });
  };

  const toggleReasonSelection = (value) => {
    setSelectedReasons((prev) => {
      if (prev.includes(value)) {
        return prev.filter((item) => item !== value);
      } else {
        return [...prev, value];
      }
    });
  };

  const handleRecoveryPeriodSelection = (value) => {
    setSelectedRecoveryPeriod(value);
  };

  const handleOpenSignaturePad = (signatureType) => {
    let initialSignature;
    switch (signatureType) {
      case 'leave':
        initialSignature = leaveSignature;
        setLeaveSignatureLoading(true);
        break;
      case 'funeral':
        initialSignature = funeralSignature;
        setFuneralSignatureLoading(true);
        break;
      case 'salaryAdvance':
        initialSignature = salaryAdvanceSignature;
        setSalaryAdvanceSignatureLoading(true);
        break;
      case 'appeal':
        initialSignature = appellantSignature;
        setAppellantSignatureLoading(true);
        break;
      default:
        initialSignature = null;
    }
    
    // Create a callback function to handle signature data
    const handleSignatureResult = (signature) => {
      if (signature) {
        switch (signatureType) {
          case 'leave':
            setLeaveSignature(signature);
            setLeaveSignatureError(false);
            break;
          case 'funeral':
            setFuneralSignature(signature);
            setFuneralSignatureError(false);
            break;
          case 'salaryAdvance':
            setSalaryAdvanceSignature(signature);
            setSalaryAdvanceSignatureError(false);
            break;
          case 'appeal':
            setAppellantSignature(signature);
            setAppellantSignatureError(false);
            break;
        }
      }
      // Clear loading state
      switch (signatureType) {
        case 'leave':
          setLeaveSignatureLoading(false);
          break;
        case 'funeral':
          setFuneralSignatureLoading(false);
          break;
        case 'salaryAdvance':
          setSalaryAdvanceSignatureLoading(false);
          break;
        case 'appeal':
          setAppellantSignatureLoading(false);
          break;
      }
    };
    
    navigation.navigate('SignatureHRappl', {
      signatureTitle: `${signatureType.charAt(0).toUpperCase() + signatureType.slice(1)} Signature`,
      signatureType,
      initialSignature,
      onSignatureSave: handleSignatureResult,
    });
  };

  // Handle form submission and save to Firebase
  const handleSubmit = async () => {
    try {
      const teamMemberId = await AsyncStorage.getItem('teamMemberId');
      if (!teamMemberId) {
        showToast('error', 'Error', 'Team Member ID not found in storage.');
        return;
      }

      if (!applicationType) {
        showToast('error', 'Error', 'Please select an application type.');
        return;
      }

      let hasError = false;

      if (applicationType === 'leave_application') {
        if (!leaveStartDate) {
          showToast('error', 'Error', 'Please select a start date for leave.');
          hasError = true;
        }
        if (!leaveEndDate) {
          showToast('error', 'Error', 'Please select an end date for leave.');
          hasError = true;
        }
        if (!reportDate) {
          showToast('error', 'Error', 'Please select a report date.');
          hasError = true;
        }
        if (!reasonOfLeave) {
          showToast('error', 'Error', 'Please provide a reason for leave.');
          hasError = true;
        }
        if (!addressWhileOnLeave) {
          showToast('error', 'Error', 'Please provide an address while on leave.');
          hasError = true;
        }
        if (!signatureDate) {
          showToast('error', 'Error', 'Please select a signature date.');
          hasError = true;
        }
        // Removed: if (!leaveSignature) {
        //   setLeaveSignatureError(true);
        //   showToast('error', 'Error', 'Please provide a signature for the leave application.');
        //   hasError = true;
        // }
      } else if (applicationType === 'funeral_grant_form') {
        if (!funeralFromDate) {
          showToast('error', 'Error', 'Please select a from date for the funeral grant.');
          hasError = true;
        }
        if (!funeralToDate) {
          showToast('error', 'Error', 'Please select a to date for the funeral grant.');
          hasError = true;
        }
        if (!selectedFuneralRelationshipType) {
          showToast('error', 'Error', 'Please select a relationship type for the funeral grant.');
          hasError = true;
        }
        if (!deceasedName) {
          showToast('error', 'Error', "Please provide the deceased's name.");
          hasError = true;
        }
        if (!deceasedRelationship) {
          showToast('error', 'Error', 'Please provide the relationship with the deceased.');
          hasError = true;
        }
        if (!deceasedDate) {
          showToast('error', 'Error', 'Please select the deceased date.');
          hasError = true;
        }
        if (!funeralEmployeeSignatureDate) {
          showToast('error', 'Error', 'Please select a signature date for the funeral grant.');
          hasError = true;
        }
        // Removed: if (!funeralSignature) {
        //   setFuneralSignatureError(true);
        //   showToast('error', 'Error', 'Please provide a signature for the funeral grant.');
        //   hasError = true;
        // }
        if (selectedFuneralRelationshipType === 'spouse_child') {
          const hasAmount = funeralSpouseChildData.some((item) => item.amount);
          if (!hasAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for spouse/child funeral grant.');
            hasError = true;
          }
        } else if (selectedFuneralRelationshipType === 'mother_father') {
          const hasAmount = funeralMotherFatherData.some((item) => item.amount);
          if (!hasAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for mother/father funeral grant.');
            hasError = true;
          }
        } else if (selectedFuneralRelationshipType === 'both') {
          const hasSpouseChildAmount = funeralSpouseChildData.some((item) => item.amount);
          const hasMotherFatherAmount = funeralMotherFatherData.some((item) => item.amount);
          if (!hasSpouseChildAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for spouse/child funeral grant.');
            hasError = true;
          }
          if (!hasMotherFatherAmount) {
            showToast('error', 'Error', 'Please provide at least one amount for mother/father funeral grant.');
            hasError = true;
          }
        }
      } else if (applicationType === 'salary_advance_application') {
        if (!salaryAdvanceDepartment) {
          showToast('error', 'Error', 'Please provide the department for salary advance.');
          hasError = true;
        }
        if (!salaryAdvanceAmount) {
          showToast('error', 'Error', 'Please provide the amount for salary advance.');
          hasError = true;
        }
        if (selectedSites.length === 0) {
          showToast('error', 'Error', 'Please select at least one site for salary advance.');
          hasError = true;
        }
        if (selectedSites.includes('others') && !salaryAdvanceOtherSite) {
          showToast('error', 'Error', 'Please specify the other site for salary advance.');
          hasError = true;
        }
        if (!selectedRecoveryPeriod) {
          showToast('error', 'Error', 'Please select a recovery period for salary advance.');
          hasError = true;
        }
        if (selectedReasons.length === 0) {
          showToast('error', 'Error', 'Please select at least one reason for salary advance.');
          hasError = true;
        }
        if (selectedReasons.includes('others') && !salaryAdvanceOtherReason) {
          showToast('error', 'Error', 'Please specify the other reason for salary advance.');
          hasError = true;
        }
        if (!salaryAdvanceSignatureDate) {
          showToast('error', 'Error', 'Please select a signature date for salary advance.');
          hasError = true;
        }
        // Removed: if (!salaryAdvanceSignature) {
        //   setSalaryAdvanceSignatureError(true);
        //   showToast('error', 'Error', 'Please provide a signature for the salary advance.');
        //   hasError = true;
        // }
      } else if (applicationType === 'application_for_appeal') {
        if (!appealFormDate) {
          showToast('error', 'Error', 'Please select the form date for appeal.');
          hasError = true;
        }
        if (!appealManNo) {
          showToast('error', 'Error', 'Please provide the man number for appeal.');
          hasError = true;
        }
        if (!appealName) {
          showToast('error', 'Error', 'Please provide the name for appeal.');
          hasError = true;
        }
        if (!appealSanction) {
          showToast('error', 'Error', 'Please provide the sanction for appeal.');
          hasError = true;
        }
        if (!appealOffence) {
          showToast('error', 'Error', 'Please provide the offence for appeal.');
          hasError = true;
        }
        if (!appealOffenceDate) {
          showToast('error', 'Error', 'Please select the offence date for appeal.');
          hasError = true;
        }
        if (!appealGrounds) {
          showToast('error', 'Error', 'Please provide the grounds for appeal.');
          hasError = true;
        }
        if (!appealDateOfAppeal) {
          showToast('error', 'Error', 'Please select the date of appeal.');
          hasError = true;
        }
        // Removed: if (!appellantSignature) {
        //   setAppellantSignatureError(true);
        //   showToast('error', 'Error', 'Please provide a signature for the appeal.');
        //   hasError = true;
        // }
      }

      if (hasError) {
        return;
      }

      let applicationData = {
        applicationType,
        fullName,
        employeeNo,
        country,
        project,
        department,
        nrcNo,
        manNo,
        submittedAt: new Date().toISOString(),
      };

      if (applicationType === 'leave_application') {
        applicationData = {
          ...applicationData,
          leaveStartDate,
          leaveEndDate,
          reportDate,
          reasonOfLeave,
          addressWhileOnLeave,
          otherContactName,
          otherContactCell,
          signatureDate,
          leaveSignature,
        };
      } else if (applicationType === 'funeral_grant_form') {
        applicationData = {
          ...applicationData,
          funeralFromDate,
          funeralToDate,
          selectedFuneralRelationshipType,
          funeralSpouseChildData: selectedFuneralRelationshipType === 'spouse_child' || selectedFuneralRelationshipType === 'both' ? funeralSpouseChildData : null,
          funeralMotherFatherData: selectedFuneralRelationshipType === 'mother_father' || selectedFuneralRelationshipType === 'both' ? funeralMotherFatherData : null,
          deceasedName,
          deceasedRelationship,
          deceasedDate,
          funeralEmployeeSignatureDate,
          funeralSignature,
        };
      } else if (applicationType === 'salary_advance_application') {
        applicationData = {
          ...applicationData,
          salaryAdvanceDepartment,
          salaryAdvanceAmount,
          selectedSites,
          salaryAdvanceOtherSite: selectedSites.includes('others') ? salaryAdvanceOtherSite : '',
          selectedRecoveryPeriod,
          selectedReasons,
          salaryAdvanceOtherReason: selectedReasons.includes('others') ? salaryAdvanceOtherReason : '',
          salaryAdvanceSignatureDate,
          salaryAdvanceSignature,
        };
      } else if (applicationType === 'application_for_appeal') {
        applicationData = {
          ...applicationData,
          appealFormDate,
          appealManNo,
          appealName,
          appealSanction,
          appealOffence,
          appealOffenceDate,
          appealGrounds,
          appellantSignature,
          appealDateOfAppeal,
        };
      }

      const applicationRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}/HrmoduleTeam`);
      await applicationRef.push(applicationData);

      showToast('success', 'Success', 'Application submitted successfully.');
      navigation.goBack();
    } catch (error) {
      console.error('Error submitting application:', error);
      showToast('error', 'Error', 'Failed to submit application. Please try again.');
    }
  };

  // Render selected items for country and project
  const renderSelectedItems = (selectedArray) => {
    if (!selectedArray || selectedArray.length === 0) {
      return <Text style={styles.noSelectionText}>No items selected</Text>;
    }
    return (
      <View style={styles.selectedItemsContainer}>
        {selectedArray.map((item, index) => (
          <Text key={index} style={styles.selectedItemText}>
            {item}
            {index < selectedArray.length - 1 ? ', ' : ''}
          </Text>
        ))}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity
          style={styles.backButton}
          onPress={() => navigation.goBack()}
          accessibilityLabel="Go back"
          accessibilityRole="button"
        >
          <MaterialIcons name="arrow-back-ios" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>Application</Text>
        <TouchableOpacity
          style={styles.menuButton}
          accessibilityLabel="Open menu"
          accessibilityRole="button"
        >
          <MaterialIcons name="menu" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
        keyboardVerticalOffset={Platform.OS === 'ios' ? height * 0.1 : 0}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={[styles.content, { paddingBottom: keyboardHeight + width * 0.1 }]}
          keyboardShouldPersistTaps="handled"
          showsVerticalScrollIndicator={false}
          scrollEnabled={true}
        >
          <CustomInputField
            label="Full name"
            value={fullName}
            placeholder="Full name"
            onChangeText={setFullName}
            editable={false}
            onFocus={handleInputFocus('fullName')}
            onLayout={handleLayout('fullName')}
          />

          <CustomInputField
            label="Employee no"
            value={employeeNo}
            placeholder="Employee no"
            onChangeText={setEmployeeNo}
            keyboardType="numeric"
            editable={false}
            onFocus={handleInputFocus('employeeNo')}
            onLayout={handleLayout('employeeNo')}
          />

          <View style={inputStyles.container} onLayout={handleLayout('country')}>
            <Text style={inputStyles.label}>Country</Text>
            <View style={styles.selectedDisplayBox}>
              {renderSelectedItems(country)}
            </View>
          </View>

          <View style={inputStyles.container} onLayout={handleLayout('project')}>
            <Text style={inputStyles.label}>Project</Text>
            <View style={styles.selectedDisplayBox}>
              {renderSelectedItems(project)}
            </View>
          </View>

          <CustomInputField
            label="Department"
            value={department}
            placeholder="Department"
            onChangeText={setDepartment}
            editable={false}
            onFocus={handleInputFocus('department')}
            onLayout={handleLayout('department')}
          />

          <CustomInputField
            label="NRC no"
            value={nrcNo}
            placeholder="NRC no"
            onChangeText={setNrcNo}
            keyboardType="numeric"
            editable={false}
            onFocus={handleInputFocus('nrcNo')}
            onLayout={handleLayout('nrcNo')}
          />

          <View style={[styles.inputRow, { zIndex: 9999 }]} onLayout={handleLayout('applicationType')}>
            <Dropdown
              style={[styles.dropdown, { zIndex: 8000 }]}
              containerStyle={[styles.dropdownContainer, { zIndex: 9000 }]}
              placeholderStyle={styles.dropdownText}
              selectedTextStyle={styles.dropdownText}
              itemTextStyle={styles.dropdownText}
              data={applicationTypes}
              labelField="label"
              valueField="value"
              placeholder="Application type"
              value={applicationType}
              onChange={(item) => {
                setApplicationType(item.value);
                setApplicationTypeOpen(false);
              }}
              onFocus={() => setApplicationTypeOpen(true)}
              onBlur={() => setApplicationTypeOpen(false)}
              maxHeight={height * 0.5}
              showsVerticalScrollIndicator={true}
              nestedScrollEnabled={true}
              renderRightIcon={() => (
                <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
              )}
              accessibilityLabel="Select application type"
              accessibilityRole="combobox"
            />
          </View>

          {applicationType === 'leave_application' && (
            <>
              <View style={styles.twoColumnRow}>
                <CustomInputField
                  label="Date"
                  value={leaveStartDate}
                  placeholder="Select Start Date"
                  onPress={() => showDatePickerDialog('leaveStartDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('leaveStartDate')}
                  onFocus={handleInputFocus('leaveStartDate')}
                  onLayout={handleLayout('leaveStartDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!leaveStartDate && leaveSignatureError}
                  errorMessage="Start date is required"
                />
                <CustomInputField
                  label="To"
                  value={leaveEndDate}
                  placeholder="Select End Date"
                  onPress={() => showDatePickerDialog('leaveEndDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('leaveEndDate')}
                  onFocus={handleInputFocus('leaveEndDate')}
                  onLayout={handleLayout('leaveEndDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!leaveEndDate && leaveSignatureError}
                  errorMessage="End date is required"
                />
              </View>

              <CustomInputField
                label="Report date"
                value={reportDate}
                placeholder="Select Report Date"
                onPress={() => showDatePickerDialog('reportDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('reportDate')}
                onFocus={handleInputFocus('reportDate')}
                onLayout={handleLayout('reportDate')}
                isRequired
                showError={!reportDate && leaveSignatureError}
                errorMessage="Report date is required"
              />

              <CustomInputField
                label="Reason of leave"
                value={reasonOfLeave}
                placeholder="Reason of leave"
                onChangeText={setReasonOfLeave}
                onFocus={handleInputFocus('reasonOfLeave')}
                onLayout={handleLayout('reasonOfLeave')}
                isRequired
                showError={!reasonOfLeave && leaveSignatureError}
                errorMessage="Reason is required"
              />

              <CustomInputField
                label="Address while on leave"
                value={addressWhileOnLeave}
                placeholder="Address while on leave"
                onChangeText={setAddressWhileOnLeave}
                onFocus={handleInputFocus('addressWhileOnLeave')}
                onLayout={handleLayout('addressWhileOnLeave')}
                isRequired
                showError={!addressWhileOnLeave && leaveSignatureError}
                errorMessage="Address is required"
              />

              <Text style={inputStyles.label}>Other contacts</Text>
              <View style={styles.twoColumnRow}>
                <CustomInputField
                  label="Name"
                  value={otherContactName}
                  placeholder="Name"
                  onChangeText={setOtherContactName}
                  onFocus={handleInputFocus('otherContactName')}
                  onLayout={handleLayout('otherContactName')}
                  style={styles.halfWidthInput}
                />
                <CustomInputField
                  label="Cell"
                  value={otherContactCell}
                  placeholder="Cell"
                  onChangeText={setOtherContactCell}
                  keyboardType="phone-pad"
                  onFocus={handleInputFocus('otherContactCell')}
                  onLayout={handleLayout('otherContactCell')}
                  style={styles.halfWidthInput}
                />
              </View>

              <View style={styles.signatureContainer} onLayout={handleLayout('employeeSignature')}>
                <Text style={inputStyles.label}>Employee's Signature</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, leaveSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('leave')}
                  accessibilityLabel="Capture employee signature"
                  accessibilityRole="button"
                >
                  {leaveSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : leaveSignature ? (
                    <Image
                      source={{ uri: leaveSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading leave signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {/* Removed: {leaveSignatureError && (
                  <Text style={inputStyles.errorText}>Signature is required</Text>
                )} */}
              </View>

              <CustomInputField
                label="Date"
                value={signatureDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('signatureDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('signatureDate')}
                onFocus={handleInputFocus('signatureDate')}
                onLayout={handleLayout('signatureDate')}
                isRequired
                showError={!signatureDate && leaveSignatureError}
                errorMessage="Signature date is required"
              />
            </>
          )}

          {applicationType === 'funeral_grant_form' && (
            <>
              <View style={styles.twoColumnRow}>
                <CustomInputField
                  label="From Date"
                  value={funeralFromDate}
                  placeholder="Select From Date"
                  onPress={() => showDatePickerDialog('funeralFromDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('funeralFromDate')}
                  onFocus={handleInputFocus('funeralFromDate')}
                  onLayout={handleLayout('funeralFromDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!funeralFromDate && funeralSignatureError}
                  errorMessage="From date is required"
                />
                <CustomInputField
                  label="To Date"
                  value={funeralToDate}
                  placeholder="Select To Date"
                  onPress={() => showDatePickerDialog('funeralToDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('funeralToDate')}
                  onFocus={handleInputFocus('funeralToDate')}
                  onLayout={handleLayout('funeralToDate')}
                  style={styles.halfWidthInput}
                  isRequired
                  showError={!funeralToDate && funeralSignatureError}
                  errorMessage="To date is required"
                />
              </View>

              <View style={styles.inputRow} onLayout={handleLayout('funeralRelationshipType')}>
                <Dropdown
                  style={[styles.dropdown, { zIndex: 7000 }]}
                  containerStyle={[styles.dropdownContainer, { zIndex: 7000 }]}
                  placeholderStyle={styles.dropdownText}
                  selectedTextStyle={styles.dropdownText}
                  itemTextStyle={styles.dropdownText}
                  data={funeralRelationshipOptions}
                  labelField="label"
                  valueField="value"
                  placeholder="Select Relationship Type"
                  value={selectedFuneralRelationshipType}
                  onChange={(item) => {
                    setSelectedFuneralRelationshipType(item.value);
                    setFuneralRelationshipTypeOpen(false);
                  }}
                  onFocus={() => setFuneralRelationshipTypeOpen(true)}
                  onBlur={() => setFuneralRelationshipTypeOpen(false)}
                  maxHeight={height * 0.5}
                  showsVerticalScrollIndicator={true}
                  nestedScrollEnabled={true}
                  renderRightIcon={() => (
                    <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#333" />
                  )}
                  accessibilityLabel="Select relationship type"
                  accessibilityRole="combobox"
                />
              </View>

              {selectedFuneralRelationshipType === 'spouse_child' && (
                <>
                  <Text style={styles.sectionTitle}>RELATIONSHIP: SPOUSE AND BIOLOGICAL CHILD (EMPLOYEE HIM/HERSELF)</Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralSpouseChildData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralSpouseChildDate', `funeralSpouseChild-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>
                              {item.date || 'Select Date'}
                            </Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('spouseChild', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralSpouseChildData)}
                      </Text>
                    </View>
                  </View>
                </>
              )}

              {selectedFuneralRelationshipType === 'mother_father' && (
                <>
                  <Text style={[styles.sectionTitle, { marginTop: width * 0.06 }]}>
                    RELATIONSHIP: MOTHER AND FATHER
                  </Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralMotherFatherData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralMotherFatherDate', `funeralMotherFather-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>
                              {item.date || 'Select Date'}
                            </Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('motherFather', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralMotherFatherData)}
                      </Text>
                    </View>
                  </View>
                </>
              )}

              {selectedFuneralRelationshipType === 'both' && (
                <>
                  <Text style={styles.sectionTitle}>RELATIONSHIP: SPOUSE AND BIOLOGICAL CHILD (EMPLOYEE HIM/HERSELF)</Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralSpouseChildData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralSpouseChildDate', `funeralSpouseChild-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>
                              {item.date || 'Select Date'}
                            </Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('spouseChild', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralSpouseChildData)}
                      </Text>
                    </View>
                  </View>

                  <Text style={[styles.sectionTitle, { marginTop: width * 0.06 }]}>
                    RELATIONSHIP: MOTHER AND FATHER
                  </Text>
                  <View style={styles.table}>
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableHeaderCell, styles.dateCell]}>DATE</Text>
                      <Text style={[styles.tableHeaderCell, styles.descriptionCell]}>DESCRIPTION</Text>
                      <Text style={[styles.tableHeaderCell, styles.amountCell]}>AMOUNT</Text>
                    </View>
                    {funeralMotherFatherData.map((item) => (
                      <View key={item.id} style={styles.tableRow}>
                        <View style={[styles.tableDataCell, styles.dateCell]}>
                          <TouchableOpacity
                            style={styles.tableDateButton}
                            onPress={() =>
                              showDatePickerDialog('funeralMotherFatherDate', `funeralMotherFather-${item.id}`)
                            }
                          >
                            <Text style={styles.tableDateText}>
                              {item.date || 'Select Date'}
                            </Text>
                            <MaterialIcons name="calendar-today" size={width * 0.04} color="#333" />
                          </TouchableOpacity>
                        </View>
                        <View style={[styles.tableDataCell, styles.descriptionCell]}>
                          <Text style={styles.tableCellText}>{item.description}</Text>
                        </View>
                        <View style={[styles.tableDataCell, styles.amountCell]}>
                          <TextInput
                            style={styles.tableInput}
                            value={item.amount}
                            onChangeText={(text) => handleFuneralAmountChange('motherFather', item.id, text)}
                            keyboardType="numeric"
                            placeholder="K"
                          />
                        </View>
                      </View>
                    ))}
                    <View style={styles.tableRow}>
                      <Text style={[styles.tableTotalCell, styles.dateCell]}></Text>
                      <Text style={[styles.tableTotalCell, styles.descriptionCell]}>TOTAL</Text>
                      <Text style={[styles.tableTotalCell, styles.amountCell]}>
                        K {calculateTotal(funeralMotherFatherData)}
                      </Text>
                    </View>
                  </View>
                </>
              )}

              <CustomInputField
                label="DECEASED'S NAME"
                value={deceasedName}
                placeholder="Deceased's Name"
                onChangeText={setDeceasedName}
                onFocus={handleInputFocus('deceasedName')}
                onLayout={handleLayout('deceasedName')}
                isRequired
                showError={!deceasedName && funeralSignatureError}
                errorMessage="Deceased's name is required"
              />
              <CustomInputField
                label="RELATIONSHIP"
                value={deceasedRelationship}
                placeholder="Relationship"
                onChangeText={setDeceasedRelationship}
                onFocus={handleInputFocus('deceasedRelationship')}
                onLayout={handleLayout('deceasedRelationship')}
                isRequired
                showError={!deceasedRelationship && funeralSignatureError}
                errorMessage="Relationship is required"
              />
              <CustomInputField
                label="DATE"
                value={deceasedDate}
                placeholder="Date"
                onPress={() => showDatePickerDialog('deceasedDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('deceasedDate')}
                onFocus={handleInputFocus('deceasedDate')}
                onLayout={handleLayout('deceasedDate')}
                isRequired
                showError={!deceasedDate && funeralSignatureError}
                errorMessage="Deceased date is required"
              />

              <View style={styles.signatureContainer} onLayout={handleLayout('funeralEmployeeSignature')}>
                <Text style={inputStyles.label}>EMPLOYEE SIGNATURE</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, funeralSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('funeral')}
                  accessibilityLabel="Capture employee signature for funeral grant"
                  accessibilityRole="button"
                >
                  {funeralSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : funeralSignature ? (
                    <Image
                      source={{ uri: funeralSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading funeral signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {/* Removed: {funeralSignatureError && (
                  <Text style={inputStyles.errorText}>Signature is required</Text>
                )} */}
              </View>
              <CustomInputField
                label="DATE"
                value={funeralEmployeeSignatureDate}
                placeholder="Date"
                onPress={() => showDatePickerDialog('funeralEmployeeSignatureDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('funeralEmployeeSignatureDate')}
                onFocus={handleInputFocus('funeralEmployeeSignatureDate')}
                onLayout={handleLayout('funeralEmployeeSignatureDate')}
                isRequired
                showError={!funeralEmployeeSignatureDate && funeralSignatureError}
                errorMessage="Signature date is required"
              />
            </>
          )}

          {applicationType === 'salary_advance_application' && (
            <>
              <Text style={styles.formTitle}>SALARY ADVANCE FORM</Text>

              <CustomInputField
                label="DEPARTMENT:"
                value={salaryAdvanceDepartment}
                onChangeText={setSalaryAdvanceDepartment}
                onFocus={handleInputFocus('salaryAdvanceDepartment')}
                onLayout={handleLayout('salaryAdvanceDepartment')}
                isRequired
                showError={!salaryAdvanceDepartment && salaryAdvanceSignatureError}
                errorMessage="Department is required"
              />
              <CustomInputField
                label="AMOUNT:"
                value={salaryAdvanceAmount}
                onChangeText={setSalaryAdvanceAmount}
                keyboardType="numeric"
                onFocus={handleInputFocus('salaryAdvanceAmount')}
                onLayout={handleLayout('salaryAdvanceAmount')}
                isRequired
                showError={!salaryAdvanceAmount && salaryAdvanceSignatureError}
                errorMessage="Amount is required"
              />

              <Text style={styles.label}>SITE:</Text>
              <View style={styles.checkboxGroup}>
                {sitesOptions.map((site) => (
                  <TouchableOpacity
                    key={site.value}
                    style={styles.checkboxContainer}
                    onPress={() => toggleSiteSelection(site.value)}
                    accessibilityLabel={`Select site ${site.label}`}
                    accessibilityRole="checkbox"
                    accessibilityState={{ checked: selectedSites.includes(site.value) }}
                  >
                    <MaterialIcons
                      name={selectedSites.includes(site.value) ? 'check-box' : 'check-box-outline-blank'}
                      size={width * 0.05}
                      color="#333"
                    />
                    <Text style={styles.checkboxText}>{site.label}</Text>
                    {site.value === 'others' && selectedSites.includes('others') && (
                      <TextInput
                        style={styles.otherInput}
                        value={salaryAdvanceOtherSite}
                        onChangeText={setSalaryAdvanceOtherSite}
                        placeholder="specify"
                        onFocus={handleInputFocus('salaryAdvanceOtherSite')}
                        accessibilityLabel="Specify other site"
                      />
                    )}
                  </TouchableOpacity>
                ))}
              </View>

              <Text style={styles.label}>RECOVERY PERIOD:</Text>
              <View style={styles.checkboxGroup}>
                {recoveryPeriodOptions.map((option) => (
                  <TouchableOpacity
                    key={option.value}
                    style={styles.checkboxContainer}
                    onPress={() => handleRecoveryPeriodSelection(option.value)}
                    accessibilityLabel={`Select recovery period ${option.label}`}
                    accessibilityRole="radio"
                    accessibilityState={{ selected: selectedRecoveryPeriod === option.value }}
                  >
                    <MaterialIcons
                      name={
                        selectedRecoveryPeriod === option.value ? 'radio-button-checked' : 'radio-button-unchecked'
                      }
                      size={width * 0.05}
                      color="#333"
                    />
                    <Text style={styles.checkboxText}>{option.label}</Text>
                  </TouchableOpacity>
                ))}
              </View>

              <Text style={styles.label}>Reasons for advance pay:</Text>
              <View style={styles.checkboxGroup}>
                {reasonsOptions.map((reason) => (
                  <TouchableOpacity
                    key={reason.value}
                    style={styles.checkboxContainer}
                    onPress={() => toggleReasonSelection(reason.value)}
                    accessibilityLabel={`Select reason ${reason.label}`}
                    accessibilityRole="checkbox"
                    accessibilityState={{ checked: selectedReasons.includes(reason.value) }}
                  >
                    <MaterialIcons
                      name={selectedReasons.includes(reason.value) ? 'check-box' : 'check-box-outline-blank'}
                      size={width * 0.05}
                      color="#333"
                    />
                    <Text style={styles.checkboxText}>{reason.label}</Text>
                    {reason.value === 'others' && selectedReasons.includes('others') && (
                      <TextInput
                        style={styles.otherInput}
                        value={salaryAdvanceOtherReason}
                        onChangeText={setSalaryAdvanceOtherReason}
                        placeholder="if others specify:"
                        onFocus={handleInputFocus('salaryAdvanceOtherReason')}
                        accessibilityLabel="Specify other reason"
                      />
                    )}
                  </TouchableOpacity>
                ))}
              </View>

              <View
                style={[styles.signatureContainer, { marginTop: width * 0.04 }]}
                onLayout={handleLayout('salaryAdvanceSignature')}
              >
                <Text style={inputStyles.label}>SIGNATURE OF APPLICANT:</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, salaryAdvanceSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('salaryAdvance')}
                  accessibilityLabel="Capture signature for salary advance"
                  accessibilityRole="button"
                >
                  {salaryAdvanceSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : salaryAdvanceSignature ? (
                    <Image
                      source={{ uri: salaryAdvanceSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading salary advance signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {/* Removed: {salaryAdvanceSignatureError && (
                  <Text style={inputStyles.errorText}>Signature is required</Text>
                )} */}
              </View>
              <CustomInputField
                label="DATE:"
                value={salaryAdvanceSignatureDate}
                placeholder="Date"
                onPress={() => showDatePickerDialog('salaryAdvanceSignatureDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('salaryAdvanceSignatureDate')}
                onFocus={handleInputFocus('salaryAdvanceSignatureDate')}
                onLayout={handleLayout('salaryAdvanceSignatureDate')}
                isRequired
                showError={!salaryAdvanceSignatureDate && salaryAdvanceSignatureError}
                errorMessage="Signature date is required"
              />
            </>
          )}

          {applicationType === 'application_for_appeal' && (
            <>
              <Text style={styles.formTitle}>APPLICATION FOR APPEAL</Text>

              <View style={styles.appealTopDateContainer}>
                <Text style={styles.appealFormLabel}>Date:</Text>
                <CustomInputField
                  value={appealFormDate}
                  placeholder="Select Date"
                  onPress={() => showDatePickerDialog('appealFormDate')}
                  editable={false}
                  icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                  onIconPress={() => showDatePickerDialog('appealFormDate')}
                  onFocus={handleInputFocus('appealFormDate')}
                  onLayout={handleLayout('appealFormDate')}
                  style={styles.appealTopDateInput}
                  isRequired
                  showError={!appealFormDate && appellantSignatureError}
                  errorMessage="Form date is required"
                />
              </View>

              <View style={styles.appealIntroContainer}>
                <Text style={styles.appealIntroText}>I, Man No: </Text>
                <TextInput
                  style={styles.appealTextInputLine}
                  value={appealManNo}
                  onChangeText={setAppealManNo}
                  onFocus={handleInputFocus('appealManNo')}
                  accessibilityLabel="Man number for appeal"
                  showError={!appealManNo && appellantSignatureError}
                />
                <Text style={styles.appealIntroText}> Name: </Text>
                <TextInput
                  style={styles.appealTextInputLine}
                  value={appealName}
                  onChangeText={setAppealName}
                  onFocus={handleInputFocus('appealName')}
                  accessibilityLabel="Name for appeal"
                  showError={!appealName && appellantSignatureError}
                />
              </View>

              <Text style={styles.appealSentence}>
                wish to lodge my appeal against the disciplinary action taken against me, namely:
              </Text>

              <Text style={styles.appealSanctionOffenceLabel}>(Sanction)</Text>
              <TextInput
                style={styles.appealTextInputMultiLine}
                value={appealSanction}
                onChangeText={setAppealSanction}
                multiline
                numberOfLines={3}
                onFocus={handleInputFocus('appealSanction')}
                accessibilityLabel="Sanction for appeal"
                showError={!appealSanction && appellantSignatureError}
              />

              <Text style={styles.appealSentence}>for</Text>
              <Text style={styles.appealSanctionOffenceLabel}>(offence)</Text>
              <TextInput
                style={styles.appealTextInputMultiLine}
                value={appealOffence}
                onChangeText={setAppealOffence}
                multiline
                numberOfLines={3}
                onFocus={handleInputFocus('appealOffence')}
                accessibilityLabel="Offence for appeal"
                showError={!appealOffence && appellantSignatureError}
              />

              <Text style={styles.appealSentence}>On</Text>
              <Text style={styles.appealSanctionOffenceLabel}>(Date)</Text>
              <CustomInputField
                value={appealOffenceDate}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('appealOffenceDate')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('appealOffenceDate')}
                onFocus={handleInputFocus('appealOffenceDate')}
                onLayout={handleLayout('appealOffenceDate')}
                isRequired
                showError={!appealOffenceDate && appellantSignatureError}
                errorMessage="Offence date is required"
              />

              <Text style={styles.appealSentence}>My grounds for appeal are:</Text>
              <TextInput
                style={styles.appealGroundsInput}
                value={appealGrounds}
                onChangeText={setAppealGrounds}
                multiline
                numberOfLines={5}
                onLayout={handleLayout('appealGrounds')}
                onFocus={handleInputFocus('appealGrounds')}
                accessibilityLabel="Grounds for appeal"
                showError={!appealGrounds && appellantSignatureError}
              />

              <View style={styles.signatureContainer} onLayout={handleLayout('appellantSignature')}>
                <Text style={inputStyles.label}>Signature of Appellant:</Text>
                <TouchableOpacity
                  style={[styles.signaturePad, appellantSignatureError && styles.signaturePadError]}
                  onPress={() => handleOpenSignaturePad('appeal')}
                  accessibilityLabel="Capture signature for appeal"
                  accessibilityRole="button"
                >
                  {appellantSignatureLoading ? (
                    <ActivityIndicator size="small" color="#015185" />
                  ) : appellantSignature ? (
                    <Image
                      source={{ uri: appellantSignature }}
                      style={styles.capturedSignatureImage}
                      resizeMode="contain"
                      onError={(e) => {
                        console.error('Error loading appellant signature:', e.nativeEvent.error);
                        showToast('error', 'Error', 'Failed to load signature.');
                      }}
                    />
                  ) : (
                    <Text style={styles.signatureText}>Tap to sign</Text>
                  )}
                </TouchableOpacity>
                {/* Removed: {appellantSignatureError && (
                  <Text style={inputStyles.errorText}>Signature is required</Text>
                )} */}
              </View>

              <Text style={[inputStyles.label, { marginTop: width * 0.04 }]}>Date of Appeal:</Text>
              <CustomInputField
                value={appealDateOfAppeal}
                placeholder="Select Date"
                onPress={() => showDatePickerDialog('appealDateOfAppeal')}
                editable={false}
                icon={<MaterialIcons name="calendar-today" size={width * 0.05} color="#333" />}
                onIconPress={() => showDatePickerDialog('appealDateOfAppeal')}
                onFocus={handleInputFocus('appealDateOfAppeal')}
                onLayout={handleLayout('appealDateOfAppeal')}
                isRequired
                showError={!appealDateOfAppeal && appellantSignatureError}
                errorMessage="Appeal date is required"
              />
            </>
          )}

          <View style={styles.buttonContainer}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <Text style={styles.buttonText}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.saveButton}
              onPress={handleSubmit}
              accessibilityLabel="Submit application"
              accessibilityRole="button"
            >
              <Text style={styles.buttonText}>Submit</Text>
            </TouchableOpacity>
          </View>

          <View style={{ height: width * 0.1 }} />
        </ScrollView>
      </KeyboardAvoidingView>

      {showDatePicker && (
        <View style={styles.datePickerContainer}>
          <DateTimePicker
            testID="dateTimePicker"
            value={datePickerValue}
            mode="date"
            display={Platform.OS === 'ios' ? 'inline' : 'default'}
            onChange={onDateChange}
            maximumDate={datePickerMaxDate}
            accentColor="#025383"
            textColor="#333"
            style={styles.datePicker}
          />
          {Platform.OS === 'ios' && (
            <TouchableOpacity
              style={styles.datePickerCloseButton}
              onPress={() => setShowDatePicker(false)}
              accessibilityLabel="Close date picker"
              accessibilityRole="button"
            >
              <Text style={styles.datePickerCloseText}>Done</Text>
            </TouchableOpacity>
          )}
        </View>
      )}
      <Toast />
    </View>
  );
};

// Styles remain unchanged as they are extensive and no modifications are needed


const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: width * 0.01,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputContainer: {
    position: 'relative',
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    backgroundColor: '#fff',
  },
  noInputContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: width * 0.02,
  },
  input: {
    flex: 1,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.035,
    fontSize: width * 0.04,
  },
  inputError: {
    borderColor: 'red',
  },
  disabledInput: {
    backgroundColor: '#f5f5f5',
  },
  noBackground: {
    backgroundColor: 'transparent',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: width * 0.005,
    marginLeft: width * 0.005,
  },
  inputIcon: {
    position: 'absolute',
    right: width * 0.03,
    top: '50%',
    transform: [{ translateY: -width * 0.03 }],
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  selectText: {
    fontSize: width * 0.04,
    color: '#015185',
    marginRight: width * 0.01,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  content: {
    padding: width * 0.04,
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? height * 0.05 : height * 0.02,
    paddingBottom: height * 0.02,
  },
  headerTitle: {
    fontSize: width * 0.05,
    color: '#fff',
    fontWeight: 'bold',
  },
  backButton: {
    alignItems: 'center',
    flex: 1,
    backgroundColor: '#888',
    paddingVertical: width * 0.03,
    borderRadius: width * 0.015,
    marginRight: width * 0.02,
  },
  menuButton: {
    padding: width * 0.02,
  },
  inputRow: {
    marginVertical: width * 0.02,
    zIndex: 6000,
  },
  dropdown: {
    flex: 1,
    height: width * 0.12,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    paddingHorizontal: width * 0.03,
    backgroundColor: '#fff',
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    backgroundColor: '#fff',
    elevation: 10,
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  twoColumnRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: width * 0.04,
  },
  halfWidthInput: {
    width: '48%',
  },
  signatureContainer: {
    marginBottom: width * 0.04,
  },
  signaturePad: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    height: height * 0.15,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f9f9f9',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  signatureText: {
    color: '#888',
    fontSize: width * 0.04,
    fontStyle: 'italic',
  },
  capturedSignatureImage: {
    width: '90%',
    height: '90%',
  },
  sectionTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.02,
    marginTop: width * 0.04,
  },
  table: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.012,
    marginBottom: width * 0.04,
  },
  tableRow: {
    flexDirection: 'row',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  tableHeaderCell: {
    backgroundColor: '#f0f0f0',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.02,
    fontWeight: 'bold',
    color: '#333',
    fontSize: width * 0.035,
    textAlign: 'center',
    borderRightWidth: 1,
    borderRightColor: '#eee',
  },
  tableDataCell: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.01,
    borderRightWidth: 1,
    borderRightColor: '#eee',
    justifyContent: 'center',
  },
  tableCellText: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
  },
  tableInput: {
    flex: 1,
    paddingHorizontal: width * 0.01,
    paddingVertical: width * 0.01,
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    height: '100%',
  },
  tableTotalCell: {
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.02,
    fontWeight: 'bold',
    color: '#015185',
    fontSize: width * 0.04,
    textAlign: 'center',
    borderRightWidth: 1,
    borderRightColor: '#eee',
  },
  dateCell: {
    width: '25%',
  },
  descriptionCell: {
    width: '50%',
  },
  amountCell: {
    width: '25%',
  },
  formTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#333',
    textAlign: 'center',
    marginBottom: width * 0.05,
    textDecorationLine: 'underline',
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: width * 0.01,
    fontWeight: '500',
    marginTop: width * 0.03,
  },
  checkboxGroup: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: width * 0.04,
  },
  checkboxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginRight: width * 0.05,
    marginBottom: width * 0.02,
  },
  checkboxText: {
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.01,
  },
  otherInput: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    width: width * 0.3,
    marginLeft: width * 0.02,
    paddingVertical: 0,
    fontSize: width * 0.04,
    color: '#333',
  },
  appealTopDateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    marginBottom: width * 0.04,
  },
  appealFormLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
    marginRight: width * 0.02,
  },
  appealTopDateInput: {
    width: width * 0.4,
  },
  appealIntroContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    alignItems: 'flex-end',
    marginBottom: width * 0.02,
    justifyContent: 'center',
  },
  appealIntroText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  appealTextInputLine: {
    flex: 1,
    borderBottomWidth: 1,
    borderBottomColor: '#333',
    borderStyle: 'dotted',
    paddingVertical: 0,
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.01,
    marginRight: width * 0.03,
    minWidth: width * 0.2,
  },
  appealSentence: {
    fontSize: width * 0.04,
    color: '#333',
    marginBottom: width * 0.01,
    marginTop: width * 0.02,
  },
  appealSanctionOffenceLabel: {
    fontSize: width * 0.035,
    color: '#888',
    fontStyle: 'italic',
    textAlign: 'center',
    marginBottom: width * 0.01,
  },
  appealTextInputMultiLine: {
    borderWidth: 1,
    borderColor: '#333',
    borderStyle: 'dotted',
    borderRadius: width * 0.012,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.01,
    fontSize: width * 0.04,
    color: '#333',
    textAlignVertical: 'top',
    marginBottom: width * 0.02,
  },
  appealGroundsInput: {
    borderWidth: 1,
    borderColor: '#333',
    borderStyle: 'dotted',
    borderRadius: width * 0.012,
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.01,
    fontSize: width * 0.04,
    color: '#333',
    textAlignVertical: 'top',
    height: height * 0.15,
    marginBottom: width * 0.02,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: width * 0.05,
    backgroundColor: '#f5f5f5',
    paddingVertical: width * 0.02,
    borderRadius: width * 0.015,
  },
  saveButton: {
    flex: 1,
    backgroundColor: '#025383',
    paddingVertical: width * 0.03,
    borderRadius: width * 0.015,
    alignItems: 'center',
  },
  buttonText: {
    fontSize: width * 0.04,
    color: '#fff',
    fontWeight: '600',
  },
  selectedDisplayBox: {
    marginTop: 5,
    padding: 10,
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
  },
  selectedItemsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  selectedItemText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  noSelectionText: {
    fontSize: width * 0.04,
    color: '#888',
    fontStyle: 'italic',
  },
  datePickerContainer: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    backgroundColor: '#fff',
    borderTopLeftRadius: width * 0.04,
    borderTopRightRadius: width * 0.04,
    padding: width * 0.04,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.2,
    shadowRadius: 5,
  },
  datePicker: {
    width: '100%',
    backgroundColor: '#f9f9f9',
  },
  datePickerCloseButton: {
    backgroundColor: '#025383',
    paddingVertical: width * 0.02,
    borderRadius: width * 0.015,
    alignItems: 'center',
    marginTop: width * 0.02,
  },
  datePickerCloseText: {
    fontSize: width * 0.04,
    color: '#fff',
    fontWeight: '600',
  },
  tableDateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.02,
    paddingVertical: width * 0.01,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.008,
    backgroundColor: '#f9f9f9',
    minHeight: width * 0.08,
  },
  tableDateText: {
    fontSize: width * 0.032,
    color: '#333',
    flex: 1,
    textAlign: 'center',
  },
});

export default TeanHrApplicationApply;