import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  Dimensions,
  TouchableOpacity,
  ScrollView,
  Platform,
  Image,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');

const TeamHrApplyHome = () => {
  const navigation = useNavigation();
  const [userName, setUserName] = useState('Team Member');
  const [isMenuOpen, setIsMenuOpen] = useState(false);

  // Fetch user data from Firebase
  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const teamMemberId = await AsyncStorage.getItem('teamMemberId');
        if (teamMemberId) {
          const userRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}`);
          const snapshot = await userRef.once('value');
          const userData = snapshot.val();
          if (userData && userData.fullName) {
            setUserName(userData.fullName);
          } else {
            Toast.show({
              type: 'warning',
              text1: 'Warning',
              text2: 'User name not found in database.',
              position: 'bottom',
            });
          }
        } else {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'Team Member ID not found in storage.',
            position: 'bottom',
          });
        }
      } catch (error) {
        console.error('Error fetching user data:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load user data.',
          position: 'bottom',
        });
      }
    };

    fetchUserData();
  }, []);

  // Handle logout
  const handleLogout = async () => {
    try {
      await AsyncStorage.removeItem('teamMemberId');
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Logged out successfully.',
        position: 'bottom',
      });
      // Navigate to login screen (assuming it's named 'Login')
      navigation.reset({
        index: 0,
        routes: [{ name: 'Login' }],
      });
    } catch (error) {
      console.error('Error during logout:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to log out. Please try again.',
        position: 'bottom',
      });
    }
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
          <MaterialIcons name="arrow-back-ios" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>Team HR Application Home</Text>
        <TouchableOpacity style={styles.menuButton} onPress={() => setIsMenuOpen(!isMenuOpen)}>
          <MaterialIcons name="menu" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
      </LinearGradient>

      {/* Menu Dropdown */}
      {isMenuOpen && (
        <View style={styles.menuDropdown}>
          <TouchableOpacity
            style={styles.menuItem}
            onPress={() => {
              setIsMenuOpen(false);
              navigation.navigate('TeanHrApplicationApply');
            }}
          >
            <Text style={styles.menuItemText}>Apply for Leave/Grant</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.menuItem}
            onPress={() => {
              setIsMenuOpen(false);
              navigation.navigate('TeamHRpreviousAppli');
            }}
          >
            <Text style={styles.menuItemText}>View Past Applications</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.menuItem, styles.logoutItem]}
            onPress={() => {
              setIsMenuOpen(false);
              handleLogout();
            }}
          >
            <Text style={[styles.menuItemText, styles.logoutText]}>Logout</Text>
          </TouchableOpacity>
        </View>
      )}

      {/* Main Content */}
      <LinearGradient
        colors={['#f8f8f8', '#e8ecef']}
        style={styles.contentContainer}
        start={{ x: 0, y: 0 }}
        end={{ x: 0, y: 1 }}
      >
        <ScrollView contentContainerStyle={styles.content}>
          {/* Welcome Section */}
          <View style={styles.welcomeContainer}>
            <Text style={styles.welcomeText}>Welcome, {userName}!</Text>
            <Text style={styles.subText}>
              Manage your HR applications with Team HR.
            </Text>
          </View>

          {/* Quick Actions */}
          <View style={styles.actionsContainer}>
            <Text style={styles.sectionTitle}>Quick Actions</Text>
            <View style={styles.actionsGrid}>
              <TouchableOpacity
                style={styles.actionCard}
                onPress={() => navigation.navigate('TeanHrApplicationApply')}
              >
                <MaterialIcons name="note-add" size={width * 0.08} color="#025383" />
                <Text style={styles.actionText}>Apply for Leave/Grant</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.actionCard}
                onPress={() => navigation.navigate('TeamHRpreviousAppli')}
              >
                <MaterialIcons name="history" size={width * 0.08} color="#025383" />
                <Text style={styles.actionText}>View Past Applications</Text>
              </TouchableOpacity>
            </View>
          </View>

          {/* HR Announcements */}
          <View style={styles.announcementsContainer}>
            <Text style={styles.sectionTitle}>HR Announcements</Text>
            <View style={styles.announcementCard}>
              <Text style={styles.announcementText}>
                Reminder: Submit your annual leave applications in testing.
              </Text>
            </View>
            <View style={styles.announcementCard}>
              <Text style={styles.announcementText}>
                New Policy Update: Funeral grant applications now require additional documentation.
              </Text>
            </View>
          </View>

          {/* Footer */}
          <View style={styles.footer}>
            <Text style={styles.footerText}>Team HR © 2025</Text>
          </View>
        </ScrollView>
      </LinearGradient>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f8f8',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? height * 0.05 : height * 0.02,
    paddingBottom: height * 0.02,
  },
  backButton: {
    padding: width * 0.02,top:"46%"
  },
  headerTitle: {
    fontSize: width * 0.05,
    color: '#fff',
    fontWeight: 'bold',
    top:"46%"
  },
  menuButton: {
    padding: width * 0.02,
  },
  menuDropdown: {
    position: 'absolute',
    top: Platform.OS === 'ios' ? height * 0.12 : height * 0.09,
    right: width * 0.04,
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    zIndex: 1000,
  },
  menuItem: {
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.05,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  menuItemText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '500',
  },
  logoutItem: {
    borderBottomWidth: 0,
  },
  logoutText: {
    color: '#d32f2f',
  },
  contentContainer: {
    flex: 1,
  },
  content: {
    flexGrow: 1,
    padding: width * 0.04,
  },
  welcomeContainer: {
    alignItems: 'center',
    marginBottom: width * 0.06,
    marginTop: width * 0.04,
  },
  welcomeText: {
    fontSize: width * 0.06,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.02,
  },
  subText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    paddingHorizontal: width * 0.05,
  },
  actionsContainer: {
    marginBottom: width * 0.06,
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.03,
  },
  actionsGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
  },
  actionCard: {
    width: '48%',
    backgroundColor: '#fff',
    borderRadius: width * 0.03,
    padding: width * 0.04,
    alignItems: 'center',
    marginBottom: width * 0.04,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  actionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginTop: width * 0.02,
    textAlign: 'center',
    fontWeight: '500',
  },
  announcementsContainer: {
    marginBottom: width * 0.06,
  },
  announcementCard: {
    backgroundColor: '#fff',
    borderRadius: width * 0.03,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 3,
  },
  announcementText: {
    fontSize: width * 0.038,
    color: '#333',
    lineHeight: width * 0.055,
  },
  footer: {
    alignItems: 'center',
    marginTop: width * 0.06,
    marginBottom: width * 0.04,
  },
  footerText: {
    fontSize: width * 0.035,
    color: '#666',
  },
});

export default TeamHrApplyHome;