import React, { useEffect, useState, useRef } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  Image,
  StyleSheet,
  SafeAreaView,
  Animated,
  ActivityIndicator,
  Modal,
  Platform,
  ToastAndroid,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import LinearGradient from 'react-native-linear-gradient';
import { GoogleSignin } from '@react-native-google-signin/google-signin';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Processing Sign Up</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.show(`${title}: ${message}`, ToastAndroid.LONG);
  } else {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

export default function AdminSignUpScreen() {
  const navigation = useNavigation();
  const [isLoadingGoogle, setIsLoadingGoogle] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const networkErrorAnim = useRef(new Animated.Value(-50)).current;

  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -50,
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  // Configure Google Sign-In when the component mounts
  useEffect(() => {
    GoogleSignin.configure({
      webClientId: '749788854975-rhkmepo20c1n038ftnreustj8br62qbj.apps.googleusercontent.com',
      androidClientId: '749788854975-1vqr2rje9a19hl56ohheqvr32hl6045p.apps.googleusercontent.com',
      iosClientId: '749788854975-fsan3f653h4fffue813uqc38h6mu7ipe.apps.googleusercontent.com',
      offlineAccess: true,
    });

    checkNetworkStatus();
  }, []);

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      }
      return true;
    } catch (error) {
      showNetworkError();
      showToast('error', 'Network Error', `Failed to check network status: ${error.message}`);
      return false;
    }
  };

  const handleGoogleSignUp = async () => {
    setIsLoadingGoogle(true);
    setSyncModalVisible(true);
    try {
      // Check network status
      if (!(await checkNetworkStatus())) {
        setIsLoadingGoogle(false);
        setSyncModalVisible(false);
        return;
      }

      // Check if Google Play Services are available (mainly for Android)
      try {
        await GoogleSignin.hasPlayServices({ showPlayServicesUpdateDialog: true });
        console.log('Google Play Services are available');
      } catch (playServicesError) {
        console.error('Google Play Services error:', playServicesError);
        throw new Error('Google Play Services are not available. Please install or update them.');
      }

      // Sign in with Google
      let signInResult;
      try {
        signInResult = await GoogleSignin.signIn();
        console.log('Google Sign-In Response:', JSON.stringify(signInResult, null, 2));
      } catch (signInError) {
        console.error('Google Sign-In failed:', signInError);
        throw new Error(`Failed to sign in with Google: ${signInError.message} (Code: ${signInError.code})`);
      }

      // Handle both new and old Google Sign-In result formats
      let idToken = signInResult.data?.idToken || signInResult.idToken;
      if (!idToken) {
        throw new Error('No ID token found in Google Sign-In response');
      }

      // Create a Google credential with the token
      const googleCredential = auth.GoogleAuthProvider.credential(idToken);

      // Sign in to Firebase with the credential
      const userCredential = await auth().signInWithCredential(googleCredential);
      const userId = userCredential.user.uid;

      // Check if user exists in database
      const userRef = database().ref(`/users/${userId}`);
      const snapshot = await userRef.once('value');
      const isNewUser = !snapshot.exists();

      if (isNewUser) {
        // Create new user entry
        await userRef.set({
          name: userCredential.user.displayName || 'Unknown',
          email: userCredential.user.email,
          loginStatus: true,
          lastLogin: new Date().toISOString(),
          selectedCountry: null,
        });
        showToast('success', 'Sign-Up Success', 'New user created and signed up with Google successfully');
      } else {
        // User already exists, update login status
        await userRef.update({
          loginStatus: true,
          lastLogin: new Date().toISOString(),
        });
        showToast('success', 'Sign-In Success', 'Existing user signed in with Google successfully');
      }

      // Store UID and other data in AsyncStorage
      await AsyncStorage.setItem('userUid', userId);
      await AsyncStorage.setItem('userEmail', userCredential.user.email);
      await AsyncStorage.setItem('Loginsuccess-01', 'true');

      navigation.replace('MainApp');
    } catch (error) {
      console.error('Google Sign-Up error:', {
        message: error.message,
        code: error.code,
        details: JSON.stringify(error, null, 2),
      });
      await GoogleSignin.signOut();
      await auth().signOut();
      const errorMessages = {
        'auth/account-exists-with-different-credential': 'An account already exists with a different credential. Please use a different sign-in method.',
        'auth/invalid-credential': 'Invalid Google credential. Please ensure your app is correctly configured in Firebase.',
        'auth/operation-not-allowed': 'Google Sign-In is not enabled in Firebase. Please enable it in the Firebase Console.',
        10: 'Developer Error: Ensure the SHA-1 fingerprint in Firebase Console matches your app’s debug or release keystore. For debug, run: keytool -list -v -keystore ~/.android/debug.keystore -alias androiddebugkey -storepass android -keypass android. Also, verify the package name (com.titan_drilling) and ensure the Google Sign-In API is enabled in Google Cloud Console.',
        'auth/network-request-failed': 'Network error: Please check your internet connection and try again.',
        default: error.message || 'An unexpected error occurred during Google Sign-Up.',
      };
      showToast('error', 'Google Sign-Up Error', errorMessages[error.code] || errorMessages.default);
    } finally {
      setIsLoadingGoogle(false);
      setSyncModalVisible(false);
    }
  };

  const navigateToLogin = () => {
    navigation.navigate('AdminSignUp');
  };

  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      <View style={styles.contentContainer}>
        {/* Logo and Sign-Up Text */}
        <View style={styles.logoContainer}>
          <Image
            source={require('../../../Images/adminlogin/logo.png')}
            style={styles.logo}
          />
          <Text style={styles.signupText}>Sign Up for Titan Drilling</Text>
        </View>

        {/* Form */}
        <View style={styles.formContainer}>
          {/* Google Sign-Up Button */}
          <TouchableOpacity
            style={styles.googleButton}
            onPress={handleGoogleSignUp}
            disabled={isLoadingGoogle}
          >
            <Image
              source={require('../../../Images/adminlogin/gg_google.png')}
              style={styles.googleIcon}
            />
            {isLoadingGoogle ? (
              <ActivityIndicator size="small" color="#015185" style={{ marginRight: 10 }} />
            ) : (
              <Text style={styles.googleButtonText}>Sign Up Using Google Account</Text>
            )}
 sincer
          </TouchableOpacity>

          {/* Navigate to Login Button with Gradient */}
          <TouchableOpacity style={styles.loginButton} onPress={navigateToLogin}>
            <LinearGradient
              colors={['#015185', '#3481BC']}
              style={styles.gradient}
              start={{ x: 0, y: 0 }}
              end={{ x: 1, y: 0 }}
            >
              <Text style={styles.loginButtonText}>Go to Sign Up</Text>
            </LinearGradient>
          </TouchableOpacity>
        </View>
      </View>

      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
      />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
    paddingHorizontal: 10,
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: 20,
  },
  logo: {
    width: '80%',
    height: 100,
    resizeMode: 'contain',
  },
  signupText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 10,
    textAlign: 'center',
  },
  formContainer: {
    width: '100%',
  },
  googleButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    height: 50,
    borderRadius: 8,
    marginTop: 10,
    marginHorizontal: 10,
    paddingHorizontal: 15,
  },
  googleIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  googleButtonText: {
    color: '#000000',
    fontSize: 16,
    fontWeight: 'bold',
  },
  loginButton: {
    marginTop: 10,
    marginHorizontal: 10,
    borderRadius: 8,
  },
  gradient: {
    height: 50,
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
  },
  loginButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff0000',
    paddingVertical: 10,
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});