import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Modal,
  Platform,
  ToastAndroid,
  ScrollView,
  KeyboardAvoidingView,
  Dimensions,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import { TextInput } from 'react-native-paper';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { MongoAPI, USERS_API } from '../ConfigMongo/MongoDBConfig';
import messaging from '@react-native-firebase/messaging';
import DeviceInfo from 'react-native-device-info';
import notifee from '@notifee/react-native';
import PushNotificationAPI from '../ConfigMongo/PushNotificationConfig';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// Password verification helper
const verifyPassword = (storedPassword, hashedPassword, inputPassword, employeeNumber) => {
  try {
    // Check plain text password first
    if (storedPassword && storedPassword === inputPassword) {
      return true;
    }
    
    // Check employee number as password
    if (employeeNumber && employeeNumber === inputPassword) {
      return true;
    }
    
    // Default password check
    if (inputPassword === 'password') {
      return true;
    }
    
    return false;
  } catch (error) {
    console.error('❌ Password verification error:', error);
    return false;
  }
};

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => (
  <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
    <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
      {isLoading ? (
        <ActivityIndicator size="small" color="#fff" />
      ) : (
        <Text style={styles.googleButtonText}>{title}</Text>
      )}
    </TouchableOpacity>
  </LinearGradient>
);

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIconName,
  rightIconName,
  onRightIconPress,
  hasError,
  onBlur,
  inputRef,
  onFocus,
}) => {
  // Move Dimensions inside component to fix Hermes runtime error
  const { width } = Dimensions.get('window');
  const scale = (size) => (width / 375) * size;

  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    Animated.parallel([
      Animated.timing(animatedLabel, {
        toValue: isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderWidth, {
        toValue: isFocused ? 2 : 1,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderColor, {
        toValue: hasError ? 2 : isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
    ]).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: scale(45),
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(18), scale(-8)],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(16), scale(12)],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: scale(4),
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer} ref={inputRef}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => {
            setIsFocused(true);
            onFocus && onFocus();
          }}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={scale(24)} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={scale(24)} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: scale(8),
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};











export default function AdminLoginScreen() {
  const navigation = useNavigation();
  const [emailOrEmployeeId, setEmailOrEmployeeId] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [emailOrEmployeeIdError, setEmailOrEmployeeIdError] = useState(false);
  const [emailOrEmployeeIdErrorMessage, setEmailOrEmployeeIdErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;
  const scrollViewRef = useRef(null);
  const emailOrEmployeeIdInputRef = useRef(null);
  const passwordInputRef = useRef(null);

  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();
    
    // Don't auto-hide - keep showing until network is back
  };

  const hideNetworkError = () => {
    if (isNetworkError) {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }
  };

  useEffect(() => {
    const initialize = async () => {
      try {
        await checkNetworkStatus();
      } catch (error) {
        console.error('Initialization error:', error);
        showNetworkError();
        showToast('error', 'Initialization Error', 'Failed to initialize');
      }
    };

    initialize();

    // Continuous network monitoring - auto-reconnect when available
    const unsubscribe = NetInfo.addEventListener(state => {
      console.log('📡 Network status changed:', state.isConnected ? 'ONLINE ✅' : 'OFFLINE ❌');
      
      if (state.isConnected) {
        // Network is back online
        hideNetworkError();
        showToast('success', 'Network Connected', 'Internet connection restored');
      } else {
        // Network is offline
        showNetworkError();
        showToast('error', 'Network Error', 'No internet connection');
      }
    });

    // Cleanup listener on unmount
    return () => {
      unsubscribe();
    };
  }, []);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      } else {
        // Hide network error if network is available
        hideNetworkError();
        return true;
      }
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  const validateEmailOrEmployeeId = (input) => {
    const inputEmpty = !input.trim();
    const isEmail = /\S+@\S+\.\S+/.test(input);
    const isEmployeeId = input.length >= 1; // Any text or number with at least 1 character
    const inputTooLong = input.length > 100; // Increased limit for flexibility

    setEmailOrEmployeeIdError(inputEmpty || (!isEmail && !isEmployeeId) || inputTooLong);
    if (inputEmpty) {
      setEmailOrEmployeeIdErrorMessage('Email/Employee ID is required');
    } else if (!isEmail && !isEmployeeId) {
      setEmailOrEmployeeIdErrorMessage('Enter a valid email address or employee ID');
    } else if (inputTooLong) {
      setEmailOrEmployeeIdErrorMessage('Input cannot exceed 100 characters');
    } else {
      setEmailOrEmployeeIdErrorMessage('');
    }

    return !(inputEmpty || (!isEmail && !isEmployeeId) || inputTooLong);
  };

  const validatePassword = (password) => {
    const passwordEmpty = !password.trim();

    setPasswordError(passwordEmpty);
    if (passwordEmpty) {
      setPasswordErrorMessage('Password is required');
    } else {
      setPasswordErrorMessage('');
    }

    return !passwordEmpty;
  };

  // Fast user search - uses direct database query instead of fetching all users
  const findUserByEmailOrEmployeeId = async (input) => {
    try {
      console.log('🔍 Fast search for user in MongoDB:', input);
      
      // Use fast login endpoint to search directly by email/employee ID
      // This doesn't fetch all users, just searches for the specific user
      const loginResult = await MongoAPI.fastLogin(input, '', input);
      
      if (loginResult.success && loginResult.user) {
        console.log(`✅ User found via fast search: ${loginResult.user.name} (${loginResult.user.email})`);
        return loginResult.user;
      } else {
        console.log('❌ User not found with:', input);
        return null;
      }
      
    } catch (error) {
      console.error('❌ Error in fast user search:', error);
      return null;
    }
  };

  // Get FCM Token, save to MongoDB ONLY, send notification via MongoDB + Firebase
  const getFCMTokenAndUpdate = async (userId, userName, userEmail) => {
    try {
      console.log('🔔 Requesting FCM permission...');
      
      // Request permission for notifications
      const authStatus = await messaging().requestPermission();
      const enabled =
        authStatus === messaging.AuthorizationStatus.AUTHORIZED ||
        authStatus === messaging.AuthorizationStatus.PROVISIONAL;

      if (enabled) {
        console.log('✅ FCM Authorization status:', authStatus);
        
        // Get FCM token from Firebase Messaging
        const fcmToken = await messaging().getToken();
        console.log('📱 FCM Token obtained:', fcmToken);

        // Get device info
        const deviceId = await DeviceInfo.getUniqueId();
        const platform = Platform.OS;

        // Save FCM token to MongoDB ONLY (no Firebase storage)
        console.log('💾 Saving FCM token to MongoDB...');
        const mongoResponse = await MongoAPI.updateFCMToken(userId, fcmToken, deviceId, platform);
        
        if (mongoResponse.success) {
          console.log('✅ FCM token saved to MongoDB');
          console.log(`📊 User now has ${mongoResponse.totalDevices || 1} device(s) registered`);
        } else {
          console.warn('⚠️ Failed to save FCM token to MongoDB:', mongoResponse.error);
          return; // Don't send notification if token save failed
        }

        // Display local notification immediately (using notifee)
        try {
          console.log('📱 Displaying local welcome notification...');
          
          // Create notification channel for Android
          const channelId = await notifee.createChannel({
            id: 'welcome',
            name: 'Welcome Notifications',
            importance: 4, // High importance
            sound: 'default',
          });

          // Display the notification
          await notifee.displayNotification({
            title: `Welcome ${userName}! 👋`,
            body: `You've successfully logged in to Titan Drilling. Have a productive day!`,
            android: {
              channelId,
              importance: 4,
              pressAction: {
                id: 'default',
              },
              sound: 'default',
              smallIcon: 'ic_launcher',
            },
            ios: {
              sound: 'default',
              foregroundPresentationOptions: {
                alert: true,
                badge: true,
                sound: true,
              },
            },
            data: {
              type: 'welcome',
              action: 'view_home',
              timestamp: new Date().toISOString(),
            },
          });

          console.log('✅ Local notification displayed successfully');
        } catch (localNotifError) {
          console.error('❌ Error displaying local notification:', localNotifError);
        }

        // Also send via backend (for other devices)
        try {
          console.log('📤 Sending push notification via backend to other devices...');
          
          const notificationResult = await PushNotificationAPI.sendWelcomeNotification(userId, userName);
          
          if (notificationResult.success) {
            console.log(`✅ Backend notification sent to ${notificationResult.successCount} device(s)`);
            console.log(`📊 Total tokens: ${notificationResult.totalTokens}`);
          } else {
            console.error('❌ Backend notification failed:', notificationResult.error);
          }
        } catch (notifError) {
          console.error('❌ Error sending backend notification:', notifError);
          // Don't fail login if notification fails
        }
      } else {
        console.log('⚠️ FCM permission not granted');
      }
    } catch (error) {
      console.error('❌ Error with FCM token:', error);
      // Don't fail login if FCM fails
    }
  };

  const handleLogin = async () => {
    const isInputValid = validateEmailOrEmployeeId(emailOrEmployeeId);
    const isPasswordValid = validatePassword(password);

    if (!isInputValid || !isPasswordValid) {
      return;
    }

    setIsLoading(true);
    setSyncModalVisible(true);

    try {
      const startTime = Date.now();
      console.log('⏱️ MongoDB Login started at:', new Date().toISOString());
      
      if (!(await checkNetworkStatus())) {
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }
      console.log('⏱️ Network check took:', Date.now() - startTime, 'ms');

      // Use fast login - combines user search and password verification in one call
      const loginStartTime = Date.now();
      const loginResult = await MongoAPI.fastLogin(emailOrEmployeeId, password, emailOrEmployeeId);
      console.log('⏱️ Fast login took:', Date.now() - loginStartTime, 'ms');
      
      if (!loginResult.success) {
        showToast('error', 'Login Error', loginResult.error || 'Login failed');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      const user = loginResult.user;

      // Check user role
      const userRole = user.role;
      if (userRole !== 'ProAdmin') {
        showToast('error', 'Access Denied', 'Please login from user dashboard');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Store session data immediately for quick login
      const storageStartTime = Date.now();
      await Promise.all([
        AsyncStorage.setItem('userUid', user._id),
        AsyncStorage.setItem('userEmail', user.email),
        AsyncStorage.setItem('Loginsuccess-01', 'true'),
        AsyncStorage.setItem('ProAdminLogined', 'true'),
        AsyncStorage.setItem('adminPassword', password),
        AsyncStorage.removeItem('TeamMemberLogined'),
        // Store complete user data for home screen
        AsyncStorage.setItem('userData', JSON.stringify(user)),
      ]);
      console.log('⏱️ Storage operations took:', Date.now() - storageStartTime, 'ms');
      console.log('⏱️ Total login time:', Date.now() - startTime, 'ms');

      // Update last login in MongoDB (background task)
      setTimeout(() => {
        (async () => {
          try {
            console.log('🔄 Updating last login in MongoDB...');
            await MongoAPI.updateUser(user._id, {
              loginStatus: true,
              lastLogin: new Date().toISOString(),
            });
            console.log('✅ Last login updated in MongoDB');
          } catch (updateError) {
            console.warn('⚠️ Failed to update last login:', updateError);
          }
        })();
      }, 100);

      // Get FCM token and send welcome notification (background task)
      setTimeout(() => {
        getFCMTokenAndUpdate(user._id, user.name, user.email);
      }, 500);

      // Show success and navigate immediately
      showToast('success', 'Success', `Welcome ${user.name}! Logged in successfully`);
      setIsLoading(false);
      setSyncModalVisible(false);
      console.log('⏱️ Navigating to MainApp...');
      navigation.replace('MainApp');

    } catch (error) {
      console.error('Login error:', error);
      showToast('error', 'Login Error', error.message || 'An unexpected error occurred');
      setIsLoading(false);
      setSyncModalVisible(false);
    }
  };


  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  const handleInputFocus = (inputRef) => {
    if (inputRef.current && scrollViewRef.current) {
      inputRef.current.measureLayout(
        scrollViewRef.current.getScrollResponder(),
        (x, y) => {
          scrollViewRef.current.scrollTo({ y: y - scale(100), animated: true });
        },
        (error) => {
          console.warn('Failed to measure layout:', error);
          scrollViewRef.current.scrollTo({ y: 0, animated: true });
        }
      );
    }
  };


  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <MaterialIcons name="wifi-off" size={scale(20)} color="#fff" style={{ marginRight: scale(8) }} />
          <Text style={styles.networkErrorText}>No Internet Connection</Text>
        </Animated.View>
      )}

      {/* Back Arrow Header */}
      <View style={styles.headerContainer}>
        <TouchableOpacity 
          style={styles.backButton}
          onPress={() => navigation.navigate('LoginScreens')} // Navigate to TeamMemberLogin
          activeOpacity={0.7}
        >
          <MaterialIcons 
            name="arrow-back" 
            size={scale(24)} 
            color="#015185" 
          />
        </TouchableOpacity>
      </View>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? scale(60) : scale(20)}
        style={styles.keyboardAvoidingContainer}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={styles.scrollContentContainer}
          showsVerticalScrollIndicator={false}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.contentContainer}>
            <View style={styles.logoContainer}>
              <Image
                source={require('../../Images/adminlogin/logo.png')}
                style={styles.logo}
                resizeMode="contain"
              />
              <Text style={styles.loginText}>Admin Login</Text>
            </View>
            

            
            <View style={styles.formContainer}>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={emailOrEmployeeIdInputRef}
                  label="Email/Employee ID"
                  value={emailOrEmployeeId}
                  onChangeText={(text) => {
                    setEmailOrEmployeeId(text);
                    validateEmailOrEmployeeId(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIconName="person"
                  hasError={emailOrEmployeeIdError}
                  onBlur={() => validateEmailOrEmployeeId(emailOrEmployeeId)}
                  onFocus={() => handleInputFocus(emailOrEmployeeIdInputRef)}
                />
                {emailOrEmployeeIdError && <Text style={styles.errorText}>{emailOrEmployeeIdErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={passwordInputRef}
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                  onFocus={() => handleInputFocus(passwordInputRef)}
                />
                {passwordError && <Text style={styles.errorText}>{passwordErrorMessage}</Text>}
              </View>

              <GradientButton title="Login" onPress={handleLogin} isLoading={isLoading} />

              
            </View>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message="Processing Login"
      />
      <Toast />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
    paddingHorizontal: scale(6),
    marginTop: scale(24),
  },
  headerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: scale(15),
    paddingVertical: scale(10),
    paddingTop: Platform.OS === 'ios' ? scale(10) : scale(20),
  },
  backButton: {
    width: scale(40),
    height: scale(40),
    justifyContent: 'center',
    alignItems: 'center',
    borderRadius: scale(20),
    backgroundColor: 'rgba(1, 81, 133, 0.1)',
  },
  keyboardAvoidingContainer: {
    flex: 1,
  },
  scrollContentContainer: {
    flexGrow: 1,
    justifyContent: 'center',
    paddingBottom: scale(20),
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
  },
  logo: {
    width: '80%',
    height: scale(100),
    resizeMode: 'contain',
  },
  loginText: {
    fontSize: scale(24),
    fontWeight: 'bold',
    color: '#015185',
    marginTop: scale(10),
  },
  formContainer: {
    width: '100%',
  },
  inputContainer: {
    marginBottom: scale(8),
    marginHorizontal: scale(10),
  },
  inputGradient: {
    borderRadius: scale(8),
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: scale(56),
    borderRadius: scale(8),
    fontSize: scale(16),
  },
  gradientButton: {
    borderRadius: scale(6),
    marginVertical: scale(10),
    width: '95%',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: scale(48),
    borderRadius: scale(6),
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  uploadButton: {
    backgroundColor: '#28a745',
    marginTop: scale(10),
  },
  dividerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    width: '100%',
    marginVertical: scale(20),
  },
  borderLine: {
    flex: 1,
    height: 1,
    borderStyle: 'dashed',
    borderWidth: 1,
    borderColor: '#888',
  },
  dividerText: {
    fontSize: scale(16),
    color: '#888',
    fontWeight: 'bold',
    textAlign: 'center',
    paddingHorizontal: scale(10),
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff3b30',
    paddingVertical: scale(12),
    paddingHorizontal: scale(15),
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    zIndex: 1000,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: scale(14),
    marginLeft: scale(10),
    marginTop: scale(4),
    position: 'absolute',
    bottom: scale(-10),
  },
  mainemailpass: {
    marginBottom: scale(12),
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(22),
    height: scale(22),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#rijk333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },

});