import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  FlatList,
  TouchableOpacity,
  Animated,
  Modal,
  Image,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';

const { width } = Dimensions.get('window');

const DROPDOWN_OPTIONS = [
  {
    label: 'Edit',
    value: 'edit',
    icon: <MaterialIcons name="edit" size={22} color="#2196f3" />,
  },
  {
    label: 'View PDF',
    value: 'view-pdf',
    icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />,
  },
  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#e74c3c" />,
  },
];

const AssignedWotab = () => {
  const navigation = useNavigation();
  const [mixedWorkOrders, setMixedWorkOrders] = useState([]);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const fadeAnim = useRef(new Animated.Value(0)).current;

  // Toast
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Load assigned and in-progress work orders as a single mixed list
  useEffect(() => {
    let workOrdersRef;
    const fetchWorkOrders = async () => {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) return;
      workOrdersRef = database().ref(`/GlobalWorkOrders/admins/${adminUid}`);
      workOrdersRef.on('value', (snapshot) => {
        const data = snapshot.val();
        let workOrderList = [];
        if (data) {
          workOrderList = Object.keys(data)
            .filter((key) => key !== 'lastRequestNumber')
            .map((key) => ({
              id: key,
              ...data[key],
            }));
        }
        // Assigned: assignTo array length > 0, not completed
        const assigned = workOrderList.filter(
          (wo) => Array.isArray(wo.assignTo) && wo.assignTo.length > 0 && wo.status !== 'Complete'
        );
        // In Progress: status contains 'progress', not completed
        const inProgress = workOrderList.filter(
          (wo) => wo.status && wo.status.toLowerCase().includes('progress') && wo.status !== 'Complete'
        );
        // Mix and deduplicate by id
        const mixed = [...assigned, ...inProgress.filter(ip => !assigned.some(a => a.id === ip.id))];
        setMixedWorkOrders(mixed);
      });
    };
    fetchWorkOrders();
    return () => { if (workOrdersRef) workOrdersRef.off('value'); };
  }, []);

  // Dropdown actions
  const handleDropdownAction = async (action, workOrder) => {
    switch (action) {
      case 'edit':
        setDropdownVisible(null);
        navigation.navigate('UpdateCreateWorkOrder', { workOrder });
        break;
      case 'view-pdf':
        handleViewPDF(workOrder.id);
        break;
      case 'delete':
        deleteWorkOrder(workOrder.id);
        break;
      default:
        break;
    }
  };

  // Delete work order
  const deleteWorkOrder = async (workOrderId) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      const workOrderRef = database().ref(`/GlobalWorkOrders/admins/${adminUid}/${workOrderId}`);
      await workOrderRef.remove();
      setMixedWorkOrders((prev) => prev.filter((wo) => wo.id !== workOrderId));
      setDropdownVisible(null);
      showToast('success', 'Success', 'Work Order deleted successfully!');
    } catch (error) {
      console.error('Error deleting work order from Firebase:', error);
      showToast('error', 'Error', 'Failed to delete work order.');
    }
  };

  // View PDF
  const handleViewPDF = async (workOrderId) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      const workOrderRef = database().ref(`/GlobalWorkOrders/admins/${adminUid}/${workOrderId}`);
      const snapshot = await workOrderRef.once('value');
      const workOrderData = snapshot.val();
      if (!workOrderData) {
        showToast('error', 'Error', 'Work Order not found in the database.');
        return;
      }
      const pdfUrl = workOrderData.pdfDownloadUrl;
      if (!pdfUrl) {
        showToast('error', 'Error', 'PDF URL not available for this Work Order.');
        return;
      }
      setDropdownVisible(null);
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.goBack(),
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      showToast('error', 'Error', 'Could not fetch the PDF URL. Please try again.');
    }
  };

  // Toggle dropdown
  const toggleDropdown = (id) => {
    if (dropdownVisible === id) {
      Animated.timing(fadeAnim, {
        toValue: 0,
        duration: 200,
        useNativeDriver: true,
      }).start(() => setDropdownVisible(null));
    } else {
      setDropdownVisible(id);
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 200,
        useNativeDriver: true,
      }).start();
    }
  };

  // Format date
  const formatDate = (date) => {
    if (!date) return 'N/A';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Get priority color
  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'Low':
        return '#2ecc71';
      case 'Medium':
        return '#f1c40f';
      case 'High':
        return '#e67e22';
      case 'Emergency':
        return '#e74c3c';
      default:
        return '#666';
    }
  };

  // Render work order item
  const renderWorkOrder = ({ item, index }) => {
    const dropdownPosition = index > mixedWorkOrders.length - 4 ? 'top' : 'auto';
    return (
      <View style={styles.workOrderItem}>
        <View style={styles.workOrderHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            <Text style={styles.workOrderTitle}>{item.title || 'Untitled Work Order'}</Text>
          </View>
          <TouchableOpacity style={[styles.statusButton, { backgroundColor: getPriorityColor(item.priority) }]}> 
            <Text style={styles.statusButtonText}>{item.priority || 'N/A'}</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.workOrderDetails}>
          {/* Status row before Track row */}
          {item.status && (
            <View style={styles.statusBadgeRow}>
              <Text style={styles.statusBadgeText}>{item.status}</Text>
            </View>
          )}
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValue}>{item.equipment?.equipmentName || 'No Equipment'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Type</Text>
            <Text style={styles.detailValue}>{item.workOrderType || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Start Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.startDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Due Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.dueDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Assigned To</Text>
            <Text style={styles.detailValue}>{item.assignTo?.map((user) => user.name).join(', ') || 'Unassigned'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={[styles.detailLabel, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.detailValue}>{formatDate(item.createdAt)}</Text>
              <Dropdown
                style={styles.dropdown}
                data={DROPDOWN_OPTIONS}
                labelField="label"
                valueField="value"
                value={null}
                onChange={(option) => handleDropdownAction(option.value, item)}
                renderRightIcon={() => (
                  <MaterialIcons name="more-vert" size={22} color="#333" />
                )}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
          <View style={styles.detailItem}>
            <TouchableOpacity
              style={styles.trackButton}
              onPress={() => navigation.navigate('WorkerDashboardhome', {
                workOrderId: item.id,
                adminUid: item.adminUid || '',
                onGoBack: () => navigation.navigate('WorkOrdertab'),
              })}
            >
              <Text style={styles.trackButtonText}>Track</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <FlatList
        data={mixedWorkOrders}
        keyExtractor={(item) => item.id}
        renderItem={renderWorkOrder}
        ListEmptyComponent={<Text style={styles.text}>No assigned or in progress work orders found.</Text>}
        contentContainerStyle={styles.workOrderList}
        style={styles.flatList}
      />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  flatList: {
    flex: 1,
  },
  workOrderList: {
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.3,
  },
  workOrderItem: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  workOrderHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingTop: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  workOrderTitle: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    minWidth: width * 0.08,
    alignItems: 'center',
    justifyContent: 'center',
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  workOrderDetails: {
    paddingHorizontal: width * 0.05,
  },
  detailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
    backgroundColor: '#fff',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  trackButton: {
    backgroundColor: '#015185',
    paddingVertical: 6,
    paddingHorizontal: 16,
    borderRadius: 6,
    marginTop: 8,
    alignSelf: 'flex-start',
  },
  trackButtonText: {
    color: '#fff',
    fontWeight: 'bold',
  },
  text: {
    fontSize: width * 0.045,
    color: '#333',
    textAlign: 'center',
    marginTop: 20,
  },
  sectionHeader: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 16,
    marginBottom: 8,
    marginLeft: 4,
  },
  statusBadgeRow: {
    backgroundColor: '#2ecc71',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    alignSelf: 'flex-start',
    marginBottom: 8,
    marginTop: 8,
  },
  statusBadgeText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
});

export default AssignedWotab;