import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  FlatList,
  Modal,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.48;

// MongoDB API URLs
const MONGODB_SERVICE_SCHEDULE_API_URL = 'http://168.231.113.219:5010';
const MONGODB_ADMIN_USERS_API_URL = 'http://168.231.113.219:5003';

const ServiceScheduleHome = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [schedules, setSchedules] = useState([]);
  const [filteredSchedules, setFilteredSchedules] = useState([]);
  const [activeTab, setActiveTab] = useState('upcoming');
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [selectedSchedule, setSelectedSchedule] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [userDataModalVisible, setUserDataModalVisible] = useState(false);
  const [userData, setUserData] = useState(null);
  const [serviceSchedulePermissions, setServiceSchedulePermissions] = useState([]);
  const [isLoadingPermissions, setIsLoadingPermissions] = useState(true);
  const [deleteConfirmModalVisible, setDeleteConfirmModalVisible] = useState(false);
  const [scheduleToDelete, setScheduleToDelete] = useState(null);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const [localUserUid, setLocalUserUid] = useState(null); // <-- Add state for local UID

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const DROPDOWN_WIDTH = 220;
  const DROPDOWN_ITEM_HEIGHT = width * 0.13;
  const DROPDOWN_ICON_SIZE = 22;
  const DROPDOWN_FONT_SIZE = 16;

  const dropdownData = [
    { label: 'Edit Service', icon: 'build', iconColor: '#2196F3', action: 'handleEditService' },
    { label: 'Mark as Completed', icon: 'check-circle', iconColor: '#4CAF50', action: 'handleMarkCompleted' },
    { label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' },
    { label: 'Inactive', icon: 'block', iconColor: '#FF9800', action: 'handleInactive' },
  ];

  const completedDropdownData = [
    { label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' },
  ];

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const fetchUserData = async () => {
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        showToast('error', 'Error', 'User not authenticated. Please log in again.');
        navigation.navigate('AdminLogin');
        return;
      }

      console.log('Fetching user data for uid:', uid);
      
      // Use MongoAPI like inspection module
      const response = await MongoAPI.getUserById(uid);
      console.log('User response:', response);
      
      if (!response.success || !response.data) {
        console.log('User not found in database');
        showToast('error', 'Error', 'User data not found.');
        setIsLoadingPermissions(false);
        return;
      }

      const data = response.data;
      console.log('User data received:', {
        name: data.name,
        email: data.email,
        countries: data.countries,
        projects: data.projects,
        modules: data.modules
      });
      
      // Normalize modules to array and extract Service Schedule permissions
      const modules = Array.isArray(data.modules) ? data.modules : (data.modules ? Object.values(data.modules) : []);
      
      // Find Service Schedule module - try multiple possible names like inspection module
      const serviceScheduleModule = modules.find(module => 
        module.module && (
          module.module === 'serviceSchedule' ||
          module.module === 'Service Schedule' ||
          module.module.toLowerCase() === 'service schedule' ||
          module.module.toLowerCase() === 'serviceschedule' ||
          module.module.toLowerCase() === 'service_schedule'
        )
      );
      
      console.log('Service Schedule module found:', serviceScheduleModule);
      console.log('All available modules:', modules.map(m => ({ module: m.module, permissions: m.permissions })));
      
      const permissions = serviceScheduleModule ? (Array.isArray(serviceScheduleModule.permissions) ? serviceScheduleModule.permissions : []) : [];
      setServiceSchedulePermissions(permissions);
      setUserData({ ...data, modules, serviceSchedulePermissions: permissions });
      setIsLoadingPermissions(false);
      
      console.log('Service Schedule Permissions:', permissions);
      console.log('All modules:', modules);
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to fetch user data.');
      setIsLoadingPermissions(false);
    }
  };

  // Check if user has specific permission
  const hasPermission = (permission) => {
    const hasSpecificPermission = serviceSchedulePermissions.includes(permission);
    const hasAllPermission = serviceSchedulePermissions.includes('all');
    const result = hasSpecificPermission || hasAllPermission;
    
    console.log(`Permission check for "${permission}":`, {
      serviceSchedulePermissions,
      hasSpecificPermission,
      hasAllPermission,
      result
    });
    
    return result;
  };

  // Check if user can create schedules
  const canCreate = () => hasPermission('create');
  
  // Check if user can edit schedules
  const canEdit = () => hasPermission('edit');
  
  // Check if user can delete schedules
  const canDelete = () => hasPermission('delete');
  
  // Check if user can view schedules
  const canView = () => hasPermission('view');
  
  // Check if user can update schedules
  const canUpdate = () => hasPermission('update');

  const fetchSchedules = async () => {
    try {
      console.log('📋 Fetching service schedules from MongoDB...');
      const isOnline = await checkNetworkStatus();
      let allSchedules = [];

      if (isOnline) {
        // Fetch from MongoDB
        const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules`);
        
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        
        if (result.success && result.data) {
          allSchedules = result.data;
          console.log(`✅ Fetched ${allSchedules.length} schedules from MongoDB`);
          
          // Save to local storage
          await AsyncStorage.setItem('globalServiceSchedules', JSON.stringify(allSchedules));
        }
      } else {
        // Load from local storage
        const storedSchedules = await AsyncStorage.getItem('globalServiceSchedules');
        allSchedules = storedSchedules ? JSON.parse(storedSchedules) : [];
        console.log(`📱 Loaded ${allSchedules.length} schedules from local storage`);
        showToast('info', 'Offline', 'Using offline data. Will sync when online.');
      }

      setSchedules(allSchedules);
      filterSchedules(allSchedules, activeTab, searchQuery);

      // Sync offline schedules if any
      const uid = await AsyncStorage.getItem('userUid');
      if (uid && isOnline) {
        const offlineSchedules = await AsyncStorage.getItem('offlineSchedules');
        if (offlineSchedules) {
          const offlineData = JSON.parse(offlineSchedules);
          if (offlineData.length > 0) {
            await syncOfflineSchedules(uid, offlineData);
          }
        }
      }
    } catch (error) {
      console.error('Error fetching schedules:', error);
      showToast('error', 'Error', 'Failed to fetch schedules.');
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const syncOfflineSchedules = async (uid, offlineSchedules) => {
    try {
      console.log(`🔄 Syncing ${offlineSchedules.length} offline schedules to MongoDB...`);
      
      for (const schedule of offlineSchedules) {
        try {
          const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({ ...schedule, adminUid: uid }),
          });

          if (!response.ok) {
            console.error(`Failed to sync schedule ${schedule.id}: ${response.status}`);
          } else {
            console.log(`✅ Synced schedule ${schedule.id}`);
          }
        } catch (error) {
          console.error(`Error syncing schedule ${schedule.id}:`, error);
        }
      }
      
      await AsyncStorage.setItem('offlineSchedules', JSON.stringify([]));
      showToast('success', 'Success', 'Offline schedules synced to database.');
    } catch (error) {
      console.error('Error syncing offline schedules:', error);
      showToast('error', 'Error', 'Failed to sync offline schedules.');
    }
  };

  useEffect(() => {
    // Load UID from AsyncStorage and fetch data
    const initializeComponent = async () => {
      const uid = await AsyncStorage.getItem('userUid');
      setLocalUserUid(uid);
      
      // Fetch user permissions first
      await fetchUserData();
      
      // Small delay to ensure permissions are loaded
      setTimeout(() => {
        fetchSchedules();
      }, 500);
    };
    
    initializeComponent();
    
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected && state.isInternetReachable) {
        AsyncStorage.getItem('userUid').then((uid) => {
          if (uid) {
            AsyncStorage.getItem('offlineSchedules').then((offlineSchedules) => {
              if (offlineSchedules) {
                const data = JSON.parse(offlineSchedules);
                if (data.length > 0) {
                  syncOfflineSchedules(uid, data);
                }
              }
            });
          }
        });
      }
    });
    
    return () => {
      unsubscribe();
    };
  }, []);

  useEffect(() => {
    // No need to fetch job cards from Firebase anymore
    // Job card info is now stored directly in service schedule records
    if (isFocused) {
      fetchSchedules();
    }
  }, [isFocused]);

  const filterSchedules = (data, tab, query) => {
    let filtered = data;

    // Get user countries and projects
    const userCountries = userData && userData.countries ? (Array.isArray(userData.countries) ? userData.countries : [userData.countries]) : [];
    const userProjects = userData && userData.projects ? (Array.isArray(userData.projects) ? userData.projects : [userData.projects]) : [];
    const onlyMineView = serviceSchedulePermissions.includes('onlyMineView');

    // Filter by country and project
    filtered = filtered.filter(schedule => {
      const countryMatch = userCountries.length === 0 || userCountries.includes(schedule.country);
      const projectMatch = userProjects.length === 0 || userProjects.includes(schedule.project);
      return countryMatch && projectMatch;
    });

    // If onlymineview permission, show only mine (created by me)
    if (onlyMineView && localUserUid) {
      filtered = filtered.filter(schedule => schedule.adminUid === localUserUid);
    }

    // Filter by schedule status and job card status
    filtered = filtered.filter((schedule) => {
      const scheduleStatus = (schedule.status ? schedule.status.toLowerCase().trim() : '');
      const jobCardStatus = (schedule.jobCardStatus ? schedule.jobCardStatus.toLowerCase().trim() : null);
      
      // Debug logging for status filtering
      console.log(`Schedule ${schedule.scheduleId} - Schedule Status: "${scheduleStatus}", Job Card Status: "${jobCardStatus}"`);
      
      if (tab.toLowerCase() === 'running') {
        // Show schedules with job card status 'in progress' or 'in_progress'
        // Handle different possible formats of the status
        const isRunning = jobCardStatus === 'in progress' || 
                         jobCardStatus === 'in_progress' ||
                         jobCardStatus === 'inprogress';
        if (isRunning) {
          console.log(`✅ Schedule ${schedule.scheduleId} shown in RUNNING tab (job card status: "${jobCardStatus}")`);
        }
        return isRunning;
      } else if (tab.toLowerCase() === 'completed') {
        // Show schedules with:
        // 1. Job card status = 'complete' OR
        // 2. Schedule status = 'completed' (even without job card)
        const isCompleted = jobCardStatus === 'complete' || scheduleStatus === 'completed';
        if (isCompleted) {
          console.log(`✅ Schedule ${schedule.scheduleId} shown in COMPLETED tab (${jobCardStatus === 'complete' ? 'job card complete' : 'schedule completed'})`);
        }
        return isCompleted;
      } else if (tab.toLowerCase() === 'upcoming') {
        // Show schedules with:
        // 1. Status = 'upcoming' AND
        // 2. No job card OR job card not in progress/complete AND
        // 3. Schedule status is NOT 'inactive' or 'completed'
        const isUpcoming = scheduleStatus === 'upcoming' && 
                          (!jobCardStatus || (jobCardStatus !== 'in progress' && jobCardStatus !== 'complete')) &&
                          scheduleStatus !== 'inactive' && 
                          scheduleStatus !== 'completed';
        if (isUpcoming) {
          console.log(`✅ Schedule ${schedule.scheduleId} shown in UPCOMING tab`);
        }
        return isUpcoming;
      } else if (tab.toLowerCase() === 'inactive') {
        // Show schedules with schedule status = 'inactive'
        const isInactive = scheduleStatus === 'inactive';
        if (isInactive) {
          console.log(`✅ Schedule ${schedule.scheduleId} shown in INACTIVE tab`);
        }
        return isInactive;
      }
      return false;
    });

    if (query) {
      filtered = filtered.filter(
        (schedule) =>
          (schedule.scheduleId && schedule.scheduleId.toLowerCase().includes(query.toLowerCase())) ||
          (schedule.title && schedule.title.toLowerCase().includes(query.toLowerCase())) ||
          (schedule.equipment?.equipmentName && schedule.equipment.equipmentName.toLowerCase().includes(query.toLowerCase()))
      );
    }

    setFilteredSchedules(filtered);
  };

  const handleSearch = (text) => {
    setSearchQuery(text);
    filterSchedules(schedules, activeTab, text);
  };

  const handleTabChange = (tab) => {
    setActiveTab(tab);
    filterSchedules(schedules, tab, searchQuery);
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    setTimeout(() => {
      fetchSchedules();
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      showToast('success', 'Success', 'Schedules refreshed.');
    }, 2000);
  };

  const handleActionPress = (schedule, event) => {
    setSelectedSchedule(schedule);
    setDropdownVisible(true);

    const { pageX, pageY } = event.nativeEvent;
    const iconHeight = 22;
    const dropdownHeight = DROPDOWN_ITEM_HEIGHT * (activeTab === 'completed' ? completedDropdownData.length : dropdownData.length);
    const screenHeight = Dimensions.get('window').height;

    let top = pageY + 2;
    let openUp = false;
    if (pageY + 2 + dropdownHeight + 20 > screenHeight) {
      top = pageY - dropdownHeight - 2;
      openUp = true;
    }

    setDropdownPosition({ x: pageX - DROPDOWN_WIDTH + iconHeight, y: top, openUp });
  };

  const handleMarkCompleted = async () => {
    if (!canUpdate() && !canEdit()) {
      showToast('error', 'Permission Denied', 'You do not have permission to update schedules.');
      setDropdownVisible(false);
      return;
    }
    
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) throw new Error('User not authenticated');

      setSyncModalVisible(true);
      const isOnline = await checkNetworkStatus();
      
      if (isOnline) {
        const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${selectedSchedule.id}`, {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({ status: 'completed' }),
        });

        if (!response.ok) {
          throw new Error(`Failed to update schedule: ${response.status}`);
        }

        const result = await response.json();
        if (result.success) {
          showToast('success', 'Success', 'Schedule marked as completed.');
          fetchSchedules(); // Refresh the list
        }
      } else {
        showToast('info', 'Offline', 'Status updated locally. Will sync when online.');
      }

      setDropdownVisible(false);
      setSyncModalVisible(false);
    } catch (error) {
      console.error('Error updating schedule status:', error);
      showToast('error', 'Error', 'Failed to mark schedule as completed.');
      setSyncModalVisible(false);
    }
  };

  const handleDelete = async () => {
    if (!canDelete()) {
      showToast('error', 'Permission Denied', 'You do not have permission to delete schedules.');
      setDropdownVisible(false);
      return;
    }
    
    // Show confirmation modal instead of deleting immediately
    setScheduleToDelete(selectedSchedule);
    setDeleteConfirmModalVisible(true);
    setDropdownVisible(false);
  };

  const confirmDelete = async () => {
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) throw new Error('User not authenticated');

      setSyncModalVisible(true);
      const isOnline = await checkNetworkStatus();

      if (isOnline) {
        const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${scheduleToDelete.id}`, {
          method: 'DELETE',
          headers: {
            'Content-Type': 'application/json',
          },
        });

        if (!response.ok) {
          throw new Error(`Failed to delete schedule: ${response.status}`);
        }

        const result = await response.json();
        if (result.success) {
          showToast('success', 'Success', 'Schedule deleted successfully.');
          fetchSchedules(); // Refresh the list
        }
      } else {
        showToast('info', 'Offline', 'Schedule deleted locally. Will sync when online.');
      }

      setDeleteConfirmModalVisible(false);
      setScheduleToDelete(null);
      setSyncModalVisible(false);
    } catch (error) {
      console.error('Error deleting schedule:', error);
      showToast('error', 'Error', 'Failed to delete schedule.');
      setDeleteConfirmModalVisible(false);
      setScheduleToDelete(null);
      setSyncModalVisible(false);
    }
  };

  const cancelDelete = () => {
    setDeleteConfirmModalVisible(false);
    setScheduleToDelete(null);
  };

  const handleInactive = async () => {
    if (!canUpdate() && !canEdit()) {
      showToast('error', 'Permission Denied', 'You do not have permission to update schedules.');
      setDropdownVisible(false);
      return;
    }
    
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) throw new Error('User not authenticated');

      setSyncModalVisible(true);
      const isOnline = await checkNetworkStatus();
      
      if (isOnline) {
        const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${selectedSchedule.id}`, {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({ status: 'inactive' }),
        });

        if (!response.ok) {
          throw new Error(`Failed to update schedule: ${response.status}`);
        }

        const result = await response.json();
        if (result.success) {
          showToast('success', 'Success', 'Schedule marked as inactive.');
          fetchSchedules(); // Refresh the list
        }
      } else {
        showToast('info', 'Offline', 'Status updated locally. Will sync when online.');
      }

      setDropdownVisible(false);
      setSyncModalVisible(false);
    } catch (error) {
      console.error('Error marking schedule as inactive:', error);
      showToast('error', 'Error', 'Failed to mark schedule as inactive.');
      setSyncModalVisible(false);
    }
  };

  const handleActivate = async () => {
    if (!canUpdate() && !canEdit()) {
      showToast('error', 'Permission Denied', 'You do not have permission to update schedules.');
      setDropdownVisible(false);
      return;
    }
    
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) throw new Error('User not authenticated');

      setSyncModalVisible(true);
      const isOnline = await checkNetworkStatus();
      
      if (isOnline) {
        const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/${selectedSchedule.id}`, {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({ status: 'upcoming' }),
        });

        if (!response.ok) {
          throw new Error(`Failed to activate schedule: ${response.status}`);
        }

        const result = await response.json();
        if (result.success) {
          showToast('success', 'Success', 'Schedule activated successfully.');
          fetchSchedules(); // Refresh the list
        }
      } else {
        showToast('info', 'Offline', 'Status updated locally. Will sync when online.');
      }

      setDropdownVisible(false);
      setSyncModalVisible(false);
    } catch (error) {
      console.error('Error activating schedule:', error);
      showToast('error', 'Error', 'Failed to activate schedule.');
      setSyncModalVisible(false);
    }
  };

  const handleEditService = () => {
    if (!canEdit()) {
      showToast('error', 'Permission Denied', 'You do not have permission to edit schedules.');
      setDropdownVisible(false);
      return;
    }
    
    setDropdownVisible(false);
    navigation.navigate('Update_service_Schedule', { schedule: selectedSchedule });
  };

  const getDropdownData = () => {
    const data = [];
    
    if (canEdit()) {
      data.push({ label: 'Edit Service', icon: 'build', iconColor: '#2196F3', action: 'handleEditService' });
    }
    
    if (canUpdate() || canEdit()) {
      data.push({ label: 'Mark as Completed', icon: 'check-circle', iconColor: '#4CAF50', action: 'handleMarkCompleted' });
    }
    
    if (canDelete()) {
      data.push({ label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' });
    }
    
    if (canUpdate() || canEdit()) {
      data.push({ label: 'Inactive', icon: 'block', iconColor: '#FF9800', action: 'handleInactive' });
    }
    
    return data.length > 0 ? data : [{ label: 'No permissions', icon: 'block', iconColor: '#999', action: 'noAction' }];
  };

  const getCompletedDropdownData = () => {
    const data = [];
    
    if (canDelete()) {
      data.push({ label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' });
    }
    
    return data.length > 0 ? data : [{ label: 'No permissions', icon: 'block', iconColor: '#999', action: 'noAction' }];
  };

  const getInactiveDropdownData = () => {
    const data = [];
    
    if (canEdit()) {
      data.push({ label: 'Activate', icon: 'check-circle', iconColor: '#4CAF50', action: 'handleActivate' });
    }
    
    if (canDelete()) {
      data.push({ label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' });
    }
    
    return data.length > 0 ? data : [{ label: 'No permissions', icon: 'block', iconColor: '#999', action: 'noAction' }];
  };

  const UserDataModal = () => (
    <Modal
      visible={userDataModalVisible}
      transparent={true}
      animationType="fade"
      onRequestClose={() => setUserDataModalVisible(false)}
    >
      <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
        <View style={{
          width: '85%',
          backgroundColor: '#fff',
          borderRadius: 18,
          padding: 28,
          alignItems: 'center',
          shadowColor: '#000',
          shadowOffset: { width: 0, height: 2 },
          shadowOpacity: 0.2,
          shadowRadius: 8,
          elevation: 8,
          maxHeight: '80%',
        }}>
          <Text style={{
            fontSize: 22,
            fontWeight: 'bold',
            color: '#025383',
            marginBottom: 18,
          }}>
            User Details
          </Text>
          <View style={{ width: '100%', marginBottom: 18 }}>
            <View style={{ marginBottom: 18 }}>
              <Text style={{ color: '#888', fontSize: 14 }}>Country</Text>
              <Text style={{ fontSize: 17, fontWeight: '600', color: userData && userData.countries && userData.countries.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData && userData.countries && userData.countries.length > 0 ? userData.countries.join(', ') : 'No countries assigned'}</Text>
            </View>
            <View style={{ marginBottom: 18 }}>
              <Text style={{ color: '#888', fontSize: 14 }}>Projects</Text>
              <Text style={{ fontSize: 17, fontWeight: '600', color: userData && userData.projects && userData.projects.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>{userData && userData.projects && userData.projects.length > 0 ? userData.projects.join(', ') : 'No projects assigned'}</Text>
            </View>
            <View style={{ marginBottom: 18 }}>
              <Text style={{ color: '#888', fontSize: 14 }}>Service Schedule Permissions</Text>
              <Text style={{ fontSize: 17, fontWeight: '600', color: serviceSchedulePermissions && serviceSchedulePermissions.length > 0 ? '#025383' : '#FF6B6B', marginTop: 2 }}>
                {serviceSchedulePermissions && serviceSchedulePermissions.length > 0 ? serviceSchedulePermissions.join(', ') : 'No permissions assigned'}
              </Text>
              {serviceSchedulePermissions && serviceSchedulePermissions.length > 0 && (
                <Text style={{ fontSize: 12, color: '#666', marginTop: 4 }}>
                  Debug: {JSON.stringify(serviceSchedulePermissions)}
                </Text>
              )}
            </View>
          </View>
          <TouchableOpacity
            style={{
              marginTop: 8,
              backgroundColor: '#025383',
              borderRadius: 8,
              paddingVertical: 10,
              paddingHorizontal: 32,
            }}
            onPress={() => setUserDataModalVisible(false)}
            accessibilityLabel="Close user details"
            accessibilityRole="button"
          >
            <Text style={{
              color: '#fff',
              fontWeight: 'bold',
              fontSize: 16,
            }}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  const getStatusColor = (status) => {
    switch ((status || '').toLowerCase()) {
      case 'initiated':
        return { bg: '#e0e0e0', text: '#333' };
      case 'in progress':
        return { bg: '#e0f7fa', text: '#3481BC' };
      case 'on hold':
        return { bg: '#fffbe6', text: '#bfa100' };
      case 'complete':
        return { bg: '#e6f4ea', text: '#28a745' };
      case 'rejected':
        return { bg: '#ffe6e6', text: '#e74c3c' };
      default:
        return { bg: '#eee', text: '#888' };
    }
  };

  const handleViewJobCardPDF = async (item) => {
    try {
      // Fetch the job card from MongoDB to get the PDF URL
      const MONGODB_JOBCARD_API_URL = 'http://168.231.113.219:5005';
      const response = await fetch(`${MONGODB_JOBCARD_API_URL}/job-cards/${item.jobCardId}`);
      
      if (!response.ok) {
        showToast('error', 'Error', 'Failed to fetch job card details');
        return;
      }
      
      const result = await response.json();
      
      if (result.success && result.data && result.data.pdfDownloadUrl) {
        navigation.navigate('PdfviewPage', { 
          pdfUrl: result.data.pdfDownloadUrl, 
          jobCardNumber: item.jobCardRequestNumber 
        });
      } else {
        showToast('info', 'Info', 'PDF not available for this job card');
      }
    } catch (error) {
      console.error('Error fetching job card PDF:', error);
      showToast('error', 'Error', 'Failed to load PDF');
    }
  };

  const renderSchedule = ({ item }) => {
    // Use job card info directly from the schedule record
    const hasJobCard = item.jobCardId && item.jobCardRequestNumber;
    const isInactive = item.status && item.status.toLowerCase() === 'inactive';
    
    return (
      <View style={[styles.scheduleCard, isInactive && styles.inactiveScheduleCard]}>
        <View style={styles.rowBetween}>
          <Text style={styles.scheduleId}>{item.scheduleId}</Text>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
            {isInactive && (
              <View style={{ 
                backgroundColor: '#FF9800', 
                borderRadius: 12, 
                paddingHorizontal: 8, 
                paddingVertical: 2,
                marginRight: 8
              }}>
                <Text style={{ color: '#fff', fontSize: 12, fontWeight: 'bold' }}>INACTIVE</Text>
              </View>
            )}
            <Text style={styles.taskCount}>{item.tasks?.length || 1}</Text>
          </View>
        </View>
        {/* Job Card Status Row - show before Country if jobCard exists */}
        {hasJobCard && (
          <View style={styles.scheduleRow}>
            <Text style={styles.label}>Job Card Status</Text>
            <View style={{
              backgroundColor: getStatusColor(item.jobCardStatus).bg,
              borderRadius: 12,
              paddingHorizontal: 12,
              paddingVertical: 4,
              alignSelf: 'flex-start',
            }}>
              <Text style={{
                color: getStatusColor(item.jobCardStatus).text,
                fontWeight: 'bold',
              }}>
                {item.jobCardStatus || 'N/A'}
              </Text>
            </View>
          </View>
        )}
        <View style={styles.scheduleRow}>
          <Text style={styles.label}>Country</Text>
          <Text style={styles.value}>{item.country || 'N/A'}</Text>
        </View>
        <View style={styles.scheduleRow}>
          <Text style={styles.label}>Project</Text>
          <Text style={styles.value}>{item.project || 'N/A'}</Text>
        </View>
        <View style={styles.scheduleRow}>
          <Text style={styles.label}>Title</Text>
          <Text style={styles.value}>{item.title}</Text>
        </View>
        <View style={styles.scheduleRow}>
          <Text style={styles.label}>Equipments</Text>
          <Text style={styles.value}>{item.equipment?.equipmentName || 'N/A'}</Text>
        </View>
        <View style={styles.scheduleRow}>
          <Text style={styles.label}>Due</Text>
          <View style={styles.dueContainer}>
            <View style={styles.dueIconContainer}>
              <MaterialIcons name="access-time" size={16} color="#015185" />
              <Text style={styles.dueText}>at {item.dueHours} hours</Text>
            </View>
            <View style={styles.dueIconContainer}>
              <MaterialIcons name="calendar-today" size={16} color="#015185" />
              <Text style={styles.dueText}>on {item.lastServiceDate}</Text>
            </View>
            <Text style={styles.dueText}>{item.hoursToGo} hours to go</Text>
            <Text style={styles.dueText}>{item.daysToGo} days to go</Text>
          </View>
        </View>
        <View style={styles.scheduleRow}>
          <Text style={styles.label}>Work Order</Text>
          {hasJobCard ? (
            <View style={{ flexDirection: 'column', flex: 1 }}>
              <View style={{ flexDirection: 'row', alignItems: 'center', flexWrap: 'wrap', marginBottom: 8 }}>
                <Text style={{ color: '#015185', fontWeight: 'bold', marginRight: 10, marginBottom: 4 }}>Job Card #: {item.jobCardRequestNumber}</Text>
              </View>
              <View style={{ flexDirection: 'row', alignItems: 'center', flexWrap: 'wrap', marginBottom: 8 }}>
                <TouchableOpacity
                  style={{ backgroundColor: '#3481BC', paddingHorizontal: 12, paddingVertical: 6, borderRadius: 4, marginRight: 8, marginBottom: 4 }}
                  onPress={() => handleViewJobCardPDF(item)}
                >
                  <Text style={{ color: '#fff', fontWeight: 'bold' }}>View PDF</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={{ backgroundColor: '#3481BC', paddingHorizontal: 12, paddingVertical: 6, borderRadius: 4, marginBottom: 4 }}
                  onPress={() => navigation.navigate('WorkerDashboardhome', {
                    workOrderId: item.jobCardId,
                    adminUid: item.adminUid,
                    jobCardNumber: item.jobCardRequestNumber,
                    schedule: item,
                  })}
                >
                  <Text style={{ color: '#fff', fontWeight: 'bold' }}>Track Job Card</Text>
                </TouchableOpacity>
              </View>
              <Text style={{ color: '#015185', fontSize: width * 0.035, marginTop: 4 }}>
                Title: {item.jobCardTitle || 'N/A'}
              </Text>
              <Text style={{ color: '#015185', fontSize: width * 0.035 }}>
                Created: {item.jobCardCreatedAt ? new Date(item.jobCardCreatedAt).toLocaleDateString() : 'N/A'}
              </Text>
            </View>
          ) : (
            <TouchableOpacity
              style={styles.createWorkOrderButton}
              onPress={() => navigation.navigate('ServiceSWorkOrderJob', { schedule: item })}
            >
              <Text style={styles.createWorkOrderText}>Create Job Card</Text>
            </TouchableOpacity>
          )}
          <TouchableOpacity onPress={(e) => handleActionPress(item, e)}>
            <MaterialIcons name="more-vert" size={24} color="#015185" />
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  // Show loading screen while permissions are being fetched
  if (isLoadingPermissions) {
    return (
      <View style={styles.container}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
        >
          <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
          <View style={styles.headerContent}>
            <View style={styles.headerLeft}>
              <TouchableOpacity onPress={() => navigation.goBack()}>
                <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
              </TouchableOpacity>
              <Text style={styles.headerTitle}>Service Schedule</Text>
            </View>
          </View>
        </LinearGradient>
        <View style={styles.loadingContainer}>
          <Text style={styles.loadingText}>Loading permissions...</Text>
        </View>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Service Schedule</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={() => setUserDataModalVisible(true)}>
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>

        <View style={styles.tabContainer}>
          {['Running', 'Upcoming', 'Completed', 'Inactive'].map((tabKey) => (
            <TouchableOpacity
              key={tabKey}
              style={[styles.tabButton, activeTab === tabKey.toLowerCase() && styles.activeTab]}
              onPress={() => handleTabChange(tabKey.toLowerCase())}
            >
              <Text style={[styles.tabText, activeTab === tabKey.toLowerCase() && styles.activeTabText]}>
                {tabKey}
              </Text>
            </TouchableOpacity>
          ))}
        </View>
      </LinearGradient>

      <FlatList
        data={filteredSchedules}
        renderItem={renderSchedule}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.mainContent}
        ListEmptyComponent={<Text style={styles.defaultText}>No schedules found for this tab</Text>}
      />

      {dropdownVisible && (
        <Modal
          visible={dropdownVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setDropdownVisible(false)}
        >
          <TouchableOpacity
            style={styles.dropdownOverlay}
            onPress={() => setDropdownVisible(false)}
            activeOpacity={1}
          >
            <View
              style={[
                styles.dropdownContainer,
                {
                  position: 'absolute',
                  top: dropdownPosition.y + 8,
                  left: dropdownPosition.x,
                },
              ]}
            >
              {(activeTab === 'completed' ? getCompletedDropdownData() : 
                activeTab === 'inactive' ? getInactiveDropdownData() : 
                getDropdownData()).map((option, index, array) => (
                <TouchableOpacity
                  key={index}
                  style={[
                    styles.dropdownItem,
                    index === array.length - 1 && { borderBottomWidth: 0 },
                    option.action === 'noAction' && styles.disabledDropdownItem,
                  ]}
                  onPress={() => {
                    if (option.action === 'noAction') {
                      setDropdownVisible(false);
                      return;
                    }
                    
                    const actions = {
                      handleEdit: () => navigation.navigate('EditServiceSchedule', { schedule: selectedSchedule }),
                      handleEditService,
                      handleMarkCompleted,
                      handleDelete,
                      handleInactive,
                      handleActivate,
                    };
                    actions[option.action]();
                    setDropdownVisible(false);
                  }}
                >
                  <MaterialIcons name={option.icon} size={DROPDOWN_ICON_SIZE} color={option.iconColor} />
                  <Text style={[styles.dropdownText, option.action === 'noAction' && styles.disabledDropdownText]}>{option.label}</Text>
                </TouchableOpacity>
              ))}
            </View>
          </TouchableOpacity>
        </Modal>
      )}

      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={[styles.startButton, !canCreate() && styles.disabledButton]}
            onPress={() => {
              if (canCreate()) {
                navigation.navigate('AddSchedule');
              } else {
                showToast('error', 'Permission Denied', 'You do not have permission to create schedules.');
              }
            }}
            disabled={!canCreate()}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>Add Schedule</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <UserDataModal />

      {/* Delete Confirmation Modal */}
      <Modal visible={deleteConfirmModalVisible} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.deleteConfirmModalContainer}>
            <View style={styles.deleteConfirmModalHeader}>
              <MaterialIcons name="warning" size={24} color="#FF9800" />
              <Text style={styles.deleteConfirmModalTitle}>Confirm Delete</Text>
            </View>
            <View style={styles.deleteConfirmModalContent}>
              <Text style={styles.deleteConfirmModalText}>
                Are you sure you want to delete this schedule?
              </Text>
              {scheduleToDelete && (
                <View style={styles.scheduleToDeleteInfo}>
                  <Text style={styles.scheduleToDeleteLabel}>Schedule ID:</Text>
                  <Text style={styles.scheduleToDeleteValue}>{scheduleToDelete.scheduleId}</Text>
                  <Text style={styles.scheduleToDeleteLabel}>Title:</Text>
                  <Text style={styles.scheduleToDeleteValue}>{scheduleToDelete.title}</Text>
                </View>
              )}
              <Text style={styles.deleteConfirmModalWarning}>
                This action cannot be undone.
              </Text>
            </View>
            <View style={styles.deleteConfirmModalButtons}>
              <TouchableOpacity 
                style={[styles.deleteConfirmModalButton, styles.cancelDeleteButton]} 
                onPress={cancelDelete}
              >
                <Text style={styles.cancelDeleteButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={[styles.deleteConfirmModalButton, styles.confirmDeleteButton]} 
                onPress={confirmDelete}
              >
                <Text style={styles.confirmDeleteButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  tabContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
  },
  tabButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: width * 0.03,
    marginTop: width * 0.02,
  },
  activeTab: {
    borderBottomWidth: 2,
    borderBottomColor: '#fff',
  },
  tabText: {
    color: '#ccc',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  activeTabText: {
    color: '#fff',
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  scheduleCard: {
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  inactiveScheduleCard: {
    backgroundColor: '#f9f9f9',
    borderColor: '#FF9800',
    borderWidth: 2,
    opacity: 0.8,
  },
  rowBetween: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  scheduleId: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#025383',
  },
  taskCount: {
    fontSize: width * 0.04,
    color: '#666',
    backgroundColor: '#f0f0f0',
    borderRadius: 12,
    paddingHorizontal: 8,
    paddingVertical: 2,
  },
  scheduleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  label: {
    fontSize: width * 0.04,
    color: '#666',
    width: '30%',
  },
  value: {
    fontSize: width * 0.04,
    color: '#015185',
    flex: 1,
  },
  dueContainer: {
    flex: 1,
  },
  dueIconContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 4,
  },
  dueText: {
    fontSize: width * 0.04,
    color: '#015185',
    marginLeft: 4,
  },
  createWorkOrderButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    paddingHorizontal: 12,
    paddingVertical: 6,
    marginRight: 8,
  },
  createWorkOrderText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  disabledButton: {
    opacity: 0.5,
  },
  disabledDropdownItem: {
    opacity: 0.5,
  },
  disabledDropdownText: {
    color: '#999',
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.2)',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#eee',
    width: 220,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    paddingVertical: 4,
    zIndex: 999,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: 16,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  disabledDropdownItem: {
    opacity: 0.5,
  },
  disabledDropdownText: {
    color: '#999',
  },
  userDataModalContainer: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    alignSelf: 'center',
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
  },
  userDataModalHeader: {
    padding: width * 0.04,
    borderTopLeftRadius: width * 0.02,
    borderTopRightRadius: width * 0.02,
  },
  userDataModalTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    textAlign: 'center',
  },
  userDataModalContent: {
    padding: width * 0.04,
  },
  userDataRow: {
    flexDirection: 'row',
    marginBottom: width * 0.02,
    alignItems: 'center',
  },
  userDataLabel: {
    fontSize: width * 0.04,
    color: '#666',
    width: '40%',
    fontWeight: 'bold',
  },
  userDataValue: {
    fontSize: width * 0.04,
    color: '#015185',
    flex: 1,
  },
  closeButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.025,
    borderBottomLeftRadius: width * 0.02,
    borderBottomRightRadius: width * 0.02,
    alignItems: 'center',
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  loadingText: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  deleteConfirmModalContainer: {
    width: '85%',
    backgroundColor: '#fff',
    borderRadius: 12,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
  },
  deleteConfirmModalHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  deleteConfirmModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginLeft: 10,
  },
  deleteConfirmModalContent: {
    padding: 20,
  },
  deleteConfirmModalText: {
    fontSize: 16,
    color: '#333',
    marginBottom: 15,
    textAlign: 'center',
  },
  scheduleToDeleteInfo: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 15,
    marginBottom: 15,
  },
  scheduleToDeleteLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
    marginBottom: 4,
  },
  scheduleToDeleteValue: {
    fontSize: 14,
    color: '#015185',
    fontWeight: 'bold',
    marginBottom: 8,
  },
  deleteConfirmModalWarning: {
    fontSize: 14,
    color: '#FF9800',
    fontWeight: '500',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  deleteConfirmModalButtons: {
    flexDirection: 'row',
    borderTopWidth: 1,
    borderTopColor: '#eee',
  },
  deleteConfirmModalButton: {
    flex: 1,
    paddingVertical: 15,
    alignItems: 'center',
  },
  cancelDeleteButton: {
    borderRightWidth: 0.5,
    borderRightColor: '#eee',
  },
  confirmDeleteButton: {
    borderLeftWidth: 0.5,
    borderLeftColor: '#eee',
  },
  cancelDeleteButtonText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '500',
  },
  confirmDeleteButtonText: {
    fontSize: 16,
    color: '#F44336',
    fontWeight: 'bold',
  },
});

export default ServiceScheduleHome;