import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  StatusBar,
  TouchableOpacity,
  Image,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NoticeAPI from '../ConfigMongo/NoticeMongoConfig';
import NetInfo from '@react-native-community/netinfo';

const Mainheader = ({ navigation }) => {
  const TOTAL_HEADER_HEIGHT = 90; // Total height including status bar
  const [unreadCount, setUnreadCount] = useState(0);
  const [unreadNoticeCount, setUnreadNoticeCount] = useState(0);
  const [currentUserId, setCurrentUserId] = useState('');
  const [isOnline, setIsOnline] = useState(true);

  // Check network status
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      const online = state.isConnected && state.isInternetReachable !== false;
      setIsOnline(online);
      console.log('📡 Mainheader network status:', { online, isConnected: state.isConnected, isInternetReachable: state.isInternetReachable });
    });
    
    return () => unsubscribe();
  }, []);

  // Get current user ID and load notification count
  useEffect(() => {
    getCurrentUserId();
  }, []);

  // Listen for notification count changes with real-time polling
  useEffect(() => {
    if (currentUserId && isOnline) {
      // Initial load - only if online
      console.log('📡 Mainheader: Loading notification counts (online)');
      loadNotificationCount();
      loadUnreadNoticeCount();
      
      // Set up real-time listener for Firebase notifications - only if online
      const notificationsRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications`);
      const unsubscribe = notificationsRef.on('value', (snapshot) => {
        if (snapshot.exists()) {
          const notifications = snapshot.val();
          const unread = Object.values(notifications).filter(notification => !notification.isRead).length;
          setUnreadCount(unread);
        } else {
          setUnreadCount(0);
        }
      });

      // Real-time polling for MongoDB notices (every 5 seconds) - only if online
      const noticeInterval = setInterval(() => {
        // Double-check online status before each poll
        NetInfo.fetch().then((state) => {
          const stillOnline = state.isConnected && state.isInternetReachable !== false;
          if (stillOnline) {
            loadUnreadNoticeCount();
          } else {
            console.log('📴 Mainheader: Skipping notice poll - offline');
          }
        });
      }, 5000); // Every 5 seconds for real-time feel

      return () => {
        notificationsRef.off('value', unsubscribe);
        clearInterval(noticeInterval);
      };
    } else if (currentUserId && !isOnline) {
      console.log('📴 Mainheader: Offline - skipping notification loads');
      // Load cached counts from AsyncStorage if available
      AsyncStorage.getItem('cachedUnreadCount').then((cached) => {
        if (cached) {
          const count = parseInt(cached, 10);
          setUnreadNoticeCount(count);
        }
      }).catch(() => {});
    }
  }, [currentUserId, isOnline, loadUnreadNoticeCount]);

  const getCurrentUserId = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (userId) {
        setCurrentUserId(userId);
      }
    } catch (error) {
      console.error('Error getting current user ID:', error);
    }
  };

  const loadNotificationCount = async () => {
    if (!currentUserId) return;
    
    // Check if online before making network call
    const netInfo = await NetInfo.fetch();
    const online = netInfo.isConnected && netInfo.isInternetReachable !== false;
    
    if (!online) {
      console.log('📴 Mainheader: Offline - skipping Firebase notification count');
      // Try to load from cache
      try {
        const cached = await AsyncStorage.getItem('cachedFirebaseNotifCount');
        if (cached) {
          const count = parseInt(cached, 10);
          setUnreadCount(count);
        }
      } catch (cacheError) {}
      return;
    }
    
    try {
      const notificationsRef = database().ref(`/Globalusers/admins/${currentUserId}/notifications`);
      const snapshot = await notificationsRef.once('value');
      
      if (snapshot.exists()) {
        const notifications = snapshot.val();
        const unread = Object.values(notifications).filter(notification => !notification.isRead).length;
        setUnreadCount(unread);
        // Cache for offline use
        await AsyncStorage.setItem('cachedFirebaseNotifCount', String(unread));
      } else {
        setUnreadCount(0);
        await AsyncStorage.setItem('cachedFirebaseNotifCount', '0');
      }
    } catch (error) {
      console.warn('⚠️ Could not load Firebase notification count (possibly offline):', error.message);
      // Try to load from cache
      try {
        const cached = await AsyncStorage.getItem('cachedFirebaseNotifCount');
        if (cached) {
          const count = parseInt(cached, 10);
          setUnreadCount(count);
        }
      } catch (cacheError) {}
    }
  };

  const loadUnreadNoticeCount = React.useCallback(async () => {
    if (!currentUserId) return;
    
    // Check if online before making network call
    const netInfo = await NetInfo.fetch();
    const online = netInfo.isConnected && netInfo.isInternetReachable !== false;
    
    if (!online) {
      console.log('📴 Mainheader: Offline - skipping unread notice count fetch');
      // Try to load from cache
      try {
        const cached = await AsyncStorage.getItem('cachedUnreadNoticeCount');
        if (cached) {
          const count = parseInt(cached, 10);
          setUnreadNoticeCount(count);
          console.log('📦 Loaded cached unread notice count:', count);
        }
      } catch (cacheError) {
        console.warn('Could not load cached notice count:', cacheError);
      }
      return;
    }
    
    try {
      const countResponse = await NoticeAPI.getUnreadCount(currentUserId);
      
      if (countResponse.success) {
        const newCount = countResponse.unreadCount || 0;
        
        // Cache the count for offline use
        await AsyncStorage.setItem('cachedUnreadNoticeCount', String(newCount));
        
        // Only update if count changed to avoid unnecessary re-renders
        setUnreadNoticeCount(prevCount => {
          if (prevCount !== newCount) {
            console.log(`🔔 [Home] Notice count updated: ${prevCount} → ${newCount}`);
            return newCount;
          }
          return prevCount;
        });
      }
    } catch (error) {
      // Silently fail in offline mode - don't show errors
      console.warn('⚠️ [Home] Could not load unread notice count (possibly offline):', error.message);
      
      // Try to load from cache
      try {
        const cached = await AsyncStorage.getItem('cachedUnreadNoticeCount');
        if (cached) {
          const count = parseInt(cached, 10);
          setUnreadNoticeCount(count);
        }
      } catch (cacheError) {}
    }
  }, [currentUserId]);

  const handleNotificationPress = () => {
    navigation.navigate('HeaderWellNotificaton');
  };

  
  return (
    <View style={styles.container}>
      {/* Single Gradient for Status Bar and Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']} // Single gradient across entire header
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        {/* Status Bar */}
        <StatusBar
          translucent
          backgroundColor="transparent" // Transparent to show gradient
          barStyle="light-content" // White icons/text for contrast
        />

        {/* Header Content */}
        <View style={styles.headerContent}>
          {/* Left: Logo */}
          {/* <TouchableOpacity  onPress={() => navigation.toggleDrawer()}> */}
            <TouchableOpacity>
            <Image
              source={require('../../Images/adminhome/homeheadlogo.png')} // Replace with your logo path
              style={styles.logo} 
            />
          </TouchableOpacity>

          {/* Right: Bell Icon and Menu Icon */}
          <View style={styles.rightContainer}>
            <TouchableOpacity 
              style={styles.notificationButton} 
              onPress={handleNotificationPress}
              activeOpacity={0.7}
            >
              <MaterialIcons name="notifications" size={24} color="white" style={styles.icon} />
              {(unreadCount + unreadNoticeCount) > 0 && (
                <View style={styles.notificationBadge}>
                  <Text style={styles.notificationBadgeText}>
                    {(unreadCount + unreadNoticeCount) > 99 ? '99+' : (unreadCount + unreadNoticeCount)}
                  </Text>
                </View>
              )}
            </TouchableOpacity>
            <TouchableOpacity 
              onPress={() => navigation.toggleDrawer()}
              activeOpacity={0.7}
            >
              <Image
                source={require('../../Images/adminhome/menuicon.png')}
                style={styles.logomenu} 
              />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    width: '100%',
  },
  header: {
    width: '100%',
  },
  headerContent: {
    flex: 1, // Takes up the full height minus status bar padding
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingTop: StatusBar.currentHeight || 44, // Ensures content starts below status bar
  },
  logo: {
    width: 100,
    height: 80,
    resizeMode: 'contain',
  },
  rightContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  icon: {
    marginLeft: 15, // Space between bell and menu icons
  },

  logomenu:{
    width: 20,
    height: 30,
    resizeMode: 'contain',
    marginLeft:20
  },
  notificationButton: {
    position: 'relative',
    padding: 8,
  },
  notificationBadge: {
    position: 'absolute',
    top: 2,
    right: 2,
    backgroundColor: '#FF4444',
    borderRadius: 10,
    minWidth: 20,
    height: 20,
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 4,
  },
  notificationBadgeText: {
    color: 'white',
    fontSize: 10,
    fontWeight: 'bold',
  },
});

export default Mainheader;