
import React, { useState, useEffect } from 'react';
import { createBottomTabNavigator } from '@react-navigation/bottom-tabs';
import { createDrawerNavigator } from '@react-navigation/drawer';
import { createStackNavigator } from '@react-navigation/stack';

import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';
import Octicons from 'react-native-vector-icons/Octicons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { View, Text, TouchableOpacity, StyleSheet, Dimensions, Image, Modal, Alert, StatusBar } from 'react-native';
import Animated from 'react-native-reanimated';
import LinearGradient from 'react-native-linear-gradient';
import AsyncStorage from '@react-native-async-storage/async-storage';
import auth from '@react-native-firebase/auth';

import HomeScreen from './HomeScreen';
import Admin_Account from './Admin_Account';
import Sync_Data from './Sync_Data';
import Mainheader from './Mainheader';
import Equipment_one from './EquipmentScreens/Equipment_one';

const Tab = createBottomTabNavigator();
const Drawer = createDrawerNavigator();
const Stack = createStackNavigator();

// Get screen dimensions for responsive design
const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

const DrawerContent = ({ navigation }) => {
  const [userEmail, setUserEmail] = useState('');
  const [userName, setUserName] = useState('');
  const [userRole, setUserRole] = useState('Admin');
  const [showLogoutModal, setShowLogoutModal] = useState(false);

  useEffect(() => {
    const getUserInfo = async () => {
      try {
        const email = await AsyncStorage.getItem('userEmail');
        const name = await AsyncStorage.getItem('userName');
        const role = await AsyncStorage.getItem('userRole');
        
        if (email) setUserEmail(email);
        if (name) setUserName(name);
        if (role) setUserRole(role);
        
        // Fallback to Firebase user data
        const user = auth().currentUser;
        if (user) {
          if (!email && user.email) {
            setUserEmail(user.email);
            await AsyncStorage.setItem('userEmail', user.email);
          }
          if (!name && user.displayName) {
            setUserName(user.displayName);
            await AsyncStorage.setItem('userName', user.displayName);
          }
        }
      } catch (error) {
        console.error('Error getting user info:', error);
      }
    };

    getUserInfo();
  }, []);

  const handleMenuPress = (screenName) => {
    navigation.navigate(screenName);
    navigation.closeDrawer();
  };

  const handleLogoutPress = () => {
    setShowLogoutModal(true);
  };

  const handleLogoutConfirm = async () => {
    try {
      setShowLogoutModal(false);
      
      // Clear all AsyncStorage data
      await AsyncStorage.clear();
      
      // Sign out from Firebase
      await auth().signOut();
      
      // Navigate to login screen
      navigation.reset({
        index: 0,
        routes: [{ name: 'LoginScreens' }],
      });
    } catch (error) {
      console.error('Logout error:', error);
      Alert.alert('Error', 'Failed to logout. Please try again.');
    }
  };

  const handleLogoutCancel = () => {
    setShowLogoutModal(false);
  };

  return (
    <View style={styles.drawerContainer}>
      {/* Status Bar */}
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      
      {/* Gradient Header with Logo */}
      <LinearGradient 
        colors={['#3481BC', '#025383']} 
        style={styles.drawerHeader}
      >
        <View style={styles.logoContainer}>
          <View style={styles.logoWrapper}>
            <Image 
              source={require('../../Images/logo/Titan_logo.jpg')} 
              style={styles.logo}
              resizeMode="contain"
              defaultSource={require('../../Images/logo/Amit-Titan-Drilling-Mobil-logo.png')}
            />
          </View>
          <Text style={styles.userName}>{userName || 'User Name'}</Text>
        </View>
        
        {/* User Info */}
        <View style={styles.userInfoContainer}>
          <Text style={styles.userRole}>{userRole}</Text>
          <Text style={styles.userEmail} numberOfLines={1}>
            {userEmail || 'Loading...'}
          </Text>
        </View>
      </LinearGradient>

      {/* Menu Items */}
      <View style={styles.menuContainer}>
        <TouchableOpacity 
          style={styles.menuItem}
          onPress={() => handleMenuPress('Notice_BoardAC')}
        >
          <MaterialIcons name="tune" size={scale(24)} color="#666" />
          <Text style={styles.menuText}>Notice Board</Text>
          <MaterialIcons name="chevron-right" size={scale(16)} color="#015185" />
        </TouchableOpacity>

        <TouchableOpacity 
          style={styles.menuItem}
          onPress={() => handleMenuPress('Conatct_US_AC')}
        >
          <MaterialIcons name="contact-mail" size={scale(24)} color="#666" />
          <Text style={styles.menuText}>Contact Us</Text>
          <MaterialIcons name="chevron-right" size={scale(16)} color="#015185" />
        </TouchableOpacity>

        <TouchableOpacity 
          style={styles.menuItem}
          onPress={() => handleMenuPress('AboutUsAC')}
        >
          <MaterialIcons name="info" size={scale(24)} color="#666" />
          <Text style={styles.menuText}>About Us</Text>
          <MaterialIcons name="chevron-right" size={scale(16)} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Logout Button */}
      <TouchableOpacity 
        style={styles.logoutButton}
        onPress={handleLogoutPress}
      >
        <MaterialIcons name="logout" size={scale(24)} color="#FF0000" />
        <Text style={styles.logoutText}>LOGOUT</Text>
      </TouchableOpacity>

      {/* Logout Confirmation Modal */}
      <Modal
        visible={showLogoutModal}
        transparent
        animationType="fade"
        onRequestClose={handleLogoutCancel}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Confirm Logout</Text>
            <Text style={styles.modalMessage}>Are you sure you want to logout?</Text>
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={handleLogoutCancel}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.confirmButton]}
                onPress={handleLogoutConfirm}
              >
                <Text style={styles.modalButtonText}>Logout</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const GradientBackground = () => (
  <LinearGradient colors={['#3481BC', '#025383']} style={{ flex: 1 }} />
);

const TabNavigator = () => (
  <Tab.Navigator
    screenOptions={({ route }) => ({
      tabBarIcon: ({ color, size }) => {
        if (route.name === 'Home') {
          return <AntDesign name="home" size={size} color={color} />;
        } else if (route.name === 'Sync Data') {
          return <Octicons name="sync" size={size} color={color} />;
        } else if (route.name === 'Account') {
          return <Ionicons name="person-outline" size={size} color={color} />;
        }
      },
      tabBarActiveTintColor: 'white',
      tabBarInactiveTintColor: 'white',
      tabBarLabelStyle: { fontSize: 11, fontWeight: 'bold' },
      tabBarStyle: { 
        backgroundColor: 'transparent',
        height: 70,
        paddingBottom: 10,
        paddingTop: 5,
        justifyContent: 'flex-start',
        alignItems: 'center',
      },
      tabBarBackground: () => <GradientBackground />,
      tabBarItemStyle: ({ focused }) => ({
        backgroundColor: focused ? '#025383' : 'transparent',
        borderRadius: 20,
        margin: 5,
      }),
      headerShown: false,
    })}
  >
    <Tab.Screen name="Home" component={HomeScreen} />
    <Tab.Screen name="Sync Data" component={Sync_Data} />
    <Tab.Screen name="Account" component={Admin_Account} />
  </Tab.Navigator>
);

// Stack Navigator for Equipment
const EquipmentStack = () => (
  <Stack.Navigator screenOptions={{ headerShown: true }}>
    <Stack.Screen
      name="EquipmentOne"
      component={Equipment_one}
      options={{ title: 'Equipment One' }}
    />
  </Stack.Navigator>
);

// Secondary Stack with HomeScreen and EquipmentOne
const SecondaryStack = () => (
  <Stack.Navigator screenOptions={{ headerShown: true }}>
    <Stack.Screen name="Home" component={HomeScreen} options={{ title: 'Home' }} />
    <Stack.Screen name="EquipmentOne" component={Equipment_one} options={{ title: 'Equipment One' }} />
  </Stack.Navigator>
);

const MainNavigation = () => (
  <Drawer.Navigator
    drawerContent={(props) => <DrawerContent {...props} />}
    screenOptions={{
      drawerPosition: 'left',
      drawerActiveTintColor: '#007AFF',
      drawerInactiveTintColor: 'gray',
      header: ({ navigation }) => <Mainheader navigation={navigation} />,
    }}
  >
    <Drawer.Screen
      name="Main"
      component={TabNavigator}
      options={{
        drawerIcon: ({ color, size }) => <MaterialIcons name="home" size={size} color={color} />,
      }}
    />
    <Drawer.Screen
      name="Equipment"
      component={EquipmentStack}
      options={{
        drawerIcon: ({ color, size }) => <MaterialIcons name="build" size={size} color={color} />,
      }}
    />
    <Drawer.Screen
      name="Secondary"
      component={SecondaryStack}
      options={{
        drawerIcon: ({ color, size }) => <MaterialIcons name="settings" size={size} color={color} />,
      }}
    />
  </Drawer.Navigator>
);

export default MainNavigation;

const styles = StyleSheet.create({
  drawerContainer: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  drawerHeader: {
    paddingTop: 0,
    alignItems: 'center',
    minHeight: scale(200),
    width: '100%',
    paddingBottom: scale(20),
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
    marginHorizontal: scale(20),
    marginTop: scale(60),
  },
  logoWrapper: {
    position: 'relative',
    width: scale(80),
    height: scale(80),
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: scale(10),
  },
  logo: {
    position: 'absolute',
    width: scale(80),
    height: scale(80),
  },
  fallbackIcon: {
    position: 'absolute',
    width: scale(80),
    height: scale(80),
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(255,255,255,0.2)',
    borderRadius: scale(40),
  },
  appName: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: 'white',
    textAlign: 'center',
  },
  userInfoContainer: {
    alignItems: 'center',
    paddingHorizontal: scale(10),
  },
  userRole: {
    fontSize: scale(16),
    color: 'white',
    fontWeight: 'bold',
    marginBottom: scale(5),
  },
  userEmail: {
    fontSize: scale(14),
    color: 'white',
    textAlign: 'center',
    opacity: 0.9,
  },
  menuContainer: {
    flex: 1,
    padding: scale(15),
  },
  menuItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: scale(15),
    paddingHorizontal: scale(15),
    backgroundColor: 'white',
    borderRadius: scale(8),
    marginBottom: scale(10),
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  menuText: {
    flex: 1,
    marginLeft: scale(15),
    fontSize: scale(16),
    color: '#333',
    fontWeight: '500',
  },
  logoutButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: scale(15),
    paddingHorizontal: scale(20),
    backgroundColor: '#FF0000',
    margin: scale(15),
    borderRadius: scale(8),
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  logoutText: {
    color: 'white',
    fontSize: scale(16),
    fontWeight: 'bold',
    marginLeft: scale(8),
  },
  userName: {
    fontSize: scale(18),
    fontWeight: 'bold',
    color: 'white',
    textAlign: 'center',
    marginTop: scale(10),
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  modalContent: {
    backgroundColor: 'white',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    width: '80%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    elevation: 5,
  },
  modalTitle: {
    fontSize: scale(20),
    fontWeight: 'bold',
    marginBottom: scale(10),
    color: '#333',
  },
  modalMessage: {
    fontSize: scale(16),
    color: '#666',
    textAlign: 'center',
    marginBottom: scale(20),
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    width: '100%',
  },
  modalButton: {
    paddingVertical: scale(10),
    paddingHorizontal: scale(20),
    borderRadius: scale(8),
  },
  cancelButton: {
    backgroundColor: '#ccc',
    borderWidth: 1,
    borderColor: '#ccc',
  },
  confirmButton: {
    backgroundColor: '#FF0000',
    borderWidth: 1,
    borderColor: '#FF0000',
  },
  modalButtonText: {
    color: 'white',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
});