# Inspection Report Email Notification System

This module provides comprehensive email notification functionality for the Titan Drilling Inspection Report system. It automatically sends emails to inspection creators and supervisors when reports are created, updated, or completed.

## Features

- **Automatic Email Notifications**: Sends emails to creators and supervisors automatically
- **Role-Based Recipients**: Targets supervisors based on country and project access
- **Professional Email Templates**: Beautiful, responsive HTML email templates
- **Email Queue Management**: Handles email queuing and retry logic
- **Database Integration**: Stores email records in `GlobalInspectionEmails` database
- **Configurable Templates**: Easy to customize email content and styling
- **Validation & Security**: Input validation and XSS protection

## File Structure

```
ModulesNotificationEmail/
├── InspectionEmailService.js    # Main email service class
├── EmailConfig.js               # Configuration and templates
├── EmailUtils.js                # Utility functions
├── index.js                     # Module exports
└── README.md                    # This documentation
```

## Quick Start

### 1. Import the Email Service

```javascript
import { sendInspectionEmails, emailService, InspectionEmailUtil } from './ModulesNotificationEmail';
```

### 2. Send Inspection Report Emails

```javascript
// After creating an inspection report
const reportData = {
  id: 'inspection_123',
  step1: {
    requestNumber: 'ZAM-KOB-IR-001',
    country: 'Zambia',
    project: 'Kobold',
    inspector: 'John Doe',
    selectedEquipment: { equipmentName: 'Drill Rig A' }
  },
  step3: {
    equipmentStatus: 'Operational',
    faultyItems: 0,
    createdAt: new Date().toISOString()
  }
};

const pdfUrl = 'https://example.com/report.pdf';
const signatureUrl = 'https://example.com/signature.png';

try {
  await sendInspectionEmails(reportData, pdfUrl, signatureUrl);
  console.log('Emails sent successfully');
} catch (error) {
  console.error('Failed to send emails:', error);
}
```

## How It Works

### 1. Email Recipients

The system automatically determines who should receive emails:

- **Creator Email**: Sent to the user who created the inspection report
- **Supervisor Emails**: Sent to all users with supervisor role who have access to the same country and project

### 2. Email Content

Each email includes:
- Professional HTML formatting with Titan Drilling branding
- Complete report details (request number, country, project, equipment, status, etc.)
- Action items and next steps
- Links to review the report in the system

### 3. Database Storage

All email records are stored in the `GlobalInspectionEmails` database with:
- Recipient information
- Email content and metadata
- Status tracking (queued, sent, failed)
- Retry attempts and timestamps

## Configuration

### Email Service Settings

Edit `EmailConfig.js` to customize:

```javascript
export const EMAIL_CONFIG = {
  service: {
    name: 'Titan Drilling Email Service',
    from: 'noreply@titandrilling.com',
    replyTo: 'support@titandrilling.com',
    supportEmail: 'support@titandrilling.com'
  },
  retry: {
    maxAttempts: 3,
    delayBetweenAttempts: 5000,
    maxDelay: 30000
  }
};
```

### Email Templates

Customize email templates in `EmailConfig.js`:

```javascript
export const DEFAULT_TEMPLATES = {
  inspectionCreated: (data) => `
    // Your custom HTML template
  `,
  supervisorNotification: (data, supervisor) => `
    // Your custom supervisor notification template
  `
};
```

## API Reference

### InspectionEmailService

#### Methods

- `sendInspectionEmails(reportData, pdfUrl, signatureUrl)`
  - Sends emails to creator and supervisors
  - Returns Promise

- `getEmailStatistics()`
  - Returns email statistics (total, sent, failed, queued)
  - Returns Promise with statistics object

- `retryFailedEmails()`
  - Retries all failed emails
  - Returns Promise with retry count

#### Properties

- `emailQueue`: Array of queued emails
- `isProcessing`: Boolean indicating if queue is being processed

### InspectionEmailUtil

#### Static Methods

- `isValidEmail(email)`: Validates email format
- `sanitizeEmailContent(content)`: Removes dangerous HTML
- `formatDateForEmail(date, locale)`: Formats dates for email display
- `generateEmailSignature(user)`: Creates user signature HTML
- `generateButton(text, url, color, size)`: Creates button HTML
- `generateAlertBox(message, type)`: Creates alert box HTML

## Database Schema

### GlobalInspectionEmails

```javascript
{
  "email_id": {
    "id": "unique_email_id",
    "reportId": "inspection_report_id",
    "requestNumber": "ZAM-KOB-IR-001",
    "recipientId": "user_id",
    "recipientEmail": "user@example.com",
    "recipientName": "User Name",
    "recipientRole": "creator|supervisor",
    "emailType": "inspection_created|inspection_notification",
    "status": "queued|sent|failed",
    "country": "Zambia",
    "project": "Kobold",
    "createdAt": "2024-01-01T00:00:00.000Z",
    "updatedAt": "2024-01-01T00:00:00.000Z",
    "attempts": 0,
    "lastAttempt": null,
    "metadata": {
      "equipment": "Drill Rig A",
      "inspector": "John Doe",
      "status": "Operational",
      "faultyItems": 0
    }
  }
}
```

## Integration with Existing Code

### In Summerystep3.js

Add email functionality after report generation:

```javascript
import { sendInspectionEmails } from './ModulesNotificationEmail';

// In handleGenerateReport function, after saving to Firebase
try {
  // ... existing report generation code ...
  
  // Send email notifications
  await sendInspectionEmails(reportData, pdfDownloadUrl, signatureDownloadUrl);
  
  // ... rest of the function ...
} catch (error) {
  console.error('Error in report generation:', error);
}
```

### In Inspection_Report_Home.js

Add email statistics display:

```javascript
import { getEmailStatistics } from './ModulesNotificationEmail';

// Add to component state
const [emailStats, setEmailStats] = useState(null);

// Load email statistics
useEffect(() => {
  const loadEmailStats = async () => {
    try {
      const stats = await getEmailStatistics();
      setEmailStats(stats);
    } catch (error) {
      console.error('Error loading email stats:', error);
    }
  };
  
  loadEmailStats();
}, []);

// Display in UI
{emailStats && (
  <View style={styles.statsContainer}>
    <Text>Email Success Rate: {emailStats.successRate}%</Text>
    <Text>Total Sent: {emailStats.sent}</Text>
    <Text>Failed: {emailStats.failed}</Text>
  </View>
)}
```

## Customization

### Adding New Email Types

1. Add new template to `EmailConfig.js`:

```javascript
export const DEFAULT_TEMPLATES = {
  // ... existing templates ...
  inspectionUpdated: (data) => `
    // Your new template
  `
};
```

2. Add new subject line:

```javascript
export const EMAIL_SUBJECTS = {
  // ... existing subjects ...
  inspectionUpdated: (requestNumber) => `Inspection Report Updated - ${requestNumber}`
};
```

3. Use in service:

```javascript
// In InspectionEmailService.js
async sendUpdateNotification(reportData) {
  const emailContent = DEFAULT_TEMPLATES.inspectionUpdated(reportData);
  // ... send email logic
}
```

### Custom Email Styling

Modify `EMAIL_STYLES` in `EmailConfig.js`:

```javascript
export const EMAIL_STYLES = {
  colors: {
    primary: '#your-primary-color',
    secondary: '#your-secondary-color',
    // ... more colors
  },
  fonts: {
    primary: 'Your Font, sans-serif',
    // ... more fonts
  }
};
```

## Troubleshooting

### Common Issues

1. **Emails not sending**
   - Check network connectivity
   - Verify user permissions and roles
   - Check email queue status

2. **Supervisors not receiving emails**
   - Verify supervisor role assignment
   - Check country/project access permissions
   - Review database user records

3. **Email content issues**
   - Validate report data structure
   - Check template syntax
   - Review email sanitization

### Debug Mode

Enable debug logging:

```javascript
// In InspectionEmailService.js
const DEBUG_MODE = true;

if (DEBUG_MODE) {
  console.log('Email service debug info:', {
    reportData,
    supervisors,
    emailContent
  });
}
```

## Security Considerations

- All email content is sanitized to prevent XSS attacks
- Email addresses are validated before sending
- User permissions are verified before sending supervisor emails
- No sensitive data is logged in production

## Performance

- Email queue processing is asynchronous
- Failed emails are automatically retried
- Database operations are batched for efficiency
- Email content is generated on-demand

## Support

For issues or questions about the email notification system:

1. Check the console logs for error messages
2. Verify database connectivity and permissions
3. Review email configuration settings
4. Contact the development team

## Future Enhancements

- Email delivery tracking and analytics
- Customizable email preferences per user
- Advanced email scheduling
- Integration with external email services (SendGrid, AWS SES)
- Email template builder interface
- Multi-language email support
