import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, StatusBar, ScrollView, TouchableOpacity, Dimensions, Alert, Platform } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Corrective_action_step1 from './Corrective_action_step1';
import Signature_Step2 from './Signature_Step2';
import MachanicSignaturedo from './MachanicSignaturedo';
import OperatorSignaturedo from './OperatorSignaturedo';

const { height } = Dimensions.get('window');
const STATUS_BAR_HEIGHT = Platform.OS === 'ios' ? 44 : StatusBar.currentHeight || 0;
const TOTAL_HEADER_HEIGHT = height * 0.15 - STATUS_BAR_HEIGHT;

const CorrectiveactionHeader = ({ route, navigation }) => {
  const { report } = route.params;
  const [activeStep, setActiveStep] = useState(1);
  const [subItemStatuses, setSubItemStatuses] = useState({});
  const [signatureScreen, setSignatureScreen] = useState(null);
  const [mechanicSignatureUri, setMechanicSignatureUri] = useState(null);
  const [operatorSignatureUri, setOperatorSignatureUri] = useState(null);
  const [hasUnsavedSignature, setHasUnsavedSignature] = useState(false);

  useEffect(() => {
    if (report?.correctiveAction?.subItemStatuses) {
      setSubItemStatuses(report.correctiveAction.subItemStatuses);
    } else {
      setSubItemStatuses({});
    }
  }, [report]);

  const resetSubItemStatuses = () => {
    setSubItemStatuses({});
  };

  const resetAllStates = () => {
    setActiveStep(1);
    setSubItemStatuses({});
    setMechanicSignatureUri(null);
    setOperatorSignatureUri(null);
    setSignatureScreen(null);
    setHasUnsavedSignature(false);
  };

  const steps = [
    {
      id: 1,
      title: 'Step 1',
      subtitle: 'Corrective Action',
      component: (
        <Corrective_action_step1
          report={report}
          setActiveStep={setActiveStep}
          navigation={navigation}
          subItemStatuses={subItemStatuses}
          setSubItemStatuses={setSubItemStatuses}
          resetSubItemStatuses={resetSubItemStatuses}
          onCancel={() => {
            Alert.alert(
              'Cancel',
              'Are you sure you want to go back? Your changes will not be saved.',
              [
                { text: 'No', style: 'cancel' },
                {
                  text: 'Yes',
                    onPress: () => {
                    setShowSuccessModal(false);
                    navigation.reset({
                      index: 1,
                      routes: [
                        { name: 'MainApp' },
                        { name: 'Inspection_Report_Home' },
                      ],
                    });
                  },
                  style: 'destructive',
                },
              ]
            );
          }}
        />
      ),
    },
    {
      id: 2,
      title: 'Step 2',
      subtitle: 'Signature',
      component: (
        <Signature_Step2
          report={report}
          setActiveStep={setActiveStep}
          navigation={navigation}
          subItemStatuses={subItemStatuses}
          setSubItemStatuses={setSubItemStatuses}
          setSignatureScreen={setSignatureScreen}
          mechanicSignatureUri={mechanicSignatureUri}
          setMechanicSignatureUri={setMechanicSignatureUri}
          operatorSignatureUri={operatorSignatureUri}
          setOperatorSignatureUri={setOperatorSignatureUri}
          onDone={() => {
            resetAllStates();
            navigation.navigate('Inspection_Report_Home');
          }}
          onBack={() => {
            Alert.alert(
              'Go Back',
              'Are you sure you want to go back? Your changes may not be saved.',
              [
                { text: 'No', style: 'cancel' },
                {
                  text: 'Yes',
                  onPress: () => {
                    resetAllStates();
                    navigation.navigate('Inspection_Report_Home');
                  },
                  style: 'destructive',
                },
              ]
            );
          }}
        />
      ),
    },
  ];

  const handleBackFromSignature = () => {
    if (hasUnsavedSignature) {
      Alert.alert(
        'Unsaved Changes',
        'You have unsaved signature changes. Are you sure you want to go back?',
        [
          { text: 'Cancel', style: 'cancel' },
          {
            text: 'Go Back',
            onPress: () => {
              setSignatureScreen(null);
              setHasUnsavedSignature(false);
              resetAllStates();
              navigation.navigate('Inspection_Report_Home');
            },
            style: 'destructive',
          },
        ]
      );
    } else {
      setSignatureScreen(null);
      resetAllStates();
      navigation.navigate('Inspection_Report_Home');
    }
  };

  const renderContent = () => {
    if (signatureScreen === 'Mechanic') {
      return (
        <MachanicSignaturedo
          report={report}
          onSave={(signatureUri) => {
            console.log('Mechanic signature saved:', signatureUri);
            setMechanicSignatureUri(signatureUri);
            setSignatureScreen(null);
            setHasUnsavedSignature(false);
          }}
          onBack={handleBackFromSignature}
          setHasUnsavedSignature={setHasUnsavedSignature}
        />
      );
    }
    if (signatureScreen === 'Operator') {
      return (
        <OperatorSignaturedo
          report={report}
          onSave={(signatureUri) => {
            console.log('Operator signature saved:', signatureUri);
            setOperatorSignatureUri(signatureUri);
            setSignatureScreen(null);
            setHasUnsavedSignature(false);
          }}
          onBack={handleBackFromSignature}
          setHasUnsavedSignature={setHasUnsavedSignature}
        />
      );
    }
    return steps.find((step) => step.id === activeStep).component;
  };

  return (
    <View style={styles.container}>
      <StatusBar barStyle="light-content" translucent backgroundColor="transparent" />
      <View style={styles.headerWrapper}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.headerGradient}>
          <View style={styles.headerContent}>
            <TouchableOpacity
              style={styles.homeButton}
              onPress={() => {
                Alert.alert(
                  'Go Back',
                  'Are you sure you want to go back? Your changes may not be saved.',
                  [
                    { text: 'No', style: 'cancel' },
                    {
                      text: 'Yes',
                      onPress: () => {
                        resetAllStates();
                        navigation.navigate('Inspection_Report_Home');
                      },
                      style: 'destructive',
                    },
                  ]
                );
              }}
            >
              <MaterialIcons name="chevron-left" size={22} color="#fff" />
              <Text style={styles.headerHomeText}>Home</Text>
            </TouchableOpacity>
            <View style={styles.stepsRow}>
              <View style={styles.stepContainer}>
                {activeStep === 2 ? (
                  <View style={styles.stepBadgeActive}>
                    <MaterialIcons name="check" size={18} color="#fff" />
                  </View>
                ) : (
                  <View style={styles.stepBadgeActive}>
                    <Text style={styles.stepBadgeTextActive}>1</Text>
                  </View>
                )}
                <View style={styles.stepTextBlock}>
                  <Text style={styles.stepLabelWhite}>Step 1</Text>
                  <Text style={styles.stepSubtitle}>Corrective Action</Text>
                </View>
              </View>
              <View style={styles.stepContainer}>
                <View style={[styles.stepBadgeInactive, activeStep === 2 && { backgroundColor: '#4cd964' }]}>
                  <Text style={[styles.stepBadgeTextInactive, activeStep === 2 && { color: '#fff' }]}>2</Text>
                </View>
                <View style={styles.stepTextBlock}>
                  <Text style={styles.stepLabelWhite}>Step 2</Text>
                  <Text style={styles.stepSubtitle}>Signature</Text>
                </View>
              </View>
            </View>
          </View>
        </LinearGradient>
      </View>
      <View style={styles.mainContent}>{renderContent()}</View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  headerWrapper: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    zIndex: 10,
  },
  headerGradient: {
    width: '100%',
    height: STATUS_BAR_HEIGHT + TOTAL_HEADER_HEIGHT,
  },
  headerContent: {
    flex: 1,
    paddingTop: STATUS_BAR_HEIGHT,
  },
  homeButton: {
    position: 'absolute',
    top: STATUS_BAR_HEIGHT + 10,
    left: 10,
    flexDirection: 'row',
    alignItems: 'center',
    zIndex: 2,
  },
  headerHomeText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 2,
  },
  stepsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'flex-end',
    marginTop: 30,
    width: '100%',
  },
  stepContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 32,
  },
  stepBadgeActive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#4cd964',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeInactive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#e0e6ed',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeTextActive: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepBadgeTextInactive: {
    color: '#3481BC',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepTextBlock: {
    flexDirection: 'column',
    alignItems: 'flex-start',
  },
  stepLabelWhite: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    lineHeight: 18,
  },
  stepSubtitle: {
    color: '#fff',
    fontSize: 12,
    marginTop: -2,
  },
  mainContent: {
    flex: 1,
    marginTop: STATUS_BAR_HEIGHT + TOTAL_HEADER_HEIGHT, // so content starts below header
  },
});

export default CorrectiveactionHeader;