import React, { useState, useImperativeHandle, forwardRef, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  Alert,
  Image,
  KeyboardAvoidingView,
  Platform,
  Modal,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
// Removed Firebase imports - now using MongoDB API
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import Toast from 'react-native-toast-message';
import { useNavigation } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import OfflineIncidentHelper from './OfflineIncidentHelper';

const { width } = Dimensions.get('window');
const IMAGE_BOX_GAP = 8;
const IMAGE_BOX_HORIZONTAL_PADDING = 50;
const IMAGE_BOX_SIZE = (width - IMAGE_BOX_HORIZONTAL_PADDING - 2 * IMAGE_BOX_GAP - 12) / 3;

// Incident Notifications API URL
const INCIDENT_NOTIFICATIONS_API_URL = 'http://168.231.113.219:5021';

// Helper function for fetch with timeout
const fetchWithTimeout = (url, options = {}, timeout = 10000) => {
  return Promise.race([
    fetch(url, options),
    new Promise((_, reject) =>
      setTimeout(() => reject(new Error('Request timeout - please check your internet connection')), timeout)
    ),
  ]);
};

// Helper function to check network connectivity
const checkNetworkConnection = async () => {
  try {
    const netInfo = await NetInfo.fetch();
    console.log('Network status:', netInfo);
    
    if (!netInfo.isConnected) {
      throw new Error('No internet connection. Please check your network settings.');
    }
    
    if (netInfo.isInternetReachable === false) {
      throw new Error('Internet not reachable. Please check your connection.');
    }
    
    return true;
  } catch (error) {
    console.error('Network check error:', error);
    throw error;
  }
};

// SyncModal Component for Animated Loading
const SyncModal = ({ visible }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Generating PDF...</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// SuccessModal Component
const SuccessModal = ({ visible, onClose, onNavigateBack, onViewPDF, pdfUrl }) => {
  const navigation = useNavigation();
  return (
    <Modal visible={visible} animationType="fade" transparent={true} onRequestClose={onClose}>
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>Success</Text>
          <Text style={styles.modalText}>Incident Report Generated and Saved</Text>
          <View style={styles.modalActions}>
            <TouchableOpacity
              style={[styles.modalButton, styles.viewButton]}
              onPress={() => {
                onClose && onClose();


                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'IncodentReport' },
                    ],
                  });
                }}


               
            >
              <Text style={styles.modalButtonText}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.modalButton, styles.viewButton]}
              onPress={() => {
                onClose && onClose();
                onViewPDF && onViewPDF(pdfUrl);
              }}
            >
              <Text style={styles.modalButtonText}>View PDF</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const IncidentReportStep2 = forwardRef(({ currentStep, formData, selectedIncidents, involvesEquipment }, ref) => {
  const navigation = useNavigation();
  const [incidentImages, setIncidentImages] = useState(Array(9).fill(null));
  const [incidentCause, setIncidentCause] = useState('');
  const [equipmentDamage, setEquipmentDamage] = useState('');
  const [additionalComment, setAdditionalComment] = useState('');
  const [immediateCorrectiveAction, setImmediateCorrectiveAction] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [pdfUrl, setPdfUrl] = useState(null);
  const [reportId, setReportId] = useState(null);

  useEffect(() => {
    loadPersistedState();
  }, []);

  const loadPersistedState = async () => {
    try {
      const state = await AsyncStorage.getItem('IncidentReportStep2State');
      if (state) {
        const parsed = JSON.parse(state);
        setIncidentImages(parsed.incidentImages || Array(9).fill(null));
        setIncidentCause(parsed.incidentCause || '');
        setEquipmentDamage(parsed.equipmentDamage || '');
        setAdditionalComment(parsed.additionalComment || '');
        setImmediateCorrectiveAction(parsed.immediateCorrectiveAction || '');
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
    }
  };

  const saveState = async () => {
    try {
      const state = {
        incidentImages,
        incidentCause,
        equipmentDamage,
        additionalComment,
        immediateCorrectiveAction,
      };
      await AsyncStorage.setItem('IncidentReportStep2State', JSON.stringify(state));
    } catch (error) {
      console.error('Error saving state:', error);
    }
  };

  const validateStep2 = () => {
    saveState();
    return true; // No required fields in Step 2
  };

  const requestStoragePermission = async () => {
    try {
      const permissions = Platform.OS === 'android'
        ? Platform.Version >= 33
          ? [PERMISSIONS.ANDROID.READ_MEDIA_IMAGES]
          : [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
        : [];

      if (permissions.length === 0) return true;

      let allGranted = true;
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            Alert.alert('Permission Denied', 'Storage permission is required to save the report.');
          }
        } else if (result === RESULTS.BLOCKED) {
          allGranted = false;
          Alert.alert('Permission Blocked', 'Please enable storage permission in Settings.');
        }
      }
      return allGranted;
    } catch (error) {
      console.error('Error requesting storage permission:', error);
      Alert.alert('Error', 'Failed to request storage permission.');
      return false;
    }
  };

  const uploadImagesToServer = async (imageUris, reportId) => {
    try {
      console.log('Starting image upload to server...');
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('User not authenticated');

      const formData = new FormData();
      
      // Add all images to formData with Android-specific handling
      imageUris.forEach((uri, index) => {
        if (uri) {
          // For Android, ensure proper file URI format
          let fileUri = uri;
          if (Platform.OS === 'android' && !uri.startsWith('file://')) {
            fileUri = 'file://' + uri;
          }
          
          formData.append('images', {
            uri: fileUri,
            type: 'image/jpeg',
            name: `incident_image_${index}_${Date.now()}.jpg`,
          });
        }
      });

      console.log(`Uploading ${imageUris.length} images to server...`);
      console.log('Platform:', Platform.OS);
      
      // Use fetchWithTimeout with 30 second timeout for image upload (larger files)
      const response = await fetchWithTimeout(
        `http://168.231.113.219:5007/incident-reports/${userUid}/${reportId}/upload-images`,
        {
          method: 'POST',
          body: formData,
          // Don't set Content-Type header - let fetch set it automatically with boundary
        },
        30000 // 30 second timeout for images
      );

      console.log('Image upload response status:', response.status);
      
      if (!response.ok) {
        throw new Error(`Server responded with status ${response.status}`);
      }

      const result = await response.json();
      console.log('Image upload result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to upload images');
      }

      return result.imageUrls; // Return all uploaded image URLs
    } catch (error) {
      console.error('Error uploading images:', error);
      console.error('Error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack
      });
      
      if (error.message.includes('timeout')) {
        throw new Error('Image upload timeout. Please check your internet connection and try again.');
      }
      
      if (error.message.includes('Network request failed')) {
        throw new Error('Network error during image upload. Please check:\n1. Internet connection\n2. Server is accessible\n3. File permissions');
      }
      
      throw error;
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const selectedIncidents = report.selectedIncidents || {};

      const formatDate = (date) => {
        if (!date) return 'N/A';
        return new Date(date).toLocaleDateString('en-GB');
      };

      const formatTime = (date) => {
        if (!date) return 'N/A';
        return new Date(date).toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit' });
      };

      // Types of Incident checkboxes (4 columns per row)
      const incidentTypes = [
        'Brake Failure', 'Chemicals Spill', 'Collision', 'Drowned in Water',
        'Electrical Failure', 'Engine Failure', 'Equipment Failure', 'Explosion/fire',
        'Fall Attachments', 'Fall from elevation', 'Leakage in Vehicle', 'Loss of Stability',
        'Oil Spill', 'Person hit by Vehicle', 'Steering Failure', 'Stuck',
        'Swamping', 'Tire Burst', 'Turned Over', 'Other'
      ];
      let incidentTypesHtml = '';
      for (let i = 0; i < incidentTypes.length; i += 4) {
        incidentTypesHtml += '<tr>';
        for (let j = 0; j < 4; j++) {
          const idx = i + j;
          if (idx < incidentTypes.length) {
            const type = incidentTypes[idx];
            incidentTypesHtml += `<td style="padding:2px 6px;"><input type="checkbox" ${selectedIncidents[type] ? 'checked' : ''}/> ${type}</td>`;
          } else {
            incidentTypesHtml += '<td></td>';
          }
        }
        incidentTypesHtml += '</tr>';
      }

      // Severity checkboxes
      const severityOptions = ['High', 'Medium', 'Low'];
      let severityHtml = '<tr>';
      severityOptions.forEach((option) => {
        severityHtml += `<td style="padding:2px 6px;"><input type="checkbox" ${step1.severity === option ? 'checked' : ''}/> ${option}</td>`;
      });
      severityHtml += '</tr>';

      // Images
      let imagesHtml = '';
      const imageUrls = step2.incidentImages || [];
      if (imageUrls.length > 0 && imageUrls.some(Boolean)) {
        imagesHtml = '<div style="display:flex;flex-wrap:wrap;gap:8px;">';
        imageUrls.forEach((url, index) => {
          if (url) {
            imagesHtml += `<div style="width:120px;height:90px;border:1px solid #ccc;display:flex;align-items:center;justify-content:center;margin:4px;"><img src="${url}" style="max-width:100%;max-height:100%;object-fit:contain;"/></div>`;
          }
        });
        imagesHtml += '</div>';
      } else {
        imagesHtml = '<div style="color:#888;">No Images</div>';
      }

      const userName = await AsyncStorage.getItem('userName') || 'N/A';

      // Helper to show dash instead of N/A or empty
      const dashIfNA = (val) => {
        if (val === undefined || val === null || val === '' || val === 'N/A') return '-';
        return val;
      };

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Incident Report</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
            .container { max-width: 900px; margin: 0 auto; padding: 12px; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px; }
            .header h1 { font-size: 22px; font-weight: bold; margin: 0; }
            .logo { font-size: 22px; font-weight: bold; color: #fff;  border-radius: 50%; width: 150px; height: 48px; display: flex; align-items: center; justify-content: center; }
            .section-title { font-weight: bold; font-size: 14px; margin: 10px 0 4px; background-color: #e6e6e6; padding: 4px; border: 1px solid #ccc; }
            table { width: 100%; border-collapse: collapse; margin-bottom: 8px; }
            th, td { border: 1px solid #ccc; padding: 6px; text-align: left; font-size: 13px; }
            th { background-color: #f2f2f2; font-weight: bold; }
            .checkbox-table td { border: none; padding: 2px 6px; font-size: 13px; }
            .checkbox-table input[type="checkbox"] { margin-right: 4px; }
            .text-section { border: 1px solid #ccc; padding: 8px; min-height: 32px; font-size: 13px; margin-bottom: 8px; }
            .footer { text-align: center; margin-top: 10px; font-size: 11px; color: #000; border-top: 1px solid #000; padding-top: 6px; }
            .reported-by { text-align: right; font-size: 13px; margin-top: 8px; }
            .reported-by span { border-bottom: 1px solid #000; padding-bottom: 2px; }
            .logo img { width: 100%; height: 100%; object-fit: contain; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Incident Report</h1>
              <div class="logo">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/></div>
            </div>
            <table>
              <tr>
                <th colspan="2">Vehicle Details</th>
                <th colspan="2">Incident Details</th>
              </tr>
              <tr>
                <td>Vehicle Category</td>
                <td>${dashIfNA(step1.vehicleName)}</td>
                <td>Incident Number</td>
                <td>${dashIfNA(step1.incidentNumber)}</td>
              </tr>
              <tr>
                <td>Vehicle Name</td>
                <td>${dashIfNA(step1.vehicleName)}</td>
                <td>Incident Date</td>
                <td>${dashIfNA(formatDate(step1.incidentDate))}</td>
              </tr>
              <tr>
                <td>Vehicle #</td>
                <td>${dashIfNA(step1.vehicleId)}</td>
                <td>Incident Time</td>
                <td>${dashIfNA(formatTime(step1.incidentDate))}</td>
              </tr>
              <tr>
                <td>VIN #</td>
                <td>-</td>
                <td>Location</td>
                <td>${dashIfNA(step1.project)}</td>
              </tr>
              <tr>
                <td>Operator</td>
                <td>${dashIfNA(step1.incidentDate2)}</td>
                <td>Incident Area</td>
                <td>${dashIfNA(step1.incidentArea)}</td>
              </tr>
              <tr>
                <td>GPS Coordinates</td>
                <td colspan="3">${dashIfNA(step1.coordinates)}</td>
              </tr>
              <tr>
                <td>GPS Address</td>
                <td colspan="3">${dashIfNA(step1.gpsAddress)}</td>
              </tr>
            </table>
            <div class="section-title">Types of Incident</div>
            <table class="checkbox-table">${incidentTypesHtml}</table>
            <div class="section-title">Incident Severity Rating</div>
            <table class="checkbox-table">${severityHtml}</table>
            <div class="section-title">How incident occurred and what were the causes?</div>
            <div class="text-section">${step2.incidentCause || '-'}</div>
            <div class="section-title">What broke on the equipment, list all damages?</div>
            <div class="text-section">${step2.equipmentDamage || '-'}</div>
            <div class="section-title">Additional Comment</div>
            <div class="text-section">${step2.additionalComment || '-'}</div>
            <div class="section-title">Incident Images</div>
            ${imagesHtml}
            <div class="reported-by">
              <span>${userName}</span><br>
              Reported By
            </div>
            <div class="footer">
              Prepared By: Titan Drilling            </div>
          </div>
        </body>
        </html>
      `;

      const safeIncidentId = (report?.step1?.incidentNumber || report?.incidentNumber || 'unknown').toString().replace(/[^a-zA-Z0-9-_]/g, '_');
      const options = {
        html,
        fileName: `Incident_Report_${safeIncidentId}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const saveReportToMongoDB = async (userUid, report, pdfDownloadUrl) => {
    try {
      const reportData = {
        ...report,
        step2: {
          ...report.step2,
          ...(pdfDownloadUrl && { pdfDownloadUrl }),
        },
        // Ensure adminUid is set for permission filtering
        adminUid: userUid,
      };

      console.log('Saving report to MongoDB:', {
        id: reportData.id,
        incidentNumber: reportData.incidentNumber,
        adminUid: reportData.adminUid
      });

      // Use fetchWithTimeout with 15 second timeout
      const response = await fetchWithTimeout(
        `http://168.231.113.219:5007/incident-reports`,
        {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(reportData),
        },
        15000 // 15 second timeout
      );

      console.log('MongoDB save response status:', response.status);
      
      if (!response.ok) {
        throw new Error(`Server responded with status ${response.status}`);
      }
      
      const result = await response.json();
      console.log('MongoDB save result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to save incident report');
      }
      
      console.log('Incident report saved successfully to MongoDB');
      
      if (pdfDownloadUrl) {
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Incident report saved successfully',
        });
      }
    } catch (error) {
      console.error('Error saving report to MongoDB:', error);
      if (error.message.includes('timeout')) {
        throw new Error('Database save timeout. Please check your internet connection and try again.');
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to save incident report',
      });
      throw error;
    }
  };

  const updateReportWithFiles = async (userUid, incidentNumber, imageUrls, pdfDownloadUrl) => {
    try {
      const updateData = {
        'step2.incidentImages': imageUrls,
        ...(pdfDownloadUrl && { 
          pdfDownloadUrl, // Store at root level for easy access
          'step2.pdfDownloadUrl': pdfDownloadUrl // Also store in step2 for backwards compatibility
        }),
      };

      console.log('Updating report with files:', {
        incidentNumber,
        imageCount: imageUrls.filter(url => url).length,
        hasPdf: !!pdfDownloadUrl,
        pdfUrl: pdfDownloadUrl
      });

      // Use fetchWithTimeout with 15 second timeout
      const response = await fetchWithTimeout(
        `http://168.231.113.219:5007/incident-reports/${userUid}/${incidentNumber}`,
        {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(updateData),
        },
        15000 // 15 second timeout
      );

      console.log('Update response status:', response.status);
      
      if (!response.ok) {
        throw new Error(`Server responded with status ${response.status}`);
      }
      
      const result = await response.json();
      console.log('Update result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to update incident report');
      }
      
      console.log('Incident report updated successfully with files. PDF URL:', pdfDownloadUrl);
    } catch (error) {
      console.error('Error updating report with files:', error);
      if (error.message.includes('timeout')) {
        throw new Error('Report update timeout. Please check your internet connection and try again.');
      }
      throw error;
    }
  };

  // Send Incident notifications (push, in-app, email)
  const sendIncidentNotifications = async (incidentData, createdByName, reporterEmail, pdfUrl) => {
    try {
      console.log('📤 Sending incident notifications...');
      console.log('📋 Incident Number:', incidentData.incidentNumber);
      console.log('👤 Reporter:', createdByName);
      console.log('📧 Reporter Email:', reporterEmail);
      console.log('📎 PDF URL:', pdfUrl);
      
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout
      
      const response = await fetch(`${INCIDENT_NOTIFICATIONS_API_URL}/api/incident-notifications/new-incident`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          incidentData,
          createdByName,
          reporterEmail,
          pdfUrl,
        }),
        signal: controller.signal,
      });
      
      clearTimeout(timeoutId);
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ Incident notifications sent successfully:', result);
        console.log(`📊 Recipients: ${result.recipients?.total || 0} users notified`);
        console.log(`📱 Push: ${result.results?.push?.sent || 0} sent`);
        console.log(`📧 Emails: ${result.results?.emails?.sent || 0} sent`);
        return result;
      } else {
        const errorText = await response.text();
        console.error('❌ Incident notifications failed:', errorText);
        return { success: false, error: errorText };
      }
    } catch (error) {
      if (error.name === 'AbortError') {
        console.error('❌ Incident notifications request timed out');
      } else {
        console.error('❌ Error sending incident notifications:', error);
      }
      // Don't fail the report generation if notifications fail
      return { success: false, error: error.message };
    }
  };



  // Generate offline report (save locally without MongoDB)
  const generateOfflineReport = async () => {
    try {
      // Get user authentication
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Load step 1 data from AsyncStorage
      const step1DataStr = await AsyncStorage.getItem('IncidentReportStep1Data');
      if (!step1DataStr) {
        throw new Error('Incident details not found. Please restart the report.');
      }
      const step1Data = JSON.parse(step1DataStr);
      const incidentNumber = step1Data.incidentNumber;
      if (!incidentNumber) {
        throw new Error('Incident number not available.');
      }

      // Load user permissions for offline storage
      let userPermissions = {};
      try {
        const cachedPermissions = await AsyncStorage.getItem('incidentModuleUserInfo');
        if (cachedPermissions) {
          userPermissions = JSON.parse(cachedPermissions);
        }
      } catch (permError) {
        console.warn('⚠️ Could not load permissions for offline report:', permError);
      }

      // Create flattened report structure
      const report = {
        id: incidentNumber,
        _id: incidentNumber,
        incidentNumber: step1Data.incidentNumber,
        incidentDate: step1Data.incidentDate,
        incidentArea: step1Data.incidentArea,
        severity: step1Data.severity,
        incidentCategory: step1Data.incidentCategory,
        country: step1Data.country,
        project: step1Data.project,
        vehicleId: step1Data.vehicleId,
        vehicleName: step1Data.vehicleName,
        personInvolved: step1Data.personInvolved,
        coordinates: step1Data.coordinates,
        gpsAddress: step1Data.gpsAddress,
        selectedIncidents: step1Data.selectedIncidents,
        involvesEquipment: step1Data.involvesEquipment,
        otherIncidentNote: step1Data.otherIncidentNote,
        createdAt: step1Data.createdAt || new Date().toISOString(),
        adminUid: userUid,
        // CRITICAL: Store permissions for offline access
        userPermissions: userPermissions.incidentReportPermissions || [],
        userCountries: userPermissions.countries || [],
        userProjects: userPermissions.projects || [],
        step1: step1Data,
        step2: {
          incidentImages: incidentImages,
          incidentCause,
          equipmentDamage,
          additionalComment,
          immediateCorrectiveAction,
        },
      };

      // Generate PDF locally
      console.log('📄 Generating PDF locally...');
      const pdfPath = await generatePDFWithHtml({ 
        step1: step1Data, 
        step2: report.step2, 
        selectedIncidents: report.selectedIncidents 
      });
      
      // Save PDF path in report
      report.step2.pdfLocalPath = pdfPath;
      // Persist local PDF path for offline access (store Android-safe URI with file:// prefix)
      try {
        const storedPath = Platform.OS === 'android' && !pdfPath.startsWith('file://') ? `file://${pdfPath}` : pdfPath;
        await AsyncStorage.setItem(`incident_pdf_local_${incidentNumber}`, storedPath);
      } catch (e) {
        console.warn('⚠️ Failed to persist local PDF path:', e?.message || e);
      }
      
      // Save to offline storage
      console.log('💾 Saving to offline storage...');
      console.log('   Incident Number:', incidentNumber);
      console.log('   Permissions stored:', report.userPermissions);
      console.log('   Countries stored:', report.userCountries);
      console.log('   Projects stored:', report.userProjects?.length || 0, 'projects');
      
      await OfflineIncidentHelper.saveOfflineReport(report);
      await OfflineIncidentHelper.saveOfflinePDF(incidentNumber, pdfPath);

      // Clear temporary data
      await AsyncStorage.removeItem('IncidentReportStep1Data');
      await AsyncStorage.removeItem('IncidentReportStep2State');

      console.log('✅ Offline incident report saved with permissions:', incidentNumber);

      // Show success message
      Toast.show({
        type: 'success',
        text1: 'Saved Offline',
        text2: `Incident ${incidentNumber} saved locally. Will sync when online.`,
        position: 'top',
        visibilityTime: 4000,
      });

      setTimeout(() => {
        Alert.alert(
          '📴 Saved Offline',
          `Incident report ${incidentNumber} has been saved locally.\n\nIt will automatically sync to MongoDB when you have internet connection.\n\nYou can view the PDF offline.`,
          [
            {
              text: 'View PDF',
              onPress: () => {
                const openUri = Platform.OS === 'android' && !pdfPath.startsWith('file://') ? `file://${pdfPath}` : pdfPath;
                navigation.navigate('PdfviewPage', {
                  pdfUrl: openUri,
                });
              }
            },
            {
              text: 'Go Back',
              onPress: () => {
                navigation.reset({
                  index: 1,
                  routes: [
                    { name: 'MainApp' },
                    { name: 'IncodentReport' },
                  ],
                });
              }
            }
          ]
        );
      }, 500);

    } catch (error) {
      console.error('❌ Error generating offline report:', error);
      throw error;
    }
  };

  const generateReport = async () => {
    if (!validateStep2()) return;

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    
    // Step 1: Check network connectivity FIRST
    console.log('Checking network connectivity...');
    const netInfo = await NetInfo.fetch();
    const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
    
    if (!isOnline) {
      // OFFLINE MODE: Save locally
      console.log('📴 Offline mode detected - saving report locally...');
      try {
        await generateOfflineReport();
        setSyncModalVisible(false);
        return;
      } catch (offlineError) {
        console.error('❌ Offline save failed:', offlineError);
        setSyncModalVisible(false);
        Alert.alert(
          'Error',
          'Failed to save offline report. Please try again.',
          [{ text: 'OK' }]
        );
        return;
      }
    }
    
    // ONLINE MODE: Continue with MongoDB save
    try {
      try {
        await checkNetworkConnection();
        console.log('Network connection verified');
      } catch (networkError) {
        setSyncModalVisible(false);
        Alert.alert(
          'Network Error',
          networkError.message || 'No internet connection. Please check your network settings and try again.',
          [{ text: 'OK' }]
        );
        return;
      }

      // Step 2: Test server connection
      console.log('Testing server connection...');
      try {
        const testResponse = await fetchWithTimeout(
          'http://168.231.113.219:5007/',
          { method: 'GET' },
          5000
        );
        console.log('Server test response status:', testResponse.status);
        
        if (!testResponse.ok) {
          throw new Error(`Server not responding properly (status: ${testResponse.status})`);
        }
      } catch (serverError) {
        console.error('Server connection test failed:', serverError);
        setSyncModalVisible(false);
        Alert.alert(
          'Server Error',
          'Cannot connect to server. Please check:\n1. Your internet connection\n2. Server status\n3. Try again later',
          [{ text: 'OK' }]
        );
        return;
      }

      // Step 3: Get user authentication
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        setSyncModalVisible(false);
        return;
      }

      // Load step 1 data from AsyncStorage
      const step1DataStr = await AsyncStorage.getItem('IncidentReportStep1Data');
      if (!step1DataStr) {
        Alert.alert('Error', 'Incident details not found. Please restart the report.');
        setSyncModalVisible(false);
        return;
      }
      const step1Data = JSON.parse(step1DataStr);
      const incidentNumber = step1Data.incidentNumber;
      if (!incidentNumber) {
        Alert.alert('Error', 'Incident number not available.');
        setSyncModalVisible(false);
        return;
      }
      setReportId(incidentNumber);

      // Create flattened report structure for home page compatibility
      const report = {
        id: incidentNumber,
        _id: incidentNumber,
        // Flatten step1 data to top level for home page compatibility
        incidentNumber: step1Data.incidentNumber,
        incidentDate: step1Data.incidentDate,
        incidentArea: step1Data.incidentArea,
        severity: step1Data.severity,
        incidentCategory: step1Data.incidentCategory,
        country: step1Data.country,
        project: step1Data.project,
        vehicleId: step1Data.vehicleId,
        vehicleName: step1Data.vehicleName,
        personInvolved: step1Data.personInvolved,
        coordinates: step1Data.coordinates,
        gpsAddress: step1Data.gpsAddress,
        selectedIncidents: step1Data.selectedIncidents,
        involvesEquipment: step1Data.involvesEquipment,
        otherIncidentNote: step1Data.otherIncidentNote,
        createdAt: step1Data.createdAt || new Date().toISOString(),
        // Keep nested structure for PDF generation
        step1: step1Data,
        step2: {
          incidentImages: Array(9).fill(null), // Initialize empty, will be updated after upload
          incidentCause,
          equipmentDamage,
          additionalComment,
          immediateCorrectiveAction,
        },
      };

      // First, save the report to MongoDB (without files)
      console.log('Saving initial report to MongoDB...');
      await saveReportToMongoDB(userUid, report, null);

      // Upload images to server (only upload non-null images)
      const imagesToUpload = incidentImages.filter(uri => uri !== null);
      let imageUrls = Array(9).fill(null);
      
      if (imagesToUpload.length > 0) {
        console.log('Uploading images to server...');
        const uploadedUrls = await uploadImagesToServer(imagesToUpload, incidentNumber);
        // Map uploaded URLs back to their original positions
        let urlIndex = 0;
        incidentImages.forEach((uri, index) => {
          if (uri) {
            imageUrls[index] = uploadedUrls[urlIndex];
            urlIndex++;
          }
        });
      }

      // Generate PDF
      console.log('Generating PDF...');
      const pdfPath = await generatePDFWithHtml({ 
        step1: step1Data, 
        step2: { ...report.step2, incidentImages: imageUrls }, 
        selectedIncidents: report.selectedIncidents 
      });
      // Persist local PDF path for offline access (even in online flow)
      try {
        const storedPath = Platform.OS === 'android' && !pdfPath.startsWith('file://') ? `file://${pdfPath}` : pdfPath;
        await AsyncStorage.setItem(`incident_pdf_local_${incidentNumber}`, storedPath);
      } catch (e) {
        console.warn('⚠️ Failed to persist local PDF path (online flow):', e?.message || e);
      }
      
      // Upload PDF to server
      console.log('Uploading PDF to server...');
      console.log('PDF Path:', pdfPath);
      console.log('Platform:', Platform.OS);
      
      const formData = new FormData();
      
      // For Android, ensure proper file URI format
      let pdfUri = pdfPath;
      if (Platform.OS === 'android') {
        // Android needs file:// prefix
        if (!pdfPath.startsWith('file://')) {
          pdfUri = 'file://' + pdfPath;
        }
        console.log('Android PDF URI:', pdfUri);
      }
      
      formData.append('pdf', {
        uri: pdfUri,
        type: 'application/pdf',
        name: `IncidentReport_${incidentNumber}_${Date.now()}.pdf`,
      });

      console.log('FormData prepared for upload');

      // Use fetchWithTimeout with 30 second timeout for PDF upload
      const pdfResponse = await fetchWithTimeout(
        `http://168.231.113.219:5007/incident-reports/${userUid}/${incidentNumber}/upload-pdf`,
        {
          method: 'POST',
          body: formData,
          // Don't set Content-Type header - let the browser/fetch set it automatically with boundary
        },
        30000 // 30 second timeout for PDF
      );

      console.log('PDF upload response status:', pdfResponse.status);
      console.log('PDF upload response OK:', pdfResponse.ok);
      
      if (!pdfResponse.ok) {
        const errorText = await pdfResponse.text().catch(() => 'Unable to read error response');
        console.error('PDF upload error response:', errorText);
        throw new Error(`PDF upload failed with status ${pdfResponse.status}: ${errorText}`);
      }
      
      const pdfResult = await pdfResponse.json();
      console.log('PDF upload result:', pdfResult);
      
      if (!pdfResult.success) {
        throw new Error(pdfResult.error || 'Failed to upload PDF');
      }

      const pdfDownloadUrl = pdfResult.pdfUrl;
      console.log('PDF uploaded successfully. URL:', pdfDownloadUrl);

      // Update the report with final file URLs
      console.log('Updating report with file URLs...');
      await updateReportWithFiles(userUid, incidentNumber, imageUrls, pdfDownloadUrl);

      // Keep local PDF so it can be opened offline later
      // await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));

      // Send notifications after successful save (fire-and-forget)
      setTimeout(async () => {
        try {
          // Get reporter name and email
          const reporterName = step1Data.personInvolved || 'Reporter';
          
          // Get current user's email for confirmation
          let reporterEmail = '';
          try {
            const userData = await AsyncStorage.getItem('userData');
            if (userData) {
              const user = JSON.parse(userData);
              reporterEmail = user.email || '';
            }
          } catch (emailError) {
            console.warn('⚠️ Could not get reporter email:', emailError);
          }
          
          // Prepare incident data for notifications
          const incidentNotificationData = {
            _id: incidentNumber,
            id: incidentNumber,
            incidentNumber: incidentNumber,
            severity: step1Data.severity,
            incidentCategory: step1Data.incidentCategory,
            country: step1Data.country,
            project: step1Data.project,
            incidentArea: step1Data.incidentArea,
          };
          
          console.log('🔔 Sending incident notifications in background...');
          const notifResult = await sendIncidentNotifications(
            incidentNotificationData,
            reporterName,
            reporterEmail,
            pdfDownloadUrl
          );
          
          if (notifResult.success) {
            console.log(`✅ Incident notifications sent to ${notifResult.recipients?.total || 0} users`);
          } else {
            console.warn('⚠️ Incident notifications failed:', notifResult.error);
          }
        } catch (notifError) {
          console.error('❌ Background notification error:', notifError);
        }
      }, 500); // 500ms delay for background execution

      setSyncModalVisible(false);
      
      // Show success toast
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Incident report generated and saved successfully',
      });
      
      setTimeout(() => {
        setPdfUrl(pdfDownloadUrl);
        setSuccessModalVisible(true);
      }, 200);

      // Clear persisted state
      await AsyncStorage.removeItem('IncidentReportStep1Data');
      await AsyncStorage.removeItem('IncidentReportStep2State');
      // Do NOT auto-navigate here; only show modal
    } catch (error) {
      console.error('Error generating report:', error);
      console.error('Error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack,
        platform: Platform.OS
      });
      setSyncModalVisible(false);
      
      // Provide detailed error messages based on error type
      let errorTitle = 'Error';
      let errorMessage = 'Failed to generate or save report.';
      
      if (error.message.includes('timeout')) {
        errorTitle = 'Timeout Error';
        errorMessage = 'The operation timed out. Please check your internet connection and try again.';
      } else if (error.message.includes('Network request failed')) {
        errorTitle = 'Network Error';
        if (Platform.OS === 'android') {
          errorMessage = 'Network error on Android. Please check:\n1. Internet connection\n2. App permissions\n3. Server is accessible at http://168.231.113.219:5007\n4. Try restarting the app';
        } else {
          errorMessage = 'Network connection failed. Please check your internet and try again.';
        }
      } else if (error.message.includes('Network') || error.message.includes('internet')) {
        errorTitle = 'Network Error';
        errorMessage = error.message || 'Network connection issue. Please check your internet and try again.';
      } else if (error.message.includes('Server')) {
        errorTitle = 'Server Error';
        errorMessage = 'Cannot connect to server. Please try again later.';
      } else if (error.message.includes('authenticated')) {
        errorTitle = 'Authentication Error';
        errorMessage = 'Your session has expired. Please log in again.';
      } else {
        errorMessage = error.message || 'An unexpected error occurred. Please try again.';
      }
      
      Alert.alert(
        errorTitle,
        errorMessage,
        [
          { 
            text: 'OK',
            onPress: () => {
              // If it's a network error, keep the modal open so user can retry
              if (error.message.includes('timeout') || error.message.includes('Network')) {
                console.log('Network error - user can retry');
              }
            }
          }
        ]
      );
      
      Toast.show({
        type: 'error',
        text1: errorTitle,
        text2: errorMessage,
        position: 'top',
        visibilityTime: 4000,
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  // Only allow navigation from the modal's Back or View PDF buttons
  const handleViewPDF = async (pdfUrl) => {
    if (!pdfUrl || !reportId) {
      Alert.alert('Error', 'PDF or Report ID not available.');
      return;
    }
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        // Do not navigate away
        return;
      }
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        // Do not auto-navigate on return
      });
    } catch (error) {
      console.error('Error viewing PDF:', error);
      Alert.alert('Error', 'Could not view the PDF. Please try again.');
    }
    // Do not close the modal automatically here
  };

  const handleNavigateBack = () => {
    setSuccessModalVisible(false);
    navigation.navigate('IncodentReport');
  };

  useImperativeHandle(ref, () => ({
    validateStep2,
    getStep2Data: () => ({
      incidentImages,
      incidentCause,
      equipmentDamage,
      additionalComment,
      immediateCorrectiveAction,
    }),
    generatePDF: generateReport,
  }));

  const handleRemoveImage = (index) => {
    Alert.alert(
      'Remove Image',
      'Are you sure you want to remove this image?',
      [
        { text: 'Cancel', style: 'cancel' },
        { 
          text: 'Remove', 
          style: 'destructive',
          onPress: () => {
            const newImages = [...incidentImages];
            newImages[index] = null;
            setIncidentImages(newImages);
            saveState();
          }
        },
      ],
      { cancelable: true }
    );
  };

  const handleFilePick = (index) => {
    Alert.alert(
      'Select Image',
      'Choose an option',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Take Picture', onPress: () => pickImage(index, 'camera') },
        { text: 'Choose from Gallery', onPress: () => pickImage(index, 'gallery') },
      ],
      { cancelable: true }
    );
  };

  const pickImage = async (index, source) => {
    const options = {
      mediaType: 'photo',
      maxWidth: 500,
      maxHeight: 500,
      quality: 0.8,
    };

    const callback = (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        console.error('Image Picker Error:', response.errorCode, response.errorMessage);
        if (response.errorCode === 'permission') {
          Alert.alert(
            'Permission Error',
            'Photo library access is denied. Please enable it in Settings > Privacy > Photos.'
          );
        } else if (response.errorCode === 'library') {
          Alert.alert('No Photos Available', 'No photos are available in your library.');
        } else {
          Alert.alert('Error', response.errorMessage || 'Failed to pick image');
        }
      } else if (response.assets && response.assets.length > 0) {
        const uri = response.assets[0].uri;
        const newImages = [...incidentImages];
        newImages[index] = uri;
        setIncidentImages(newImages);
        saveState();
      } else {
        Alert.alert('No Selection', 'No image was selected.');
      }
    };

    if (source === 'camera') {
      launchCamera(options, callback);
    } else {
      launchImageLibrary(options, callback);
    }
  };

  const chunkArray = (arr, size) => {
    const result = [];
    for (let i = 0; i < arr.length; i += size) {
      result.push(arr.slice(i, i + size));
    }
    return result;
  };

  const renderStep2 = () => (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
      enabled
    >
      <ScrollView
        style={styles.formContainer}
        contentContainerStyle={{ paddingBottom: 100, flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
        bounces={true}
        automaticallyAdjustKeyboardInsets={true}
      >
        <TouchableOpacity style={styles.incidentImagesButton}>
          <Text style={styles.incidentImagesButtonText}>Click Incident Images</Text>
        </TouchableOpacity>
        <View style={styles.incidentImagesGridWrapper}>
          <View style={styles.incidentImagesGrid}>
            {incidentImages.map((img, idx) => (
              <View key={idx} style={styles.incidentImageBox}>
                {img ? (
                  <View style={styles.imageContainer}>
                    <Image
                      source={{ uri: img }}
                      style={styles.incidentImage}
                      onError={(e) => console.log('Image error:', e.nativeEvent.error)}
                    />
                    <TouchableOpacity 
                      style={styles.removeImageButton} 
                      onPress={() => handleRemoveImage(idx)}
                    >
                      <MaterialIcons name="close" size={16} color="#fff" />
                    </TouchableOpacity>
                    <TouchableOpacity 
                      style={styles.cameraOverlay} 
                      onPress={() => handleFilePick(idx)}
                    >
                      <MaterialIcons name="camera-alt" size={24} color="#fff" />
                    </TouchableOpacity>
                  </View>
                ) : (
                  <TouchableOpacity style={styles.emptyImageBox} onPress={() => handleFilePick(idx)}>
                    <MaterialIcons name="camera-alt" size={36} color="#015185" />
                  </TouchableOpacity>
                )}
              </View>
            ))}
          </View>
        </View>
      </ScrollView>
    </KeyboardAvoidingView>
  );

  const renderStep3 = () => (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
      enabled
    >
      <ScrollView
        style={{ flex: 1, backgroundColor: '#F5F6FA' }}
        contentContainerStyle={{ paddingBottom: 100, flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
        bounces={true}
        automaticallyAdjustKeyboardInsets={true}
      >
        <View style={styles.summaryCard}>
          <View style={styles.incidentCauseGroup}>
            <Text style={styles.incidentCauseLabel}>Description of incident</Text>
            <View style={styles.incidentCauseRelative}>
              <TextInput
                style={styles.incidentCauseInput}
                multiline
                value={incidentCause}
                onChangeText={(text) => { setIncidentCause(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.incidentCauseMic} />
            </View>
          </View>
          <View style={styles.equipmentDamageGroup}>
            <Text style={styles.equipmentDamageLabel}>Details of damage / injury / loss</Text>
            <View style={styles.equipmentDamageRelative}>
              <TextInput
                style={styles.equipmentDamageInput}
                multiline
                value={equipmentDamage}
                onChangeText={(text) => { setEquipmentDamage(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.equipmentDamageMic} />
            </View>
          </View>
          <View style={styles.additionalCommentGroup}>
            <Text style={styles.additionalCommentLabel}>Possible direct causes</Text>
            <View style={styles.additionalCommentRelative}>
              <TextInput
                style={styles.additionalCommentInput}
                multiline
                value={additionalComment}
                onChangeText={(text) => { setAdditionalComment(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.additionalCommentMic} />
            </View>
          </View>
          <View style={styles.immediateCorrectiveActionGroup}>
            <Text style={styles.immediateCorrectiveActionLabel}>Immediate Corrective Action taken</Text>
            <View style={styles.immediateCorrectiveActionRelative}>
              <TextInput
                style={styles.immediateCorrectiveActionInput}
                multiline
                value={immediateCorrectiveAction}
                onChangeText={(text) => { setImmediateCorrectiveAction(text); saveState(); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <MaterialIcons name="mic" size={22} color="#015185" style={styles.immediateCorrectiveActionMic} />
            </View>
          </View>
        </View>
      </ScrollView>
    </KeyboardAvoidingView>
  );

  return (
    <View style={styles.container}>
      {currentStep === 2 && renderStep3()}
      {currentStep === 3 && renderStep2()}
      <SyncModal visible={syncModalVisible} />
      <SuccessModal
        visible={successModalVisible}
        onClose={() => setSuccessModalVisible(false)}
        onNavigateBack={handleNavigateBack}
        onViewPDF={handleViewPDF}
        pdfUrl={pdfUrl}
      />
      <Toast />
    </View>
  );
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  formContainer: {
    flex: 1,
    backgroundColor: '#fff',
    padding: 15,
  },
  incidentImagesButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 8,
    marginBottom: 18,
    marginHorizontal: 15,
  },
  incidentImagesButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 14,
  },
  incidentImagesGridWrapper: {
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
  },
  incidentImagesGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'center',
    alignItems: 'center',
    width: 320, // or use width: '90%' for responsiveness
  },
  incidentImageBox: {
    width: 90,
    height: 90,
    margin: 8,
    backgroundColor: '#F5F8FA',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#B0C4DE',
    position: 'relative',
    overflow: 'hidden',
  },
  imageContainer: {
    position: 'relative',
    width: '100%',
    height: '100%',
  },
  incidentImage: {
    width: '100%',
    height: '100%',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#B0C4DE',
  },
  removeImageButton: {
    position: 'absolute',
    top: 5,
    right: 5,
    backgroundColor: 'rgba(255, 0, 0, 0.8)',
    borderRadius: 12,
    width: 24,
    height: 24,
    alignItems: 'center',
    justifyContent: 'center',
    zIndex: 2,
  },
  cameraOverlay: {
    position: 'absolute',
    bottom: 5,
    right: 5,
    backgroundColor: 'rgba(0, 0, 0, 0.6)',
    borderRadius: 12,
    padding: 5,
    zIndex: 1,
  },
  emptyImageBox: {
    width: '100%',
    height: '100%',
    backgroundColor: '#F5F8FA',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#B0C4DE',
  },
  summaryCard: {
    backgroundColor: '#fff',
    borderRadius: 14,
    margin: 18,
    padding: 18,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.08,
    shadowRadius: 8,
    elevation: 4,
  },
  incidentCauseGroup: {
    marginBottom: 18,
  },
  incidentCauseLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  incidentCauseRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  incidentCauseInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 16,
    paddingVertical: 14,
    fontSize: 15,
    color: '#222',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  incidentCauseMic: {
    position: 'absolute',
    right: 16,
    top: 14,
  },
  equipmentDamageGroup: {
    marginBottom: 18,
  },
  equipmentDamageLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  equipmentDamageRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  equipmentDamageInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 15,
    paddingVertical: 13,
    fontSize: 15,
    color: '#222',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  equipmentDamageMic: {
    position: 'absolute',
    right: 15,
    top: 13,
  },
  additionalCommentGroup: {
    marginBottom: 18,
  },
  additionalCommentLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  additionalCommentRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  additionalCommentInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 14,
    paddingVertical: 14,
    fontSize: 14,
    color: '#333',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  additionalCommentMic: {
    position: 'absolute',
    right: 14,
    top: 14,
  },
  immediateCorrectiveActionGroup: {
    marginBottom: 18,
  },
  immediateCorrectiveActionLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  immediateCorrectiveActionRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  immediateCorrectiveActionInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 14,
    paddingVertical: 14,
    fontSize: 14,
    color: '#333',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  immediateCorrectiveActionMic: {
    position: 'absolute',
    right: 14,
    top: 14,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
  },
  modalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  modalActions: {
    flexDirection: 'row',
    justifyContent: 'center',
  },
  modalButton: {
    backgroundColor: '#d3d3d3',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  viewButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  successIcon: {
    alignSelf: 'center',
    marginBottom: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
});

export default IncidentReportStep2;