import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  KeyboardAvoidingView,
  Platform,
  Dimensions,
  PermissionsAndroid,
  Animated,
  Modal,
  Modal as RNModal,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import DateTimePicker from '@react-native-community/datetimepicker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { Dropdown } from 'react-native-element-dropdown';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import IncidentReportHeader from './IncidentReportHeader';
import IncidentReportSteps from './IncidentReportStep2';
import Geolocation from '@react-native-community/geolocation';
import OfflineIncidentHelper from './OfflineIncidentHelper';
// Removed Firebase import - now using MongoDB API

const { width } = Dimensions.get('window');

const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  isDate = false,
  onPress,
  editable = true,
  placeholder,
  ...props
}) => (
  <View style={styles.inputContainer}>
    <Text style={styles.label}>
      {label}
      {isRequired && <Text style={styles.requiredError}> *</Text>}
    </Text>
    {isDate ? (
      <TouchableOpacity
        style={[
          styles.input,
          showError ? styles.inputError : null,
          { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between' },
        ]}
        onPress={onPress}
      >
        <Text style={styles.dateText}>{value}</Text>
        <MaterialIcons name="calendar-today" size={20} color="#015185" />
      </TouchableOpacity>
    ) : (
      <TextInput
        style={[styles.input, showError ? styles.inputError : null]}
        value={value}
        onChangeText={onChangeText}
        editable={editable}
        placeholder={placeholder}
        placeholderTextColor="#888"
        {...props}
      />
    )}
    {showError && <Text style={styles.errorText}>{errorMessage}</Text>}
  </View>
);

// SyncModal Component
const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')} // Ensure this image path is correct
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const CreateIncidentReport = () => {
  const navigation = useNavigation();
  const [currentStep, setCurrentStep] = useState(1);
  const [pickerMode, setPickerMode] = useState('date'); // 'date' or 'time'
  const [formData, setFormData] = useState({
    incidentNumber: '',
    incidentDate: null,
    country: null,
    project: null,
    vehicleId: '',
    vehicleName: '',
    incidentDate2: '',
    otherIncidentNote: '',
    incidentArea: '',
    severity: '',
    incidentCategory: null,
    otherIncidentCategory: '',
    coordinates: '',
    gpsAddress: '',
  });
  const [showErrors, setShowErrors] = useState(false);
  const [errors, setErrors] = useState({
    incidentTypes: '',
    incidentNumber: '',
    incidentDate: '',
    country: '',
    project: '',
    involvesEquipment: '',
    incidentArea: '',
    incidentCategory: '',
    otherIncidentCategory: '',
  });
  const stepsRef = useRef(null);
  const [involvesEquipment, setInvolvesEquipment] = useState(null);
  const [countryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);
  const [projectItems, setProjectItems] = useState([]);
  const [selectedIncidents, setSelectedIncidents] = useState({
    'Brake Failure': false,
    'Chemical Spill': false,
    'Collision': false,
    'Drowned in Water': false,
    'Electrical Failure': false,
    'Engine Failure': false,
    'Equipment Failure': false,
    'Fall from elevation': false,
    'Loss of control': false,
    'Oil Spill': false,
    'Person hit by Vehicle': false,
    'Steering Failure': false,
    'Stuck': false,
    'Puncture': false,
    'Tire Burst': false,
    'Roll over': false,
    'Fell on side': false,
    'Other': false,
  });
  const [coordinates, setCoordinates] = useState('');
  const [gpsAddress, setGpsAddress] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const isMounted = useRef(true);
  const severityOptions = [
    { label: 'High', value: 'High' },
    { label: 'Medium', value: 'Medium' },
    { label: 'Low', value: 'Low' },
  ];
  const incidentCategoryOptions = [
    { label: 'Personnel', value: 'Personnel' },
    { label: 'Equipment', value: 'Equipment' },
    { label: 'Environmental', value: 'Environmental' },
    { label: 'Other', value: 'Other' },
  ];
  const [dateTimeModalVisible, setDateTimeModalVisible] = useState(false);
  const [tempDateTime, setTempDateTime] = useState(new Date());
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [showTimePicker, setShowTimePicker] = useState(false);
  const [datePickerValue, setDatePickerValue] = useState(new Date());
  const [timePickerValue, setTimePickerValue] = useState(new Date());
  const [totalIncidents, setTotalIncidents] = useState(0);

  // Country code mapping for incident number prefix
  const countryCodeMap = {
    Zambia: 'ZAM',
    DRC: 'DRC',
  };

  // Project code mapping for incident number prefix
  const projectCodeMap = {
    Kobold: 'KOB',
    Kalumbila: 'KAL',
    Kansanshi: 'KAN',
    Kimiteto: 'KIM',
    Mimosa: 'MIM',
    Musompo: 'MUS',
    IME: 'IME',
  };

  useEffect(() => {
    loadPersistedState();
    fetchTotalIncidents();

    if (formData.country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (formData.country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }

    if (!formData.country || (formData.project && !projectItems.some((item) => item.value === formData.project))) {
      setFormData((prev) => ({ ...prev, project: null, incidentNumber: '' }));
    }

    // Generate incident number whenever country or project changes
    if (formData.country && formData.project) {
      generateIncidentNumber(formData.country, formData.project);
    } else {
      setFormData((prev) => ({ ...prev, incidentNumber: '' }));
    }

    return () => {
      isMounted.current = false;
    };
  }, [formData.country, formData.project]);

  const loadPersistedState = async () => {
    try {
      const state = await AsyncStorage.getItem('IncidentReportStep1State');
      if (state) {
        const parsed = JSON.parse(state);
        setFormData((prev) => ({
          ...prev,
          incidentArea: parsed.incidentArea || '',
          severity: parsed.severity || '',
          coordinates: parsed.coordinates || '',
          gpsAddress: parsed.gpsAddress || '',
        }));
        setCoordinates(parsed.coordinates || '');
        setGpsAddress(parsed.gpsAddress || '');
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
    }
  };

  const fetchTotalIncidents = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;

      const response = await fetch(`http://168.231.113.219:5007/incident-reports/${userUid}`);
      const result = await response.json();
      
      if (result.success) {
        setTotalIncidents(result.count || 0);
      }
    } catch (error) {
      console.error('Error fetching total incidents:', error);
    }
  };

  const saveState = async () => {
    try {
      const state = {
        incidentArea: formData.incidentArea,
        severity: formData.severity,
        coordinates: formData.coordinates,
        gpsAddress: formData.gpsAddress,
      };
      await AsyncStorage.setItem('IncidentReportStep1State', JSON.stringify(state));
    } catch (error) {
      console.error('Error saving state:', error);
    }
  };

  // Generate incident number using MongoDB (with offline support)
  const generateIncidentNumber = async (country, project) => {
    if (!country || !project) return;

    setSyncModalVisible(true);
    try {
      const countryCode = countryCodeMap[country] || country.toUpperCase().slice(0, 3);
      const projectCode = projectCodeMap[project] || project.toUpperCase().slice(0, 3);
      const prefix = `${countryCode}-${projectCode}-INC-`;

      // Check if online
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;

      let incidentNumber = '';

      const getNextGlobalNumber = async () => {
        try {
          let maxGlobal = 0;
          // Cached cloud reports - check ALL incidents
          const cachedRaw = await AsyncStorage.getItem('incidentReports');
          const cached = cachedRaw ? JSON.parse(cachedRaw) : [];
          cached.forEach((r) => {
            const inc = r?.incidentNumber;
            if (inc) {
              const match = inc.match(/INC-(\d+)/);
              if (match) {
                const n = parseInt(match[1], 10);
                if (!isNaN(n)) maxGlobal = Math.max(maxGlobal, n);
              }
            }
          });
          // Offline reports saved locally - check ALL incidents
          try {
            const offline = await OfflineIncidentHelper.getOfflineReports();
            offline.forEach((r) => {
              const inc = r?.incidentNumber;
              if (inc) {
                const match = inc.match(/INC-(\d+)/);
                if (match) {
                  const n = parseInt(match[1], 10);
                  if (!isNaN(n)) maxGlobal = Math.max(maxGlobal, n);
                }
              }
            });
          } catch (e) {
            // ignore
          }
          return (maxGlobal + 1).toString().padStart(3, '0');
        } catch (e) {
          return '001';
        }
      };

      if (isOnline) {
        // ONLINE MODE: Get next number based on TOTAL count in MongoDB
        try {
          const response = await fetch('http://168.231.113.219:5007/incident-reports', {
            timeout: 5000
          });
          const result = await response.json();
          
          let maxGlobalNumber = 0;
          if (result.success && result.data) {
            // Check ALL reports across all prefixes to find highest number
            result.data.forEach(report => {
              if (report.incidentNumber && !report.incidentNumber.includes('-OFFLINE')) {
                const match = report.incidentNumber.match(/INC-(\d+)/);
                if (match) {
                  const num = parseInt(match[1], 10);
                  if (!isNaN(num)) {
                    maxGlobalNumber = Math.max(maxGlobalNumber, num);
                  }
                }
              }
            });
          }

          const nextNumber = (maxGlobalNumber + 1).toString().padStart(3, '0');
          incidentNumber = `${prefix}${nextNumber}`;

          console.log('✅ Generated ONLINE incident number (global count):', incidentNumber);
        } catch (onlineError) {
          console.warn('⚠️ Online number generation failed, creating offline number:', onlineError.message);
          // Fallback to offline mode
          const nextGlobal = await getNextGlobalNumber();
          incidentNumber = `${prefix}${nextGlobal}-OFFLINE`;
        }
      } else {
        // OFFLINE MODE: Generate temporary offline number based on local global count
        const nextGlobal = await getNextGlobalNumber();
        incidentNumber = `${prefix}${nextGlobal}-OFFLINE`;
        console.log('📴 Generated OFFLINE incident number (global count):', incidentNumber);
        
        Toast.show({
          type: 'info',
          text1: 'Offline Mode',
          text2: 'Creating offline incident report. Will sync when online.',
          position: 'top',
          visibilityTime: 3000,
        });
      }

      setFormData((prev) => ({ ...prev, incidentNumber }));
    } catch (error) {
      console.error('Error generating incident number:', error);
      
      // Fallback: Create offline number based on global count
      const countryCode = countryCodeMap[country] || country.toUpperCase().slice(0, 3);
      const projectCode = projectCodeMap[project] || project.toUpperCase().slice(0, 3);
      let fallbackNumber = `${countryCode}-${projectCode}-INC-001-OFFLINE`;
      try {
        const nextGlobal = await (async () => {
          let maxGlobal = 0;
          const cachedRaw = await AsyncStorage.getItem('incidentReports');
          const cached = cachedRaw ? JSON.parse(cachedRaw) : [];
          cached.forEach((r) => {
            const inc = r?.incidentNumber;
            if (inc) {
              const match = inc.match(/INC-(\d+)/);
              if (match) {
                const n = parseInt(match[1], 10);
                if (!isNaN(n)) maxGlobal = Math.max(maxGlobal, n);
              }
            }
          });
          try {
            const offline = await OfflineIncidentHelper.getOfflineReports();
            offline.forEach((r) => {
              const inc = r?.incidentNumber;
              if (inc) {
                const match = inc.match(/INC-(\d+)/);
                if (match) {
                  const n = parseInt(match[1], 10);
                  if (!isNaN(n)) maxGlobal = Math.max(maxGlobal, n);
                }
              }
            });
          } catch {}
          return (maxGlobal + 1).toString().padStart(3, '0');
        })();
        fallbackNumber = `${countryCode}-${projectCode}-INC-${nextGlobal}-OFFLINE`;
      } catch {}
      
      console.log('⚠️ Using fallback offline number (global count):', fallbackNumber);
      setFormData((prev) => ({ ...prev, incidentNumber: fallbackNumber }));
      
      Toast.show({
        type: 'info',
        text1: 'Offline Mode',
        text2: 'Creating offline incident report.',
        position: 'top',
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  // Save incident report to AsyncStorage only when moving to step 2
  const saveIncidentReportToAsyncStorage = async () => {
    try {
      if (!formData.incidentNumber) {
        throw new Error('Incident number not generated.');
      }
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        throw new Error('Admin not authenticated. Please log in again.');
      }
      const incidentData = {
        incidentNumber: formData.incidentNumber,
        incidentDate: formData.incidentDate ? formData.incidentDate.toISOString() : null,
        country: formData.country,
        project: formData.project,
        vehicleId: formData.vehicleId,
        vehicleName: formData.vehicleName,
        personInvolved: formData.incidentDate2,
        otherIncidentNote: formData.otherIncidentNote,
        incidentArea: formData.incidentArea,
        severity: formData.severity,
        incidentCategory: formData.incidentCategory === 'Other' ? formData.otherIncidentCategory : formData.incidentCategory,
        selectedIncidents,
        involvesEquipment,
        coordinates: formData.coordinates,
        gpsAddress: formData.gpsAddress,
        createdAt: new Date().toISOString(),
        adminUid, // Optionally keep track of who created it
      };
      await AsyncStorage.setItem('IncidentReportStep1Data', JSON.stringify(incidentData));
    } catch (error) {
      console.error('Error saving incident report to AsyncStorage:', error);
      throw new Error('Failed to save incident report to local storage.');
    }
  };

  const requestLocationPermission = async () => {
    try {
      if (Platform.OS === 'android') {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.ACCESS_FINE_LOCATION,
          {
            title: 'Location Permission',
            message: 'This app needs access to your location to fetch your current address.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        if (granted === PermissionsAndroid.RESULTS.GRANTED) {
          return true;
        } else if (granted === PermissionsAndroid.RESULTS.DENIED) {
          Toast.show({
            type: 'error',
            text1: 'Permission Denied',
            text2: 'Location permission is required to fetch your current location. Please enable it in settings.',
            position: 'top',
          });
          return false;
        } else if (granted === PermissionsAndroid.RESULTS.NEVER_ASK_AGAIN) {
          Toast.show({
            type: 'error',
            text1: 'Permission Blocked',
            text2: 'Location permission is blocked. Please enable it in your device settings.',
            position: 'top',
          });
          return false;
        }
      } else {
        return new Promise((resolve, reject) => {
          Geolocation.requestAuthorization(
            () => resolve(true),
            (error) => {
              console.error('iOS permission error:', error);
              Toast.show({
                type: 'error',
                text1: 'Permission Error',
                text2: 'Unable to access location permission. Please enable it in settings.',
                position: 'top',
              });
              resolve(false);
            }
          );
        });
      }
    } catch (error) {
      console.error('Error requesting location permission:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to request location permission. Please try again.',
        position: 'top',
      });
      return false;
    }
  };

  const fetchAddressFromCoordinates = async (latitude, longitude) => {
    const apiKey = 'AIzaSyDDEh8W8buXtPi7uhhwJx4e5iK7Gb-vbe8'; // Replace with your actual API key
    const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${latitude},${longitude}&key=${apiKey}`;

    try {
      console.log('Fetching address with URL:', url);
      const response = await fetch(url);
      const data = await response.json();
      console.log('Geocoding API Response:', JSON.stringify(data, null, 2));

      if (data.status === 'OK' && data.results.length > 0) {
        return data.results[0].formatted_address;
      } else if (data.status === 'REQUEST_DENIED') {
        throw new Error('Request denied: Invalid API key or billing issue.');
      } else if (data.status === 'ZERO_RESULTS') {
        throw new Error('No address found for the given coordinates.');
      } else {
        throw new Error(`Geocoding API error: ${data.status}`);
      }
    } catch (error) {
      console.error('Error fetching address:', error.message);
      Toast.show({
        type: 'error',
        text1: 'Address Fetch Failed',
        text2: error.message || 'Failed to fetch address. Using coordinates only.',
        position: 'top',
      });
      return `Latitude: ${latitude}, Longitude: ${longitude} (Address unavailable)`;
    }
  };

  const handleGetLocation = async () => {
    setSyncModalVisible(true);
    const timeoutId = setTimeout(() => {
      if (isMounted.current && syncModalVisible) {
        setSyncModalVisible(false);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Location fetching timed out. Please try again.',
          position: 'top',
        });
      }
    }, 15000);

    try {
      const hasPermission = await requestLocationPermission();
      if (!hasPermission) {
        setSyncModalVisible(false);
        throw new Error('Location permission denied');
      }

      await new Promise((resolve, reject) => {
        Geolocation.getCurrentPosition(
          async (position) => {
            const { latitude, longitude } = position.coords;
            const coords = `${latitude}, ${longitude}`;
            const address = await fetchAddressFromCoordinates(latitude, longitude);

            // Update formData and separate states together
            setFormData((prev) => ({
              ...prev,
              coordinates: coords,
              gpsAddress: address,
            }));
            setCoordinates(coords);
            setGpsAddress(address);

            await saveState();

            Toast.show({
              type: 'success',
              text1: 'Location Fetched',
              text2: `Coordinates: ${coords}\nAddress: ${address}`,
              position: 'top',
              visibilityTime: 4000,
            });
            setSyncModalVisible(false);
            clearTimeout(timeoutId);
            resolve();
          },
          (error) => {
            console.error('Error getting location:', error);
            let errorMessage = '';
            if (error.code === error.PERMISSION_DENIED) {
              errorMessage += 'Please ensure location permissions are granted.';
            } else if (error.code === error.POSITION_UNAVAILABLE) {
              errorMessage += 'Please turn on your GPS.';
            } else if (error.code === error.TIMEOUT) {
              errorMessage += 'The request timed out. Please try again.';
            } else {
              errorMessage += 'An unexpected error occurred.';
            }
            Toast.show({
              type: 'error',
              text1: 'Location Error',
              text2: errorMessage,
              position: 'top',
            });
            setSyncModalVisible(false);
            clearTimeout(timeoutId);
            reject(error);
          },
          { enableHighAccuracy: true, timeout: 15000, maximumAge: 10000 }
        );
      });
    } catch (error) {
      console.error('Error in handleGetLocation:', error);
      setSyncModalVisible(false);
      clearTimeout(timeoutId);
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
      clearTimeout(timeoutId);
    }
  };

  const handleIncidentToggle = (incident) => {
    setSelectedIncidents((prev) => {
      const newSelectedIncidents = { ...prev, [incident]: !prev[incident] };
      const hasSelectedIncident = Object.values(newSelectedIncidents).some((value) => value);
      setErrors((prev) => ({
        ...prev,
        incidentTypes: hasSelectedIncident ? '' : 'Please select at least one type of incident',
      }));
      if (incident === 'Other' && !newSelectedIncidents[incident]) {
        setFormData((prev) => ({ ...prev, otherIncidentNote: '' }));
      }
      return newSelectedIncidents;
    });
  };

  const handleEquipmentSelect = (equipmentData) => {
    setFormData((prev) => ({
      ...prev,
      vehicleId:
        equipmentData.equipmentNumber ||
        equipmentData.vehicleNumber ||
        equipmentData.trailerNumber ||
        equipmentData.id ||
        equipmentData.entryNumber ||
        '',
      vehicleName:
        equipmentData.equipmentName ||
        equipmentData.rigName ||
        equipmentData.vehicleNumber ||
        '',
    }));
  };

  // Date & Time Picker logic
  const handleIncidentDatePress = () => {
    try {
      if (Platform.OS === 'ios') {
        // Initialize with current value or now - ensure it's a valid Date
        let initialDate = new Date();
        
        if (formData.incidentDate) {
          try {
            const tempDate = new Date(formData.incidentDate);
            if (tempDate instanceof Date && !isNaN(tempDate.getTime())) {
              initialDate = tempDate;
            }
          } catch (dateError) {
            console.warn('Invalid incident date, using current date:', dateError);
          }
        }
        
        console.log('Opening iOS date picker with date:', initialDate);
        setTempDateTime(initialDate);
        setDateTimeModalVisible(true);
      } else {
        // Android - simple date picker like Add_equipment.js
        setDatePickerValue(formData.incidentDate || new Date());
        setShowDatePicker(true);
      }
    } catch (error) {
      console.error('Error in handleIncidentDatePress:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to open date picker. Please try again.',
        position: 'top',
      });
    }
  };

  const handleDateChange = (event, selectedDate) => {
    if (Platform.OS === 'android') {
      setShowDatePicker(false);
      if (event.type === 'set' && selectedDate) {
        setDatePickerValue(selectedDate);
        // After date selection, show time picker
        setTimePickerValue(selectedDate);
        setShowTimePicker(true);
      }
    } else {
      // iOS logic - freely update tempDateTime as user scrolls (validation happens on "Done" click)
      try {
        if (selectedDate && selectedDate instanceof Date && !isNaN(selectedDate.getTime())) {
          setTempDateTime((prev) => {
            try {
              // Ensure prev is a valid date
              const prevDate = (prev && prev instanceof Date && !isNaN(prev.getTime())) ? prev : new Date();
              
              // Create new date from selected date
              const newDate = new Date(selectedDate.getTime());
              
              // Keep the time part from previous value
              newDate.setHours(prevDate.getHours());
              newDate.setMinutes(prevDate.getMinutes());
              newDate.setSeconds(0);
              newDate.setMilliseconds(0);
              
              return newDate;
            } catch (innerError) {
              console.error('Error in setTempDateTime callback:', innerError);
              return prev || new Date();
            }
          });
        }
      } catch (error) {
        console.error('Error in handleDateChange (iOS):', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to update date. Please try again.',
          position: 'top',
        });
      }
    }
  };

  const handleTimeChange = (event, selectedTime) => {
    if (Platform.OS === 'android') {
      setShowTimePicker(false);
      if (event.type === 'set' && selectedTime) {
        // Combine date and time for Android
        const combinedDateTime = new Date(datePickerValue);
        combinedDateTime.setHours(selectedTime.getHours());
        combinedDateTime.setMinutes(selectedTime.getMinutes());
        setFormData((prev) => ({ ...prev, incidentDate: combinedDateTime }));
      }
    } else {
      // iOS logic - freely update time as user scrolls
      try {
        if (selectedTime && selectedTime instanceof Date && !isNaN(selectedTime.getTime())) {
          setTempDateTime((prev) => {
            try {
              // Ensure prev is a valid date
              const prevDate = (prev && prev instanceof Date && !isNaN(prev.getTime())) ? prev : new Date();
              
              // Create new date from previous date
              const newDate = new Date(prevDate.getTime());
              
              // Update time part
              newDate.setHours(selectedTime.getHours());
              newDate.setMinutes(selectedTime.getMinutes());
              newDate.setSeconds(0);
              newDate.setMilliseconds(0);
              
              return newDate;
            } catch (innerError) {
              console.error('Error in setTempDateTime callback:', innerError);
              return prev || new Date();
            }
          });
        }
      } catch (error) {
        console.error('Error in handleTimeChange (iOS):', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to update time. Please try again.',
          position: 'top',
        });
      }
    }
  };

  const handleDateTimeDone = () => {
    try {
      // Validate tempDateTime exists and is valid
      if (!tempDateTime || !(tempDateTime instanceof Date) || isNaN(tempDateTime.getTime())) {
        console.error('tempDateTime is invalid:', tempDateTime);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Please select a valid date and time.',
          position: 'top',
        });
        return;
      }

      // Validate date is not in the future
      const today = new Date();
      today.setHours(23, 59, 59, 999);
      
      // Create a date without time for comparison
      const selectedDateOnly = new Date(tempDateTime.getTime());
      selectedDateOnly.setHours(23, 59, 59, 999);
      
      if (selectedDateOnly > today) {
        Toast.show({
          type: 'error',
          text1: 'Invalid Date',
          text2: 'Incident date cannot be in the future. Please select today or a past date.',
          position: 'top',
        });
        return;
      }

      console.log('Date & Time confirmed, saving:', tempDateTime);
      
      // Save the selected date and time
      setFormData((prev) => ({ ...prev, incidentDate: tempDateTime }));
      setDateTimeModalVisible(false);
      
      Toast.show({
        type: 'success',
        text1: 'Date & Time Set',
        text2: `${formatDate(tempDateTime)}`,
        position: 'top',
        visibilityTime: 2000,
      });
    } catch (error) {
      console.error('Error in handleDateTimeDone:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save date and time. Please try again.',
        position: 'top',
      });
    }
  };

  const handleDateTimeCancel = () => {
    try {
      // Reset to original value if cancelled
      const resetDate = formData.incidentDate ? new Date(formData.incidentDate) : new Date();
      setTempDateTime(resetDate);
      setDateTimeModalVisible(false);
    } catch (error) {
      console.error('Error in handleDateTimeCancel:', error);
      setDateTimeModalVisible(false);
    }
  };

  const formatDate = (date) => {
    if (!date) return 'Select Incident Date';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const validateStep1 = () => {
    const newErrors = {};
    if (!formData.incidentNumber) newErrors.incidentNumber = 'Incident Number is required';
    if (!formData.incidentDate) newErrors.incidentDate = 'Incident Date is required';
    
    // Check if incident date is in the future
    if (formData.incidentDate) {
      const today = new Date();
      today.setHours(23, 59, 59, 999); // Set to end of today
      if (formData.incidentDate > today) {
        newErrors.incidentDate = 'Incident date cannot be in the future';
      }
    }
    
    if (!formData.country) newErrors.country = 'Please select a country';
    if (!formData.project) newErrors.project = 'Please select a project';
    if (!formData.incidentArea) newErrors.incidentArea = 'Incident Area is required';
    if (!formData.incidentCategory) newErrors.incidentCategory = 'Please select an incident category';
    if (formData.incidentCategory === 'Other' && !formData.otherIncidentCategory) {
      newErrors.otherIncidentCategory = 'Please enter the incident category';
    }
    const hasSelectedIncident = Object.values(selectedIncidents).some((value) => value);
    if (!hasSelectedIncident) newErrors.incidentTypes = 'Please select at least one type of incident';
    if (involvesEquipment === null) newErrors.involvesEquipment = 'Please specify if the incident involves equipment';

    setErrors(newErrors);
    setShowErrors(true);
    return Object.keys(newErrors).length === 0;
  };

  const handleNext = async () => {
    if (currentStep === 1) {
      if (validateStep1()) {
        // Save to AsyncStorage instead of database
        await saveIncidentReportToAsyncStorage();
        setCurrentStep(2);
        setShowErrors(false);
        await AsyncStorage.removeItem('IncidentReportStep1State');
      } else {
        Toast.show({
          type: 'error',
          text1: 'Validation Error',
          text2: 'Please fill all required fields in Step 1',
        });
      }
    } else if (currentStep === 2) {
      if (stepsRef.current && stepsRef.current.validateStep2()) {
        setCurrentStep(3);
        setShowErrors(false);
      } else {
        setShowErrors(true);
        Toast.show({
          type: 'error',
          text1: 'Validation Error',
          text2: 'Please fill all required fields in Step 2',
        });
      }
    } else if (currentStep === 3) {
      if (stepsRef.current && stepsRef.current.validateStep2()) {
        try {
          // Do not save to database here, just trigger PDF generation in step 2
          await stepsRef.current.generatePDF(formData.incidentNumber); // Pass incidentId if needed for PDF generation
          
          // Refresh total incidents count after successful generation
          await fetchTotalIncidents();
          
          Toast.show({
            type: 'success',
            text1: 'Success',
            text2: `Incident Report ${formData.incidentNumber} generated successfully.`,
            position: 'top',
          });
          // navigation.goBack(); // REMOVE this line to prevent auto-navigation
        } catch (err) {
          console.error('Error generating report:', err);
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: err.message || 'Failed to generate report.',
            position: 'top',
          });
        }
      } else {
        setShowErrors(true);
        Toast.show({
          type: 'error',
          text1: 'Validation Error',
          text2: 'Please fill all required fields in Step 3',
        });
      }
    }
  };

  const handleCancel = () => {
    if (currentStep > 1) {
      setCurrentStep((prev) => prev - 1);
      setShowErrors(false);
    } else {
      navigation.goBack();
    }
  };

  const handleChange = (field, value) => {
    setFormData((prev) => ({ ...prev, [field]: value }));
    if (showErrors && value) {
      setErrors((prev) => ({ ...prev, [field]: '' }));
    }
    saveState();
  };

  const handleEquipmentToggle = (value) => {
    setInvolvesEquipment(value);
    if (!value) {
      setFormData((prev) => ({
        ...prev,
        vehicleId: '',
        vehicleName: '',
        incidentDate2: '',
      }));
    }
    if (showErrors) {
      setErrors((prev) => ({ ...prev, involvesEquipment: '' }));
    }
  };

  const renderStep1 = () => (
    <ScrollView
      style={styles.formContainer}
      contentContainerStyle={styles.formContentContainer}
      keyboardShouldPersistTaps="handled"
    >
      {/* Total Incidents Counter */}
    

      <View style={styles.section}>
        <Text style={styles.sectionTitle}>Incident Details</Text>
        <CustomInputField
          label="Incident Number"
          value={formData.incidentNumber}
          onChangeText={(text) => handleChange('incidentNumber', text)}
          placeholder="Auto-generated"
          editable={false}
          showError={showErrors && errors.incidentNumber}
          errorMessage={errors.incidentNumber}
        />
        <CustomInputField
          label="Incident Date"
          value={formatDate(formData.incidentDate)}
          onPress={handleIncidentDatePress}
          isRequired
          isDate
          showError={showErrors && errors.incidentDate}
          errorMessage={errors.incidentDate}
          placeholder="Incident Date"
        />
        <View style={styles.inputContainer}>
          <Text style={styles.label}>
            Country
            <Text style={styles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[
              styles.input,
              showErrors && errors.country ? styles.inputError : null,
            ]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            inputSearchStyle={styles.inputSearchStyle}
            data={countryItems}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Country"
            value={formData.country}
            onChange={(item) => handleChange('country', item.value)}
            containerStyle={styles.dropdownContainer}
            itemContainerStyle={styles.dropdownItemContainer}
            itemTextStyle={styles.dropdownItemText}
            activeColor="#e0e0e0"
          />
          {showErrors && errors.country && (
            <Text style={styles.errorText}>{errors.country}</Text>
          )}
        </View>
        <View style={styles.inputContainer}>
          <Text style={styles.label}>
            Project
            <Text style={styles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[
              styles.input,
              !formData.country && styles.disabledDropdown,
              showErrors && errors.project ? styles.inputError : null,
            ]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            inputSearchStyle={styles.inputSearchStyle}
            data={projectItems}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Project"
            value={formData.project}
            onChange={(item) => handleChange('project', item.value)}
            containerStyle={styles.dropdownContainer}
            itemContainerStyle={styles.dropdownItemContainer}
            itemTextStyle={styles.dropdownItemText}
            activeColor="#e0e0e0"
            disable={!formData.country}
          />
          {showErrors && errors.project && (
            <Text style={styles.errorText}>{errors.project}</Text>
          )}
        </View>
        <View style={styles.inputContainer}>
          <Text style={styles.label}>
            Incident Category
            <Text style={styles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[
              styles.input,
              showErrors && errors.incidentCategory ? styles.inputError : null,
            ]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            inputSearchStyle={styles.inputSearchStyle}
            data={incidentCategoryOptions}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Incident Category"
            value={formData.incidentCategory}
            onChange={(item) => handleChange('incidentCategory', item.value)}
            containerStyle={styles.dropdownContainer}
            itemContainerStyle={styles.dropdownItemContainer}
            itemTextStyle={styles.dropdownItemText}
            activeColor="#e0e0e0"
          />
          {showErrors && errors.incidentCategory && (
            <Text style={styles.errorText}>{errors.incidentCategory}</Text>
          )}
        </View>
        {formData.incidentCategory === 'Other' && (
          <View style={styles.inputContainer}>
            <Text style={styles.label}>
              Enter Incident Category
              <Text style={styles.requiredError}> *</Text>
            </Text>
            <TextInput
              style={[
                styles.input,
                showErrors && errors.otherIncidentCategory ? styles.inputError : null,
              ]}
              placeholder="Enter custom incident category"
              placeholderTextColor="#888"
              value={formData.otherIncidentCategory}
              onChangeText={(text) => handleChange('otherIncidentCategory', text)}
            />
            {showErrors && errors.otherIncidentCategory && (
              <Text style={styles.errorText}>{errors.otherIncidentCategory}</Text>
            )}
          </View>
        )}
      </View>
      <View style={styles.section}>
        <Text style={styles.incidentInfoTitle}>Other Information</Text>
        <View style={styles.inputContainer}>
          <Text style={styles.label}>
            Incident Area
            <Text style={styles.requiredError}> *</Text>
          </Text>
          <TextInput
            style={[styles.input, showErrors && errors.incidentArea ? styles.inputError : null]}
            placeholder="Incident Area"
            placeholderTextColor="#A0A0A0"
            value={formData.incidentArea}
            onChangeText={(text) => handleChange('incidentArea', text)}
          />
          {showErrors && errors.incidentArea && (
            <Text style={styles.errorText}>{errors.incidentArea}</Text>
          )}
        </View>
        <CustomInputField
          label="Person involved in incident"
          value={formData.incidentDate2}
          onChangeText={(text) => handleChange('incidentDate2', text)}
          placeholder="Person involved in incident"
        />
        <View style={[styles.inputContainer, { zIndex: 10 }]}>
          <Text style={styles.label}>Incident Severity Rating</Text>
          <Dropdown
            style={[styles.input, !formData.severity && styles.placeholder]}
            data={severityOptions}
            labelField="label"
            valueField="value"
            placeholder="Select Severity"
            value={formData.severity}
            onChange={(item) => handleChange('severity', item.value)}
            renderRightIcon={() => (
              <MaterialIcons name="arrow-drop-down" size={22} color="#015185" style={{ marginRight: 8 }} />
            )}
            dropdownPosition="bottom"
            containerStyle={styles.dropdownContainer}
            itemTextStyle={styles.dropdownItemText}
            selectedTextStyle={styles.dropdownText}
            placeholderStyle={[styles.dropdownText, styles.placeholderStyle]}
          />
        </View>
      </View>
      <View style={styles.selectRow}>
        <Text style={styles.selectLabel}>GPS Location</Text>
        <TouchableOpacity onPress={handleGetLocation} style={styles.selectButton}>
          <MaterialIcons name="location-on" size={16} color="#0078D4" />
          <Text style={styles.selectText}>Get Location</Text>
        </TouchableOpacity>
      </View>
      <View style={styles.gpsDetailsBox}>
        {formData.coordinates ? (
          <View style={styles.gpsInfoRow}>
            <Text style={styles.gpsLabel}>Coordinates: </Text>
            <Text style={styles.gpsValue}>{formData.coordinates}</Text>
          </View>
        ) : null}
        {formData.gpsAddress ? (
          <View style={styles.gpsInfoRow}>
            <Text style={styles.gpsLabel}>GPS Address: </Text>
            <Text style={styles.gpsValue}>{formData.gpsAddress}</Text>
          </View>
        ) : null}
        {!formData.coordinates && !formData.gpsAddress && (
          <Text style={styles.noGpsText}>Click "Get Location" to fetch your current location</Text>
        )}
      </View>
      <View style={styles.section}>
        <View style={styles.titleContainer}>
          <Text style={[styles.sectionTitle, styles.blueTitle]}>
            <Text style={styles.asterisk}>*</Text> Does the incident involve any equipment?
          </Text>
          {showErrors && errors.involvesEquipment && (
            <Text style={[styles.errorText, styles.incidentTypeError]}>{errors.involvesEquipment}</Text>
          )}
        </View>
        <View style={styles.checkboxContainer}>
          <View style={styles.checkboxRow}>
            <TouchableOpacity
              onPress={() => handleEquipmentToggle(true)}
              style={styles.customCheckbox}
            >
              <MaterialIcons
                name={involvesEquipment === true ? 'check-box' : 'check-box-outline-blank'}
                size={20}
                color="#015185"
              />
            </TouchableOpacity>
            <Text style={styles.checkboxLabel}>Yes</Text>
          </View>
          <View style={styles.checkboxRow}>
            <TouchableOpacity
              onPress={() => handleEquipmentToggle(false)}
              style={styles.customCheckbox}
            >
              <MaterialIcons
                name={involvesEquipment === false ? 'check-box' : 'check-box-outline-blank'}
                size={20}
                color="#015185"
              />
            </TouchableOpacity>
            <Text style={styles.checkboxLabel}>No</Text>
          </View>
        </View>
      </View>
      {involvesEquipment && (
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>Equipment Detail</Text>
            <TouchableOpacity
              style={styles.selectButton}
              onPress={() =>
                navigation.navigate('IncidentPickEquipment', {
                  onSelect: (equipment) => {
                    handleEquipmentSelect(equipment);
                  },
                })
              }
            >
              <Text style={styles.selectButtonText}>Select</Text>
            </TouchableOpacity>
          </View>
          <CustomInputField
            label="Equipment #"
            value={formData.vehicleId}
            onChangeText={(text) => handleChange('vehicleId', text)}
            placeholder="Enter or Select Equipment #"
          />
          <CustomInputField
            label="Equipment Name"
            value={formData.vehicleName}
            onChangeText={(text) => handleChange('vehicleName', text)}
            placeholder="Enter or Select Equipment Name"
          />
        </View>
      )}
      <View style={styles.section}>
        <View style={styles.titleContainer}>
          <Text style={[styles.sectionTitle, styles.blueTitle]}>
            <Text style={styles.asterisk}>*</Text> Type of event (Check one or more)
          </Text>
          {showErrors && errors.incidentTypes && (
            <Text style={[styles.errorText, styles.incidentTypeError]}>{errors.incidentTypes}</Text>
          )}
        </View>
        <View style={styles.checkboxContainer}>
          {Object.keys(selectedIncidents).map((incident, index) => (
            <View key={index} style={styles.checkboxRow}>
              <TouchableOpacity
                onPress={() => handleIncidentToggle(incident)}
                style={styles.customCheckbox}
              >
                <MaterialIcons
                  name={selectedIncidents[incident] ? 'check-box' : 'check-box-outline-blank'}
                  size={20}
                  color="#015185"
                />
              </TouchableOpacity>
              <Text style={styles.checkboxLabel}>{incident}</Text>
            </View>
          ))}
        </View>
        {selectedIncidents['Other'] && (
          <CustomInputField
            label="Note"
            value={formData.otherIncidentNote}
            onChangeText={(text) => handleChange('otherIncidentNote', text)}
            placeholder="Add Note"
            multiline
            numberOfLines={4}
            style={{
              height: 100,
              textAlignVertical: 'top',
              borderWidth: 1,
              borderColor: '#BDBDBD',
              borderRadius: 6,
            }}
          />
        )}
      </View>
    </ScrollView>
  );

  return (
    <View style={styles.container}>
      <IncidentReportHeader currentStep={currentStep} setCurrentStep={setCurrentStep} />
      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : undefined}
        keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
        enabled
      >
        <ScrollView 
          style={styles.formContainer}
          contentContainerStyle={{ ...styles.formContentContainer, paddingBottom: 10 }}
          keyboardShouldPersistTaps="handled"
          showsVerticalScrollIndicator={true}
          bounces={true}
          automaticallyAdjustKeyboardInsets={true}
        >
          {currentStep === 1 && renderStep1()}
          {(currentStep === 2 || currentStep === 3) && (
            <IncidentReportSteps
              ref={stepsRef}
              currentStep={currentStep}
              formData={formData}
              selectedIncidents={selectedIncidents}
              involvesEquipment={involvesEquipment}
            />
          )}
        </ScrollView>
      </KeyboardAvoidingView>
      <View style={[styles.buttonContainer, { paddingBottom: 10 }]}> 
        <TouchableOpacity style={styles.cancelButton} onPress={handleCancel}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>
            {currentStep === 3 ? 'Generate Report' : 'Next'}
          </Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      {/* iOS Date & Time Picker Modal - Both in one popup */}
      {Platform.OS === 'ios' && dateTimeModalVisible && (
        <RNModal
          visible={dateTimeModalVisible}
          transparent
          animationType="slide"
          onRequestClose={handleDateTimeCancel}
          presentationStyle="pageSheet"
        >
          <View style={styles.modalOverlay}>
            <View style={styles.datePickerContainer}>
              <View style={styles.datePickerHeader}>
                <TouchableOpacity onPress={handleDateTimeCancel} style={styles.datePickerCancelButton}>
                  <Text style={styles.datePickerButtonText}>Cancel</Text>
                </TouchableOpacity>
                <Text style={styles.datePickerTitle}>Select Date & Time</Text>
                <TouchableOpacity onPress={handleDateTimeDone} style={styles.datePickerDoneButton}>
                  <Text style={styles.datePickerDoneText}>Done</Text>
                </TouchableOpacity>
              </View>
              
              <View style={styles.iosPickerContainer}>
                {/* Date Picker */}
                <Text style={styles.pickerLabel}>Date</Text>
                <DateTimePicker
                  value={tempDateTime instanceof Date && !isNaN(tempDateTime.getTime()) ? tempDateTime : new Date()}
                  mode="date"
                  display="inline"
                  onChange={handleDateChange}
                  maximumDate={new Date()}
                  style={styles.datePicker}
                  accentColor="#015185"
                  textColor="#333"
                />
                
                {/* Time Picker */}
                <Text style={styles.pickerLabel}>Time</Text>
                <DateTimePicker
                  value={tempDateTime instanceof Date && !isNaN(tempDateTime.getTime()) ? tempDateTime : new Date()}
                  mode="time"
                  display="spinner"
                  onChange={handleTimeChange}
                  style={styles.timePicker}
                  is24Hour={true}
                  accentColor="#015185"
                  textColor="#333"
                />
              </View>
            </View>
          </View>
        </RNModal>
      )}

      {/* Android Date Picker - Simple like Add_equipment.js */}
      {Platform.OS === 'android' && showDatePicker && (
        <DateTimePicker
          testID="dateTimePicker"
          value={datePickerValue}
          mode="date"
          display="default"
          onChange={handleDateChange}
          maximumDate={new Date()}
          minimumDate={new Date(1900, 0, 1)}
          accentColor="#015185"
          textColor="#333"
        />
      )}

      {/* Android Time Picker */}
      {Platform.OS === 'android' && showTimePicker && (
        <DateTimePicker
          testID="timePicker"
          value={timePickerValue}
          mode="time"
          display="default"
          onChange={handleTimeChange}
          is24Hour={true}
          accentColor="#015185"
          textColor="#333"
        />
      )}
      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message="Fetching Incident Number"
      />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  keyboardContainer: {
    flex: 1,
  },
  formContainer: {
    flex: 1,
    backgroundColor: '#fff',
    padding: 10,
  },
  formContentContainer: {
    paddingBottom: 10, // Reduced from 100
    flexGrow: 1,
  },
  section: {
    backgroundColor: '#F5F5F5',
    borderRadius: 8,
    padding: 12,
    marginBottom: 15,
  },
  sectionHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  inputContainer: {
    marginBottom: 10,
    width: '100%',
  },
  input: {
    flex: 1,
    height: 44,
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#BDBDBD',
    borderRadius: 4,
    paddingHorizontal: 12,
    fontSize: 16,
  },
  inputError: {
    borderColor: 'red',
  },
  dateText: {
    fontSize: 14,
    color: '#333',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  selectButton: {
    paddingHorizontal: 15,
    paddingVertical: 5,
    borderRadius: 4,
    flexDirection: 'row',
    alignContent: 'center',
    alignItems: 'baseline',
  },
  selectButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '500',
  },
  checkboxContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    width: '50%',
    marginBottom: 8,
  },
  customCheckbox: {
    marginRight: 8,
  },
  checkboxLabel: {
    fontSize: 14,
    color: '#707173',
    marginRight: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 10, // Reduced from 15
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    // paddingBottom: 30, // Remove or reduce this if present
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  label: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 3,
    fontWeight: '500',
  },
  errorText: {
    color: 'red',
    fontSize: 12,
    marginTop: 2,
  },
  blueTitle: {
    color: '#015185',
  },
  titleContainer: {
    marginBottom: 15,
  },
  incidentTypeError: {
    marginTop: 4,
    marginBottom: 0,
  },
  asterisk: {
    color: 'red',
    marginLeft: 5,
    fontSize: 16,
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#BDBDBD',
    borderRadius: 4,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  dropdownItemContainer: {},
  dropdownItemText: {
    fontSize: 16,
    color: '#333',
  },
  dropdownText: {
    fontSize: 16,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  inputSearchStyle: {
    height: 40,
    fontSize: 16,
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
    opacity: 0.6,
  },
  placeholder: {
    borderColor: '#BDBDBD',
  },
  incidentInfoCard: {
    backgroundColor: '#F5F5F5',
    borderRadius: 8,
    padding: 15,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  incidentInfoTitle: {
    fontSize: 16,
    fontWeight: '700',
    color: '#015185',
    marginBottom: 10,
  },
  selectRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 5,
  },
  selectLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  selectText: {
    fontSize: 16,
    color: '#0078D4',
    marginLeft: 5,
  },
  gpsDetailsBox: {
    width: '100%',
    backgroundColor: '#f0f0f0',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginBottom: 10,
    borderRadius: 5,
    alignSelf: 'center',
  },
  gpsInfoRow: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: 5,
  },
  gpsLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  gpsValue: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  noGpsText: {
    fontSize: 14,
    color: '#888',
    textAlign: 'center',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  // Date Picker Modal Styles
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    marginHorizontal: width * 0.05,
    marginVertical: Platform.OS === 'ios' ? width * 0.125 : width * 0.25,
    maxHeight: Platform.OS === 'ios' ? '80%' : undefined,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
    overflow: 'hidden',
  },
  datePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: width * 0.05,
    paddingVertical: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    backgroundColor: '#f8f8f8',
  },
  datePickerTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#333',
  },
  datePickerCancelButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
  },
  datePickerDoneButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    backgroundColor: '#015185',
    borderRadius: width * 0.015,
  },
  datePickerDoneText: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#fff',
  },
  datePickerButtonText: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
  },
  iosPickerContainer: {
    paddingHorizontal: width * 0.05,
    paddingVertical: width * 0.03,
  },
  pickerLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    marginTop: width * 0.02,
    marginBottom: width * 0.015,
  },
  androidPickerContainer: {
    paddingHorizontal: width * 0.05,
    paddingVertical: width * 0.03,
    alignItems: 'center',
  },
  datePicker: {
    width: Platform.OS === 'ios' ? '100%' : width * 0.75,
    height: Platform.OS === 'ios' ? width * 0.4 : undefined,
  },
  timePicker: {
    width: Platform.OS === 'ios' ? '100%' : width * 0.75,
    height: Platform.OS === 'ios' ? width * 0.3 : undefined,
    marginTop: Platform.OS === 'ios' ? width * 0.02 : 0,
  },
  // Total Incidents Counter Styles
  totalIncidentsContainer: {
    marginBottom: 15,
  },
  totalIncidentsCard: {
    backgroundColor: '#E3F2FD',
    borderRadius: 8,
    padding: 15,
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  totalIncidentsTextContainer: {
    marginLeft: 12,
    flex: 1,
  },
  totalIncidentsLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 2,
  },
  totalIncidentsCount: {
    fontSize: 24,
    color: '#015185',
    fontWeight: 'bold',
  },
});

export default CreateIncidentReport;