import React, { useState, useEffect, useCallback, useRef, useMemo } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Modal,
  Animated,
  Dimensions,
  Alert,
  BackHandler,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { MongoAPI, USERS_API } from '../ConfigMongo/MongoDBConfig';

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message || 'Processing...'}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Skeleton Component for Loading State
const Skeleton = () => {
  const deviceWidth = Dimensions.get('window').width;
  const boxSize = Math.floor((deviceWidth - 5 * 20) / 4);
  const minBoxSize = Math.max(boxSize, 70);
  const fadeGray = '#f5f6fa';
  const fadeGrayDark = '#e6e6e6';

  return (
    <ScrollView style={{ flex: 1, backgroundColor: '#fff' }}>
      <View style={{ paddingHorizontal: 5, paddingVertical: 10, alignItems: 'center', width: '100%' }}>
        {[
          { title: 'Master Data', items: 3 },
          { title: 'Inspection and Compliances', items: 5 },
          { title: 'Maintenance & Premium Services', items: 3 },
          { title: 'Logbook', items: 2 },
          { title: 'HR', items: 3 },
        ].map((section, index) => {
          const itemsPerRow = 4;
          const rows = Math.ceil(section.items / itemsPerRow);
          const itemsArray = Array.from({ length: section.items });
          let renderedRows = [];
          for (let row = 0; row < rows; row++) {
            const startIdx = row * itemsPerRow;
            const endIdx = startIdx + itemsPerRow;
            const rowItems = itemsArray.slice(startIdx, endIdx);
            renderedRows.push(
              <View
                key={row}
                style={{
                  flexDirection: 'row',
                  justifyContent: 'center',
                  alignItems: 'flex-start',
                  marginBottom: 18,
                  width: '100%',
                }}
              >
                {rowItems.map((_, idx) => (
                  <View
                    key={idx}
                    style={{
                      alignItems: 'center',
                      marginHorizontal: 16,
                      width: minBoxSize,
                    }}
                  >
                    <View
                      style={{
                        backgroundColor: fadeGray,
                        borderRadius: 12,
                        width: minBoxSize,
                        height: minBoxSize,
                        justifyContent: 'center',
                        alignItems: 'center',
                        marginBottom: 8,
                      }}
                    />
                    <View
                      style={{
                        height: 13,
                        width: '80%',
                        backgroundColor: fadeGrayDark,
                        borderRadius: 4,
                        alignSelf: 'center',
                      }}
                    />
                  </View>
                ))}
              </View>
            );
          }
          return (
            <View key={section.title} style={{ width: '100%' }}>
              <View
                style={{
                  height: 24,
                  width: '60%',
                  backgroundColor: fadeGrayDark,
                  borderRadius: 4,
                  alignSelf: 'center',
                  marginBottom: 13,
                }}
              />
              {renderedRows}
              <View style={{ alignItems: 'center', marginVertical: 15, width: '100%' }}>
                <View
                  style={{
                    width: 160,
                    height: 40,
                    borderRadius: 5,
                    backgroundColor: fadeGrayDark,
                    alignSelf: 'center',
                  }}
                />
              </View>
            </View>
          );
        })}
      </View>
    </ScrollView>
  );
};

// Get screen dimensions outside component
const { width: screenWidth, height: screenHeight } = Dimensions.get('window');

const HomeScreen = React.memo(() => {
  const width = screenWidth;
  const height = screenHeight;
  
  const navigation = useNavigation();
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [modalVisible, setModalVisible] = useState(false);
  const [userId, setUserId] = useState(null);
  const [userData, setUserData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncModalMessage, setSyncModalMessage] = useState('Processing...');
  const [dataLoaded, setDataLoaded] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [hasShownOfflineToast, setHasShownOfflineToast] = useState(false);
  const [hasShownOnlineToast, setHasShownOnlineToast] = useState(false);
  const [modules, setModules] = useState([]);
  const [selectedCountries, setSelectedCountries] = useState([]);
  const isMounted = useRef(true);

  // Valid country options
  const countryOptions = ['Zambia', 'DRC', 'Both Countries'];
  const memoizedCountryOptions = useMemo(() => countryOptions, []);

  // Module definitions with categories, navigation routes, and icons
  const moduleDefinitions = {
    Equipments: {
      category: 'Master Data',
      label: 'Equipments',
      navigation: 'Equipment_one',
      icon: require('../../Images/adminhome/icosn/eq101.png'),
    },
    Documents: {
      category: 'Master Data',
      label: 'Documents',
      navigation: 'Documenthome',
      icon: require('../../Images/adminhome/icosn/eq103.png'),
    },
    teamEmploy: {
      category: 'Master Data',
      label: 'Team/Employee',
      navigation: 'Team_employment_Home',
      icon: require('../../Images/adminhome/icosn/eq102.png'),
    },
    inspectionReport: {
      category: 'Inspection and Compliances',
      label: 'Inspection Report',
      navigation: 'Inspection_Report_Home',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    hseInspection: {
      category: 'Inspection and Compliances',
      label: 'HSE Inspection',
      navigation: 'HSEInspectionHome',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    dvirReport: {
      category: 'Inspection and Compliances',
      label: 'DVIR Inspection',
      navigation: 'DviranddotInHome',
      icon: require('../../Images/adminhome/icosn/eq103.png'),
    },
    checklist: {
      category: 'Inspection and Compliances',
      label: 'Check List',
      navigation: 'Checklist_home',
      icon: require('../../Images/adminhome/icosn/checklist1.png'),
    },
    incidentReport: {
      category: 'Inspection and Compliances',
      label: 'Incident Report',
      navigation: 'IncodentReport',
      icon: require('../../Images/adminhome/icosn/ins0101.png'),
    },
    dailyReport: {
      category: 'Inspection and Compliances',
      label: 'Daily Inspection',
      navigation: 'DailyIns_report_home',
      icon: require('../../Images/adminhome/icosn/solar_calendar-linear.png'),
    },
    requestMaintenance: {
      category: 'Maintenance & Premium Services',
      label: 'Request Maintenance',
      navigation: 'RequestmaintenanceHome',
      icon: require('../../Images/adminhome/icosn/RM01.png'),
    },
    jobcard: {
      category: 'Maintenance & Premium Services',
      label: 'Job Card',
      navigation: 'WorkOrdertab',
      icon: require('../../Images/adminhome/icosn/jobcard1.png'),
    },
    serviceSchedule: {
      category: 'Maintenance & Premium Services',
      label: 'Service Schedule',
      navigation: 'ServiceScheduleHome',
      icon: require('../../Images/adminhome/icosn/schdule01.png'),
    },
    fuelLog: {
      category: 'Logbook',
      label: 'Fuel Log',
      navigation: 'Fuel_Log_List',
      icon: require('../../Images/adminhome/icosn/fuel-log.png'),
    },
    logbook: {
      category: 'Logbook',
      label: 'Logbook',
      navigation: 'LogbookList',
      icon: require('../../Images/adminhome/icosn/logbook1.png'),
    },
    hrForm: {
      category: 'HR',
      label: 'HR Form',
      navigation: 'AdminCheckAllApli',
      icon: require('../../Images/adminhome/icosn/hrform01.png'),
    },
    hrApplications: {
      category: 'HR',
      label: 'HR Applications',
      navigation: 'AdminHrDocument',
      icon: require('../../Images/adminhome/icosn/hrdocuments01.png'),
    },
    teamKpi: {
      category: 'HR',
      label: 'Team KPI',
      navigation: 'HRReportingHome',
      icon: require('../../Images/adminhome/icosn/Hrreporting01.png'),
    },
  };

  // Check network status
  const checkNetworkStatus = useCallback(async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  }, []);

  // Show toast notification
  const showToast = useCallback((type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  }, []);

  // Fetch user data from MongoDB
  const fetchUserDataFromMongoDB = useCallback(
    async (uid) => {
      if (!isMounted.current) return;
      try {
        console.log('🔄 Fetching user data from MongoDB...');
        setSyncModalVisible(true);
        setSyncModalMessage('Loading User Data');

        // Use MongoAPI from config
        const result = await MongoAPI.getUserById(uid);
        if (result.success && result.data) {
          const user = result.data;
          setUserData(user);
          
          // Extract modules from user data
          if (user.modules && Array.isArray(user.modules)) {
            const moduleNames = user.modules
              .map((mod) => mod.module)
              .filter((moduleName) => moduleDefinitions[moduleName]);
            
            if (moduleNames.length > 0) {
              setModules(moduleNames);
              await AsyncStorage.setItem('modules', JSON.stringify(moduleNames));
              console.log('✅ Modules loaded from MongoDB:', moduleNames);
            } else {
              const defaultModules = ['Equipments', 'Documents', 'teamEmploy', 'hseInspection'];
              setModules(defaultModules);
              await AsyncStorage.setItem('modules', JSON.stringify(defaultModules));
              console.log('⚠️ No valid modules, using defaults:', defaultModules);
            }
          }

          // Set countries
          if (user.countries && Array.isArray(user.countries)) {
            setSelectedCountries(user.countries);
          }

          // Store user data locally
          await AsyncStorage.setItem('userData', JSON.stringify(user));
          console.log('✅ User data loaded from MongoDB:', user.name);
        }
      } catch (error) {
        console.error('❌ Error fetching user data from MongoDB:', error);
        showToast('error', 'Error', 'Failed to load user data');
        
        // Try to load from AsyncStorage as fallback
        const storedData = await AsyncStorage.getItem('userData');
        if (storedData) {
          const user = JSON.parse(storedData);
          setUserData(user);
          if (user.modules) {
            const moduleNames = user.modules
              .map((mod) => mod.module)
              .filter((moduleName) => moduleDefinitions[moduleName]);
            setModules(moduleNames);
          }
          if (user.countries) {
            setSelectedCountries(user.countries);
          }
        }
      } finally {
        if (isMounted.current) {
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [showToast]
  );

  // Update country in MongoDB
  const updateCountryInMongoDB = useCallback(
    async (userId, countries) => {
      if (!isMounted.current || !userId) return;

      setLoading(true);
      setSyncModalVisible(true);
      setSyncModalMessage('Updating Countries');

      try {
        console.log('🔄 Updating countries in MongoDB...');
        
        // Use MongoAPI from config
        const result = await MongoAPI.updateUser(userId, {
          countries: countries,
        });

        if (result.success) {
          await AsyncStorage.setItem('selectedCountries', JSON.stringify(countries));
          console.log('✅ Countries updated in MongoDB:', countries);
          showToast('success', 'Success', 'Countries updated successfully');
        }
      } catch (error) {
        console.error('❌ Error updating countries:', error);
        showToast('error', 'Error', 'Failed to update countries');
      } finally {
        if (isMounted.current) {
          setLoading(false);
          setSyncModalVisible(false);
          setSyncModalMessage('Processing...');
        }
      }
    },
    [showToast]
  );

  // Handle country selection
  const handleCountrySelect = useCallback(
    async (country) => {
      if (!isMounted.current) return;
      try {
        if (!countryOptions.includes(country)) {
          showToast('error', 'Error', 'Invalid country selected');
          return;
        }

        setSelectedCountry(country);
        setModalVisible(false);

        // Update in local state
        await AsyncStorage.setItem('selectedCountry', country);
        console.log(`✅ Country selected: ${country}`);
        showToast('success', 'Success', `Country updated to ${country}`);
      } catch (error) {
        console.error('Error selecting country:', error);
        showToast('error', 'Error', error.message);
      }
    },
    [showToast]
  );

  // Handle module press
  const handlePress = useCallback(
    (action, actionName) => {
      if (!isMounted.current) return;
      if (!selectedCountry) {
        setModalVisible(true);
        showToast('error', 'Error', 'Please select a country first');
        return;
      }
      action();
    },
    [selectedCountry, showToast]
  );

  // Handle back button press
  useFocusEffect(
    useCallback(() => {
      const backAction = () => {
        Alert.alert(
          'Exit App',
          'Are you sure you want to exit the app?',
          [
            {
              text: 'Cancel',
              style: 'cancel',
            },
            {
              text: 'Exit',
              style: 'destructive',
              onPress: () => BackHandler.exitApp(),
            },
          ]
        );
        return true;
      };

      const backHandler = BackHandler.addEventListener('hardwareBackPress', backAction);
      return () => backHandler.remove();
    }, [])
  );

  // Load data on mount
  useEffect(() => {
    let netInfoUnsubscribe;
    let loadingTimeout;

    const loadData = async () => {
      try {
        const onlineStatus = await checkNetworkStatus();
        setIsOnline(onlineStatus);

        if (!onlineStatus && !hasShownOfflineToast) {
          showToast('info', 'Offline Mode', 'Operating in offline mode');
          setHasShownOfflineToast(true);
        }

        // Get user ID from AsyncStorage
        const uid = await AsyncStorage.getItem('userUid');
        setUserId(uid);

        if (uid) {
          if (onlineStatus) {
            // Fetch from MongoDB
            await fetchUserDataFromMongoDB(uid);
          } else {
            // Load from AsyncStorage
            const storedData = await AsyncStorage.getItem('userData');
            if (storedData) {
              const user = JSON.parse(storedData);
              setUserData(user);
              
              if (user.modules) {
                const moduleNames = user.modules
                  .map((mod) => mod.module)
                  .filter((moduleName) => moduleDefinitions[moduleName]);
                setModules(moduleNames);
              }
              
              if (user.countries) {
                setSelectedCountries(user.countries);
              }
            }
          }
        }

        // Get selected country from storage
        const storedCountry = await AsyncStorage.getItem('selectedCountry');
        if (storedCountry && countryOptions.includes(storedCountry)) {
          setSelectedCountry(storedCountry);
        }

        setDataLoaded(true);
        setLoading(false);
        
      } catch (error) {
        console.error('Error loading data:', error);
        setDataLoaded(true);
        setLoading(false);
        showToast('error', 'Error', 'Failed to load user data');
      }
    };

    // Set timeout to prevent infinite loading
    loadingTimeout = setTimeout(() => {
      if (isMounted.current && loading) {
        console.log('Loading timeout reached');
        setDataLoaded(true);
        setLoading(false);
        showToast('info', 'Loading Timeout', 'Data loading took too long');
      }
    }, 15000);

    loadData().catch((error) => {
      console.error('Load data failed:', error);
      setDataLoaded(true);
      setLoading(false);
    });

    // Network listener
    netInfoUnsubscribe = NetInfo.addEventListener((state) => {
      if (!isMounted.current) return;

      const newOnlineStatus = state.isConnected;
      setIsOnline(newOnlineStatus);

      if (!newOnlineStatus && !hasShownOfflineToast) {
        showToast('info', 'Offline Mode', 'Operating in offline mode');
        setHasShownOfflineToast(true);
      } else if (newOnlineStatus && !hasShownOnlineToast) {
        showToast('success', 'Online', 'Now online');
        setHasShownOnlineToast(true);
        setHasShownOfflineToast(false);
        
        // Refresh data when back online
        if (userId) {
          fetchUserDataFromMongoDB(userId).catch((error) => 
            console.error('Fetch user data failed:', error)
          );
        }
      }
    });

    return () => {
      isMounted.current = false;
      if (netInfoUnsubscribe) netInfoUnsubscribe();
      if (loadingTimeout) clearTimeout(loadingTimeout);
    };
  }, [checkNetworkStatus, showToast, fetchUserDataFromMongoDB]);

  // Handler for toggling country selection
  const handleToggleCountry = (country) => {
    setSelectedCountries((prev) => {
      if (country === 'Both Countries') {
        const hasBoth = prev.includes('Zambia') && prev.includes('DRC');
        if (hasBoth) {
          return prev.filter((c) => c !== 'Zambia' && c !== 'DRC' && c !== 'Both Countries');
        } else {
          return ['Zambia', 'DRC'];
        }
      } else {
        let newSelected;
        if (prev.includes(country)) {
          newSelected = prev.filter((c) => c !== country && c !== 'Both Countries');
        } else {
          newSelected = [...prev.filter((c) => c !== 'Both Countries'), country];
        }
        if (newSelected.includes('Zambia') && newSelected.includes('DRC')) {
          return ['Zambia', 'DRC'];
        }
        return newSelected;
      }
    });
  };

  // Handler for saving selected countries
  const handleSaveCountries = async () => {
    if (!userId) return;
    setSyncModalVisible(true);
    setSyncModalMessage('Updating Countries...');
    try {
      await updateCountryInMongoDB(userId, selectedCountries);
      setModalVisible(false);
    } catch (error) {
      showToast('error', 'Error', 'Failed to update countries');
    } finally {
      setSyncModalVisible(false);
      setSyncModalMessage('Processing...');
    }
  };

  // Render category with dynamic modules
  const renderCategory = (category, navigationRoute) => {
    const boxSize = Math.floor((width - 5 * 10) / 4);
    const minBoxSize = Math.max(boxSize, 70);

    // Get modules for this category
    const categoryModules = modules
      .filter((mod) => moduleDefinitions[mod]?.category === category)
      .map((mod) => ({
        icon: moduleDefinitions[mod].icon,
        label: moduleDefinitions[mod].label,
        onPress: () => handlePress(() => navigation.navigate(moduleDefinitions[mod].navigation), mod),
      }));

    // Special handling for Inspection and Compliances category
    if (category === 'Inspection and Compliances') {
      const hseModule = {
        icon: moduleDefinitions.hseInspection.icon,
        label: moduleDefinitions.hseInspection.label,
        onPress: () => handlePress(() => navigation.navigate(moduleDefinitions.hseInspection.navigation), 'hseInspection'),
      };

      const otherModules = categoryModules.filter(mod => mod.label !== 'HSE Inspection');
      const allModules = [hseModule, ...otherModules];

      if (allModules.length === 0) return null;

      const firstRow = allModules.slice(0, 4);
      const secondRow = allModules.slice(4);

      return (
        <>
          <Text style={styles.heading}>{category}</Text>
          <View style={{ alignItems: 'center', width: '100%', paddingHorizontal: 5 }}>
            <View style={{
              flexDirection: 'row',
              justifyContent: 'center',
              alignItems: 'flex-start',
              marginBottom: 12,
              width: '100%',
            }}>
              {firstRow.map((item) => (
                <TouchableOpacity
                  key={item.label}
                  style={{ alignItems: 'center', marginHorizontal: 8 }}
                  onPress={item.onPress}
                  activeOpacity={0.8}
                >
                  <View style={{
                    backgroundColor: '#035484',
                    borderRadius: 12,
                    width: minBoxSize,
                    height: minBoxSize,
                    justifyContent: 'center',
                    alignItems: 'center',
                    marginBottom: 8,
                  }}>
                    <Image source={item.icon} style={{
                      width: item.label === 'Daily Report' ? 42 : 48,
                      height: item.label === 'Daily Report' ? 42 : 48,
                      resizeMode: 'contain',
                    }} />
                  </View>
                  <Text style={{
                    color: '#29486A',
                    fontWeight: 'bold',
                    fontSize: 12,
                    textAlign: 'center',
                    width: minBoxSize,
                    lineHeight: 16,
                    flexWrap: 'wrap',
                    numberOfLines: 2,
                  }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                    {item.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
            {secondRow.length > 0 && (
              <View style={{
                flexDirection: 'row',
                justifyContent: 'center',
                alignItems: 'flex-start',
                marginBottom: 12,
                width: '100%',
              }}>
                {secondRow.map((item) => (
                  <TouchableOpacity
                    key={item.label}
                    style={{ alignItems: 'center', marginHorizontal: 8 }}
                    onPress={item.onPress}
                    activeOpacity={0.8}
                  >
                    <View style={{
                      backgroundColor: '#035484',
                      borderRadius: 12,
                      width: minBoxSize,
                      height: minBoxSize,
                      justifyContent: 'center',
                      alignItems: 'center',
                      marginBottom: 8,
                    }}>
                      <Image source={item.icon} style={{
                        width: item.label === 'Daily Report' ? 42 : 48,
                        height: item.label === 'Daily Report' ? 42 : 48,
                        resizeMode: 'contain',
                      }} />
                    </View>
                    <Text style={{
                      color: '#29486A',
                      fontWeight: 'bold',
                      fontSize: 12,
                      textAlign: 'center',
                      width: minBoxSize,
                      lineHeight: 16,
                      flexWrap: 'wrap',
                      numberOfLines: 2,
                    }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                      {item.label}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            )}
            <View style={styles.buttonContainer}>
              <TouchableOpacity
                style={styles.gradientButton}
                onPress={() => handlePress(() => navigation.navigate(navigationRoute), `View${category.replace(/ & /g, '').replace(/\s/g, '')}KPI`)}
                activeOpacity={0.8}
              >
                <LinearGradient colors={['#3481BC', '#025383']} style={styles.gradientButtonContent}>
                  <Text style={styles.buttonText}>View KPI</Text>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          </View>
        </>
      );
    }

    // For other categories
    if (categoryModules.length === 0) return null;

    const firstRow = categoryModules.slice(0, 4);
    const secondRow = categoryModules.slice(4);

    return (
      <>
        <Text style={styles.heading}>{category}</Text>
        <View style={{ alignItems: 'center', width: '100%', paddingHorizontal: 5 }}>
          <View style={{
            flexDirection: 'row',
            justifyContent: 'center',
            alignItems: 'flex-start',
            marginBottom: 12,
            width: '100%',
          }}>
            {firstRow.map((item, idx) => (
              <TouchableOpacity
                key={item.label}
                style={{ alignItems: 'center', marginHorizontal: 8 }}
                onPress={item.onPress}
                activeOpacity={0.8}
              >
                <View style={{
                  backgroundColor: '#035484',
                  borderRadius: 12,
                  width: minBoxSize,
                  height: minBoxSize,
                  justifyContent: 'center',
                  alignItems: 'center',
                  marginBottom: 8,
                }}>
                  <Image source={item.icon} style={{
                    width: item.label === 'Daily Report' ? 42 : 48,
                    height: item.label === 'Daily Report' ? 42 : 48,
                    resizeMode: 'contain'
                  }} />
                </View>
                <Text style={{
                  color: '#29486A',
                  fontWeight: 'bold',
                  fontSize: 12,
                  textAlign: 'center',
                  width: minBoxSize,
                  lineHeight: 16,
                  flexWrap: 'wrap',
                  numberOfLines: 2,
                }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                  {item.label}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
          {secondRow.length > 0 && (
            <View style={{
              flexDirection: 'row',
              justifyContent: 'center',
              alignItems: 'flex-start',
              marginBottom: 12,
              width: '100%',
            }}>
              {secondRow.map((item, idx) => (
                <TouchableOpacity
                  key={item.label}
                  style={{ alignItems: 'center', marginHorizontal: 8 }}
                  onPress={item.onPress}
                  activeOpacity={0.8}
                >
                  <View style={{
                    backgroundColor: '#035484',
                    borderRadius: 12,
                    width: minBoxSize,
                    height: minBoxSize,
                    justifyContent: 'center',
                    alignItems: 'center',
                    marginBottom: 8,
                  }}>
                    <Image source={item.icon} style={{
                      width: item.label === 'Daily Report' ? 42 : 48,
                      height: item.label === 'Daily Report' ? 42 : 48,
                      resizeMode: 'contain'
                    }} />
                  </View>
                  <Text style={{
                    color: '#29486A',
                    fontWeight: 'bold',
                    fontSize: 12,
                    textAlign: 'center',
                    width: minBoxSize,
                    lineHeight: 16,
                    flexWrap: 'wrap',
                    numberOfLines: 2,
                  }} numberOfLines={2} adjustsFontSizeToFit={true} minimumFontScale={0.8}>
                    {item.label}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
          )}
          <View style={styles.buttonContainer}>
            <TouchableOpacity
              style={styles.gradientButton}
              onPress={() => handlePress(() => navigation.navigate(navigationRoute), `View${category.replace(/ & /g, '').replace(/\s/g, '')}KPI`)}
              activeOpacity={0.8}
            >
              <LinearGradient colors={['#3481BC', '#025383']} style={styles.gradientButtonContent}>
                <Text style={styles.buttonText}>View KPI</Text>
              </LinearGradient>
            </TouchableOpacity>
          </View>
        </View>
      </>
    );
  };

  // Country dropdown
  const CountryDropdown = useMemo(() => (
    <TouchableOpacity
      style={{
        borderWidth: 2,
        borderColor: '#035484',
        paddingHorizontal: 7,
        paddingVertical: 3,
        minWidth: '40%',
        backgroundColor: '#fff',
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'space-between',
        shadowColor: '#000',
        shadowOpacity: 0.08,
        shadowRadius: 4,
        shadowOffset: { width: 0, height: 2 },
        elevation: 2,
        marginBottom: 10,
      }}
      onPress={() => setModalVisible(true)}
      activeOpacity={0.8}
    >
      <Text style={{ color: selectedCountries.length > 0 ? '#222' : '#888', fontSize: 16, flex: 1 }}>
        {selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')
          ? 'Both Countries'
          : selectedCountries.length > 0
            ? selectedCountries.join(', ')
            : 'Choose Country'}
      </Text>
      <MaterialIcons name="keyboard-arrow-down" size={28} color="#035484" />
    </TouchableOpacity>
  ), [selectedCountries]);

  // Country modal
  const CountryModal = useMemo(() => (
    <Modal
      animationType="fade"
      transparent={true}
      visible={modalVisible}
      onRequestClose={() => setModalVisible(false)}
    >
      <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.10)', justifyContent: 'center', alignItems: 'center' }}>
        <View style={{ backgroundColor: '#fff', borderRadius: 14, width: 320, paddingBottom: 18, shadowColor: '#000', shadowOpacity: 0.12, shadowRadius: 12, shadowOffset: { width: 0, height: 4 }, elevation: 8 }}>
          <View style={{ backgroundColor: '#F6F7F9', borderTopLeftRadius: 14, borderTopRightRadius: 14, paddingVertical: 16, alignItems: 'center', borderBottomWidth: 1, borderBottomColor: '#eee' }}>
            <Text style={{ color: '#035484', fontWeight: 'bold', fontSize: 20 }}>Choose Countries</Text>
          </View>
          {memoizedCountryOptions.map((option, idx) => (
            <TouchableOpacity
              key={option}
              style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 16, paddingHorizontal: 22, borderBottomWidth: idx !== memoizedCountryOptions.length - 1 ? 1 : 0, borderBottomColor: '#eee', backgroundColor: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? '#F6F7F9' : '#fff' }}
              onPress={() => handleToggleCountry(option)}
              activeOpacity={0.8}
            >
              <Text style={{ color: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? '#035484' : '#222', fontWeight: (option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? 'bold' : 'normal', fontSize: 18 }}>{option}</Text>
              <View style={{ width: 28, height: 28, borderRadius: 14, borderWidth: 2, borderColor: '#035484', alignItems: 'center', justifyContent: 'center', backgroundColor: '#fff' }}>
                {(option === 'Both Countries' ? (selectedCountries.includes('Zambia') && selectedCountries.includes('DRC')) : selectedCountries.includes(option)) ? (
                  <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: '#035484' }} />
                ) : null}
              </View>
            </TouchableOpacity>
          ))}
          <View style={{ flexDirection: 'row', justifyContent: 'center', marginTop: 10 }}>
            <TouchableOpacity
              style={{ backgroundColor: '#035484', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 36, marginTop: 6, marginRight: 10 }}
              onPress={handleSaveCountries}
              activeOpacity={0.8}
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Save</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={{ backgroundColor: '#888', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 36, marginTop: 6 }}
              onPress={() => setModalVisible(false)}
              activeOpacity={0.8}
            >
              <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 16 }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  ), [modalVisible, selectedCountries, memoizedCountryOptions, handleToggleCountry, handleSaveCountries]);

  // Conditional rendering
  if (loading || !dataLoaded) {
    return <Skeleton />;
  }

  if (modules.length === 0) {
    return <Skeleton />;
  }

  return (
    <>
      <ScrollView style={styles.scrollContainer}>
        <View style={styles.container}>
          {/* Top bar with country dropdown */}
          {CountryDropdown}

          {/* Country Selection Modal */}
          {CountryModal}

          {/* SyncModal */}
          <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message={syncModalMessage} />

          {/* Render Categories Dynamically */}
          {renderCategory('Master Data', 'masterdatakpi')}
          {renderCategory('Inspection and Compliances', 'InspectionKpi')}
          {renderCategory('Maintenance & Premium Services', 'MaintenanceKpiHome')}
          {renderCategory('Logbook', 'LogbookKPI')}
          {renderCategory('HR', 'HrModuleKpI')}

          {/* Quick Access: Speech to Text button */}
          <View style={{ width: '100%', alignItems: 'center', marginTop: 12, marginBottom: 18 }}>
            <TouchableOpacity
              style={[styles.initializeButton, { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 18 }]}
              onPress={() => navigation.navigate('SpeechToText')}
              activeOpacity={0.8}
            >
              <MaterialIcons name="keyboard-voice" size={20} color="#fff" />
              <Text style={[styles.initializeButtonText, { marginLeft: 10, fontWeight: '700' }]}>Speech To Text</Text>
            </TouchableOpacity>
          </View>

          {/* Equipment MongoDB Upload button */}
          <View style={{ width: '100%', alignItems: 'center', marginTop: 12, marginBottom: 18 }}>
            <TouchableOpacity
              style={[styles.initializeButton, { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 18, backgroundColor: '#0055A5' }]}
              onPress={() => navigation.navigate('UploadEquipmentToMongo')}
              activeOpacity={0.8}
            >
              <MaterialIcons name="cloud-upload" size={20} color="#fff" />
              <Text style={[styles.initializeButtonText, { marginLeft: 10, fontWeight: '700' }]}>Equipment Mongo</Text>
            </TouchableOpacity>
          </View>
        </View>
      </ScrollView>
      <Toast />
    </>
  );
});

const styles = StyleSheet.create({
  scrollContainer: {
    flex: 1,
    backgroundColor: '#fff',
    width: '100%',
  },
  container: {
    paddingHorizontal: 5,
    paddingVertical: 10,
    alignItems: 'center',
    width: '100%',
  },
  heading: {
    fontSize: 24,
    fontWeight: 'bold',
    textAlign: 'center',
    marginBottom: 10,
    color: '#015185',
    alignSelf: 'center',
  },
  buttonContainer: {
    alignItems: 'center',
    marginVertical: 10,
    width: '100%',
  },
  gradientButton: {
    width: 160,
    borderRadius: 5,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  gradientButtonContent: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    height: 40,
  },
  buttonText: {
    color: 'white',
    fontSize: 16,
    fontWeight: 'bold',
    marginRight: 5,
    textAlign: 'center',
    alignSelf: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  initializeButton: {
    backgroundColor: '#035484',
    borderRadius: 6,
    paddingVertical: 8,
    paddingHorizontal: 15,
    marginLeft: 10,
  },
  initializeButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
});

export default HomeScreen;
