import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Image,
  StatusBar,
  Dimensions,
  RefreshControl,
  Alert,
  Platform,
  Animated,
  Modal,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useFocusEffect, useRoute } from '@react-navigation/native';
import NoticeAPI from '../ConfigMongo/NoticeMongoConfig';
import NotificationAPI from '../ConfigMongo/NotificationMongoConfig';

const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;
const TOTAL_HEADER_HEIGHT = width * 0.25;
const NOTIFICATIONS_PER_PAGE = 16;

// SyncModal Component with animated sync.png
const SyncModal = ({ visible, message = 'Loading Notifications...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalOverlay}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HeaderWellNotificaton = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const [items, setItems] = useState([]); // Combined list of notifications and notices
  const [loading, setLoading] = useState(true);
  const [refreshing, setRefreshing] = useState(false);
  const [currentUserId, setCurrentUserId] = useState('');
  const [unreadCount, setUnreadCount] = useState(0);
  const [filter, setFilter] = useState('all'); // all, unread, read
  const [highlightedNotificationId, setHighlightedNotificationId] = useState(null);
  const scrollViewRef = useRef(null);
  const [unreadNoticeCount, setUnreadNoticeCount] = useState(0);

  // Pagination states
  const [lastItemKey, setLastItemKey] = useState(null);
  const [hasMoreItems, setHasMoreItems] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const [allItems, setAllItems] = useState([]);

  useEffect(() => {
    getCurrentUserId();
  }, []);

  useEffect(() => {
    if (route.params?.highlightedNotificationId) {
      setHighlightedNotificationId(route.params.highlightedNotificationId);
      console.log('🔴 Highlighted notification ID:', route.params.highlightedNotificationId);
    }
  }, [route.params]);

  useEffect(() => {
    if (currentUserId) {
      loadAllItems(false, false); // Initial load with modal
      const interval = setInterval(() => loadAllItems(false, true), 5000); // Background refresh without modal
      return () => clearInterval(interval);
    }
  }, [currentUserId]);

  useFocusEffect(
    useCallback(() => {
      if (currentUserId) {
        loadAllItems(false, false); // Show modal when screen comes into focus
      }
    }, [currentUserId])
  );

  const getCurrentUserId = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (userId) {
        setCurrentUserId(userId);
        console.log('Current user ID:', userId);
      }
    } catch (error) {
      console.error('Error getting current user ID:', error);
    }
  };

  const loadAllItems = async (loadMore = false, isBackgroundRefresh = false) => {
    if (!currentUserId || (loadMore && !hasMoreItems)) return;

    if (loadMore) {
      setLoadingMore(true);
    } else if (!isBackgroundRefresh) {
      // Only show loading modal if NOT a background refresh
      setLoading(true);
    }

    try {
      console.log(isBackgroundRefresh ? '🔄 Background refresh...' : '📱 Loading items from MongoDB...');

      // Load notifications from MongoDB (with fallback if service is down)
      let notificationsArray = [];
      try {
        const notificationsResponse = await NotificationAPI.getAllNotifications(currentUserId);
        if (notificationsResponse.success && notificationsResponse.data) {
          notificationsArray = notificationsResponse.data.map(notification => ({
            id: notification.notificationId || notification._id,
            ...notification,
            timestamp: notification.timestamp || Date.now(),
            isRead: notification.isRead || false,
            type: 'notification',
          }));
          console.log(`✅ Loaded ${notificationsArray.length} notifications from MongoDB`);
        } else {
          console.warn('⚠️ Notifications service returned no data');
        }
      } catch (notifError) {
        console.warn('⚠️ Notifications service unavailable, continuing with notices only:', notifError.message);
        // Continue loading notices even if notifications fail
      }

      // Load recent notices from MongoDB
      let noticesArray = [];
      try {
        const noticesResponse = await NoticeAPI.getAllNotices(currentUserId);
        if (noticesResponse.success && noticesResponse.data) {
          noticesArray = noticesResponse.data.map(notice => ({
            noticeId: notice.noticeId,
            title: notice.title,
            description: notice.description,
            imageUrl: notice.imageUrl || null,
            createdAt: notice.createdAt,
            createdBy: notice.createdBy,
            isRead: notice.isRead || false,
            timestamp: new Date(notice.createdAt).getTime(),
            type: 'notice',
          })).slice(0, 10);
          console.log(`✅ Loaded ${noticesArray.length} notices from MongoDB`);
          setUnreadNoticeCount(noticesResponse.unreadCount || 0);
        } else {
          console.warn('⚠️ Notices service returned no data');
        }
      } catch (noticeError) {
        console.warn('⚠️ Notices service unavailable:', noticeError.message);
      }

      // Combine and sort by timestamp (newest first)
      const allItems = [...notificationsArray, ...noticesArray].sort((a, b) => b.timestamp - a.timestamp);

      console.log(`✅ Total items: ${allItems.length} (${notificationsArray.length} notifications + ${noticesArray.length} notices)`);

      if (loadMore) {
        const startIndex = allItems.length;
        const endIndex = startIndex + NOTIFICATIONS_PER_PAGE;
        const nextBatch = allItems.slice(startIndex, endIndex);
        setAllItems(prev => [...prev, ...nextBatch]);
        setItems([...prev, ...nextBatch]);
        setHasMoreItems(endIndex < allItems.length);
      } else {
        const initialBatch = allItems.slice(0, NOTIFICATIONS_PER_PAGE);
        setAllItems(initialBatch);
        setItems(initialBatch);
        setHasMoreItems(allItems.length > NOTIFICATIONS_PER_PAGE);
      }

      if (allItems.length > 0) {
        setLastItemKey(allItems[allItems.length - 1].timestamp);
      }

      // Calculate unread counts
      const notificationUnreadCount = notificationsArray.filter(item => !item.isRead).length;
      setUnreadCount(notificationUnreadCount);
    } catch (error) {
      console.error('❌ Error loading items:', error);
      // Don't show alert, just log error and continue
    } finally {
      if (loadMore) {
        setLoadingMore(false);
      } else if (!isBackgroundRefresh) {
        // Only hide loading modal if it was shown (not a background refresh)
        setLoading(false);
      }
    }
  };

  const markAsRead = async (itemId, isNotice = false) => {
    try {
      if (isNotice) {
        await NoticeAPI.markAsRead(itemId, currentUserId);
        await loadAllItems(false, true); // Silent background refresh
      } else {
        // Mark notification as read in MongoDB
        await NotificationAPI.markAsRead(itemId, currentUserId);

        const updatedItems = items.map(item =>
          item.id === itemId ? { ...item, isRead: true } : item
        );
        setItems(updatedItems);
        updateUnreadCount(updatedItems);
      }
    } catch (error) {
      console.error('Error marking item as read:', error);
    }
  };

  const markAllAsRead = async () => {
    try {
      // Mark all notifications as read in MongoDB
      await NotificationAPI.markAllAsRead(currentUserId);

      const updatedItems = items.map(item => ({ ...item, isRead: true }));
      setItems(updatedItems);
      setUnreadCount(0);
      Alert.alert('Success', 'All notifications marked as read');
    } catch (error) {
      console.error('Error marking all notifications as read:', error);
      Alert.alert('Error', 'Failed to mark all notifications as read');
    }
  };

  const deleteNotification = async (notificationId) => {
    Alert.alert(
      'Delete Notification',
      'Are you sure you want to delete this notification?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              // Delete notification from MongoDB
              await NotificationAPI.deleteNotification(notificationId, currentUserId);

              const updatedItems = items.filter(item => item.id !== notificationId);
              setItems(updatedItems);
              updateUnreadCount(updatedItems);
            } catch (error) {
              console.error('Error deleting notification:', error);
              Alert.alert('Error', 'Failed to delete notification');
            }
          },
        },
      ]
    );
  };

  const onRefresh = async () => {
    setRefreshing(true);
    setLastItemKey(null);
    setHasMoreItems(true);
    setAllItems([]);
    console.log('🔄 Refreshing items...');
    await loadAllItems(false);
    setRefreshing(false);
  };

  const handleScroll = ({ nativeEvent }) => {
    const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
    const paddingToBottom = 20;
    const isCloseToBottom = layoutMeasurement.height + contentOffset.y >= contentSize.height - paddingToBottom;

    if (isCloseToBottom && hasMoreItems && !loadingMore && !loading) {
      console.log('Loading more items...');
      loadAllItems(true);
    }
  };

  const updateUnreadCount = (itemsList) => {
    const unread = itemsList.filter(item => !item.isRead).length;
    setUnreadCount(unread - unreadNoticeCount);
  };

  const handleNoticeClick = async (notice) => {
    try {
      // Mark as read in MongoDB
      if (!notice.isRead) {
        await NoticeAPI.markAsRead(notice.noticeId, currentUserId);
        await loadAllItems(false, true); // Silent background refresh
      }
    } catch (error) {
      console.error('❌ Error marking notice as read:', error);
    }
    
    navigation.navigate('ViewNotificationsadded', {
      title: notice.title,
      description: notice.description,
      imageUrl: notice.imageUrl,
      createdAt: notice.createdAt,
    });
  };

  const getFilteredItems = () => {
    let filteredItems = [...items];

    switch (filter) {
      case 'unread':
        filteredItems = filteredItems.filter(item => !item.isRead);
        break;
      case 'read':
        filteredItems = filteredItems.filter(item => item.isRead);
        break;
      default:
        filteredItems = filteredItems;
    }

    return filteredItems.sort((a, b) => b.timestamp - a.timestamp);
  };

  const formatTimestamp = (timestamp) => {
    const now = Date.now();
    const diff = now - timestamp;
    const minutes = Math.floor(diff / (1000 * 60));
    const hours = Math.floor(diff / (1000 * 60 * 60));
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));

    if (minutes < 1) return 'Just now';
    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    if (days < 7) return `${days}d ago`;

    return new Date(timestamp).toLocaleDateString();
  };

  const getNotificationIcon = (type) => {
    switch (type) {
      case 'welcome':
        return '🎉';
      case 'safety_alert':
        return '🚨';
      case 'maintenance':
        return '🔧';
      case 'update':
        return '📢';
      case 'test':
        return '🧪';
      case 'notice_board':
        return '📢';
      case 'user_created':
        return '👤';
      case 'equipment_added':
        return '🚜';
      case 'dvir_report':
        return '🚛';
      case 'inspection_report':
        return '🔍';
      case 'hse_inspection_report':
        return '🏥';
      case 'incident_report':
        return '🚨';
      case 'request_maintenance_created':
        return '🔧';
      case 'work_order_created':
        return '📋';
      case 'job_card_created':
        return '📋';
      case 'service_schedule_created':
        return '📅';
      case 'hr_application':
        return '📋';
      case 'hr_application_approval_rejection':
        return '✅';
      case 'hr_application_supervisor_approved':
        return '📢';
      default:
        return '📱';
    }
  };

  const getNotificationColor = (type) => {
    switch (type) {
      case 'welcome':
        return '#4CAF50';
      case 'safety_alert':
        return '#F44336';
      case 'maintenance':
        return '#FF9800';
      case 'update':
        return '#2196F3';
      case 'test':
        return '#9C27B0';
      case 'notice_board':
        return '#3481BC';
      case 'user_created':
        return '#4CAF50';
      case 'equipment_added':
        return '#4CAF50';
      case 'dvir_report':
        return '#FF6B35';
      case 'inspection_report':
        return '#FF9800';
      case 'hse_inspection_report':
        return '#9C27B0';
      case 'incident_report':
        return '#F44336';
      case 'request_maintenance_created':
        return '#015185';
      case 'work_order_created':
        return '#4CAF50';
      case 'job_card_created':
        return '#4CAF50';
      case 'service_schedule_created':
        return '#3481BC';
      case 'hr_application':
        return '#015185';
      case 'hr_application_approval_rejection':
        return '#4CAF50';
      case 'hr_application_supervisor_approved':
        return '#FF9800';
      default:
        return '#607D8B';
    }
  };

  const renderItem = (item) => {
    const isHighlighted = highlightedNotificationId &&
      (item.id === highlightedNotificationId || item.noticeId === highlightedNotificationId);

    if (item.type === 'notice') {
      return (
        <TouchableOpacity
          key={item.noticeId}
          style={[styles.noticeItem, !item.isRead && styles.unreadNotice]}
          onPress={() => {
            handleNoticeClick(item);
            markAsRead(item.noticeId, true);
          }}
          activeOpacity={0.7}
        >
          <View style={styles.noticeHeader}>
            <LinearGradient
              colors={['#3481BC', '#0055A5']}
              style={styles.noticeIconContainer}
            >
              <MaterialIcons name="campaign" size={28} color="#FFFFFF" />
            </LinearGradient>
            <View style={styles.noticeContent}>
              <View style={styles.noticeTitleRow}>
                <Text
                  style={[styles.noticeTitle, !item.isRead && styles.unreadNoticeTitle]}
                  numberOfLines={2}
                >
                  {item.title}
                </Text>
              </View>
              <Text style={styles.noticeBody} numberOfLines={3}>
                {item.description}
              </Text>
            </View>
            <View style={styles.noticeRightSection}>
              <View style={styles.noticeTimeContainer}>
                <MaterialIcons name="access-time" size={12} color="#9CA3AF" />
                <Text style={styles.noticeTime}>
                  {formatTimestamp(item.timestamp)}
                </Text>
              </View>
              {!item.isRead && <View style={styles.unreadNoticeDot} />}
            </View>
          </View>
        </TouchableOpacity>
      );
    }

    return (
      <TouchableOpacity
        key={item.id}
        style={[
          styles.notificationItem,
          !item.isRead && styles.unreadNotification,
          item.type === 'user_created' && styles.userCreatedNotification,
          item.type === 'inspection_report' && styles.inspectionNotification,
          item.type === 'hse_inspection_report' && styles.hseInspectionNotification,
          item.type === 'dvir_report' && styles.dvirNotification,
          item.type === 'incident_report' && styles.incidentNotification,
          item.type === 'equipment_added' && styles.equipmentNotification,
          (item.type === 'work_order_created' || item.type === 'job_card_created') && styles.workOrderNotification,
          item.type === 'service_schedule_created' && styles.serviceScheduleNotification,
          item.type === 'hr_application' && styles.hrApplicationNewNotification,
          item.type === 'hr_application_approval_rejection' && styles.hrApplicationApprovedNotification,
          item.type === 'hr_application_supervisor_approved' && styles.hrApplicationPendingNotification,
          isHighlighted && styles.highlightedNotification,
        ]}
        onPress={() => {
          markAsRead(item.id);
          if (item.type === 'user_created') {
            navigation.navigate('Admin_Accounts', { notificationData: item, fromNotification: true });
          } else if (item.type === 'equipment_added') {
            navigation.navigate('EquipmentACHomePage', { notificationData: item, fromNotification: true });
          } else if (item.type === 'inspection_report') {
            navigation.navigate('Inspection_Report_Home', { notificationData: item, fromNotification: true });
          } else if (item.type === 'hse_inspection_report') {
            navigation.navigate('HSEInspection', { notificationData: item, fromNotification: true });
          } else if (item.type === 'dvir_report') {
            navigation.navigate('DviranddotInHome', { notificationData: item, fromNotification: true });
          } else if (item.type === 'incident_report') {
            navigation.navigate('IncodentReport', { notificationData: item, fromNotification: true });
          } else if (item.type === 'work_order_created' || item.type === 'job_card_created') {
            navigation.navigate('WorkOrdertab', { notificationData: item, fromNotification: true });
          } else if (item.type === 'service_schedule_created') {
            navigation.navigate('ServiceScheduleHome', { notificationData: item, fromNotification: true });
          } else if (item.type === 'hr_application' || item.type === 'hr_application_approval_rejection' || item.type === 'hr_application_supervisor_approved') {
            navigation.navigate('AdminHrApplyHome', { notificationData: item, fromNotification: true });
          } else {
            navigation.navigate('ViewNotificationWell', {
              title: item.title || 'No Title',
              description: item.body || 'No Description',
              timestamp: item.timestamp,
              type: item.type || 'general',
              notificationId: item.id,
            });
          }
        }}
        activeOpacity={0.7}
      >
        <View style={styles.notificationHeader}>
          <View style={styles.notificationIconContainer}>
            <MaterialIcons
              name={
                item.type === 'user_created' ? 'person-add' :
                item.type === 'inspection_report' ? 'assignment' :
                item.type === 'hse_inspection_report' ? 'health-and-safety' :
                item.type === 'dvir_report' ? 'local-shipping' :
                item.type === 'incident_report' ? 'warning' :
                item.type === 'equipment_added' ? 'precision-manufacturing' :
                item.type === 'request_maintenance_created' ? 'build' :
                item.type === 'work_order_created' || item.type === 'job_card_created' ? 'work' :
                item.type === 'service_schedule_created' ? 'event' :
                item.type === 'hr_application' ? 'assignment' :
                item.type === 'hr_application_approval_rejection' ? 'check-circle' :
                item.type === 'hr_application_supervisor_approved' ? 'notifications-active' :
                'notifications'
              }
              size={28}
              color="#FFFFFF"
            />
          </View>
          <View style={styles.notificationContent}>
            <Text style={[styles.notificationTitle, !item.isRead && styles.unreadTitle]}>
              {item.title ? item.title.substring(0, 8) + (item.title.length > 8 ? '...' : '') : 'No Title'}
            </Text>
            <Text style={styles.notificationBody} numberOfLines={1}>
              {item.body ? item.body.substring(0, 10) + (item.body.length > 10 ? '...' : '') : 'No Description'}
            </Text>
            <Text style={styles.notificationTime}>
              {formatTimestamp(item.timestamp)}
            </Text>
            {(item.type === 'user_created' || item.type === 'inspection_report' || item.type === 'hse_inspection_report' || item.type === 'dvir_report' || item.type === 'incident_report' ||
              item.type === 'equipment_added' || item.type === 'work_order_created' || item.type === 'job_card_created' ||
              item.type === 'service_schedule_created' || item.type === 'hr_application' || item.type === 'hr_application_approval_rejection' ||
              item.type === 'hr_application_supervisor_approved') && (
              <View style={styles.notificationData}>
                {item.type === 'user_created' && (item.data?.userName || item.data?.userRole) && (
                  <>
                    {item.data?.userName && (
                      <Text style={styles.dataText}>User: {item.data.userName}</Text>
                    )}
                    {item.data?.userRole && (
                      <Text style={styles.dataText}>Role: {item.data.userRole}</Text>
                    )}
                  </>
                )}
                {item.type === 'inspection_report' && item.requestNumber && (
                  <Text style={styles.dataText}>Request: {item.requestNumber}</Text>
                )}
                {item.type === 'hse_inspection_report' && (item.data?.requestNumber || item.data?.checklistName) && (
                  <>
                    {item.data?.requestNumber && (
                      <Text style={styles.dataText}>Request: {item.data.requestNumber}</Text>
                    )}
                    {item.data?.checklistName && (
                      <Text style={styles.dataText}>Checklist: {item.data.checklistName}</Text>
                    )}
                  </>
                )}
                {item.type === 'dvir_report' && item.requestNumber && (
                  <Text style={styles.dataText}>Request: {item.requestNumber}</Text>
                )}
                {item.type === 'incident_report' && item.incidentNumber && (
                  <Text style={styles.dataText}>Incident: {item.incidentNumber}</Text>
                )}
                {item.type === 'equipment_added' && item.equipmentName && (
                  <Text style={styles.dataText}>Equipment: {item.equipmentName}</Text>
                )}
                {(item.type === 'work_order_created' || item.type === 'job_card_created') && item.requestNumber && (
                  <Text style={styles.dataText}>Work Order: {item.requestNumber}</Text>
                )}
                {item.type === 'hr_application' && (item.data?.applicationType || item.data?.applicantName) && (
                  <>
                    {item.data?.applicationType && (
                      <Text style={styles.dataText}>Type: {item.data.applicationType}</Text>
                    )}
                    {item.data?.applicantName && (
                      <Text style={styles.dataText}>Applicant: {item.data.applicantName}</Text>
                    )}
                  </>
                )}
                {item.type === 'hr_application_approval_rejection' && (item.data?.applicationType || item.data?.applicantName) && (
                  <>
                    {item.data?.applicationType && (
                      <Text style={styles.dataText}>Type: {item.data.applicationType}</Text>
                    )}
                    {item.data?.applicantName && (
                      <Text style={styles.dataText}>Applicant: {item.data.applicantName}</Text>
                    )}
                  </>
                )}
                {item.type === 'hr_application_supervisor_approved' && (item.data?.applicationType || item.data?.applicantName) && (
                  <>
                    {item.data?.applicationType && (
                      <Text style={styles.dataText}>Type: {item.data.applicationType}</Text>
                    )}
                    {item.data?.applicantName && (
                      <Text style={styles.dataText}>Applicant: {item.data.applicantName}</Text>
                    )}
                  </>
                )}
              </View>
            )}
          </View>
          <View style={styles.notificationActions}>
            {!item.isRead && <View style={styles.unreadIndicator} />}
            <TouchableOpacity
              style={styles.viewButton}
              onPress={() => {
                markAsRead(item.id);
                if (item.type === 'user_created') {
                  navigation.navigate('Admin_Accounts', { notificationData: item, fromNotification: true });
                } else if (item.type === 'equipment_added') {
                  navigation.navigate('EquipmentACHomePage', { notificationData: item, fromNotification: true });
                } else if (item.type === 'inspection_report') {
                  navigation.navigate('Inspection_Report_Home', { notificationData: item, fromNotification: true });
                } else if (item.type === 'hse_inspection_report') {
                  navigation.navigate('HSEInspection', { notificationData: item, fromNotification: true });
                } else if (item.type === 'dvir_report') {
                  navigation.navigate('DviranddotInHome', { notificationData: item, fromNotification: true });
                } else if (item.type === 'incident_report') {
                  navigation.navigate('IncodentReport', { notificationData: item, fromNotification: true });
                } else if (item.type === 'work_order_created' || item.type === 'job_card_created') {
                  navigation.navigate('WorkOrdertab', { notificationData: item, fromNotification: true });
                } else if (item.type === 'service_schedule_created') {
                  navigation.navigate('ServiceScheduleHome', { notificationData: item, fromNotification: true });
                } else if (item.type === 'hr_application' || item.type === 'hr_application_approval_rejection' || item.type === 'hr_application_supervisor_approved') {
                  navigation.navigate('AdminHrApplyHome', { notificationData: item, fromNotification: true });
                } else {
                  navigation.navigate('ViewNotificationWell', {
                    title: item.title || 'No Title',
                    description: item.body || 'No Description',
                    timestamp: item.timestamp,
                    type: item.type || 'general',
                    notificationId: item.id,
                  });
                }
              }}
            >
              <Text style={styles.viewButtonText}>
                {item.type === 'user_created' ? 'View User' :
                 item.type === 'equipment_added' ? 'View Equipment' :
                 item.type === 'inspection_report' ? 'View Report' :
                 item.type === 'hse_inspection_report' ? 'View HSE' :
                 item.type === 'dvir_report' ? 'View DVIR' :
                 item.type === 'incident_report' ? 'View Incident' :
                 (item.type === 'work_order_created' || item.type === 'job_card_created') ? 'View Job Order' :
                 item.type === 'service_schedule_created' ? 'View Schedule' :
                 (item.type === 'hr_application' || item.type === 'hr_application_approval_rejection' || item.type === 'hr_application_supervisor_approved') ? 'View Application' : 'View'}
              </Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.deleteButton}
              onPress={() => deleteNotification(item.id)}
            >
              <MaterialIcons name="delete-outline" size={20} color="#FF6B6B" />
            </TouchableOpacity>
          </View>
        </View>
      </TouchableOpacity>
    );
  };

  const renderEmptyState = () => (
    <View style={styles.emptyState}>
      <MaterialIcons name="notifications-none" size={80} color="#BDBDBD" />
      <Text style={styles.emptyStateTitle}>No Notifications</Text>
      <Text style={styles.emptyStateSubtitle}>
        {filter === 'all'
          ? "You're all caught up! No items yet."
          : filter === 'unread'
          ? "No unread items."
          : "No read items."
        }
      </Text>
    </View>
  );

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />

      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
          >
            <MaterialIcons name="arrow-back" size={width * 0.07} color="white" />
          </TouchableOpacity>

          <Text style={styles.headerTitle}>Notifications</Text>

          <View style={styles.headerActions}>
            {(unreadCount > 0 || unreadNoticeCount > 0) && (
              <View style={styles.unreadBadgeContainer}>
                <View style={styles.unreadBadge}>
                  <Text style={styles.unreadBadgeText}>
                    {unreadCount + unreadNoticeCount}
                  </Text>
                </View>
              </View>
            )}
            {unreadCount > 0 && (
              <TouchableOpacity
                style={styles.markAllReadButton}
                onPress={markAllAsRead}
              >
                <MaterialIcons name="done-all" size={20} color="white" />
              </TouchableOpacity>
            )}
          </View>
        </View>
      </LinearGradient>

      {/* Filter Tabs */}
      <View style={styles.filterContainer}>
        <ScrollView horizontal showsHorizontalScrollIndicator={false}>
          <TouchableOpacity
            style={[styles.filterTab, filter === 'all' && styles.activeFilterTab]}
            onPress={() => setFilter('all')}
          >
            <Text style={[styles.filterText, filter === 'all' && styles.activeFilterText]}>
              All ({items.length})
            </Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={styles.allNoticesFilterButton}
            onPress={() => navigation.navigate('Notice_BoardAC')}
            activeOpacity={0.7}
          >
            <MaterialIcons name="list-alt" size={16} color="#3481BC" />
            <Text style={styles.allNoticesFilterText}>
              All Notices {unreadNoticeCount > 0 && `(${unreadNoticeCount})`}
            </Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={[styles.filterTab, filter === 'unread' && styles.activeFilterTab]}
            onPress={() => setFilter('unread')}
          >
            <Text style={[styles.filterText, filter === 'unread' && styles.activeFilterText]}>
              Unread ({unreadCount})
            </Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={[styles.filterTab, filter === 'read' && styles.activeFilterTab]}
            onPress={() => setFilter('read')}
          >
            <Text style={[styles.filterText, filter === 'read' && styles.activeFilterText]}>
              Read ({items.filter(n => n.isRead).length})
            </Text>
          </TouchableOpacity>
        </ScrollView>
      </View>

      {/* Items List */}
      <ScrollView
        ref={scrollViewRef}
        style={styles.notificationsContainer}
        refreshControl={<RefreshControl refreshing={refreshing} onRefresh={onRefresh} />}
        showsVerticalScrollIndicator={false}
        onScroll={handleScroll}
        scrollEventThrottle={400}
      >
        {getFilteredItems().length > 0 && (
          <View style={styles.recentNoticesSection}>
            <View style={styles.sectionTitleContainer}>
              <MaterialIcons name="campaign" size={22} color="#3481BC" />
              <Text style={styles.recentNoticesSectionTitle}>Recent Items</Text>
            </View>
            {getFilteredItems().map(renderItem)}
          </View>
        )}

        {!getFilteredItems().length && renderEmptyState()}

        {loadingMore && (
          <View style={styles.loadingMoreContainer}>
            <ActivityIndicator size="small" color="#3481BC" />
            <Text style={styles.loadingMoreText}>Loading more...</Text>
          </View>
        )}

        {!hasMoreItems && items.length > 0 && (
          <View style={styles.endOfListContainer}>
            <Text style={styles.endOfListText}>No more items</Text>
          </View>
        )}

        <View style={styles.viewAllNoticesContainer}>
          <TouchableOpacity
            style={styles.viewAllNoticesButton}
            onPress={() => navigation.navigate('Notice_BoardAC')}
            activeOpacity={0.7}
          >
            <MaterialIcons name="list-alt" size={20} color="#3481BC" />
            <Text style={styles.viewAllNoticesText}>View All Notices</Text>
            <MaterialIcons name="arrow-forward" size={16} color="#3481BC" />
          </TouchableOpacity>
        </View>
      </ScrollView>

      {/* Sync Modal Loader */}
      <SyncModal visible={loading} message="Loading Notifications..." />
    </View>
  );
};

const styles = StyleSheet.create({
  // [Keep all existing styles unchanged]
  container: {
    flex: 1,
    backgroundColor: '#F8F9FA',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
    height: '100%',
    top: 7,
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    fontSize: scale(20),
    fontWeight: 'bold',
    color: 'white',
  },
  headerActions: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  markAllReadButton: {
    padding: 8,
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    borderRadius: 20,
  },
  filterContainer: {
    backgroundColor: '#FFFFFF',
    paddingVertical: 16,
    paddingHorizontal: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#E8EEF5',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 3,
  },
  filterTab: {
    paddingHorizontal: 18,
    paddingVertical: 11,
    marginHorizontal: 6,
    borderRadius: 25,
    backgroundColor: '#F8F9FA',
    borderWidth: 1.5,
    borderColor: '#E8EEF5',
  },
  activeFilterTab: {
    backgroundColor: '#3481BC',
    borderColor: '#3481BC',
    elevation: 3,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  filterText: {
    fontSize: scale(14),
    color: '#4a5568',
    fontWeight: '600',
    letterSpacing: 0.2,
  },
  activeFilterText: {
    color: '#FFFFFF',
    fontWeight: '700',
  },
  allNoticesFilterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#E8F4FD',
    paddingHorizontal: 18,
    paddingVertical: 11,
    marginHorizontal: 6,
    borderRadius: 25,
    borderWidth: 1.5,
    borderColor: '#3481BC',
    minWidth: 100,
    elevation: 2,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 3,
  },
  allNoticesFilterText: {
    fontSize: scale(14),
    color: '#0055A5',
    fontWeight: '700',
    marginLeft: 6,
    letterSpacing: 0.2,
  },
  notificationsContainer: {
    flex: 1,
    paddingHorizontal: 20,
  },
  notificationItem: {
    backgroundColor: 'white',
    borderRadius: 12,
    marginVertical: 8,
    padding: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  unreadNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#3481BC',
    backgroundColor: '#F8F9FF',
  },
  userCreatedNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50',
    backgroundColor: '#F1F8E9',
  },
  inspectionNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#FF9800',
    backgroundColor: '#FFFBE6',
  },
  hseInspectionNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#9C27B0',
    backgroundColor: '#F3E5F5',
  },
  dvirNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#FF6B35',
    backgroundColor: '#FFF5F0',
  },
  incidentNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#F44336',
    backgroundColor: '#FFEBEE',
  },
  equipmentNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50',
    backgroundColor: '#F1F8E9',
  },
  workOrderNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50',
    backgroundColor: '#F1F8E9',
  },
  serviceScheduleNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#3481BC',
    backgroundColor: '#E3F2FD',
  },
  hrApplicationNewNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
    backgroundColor: '#E8F4FD',
  },
  hrApplicationApprovedNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50',
    backgroundColor: '#E8F5E9',
  },
  hrApplicationPendingNotification: {
    borderLeftWidth: 4,
    borderLeftColor: '#FF9800',
    backgroundColor: '#FFF3E0',
  },
  highlightedNotification: {
    borderWidth: 3,
    borderColor: '#3481BC',
    backgroundColor: '#F0F8FF',
    elevation: 8,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
  },
  notificationHeader: {
    flexDirection: 'row',
    alignItems: 'flex-start',
  },
  notificationIconContainer: {
    width: 50,
    height: 50,
    borderRadius: 25,
    backgroundColor: '#F0F0F0',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 15,
  },
  notificationContent: {
    flex: 1,
    marginRight: 10,
  },
  notificationTitle: {
    fontSize: scale(16),
    fontWeight: '600',
    color: '#333',
    marginBottom: 5,
  },
  unreadTitle: {
    fontWeight: '700',
    color: '#3481BC',
  },
  notificationBody: {
    fontSize: scale(14),
    color: '#666',
    lineHeight: 20,
    marginBottom: 8,
  },
  notificationTime: {
    fontSize: scale(12),
    color: '#999',
  },
  notificationActions: {
    alignItems: 'center',
  },
  unreadIndicator: {
    width: 8,
    height: 8,
    borderRadius: 4,
    backgroundColor: '#3481BC',
    marginBottom: 8,
  },
  deleteButton: {
    padding: 5,
  },
  notificationData: {
    marginTop: 12,
    paddingTop: 12,
    borderTopWidth: 1,
    borderTopColor: '#F0F0F0',
  },
  dataText: {
    fontSize: scale(12),
    color: '#888',
    fontFamily: 'monospace',
    marginBottom: 2,
  },
  emptyState: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 60,
  },
  emptyStateTitle: {
    fontSize: scale(18),
    fontWeight: '600',
    color: '#666',
    marginTop: 20,
    marginBottom: 10,
  },
  emptyStateSubtitle: {
    fontSize: scale(14),
    color: '#999',
    textAlign: 'center',
    paddingHorizontal: 40,
    lineHeight: 20,
  },
  loadingMoreContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 20,
    backgroundColor: 'white',
    borderRadius: 12,
    marginVertical: 10,
  },
  loadingMoreText: {
    fontSize: scale(14),
    color: '#666',
    marginLeft: 10,
  },
  endOfListContainer: {
    alignItems: 'center',
    paddingVertical: 20,
    backgroundColor: 'white',
    borderRadius: 12,
    marginVertical: 10,
  },
  endOfListText: {
    fontSize: scale(14),
    color: '#999',
  },
  unreadBadgeContainer: {
    marginRight: 10,
  },
  unreadBadge: {
    backgroundColor: '#F44336',
    borderRadius: 12,
    minWidth: 24,
    height: 24,
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 6,
  },
  unreadBadgeText: {
    color: 'white',
    fontSize: 12,
    fontWeight: 'bold',
  },
  recentNoticesSection: {
    marginTop: 10,
    marginBottom: 20,
  },
  sectionTitleContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 15,
    paddingHorizontal: 5,
  },
  recentNoticesSectionTitle: {
    fontSize: scale(18),
    fontWeight: '800',
    color: '#1a1a1a',
    marginLeft: 8,
    letterSpacing: 0.3,
  },
  noticeItem: {
    backgroundColor: '#FFFFFF',
    borderRadius: 16,
    marginVertical: 6,
    padding: 18,
    elevation: 3,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 3 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    borderWidth: 1,
    borderColor: '#E8EEF5',
  },
  unreadNotice: {
    borderLeftWidth: 5,
    borderLeftColor: '#3481BC',
    backgroundColor: '#F7FAFF',
    elevation: 4,
    shadowOpacity: 0.18,
    borderColor: '#D0E1F9',
  },
  noticeHeader: {
    flexDirection: 'row',
    alignItems: 'flex-start',
  },
  noticeIconContainer: {
    width: 56,
    height: 56,
    borderRadius: 28,
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 14,
    elevation: 4,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 3 },
    shadowOpacity: 0.35,
    shadowRadius: 5,
  },
  noticeContent: {
    flex: 1,
    marginRight: 12,
  },
  noticeTitleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 6,
  },
  noticeTitle: {
    fontSize: scale(17),
    fontWeight: '700',
    color: '#1a1a1a',
    letterSpacing: 0.2,
    lineHeight: scale(22),
    flex: 1,
  },
  unreadNoticeTitle: {
    fontWeight: '800',
    color: '#0055A5',
  },
  noticeBody: {
    fontSize: scale(15),
    color: '#4a5568',
    lineHeight: scale(21),
  },
  noticeRightSection: {
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    paddingLeft: 8,
  },
  noticeTimeContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F1F5F9',
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    marginBottom: 8,
  },
  noticeTime: {
    fontSize: scale(11),
    color: '#64748B',
    fontWeight: '600',
    marginLeft: 4,
  },
  unreadNoticeDot: {
    width: 12,
    height: 12,
    borderRadius: 6,
    backgroundColor: '#3481BC',
    borderWidth: 2,
    borderColor: '#FFFFFF',
    elevation: 2,
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.4,
    shadowRadius: 3,
  },
  viewAllNoticesContainer: {
    backgroundColor: 'white',
    marginTop: 20,
    marginHorizontal: 20,
    borderRadius: 12,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  viewAllNoticesButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 16,
    paddingHorizontal: 20,
    backgroundColor: '#f8f9ff',
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 12,
  },
  viewAllNoticesText: {
    fontSize: scale(16),
    fontWeight: '600',
    color: '#3481BC',
    marginHorizontal: 12,
  },
  viewButton: {
    paddingHorizontal: 10,
    paddingVertical: 5,
    backgroundColor: '#3481BC',
    borderRadius: 15,
    marginRight: 10,
  },
  viewButtonText: {
    color: 'white',
    fontSize: scale(12),
    fontWeight: '600',
  },
  syncModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default HeaderWellNotificaton;