import { View, Text, TouchableOpacity, TextInput, StyleSheet, Dimensions, ScrollView, Alert, ActivityIndicator } from 'react-native';
import React, { useState } from 'react';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

const { width } = Dimensions.get('window');

// HR Application Notifications API URL
const HR_NOTIFICATIONS_API_URL = 'http://168.231.113.219:5027';

const ApplicationApprovReject = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { application, applicationType, action, approverRole, applicationId, existingReason, isEdit } = route.params;
  const [rejectReason, setRejectReason] = useState(existingReason || '');
  const [loading, setLoading] = useState(false);

  const handleSubmit = async () => {
    // Validate reason is provided
    if (!rejectReason.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Reason Required',
        text2: `Please provide a reason for ${action === 'reject' ? 'rejection' : 'approval'}`,
        position: 'bottom',
      });
      return;
    }

    // Show confirmation dialog
    Alert.alert(
      `${action === 'approve' ? 'Approve' : 'Reject'} Application`,
      `Are you sure you want to ${action === 'approve' ? 'approve' : 'reject'} this application?\n\nReason: ${rejectReason}`,
      [
        {
          text: 'Cancel',
          style: 'cancel',
        },
        {
          text: action === 'approve' ? 'Approve' : 'Reject',
          style: action === 'approve' ? 'default' : 'destructive',
          onPress: () => performAction(),
        },
      ]
    );
  };

  const performAction = async () => {
    setLoading(true);
    try {
      // Get user data from AsyncStorage
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) {
        throw new Error('User not authenticated. Please log in again.');
      }

      const user = JSON.parse(userData);
      const userName = user.name || user.fullName || (approverRole === 'HrManager' ? 'HR Manager' : approverRole === 'FinanceManager' ? 'Finance Manager' : 'Supervisor');
      const userUid = user._id || user.id;

      if (!userUid) {
        throw new Error('User ID not found. Please log in again.');
      }

      console.log('🔄 Updating application status:', { 
        applicationId, 
        action, 
        approverRole, 
        userName, 
        userUid 
      });

      // Use application ID from application object if params one is not available
      const finalApplicationId = applicationId || application?.id || application?.applicationId;
      console.log('🔍 Final application ID to use:', finalApplicationId);

      if (!finalApplicationId) {
        throw new Error('No application ID found for updating status');
      }

      // Prepare the reason text
      const reasonText = action === 'reject' ? rejectReason : (action === 'approve' ? `Approved by ${approverRole}` : `Rejected by ${approverRole}`);
      
      console.log('📝 Reason text to store:', reasonText);
      console.log('📝 Action:', action);
      console.log('📝 Approver role:', approverRole);

      // Call MongoDB API to update status
      const response = await fetch(`http://168.231.113.219:5013/hr-applications/${finalApplicationId}/status`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          status: action === 'approve' ? 'Approved' : 'Rejected',
          comments: reasonText,
          approverUid: userUid,
          approverName: userName,
          approverRole: approverRole,
          reason: reasonText, // Store reason separately for easy access
          timestamp: new Date().toISOString()
        }),
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
      }

      const result = await response.json();
      console.log('✅ Status update result:', result);

      if (result.success) {
        // Send notifications based on approval flow
        setTimeout(async () => {
          try {
            console.log('🔔 Sending notifications for approval/rejection...');
            console.log('🔍 Approval context:', { approverRole, action, userName });
            
            // Determine which notification endpoint to call
            if (approverRole === 'Supervisor' && action === 'approve') {
              // Supervisor approved → Notify HR Managers (filtered by country + project)
              console.log('📧 Supervisor approved → Notifying HR Managers');
              const notifResponse = await fetch(`${HR_NOTIFICATIONS_API_URL}/api/hr-notifications/supervisor-approval`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                  applicationData: {
                    _id: application._id || application.id || application.applicationId,
                    id: application.id || application._id || application.applicationId,
                    applicationNumber: application.applicationNumber,
                    applicationType: application.applicationType || applicationType,
                    fullName: application.fullName,
                    employeeNo: application.employeeNo,
                    department: application.department,
                    country: application.country,
                    project: application.project,
                    status: 'Approved by Supervisor',
                  },
                  approverName: userName,
                }),
              });
              if (notifResponse.ok) {
                const notifResult = await notifResponse.json();
                console.log('✅ HR Managers notified:', notifResult);
              } else {
                console.warn('⚠️ Failed to notify HR Managers:', notifResponse.status);
              }
            } else {
              // All other cases → Notify Applicant (who created the application)
              console.log('📧 Notifying applicant of decision');
              console.log('📋 Applicant ID:', application.adminUid || application.userUid);
              const notifResponse = await fetch(`${HR_NOTIFICATIONS_API_URL}/api/hr-notifications/application-decision`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                  applicationData: {
                    _id: application._id || application.id || application.applicationId,
                    id: application.id || application._id || application.applicationId,
                    applicationNumber: application.applicationNumber,
                    applicationType: application.applicationType || applicationType,
                    fullName: application.fullName,
                    employeeNo: application.employeeNo,
                    department: application.department,
                    country: application.country,
                    project: application.project,
                  },
                  status: action === 'approve' ? 'Approved' : 'Rejected',
                  approverName: userName,
                  approverRole: approverRole,
                  reason: reasonText,
                  applicantId: application.adminUid || application.userUid,
                }),
              });
              if (notifResponse.ok) {
                const notifResult = await notifResponse.json();
                console.log('✅ Applicant notified:', notifResult);
              } else {
                console.warn('⚠️ Failed to notify applicant:', notifResponse.status);
              }
            }
            console.log('✅ Notification process completed');
          } catch (notifError) {
            console.error('❌ Error sending notifications:', notifError);
            console.error('❌ Error details:', notifError.message);
          }
        }, 500);

        Toast.show({
          type: 'success',
          text1: `Application ${action === 'approve' ? 'Approved' : 'Rejected'}`,
          text2: `Successfully ${action === 'approve' ? 'approved' : 'rejected'} by ${approverRole}`,
          position: 'bottom',
        });
        // Go back to ApplicationDetail, which will refresh on focus
        navigation.goBack();
      } else {
        throw new Error(result.message || 'Failed to update application status');
      }
    } catch (error) {
      console.error('❌ Error updating application status:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to update application status',
        position: 'bottom',
      });
    } finally {
      setLoading(false);
    }
  };

  const getDetails = () => {
    const details = [
      { label: 'Full Name', value: application.fullName },
      { label: 'Employee No', value: application.employeeNo },
    ];
    switch (applicationType) {
      case 'salary_advance_application':
        details.push(
          { label: 'Amount', value: application.salaryAdvanceAmount },
          { label: 'Department', value: application.salaryAdvanceDepartment },
          { label: 'Recovery Period', value: application.selectedRecoveryPeriod },
          { label: 'Sites', value: application.selectedSites?.join(', ') },
          { label: 'Reasons', value: application.selectedReasons?.join(', ') },
        );
        break;
      case 'leave_application':
        details.push(
          { label: 'Start Date', value: application.leaveStartDate },
          { label: 'End Date', value: application.leaveEndDate },
          { label: 'Reason', value: application.reasonOfLeave },
        );
        break;
      case 'funeral_grant_form':
        details.push(
          { label: 'Deceased Name', value: application.deceasedName },
          { label: 'Relationship', value: application.deceasedRelationship },
          { label: 'Deceased Date', value: application.deceasedDate },
        );
        break;
      case 'application_for_appeal':
        details.push(
          { label: 'Offence', value: application.appealOffence },
          { label: 'Offence Date', value: application.appealOffenceDate },
        );
        break;
      case 'grievance_record':
        details.push(
          { label: 'Place of Work', value: application.grievancePlaceOfWork },
          { label: 'Details', value: application.grievanceDetails },
        );
        break;
      case 'hospital_attendance_form':
        details.push(
          { label: 'Clinics', value: application.selectedClinics?.join(', ') },
          { label: 'Shift', value: application.selectedShift },
        );
        break;
      default:
        break;
    }
    return details;
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
          <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>
          Application Detail
        </Text>
        <View style={{ width: width * 0.06 }} />
      </LinearGradient>

      <ScrollView contentContainerStyle={styles.scrollContent}>
        {/* Application Details Card */}
        <View style={styles.card}>
          <Text style={styles.cardTitle}>Application Details</Text>
          {getDetails().map((item, idx) => (
            <View key={idx} style={styles.detailRow}>
              <Text style={styles.detailLabel}>{item.label}:</Text>
              <Text style={styles.detailValue}>{item.value || 'N/A'}</Text>
            </View>
          ))}
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Current Status:</Text>
            <Text style={[
              styles.detailValue,
              {
                color: approverRole === 'HrManager'
                  ? (application.HrManager?.status === 'Approved' ? '#34C759' : 
                     application.HrManager?.status === 'Rejected' ? '#FF3B30' : '#F57C00')
                  : approverRole === 'FinanceManager'
                  ? (application.FinanceManager?.status === 'Approved' ? '#34C759' : 
                     application.FinanceManager?.status === 'Rejected' ? '#FF3B30' : '#F57C00')
                  : (application.Supervisor?.status === 'Approved' ? '#34C759' : 
                     application.Supervisor?.status === 'Rejected' ? '#FF3B30' : '#F57C00')
              }
            ]}>
              {approverRole === 'HrManager'
                ? (application.HrManager?.status || 'Pending')
                : approverRole === 'FinanceManager'
                ? (application.FinanceManager?.status || 'Pending')
                : (application.Supervisor?.status || 'Pending')}
            </Text>
          </View>
        </View>

        {/* Reason Input Card */}
        <View style={styles.card}>
          <Text style={styles.cardTitle}>
            Reason {action === 'reject' ? 'for Rejection' : 'for Approval'} 
            <Text style={{ color: '#FF3B30' }}> *</Text>
          </Text>
          <TextInput
            style={styles.input}
            placeholder={`Enter detailed reason for ${action === 'reject' ? 'rejection' : 'approval'}...`}
            value={rejectReason}
            onChangeText={(text) => {
              if (text.length <= 500) {
                setRejectReason(text);
              }
            }}
            multiline
            numberOfLines={4}
            editable={!loading}
            textAlignVertical="top"
            maxLength={500}
          />
          <Text style={styles.characterCount}>
            {rejectReason.length}/500 characters
          </Text>
        </View>

        {/* Action Buttons */}
        <View style={styles.buttonContainer}>
          <TouchableOpacity
            style={[styles.cancelBtn, { opacity: loading ? 0.6 : 1 }]}
            onPress={() => navigation.goBack()}
            disabled={loading}
          >
            <MaterialIcons name="close" size={width * 0.05} color="#666" />
            <Text style={styles.cancelBtnText}>Cancel</Text>
          </TouchableOpacity>

          <TouchableOpacity
            style={[
              styles.submitBtn, 
              {
                backgroundColor: action === 'approve' ? '#34C759' : '#FF3B30',
                opacity: loading || !rejectReason.trim() ? 0.6 : 1
              }
            ]}
            onPress={handleSubmit}
            disabled={loading || !rejectReason.trim()}
          >
            {loading ? (
              <ActivityIndicator color="#fff" size="small" />
            ) : (
              <>
                <MaterialIcons 
                  name={action === 'approve' ? 'check' : 'close'} 
                  size={width * 0.05} 
                  color="#fff" 
                />
                <Text style={styles.submitBtnText}>
                  {action === 'approve' ? 'Approve' : 'Reject'}
                </Text>
              </>
            )}
          </TouchableOpacity>
        </View>
      </ScrollView>
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#F4F6F8',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.04,
  },
  backButton: {
    padding: width * 0.02,
  },
  headerTitle: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: '700',
    color: '#fff',
    textAlign: 'center',
  },
  scrollContent: {
    padding: width * 0.04,
    paddingBottom: width * 0.08,
  },
  card: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.05,
    marginBottom: width * 0.04,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  cardTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#025383',
    marginBottom: width * 0.04,
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    marginBottom: width * 0.03,
    paddingVertical: width * 0.01,
  },
  detailLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#4A4A4A',
    flex: 1,
  },
  detailValue: {
    fontSize: width * 0.04,
    fontWeight: '500',
    color: '#1A1A1A',
    flex: 1,
    textAlign: 'right',
  },
  input: {
    borderWidth: 1,
    borderColor: '#E0E0E0',
    borderRadius: 8,
    padding: width * 0.04,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
    minHeight: 100,
    textAlignVertical: 'top',
    marginTop: width * 0.02,
  },
  characterCount: {
    fontSize: width * 0.035,
    color: '#666',
    textAlign: 'right',
    marginTop: width * 0.02,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: width * 0.04,
    gap: width * 0.03,
  },
  cancelBtn: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#f5f5f5',
    borderRadius: 8,
    paddingVertical: width * 0.04,
    borderWidth: 1,
    borderColor: '#E0E0E0',
  },
  cancelBtnText: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#666',
    marginLeft: width * 0.02,
  },
  submitBtn: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    borderRadius: 8,
    paddingVertical: width * 0.04,
  },
  submitBtnText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.04,
    marginLeft: width * 0.02,
  },
});

export default ApplicationApprovReject;
