import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Dimensions,
  SafeAreaView,
  StatusBar,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Icon from 'react-native-vector-icons/MaterialIcons';
import FontAwesome from 'react-native-vector-icons/FontAwesome';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Modal from 'react-native-modal';

const { width, height } = Dimensions.get('window');

const ApplicationDetail = ({ route }) => {
  const { application: initialApplication, applicationType } = route.params;
  const navigation = useNavigation();
  const [application, setApplication] = useState(initialApplication); // Start with initial data
  const [userRole, setUserRole] = useState('');
  const [showEdit, setShowEdit] = useState(false);
  const [showReasonModal, setShowReasonModal] = useState(false);
  const [showEditHr, setShowEditHr] = useState(false);
  const [loading, setLoading] = useState(false); // Start as false since we have initial data

  // Debug: Log the initial application data
  console.log('🔍 ApplicationDetail - Initial Application Data:', initialApplication);
  console.log('🔍 ApplicationDetail - Application Type:', applicationType);

  // Fetch user role from AsyncStorage (MongoDB-based)
  useEffect(() => {
    const fetchUserRole = async () => {
      try {
        // Get user data from AsyncStorage (set during login)
        const userData = await AsyncStorage.getItem('userData');
        if (!userData) {
          console.log('❌ No user data found in AsyncStorage');
          return;
        }

        const user = JSON.parse(userData);
        console.log('👤 User data from AsyncStorage for role:', user);

        // Get role - check multiple possible field names
        let role = 'N/A';
        if (user.role) {
          role = user.role.toLowerCase();
        } else if (Array.isArray(user.roles) && user.roles.length > 0) {
          role = user.roles[0].toLowerCase();
        } else if (user.userRole) {
          role = user.userRole.toLowerCase();
        }

        console.log('🎭 User role:', role);
        setUserRole(role);
      } catch (e) {
        console.error('❌ Error fetching user role:', e);
      }
    };
    fetchUserRole();
  }, []);

  // Fetch latest application data from MongoDB
  const fetchLatestApplication = async () => {
    try {
      // Always ensure we have initial data first
      if (!application) {
        console.log('🔄 Setting initial application data');
        setApplication(initialApplication);
        setLoading(false);
        return; // Exit early if no application data
      }
      
      const applicationId = initialApplication.id || initialApplication.applicationId;
      console.log('🔍 Application ID for MongoDB lookup:', applicationId);
      
      if (!applicationId) {
        console.log('❌ No application ID found, keeping initial data');
        return;
      }

      console.log('🔄 Attempting to fetch latest application data from MongoDB:', applicationId);

      // Try to fetch from MongoDB API with timeout
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 2000); // 2 second timeout

      try {
        const response = await fetch(`http://168.231.113.219:5013/hr-applications/entry/${applicationId}`, {
          signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
          console.log('❌ MongoDB API error:', response.status, response.statusText);
          console.log('🔄 Keeping initial data due to API error');
          return;
        }

        const result = await response.json();
        console.log('✅ MongoDB response:', result);

        // Only update if we get valid data that's not null/empty
        if (result.success && result.data && result.data !== null && Object.keys(result.data).length > 0) {
          console.log('✅ Using MongoDB data');
          setApplication(result.data);
        } else {
          console.log('❌ MongoDB response not successful or empty, keeping initial data');
        }
      } catch (fetchError) {
        clearTimeout(timeoutId);
        if (fetchError.name === 'AbortError') {
          console.log('⏰ MongoDB fetch timeout, keeping initial data');
        } else {
          console.log('❌ MongoDB fetch error:', fetchError.message);
          console.log('🔄 Keeping initial data due to fetch error');
        }
      }
    } catch (e) {
      console.error('❌ Error in fetchLatestApplication:', e);
      console.log('🔄 Keeping initial data due to error');
    }
  };

  // Only fetch from MongoDB on screen focus, not on mount
  // This prevents overriding initial data immediately

  // Only refresh on screen focus, no polling to prevent blinking
  useEffect(() => {
    const unsubscribe = navigation.addListener('focus', () => {
      console.log('🔄 Screen focused, refreshing application data');
      fetchLatestApplication();
      setShowEdit(false); // Always reset edit mode on focus
      setShowEditHr(false); // Reset HR edit mode on focus
    });
    return unsubscribe;
  }, [navigation, initialApplication, applicationType]);

  const isSupervisor = userRole === 'supervisor' || userRole === 'supervisormanager';
  const isHrManager = userRole === 'hrmanager' || userRole === 'hr manager';
  const isFinanceManager = userRole === 'financemanager' || userRole === 'finance manager';

  // Debug logging
  console.log('🔍 Role check:', { userRole, isSupervisor, isHrManager, isFinanceManager });

  const handleApprovalAction = (action) => {
    console.log('🔄 Navigating to approval screen for Supervisor action:', action);
    navigation.navigate('ApplicationApprovReject', {
      application: application || initialApplication,
      applicationType,
      action, // 'approve' or 'reject'
      approverRole: 'Supervisor',
      applicationId: (application || initialApplication).applicationId,
      existingReason: application?.Supervisor?.reason || application?.Supervisor?.comments, // Pre-fill existing reason
      isEdit: !!application?.Supervisor?.status // Indicate if this is an edit
    });
  };
  const handleHrApprovalAction = (action) => {
    console.log('🔄 Navigating to approval screen for HR Manager action:', action);
    navigation.navigate('ApplicationApprovReject', {
      application: application || initialApplication,
      applicationType,
      action, // 'approve' or 'reject'
      approverRole: 'HrManager',
      applicationId: (application || initialApplication).applicationId,
      existingReason: application?.HrManager?.reason || application?.HrManager?.comments, // Pre-fill existing reason
      isEdit: !!application?.HrManager?.status // Indicate if this is an edit
    });
  };

  const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  const getShortReason = (reason) => {
    if (!reason) return '';
    return reason.length <= 3 ? reason : reason.slice(0, 3) + '...';
  };

  const renderApplicationDetails = () => {
    if (!application) {
      console.log('❌ No application data available');
      return (
        <View style={styles.card}>
          <Text style={styles.cardTitle}>Application Details</Text>
          <Text style={{ color: '#FF3B30', textAlign: 'center', padding: 20 }}>
            No application data available. Please try refreshing.
          </Text>
        </View>
      );
    }
    
    // Debug: Log the application data structure
    console.log('🔍 Application data structure:', application);
    console.log('🔍 Available fields:', Object.keys(application));
    console.log('🔍 Application type:', applicationType);
    
    // Show all available fields for debugging
    Object.keys(application).forEach(key => {
      console.log(`🔍 Field: ${key} = ${JSON.stringify(application[key])}`);
    });
    
    // Determine the correct status to display
    const getDisplayStatus = () => {
      // If supervisor has acted, show their decision
      if (application.Supervisor && application.Supervisor.status && application.Supervisor.status !== 'Pending') {
        return application.Supervisor.status;
      }
      // If HR Manager has acted, show their decision
      if (application.HrManager && application.HrManager.status && application.HrManager.status !== 'Pending') {
        return application.HrManager.status;
      }
      // If Finance Manager has acted, show their decision
      if (application.FinanceManager && application.FinanceManager.status && application.FinanceManager.status !== 'Pending') {
        return application.FinanceManager.status;
      }
      // Otherwise show the main status or default to Pending
      return application.status || application.applicationStatus || 'Pending';
    };

    const commonFields = [
      { label: 'Full Name', value: application.fullName || application.applicantName || application.name || application.applicant_name || application.employeeName || 'N/A' },
      { label: 'Employee No', value: application.employeeNo || application.employeeNumber || application.employee_no || application.empNo || application.employeeId || 'N/A' },
      { label: 'Country', value: Array.isArray(application.country) ? application.country.join(', ') : (application.country || application.countries || application.location || 'N/A') },
      { label: 'Project', value: Array.isArray(application.project) ? application.project.join(', ') : (application.project || application.projects || application.site || 'N/A') },
      { label: 'Department', value: application.department || application.dept || application.division || 'N/A' },
      { label: 'NRC No', value: application.nrcNo || application.nrcIdNumber || application.nrc_no || application.nrc || application.idNumber || 'N/A' },
      { label: 'Submitted At', value: formatDate(application.submittedAt || application.createdAt || application.submitted_at || application.dateSubmitted || application.dateCreated) },
      { label: 'Status', value: getDisplayStatus() },
    ];

    const applicationSpecificFields = () => {
      switch (applicationType) {
        case 'leave_application':
          return [
            { label: 'Start Date', value: formatDate(application.leaveStartDate) },
            { label: 'End Date', value: formatDate(application.leaveEndDate) },
            { label: 'Report Date', value: formatDate(application.reportDate) },
            { label: 'Reason', value: application.reasonOfLeave || 'N/A' },
            { label: 'Address While on Leave', value: application.addressWhileOnLeave || 'N/A' },
            { label: 'Other Contact Name', value: application.otherContactName || 'N/A' },
            { label: 'Other Contact Cell', value: application.otherContactCell || 'N/A' },
            { label: 'Signature Date', value: formatDate(application.signatureDate) },
          ];
        case 'funeral_grant_form':
          return [
            { label: 'From Date', value: formatDate(application.funeralFromDate) },
            { label: 'To Date', value: formatDate(application.funeralToDate) },
            { label: 'Relationship Type', value: application.selectedFuneralRelationshipType || 'N/A' },
            { label: 'Deceased Name', value: application.deceasedName || 'N/A' },
            { label: 'Deceased Relationship', value: application.deceasedRelationship || 'N/A' },
            { label: 'Deceased Date', value: formatDate(application.deceasedDate) },
            { label: 'Signature Date', value: formatDate(application.funeralEmployeeSignatureDate) },
          ];
        case 'salary_advance_application':
          return [
            { label: 'Department', value: application.salaryAdvanceDepartment || application.department || application.dept || 'N/A' },
            { label: 'Amount', value: application.salaryAdvanceAmount || application.amount || application.advanceAmount || 'N/A' },
            { label: 'Sites', value: application.selectedSites?.join(', ') || application.sites || application.project || 'N/A' },
            { label: 'Other Site', value: application.salaryAdvanceOtherSite || application.otherSite || 'N/A' },
            { label: 'Recovery Period', value: application.selectedRecoveryPeriod || application.recoveryPeriod || application.repaymentPeriod || 'N/A' },
            { label: 'Reasons', value: application.selectedReasons?.join(', ') || application.reasons || application.purpose || 'N/A' },
            { label: 'Other Reason', value: application.salaryAdvanceOtherReason || application.otherReason || 'N/A' },
            { label: 'Signature Date', value: formatDate(application.salaryAdvanceSignatureDate || application.signatureDate || application.dateSigned) },
          ];
        case 'application_for_appeal':
          return [
            { label: 'Form Date', value: formatDate(application.appealFormDate || application.formDate || application.form_date || application.date) },
            { label: 'Man No', value: application.appealManNo || application.manNo || application.employeeNo || application.man_no || application.empNo || 'N/A' },
            { label: 'Name', value: application.appealName || application.applicantName || application.fullName || application.name || application.applicant_name || 'N/A' },
            { label: 'Sanction', value: application.appealSanction || application.sanction || application.sanctionType || 'N/A' },
            { label: 'Offence', value: application.appealOffence || application.offence || application.offenceType || application.offense || 'N/A' },
            { label: 'Offence Date', value: formatDate(application.appealOffenceDate || application.offenceDate || application.offence_date || application.offenseDate) },
            { label: 'Grounds for Appeal', value: application.appealGrounds || application.grounds || application.reason || application.appealReason || application.groundsForAppeal || 'N/A' },
            { label: 'Date of Appeal', value: formatDate(application.appealDateOfAppeal || application.appealDate || application.appeal_date || application.dateOfAppeal) },
          ];
        case 'grievance_record':
          return [
            { label: 'Date', value: formatDate(application.grievanceDate) },
            { label: 'Place of Work', value: application.grievancePlaceOfWork || 'N/A' },
            { label: 'Supervisor Name', value: application.grievanceSupervisorName || 'N/A' },
            { label: 'Details of Grievance', value: application.grievanceDetails || 'N/A' },
          ];
        case 'hospital_attendance_form':
          return [
            { label: 'Clinics', value: application.selectedClinics?.join(', ') || 'N/A' },
            { label: 'Other Clinic', value: application.otherClinic || 'N/A' },
            { label: 'Shift', value: application.selectedShift || 'N/A' },
          ];
        default:
          return [];
      }
    };

    return (
      <>
        <View style={styles.card}>
          <Text style={styles.cardTitle}>Application Details</Text>
          <View style={styles.fieldContainer}>
            <Text style={styles.detailLabel}>Application Type</Text>
            <Text style={styles.detailValue}>{
              (application.applicationType || applicationType || '')
                .split('_')
                .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                .join(' ')
            }</Text>
          </View>
          {commonFields.map((field, index) => {
            // Special styling for status field
            const isStatusField = field.label === 'Status';
            const statusValue = field.value;
            const statusColor = statusValue === 'Approved' ? '#34C759' : 
                              statusValue === 'Rejected' ? '#FF3B30' : 
                              statusValue === 'Pending' ? '#F57C00' : '#1A1A1A';
            
            return (
              <View key={index} style={styles.fieldContainer}>
                <Text style={styles.detailLabel}>{field.label}</Text>
                <Text style={[
                  styles.detailValue,
                  isStatusField && { color: statusColor, fontWeight: '600' }
                ]}>{field.value}</Text>
              </View>
            );
          })}
        </View>

        {applicationType === 'funeral_grant_form' && (application.funeralSpouseChildData || application.funeralMotherFatherData) && (
          <View style={styles.card}>
            <Text style={styles.cardTitle}>Expenses</Text>
            <View style={styles.table}>
              <View style={styles.tableHeaderRow}>
                <Text style={styles.tableHeaderCell}>Date</Text>
                <Text style={styles.tableHeaderCell}>Description</Text>
                <Text style={styles.tableHeaderCell}>Amount</Text>
              </View>
              {(application.funeralSpouseChildData || []).map((item, index) => (
                <View key={`spouseChild-${index}`} style={styles.tableRow}>
                  <Text style={styles.tableCell}>{formatDate(item.date)}</Text>
                  <Text style={styles.tableCell}>{item.description}</Text>
                  <Text style={styles.tableCell}>K {item.amount || '0.00'}</Text>
                </View>
              ))}
              {(application.funeralMotherFatherData || []).map((item, index) => (
                <View key={`motherFather-${index}`} style={styles.tableRow}>
                  <Text style={styles.tableCell}>{formatDate(item.date)}</Text>
                  <Text style={styles.tableCell}>{item.description}</Text>
                  <Text style={styles.tableCell}>K {item.amount || '0.00'}</Text>
                </View>
              ))}
            </View>
          </View>
        )}

        {applicationType === 'grievance_record' && application.grievanceComplaints && (
          <View style={styles.card}>
            <Text style={styles.cardTitle}>Complainants</Text>
            <View style={styles.table}>
              <View style={styles.tableHeaderRow}>
                <Text style={styles.tableHeaderCell}>Name</Text>
                <Text style={styles.tableHeaderCell}>Mine No</Text>
                <Text style={styles.tableHeaderCell}>Occupation/Grade</Text>
              </View>
              {(application.grievanceComplaints || []).map((complainant, index) => (
                <View key={index} style={styles.tableRow}>
                  <Text style={styles.tableCell}>{complainant.name || 'N/A'}</Text>
                  <Text style={styles.tableCell}>{complainant.mineNo || 'N/A'}</Text>
                  <Text style={styles.tableCell}>{complainant.occupationGrade || 'N/A'}</Text>
                </View>
              ))}
            </View>
          </View>
        )}

        {applicationSpecificFields().length > 0 && (
          <View style={styles.card}>
            <Text style={styles.cardTitle}>
              {applicationType
                .split('_')
                .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                .join(' ')}{' '}
              Details
            </Text>
            {applicationSpecificFields().map((field, index) => (
              <View key={index} style={styles.fieldContainer}>
                <Text style={styles.detailLabel}>{field.label}</Text>
                <Text style={styles.detailValue}>{field.value}</Text>
              </View>
            ))}
          </View>
        )}

        {/* Debug: Show raw data if no specific fields found */}
        {applicationSpecificFields().length === 0 && (
          <View style={styles.card}>
            <Text style={styles.cardTitle}>Raw Application Data (Debug)</Text>
            {Object.keys(application).map((key, index) => {
              const value = application[key];
              if (value !== null && value !== undefined && value !== '') {
                return (
                  <View key={index} style={styles.fieldContainer}>
                    <Text style={styles.detailLabel}>{key}:</Text>
                    <Text style={styles.detailValue}>
                      {typeof value === 'object' ? JSON.stringify(value) : String(value)}
                    </Text>
                  </View>
                );
              }
              return null;
            })}
          </View>
        )}

        <View style={styles.card}>
          <Text style={styles.cardTitle}>Approval Status</Text>
          {loading ? (
            <View style={{ padding: 20, alignItems: 'center' }}>
              <Text style={{ color: '#015185', fontSize: width * 0.045 }}>Loading status...</Text>
            </View>
          ) : (
          <View style={styles.statusTimeline}>
            {(application.Supervisor || isSupervisor) && (
              console.log('📋 Rendering Supervisor section:', { 
                hasSupervisor: !!application.Supervisor, 
                isSupervisor, 
                supervisorStatus: application.Supervisor?.status 
              }) || true
            ) && (
              <View style={styles.statusItem}>
                <View style={styles.statusIconContainer}>
                  <MaterialIcons
                    name={application.Supervisor?.status === 'Approved' ? 'check-circle' : application.Supervisor?.status === 'Rejected' ? 'cancel' : 'radio-button-unchecked'}
                    size={width * 0.05}
                    color={application.Supervisor?.status === 'Approved' ? '#34C759' : application.Supervisor?.status === 'Rejected' ? '#FF3B30' : '#FF3B30'}
                  />
                  <View
                    style={[
                      styles.statusLine,
                      application.Supervisor?.status === 'Approved' && styles.statusLineApproved,
                    ]}
                  />
                </View>
                <View style={{ flex: 1 }}>
                  <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', marginBottom: 4 }}>
                    <Text style={styles.statusLabel}>Supervisor</Text>
                    <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                      <Text style={styles.statusName}>
                        {application.Supervisor?.name && application.Supervisor.name !== 'Pending:Supervisor'
                          ? application.Supervisor.name
                          : 'N/A'}
                      </Text>
                      {isSupervisor && (
                        <MaterialIcons name="star" size={width * 0.045} color="#34C759" style={{ marginLeft: 4 }} />
                      )}
                    </View>
                  </View>
                  <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 8 }}>
                    <Text
                      style={[
                        styles.statusValue,
                        application.Supervisor?.status === 'Approved'
                          ? { color: '#34C759' }
                          : application.Supervisor?.status === 'Rejected'
                          ? { color: '#FF3B30' }
                          : {},
                      ]}
                    >
                      {application.Supervisor?.status || 'Pending'}
                    </Text>
                    {application.Supervisor?.reason && (
                      <TouchableOpacity
                        style={{ marginLeft: 8 }}
                        onPress={() => setShowReasonModal(true)}
                      >
                        <Text style={styles.rejectionReason}>
                          {getShortReason(application.Supervisor.reason)}
                        </Text>
                      </TouchableOpacity>
                    )}
                  </View>
                  {/* Show edit icon for Supervisor when there's an existing status */}
                  {isSupervisor && application.Supervisor?.status && !(showEdit || application.Supervisor.status === 'Pending' || !application.Supervisor.status) && (
                    <TouchableOpacity
                      style={styles.editStatusBtn}
                      onPress={() => setShowEdit(true)}
                    >
                      <Icon name="edit" size={width * 0.06} color="#015185" />
                    </TouchableOpacity>
                  )}
                  {/* Show Approve/Reject if no Supervisor status exists, status is Pending, or showEdit is true */}
                  {isSupervisor && (!application.Supervisor || application.Supervisor?.status === 'Pending' || !application.Supervisor?.status || showEdit) && (
                    <View style={{ flexDirection: 'row', marginTop: 10, alignItems: 'center' }}>
                      <TouchableOpacity
                        style={[styles.approvalBtn, { backgroundColor: '#34C759', marginRight: 10 }]}
                        onPress={() => { setShowEdit(false); handleApprovalAction('approve'); }}
                      >
                        <Text style={styles.approvalBtnText}>Approve</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={[styles.approvalBtn, { backgroundColor: '#FF3B30' }]}
                        onPress={() => { setShowEdit(false); handleApprovalAction('reject'); }}
                      >
                        <Text style={styles.approvalBtnText}>Reject</Text>
                      </TouchableOpacity>
                    </View>
                  )}
                </View>
              </View>
            )}
            {(application.HrManager || isHrManager) && (
              <View style={styles.statusItem}>
                <View style={styles.statusIconContainer}>
                  <MaterialIcons
                    name={application.HrManager?.status === 'Approved' ? 'check-circle' : 
                          application.HrManager?.status === 'Rejected' ? 'cancel' : 'radio-button-unchecked'}
                    size={width * 0.05}
                    color={application.HrManager?.status === 'Approved' ? '#34C759' : 
                           application.HrManager?.status === 'Rejected' ? '#FF3B30' : '#FF3B30'}
                  />
                  <View
                    style={[
                      styles.statusLine,
                      application.HrManager?.status === 'Approved' && styles.statusLineApproved,
                      !application.FinanceManager && styles.statusLineLast,
                    ]}
                  />
                </View>
                <View style={{ flex: 1 }}>
                  <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', marginBottom: 4 }}>
                    <Text style={styles.statusLabel}>HR Manager</Text>
                    <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                      <Text style={styles.statusName}>
                        {application.HrManager?.name && application.HrManager.name !== 'Pending:HrManager'
                          ? application.HrManager.name
                          : 'N/A'}
                      </Text>
                      {isHrManager && (
                        <MaterialIcons name="star" size={width * 0.045} color="#34C759" style={{ marginLeft: 4 }} />
                      )}
                    </View>
                  </View>
                  <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 8 }}>
                    <Text
                      style={[
                        styles.statusValue,
                        application.HrManager?.status === 'Approved'
                          ? { color: '#34C759' }
                          : application.HrManager?.status === 'Rejected'
                          ? { color: '#FF3B30' }
                          : {},
                      ]}
                    >
                      {application.HrManager?.status || 'Pending'}
                    </Text>
                    {application.HrManager?.reason && (
                      <TouchableOpacity
                        style={{ marginLeft: 8 }}
                        onPress={() => setShowReasonModal(true)}
                      >
                        <Text style={styles.rejectionReason}>
                          {getShortReason(application.HrManager.reason)}
                        </Text>
                      </TouchableOpacity>
                    )}
                  </View>
                  
                  {/* Show waiting message for HR Manager if supervisor hasn't approved */}
                  {isHrManager && (!application.Supervisor || !application.Supervisor.status || application.Supervisor.status === 'Pending') && (
                    <Text style={[styles.statusValue, { color: '#F57C00', fontStyle: 'italic' }]}>
                      Waiting for supervisor to approve
                    </Text>
                  )}
                  
                  {/* Show edit icon for HR Manager when there's an existing status and supervisor has approved */}
                  {isHrManager && application.Supervisor && application.Supervisor.status === 'Approved' && 
                   application.HrManager?.status && !(showEditHr || application.HrManager.status === 'Pending' || !application.HrManager.status) && (
                    <TouchableOpacity
                      style={styles.editStatusBtn}
                      onPress={() => setShowEditHr(true)}
                    >
                      <Icon name="edit" size={width * 0.06} color="#015185" />
                    </TouchableOpacity>
                  )}
                  
                  {/* Show Approve/Reject buttons for HR Manager only if supervisor has approved */}
                  {isHrManager && application.Supervisor && application.Supervisor.status === 'Approved' && 
                   (!application.HrManager || application.HrManager?.status === 'Pending' || !application.HrManager?.status || showEditHr) && (
                    <View style={{ flexDirection: 'row', marginTop: 10, alignItems: 'center' }}>
                      <TouchableOpacity
                        style={[styles.approvalBtn, { backgroundColor: '#34C759', marginRight: 10 }]}
                        onPress={() => { setShowEditHr(false); handleHrApprovalAction('approve'); }}
                      >
                        <Text style={styles.approvalBtnText}>Approve</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={[styles.approvalBtn, { backgroundColor: '#FF3B30' }]}
                        onPress={() => { setShowEditHr(false); handleHrApprovalAction('reject'); }}
                      >
                        <Text style={styles.approvalBtnText}>Reject</Text>
                      </TouchableOpacity>
                    </View>
                  )}
                </View>
              </View>
            )}
            {applicationType === 'salary_advance_application' && (application.FinanceManager || isFinanceManager) && (
              <View style={styles.statusItem}>
                <View style={styles.statusIconContainer}>
                  <MaterialIcons
                    name={application.FinanceManager?.status === 'Approved' ? 'check-circle' : 
                          application.FinanceManager?.status === 'Rejected' ? 'cancel' : 'radio-button-unchecked'}
                    size={width * 0.05}
                    color={application.FinanceManager?.status === 'Approved' ? '#34C759' : 
                           application.FinanceManager?.status === 'Rejected' ? '#FF3B30' : '#FF3B30'}
                  />
                  <View style={[styles.statusLine, styles.statusLineLast]} />
                </View>
                <View style={{ flex: 1 }}>
                  <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', marginBottom: 4 }}>
                    <Text style={styles.statusLabel}>Finance Manager</Text>
                    <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                      <Text style={styles.statusName}>
                        {application.FinanceManager?.name && application.FinanceManager.name !== 'Pending:FinanceManager'
                          ? application.FinanceManager.name
                          : 'N/A'}
                      </Text>
                      {isFinanceManager && (
                        <MaterialIcons name="star" size={width * 0.045} color="#34C759" style={{ marginLeft: 4 }} />
                      )}
                    </View>
                  </View>
                  <View style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 8 }}>
                    <Text
                      style={[
                        styles.statusValue,
                        application.FinanceManager?.status === 'Approved'
                          ? { color: '#34C759' }
                          : application.FinanceManager?.status === 'Rejected'
                          ? { color: '#FF3B30' }
                          : {},
                      ]}
                    >
                      {application.FinanceManager?.status || 'Pending'}
                    </Text>
                    {application.FinanceManager?.reason && (
                      <TouchableOpacity
                        style={{ marginLeft: 8 }}
                        onPress={() => setShowReasonModal(true)}
                      >
                        <Text style={styles.rejectionReason}>
                          {getShortReason(application.FinanceManager.reason)}
                        </Text>
                      </TouchableOpacity>
                    )}
                  </View>
                  
                  {/* Show waiting message for Finance Manager if HR Manager hasn't approved */}
                  {isFinanceManager && (!application.HrManager || !application.HrManager.status || application.HrManager.status === 'Pending') && (
                    <Text style={[styles.statusValue, { color: '#F57C00', fontStyle: 'italic' }]}>
                      Waiting for HR Manager to approve
                    </Text>
                  )}
                  
                  {/* Show edit icon for Finance Manager when there's an existing status and HR Manager has approved */}
                  {isFinanceManager && application.HrManager && application.HrManager.status === 'Approved' && 
                   application.FinanceManager?.status && !(showEditHr || application.FinanceManager.status === 'Pending' || !application.FinanceManager.status) && (
                    <TouchableOpacity
                      style={styles.editStatusBtn}
                      onPress={() => setShowEditHr(true)}
                    >
                      <Icon name="edit" size={width * 0.06} color="#015185" />
                    </TouchableOpacity>
                  )}
                  
                  {/* Show Approve/Reject buttons for Finance Manager only if HR Manager has approved */}
                  {isFinanceManager && application.HrManager && application.HrManager.status === 'Approved' && 
                   (!application.FinanceManager || application.FinanceManager?.status === 'Pending' || !application.FinanceManager?.status || showEditHr) && (
                    <View style={{ flexDirection: 'row', marginTop: 10, alignItems: 'center' }}>
                      <TouchableOpacity
                        style={[styles.approvalBtn, { backgroundColor: '#34C759', marginRight: 10 }]}
                        onPress={() => { 
                          setShowEditHr(false); 
                          console.log('🔄 Navigating to approval screen for Finance Manager action: approve');
                          navigation.navigate('ApplicationApprovReject', {
                            application: application || initialApplication,
                            applicationType,
                            action: 'approve',
                            approverRole: 'FinanceManager',
                            applicationId: (application || initialApplication).applicationId,
                            existingReason: application?.FinanceManager?.reason || application?.FinanceManager?.comments,
                            isEdit: !!application?.FinanceManager?.status
                          });
                        }}
                      >
                        <Text style={styles.approvalBtnText}>Approve</Text>
                      </TouchableOpacity>
                      <TouchableOpacity
                        style={[styles.approvalBtn, { backgroundColor: '#FF3B30' }]}
                        onPress={() => { 
                          setShowEditHr(false); 
                          console.log('🔄 Navigating to approval screen for Finance Manager action: reject');
                          navigation.navigate('ApplicationApprovReject', {
                            application: application || initialApplication,
                            applicationType,
                            action: 'reject',
                            approverRole: 'FinanceManager',
                            applicationId: (application || initialApplication).applicationId,
                            existingReason: application?.FinanceManager?.reason || application?.FinanceManager?.comments,
                            isEdit: !!application?.FinanceManager?.status
                          });
                        }}
                      >
                        <Text style={styles.approvalBtnText}>Reject</Text>
                      </TouchableOpacity>
                    </View>
                  )}
                </View>
              </View>
            )}
          </View>
          )}
        </View>
        {/* Modal for full reason */}
        <Modal isVisible={showReasonModal} onBackdropPress={() => setShowReasonModal(false)}>
          <View style={{ backgroundColor: '#fff', borderRadius: 10, padding: 20 }}>
            <Text style={{ fontWeight: 'bold', fontSize: width * 0.045, marginBottom: 10 }}>
              {application.Supervisor?.status === 'Rejected' ? 'Supervisor Rejection Reason' : 
               application.Supervisor?.status === 'Approved' ? 'Supervisor Approval Reason' :
               application.HrManager?.status === 'Rejected' ? 'HR Manager Rejection Reason' :
               application.HrManager?.status === 'Approved' ? 'HR Manager Approval Reason' :
               application.FinanceManager?.status === 'Rejected' ? 'Finance Manager Rejection Reason' :
               application.FinanceManager?.status === 'Approved' ? 'Finance Manager Approval Reason' : 'Reason'}
            </Text>
            <Text style={{ fontSize: width * 0.04, color: '#333' }}>
              {application.Supervisor?.reason || application.Supervisor?.comments || 
               application.HrManager?.reason || application.HrManager?.comments ||
               application.FinanceManager?.reason || application.FinanceManager?.comments || 'No reason provided.'}
            </Text>
            <TouchableOpacity
              style={{ alignSelf: 'flex-end', marginTop: 20 }}
              onPress={() => setShowReasonModal(false)}
            >
              <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: width * 0.04 }}>Close</Text>
            </TouchableOpacity>
          </View>
        </Modal>
      </>
    );
  };

  return (
    <View style={styles.flex1}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.headerGradient}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <SafeAreaView style={styles.safeAreaHeader}>
          <View style={styles.headerContentFixed}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitleFixed} numberOfLines={1} ellipsizeMode="tail">
              {applicationType
                .split('_')
                .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
                .join(' ')}
            </Text>
            <TouchableOpacity
              style={styles.refreshButton}
              onPress={() => {
                console.log('☁️ Cloud refresh triggered - fetching from MongoDB');
                fetchLatestApplication();
              }}
              accessibilityLabel="Refresh from MongoDB cloud"
              accessibilityRole="button"
            >
              <MaterialIcons name="cloud-download" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </SafeAreaView>
      </LinearGradient>
      <ScrollView contentContainerStyle={styles.detailContent}>
        {renderApplicationDetails()}
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#F4F6F8',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  safeAreaHeader: {
    backgroundColor: 'transparent',
  },
  headerContentFixed: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop:
      Platform.OS === 'android'
        ? Math.max((StatusBar.currentHeight || width * 0.1) - 10, 0)
        : Math.max((StatusBar.currentHeight || width * 0.1) - 15, 0),
    paddingBottom:
      Platform.OS === 'android'
        ? Math.max(width * 0.03 - 10, 0)
        : Math.max(width * 0.03 - 15, 0),
    minHeight:
      Platform.OS === 'android'
        ? Math.max(width * 0.01 - 30, 0)
        : Math.max(width * 0.01 - 15, 0),
  },
  headerTitleFixed: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: '700',
    color: '#fff',
    textAlign: 'center',
  },
  backButton: {
    padding: width * 0.02,
  },
  refreshButton: {
    padding: width * 0.02,
  },
  detailContent: {
    padding: width * 0.04,
    paddingBottom: height * 0.05,
  },
  card: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    padding: width * 0.04,
    marginBottom: width * 0.04,
    shadowColor: 'transparent',
    elevation: 0,
  },
  cardTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#025383', // blue heading
    marginBottom: height * 0.02,
  },
  fieldContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: height * 0.015,
  },
  detailLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#4A4A4A',
    flex: 1,
  },
  detailValue: {
    fontSize: width * 0.04,
    fontWeight: '400',
    color: '#1A1A1A',
    flex: 1,
    textAlign: 'right',
  },
  table: {
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    padding: width * 0.03,
    marginTop: height * 0.01,
  },
  tableRow: {
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    paddingVertical: height * 0.01,
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  tableCell: {
    fontSize: width * 0.038,
    color: '#1A1A1A',
    marginBottom: height * 0.005,
    flex: 1,
    textAlign: 'right',
  },
  tableHeaderRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#F4F6F8',
    paddingVertical: height * 0.008,
  },
  tableHeaderCell: {
    flex: 1,
    fontSize: width * 0.038,
    fontWeight: '700',
    color: '#025383',
    textAlign: 'right',
  },
  statusTimeline: {
    marginTop: height * 0.015,
    paddingLeft: width * 0.04,
  },
  statusItem: {
    flexDirection: 'row',
    alignItems: 'flex-start', // Changed to flex-start for better vertical alignment
    marginBottom: height * 0.02,
  },
  statusIconContainer: {
    alignItems: 'center',
    marginRight: width * 0.03,
  },
  statusLine: {
    position: 'absolute',
    top: width * 0.05,
    width: 2,
    height: height * 0.04,
    backgroundColor: '#E0E0E0',
  },
  statusLineApproved: {
    backgroundColor: '#34C759',
  },
  statusLineLast: {
    height: 0,
  },
  statusLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#4A4A4A',
  },
  statusValue: {
    fontSize: width * 0.038,
    fontWeight: '400',
    color: '#1A1A1A',
  },
  statusName: {
    fontSize: width * 0.035,
    fontWeight: '400',
    color: '#6B7280',
  },
  approvalBtn: {
    paddingVertical: 8,
    paddingHorizontal: 18,
    borderRadius: 6,
    alignItems: 'center',
    justifyContent: 'center',
  },
  approvalBtnText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.04,
  },
  rejectionReason: {
    color: '#FF3B30',
    fontSize: width * 0.038,
    marginTop: 4,
    fontWeight: '600',
  },
  editStatusBtn: {
    marginLeft: 10,
    padding: 6,
    borderRadius: 20,
    backgroundColor: '#e6f0fa',
  },
  flex1: {
    flex: 1,
    backgroundColor: '#F4F6F8',
  },
  headerGradient: {
    width: '100%',
    paddingBottom:
      Platform.OS === 'android'
        ? Math.max((StatusBar.currentHeight || 24) - 30, 0)
        : Math.max((StatusBar.currentHeight || 24) - 15, 0),
  },
});

export default ApplicationDetail;