import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  SafeAreaView,
  StatusBar,
  ScrollView,
  Platform,
  Modal,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../../../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');

// MongoDB API URL
const MONGODB_HRFORM_API_URL = 'http://168.231.113.219:5013';

const AdminCheckAllApli = () => {
  const navigation = useNavigation();
  const [unapprovedCounts, setUnapprovedCounts] = useState({
    leave_application: 0,
    application_for_appeal: 0,
    funeral_grant_form: 0,
    grievance_record: 0,
    hospital_attendance_form: 0,
    salary_advance_application: 0,
  });
  const [totalCounts, setTotalCounts] = useState({
    leave_application: 0,
    application_for_appeal: 0,
    funeral_grant_form: 0,
    grievance_record: 0,
    hospital_attendance_form: 0,
    salary_advance_application: 0,
  });
  const [isOnline, setIsOnline] = useState(true);
  const [isLoading, setIsLoading] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [userDetails, setUserDetails] = useState(null);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [allowedCountries, setAllowedCountries] = useState([]);
  const [allowedProjects, setAllowedProjects] = useState([]);
  const [hrPermissions, setHrPermissions] = useState([]);
  const [permissionsLoaded, setPermissionsLoaded] = useState(false);

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const buttons = [
    {
      title: 'Leave Applications',
      icon: 'event-note', // Represents leave planning
      screen: 'AdminLeaveApli',
      type: 'leave_application',
    },
    {
      title: 'Appeal Applications',
      icon: 'gavel', // Represents legal/appeal process
      screen: 'AdminAllAppealApli',
      type: 'application_for_appeal',
    },
    {
      title: 'Funeral Applications',
      icon: 'sentiment-dissatisfied', // Represents condolence/funeral
      screen: 'AdminAllFuneralApl',
      type: 'funeral_grant_form',
    },
    {
      title: 'Grievance Applications',
      icon: 'report-problem', // Represents complaints/grievances
      screen: 'AdminGrievanceApl',
      type: 'grievance_record',
    },
    {
      title: 'Hospital Attendance',
      icon: 'local-hospital', // Represents medical/hospital
      screen: 'AdminHospitalAtteAppl',
      type: 'hospital_attendance_form',
    },
    {
      title: 'Salary Advance',
      icon: 'account-balance-wallet', // Represents financial transactions
      screen: 'AdminSalaryAdv',
      type: 'salary_advance_application',
    },
    {
      title: 'Apply New',
      icon: 'add-circle-outline', // Represents adding new application
      screen: 'AdminHrAppliApply',
      type: null, // No count for this button
    },
  ];

  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      // Get userUid
      let userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        const userData = await AsyncStorage.getItem('userData');
        if (userData) {
          const user = JSON.parse(userData);
          userUid = user._id || user.id;
          if (userUid) {
            await AsyncStorage.setItem('userUid', userUid);
          }
        }
      }

      if (!userUid) {
        setModalError('User ID not found. Please log in again.');
        setModalLoading(false);
        return;
      }

      console.log('🔍 Fetching fresh user details for modal:', userUid);

      // Try to fetch fresh data from MongoDB
      let user;
      try {
        const response = await MongoAPI.getUserById(userUid);
        if (response && response.success) {
          user = response.user || response.data;
          console.log('✅ Using fresh data from MongoDB for modal');
          // Update AsyncStorage
          await AsyncStorage.setItem('userData', JSON.stringify(user));
        } else {
          throw new Error('MongoDB fetch failed');
        }
      } catch (mongoError) {
        console.log('⚠️ MongoDB fetch failed, using cached data:', mongoError.message);
        // Fallback to cached data
        const cachedData = await AsyncStorage.getItem('userData');
        if (!cachedData) {
          setModalError('User data not found. Please log in again.');
          setModalLoading(false);
          return;
        }
        user = JSON.parse(cachedData);
      }

      console.log('👤 User data for modal:', user);

      // Get country and project (handle array or string)
      const countries = Array.isArray(user.countries) ? user.countries : 
                       (user.country ? [user.country] : []);
      
      // Projects might be stored in different fields
      let projects = [];
      if (Array.isArray(user.projects)) {
        projects = user.projects;
      } else if (user.project) {
        projects = Array.isArray(user.project) ? user.project : [user.project];
      } else if (user.selectedProjects) {
        projects = Array.isArray(user.selectedProjects) ? user.selectedProjects : [user.selectedProjects];
      }
      
      // Get role - check multiple possible field names
      let role = 'N/A';
      if (user.role) {
        role = user.role;
      } else if (Array.isArray(user.roles) && user.roles.length > 0) {
        role = user.roles[0];
      } else if (user.userRole) {
        role = user.userRole;
      }

      // Normalize modules to array
      const modules = Array.isArray(user.modules) ? user.modules : 
                      (user.modules ? Object.values(user.modules) : []);

      console.log('📦 All modules from user data:', modules);

      // Find hrForm module
      const hrModule = modules.find(m => 
        m.module && (m.module === 'hrForm' || m.module.toLowerCase() === 'hrform')
      );

      console.log('📋 HR Form module found:', hrModule);

      let permissions = [];
      let viewMode = 'No Access';
      
      if (hrModule && hrModule.permissions && Array.isArray(hrModule.permissions)) {
        permissions = hrModule.permissions;
        
        if (permissions.includes('onlyMineView')) {
          viewMode = 'Only My Applications';
        } else if (permissions.includes('view') || permissions.includes('all')) {
          viewMode = 'Country & Project Access';
        }
      }

      console.log('✅ HR Form Permissions:', permissions);
      console.log('🔍 View Mode:', viewMode);

      const userDetailsData = {
        name: user.name || user.fullName || 'N/A',
        email: user.email || 'N/A',
        countries,
        projects,
        role,
        hrFormPermissions: permissions,
        viewMode,
      };
      
      console.log('📋 Setting user details:', userDetailsData);
      setUserDetails(userDetailsData);
    } catch (err) {
      console.error('❌ Error fetching user details:', err);
      setModalError('Failed to load user details.');
      setUserDetails(null);
    }
    setModalLoading(false);
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected || false);
    });

    return () => unsubscribe();
  }, []);

  // Auto-refresh user permissions from MongoDB
  const refreshUserPermissionsFromMongoDB = async () => {
    try {
      console.log('🔄 Auto-refreshing permissions from MongoDB...');
      
      // Get userUid from AsyncStorage
      let userUid = await AsyncStorage.getItem('userUid');
      
      if (!userUid) {
        const userData = await AsyncStorage.getItem('userData');
        if (userData) {
          const user = JSON.parse(userData);
          userUid = user._id || user.id;
          if (userUid) {
            await AsyncStorage.setItem('userUid', userUid);
          }
        }
      }

      if (!userUid) {
        console.log('❌ No userUid found for refresh');
        setPermissionsLoaded(true);
        return;
      }

      console.log('🔍 Fetching fresh data for userUid:', userUid);

      // Fetch fresh user data from MongoDB
      const response = await MongoAPI.getUserById(userUid);
      
      if (!response || !response.success) {
        console.log('⚠️ Failed to fetch from MongoDB, using cached data');
        // Fallback to cached data
        await loadCachedPermissions();
        return;
      }

      console.log('✅ Fresh user data received from MongoDB');
      const freshUser = response.user || response.data;
      
      // Update AsyncStorage with fresh data
      await AsyncStorage.setItem('userData', JSON.stringify(freshUser));
      console.log('💾 Updated AsyncStorage with fresh user data');

      // Parse and set permissions
      const countries = Array.isArray(freshUser.countries) ? freshUser.countries : 
                       (freshUser.countries ? [freshUser.countries] : []);
      setAllowedCountries(countries);
      
      // Handle projects from different fields
      let projects = [];
      if (Array.isArray(freshUser.projects)) {
        projects = freshUser.projects;
      } else if (freshUser.project) {
        projects = Array.isArray(freshUser.project) ? freshUser.project : [freshUser.project];
      } else if (freshUser.selectedProjects) {
        projects = Array.isArray(freshUser.selectedProjects) ? freshUser.selectedProjects : [freshUser.selectedProjects];
      }
      setAllowedProjects(projects);
      
      // Normalize modules to array
      const modules = Array.isArray(freshUser.modules) ? freshUser.modules : 
                      (freshUser.modules ? Object.values(freshUser.modules) : []);

      console.log('📦 Fresh modules from MongoDB:', modules);

      // Get HR permissions
      const hrModule = modules.find((mod) => 
        mod.module && (mod.module === 'hrForm' || mod.module.toLowerCase() === 'hrform')
      );
      
      console.log('📋 Fresh HR Form module:', hrModule);
      
      const permissions = hrModule?.permissions || [];
      setHrPermissions(permissions);
      
      console.log('✅ Fresh HR Permissions:', permissions);
      console.log('🔒 onlyMineView:', permissions.includes('onlyMineView'));
      console.log('👁️ view:', permissions.includes('view'));
      console.log('🌟 all:', permissions.includes('all'));
      
      setPermissionsLoaded(true);
      
    } catch (error) {
      console.error('❌ Error refreshing from MongoDB:', error);
      // Fallback to cached data
      await loadCachedPermissions();
    }
  };

  // Load cached permissions from AsyncStorage (fallback)
  const loadCachedPermissions = async () => {
    try {
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) {
        console.log('❌ No cached user data found');
        setPermissionsLoaded(true);
        return;
      }

      const user = JSON.parse(userData);
      console.log('📦 Using cached user data');
      
      setAllowedCountries(Array.isArray(user.countries) ? user.countries : (user.countries ? [user.countries] : []));
      
      let projects = [];
      if (Array.isArray(user.projects)) {
        projects = user.projects;
      } else if (user.project) {
        projects = Array.isArray(user.project) ? user.project : [user.project];
      } else if (user.selectedProjects) {
        projects = Array.isArray(user.selectedProjects) ? user.selectedProjects : [user.selectedProjects];
      }
      setAllowedProjects(projects);
      
      const modules = Array.isArray(user.modules) ? user.modules : 
                      (user.modules ? Object.values(user.modules) : []);

      const hrModule = modules.find((mod) => 
        mod.module && (mod.module === 'hrForm' || mod.module.toLowerCase() === 'hrform')
      );
      
      const permissions = hrModule?.permissions || [];
      setHrPermissions(permissions);
      
      console.log('✅ Cached HR Permissions loaded:', permissions);
      setPermissionsLoaded(true);
      
    } catch (error) {
      console.error('❌ Error loading cached permissions:', error);
      setPermissionsLoaded(true);
    }
  };

  // Auto-refresh on screen focus
  useFocusEffect(
    React.useCallback(() => {
      console.log('📱 Screen focused - refreshing permissions...');
      refreshUserPermissionsFromMongoDB();
    }, [])
  );

  useEffect(() => {
    const fetchUnapprovedCounts = async () => {
      // Wait for permissions to be loaded
      if (!permissionsLoaded) {
        return; // Don't fetch until permissions are loaded
      }
      
      setIsLoading(true);
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
          setIsLoading(false);
          return;
        }

        console.log('📊 Fetching application counts from MongoDB...');
        console.log('🔍 Filters:', {
          countries: allowedCountries,
          projects: allowedProjects,
          permissions: hrPermissions
        });

        if (!isOnline) {
          const cachedApplications = await AsyncStorage.getItem('offlineApplications');
          if (cachedApplications) {
            const parsedApplications = JSON.parse(cachedApplications);
            const counts = {};
            const totalCountsObj = {};
            buttons.forEach((button) => {
              if (button.type) {
                let filteredApps = parsedApplications.filter(
                  (app) => app.applicationType === button.type
                );

                // Apply onlyMineView filter
                if (hrPermissions.includes('onlyMineView')) {
                  console.log(`Filtering ${button.type}: onlyMineView mode`);
                  filteredApps = filteredApps.filter(app => app.userUid === userUid);
                } else {
                  console.log(`Filtering ${button.type}: country/project mode`);
                  // Apply country/project filter
                  filteredApps = filteredApps.filter(app => 
                    (allowedCountries.length === 0 || allowedCountries.includes(app.country)) &&
                    (allowedProjects.length === 0 || allowedProjects.includes(app.project))
                  );
                }

                counts[button.type] = filteredApps.filter(app => app.status !== 'Approved').length;
                totalCountsObj[button.type] = filteredApps.length;
              }
            });
            setUnapprovedCounts((prev) => ({ ...prev, ...counts }));
            setTotalCounts((prev) => ({ ...prev, ...totalCountsObj }));
            showToast('info', 'Offline Mode', 'Loaded cached unapproved application counts.');
          } else {
            showToast('info', 'Offline Mode', 'No cached applications available.');
          }
          setIsLoading(false);
          return;
        }

        // Fetch counts from MongoDB
        const queryParams = new URLSearchParams({
          countries: allowedCountries.join(','),
          projects: allowedProjects.join(','),
          onlyMineView: hrPermissions.includes('onlyMineView').toString()
        });

        const response = await fetch(`${MONGODB_HRFORM_API_URL}/hr-applications/counts/${userUid}?${queryParams}`);
        
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        console.log('✅ MongoDB counts response:', result);

        if (result.success && result.data) {
          setUnapprovedCounts(result.data.unapprovedCounts);
          setTotalCounts(result.data.totalCounts);
          
          // Cache for offline use
          await AsyncStorage.setItem('hrApplicationCounts', JSON.stringify(result.data));
          
          console.log('📊 Counts updated:', {
            unapproved: result.data.unapprovedCounts,
            total: result.data.totalCounts
          });
        }
        
        setIsLoading(false);
      } catch (error) {
        console.error('Error fetching unapproved counts:', error);
        showToast('error', 'Error', 'Failed to load unapproved application counts.');
        const cachedApplications = await AsyncStorage.getItem('offlineApplications');
        if (cachedApplications) {
          const parsedApplications = JSON.parse(cachedApplications);
          const counts = {};
          const totalCountsObj = {};
          buttons.forEach((button) => {
            if (button.type) {
              let filteredApps = parsedApplications.filter(
                (app) => app.applicationType === button.type
              );

              // Apply onlyMineView filter
              if (hrPermissions.includes('onlyMineView')) {
                filteredApps = filteredApps.filter(app => app.userUid === userUid);
              } else {
                // Apply country/project filter
                filteredApps = filteredApps.filter(app => 
                  (allowedCountries.length === 0 || allowedCountries.includes(app.country)) &&
                  (allowedProjects.length === 0 || allowedProjects.includes(app.project))
                );
              }

              counts[button.type] = filteredApps.filter(app => app.status !== 'Approved').length;
              totalCountsObj[button.type] = filteredApps.length;
            }
          });
          setUnapprovedCounts((prev) => ({ ...prev, ...counts }));
          setTotalCounts((prev) => ({ ...prev, ...totalCountsObj }));
          showToast('info', 'Offline Mode', 'Loaded cached unapproved counts due to error.');
        }
        setIsLoading(false);
      }
    };

    fetchUnapprovedCounts();
  }, [isOnline, navigation, allowedCountries, allowedProjects, hrPermissions, permissionsLoaded]);

  return (
    <View style={styles.flex1}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.headerGradient}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <SafeAreaView style={styles.safeAreaHeader}>
          <View style={styles.headerContentFixed}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitleFixed}>HR Admin Dashboard</Text>
            <TouchableOpacity
              style={styles.lockButton}
              onPress={handleLockPress}
              accessibilityLabel="Show user details"
              accessibilityRole="button"
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </SafeAreaView>
      </LinearGradient>
      {/* Permission Modal */}
      <Modal
        visible={modalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <MaterialIcons name="lock" size={width * 0.08} color="#015185" />
              <Text style={styles.modalTitle}>Your Permissions</Text>
              <TouchableOpacity onPress={() => setModalVisible(false)}>
                <MaterialIcons name="close" size={width * 0.06} color="#666" />
              </TouchableOpacity>
            </View>
            
            {modalLoading ? (
              <View style={styles.loadingContainer}>
                <ActivityIndicator size="large" color="#015185" />
                <Text style={styles.loadingText}>Loading permissions...</Text>
              </View>
            ) : modalError ? (
              <View style={styles.errorContainer}>
                <MaterialIcons name="error-outline" size={width * 0.12} color="#FF3B30" />
                <Text style={styles.modalError}>{modalError}</Text>
              </View>
            ) : userDetails ? (
              <ScrollView style={styles.modalDetailsBox} showsVerticalScrollIndicator={false}>
                {/* User Info */}
                <View style={styles.permissionSection}>
                  <Text style={styles.sectionTitle}>
                    <MaterialIcons name="person" size={width * 0.05} color="#015185" /> User Information
                  </Text>
                  <View style={styles.infoRow}>
                    <Text style={styles.infoLabel}>Name:</Text>
                    <Text style={styles.infoValue}>{userDetails.name || 'N/A'}</Text>
                  </View>
                  <View style={styles.infoRow}>
                    <Text style={styles.infoLabel}>Email:</Text>
                    <Text style={styles.infoValue}>{userDetails.email || 'N/A'}</Text>
                  </View>
                  <View style={styles.infoRow}>
                    <Text style={styles.infoLabel}>Role:</Text>
                    <Text style={styles.infoValue}>{userDetails.role || 'N/A'}</Text>
                  </View>
                </View>

                {/* Countries */}
                <View style={styles.permissionSection}>
                  <Text style={styles.sectionTitle}>
                    <MaterialIcons name="public" size={width * 0.05} color="#015185" /> Countries
                  </Text>
                  {userDetails.countries && userDetails.countries.length > 0 ? (
                    <View style={styles.chipsContainer}>
                      {userDetails.countries.map((country, index) => (
                        <View key={index} style={styles.chip}>
                          <Text style={styles.chipText}>{country}</Text>
                        </View>
                      ))}
                    </View>
                  ) : (
                    <Text style={styles.noDataText}>No countries assigned</Text>
                  )}
                </View>

                {/* Projects */}
                <View style={styles.permissionSection}>
                  <Text style={styles.sectionTitle}>
                    <MaterialIcons name="business" size={width * 0.05} color="#015185" /> Projects
                  </Text>
                  {userDetails.projects && userDetails.projects.length > 0 ? (
                    <View style={styles.chipsContainer}>
                      {userDetails.projects.map((project, index) => (
                        <View key={index} style={styles.chip}>
                          <Text style={styles.chipText}>{project}</Text>
                        </View>
                      ))}
                    </View>
                  ) : (
                    <Text style={styles.noDataText}>No projects assigned</Text>
                  )}
                </View>

                {/* HR Form Module Permissions */}
                <View style={styles.permissionSection}>
                  <Text style={styles.sectionTitle}>
                    <MaterialIcons name="assignment" size={width * 0.05} color="#015185" /> HR Form Module
                  </Text>
                  {userDetails.hrFormPermissions && userDetails.hrFormPermissions.length > 0 ? (
                    <View style={styles.chipsContainer}>
                      {userDetails.hrFormPermissions.map((permission, index) => (
                        <View key={index} style={[styles.chip, styles.permissionChip]}>
                          <MaterialIcons name="check" size={width * 0.04} color="#00B894" />
                          <Text style={styles.permissionChipText}>{permission}</Text>
                        </View>
                      ))}
                    </View>
                  ) : (
                    <Text style={styles.noDataText}>No permissions assigned</Text>
                  )}
                </View>

                {/* View Mode Info */}
                <View style={styles.infoBox}>
                  <MaterialIcons name="info" size={width * 0.05} color="#015185" />
                  <View style={styles.infoBoxContent}>
                    <Text style={styles.infoBoxTitle}>Current View Mode</Text>
                    <Text style={styles.infoBoxText}>{userDetails.viewMode}</Text>
                  </View>
                </View>
              </ScrollView>
            ) : null}
            
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setModalVisible(false)}
              accessibilityLabel="Close permissions"
              accessibilityRole="button"
            >
              <LinearGradient
                colors={['#015185', '#3481BC']}
                style={styles.modalCloseButtonGradient}
                start={{ x: 0, y: 0 }}
                end={{ x: 1, y: 0 }}
              >
                <Text style={styles.modalCloseText}>Close</Text>
              </LinearGradient>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      <ScrollView contentContainerStyle={styles.scrollContent} bounces={false} showsVerticalScrollIndicator={false}>
        {/* Filter Mode Indicator */}
        <View style={styles.filterModeContainer}>
          <MaterialIcons 
            name={hrPermissions.includes('onlyMineView') ? 'person' : 'filter-list'} 
            size={16} 
            color="#035484" 
          />
          <Text style={styles.filterModeText}>
            {hrPermissions.includes('onlyMineView')
              ? 'Showing only your HR applications'
              : allowedCountries.length > 0 || allowedProjects.length > 0
              ? `Access: ${allowedCountries.join(', ') || 'All'} | ${allowedProjects.join(', ') || 'All'}`
              : 'Loading permissions...'}
          </Text>
        </View>
        
        <View style={styles.buttonContainer}>
          {buttons.map((button, index) => (
            <View key={index} style={styles.buttonWrapper}>
              <TouchableOpacity
                style={styles.cardButton}
                onPress={() => navigation.navigate(button.screen)}
                accessibilityLabel={`Navigate to ${button.title}`}
                accessibilityRole="button"
                disabled={isLoading}
                activeOpacity={0.85}
              >
                <LinearGradient
                  colors={['#015185', '#3481BC']}
                  style={styles.gradientButton}
                  start={{ x: 0.5, y: 0 }}
                  end={{ x: 0.5, y: 1 }}
                >
                  <View style={styles.buttonContent}>
                    <MaterialIcons
                      name={button.icon}
                      size={width * 0.12}
                      color="#fff"
                      style={styles.buttonIcon}
                    />
                    <Text style={styles.buttonText}>{button.title}</Text>
                    {button.type && (
                      <View>
                        <Text style={styles.totalCount}>
                          {totalCounts[button.type]} Total
                        </Text>
                        {unapprovedCounts[button.type] > 0 && (
                          <Text style={styles.unapprovedCount}>
                            {unapprovedCounts[button.type]} Unapproved
                          </Text>
                        )}
                      </View>
                    )}
                  </View>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          ))}
        </View>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  flex1: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  headerGradient: {
    width: '100%',
    paddingBottom: Platform.OS === 'android' ? StatusBar.currentHeight || 24 : 0,
    // Ensures gradient covers status bar on Android
  },
  safeAreaHeader: {
    backgroundColor: 'transparent',
  },
  headerContentFixed: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'android' ? ((StatusBar.currentHeight || width * 0.4) - 30) : 0,
    paddingBottom: width * 0.03,
    minHeight: Platform.OS === 'android' ? width * 0.01 - 10 : width * 0.01,
  },
  headerTitleFixed: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  scrollContent: {
    flexGrow: 1,
    paddingBottom: width * 0.04,
  },
  buttonContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
    padding: width * 0.04,
  },
  buttonWrapper: {
    width: '48%',
    marginBottom: width * 0.04,
    alignItems: 'center',
  },
  cardButton: {
    width: '100%',
    borderRadius: 16,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    overflow: 'visible',
  },
  gradientButton: {
    borderRadius: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.12,
    shadowRadius: 6,
    elevation: 3,
    paddingVertical: 0,
  },
  buttonContent: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 28,
  },
  buttonIcon: {
    marginBottom: 10,
  },
  buttonText: {
    color: '#fff',
    fontSize: 18,
    fontWeight: '600',
    textAlign: 'center',
    marginTop: 10,
    letterSpacing: 0.2,
  },
  unapprovedCount: {
    fontSize: 14,
    fontWeight: '600',
    color: '#FF6B6B',
    marginTop: 8,
    textAlign: 'center',
  },
  totalCount: {
    fontSize: 14,
    fontWeight: '600',
    color: '#fff',
    marginTop: 4,
    textAlign: 'center',
  },
  backButton:{
    padding: width * 0.02,
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  lockButton: {
    padding: width * 0.02,
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: width * 0.9,
    maxWidth: 420,
    maxHeight: '80%',
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: width * 0.05,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 8,
    elevation: 5,
  },
  modalHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.04,
    paddingBottom: width * 0.03,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
    textAlign: 'center',
  },
  loadingContainer: {
    padding: width * 0.08,
    alignItems: 'center',
  },
  loadingText: {
    marginTop: width * 0.04,
    fontSize: width * 0.04,
    color: '#666',
  },
  errorContainer: {
    padding: width * 0.08,
    alignItems: 'center',
  },
  modalError: {
    color: '#FF3B30',
    fontSize: width * 0.04,
    marginTop: width * 0.04,
    textAlign: 'center',
  },
  modalDetailsBox: {
    maxHeight: '70%',
  },
  permissionSection: {
    marginBottom: width * 0.05,
  },
  sectionTitle: {
    fontSize: width * 0.042,
    fontWeight: '700',
    color: '#015185',
    marginBottom: width * 0.025,
    flexDirection: 'row',
    alignItems: 'center',
  },
  infoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  infoLabel: {
    fontSize: width * 0.038,
    fontWeight: '600',
    color: '#666',
  },
  infoValue: {
    fontSize: width * 0.038,
    color: '#333',
    fontWeight: '500',
  },
  chipsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  chip: {
    backgroundColor: '#e8f4f8',
    borderRadius: 16,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
    marginRight: width * 0.02,
    marginBottom: width * 0.02,
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: '600',
  },
  permissionChip: {
    backgroundColor: '#e8f8f5',
    borderColor: '#00B894',
    flexDirection: 'row',
    alignItems: 'center',
  },
  permissionChipText: {
    fontSize: width * 0.035,
    color: '#00B894',
    fontWeight: '600',
    marginLeft: width * 0.01,
  },
  noDataText: {
    fontSize: width * 0.035,
    color: '#888',
    fontStyle: 'italic',
  },
  infoBox: {
    flexDirection: 'row',
    backgroundColor: '#f0f8ff',
    padding: width * 0.03,
    borderRadius: 8,
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
    alignItems: 'center',
    marginTop: width * 0.02,
  },
  infoBoxContent: {
    flex: 1,
    marginLeft: width * 0.02,
  },
  infoBoxTitle: {
    fontSize: width * 0.038,
    fontWeight: '600',
    color: '#015185',
    marginBottom: width * 0.01,
  },
  infoBoxText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  modalCloseButton: {
    marginTop: width * 0.04,
    borderRadius: 8,
    overflow: 'hidden',
  },
  modalCloseButtonGradient: {
    paddingVertical: width * 0.035,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.04,
  },
  filterModeContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#E0F2F7',
    borderRadius: 12,
    paddingVertical: 8,
    paddingHorizontal: 12,
    marginHorizontal: width * 0.04,
    marginBottom: width * 0.04,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  filterModeText: {
    fontSize: 14,
    color: '#035484',
    marginLeft: 8,
    fontWeight: '500',
  },
});

export default AdminCheckAllApli;