import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  TextInput,
  ScrollView,
  Modal,
  Image,
  Animated,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

// MongoDB API URL
const MONGODB_FUEL_LOG_API_URL = 'http://168.231.113.219:5011';

const CreateFuelLog = () => {
  const navigation = useNavigation();
  
  // Country and Project states
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [selectedProject, setSelectedProject] = useState(null);
  const [countryItems, setCountryItems] = useState([]);
  const [projectItems, setProjectItems] = useState([]);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    fuelLogPermissions: [],
  });
  
  const [meterReading, setMeterReading] = useState('');
  const [meterUnit, setMeterUnit] = useState('Miles');
  const [itemsMeterUnit] = useState([
    { label: 'Miles', value: 'Miles' },
    { label: 'KM', value: 'KM' },
    { label: 'Hour', value: 'Hour' },
  ]);
  const [fuelQuantity, setFuelQuantity] = useState('');
  const [fuelUnit, setFuelUnit] = useState('Gallon');
  const [itemsFuelUnit] = useState([
    { label: 'Gallon', value: 'Gallon' },
    { label: 'Liter', value: 'Liter' },
    { label: 'KG', value: 'KG' },
  ]);
  const [selectedSource, setSelectedSource] = useState('Fuel Bunker');
  const [selectedBunker, setSelectedBunker] = useState(null);
  const [itemsBunker] = useState([
    { label: 'Fuel Bunker', value: 'Fuel Bunker' },
    { label: 'Fuel Truck', value: 'Fuel Truck' },
    { label: 'Out Station', value: 'Out Station' },
  ]);
  const [valueFuelType, setValueFuelType] = useState(null);
  const [itemsFuelType] = useState([
    { label: 'Diesel', value: 'Diesel' },
    { label: 'Petrol', value: 'Petrol' },
    { label: 'Gas', value: 'Gas' },
    { label: 'CNG', value: 'CNG' },
    { label: 'LPG', value: 'LPG' },
    { label: 'AdBlue', value: 'AdBlue' },
  ]);
  const [equipmentName, setEquipmentName] = useState('');
  const [equipmentNo, setEquipmentNo] = useState('');
  const [fuelCost, setFuelCost] = useState('');
  const [note, setNote] = useState('');
  const [receiverName, setReceiverName] = useState('');
  const [showErrors, setShowErrors] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [requestNumber, setRequestNumber] = useState('FL-001');
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Fetch user data and set up country/project dropdowns
  const fetchUserData = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      console.log('Fetching user data for uid:', userUid);
      
      // Use MongoAPI to get user data
      const response = await MongoAPI.getUserById(userUid);
      console.log('User response:', response);
      
      if (!response.success || !response.data) {
        console.log('User not found in database');
        showToast('error', 'Error', 'User data not found.');
        return;
      }

      const userDataFromDB = response.data;
      console.log('User data received:', {
        name: userDataFromDB.name,
        email: userDataFromDB.email,
        countries: userDataFromDB.countries,
        projects: userDataFromDB.projects,
        modules: userDataFromDB.modules
      });

      // Normalize modules to array and extract Fuel Log permissions
      const modules = Array.isArray(userDataFromDB.modules) ? userDataFromDB.modules : 
                      (userDataFromDB.modules ? Object.values(userDataFromDB.modules) : []);
      
      // Find Fuel Log module
      const fuelLogModule = modules.find(module => 
        module.module && (
          module.module === 'fuelLog' ||
          module.module === 'Fuel Log' ||
          module.module.toLowerCase() === 'fuel log' ||
          module.module.toLowerCase() === 'fuellog' ||
          module.module.toLowerCase() === 'fuel_log'
        )
      );
      
      console.log('Fuel Log module found:', fuelLogModule);
      console.log('All available modules:', modules.map(m => ({ module: m.module, permissions: m.permissions })));
      
      const permissions = fuelLogModule ? (Array.isArray(fuelLogModule.permissions) ? fuelLogModule.permissions : []) : [];
      
      const userDataObj = {
        countries: userDataFromDB.countries || [],
        projects: userDataFromDB.projects || [],
        fuelLogPermissions: permissions,
      };
      
      setUserData(userDataObj);
      console.log('Fuel Log Permissions:', permissions);
      
      // Set up country dropdown items
      const countryDropdownItems = userDataObj.countries.map(country => ({
        label: country,
        value: country,
      }));
      setCountryItems(countryDropdownItems);
      
      // Set up project dropdown items
      const projectDropdownItems = userDataObj.projects.map(project => ({
        label: project,
        value: project,
      }));
      setProjectItems(projectDropdownItems);
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to load user data.');
    }
  };

  // Update project items based on country selection
  const updateProjectItems = (country) => {
    if (country === 'Zambia') {
      const zambiaProjects = userData.projects.filter(project => 
        ['Kobold', 'Kalumbila', 'Kansanshi', 'Kimiteto', 'Mimosa'].includes(project)
      );
      setProjectItems(zambiaProjects.map(project => ({ label: project, value: project })));
    } else if (country === 'DRC') {
      const drcProjects = userData.projects.filter(project => 
        ['Musompo', 'IME'].includes(project)
      );
      setProjectItems(drcProjects.map(project => ({ label: project, value: project })));
    } else {
      // If no country selected or "Both Countries", show all projects
      setProjectItems(userData.projects.map(project => ({ label: project, value: project })));
    }
    
    // Reset project selection when country changes
    setSelectedProject(null);
  };

  // Generate request number preview
  const updateRequestNumber = () => {
    if (selectedCountry && selectedProject) {
      let countryCode = '';
      if (selectedCountry === 'Zambia') {
        countryCode = 'ZMB';
      } else if (selectedCountry === 'DRC') {
        countryCode = 'DRC';
      } else {
        countryCode = selectedCountry.substring(0, 3).toUpperCase();
      }
      const projectCode = selectedProject.substring(0, 3).toUpperCase();
      setRequestNumber(`${countryCode}-${projectCode}-FL-###`);
    }
  };

  // Fetch user data after component mounts
  useEffect(() => {
    fetchUserData();
  }, []);

  // Animation effect for loading modal
  useEffect(() => {
    if (syncModalVisible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [syncModalVisible]);

  // Update project items when country changes
  useEffect(() => {
    if (selectedCountry) {
      updateProjectItems(selectedCountry);
    }
  }, [selectedCountry, userData.projects]);

  // Update request number when both country and project are selected
  useEffect(() => {
    updateRequestNumber();
  }, [selectedCountry, selectedProject]);

  // Callback to handle equipment selection
  const handleEquipmentSelect = (equipmentData) => {
    setEquipmentName(equipmentData.equipmentName || '');
    setEquipmentNo(equipmentData.equipmentNumber || '');
    setMeterReading(equipmentData.meterReading?.toString() || '');
    setMeterUnit(equipmentData.meterUnit || 'Miles');
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) throw new Error('Internet connection not available');
      return true;
    } catch (error) {
      throw new Error('Failed to check network status: ' + error.message);
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const handleRadioChange = (source) => {
    setSelectedSource(source);
  };

  const validateForm = () => {
    return (
      selectedCountry &&
      selectedProject &&
      equipmentName &&
      equipmentNo &&
      meterReading &&
      selectedSource &&
      selectedBunker &&
      fuelQuantity
    );
  };

  const saveFuelLog = async () => {
    setShowErrors(true);
    if (!validateForm()) {
      showToast('error', 'Validation Error', 'Please fill in all required fields.');
      return;
    }

    setSyncModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('No user is logged in');
      if (!(await checkNetworkStatus())) throw new Error('Internet connection not available');

      console.log('📋 Creating fuel log in MongoDB...');

      const fuelLogData = {
        adminUid: userUid,
        country: selectedCountry,
        project: selectedProject,
        equipmentNo,
        equipmentName,
        meterReading: parseFloat(meterReading) || 0,
        meterUnit,
        fuelQuantity: parseFloat(fuelQuantity) || 0,
        fuelUnit,
        selectedSource,
        selectedBunker,
        fuelType: valueFuelType || 'N/A',
        fuelCost: parseFloat(fuelCost) || 0,
        note: note || 'N/A',
        receiverName: receiverName || 'N/A',
        createdAt: new Date().toISOString(),
      };

      console.log('Fuel log data to save:', JSON.stringify(fuelLogData, null, 2));

      // Save to MongoDB
      const response = await fetch(`${MONGODB_FUEL_LOG_API_URL}/fuel-logs`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(fuelLogData),
      });

      if (!response.ok) {
        throw new Error(`Failed to create fuel log: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.success) {
        console.log('✅ Fuel log saved successfully:', result.data);
        showToast('success', 'Success', `Fuel log ${result.data.requestNumber} created successfully.`);
        navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'Fuel_Log_List' },
          ],
        });
      } else {
        throw new Error(result.message || 'Failed to create fuel log');
      }
    } catch (error) {
      console.error('Error saving fuel log to MongoDB:', error);
      let errorMessage = 'Failed to save fuel log.';
      if (error.message.includes('Internet connection not available')) {
        errorMessage = 'No internet connection. Please try again later.';
      } else if (error.message.includes('No user is logged in')) {
        errorMessage = 'User not authenticated. Please log in.';
        navigation.navigate('AdminLogin');
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
    }
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="chevron-left" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Fuel Log Entry</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.saveButton} onPress={saveFuelLog}>
              <Text style={styles.saveButtonText}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView contentContainerStyle={styles.content}>
        <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', marginBottom: 16 }}>
          <Text style={{ fontSize: width * 0.04, color: '#015185', fontWeight: '600' }}>Request#</Text>
          <Text style={{ fontSize: width * 0.04, color: '#015185', fontWeight: '600' }}>
            {requestNumber}
          </Text>
        </View>
        
        <View style={styles.labelContainer}>
          <Text style={styles.label}>Country</Text>
          <Text style={styles.requiredStar}>*</Text>
        </View>
        <Dropdown
          data={countryItems}
          labelField="label"
          valueField="value"
          value={selectedCountry}
          placeholder="Select Country"
          placeholderStyle={styles.dropdownPlaceholder}
          selectedTextStyle={styles.dropdownSelectedText}
          onChange={item => {
            setSelectedCountry(item.value);
            updateProjectItems(item.value);
          }}
          maxHeight={300}
          style={[styles.dropdown, showErrors && !selectedCountry ? styles.inputError : null]}
          containerStyle={styles.dropdownContainer}
          activeColor="#025383"
          itemContainerStyle={styles.dropdownItemContainer}
          renderItem={(item, selected) => (
            <View style={[
              styles.dropdownItemContainer,
              selected && styles.dropdownItemContainerSelected
            ]}>
              <Text style={[
                styles.dropdownItemText,
                selected && { color: '#fff', fontWeight: '600' }
              ]}>
                {item.label}
              </Text>
            </View>
          )}
          renderRightIcon={() => (
            <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
          )}
          showsVerticalScrollIndicator={false}
        />
        {showErrors && !selectedCountry && (
          <Text style={styles.errorText}>Country selection is required</Text>
        )}

        <View style={styles.labelContainer}>
          <Text style={styles.label}>Project</Text>
          <Text style={styles.requiredStar}>*</Text>
        </View>
        <Dropdown
          data={projectItems}
          labelField="label"
          valueField="value"
          value={selectedProject}
          placeholder="Select Project"
          placeholderStyle={styles.dropdownPlaceholder}
          selectedTextStyle={styles.dropdownSelectedText}
          onChange={item => {
            setSelectedProject(item.value);
          }}
          maxHeight={300}
          style={[styles.dropdown, showErrors && !selectedProject ? styles.inputError : null]}
          containerStyle={styles.dropdownContainer}
          activeColor="#025383"
          itemContainerStyle={styles.dropdownItemContainer}
          renderItem={(item, selected) => (
            <View style={[
              styles.dropdownItemContainer,
              selected && styles.dropdownItemContainerSelected
            ]}>
              <Text style={[
                styles.dropdownItemText,
                selected && { color: '#fff', fontWeight: '600' }
              ]}>
                {item.label}
              </Text>
            </View>
          )}
          renderRightIcon={() => (
            <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
          )}
          showsVerticalScrollIndicator={false}
        />
        {showErrors && !selectedProject && (
          <Text style={styles.errorText}>Project selection is required</Text>
        )}

        <View style={styles.labelContainer}>
          <Text style={styles.label}>Equipments</Text>
          <Text style={styles.requiredStar}>*</Text>
        </View>
        <TouchableOpacity
          style={[styles.input, styles.inputWithIcon, showErrors && !equipmentName ? styles.inputError : null]}
          onPress={() =>
            navigation.navigate('Fuel_equipment_select', {
              source: 'CreateFuelLog',
              onSelect: handleEquipmentSelect,
            })
          }
        >
          <Text style={[styles.inputPlaceholder, equipmentName ? styles.inputText : null]}>
            {equipmentName || 'Select Equipment'}
          </Text>
          <Image
            source={require('../../../Images/adminhome/FuelLog/selectequipment.png')}
            style={[styles.inputIcon]}
          />
        </TouchableOpacity>
        {showErrors && !equipmentName && (
          <Text style={styles.errorText}>Equipment selection is required</Text>
        )}

        <View style={styles.labelContainer}>
          <Text style={styles.label}>Equipment Number</Text>
          <Text style={styles.requiredStar}>*</Text>
        </View>
        <TextInput
          style={[styles.input, showErrors && !equipmentNo ? styles.inputError : null]}
          value={equipmentNo}
          onChangeText={setEquipmentNo}
          placeholder="Equipment Number"
          editable={false}
        />
        {showErrors && !equipmentNo && (
          <Text style={styles.errorText}>Equipment Number is required</Text>
        )}

        <View style={styles.labelContainer}>
          <Text style={styles.label}>Equipment Name</Text>
          <Text style={styles.requiredStar}>*</Text>
        </View>
        <TextInput
          style={[styles.input, showErrors && !equipmentName ? styles.inputError : null]}
          value={equipmentName}
          onChangeText={setEquipmentName}
          placeholder="Equipment Name"
          editable={false}
        />
        {showErrors && !equipmentName && (
          <Text style={styles.errorText}>Equipment Name is required</Text>
        )}

        <View style={styles.lablesReading}>
          <View style={styles.labelContainer}>
            <Text style={styles.label}>Meter Reading</Text>
            <Text style={styles.requiredStar}>*</Text>
          </View>
          <Text style={styles.label1}>{meterUnit}</Text>
        </View>
        <View style={styles.row}>
          <TextInput
            style={[styles.input, styles.meterInput, showErrors && !meterReading ? styles.inputError : null]}
            value={meterReading}
            onChangeText={setMeterReading}
            placeholder="Meter Reading"
            keyboardType="numeric"
          />
          <Dropdown
            data={itemsMeterUnit}
            labelField="label"
            valueField="value"
            value={meterUnit}
            placeholder="Miles"
            placeholderStyle={styles.dropdownPlaceholder}
            selectedTextStyle={styles.dropdownSelectedText}
            onChange={item => setMeterUnit(item.value)}
            style={[styles.dropdown, styles.unitDropdown]}
            containerStyle={styles.dropdownContainer}
            activeColor="#025383"
            itemContainerStyle={styles.dropdownItemContainer}
            renderItem={(item, selected) => (
              <View style={[
                styles.dropdownItemContainer,
                selected && styles.dropdownItemContainerSelected
              ]}>
                <Text style={[
                  styles.dropdownItemText,
                  selected && { color: '#fff', fontWeight: '600' }
                ]}>
                  {item.label}
                </Text>
              </View>
            )}
            renderRightIcon={() => (
              <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
            )}
            showsVerticalScrollIndicator={false}
          />
        </View>
        {showErrors && !meterReading && (
          <Text style={styles.errorText}>Meter Reading is required</Text>
        )}

        <View style={styles.radioRow}>
          <View style={styles.labelContainer}>
            <Text style={styles.label}>Filling Source</Text>
            <Text style={styles.requiredStar}>*</Text>
          </View>
          <View style={styles.radioOptions}>
            {['Fuel Bunker', 'Fuel Truck', 'Out Station'].map((source) => (
              <TouchableOpacity
                key={source}
                style={styles.radioContainer}
                onPress={() => handleRadioChange(source)}
              >
                <View style={styles.checkbox}>
                  {selectedSource === source && <View style={styles.checkboxSelected} />}
                </View>
                <Text style={styles.radioLabel}>{source}</Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>
        {showErrors && !selectedSource && (
          <Text style={styles.errorText}>Filling Source is required</Text>
        )}

        <View style={styles.labelContainer}>
          <Text style={styles.label}>Select Bunker</Text>
          <Text style={styles.requiredStar}>*</Text>
        </View>
        <Dropdown
          data={itemsBunker}
          labelField="label"
          valueField="value"
          value={selectedBunker}
          placeholder="Select Bunker"
          placeholderStyle={styles.dropdownPlaceholder}
          selectedTextStyle={styles.dropdownSelectedText}
          onChange={item => setSelectedBunker(item.value)}
          style={[styles.dropdown, showErrors && !selectedBunker ? styles.inputError : null]}
          containerStyle={styles.dropdownContainer}
          activeColor="#025383"
          itemContainerStyle={styles.dropdownItemContainer}
          renderItem={(item, selected) => (
            <View style={[
              styles.dropdownItemContainer,
              selected && styles.dropdownItemContainerSelected
            ]}>
              <Text style={[
                styles.dropdownItemText,
                selected && { color: '#fff', fontWeight: '600' }
              ]}>
                {item.label}
              </Text>
            </View>
          )}
          renderRightIcon={() => (
            <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
          )}
          showsVerticalScrollIndicator={false}
        />
        {showErrors && !selectedBunker && (
          <Text style={styles.errorText}>Bunker selection is required</Text>
        )}

        <Text style={styles.label}>Fuel Type</Text>
        <Dropdown
          data={itemsFuelType}
          labelField="label"
          valueField="value"
          value={valueFuelType}
          placeholder="Select Fuel Type"
          placeholderStyle={styles.dropdownPlaceholder}
          selectedTextStyle={styles.dropdownSelectedText}
          onChange={item => setValueFuelType(item.value)}
          style={styles.dropdown}
          containerStyle={styles.dropdownContainer}
          activeColor="#025383"
          itemContainerStyle={styles.dropdownItemContainer}
          renderItem={(item, selected) => (
            <View style={[
              styles.dropdownItemContainer,
              selected && styles.dropdownItemContainerSelected
            ]}>
              <Text style={[
                styles.dropdownItemText,
                selected && { color: '#fff', fontWeight: '600' }
              ]}>
                {item.label}
              </Text>
            </View>
          )}
          renderRightIcon={() => (
            <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
          )}
          showsVerticalScrollIndicator={false}
        />

        <View style={styles.lablesReading}>
          <View style={styles.labelContainer}>
            <Text style={styles.label}>Fuel Quantity</Text>
            <Text style={styles.requiredStar}>*</Text>
          </View>
          <Text style={styles.label1}>{fuelUnit}</Text>
        </View>
        <View style={styles.row}>
          <TextInput
            style={[styles.input, styles.meterInput, showErrors && !fuelQuantity ? styles.inputError : null]}
            value={fuelQuantity}
            onChangeText={setFuelQuantity}
            placeholder="Fuel Quantity"
            keyboardType="numeric"
          />
          <Dropdown
            data={itemsFuelUnit}
            labelField="label"
            valueField="value"
            value={fuelUnit}
            placeholder="Gallon"
            placeholderStyle={styles.dropdownPlaceholder}
            selectedTextStyle={styles.dropdownSelectedText}
            onChange={item => setFuelUnit(item.value)}
            style={[styles.dropdown, styles.unitDropdown]}
            containerStyle={styles.dropdownContainer}
            activeColor="#025383"
            itemContainerStyle={styles.dropdownItemContainer}
            renderItem={(item, selected) => (
              <View style={[
                styles.dropdownItemContainer,
                selected && styles.dropdownItemContainerSelected
            ]}>
              <Text style={[
                styles.dropdownItemText,
                selected && { color: '#fff', fontWeight: '600' }
              ]}>
                {item.label}
              </Text>
            </View>
          )}
          renderRightIcon={() => (
            <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
          )}
          showsVerticalScrollIndicator={false}
        />
        </View>
        {showErrors && !fuelQuantity && (
          <Text style={styles.errorText}>Fuel Quantity is required</Text>
        )}

        <Text style={styles.label}>Fuel Cost</Text>
        <TextInput
          style={styles.input}
          value={fuelCost}
          onChangeText={setFuelCost}
          keyboardType="numeric"
          placeholder="Fuel Cost"
        />

        <Text style={styles.label}>Note</Text>
        <TextInput
          style={[styles.input, styles.noteInput]}
          value={note}
          onChangeText={setNote}
          placeholder="Note"
          multiline
        />

        <Text style={styles.label}>Receiver Name</Text>
        <TextInput
          style={styles.input}
          value={receiverName}
          onChangeText={setReceiverName}
          placeholder="Receiver Name"
        />
      </ScrollView>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.loadingModalContainer}>
          <View style={styles.loadingModalContent}>
            <Text style={styles.loadingModalText}>Please Wait</Text>
            <View style={styles.loadingIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.loadingModalTextadd}>Saving Fuel Log to Cloud</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '600',
    marginLeft: width * 0.025,
  },
  saveButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    backgroundColor: '#fff',
    borderRadius: width * 0.015,
  },
  saveButtonText: {
    color: '#025383',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  content: {
    padding: width * 0.04,
    paddingBottom: width * 0.2,
  },
  labelContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredStar: {
    color: 'red',
    fontWeight: 'bold',
    fontSize: width * 0.038,
    marginLeft: 2,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.015,
    padding: width * 0.025,
    fontSize: width * 0.04,
    marginBottom: width * 0.02,
    height: width * 0.12,
  },
  inputWithIcon: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingRight: width * 0.025,
  },
  inputError: {
    borderColor: 'red',
  },
  inputPlaceholder: {
    fontSize: width * 0.04,
    color: '#999',
  },
  inputText: {
    fontSize: width * 0.04,
    color: '#000',
  },
  inputIcon: {
    marginRight: width * 0.01,
    width: width * 0.08,
    height: width * 0.04,
  },
  meterInput: {
    width: '65%',
    marginRight: width * 0.03,
    marginTop: -8,
  },
  noteInput: {
    height: width * 0.3,
    textAlignVertical: 'top',
  },
  dropdown: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: width * 0.015,
    paddingHorizontal: width * 0.02,
    marginBottom: width * 0.02,
    height: width * 0.12,
  },
  unitDropdown: {
    width: '30%',
    height: width * 0.12,
    marginTop: -8,
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  dropdownPlaceholder: {
    fontSize: width * 0.04,
    color: '#888',
    fontWeight: '400',
  },
  dropdownSelectedText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
  },
  dropdownItemText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '400',
  },
  dropdownItemContainer: {
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  dropdownItemContainerSelected: {
    backgroundColor: '#025383',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  row: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.01,
  },
  radioRow: {
    flexDirection: 'column',
    marginBottom: width * 0.02,
  },
  radioOptions: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: width * 0.01,
  },
  radioContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginRight: width * 0.04,
  },
  checkbox: {
    width: width * 0.04,
    height: width * 0.04,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: width * 0.02,
  },
  checkboxSelected: {
    width: width * 0.04,
    height: width * 0.04,
    backgroundColor: '#015185',
    borderRadius: 4,
  },
  radioLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  lablesReading: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  label1: {
    paddingHorizontal: 10,
    paddingVertical: 4,
    overflow: 'hidden',
    marginRight: width * 0.18,
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginBottom: width * 0.02,
    marginLeft: 2,
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  loadingModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  loadingIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  loadingModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default CreateFuelLog;
