import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Modal,
  FlatList,
  Alert,
  KeyboardAvoidingView,
  Platform,
  Animated,
  ActivityIndicator,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { Dropdown } from 'react-native-element-dropdown';
import NetInfo from '@react-native-community/netinfo';
// Firebase imports removed - using MongoDB only
import Toast from 'react-native-toast-message';
import EquipmentAPI from '../../ConfigMongo/EquipmentMongoConfig';

const { width } = Dimensions.get('window');

// MongoDB API URLs
const MONGODB_EQUIPMENT_API_URL = 'http://168.231.113.219:5001';
const MONGODB_DVIR_API_URL = 'http://168.231.113.219:5006';
const MONGODB_ADMIN_USERS_API_URL = 'http://168.231.113.219:5003';

const DvirInspectioninfoStep1 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [country, setCountry] = useState(null);
  const [countryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);
  const [project, setProject] = useState(null);
  const [projectItems, setProjectItems] = useState([]);
  const [inspector, setInspector] = useState('');
  const [checklistModalVisible, setChecklistModalVisible] = useState(false);
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [checklists, setChecklists] = useState([]);
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [savedEquipmentList, setSavedEquipmentList] = useState([]);
  const [checklistSearchQuery, setChecklistSearchQuery] = useState('');
  const [isChecklistSearchActive, setIsChecklistSearchActive] = useState(false);
  const [selectedTab, setSelectedTab] = useState('ALL');
  const [selectedChecklistId, setSelectedChecklistId] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [meterReading, setMeterReading] = useState('');
  const [currentMeterReading, setCurrentMeterReading] = useState('');
  const [requestNumber, setRequestNumber] = useState('');
  const [inspectionTypes, setInspectionTypes] = useState({
    DVIR: false,
    DOT: false,
  });
  const [equipmentSearchQuery, setEquipmentSearchQuery] = useState('');
  const [isEquipmentSearchActive, setIsEquipmentSearchActive] = useState(false);
  const [userInfo, setUserInfo] = useState({
    countries: [],
    equipmentPermissions: [],
    userUid: '',
  });
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isLoadingEquipment, setIsLoadingEquipment] = useState(false);
  const [selectedCountry, setSelectedCountry] = useState('All Countries');
  const [totalDvirCount, setTotalDvirCount] = useState('Loading...');
  const equipmentSearchInputRef = useRef(null);
  const isMounted = useRef(true);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({ type, text1, text2, position: 'top' });
  };

  const generateRequestNumber = async () => {
    try {
      let requestCount = 0;
      const isOnline = await checkNetworkStatus();
      
      if (isOnline) {
        try {
          // Use MongoDB count for dynamic request number
          const controller = new AbortController();
          const timeoutId = setTimeout(() => controller.abort(), 10000);
          
          const response = await fetch(`${MONGODB_DVIR_API_URL}/reports/count`, {
            method: 'GET',
            headers: { 'Content-Type': 'application/json' },
            signal: controller.signal,
          });
          
          clearTimeout(timeoutId);
          
          if (response.ok) {
            const result = await response.json();
            if (result.success && typeof result.count === 'number') {
              requestCount = result.count + 1; // Add 1 for next report number
              console.log('📊 Using MongoDB count for request number:', requestCount);
            } else {
              throw new Error('Invalid count response');
            }
          } else {
            throw new Error(`API error: ${response.status}`);
          }
        } catch (mongoError) {
          console.error('❌ MongoDB count failed:', mongoError.message);
          // Use default fallback since Firebase is no longer used
          requestCount = 1;
        }
      } else {
        // Offline fallback
        const lastRequestRef = await AsyncStorage.getItem('lastDvirInspectionNumber');
        const lastNumber = parseInt(lastRequestRef, 10) || 0;
        requestCount = lastNumber + 1;
      }
      
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
      return `${countryCode}-${projectCode}-DVIR-${String(requestCount).padStart(3, '0')}`;
    } catch (error) {
      console.error('Error generating request number:', error);
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
      return `${countryCode}-${projectCode}-DVIR-001`;
    }
  };

  const fetchUserInfo = async () => {
    try {
      let userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        return null;
      }

      console.log('🔍 Fetching user info from MongoDB for UID:', userUid);
      
      // Fetch user data from MongoDB API
      const response = await fetch(`${MONGODB_ADMIN_USERS_API_URL}/users/${userUid}`);
      
      if (!response.ok) {
        if (response.status === 404) {
          showToast('error', 'Error', 'User not found in database');
          return null;
        }
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      const result = await response.json();
      const userData = result.data || result;
      
      if (!userData) {
        showToast('error', 'Error', 'No user data received');
        return null;
      }

      console.log('✅ User data fetched:', {
        name: userData.name,
        email: userData.email,
        countries: userData.countries?.length || 0,
        modules: userData.modules?.length || 0
      });

      const equipmentModule = userData.modules?.find(
        (module) => module.module?.toLowerCase() === 'equipments'
      ) || { permissions: [] };

      const userInfo = {
        countries: userData.countries || [],
        equipmentPermissions: equipmentModule.permissions || [],
        userUid: userUid,
      };

      await AsyncStorage.setItem('userInfo', JSON.stringify(userInfo));
      setUserInfo(userInfo);
      return userInfo;
    } catch (error) {
      console.error('❌ Error fetching user info from MongoDB:', error);
      showToast('error', 'Error', 'Failed to fetch user permissions.');
      return null;
    }
  };

  const filterEquipmentByPermission = useCallback((equipmentArr, userInfoObj) => {
    console.log('🔍 filterEquipmentByPermission called with:', equipmentArr?.length, 'items');
    console.log('🔍 userInfoObj:', userInfoObj);
    
    if (!equipmentArr || !userInfoObj) {
      console.log('🔍 No equipment or user info, returning empty array');
      return [];
    }
    
    console.log('🔍 Equipment permissions:', userInfoObj.equipmentPermissions);
    console.log('🔍 User countries:', userInfoObj.countries);
    
    // Show all equipment the user has access to, regardless of category
    if (userInfoObj.equipmentPermissions.includes('onlyMineView')) {
      console.log('🔍 Using onlyMineView filter');
      const creatorIdentifier = userInfoObj.userUid;
      const filtered = equipmentArr.filter(
        (item) =>
          (item.createdBy === creatorIdentifier || !item.createdBy) &&
          userInfoObj.countries.includes(item.country)
      );
      console.log('🔍 onlyMineView filtered result:', filtered.length, 'items');
      return filtered;
    } else if (userInfoObj.equipmentPermissions.includes('view')) {
      console.log('🔍 Using view filter');
      const filtered = equipmentArr.filter(
        (item) => userInfoObj.countries.includes(item.country)
      );
      console.log('🔍 view filtered result:', filtered.length, 'items');
      return filtered;
    } else {
      console.log('🔍 No valid permissions found:', userInfoObj.equipmentPermissions);
      console.log('🔍 User countries:', userInfoObj.countries);
      
      // Try to filter by countries if they exist, otherwise return all equipment
      if (userInfoObj.countries && userInfoObj.countries.length > 0) {
        console.log('🔍 Trying to filter by countries:', userInfoObj.countries);
        const filtered = equipmentArr.filter((item) => 
          userInfoObj.countries.includes(item.country)
        );
        console.log('🔍 Country filtered result:', filtered.length, 'items');
        
        // If country filtering results in no items, return all equipment
        if (filtered.length === 0) {
          console.log('🔍 No items match user countries, returning all equipment');
          return equipmentArr;
        }
        return filtered;
      } else {
        console.log('🔍 No user countries defined, returning all equipment');
        return equipmentArr;
      }
    }
  }, []);

  // Load equipment data from MongoDB - same approach as Inspection Report
  const loadEquipmentData = async (showSyncModal = true) => {
    if (showSyncModal) setSyncModalVisible(true);
    try {
      const netInfo = await NetInfo.fetch();
      let data = [];
      
      if (netInfo.isConnected) {
        try {
          // Try MongoDB first using EquipmentAPI
          console.log('🔄 Loading equipment from MongoDB...');
          const response = await EquipmentAPI.getAllEquipment();
          
          if (response.success && response.data) {
            data = response.data;
            console.log(`✅ Loaded ${data.length} equipment items from MongoDB`);
            
            // Deduplicate by id
            data = data.filter((item, index, self) => 
              self.findIndex((t) => t.id === item.id) === index
            );
            
            // Sort by updated date
            data.sort((a, b) => {
              const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
              const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
              return dateB - dateA;
            });
            
            // Cache to local storage
            await AsyncStorage.setItem('equipment', JSON.stringify(data));
          } else {
            throw new Error('No equipment data from MongoDB');
          }
        } catch (mongoError) {
          console.error('❌ MongoDB equipment fetch failed:', mongoError.message);
          // Fallback to direct fetch if EquipmentAPI fails
          try {
            const response = await fetch(`${MONGODB_EQUIPMENT_API_URL}/EquipmentsDataJSON/GlobalEquipmentJsonData`, {
              method: 'GET',
              headers: { 'Content-Type': 'application/json' },
            });
            
            if (response.ok) {
              const result = await response.json();
              if (result.success && result.data && Array.isArray(result.data)) {
                data = result.data;
                console.log(`✅ Loaded ${data.length} equipment items from MongoDB (fallback)`);
                
                // Deduplicate and sort
                data = data.filter((item, index, self) => 
                  self.findIndex((t) => t.id === item.id) === index
                );
                data.sort((a, b) => {
                  const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
                  const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
                  return dateB - dateA;
                });
                
                await AsyncStorage.setItem('equipment', JSON.stringify(data));
              } else {
                throw new Error('Invalid MongoDB response format');
              }
            } else {
              throw new Error(`MongoDB API error: ${response.status}`);
            }
          } catch (directError) {
            console.error('❌ Direct MongoDB fetch also failed:', directError.message);
            throw new Error('Failed to load equipment from MongoDB');
          }
        }
      } else {
        // Offline mode - load from local storage
        const storedEquipment = await AsyncStorage.getItem('equipment');
        data = storedEquipment ? JSON.parse(storedEquipment) : [];
        data.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        Toast.show({ type: 'info', text1: 'Offline', text2: 'Loaded equipment from local storage.', position: 'top' });
      }
      
      setFirebaseEquipment(data);
    } catch (error) {
      console.error('❌ Error loading equipment:', error);
      const storedEquipment = await AsyncStorage.getItem('equipment');
      const parsedEquipment = storedEquipment ? JSON.parse(storedEquipment) : [];
      setFirebaseEquipment(parsedEquipment);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load equipment. Using cached data.',
        position: 'top',
      });
    } finally {
      if (showSyncModal) setSyncModalVisible(false);
      setEquipmentSearchQuery('');
      setIsEquipmentSearchActive(false);
    }
  };

  const loadEquipment = async (isRefresh = false) => {
    setSyncModalVisible(true);
    setIsLoadingEquipment(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      // Check user authentication from AsyncStorage instead of Firebase Auth
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setSyncModalVisible(false);
        setIsLoadingEquipment(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        return;
      }

      const userInfoData = await fetchUserInfo();
      if (!userInfoData) {
        setSyncModalVisible(false);
        setIsLoadingEquipment(false);
        rotateAnim.setValue(0);
        return;
      }

      console.log('User Info:', userInfoData); // Debug: Check user info and permissions
      const isOnline = await checkNetworkStatus();
      let parsedData = [];

      if (isOnline) {
        // Try MongoDB first
        try {
          console.log('🔄 Attempting to load equipment from MongoDB...');
          parsedData = await loadEquipmentFromMongoDB(userInfoData);
          
          if (parsedData.length === 0) {
            throw new Error('No data returned from MongoDB');
          }
          
          console.log('✅ Successfully loaded equipment from MongoDB');
        } catch (mongoError) {
          console.error('❌ MongoDB load failed:', mongoError.message);
          // Use empty data if MongoDB fails - no Firebase fallback
          parsedData = [];
          showToast('error', 'Error', 'Failed to load equipment data from server.');
        }

        // Process and filter the data regardless of source
        parsedData = parsedData
          .filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index)
          .sort((a, b) => {
            const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
            const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
            return dateB - dateA;
          });

        console.log('🔍 Data BEFORE filterEquipmentByPermission:', parsedData.length, 'items');
        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('🔍 Data AFTER filterEquipmentByPermission:', filteredData.length, 'items');
        console.log('🔍 Filtered data sample:', filteredData.slice(0, 2));

        parsedData = filteredData;
        await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
      } else {
        const savedData = await AsyncStorage.getItem('equipment');
        parsedData = savedData ? JSON.parse(savedData) : [];
        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('Offline Data after filtering:', filteredData); // Debug: Check offline data
        parsedData = filteredData;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        showToast('info', 'Offline', 'Equipment data refreshed from local storage.');
      }

      console.log('🔍 About to set firebaseEquipment with:', parsedData.length, 'items');
      console.log('🔍 Sample parsedData before setFirebaseEquipment:', parsedData.slice(0, 2));
      
      if (isRefresh) {
        console.log('🔍 Setting firebaseEquipment (refresh mode)');
        setFirebaseEquipment(parsedData);
      } else {
        console.log('🔍 Setting firebaseEquipment (append mode)');
        setFirebaseEquipment((prev) => [...prev, ...parsedData]);
      }

      setSavedEquipmentList(parsedData);
      
      // Additional state management like EquipmentOne
      const storedCountry = await AsyncStorage.getItem('selectedCountry');
      setSelectedCountry(storedCountry || 'All Countries');
    } catch (error) {
      console.error('Error loading equipment:', error.code, error.message);
      let errorMessage = 'Failed to load equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check database security rules.';
      } else if (error.code === 'NETWORK_ERROR' || error.message?.includes('MongoDB API error')) {
        errorMessage = 'Offline: Loading from local storage.';
        const savedData = await AsyncStorage.getItem('equipment');
        const parsedData = savedData ? JSON.parse(savedData) : [];
        const userInfoData = await AsyncStorage.getItem('userInfo');
        const userInfoParsed = userInfoData ? JSON.parse(userInfoData) : userInfo;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        const filteredData = filterEquipmentByPermission(parsedData, userInfoParsed);
        setFirebaseEquipment(filteredData);
        setSavedEquipmentList(filteredData);
      }
      const isNetworkError = error.code === 'NETWORK_ERROR' || error.message?.includes('MongoDB API error');
      showToast(
        isNetworkError ? 'info' : 'error',
        isNetworkError ? 'Offline' : 'Error',
        errorMessage
      );
    } finally {
      setSyncModalVisible(false);
      setIsLoadingEquipment(false);
      rotateAnim.setValue(0);
    }
  };

  const loadChecklists = async () => {
    try {
      // Load checklists from local storage only - Firebase removed
      const storedChecklists = await AsyncStorage.getItem('checklists');
      const parsedChecklists = storedChecklists ? JSON.parse(storedChecklists) : [];
      setChecklists(parsedChecklists);
      console.log('Loaded checklists from local storage:', parsedChecklists.length);
    } catch (error) {
      console.error('Error loading checklists:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load checklists.',
        position: 'top',
      });
    }
  };

  // Function to fetch total DVIR count from MongoDB
  const fetchTotalDvirCount = async (retryCount = 0) => {
    const maxRetries = 3;
    
    try {
      console.log(`🔄 Fetching total DVIR count from MongoDB... (attempt ${retryCount + 1})`);
      console.log('🔗 API URL:', `${MONGODB_DVIR_API_URL}/reports/count`);
      
      // Add timeout to prevent hanging requests
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 15000); // 15 second timeout
      
      const startTime = Date.now();
      const response = await fetch(`${MONGODB_DVIR_API_URL}/reports/count`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        signal: controller.signal,
      });
      
      clearTimeout(timeoutId);
      const endTime = Date.now();
      
      console.log(`📡 Response received in ${endTime - startTime}ms`);
      console.log('📡 Response status:', response.status);
      console.log('📡 Response ok:', response.ok);
      console.log('📡 Response headers:', response.headers);

      if (response.ok) {
        const result = await response.json();
        console.log('📊 Response body:', JSON.stringify(result, null, 2));
        
        if (result.success !== false && result.count !== undefined) {
          setTotalDvirCount(result.count);
          console.log('✅ Total DVIR count fetched successfully:', result.count);
        } else {
          console.error('❌ Invalid response format:', result);
          if (retryCount < maxRetries) {
            console.log(`🔄 Retrying... (${retryCount + 1}/${maxRetries})`);
            setTimeout(() => fetchTotalDvirCount(retryCount + 1), 2000);
          } else {
            setTotalDvirCount('Error');
          }
        }
      } else {
        const errorText = await response.text();
        console.error(`❌ API request failed with status ${response.status}:`, errorText);
        
        if (retryCount < maxRetries && response.status >= 500) {
          console.log(`🔄 Retrying due to server error... (${retryCount + 1}/${maxRetries})`);
          setTimeout(() => fetchTotalDvirCount(retryCount + 1), 2000);
        } else {
          setTotalDvirCount(`Error ${response.status}`);
        }
      }
    } catch (error) {
      console.error('❌ Error fetching DVIR count:', error);
      console.error('❌ Error details:', {
        name: error.name,
        message: error.message,
        stack: error.stack
      });
      
      if (error.name === 'AbortError') {
        console.warn('⚠️ DVIR count request timed out');
        if (retryCount < maxRetries) {
          console.log(`🔄 Retrying after timeout... (${retryCount + 1}/${maxRetries})`);
          setTimeout(() => fetchTotalDvirCount(retryCount + 1), 2000);
        } else {
          setTotalDvirCount('Timeout');
        }
      } else if (retryCount < maxRetries) {
        console.log(`🔄 Retrying after error... (${retryCount + 1}/${maxRetries})`);
        setTimeout(() => fetchTotalDvirCount(retryCount + 1), 2000);
      } else {
        setTotalDvirCount('Failed');
      }
    }
  };

  const getFilteredEquipment = () => {
    let filtered = [...firebaseEquipment];
    
    console.log('🔍 getFilteredEquipment called with:', firebaseEquipment.length, 'items');
    console.log('🔍 Selected tab:', selectedTab);
    if (firebaseEquipment.length > 0) {
      console.log('🔍 Sample equipment data:', {
        id: firebaseEquipment[0].id,
        category: firebaseEquipment[0].category,
        mainCategory: firebaseEquipment[0].mainCategory,
        vehicleType: firebaseEquipment[0].vehicleType
      });
    }
    
    // Only show Light Vehicles and Heavy Vehicles
    // Check both category (MongoDB) and mainCategory (Firebase) fields for compatibility
    filtered = filtered.filter((item) => {
      const category = item.category || '';
      const vehicleType = item.vehicleType || '';
      const mainCategory = item.mainCategory || '';
      
      // Check all category fields and return true if any match
      
      // Check mainCategory first
      if (mainCategory) {
        const mainCatLower = mainCategory.toLowerCase();
        if (mainCatLower.includes('light vehicle') ||
            mainCatLower.includes('heavy vehicle') ||
            mainCatLower.includes('light vehicles') ||
            mainCatLower.includes('heavy vehicles') ||
            mainCatLower.includes('heavy equipment') ||
            mainCatLower.includes('light equipment')) {
          return true;
        }
      }
      
      // Check MongoDB category field (this is the primary field most items use)
      if (category) {
        const catLower = category.toLowerCase();
        if (catLower.includes('light vehicle') ||
            catLower.includes('heavy vehicle') ||
            catLower.includes('light vehicles') ||
            catLower.includes('heavy vehicles') ||
            catLower.includes('light equipment') ||
            catLower.includes('heavy equipment')) {
          return true;
        }
      }
      
      // Check vehicleType field as additional backup
      if (vehicleType) {
        const vtLower = vehicleType.toLowerCase();
        if (vtLower.includes('light vehicle') ||
            vtLower.includes('heavy vehicle') ||
            vtLower.includes('light vehicles') ||
            vtLower.includes('heavy vehicles') ||
            vtLower.includes('light equipment') ||
            vtLower.includes('heavy equipment')) {
          return true;
        }
      }
      
      // If no category fields match, return false to exclude the item
      return false;
    });

    // Apply tab filter if not ALL
    if (selectedTab !== 'ALL') {
      filtered = filtered.filter((item) => {
        const category = item.category || '';
        const vehicleType = item.vehicleType || '';
        const mainCategory = item.mainCategory || '';
        
        switch (selectedTab) {
          case 'LV':
            // Check all fields for light vehicle matches
            if (mainCategory) {
              const mainCatLower = mainCategory.toLowerCase();
              if (mainCatLower.includes('light vehicle') || 
                  mainCatLower.includes('light vehicles') || 
                  mainCatLower.includes('light equipment')) {
                return true;
              }
            }
            if (category) {
              const catLower = category.toLowerCase();
              if (catLower.includes('light vehicle') || 
                  catLower.includes('light vehicles') || 
                  catLower.includes('light equipment')) {
                return true;
              }
            }
            if (vehicleType) {
              const vtLower = vehicleType.toLowerCase();
              if (vtLower.includes('light vehicle') || 
                  vtLower.includes('light vehicles') || 
                  vtLower.includes('light equipment')) {
                return true;
              }
            }
            return false;
          case 'HV':
            // Check all fields for heavy vehicle matches
            if (mainCategory) {
              const mainCatLower = mainCategory.toLowerCase();
              if (mainCatLower.includes('heavy vehicle') || 
                  mainCatLower.includes('heavy vehicles') || 
                  mainCatLower.includes('heavy equipment')) {
                return true;
              }
            }
            if (category) {
              const catLower = category.toLowerCase();
              if (catLower.includes('heavy vehicle') || 
                  catLower.includes('heavy vehicles') || 
                  catLower.includes('heavy equipment')) {
                return true;
              }
            }
            if (vehicleType) {
              const vtLower = vehicleType.toLowerCase();
              if (vtLower.includes('heavy vehicle') || 
                  vtLower.includes('heavy vehicles') || 
                  vtLower.includes('heavy equipment')) {
                return true;
              }
            }
            return false;
          default:
            return true;
        }
      });
    }
    
    if (equipmentSearchQuery.trim()) {
      filtered = filtered.filter((item) =>
        [
          item.equipmentName || '',
          item.mainCategory || '',
          item.category || '',
          item.vehicleType || '',
          item.model || '',
          item.equipmentNumber || '',
        ].some((field) =>
          field.toLowerCase().includes(equipmentSearchQuery.toLowerCase())
        )
      );
    }
    console.log('🔍 Final filtered equipment count:', filtered.length);
    return filtered;
  };

  useEffect(() => {
    const loadInitialData = async () => {
      await loadChecklists();
      await loadEquipmentData(); // Use the new MongoDB-based approach
      const newRequestNumber = await generateRequestNumber();
      setRequestNumber(newRequestNumber);
    };
    loadInitialData();

    // Fetch total DVIR count separately with a small delay to ensure component is ready
    setTimeout(() => {
      if (isMounted.current) {
        fetchTotalDvirCount();
      }
    }, 500);

    return () => {
      isMounted.current = false;
    };
  }, [country, project]);

  // Separate useEffect to fetch DVIR count on component mount
  useEffect(() => {
    console.log('🚀 Component mounted, fetching DVIR count...');
    const timer = setTimeout(() => {
      if (isMounted.current) {
        fetchTotalDvirCount();
      }
    }, 1000); // 1 second delay to ensure everything is ready

    return () => clearTimeout(timer);
  }, []); // Empty dependency array means this runs once on mount

  // Update request number when total count changes or country/project changes
  useEffect(() => {
    const updateRequestNumber = async () => {
      if ((country && project) && typeof totalDvirCount === 'number') {
        try {
          const newRequestNumber = await generateRequestNumber();
          setRequestNumber(newRequestNumber);
        } catch (error) {
          console.error('Error updating request number:', error);
        }
      }
    };
    
    updateRequestNumber();
  }, [totalDvirCount, country, project]);

  // Sync functionality updated to use MongoDB only
  useEffect(() => {
    const syncPendingChanges = async () => {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) return;

      // Check user authentication from AsyncStorage instead of Firebase Auth
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;

      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        const pending = equipment.filter((item) => item.pendingSync);

        if (pending.length > 0) {
          setSyncModalVisible(true);
          Animated.loop(
            Animated.timing(rotateAnim, {
              toValue: 1,
              duration: 1000,
              useNativeDriver: true,
            })
          ).start();

          // Sync to MongoDB instead of Firebase
          for (const item of pending) {
            try {
              if (item.pendingDelete) {
                // Delete from MongoDB equipment API
                console.log('Deleting equipment item:', item.id);
                // Add MongoDB delete call here if needed
              } else {
                // Update/add to MongoDB equipment API
                const updatedItem = { ...item, pendingSync: false, pendingDelete: false };
                console.log('Syncing equipment item to MongoDB:', updatedItem.id);
                // Add MongoDB sync call here if needed
              }
            } catch (syncError) {
              console.error('Error syncing item:', item.id, syncError);
            }
          }

          const updatedEquipment = equipment
            .filter((item) => !item.pendingDelete)
            .map((item) => ({
              ...item,
              pendingSync: false,
              pendingDelete: false,
            }));
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('success', 'Success', 'Pending changes synced to database.');
        }
      } catch (error) {
        console.error('Error syncing pending changes:', error);
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'Failed to sync pending changes.');
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncPendingChanges();
      }
    });

    return () => unsubscribe();
  }, []);

  useEffect(() => {
    if (country === 'Zambia') {
      setProjectItems([
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ]);
    } else if (country === 'DRC') {
      setProjectItems([
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ]);
    } else {
      setProjectItems([]);
    }
    setProject(null);
  }, [country]);

  const handleSelectPress = async () => {
    setEquipmentModalVisible(true); // Open modal immediately
    setIsLoadingEquipment(true); // Show loading indicator
    setEquipmentSearchQuery('');
    setIsEquipmentSearchActive(false);
    setSelectedTab('ALL'); // Reset to show all vehicles
  };

  // Load equipment data when modal is opened and loading is true - same as Inspection Report
  useEffect(() => {
    if (equipmentModalVisible && isLoadingEquipment) {
      (async () => {
        try {
          await loadEquipmentData(false); // Don't show SyncModal when opening equipment modal
        } catch (error) {
          console.error('Error opening equipment modal:', error);
          Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load equipment data. Please try again.', position: 'top' });
        } finally {
          setIsLoadingEquipment(false);
        }
      })();
    }
  }, [equipmentModalVisible, isLoadingEquipment]);

  const handleChecklistSelectPress = () => {
    setChecklistModalVisible(true);
  };

  const handleBackPress = () => {
    Alert.alert(
      'Exit Step',
      'Do you want to exit this step?',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'OK', onPress: () => navigation.goBack(), style: 'default' },
      ],
      { cancelable: false }
    );
  };

  const isNextDisabled =
    !country ||
    !project ||
    !inspector.trim() ||
    !selectedEquipment ||
    (!inspectionTypes.DVIR && !inspectionTypes.DOT);

  const validateChecklistItems = (checklistId) => {
    const checklist = checklists.find((c) => c.id === checklistId);
    if (!checklist) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'No checklist found with the selected ID.',
        position: 'top',
      });
      return false;
    }
    if (!checklist.items || !Array.isArray(checklist.items)) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'Checklist items are missing or not an array.',
        position: 'top',
      });
      return false;
    }
    if (
      !checklist.items.every(
        (item) =>
          item &&
          typeof item.text === 'string' &&
          Array.isArray(item.subItems) &&
          item.subItems.every(
            (subItem) =>
              subItem &&
              typeof subItem.itemName === 'string' &&
              typeof subItem.inputType === 'string'
          )
      )
    ) {
      Toast.show({
        type: 'error',
        text1: 'Invalid Checklist',
        text2: 'Checklist contains invalid item or sub-item structure.',
        position: 'top',
      });
      return false;
    }
    return true;
  };

  const saveStep1Data = async () => {
    try {
      const step1Data = {
        requestNumber,
        country,
        project,
        inspector,
        selectedEquipment,
        meterReading,
        currentMeterReading,
        selectedChecklistId,
        inspectionTypes,
        timestamp: new Date().getTime(),
      };
      const key = `step1-${step1Data.timestamp}`;
      await AsyncStorage.setItem(key, JSON.stringify(step1Data));
      return step1Data;
    } catch (error) {
      console.error('Error saving Step 1 data:', error);
      throw error;
    }
  };

  const handleNextPress = async () => {
    if (isNextDisabled) {
      Alert.alert(
        'Error',
        'Please fill all required fields: Country, Project, Inspector, Equipment, and at least one Inspection Type.'
      );
      return;
    }
    if (meterReading && parseFloat(meterReading) <= parseFloat(selectedEquipment.meterReading)) {
      Alert.alert(
        'Error',
        `Meter reading must be greater than ${selectedEquipment.meterReading || '0'}.`
      );
      return;
    }
    try {
      if (meterReading && parseFloat(meterReading) !== parseFloat(selectedEquipment.meterReading)) {
        await updateMeterReading();
      }
      await saveStep1Data();
      setActiveStep(2);
    } catch (error) {
      console.error('Error saving inspection data:', error);
      Alert.alert('Error', 'Failed to save inspection data. Please try again.');
    }
  };

  const updateMeterReading = async () => {
    if (!selectedEquipment) return;
    if (!userInfo.equipmentPermissions.includes('update')) {
      Toast.show({
        type: 'error',
        text1: 'Permission Denied',
        text2: 'You do not have permission to update equipment.',
        position: 'top',
      });
      return;
    }

    const updatedEquipment = {
      ...selectedEquipment,
      meterReading: parseFloat(meterReading),
      updatedAt: new Date().toISOString(),
      createdBy: userInfo.userUid,
    };

    try {
      const state = await NetInfo.fetch();
      let updatedEquipmentList = [...firebaseEquipment];

      if (state.isConnected) {
        // Update equipment via MongoDB API instead of Firebase
        try {
          console.log('Updating equipment meter reading via MongoDB API:', updatedEquipment.id);
          // Add MongoDB equipment update call here if needed
          // const response = await fetch(`${MONGODB_EQUIPMENT_API_URL}/equipment/${updatedEquipment.id}`, {
          //   method: 'PUT',
          //   headers: { 'Content-Type': 'application/json' },
          //   body: JSON.stringify(updatedEquipment)
          // });
        } catch (mongoError) {
          console.error('MongoDB equipment update failed:', mongoError);
        }

        updatedEquipmentList = updatedEquipmentList.map((item) =>
          item.id === updatedEquipment.id ? updatedEquipment : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipmentList));
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Meter reading updated in database.',
          position: 'top',
        });
      } else {
        updatedEquipmentList = updatedEquipmentList.map((item) =>
          item.id === updatedEquipment.id
            ? { ...updatedEquipment, pendingSync: true }
            : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipmentList));
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: 'Meter reading updated locally. Will sync when online.',
          position: 'top',
        });
      }

      setFirebaseEquipment(updatedEquipmentList);
      setSavedEquipmentList(updatedEquipmentList);
      setSelectedEquipment(updatedEquipment);
      setCurrentMeterReading(meterReading);
    } catch (error) {
      console.error('Error updating meter reading:', error);
      let errorMessage = 'Failed to update meter reading.';
      if (error.message && error.message.includes('Network')) {
        errorMessage = 'Offline: Meter reading updated locally, will sync when online.';
        updatedEquipmentList = firebaseEquipment.map((item) =>
          item.id === updatedEquipment.id
            ? { ...updatedEquipment, pendingSync: true }
            : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipmentList));
        setFirebaseEquipment(updatedEquipmentList);
        setSavedEquipmentList(updatedEquipmentList);
        setSelectedEquipment(updatedEquipment);
        setCurrentMeterReading(meterReading);
        Toast.show({
          type: 'info',
          text1: 'Offline',
          text2: errorMessage,
          position: 'top',
        });
        return;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'top',
      });
    }
  };

  const handleChecklistSearch = (query) => {
    setChecklistSearchQuery(query);
  };

  const handleEquipmentSearch = (query) => {
    setEquipmentSearchQuery(query);
  };

  const filteredChecklists = checklists.filter((checklist) =>
    checklist.title?.toLowerCase().includes(checklistSearchQuery.toLowerCase())
  );

  const toggleChecklistSelection = async (checklistId) => {
    if (!validateChecklistItems(checklistId)) return;
    setSelectedChecklistId((prev) => {
      const newId = prev === checklistId ? null : checklistId;
      AsyncStorage.setItem('selectedChecklistIds', JSON.stringify(newId ? [newId] : []));
      return newId;
    });
    setChecklistModalVisible(false);
  };

  const selectEquipment = async (equipment) => {
    setSelectedEquipment(equipment);
    setMeterReading(equipment.meterReading?.toString() || '');
    setCurrentMeterReading(equipment.meterReading?.toString() || '');
    setEquipmentModalVisible(false);
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity
      style={styles.checklistItem}
      onPress={() => toggleChecklistSelection(item.id)}
    >
      <View style={styles.checklistItemContent}>
        <MaterialIcons
          name={selectedChecklistId === item.id ? 'check-box' : 'check-box-outline-blank'}
          size={24}
          color={selectedChecklistId === item.id ? '#0078D4' : '#666'}
        />
        <Text style={styles.checklistItemText}>{item.title || 'Untitled Checklist'}</Text>
      </View>
    </TouchableOpacity>
  );

  const renderEquipmentItem = ({ item }) => (
    <View style={styles.equipmentRow}>
      <TouchableOpacity
        style={styles.equipmentInfo}
        onPress={() => selectEquipment(item)}
      >
        <View style={styles.equipmentDetailItemheader}>
          <Text style={styles.detailLabelheader}>{item.equipmentName || 'Equipment'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Category</Text>
          <Text style={styles.detailValue}>
            {item.category || item.vehicleType || item.mainCategory || 'N/A'}
          </Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Model</Text>
          <Text style={styles.detailValue}>{item.model || 'N/A'}</Text>
        </View>
        <View style={styles.equipmentDetailItem}>
          <Text style={styles.detailLabel}>Meter Reading</Text>
          <Text style={styles.detailValue}>
            {item.meterReading || '0'} {item.meterUnit || 'Miles'}
          </Text>
        </View>
        <View style={styles.equipmentDetailItemNoBorder}>
          <Text
            style={[styles.detailLabelstatuss, { color: getStatusColor(item.status || 'Available') }]}
          >
            {item.status || 'Available'}
          </Text>
        </View>
      </TouchableOpacity>
      <TouchableOpacity
        style={styles.selectEquipmentButton}
        onPress={() => selectEquipment(item)}
      >
        <Text style={styles.selectEquipmentText}>
          {selectedEquipment && selectedEquipment.id === item.id ? 'Selected' : 'Select'}
        </Text>
      </TouchableOpacity>
    </View>
  );

  return (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 0}
    >
      <View style={styles.container}>
        <ScrollView
          style={styles.mainContent}
          contentContainerStyle={styles.scrollContent}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.headingContainer}>
            <Text style={styles.headingText}>New Inspection</Text>
          </View>
          <View style={styles.requestRow}>
            <Text style={styles.requestLabel}>Request #</Text>
            <Text style={styles.requestValue}>{requestNumber}</Text>
          </View>
         
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>
              Country <Text style={styles.requiredLabel}>*</Text>
            </Text>
            <Dropdown
              style={styles.dropdown}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.selectedTextStyle}
              data={countryItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Country"
              value={country}
              onChange={(item) => setCountry(item.value)}
            />
          </View>
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>
              Project <Text style={styles.requiredLabel}>*</Text>
            </Text>
            <Dropdown
              style={[styles.dropdown, !country && styles.disabledDropdown]}
              placeholderStyle={styles.placeholderStyle}
              selectedTextStyle={styles.selectedTextStyle}
              data={projectItems}
              maxHeight={300}
              labelField="label"
              valueField="value"
              placeholder="Select Project"
              value={project}
              onChange={(item) => setProject(item.value)}
              disable={!country}
            />
          </View>
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>
              Inspector <Text style={styles.requiredLabel}>*</Text>
            </Text>
            <TextInput
              style={styles.input}
              value={inspector}
              onChangeText={setInspector}
              placeholder="Enter Inspector"
              placeholderTextColor="#888"
            />
          </View>
          <View style={styles.selectRow}>
            <Text style={styles.selectLabel}>Equipment</Text>
            <TouchableOpacity onPress={handleSelectPress} style={styles.selectButton}>
              <Text style={styles.selectText}>
                {selectedEquipment ? 'Change equipment' : 'Select'}
              </Text>
              <MaterialIcons name="chevron-right" size={16} color="#0078D4" />
            </TouchableOpacity>
          </View>
          {selectedEquipment && (
            <View style={styles.equipmentDetailsBox}>
              <View style={styles.vehicleEquipmentRow}>
                <Text style={styles.vehicleEquipmentText}>
                  Equipment Number: {selectedEquipment.equipmentNumber || 'N/A'}, Category:{' '}
                  {selectedEquipment.category || selectedEquipment.vehicleType || selectedEquipment.mainCategory || 'N/A'}, Model: {selectedEquipment.model || 'N/A'}
                </Text>
              </View>
              <View style={styles.labelRow}>
                <Text style={styles.meterReadingLabel}>
                  Meter Reading <Text style={styles.requiredLabel}>*</Text>
                </Text>
              </View>
              <View style={styles.inputRow}>
                <TextInput
                  style={styles.meterReadingInput}
                  value={meterReading}
                  onChangeText={setMeterReading}
                  placeholder="Enter meter reading"
                  placeholderTextColor="#888"
                  keyboardType="numeric"
                />
                <Text style={styles.meterUnit}>{selectedEquipment.meterUnit || 'Miles'}</Text>
              </View>
              {meterReading && parseFloat(meterReading) <= parseFloat(selectedEquipment.meterReading) && (
                <Text style={styles.errorText}>
                  Meter reading must be greater than {selectedEquipment.meterReading || '0'}.
                </Text>
              )}
              <View style={styles.currentMeterRow}>
                <Text style={styles.currentMeterLabel}>Current Meter Reading: </Text>
                <Text style={styles.currentMeterValue}>
                  {selectedEquipment.meterReading || '0'} {selectedEquipment.meterUnit || 'Miles'}
                </Text>
              </View>
            </View>
          )}
          <View style={{ borderTopWidth: 1, borderTopColor: '#e0e0e0', marginTop: 2, marginBottom: 8 }} />
          <TouchableOpacity
            style={styles.checkboxRow}
            onPress={() =>
              setInspectionTypes((prev) => ({ ...prev, DVIR: !prev.DVIR }))
            }
          >
            <View style={[styles.checkbox, inspectionTypes.DVIR && styles.checkboxChecked]}>
              {inspectionTypes.DVIR && (
                <MaterialIcons name="check" size={16} color="#fff" />
              )}
            </View>
            <Text style={styles.checkboxLabel}>DVIR - Driver Vehicle Inspection Report</Text>
          </TouchableOpacity>
        </ScrollView>
        <View style={styles.buttonContainer}>
          <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
            <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
            <Text style={styles.cancelButtonText}>Cancel</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.nextButton, isNextDisabled && styles.disabledButton]}
            onPress={handleNextPress}
            disabled={isNextDisabled}
          >
            <Text style={styles.nextButtonText}>Next</Text>
            <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
          </TouchableOpacity>
        </View>
        <Modal
          visible={checklistModalVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setChecklistModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.checklistModalContent}>
              <View style={styles.checklistModalHeader}>
                <Text style={styles.checklistModalTitle}>Select Checklist</Text>
                <TouchableOpacity
                  onPress={() => setChecklistModalVisible(false)}
                  style={styles.modalCloseIcon}
                >
                  <MaterialIcons name="close" size={24} color="#666" />
                </TouchableOpacity>
              </View>
              <View style={styles.checklistModalTabs}>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'ALL' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('ALL')}
                >
                  <Text style={[styles.tabText, selectedTab === 'ALL' && styles.activeTabText]}>
                    ALL
                  </Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'Assign' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('Assign')}
                >
                  <Text style={[styles.tabText, selectedTab === 'Assign' && styles.activeTabText]}>
                    Assign
                  </Text>
                </TouchableOpacity>
              </View>
              <View style={styles.checklistSearchContainer}>
                {isChecklistSearchActive ? (
                  <TextInput
                    style={styles.checklistSearchInput}
                    placeholder="Search checklists..."
                    placeholderTextColor="#888"
                    value={checklistSearchQuery}
                    onChangeText={handleChecklistSearch}
                    autoFocus
                  />
                ) : (
                  <Text style={styles.checklistSelectText}>Search Checklists</Text>
                )}
                <TouchableOpacity
                  onPress={() => {
                    setIsChecklistSearchActive(!isChecklistSearchActive);
                    if (isChecklistSearchActive) setChecklistSearchQuery('');
                  }}
                >
                  <MaterialIcons
                    name={isChecklistSearchActive ? 'close' : 'search'}
                    size={24}
                    color="#666"
                  />
                </TouchableOpacity>
              </View>
              <View style={styles.divider} />
              <FlatList
                data={filteredChecklists}
                keyExtractor={(item) => item.id.toString()}
                renderItem={renderChecklistItem}
                ListEmptyComponent={
                  <Text style={styles.checklistEmptyText}>No checklists found</Text>
                }
                style={styles.checklistList}
              />
              <View style={styles.divider} />
              <View style={styles.checklistModalFooter}>
                <TouchableOpacity
                  onPress={() => {
                    setChecklistModalVisible(false);
                  }}
                >
                  <Text style={styles.createChecklistText}>Create New Checklist</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={styles.checklistCloseButton}
                  onPress={() => setChecklistModalVisible(false)}
                >
                  <Text style={styles.checklistCloseText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
        <Modal
          visible={equipmentModalVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setEquipmentModalVisible(false)}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.equipmentModalContent}>
              <View style={styles.equipmentModalHeader}>
                <Text style={styles.equipmentModalTitle}>Select Equipment</Text>
                <TouchableOpacity
                  onPress={() => setEquipmentModalVisible(false)}
                  style={styles.modalCloseIcon}
                >
                  <MaterialIcons name="close" size={24} color="#666" />
                </TouchableOpacity>
              </View>
              <View style={styles.equipmentModalTabs}>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'ALL' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('ALL')}
                >
                  <Text style={[styles.tabText, selectedTab === 'ALL' && styles.activeTabText, { fontSize: 12 }]}>
                    ALL VEHICLES
                  </Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'LV' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('LV')}
                >
                  <Text style={[styles.tabText, selectedTab === 'LV' && styles.activeTabText, { fontSize: 12 }]}>
                    LIGHT VEHICLES
                  </Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.tabButton, selectedTab === 'HV' && styles.activeTabButton]}
                  onPress={() => setSelectedTab('HV')}
                >
                  <Text style={[styles.tabText, selectedTab === 'HV' && styles.activeTabText, { fontSize: 12 }]}>
                    HEAVY VEHICLES
                  </Text>
                </TouchableOpacity>
              </View>
              <TouchableOpacity
                style={[
                  styles.equipmentSearchContainer,
                  isEquipmentSearchActive && styles.equipmentSearchContainerActive,
                ]}
                onPress={() => {
                  setIsEquipmentSearchActive(true);
                  equipmentSearchInputRef.current?.focus();
                }}
              >
                {isEquipmentSearchActive ? (
                  <TextInput
                    ref={equipmentSearchInputRef}
                    style={[
                      styles.equipmentSearchInput,
                      isEquipmentSearchActive && styles.equipmentSearchInputActive,
                    ]}
                    placeholder={isEquipmentSearchActive ? 'Search...' : 'Search equipment...'}
                    placeholderTextColor="#888"
                    value={equipmentSearchQuery}
                    onChangeText={handleEquipmentSearch}
                    autoFocus={isEquipmentSearchActive}
                  />
                ) : (
                  <Text style={styles.equipmentSelectText}>Search Equipment</Text>
                )}
                <TouchableOpacity
                  onPress={() => {
                    if (isEquipmentSearchActive) {
                      setIsEquipmentSearchActive(false);
                      setEquipmentSearchQuery('');
                      equipmentSearchInputRef.current?.blur();
                    } else {
                      setIsEquipmentSearchActive(true);
                      equipmentSearchInputRef.current?.focus();
                    }
                  }}
                >
                  <MaterialIcons
                    name={isEquipmentSearchActive ? 'close' : 'search'}
                    size={24}
                    color="#666"
                  />
                </TouchableOpacity>
              </TouchableOpacity>
              <View style={styles.divider} />
              {isLoadingEquipment ? (
                <View style={styles.loadingContainer}>
                  <ActivityIndicator size="large" color="#3481BC" />
                  <Text style={styles.loadingText}>Loading Equipment...</Text>
                </View>
              ) : (
                <FlatList
                  data={getFilteredEquipment()}
                  keyExtractor={(item) => item.id.toString()}
                  renderItem={renderEquipmentItem}
                  ListEmptyComponent={
                    <Text style={styles.equipmentEmptyText}>No equipment found</Text>
                  }
                  style={styles.equipmentList}
                />
              )}
              <View style={styles.divider} />
              <View style={styles.equipmentModalFooter}>
                <TouchableOpacity
                  style={styles.equipmentCloseButton}
                  onPress={() => setEquipmentModalVisible(false)}
                >
                  <Text style={styles.equipmentCloseText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
        <Modal
          visible={syncModalVisible}
          transparent
          animationType="fade"
        >
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>Processing Equipment Data</Text>
              </View>
            </View>
          </View>
        </Modal>
        <Toast />
      </View>
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: 15,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 10,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  requestRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    marginBottom: 20,
    borderRadius: 5,
  },
  requestLabel: {
    fontSize: 16,
    color: '#015185',
  },
  requestValue: {
    fontSize: 16,
    color: '#015185',
  },
  fieldContainer: {
    marginBottom: 15,
  },
  label: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 5,
    fontWeight: '500',
  },
  requiredLabel: {
    color: 'red',
    fontSize: 16,
  },
  input: {
    height: 40,
    borderWidth: 1,
    borderColor: '#0078D4',
    borderRadius: 5,
    paddingHorizontal: 10,
    fontSize: 16,
    color: '#000',
    backgroundColor: '#fff',
  },
  dropdown: {
    height: 40,
    borderWidth: 1,
    borderColor: '#0078D4',
    borderRadius: 5,
    backgroundColor: '#fff',
    paddingHorizontal: 10,
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  selectedTextStyle: {
    fontSize: 16,
    color: '#333',
  },
  selectRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    height: 50,
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 15,
    marginTop: 10,
    borderRadius: 5,
  },
  selectLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectText: {
    fontSize: 16,
    color: '#0078D4',
    marginLeft: 5,
  },
  equipmentDetailsBox: {
    width: '100%',
    backgroundColor: '#f0f0f0',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginBottom: 10,
    borderRadius: 5,
  },
  vehicleEquipmentRow: {
    paddingVertical: 10,
  },
  vehicleEquipmentText: {
    fontSize: 16,
    color: '#333',
  },
  labelRow: {
    paddingVertical: 5,
  },
  meterReadingLabel: {
    fontSize: 16,
    color: '#333',
    fontWeight: '600',
  },
  inputRow: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 5,
  },
  meterReadingInput: {
    flex: 1,
    fontSize: 16,
    color: '#333',
    padding: 0,
  },
  meterUnit: {
    color: '#666',
    fontSize: 14,
    marginLeft: 8,
  },
  currentMeterRow: {
    flexDirection: 'row',
    marginTop: 8,
    alignItems: 'center',
  },
  currentMeterLabel: {
    fontSize: 13,
    color: '#666',
  },
  currentMeterValue: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  errorText: {
    color: 'red',
    fontSize: 12,
    marginTop: 5,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  disabledButton: {
    opacity: 0.6,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  checklistModalContent: {
    width: '90%',
    maxHeight: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    elevation: 5,
  },
  checklistModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  checklistModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  modalCloseIcon: {
    padding: 5,
  },
  checklistModalTabs: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
  },
  tabButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 10,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  activeTabButton: {
    borderBottomColor: '#0078D4',
  },
  tabText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '600',
  },
  activeTabText: {
    color: '#0078D4',
  },
  checklistSearchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
  },
  checklistSelectText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  checklistSearchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  divider: {
    height: 1,
    backgroundColor: '#eee',
    marginVertical: 10,
  },
  checklistList: {
    maxHeight: 200,
  },
  checklistItem: {
    paddingVertical: 10,
  },
  checklistItemContent: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  checklistItemText: {
    fontSize: 16,
    color: '#333',
    marginLeft: 10,
  },
  checklistEmptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  checklistModalFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  createChecklistText: {
    fontSize: 16,
    color: '#0078D4',
    fontWeight: '600',
  },
  checklistCloseButton: {
    padding: 10,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 20,
    marginBottom: 8,
    paddingVertical: 8,
  },
  checkbox: {
    width: 24,
    height: 24,
    borderWidth: 2,
    borderColor: '#0078D4',
    borderRadius: 4,
    marginRight: 12,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  checkboxChecked: {
    backgroundColor: '#0078D4',
    borderColor: '#0078D4',
  },
  checkboxLabel: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '500',
  },
  checklistCloseText: {
    fontSize: 16,
    color: '#3481BC',
    fontWeight: '600',
  },
  equipmentModalContent: {
    width: '90%',
    maxHeight: '70%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    elevation: 5,
  },
  equipmentModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  equipmentModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  equipmentModalTabs: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
  },
  equipmentSearchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: 10,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 10,
  },
  equipmentSearchContainerActive: {
    paddingVertical: 2,
  },
  equipmentSelectText: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  equipmentSearchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  equipmentSearchInputActive: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#333',
  },
  equipmentList: {
    maxHeight: 300,
    paddingHorizontal: 12,
  },
  equipmentRow: {
    marginVertical: 5,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
    paddingRight: 10,
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: '5%',
    paddingRight: '10%',
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingLeft: '5%',
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: '5%',
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatuss: {
    fontSize: 14,
    fontWeight: '400',
    width: '60%',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: 18,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: 16,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  selectEquipmentButton: {
    position: 'absolute',
    right: 10,
    bottom: 10,
    backgroundColor: '#0078D4',
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 5,
  },
  selectEquipmentText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  equipmentEmptyText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    marginVertical: 20,
  },
  equipmentModalFooter: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  addEquipmentText: {
    fontSize: 16,
    color: '#0078D4',
    fontWeight: '600',
  },
  equipmentCloseButton: {
    padding: 10,
  },
  equipmentCloseText: {
    fontSize: 16,
    color: '#3481BC',
    fontWeight: '600',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  loadingText: {
    marginTop: 10,
    color: '#666',
    fontSize: 16,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
});

export default DvirInspectioninfoStep1;