import React from 'react';
import { View, Text, StyleSheet, TouchableOpacity, ScrollView, Image, ActivityIndicator, StatusBar, Dimensions } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';

const ViewNotificationsadded = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { title, description, imageUrl, createdAt } = route.params || {};

  // Move Dimensions inside component to fix Hermes runtime error
  const { width } = Dimensions.get('window');
  const TOTAL_HEADER_HEIGHT = width * 0.25;

  // State to handle image loading
  const [imageLoading, setImageLoading] = React.useState(true);
  const [imageError, setImageError] = React.useState(false);

  // Format the publication date
  const formatDetailDate = (isoDate) => {
    if (!isoDate) return 'Unknown Date';
    const date = new Date(isoDate);
    if (isNaN(date.getTime())) return 'Invalid Date';
    const options = { year: 'numeric', month: 'short', day: 'numeric' };
    return date.toLocaleDateString('en-US', options);
  };

  const publishedDate = formatDetailDate(createdAt);

  return (
    <View style={styles.screen}>
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            accessibilityRole="button"
            accessibilityLabel="Go Back"
          >
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Notice Details</Text>
        </View>
      </LinearGradient>
      <ScrollView style={styles.content}>
        {/* Title */}
        <Text style={styles.title}>{title || 'Untitled Notice'}</Text>

        {/* Publication Date */}
        <Text style={styles.publishedDate}>Published on: {publishedDate}</Text>

        {/* Image (only if available) */}
        {imageUrl && (
          <View style={styles.imageContainer}>
            {imageLoading && !imageError && (
              <ActivityIndicator size="large" color="#3481BC" style={styles.imageLoader} />
            )}
            <Image
              source={{ uri: imageUrl }}
              style={[styles.image, imageError && styles.imageError]}
              resizeMode="contain"
              onLoad={() => setImageLoading(false)}
              onError={() => {
                setImageLoading(false);
                setImageError(true);
              }}
            />
            {imageError && (
              <Text style={styles.imageErrorText}>Failed to load image</Text>
            )}
          </View>
        )}

        {/* Description */}
        <Text style={styles.description}>{description || 'No description provided'}</Text>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  screen: {
    flex: 1,
    backgroundColor: '#f6f7fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingTop: Platform.OS === 'ios' ? 50 : 20,
    height: '100%',
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: 'white',
    flex: 1,
    textAlign: 'center',
    marginRight: 40, // Compensate for back button
  },
  content: {
    flex: 1,
    padding: 20,
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
    textAlign: 'center',
  },
  publishedDate: {
    fontSize: 14,
    color: '#666',
    marginBottom: 20,
    textAlign: 'center',
    fontStyle: 'italic',
  },
  imageContainer: {
    marginBottom: 20,
    alignItems: 'center',
  },
  image: {
    width: '100%',
    height: 200,
    borderRadius: 8,
  },
  imageLoader: {
    position: 'absolute',
    top: '50%',
    left: '50%',
    marginLeft: -20,
    marginTop: -20,
    zIndex: 1,
  },
  imageError: {
    opacity: 0.5,
  },
  imageErrorText: {
    color: '#ff0000',
    fontSize: 14,
    marginTop: 10,
    textAlign: 'center',
  },
  description: {
    fontSize: 16,
    color: '#333',
    lineHeight: 24,
    textAlign: 'justify',
  },
});

export default ViewNotificationsadded;