import React, { useState, useEffect, useCallback } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, ScrollView, Image, Platform, Alert, StatusBar, Dimensions } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NoticeAPI from '../../ConfigMongo/NoticeMongoConfig';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';
import ToastAndroid from 'react-native/Libraries/Components/ToastAndroid/ToastAndroid';
import { Modal, Animated } from 'react-native';

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Loading Notices</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.show(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

// Helper function to format date
const formatDate = (isoDate) => {
  const date = new Date(isoDate);
  const options = { year: 'numeric', month: 'short', day: 'numeric', weekday: 'short' };
  return date.toLocaleDateString('en-US', options).toUpperCase();
};

// Helper function to format date for detail view
const formatDetailDate = (isoDate) => {
  const date = new Date(isoDate);
  const options = { year: 'numeric', month: 'short', day: 'numeric' };
  return date.toLocaleDateString('en-US', options);
};

// Helper function to calculate time ago
const timeAgo = (isoDate) => {
  const now = new Date();
  const past = new Date(isoDate);
  const diffMs = now - past;
  const diffDays = Math.floor(diffMs / (1000 * 60 * 60 * 24));

  if (diffDays === 0) {
    return 'Today';
  } else if (diffDays === 1) {
    return 'Yesterday';
  } else {
    return `${diffDays} days ago`;
  }
};

const Notice_BoardAC = () => {
  // Move Dimensions inside component to fix Hermes runtime error
  const { width } = Dimensions.get('window');
  const TOTAL_HEADER_HEIGHT = width * 0.25;

  const navigation = useNavigation();
  const [notices, setNotices] = useState([]);
  const [loading, setLoading] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [userRole, setUserRole] = useState('');
  const [currentUserId, setCurrentUserId] = useState('');
  const [allNotices, setAllNotices] = useState([]);
  const [page, setPage] = useState(0);
  const [hasMore, setHasMore] = useState(true);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const noticesPerPage = 20;

  // Fetch user UID, role and all notices from MongoDB
  const fetchNotices = useCallback(async () => {
    setLoading(true);
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        showToast('error', 'Error', 'No user logged in. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }
      setCurrentUserId(uid);

      console.log('📋 Fetching user data and notices from MongoDB...');

      // Get user role from MongoDB
      const userResponse = await MongoAPI.getUserById(uid);
      if (userResponse && userResponse.success && userResponse.data) {
        setUserRole(userResponse.data.role || '');
      }

      // Get all notices from MongoDB
      const noticesResponse = await NoticeAPI.getAllNotices(uid);
      
      if (!noticesResponse.success || !noticesResponse.data) {
        setNotices([]);
        setAllNotices([]);
        setLoading(false);
        return;
      }

      console.log(`✅ Loaded ${noticesResponse.data.length} notices from MongoDB`);

      // Process notices
      const allNoticesData = noticesResponse.data.map((notice) => ({
        noticeId: notice.noticeId,
        title: notice.title,
        description: notice.description,
        subtitle: notice.description.substring(0, 50) + (notice.description.length > 50 ? '...' : ''),
        time: timeAgo(notice.createdAt),
        imageUrl: notice.imageUrl || null,
        createdAt: notice.createdAt,
        createdBy: notice.createdBy,
        canDelete: notice.createdBy === uid,
        isRead: notice.isRead || false,
      }));

      // Group notices by date
      const grouped = {};
      allNoticesData.forEach((notice) => {
        const date = formatDate(notice.createdAt);
        if (!grouped[date]) {
          grouped[date] = [];
        }
        grouped[date].push(notice);
      });

      // Convert to array and sort by date with custom ordering
      const sortedNotices = Object.keys(grouped)
        .map((date) => ({
          date,
          notices: grouped[date],
        }))
        .sort((a, b) => {
          // Custom sorting: Today first, then Yesterday, then by actual date
          const today = new Date();
          const todayStr = formatDate(today.toISOString());
          const yesterday = new Date(today);
          yesterday.setDate(yesterday.getDate() - 1);
          const yesterdayStr = formatDate(yesterday.toISOString());
          
          if (a.date === todayStr) return -1; // Today first
          if (b.date === todayStr) return 1;
          if (a.date === yesterdayStr) return -1; // Yesterday second
          if (b.date === yesterdayStr) return 1;
          
          // For other dates, sort by actual date (newest first)
          return new Date(b.date) - new Date(a.date);
        });

      // Store all notices and set initial page
      setAllNotices(sortedNotices);
      setPage(1);
      setHasMore(sortedNotices.length > noticesPerPage);
      
      // Load first page
      loadPage(1, sortedNotices);
    } catch (error) {
      console.error('❌ Error fetching notices from MongoDB:', error);
      showToast('error', 'Error', 'Failed to load notices.');
    } finally {
      setLoading(false);
    }
  }, [navigation]);

  // Function to load notices by page
  const loadPage = useCallback((pageNum, notices = null) => {
    const noticesToUse = notices || allNotices;
    const startIndex = (pageNum - 1) * noticesPerPage;
    const endIndex = startIndex + noticesPerPage;
    
    const paginatedNotices = noticesToUse.slice(startIndex, endIndex);
    setNotices(paginatedNotices); // Changed from setNoticesByDate to setNotices
  }, [allNotices]);

  // Function to load more notices
  const loadMoreNotices = useCallback(async () => {
    if (isLoadingMore || !hasMore) return;
    
    setIsLoadingMore(true);
    const nextPage = page + 1;
    const startIndex = (nextPage - 1) * noticesPerPage;
    const endIndex = startIndex + noticesPerPage;
    
    const nextPageNotices = allNotices.slice(startIndex, endIndex);
    
    if (nextPageNotices.length > 0) {
      setNotices(prev => [...prev, ...nextPageNotices]); // Changed from setNoticesByDate to setNotices
      setPage(nextPage);
      setHasMore(endIndex < allNotices.length);
    } else {
      setHasMore(false);
    }
    
    setIsLoadingMore(false);
  }, [page, hasMore, isLoadingMore, allNotices]);

  // Function to handle scroll end
  const handleScrollEnd = useCallback(() => {
    if (hasMore && !isLoadingMore) {
      loadMoreNotices();
    }
  }, [hasMore, isLoadingMore, loadMoreNotices]);

  useEffect(() => {
    fetchNotices();
    
    // Real-time polling for new notices (every 5 seconds)
    const noticeInterval = setInterval(() => {
      if (currentUserId) {
        fetchNotices();
      }
    }, 5000); // Every 5 seconds for real-time updates
    
    return () => {
      clearInterval(noticeInterval);
    };
  }, [fetchNotices, currentUserId]);

  const handleAddNotice = () => {
    navigation.navigate('Add_NoticeONNB');
  };

  const handleViewNotice = (notice) => {
    // setDropdownVisible(null); // Close any open dropdown - This line was removed as per the edit hint
    navigation.navigate('ViewNotificationsadded', {
      title: notice.title,
      description: notice.description,
      imageUrl: notice.imageUrl,
      createdAt: notice.createdAt,
    });
  };

  const handleDeleteNotice = (noticeId, createdBy) => {
    Alert.alert(
      'Delete Notice',
      'Are you sure you want to delete this notice? This action cannot be undone.',
      [
        {
          text: 'Cancel',
          style: 'cancel',
        },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            setLoading(true);
            try {
              console.log('🗑️ Deleting notice:', { noticeId, createdBy });
              
              // Delete from MongoDB
              const deleteResponse = await NoticeAPI.deleteNotice(noticeId, createdBy);
              
              if (!deleteResponse.success) {
                throw new Error(deleteResponse.error || 'Failed to delete notice');
              }

              console.log('✅ Notice deleted successfully');
              showToast('success', 'Success', 'Notice deleted successfully.');
              
              // Refresh notices
              fetchNotices();
            } catch (error) {
              console.error('❌ Error deleting notice:', error);
              showToast('error', 'Error', 'Failed to delete notice.');
            } finally {
              setLoading(false);
            }
          },
        },
      ]
    );
  };

  // const toggleDropdown = (noticeId) => { // This line was removed as per the edit hint
  //   setDropdownVisible(dropdownVisible === noticeId ? null : noticeId);
  // };

  return (
    <View style={styles.screen}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            activeOpacity={0.8}
          >
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Notice Board</Text>
          {userRole === 'ProAdmin' && (
            <TouchableOpacity
              style={styles.addButton}
              onPress={handleAddNotice}
              accessibilityRole="button"
              accessibilityLabel="Add Notice"
            >
              <MaterialIcons name="add" size={20} color="#015185" />
              <LinearGradient
                colors={['#fff', '#fff']}
                style={styles.addButtonTextGradient}
              >
                <Text style={styles.addButtonText}>Add Notice</Text>
              </LinearGradient>
            </TouchableOpacity>
          )}
        </View>
      </LinearGradient>
      <ScrollView 
        style={styles.scroll} 
        contentContainerStyle={{ paddingBottom: 20 }}
        onScroll={({ nativeEvent }) => {
          const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
          const paddingToBottom = 20;
          if (layoutMeasurement.height + contentOffset.y >= contentSize.height - paddingToBottom) {
            handleScrollEnd();
          }
        }}
        scrollEventThrottle={400}
      >
        {notices.length === 0 && !loading ? (
          <Text style={styles.noNoticesText}>No notices available.</Text>
        ) : (
          notices.map((section, idx) => (
            <View key={`${section.date}-${idx}`} style={styles.section}>
              <Text style={styles.sectionDate}>{section.date}</Text>
              {section.notices.map((notice, nidx) => (
                <TouchableOpacity
                  key={`${notice.noticeId}-${nidx}`}
                  style={styles.noticeCard}
                  onPress={() => handleViewNotice(notice)}
                  accessibilityRole="button"
                  accessibilityLabel={`View notice: ${notice.title}`}
                >
                  <View style={styles.titleRow}>
                    <Text style={styles.noticeTitle}>{notice.title}</Text>
                    {userRole === 'ProAdmin' && notice.canDelete && (
                      <TouchableOpacity
                        onPress={() => handleDeleteNotice(notice.noticeId, notice.createdBy)}
                        accessibilityRole="button"
                        accessibilityLabel={`Delete ${notice.title}`}
                      >
                        <MaterialIcons name="delete" size={20} color="#F44336" />
                      </TouchableOpacity>
                    )}
                  </View>
                  {/* {dropdownVisible === notice.noticeId && userRole === 'ProAdmin' && notice.canDelete && ( // This block was removed as per the edit hint
                    <View style={styles.dropdown}>
                      <TouchableOpacity
                        style={styles.dropdownItem}
                        onPress={() => handleDeleteNotice(notice.noticeId, notice.imageUrl)}
                        accessibilityRole="button"
                        accessibilityLabel={`Delete notice: ${notice.title}`}
                      >
                        <MaterialIcons name="delete" size={18} color="#FF0000" />
                        <Text style={styles.dropdownText}>Delete</Text>
                      </TouchableOpacity>
                    </View>
                  )} */}
                  <View style={styles.noticeRow}>
                    <Text style={styles.noticeSubtitle}>{notice.subtitle}</Text>
                    <Text style={styles.noticeTime}>{notice.time}</Text>
                  </View>
                  {notice.imageUrl && (
                    <Image
                      source={{ uri: notice.imageUrl }}
                      style={styles.noticeImage}
                      resizeMode="cover"
                    />
                  )}
                </TouchableOpacity>
              ))}
            </View>
          ))
        )}
        
        {/* Loading indicator for more notices */}
        {isLoadingMore && (
          <View style={styles.loadingMoreContainer}>
            <Text style={styles.loadingMoreText}>Loading more notices...</Text>
          </View>
        )}
        
        {/* End of list indicator */}
        {!hasMore && notices.length > 0 && (
          <View style={styles.endOfListContainer}>
            <Text style={styles.endOfListText}>No more notices to load</Text>
          </View>
        )}
      </ScrollView>
      <SyncModal visible={loading} onClose={() => setLoading(false)} />
    </View>
  );
};

const styles = StyleSheet.create({
  screen: {
    flex: 1,
    backgroundColor: '#f6f7fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingTop: Platform.OS === 'ios' ? 50 : 20,
    height: '100%',
    top:7
  },
  backButton: {
    padding: 4,
    marginRight: 8,
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
    letterSpacing: 0.5,
    flex: 1,
    textAlign: 'center',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 2,
    paddingVertical: 6,
    paddingHorizontal: 10,
    marginLeft: 8,
  },
  addButtonTextGradient: {
    marginLeft: 4,
    paddingVertical: 2,
    paddingHorizontal: 4,
    borderRadius: 4,
  },
  addButtonText: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#025383',
  },
  scroll: {
    flex: 1,
    backgroundColor: '#f6f7fa',
    paddingHorizontal: 0,
  },
  section: {
    marginBottom: 18,
    paddingHorizontal: 10,
  },
  sectionDate: {
    color: '#8ca0b3',
    fontWeight: 'bold',
    fontSize: 13,
    marginBottom: 8,
    marginTop: 10,
    letterSpacing: 0.5,
  },
  noticeCard: {
    backgroundColor: '#f4f4f4f4',
    borderRadius: 2,
    paddingVertical: 10,
    paddingHorizontal: 14,
    marginBottom: 10,
    borderWidth: 1,
    borderColor: '#ccc',
  },
  titleRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 2,
  },
  noticeTitle: {
    color: '#29486A',
    fontWeight: 'bold',
    fontSize: 15,
    flex: 1,
  },
  noticeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  noticeSubtitle: {
    color: '#8ca0b3',
    fontSize: 13,
    flex: 1,
    flexWrap: 'wrap',
  },
  noticeTime: {
    color: '#8ca0b3',
    fontSize: 13,
    marginLeft: 10,
  },
  noticeImage: {
    width: '100%',
    height: 150,
    borderRadius: 8,
    marginTop: 8,
  },
  noNoticesText: {
    color: '#29486A',
    fontSize: 16,
    textAlign: 'center',
    marginTop: 20,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dropdown: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#ccc',
    position: 'absolute',
    top: 30,
    right: 10,
    width: 120,
    zIndex: 1000,
    elevation: 5,
    shadowColor: '#000',
    shadowOpacity: 0.2,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  dropdownText: {
    color: '#FF0000',
    fontSize: 14,
    marginLeft: 8,
  },
  loadingMoreContainer: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  loadingMoreText: {
    color: '#8ca0b3',
    fontSize: 14,
    fontStyle: 'italic',
  },
  endOfListContainer: {
    paddingVertical: 20,
    alignItems: 'center',
  },
  endOfListText: {
    color: '#8ca0b3',
    fontSize: 12,
    fontStyle: 'italic',
  },
});

export default Notice_BoardAC;