import React, { useState } from 'react'
import { View, Text, TouchableOpacity, Alert, StyleSheet, ScrollView } from 'react-native'
import database from '@react-native-firebase/database'
import auth from '@react-native-firebase/auth'
import userData from './Userdatajson.json'
import drcData from './DRCJSON.json'
// Lazy-load large Zambia JSONs inside the handler to avoid heavy memory at screen load

const UserdataUp = () => {
  const [isUploading, setIsUploading] = useState(false)
  const [statusText, setStatusText] = useState('')
  const [isUpdatingPw, setIsUpdatingPw] = useState(false)
  const [isUploadingDRC, setIsUploadingDRC] = useState(false)
  const [isBulkDeleting, setIsBulkDeleting] = useState(false)
  const [isUploadingZambia, setIsUploadingZambia] = useState(false)

  const uploadAllUsers = async () => {
    if (isUploading) return
    try {
      setIsUploading(true)
      setStatusText('Preparing data...')

      const admins = userData?.admins || {}
      const adminEntries = Object.entries(admins)
      if (adminEntries.length === 0) {
        Alert.alert('No Data', 'No admins found in JSON to upload.')
        setIsUploading(false)
        return
      }

      const updates = {}
      let prepared = 0
      adminEntries.forEach(([key, value]) => {
        const employeeNumber = value?.employeeNumber || ''
        const userWithPassword = { ...value, password: employeeNumber }
        updates[`/Globalusers/admins/${key}`] = userWithPassword
        prepared += 1
      })

      setStatusText(`Uploading ${prepared} admins...`)
      await database().ref('/').update(updates)

      // Create missing Authentication users with password = employeeNumber
      let createdCount = 0
      let skippedCount = 0
      for (const [key, value] of adminEntries) {
        const employeeId = value?.employeeNumber?.trim()
        const email = (value?.email?.trim()) || (employeeId ? `${employeeId}@gmail.com` : undefined)
        const password = employeeId
        if (!email || !password) { skippedCount++; continue }
        try {
          const methods = await auth().fetchSignInMethodsForEmail(email)
          if (methods && methods.length > 0) {
            // Exists already; we cannot change other users' passwords from client SDK
            skippedCount++
            continue
          }
          try { await auth().signOut() } catch (e) {}
          const cred = await auth().createUserWithEmailAndPassword(email, password)
          if (cred?.user) {
            try { await cred.user.updateProfile({ displayName: value?.name || value?.employeeNumber || '' }) } catch (e) {}
          }
          try { await auth().signOut() } catch (e) {}
          createdCount++
          setStatusText(`Auth created: ${createdCount}, existing skipped: ${skippedCount}`)
        } catch (e) {
          skippedCount++
        }
      }

      setStatusText(`Realtime saved: ${prepared}. Auth created: ${createdCount}, existing skipped: ${skippedCount}`)
      Alert.alert('Success', `Realtime saved: ${prepared}. Auth created: ${createdCount}, existing skipped: ${skippedCount}`)
    } catch (e) {
      console.error('Upload error:', e)
      Alert.alert('Error', e?.message || 'Failed to upload users')
    } finally {
      setIsUploading(false)
    }
  }

  const uploadDRCUsers = async () => {
    if (isUploadingDRC) return
    try {
      setIsUploadingDRC(true)
      setStatusText('Preparing DRC data...')

      const admins = drcData?.admins || {}
      const adminEntries = Object.entries(admins)
      if (adminEntries.length === 0) {
        Alert.alert('No Data', 'No DRC admins found in JSON to upload.')
        setIsUploadingDRC(false)
        return
      }

      // New group path for DRC upload
      const DRC_GROUP_ID = '-OUE2Gn_DBD_XXS54ARz'

      // 1) Upload all user records under the group path and set countries list
      const updates = {}
      let prepared = 0
      const allCountriesSet = new Set()
      adminEntries.forEach(([key, value]) => {
        const employeeNumber = value?.employeeNumber || ''
        const userWithPassword = { ...value, password: employeeNumber }
        updates[`/Globalusers/admins/${DRC_GROUP_ID}/${key}`] = userWithPassword
        ;(value?.countries || []).forEach((c) => allCountriesSet.add(c))
        prepared += 1
      })
      const allCountries = Array.from(allCountriesSet)
      updates[`/Globalusers/admins/${DRC_GROUP_ID}/countries`] = allCountries.length > 0 ? allCountries : ['DRC']

      setStatusText(`Uploading ${prepared} DRC admins...`)
      await database().ref('/').update(updates)

      // Helper: wait for cloud function to write back authUid mapping
      const waitForAuthUid = async (userId, timeoutMs = 20000) => {
        const start = Date.now()
        const path = `/Globalusers/admins/${userId}/authUid`
        while (Date.now() - start < timeoutMs) {
          const snap = await database().ref(path).once('value')
          const uid = snap.val()
          if (uid) return uid
          // brief delay
          await new Promise((r) => setTimeout(r, 600))
        }
        return null
      }

      // 2) For each admin, trigger server-side Auth user creation, then mirror data under /Globalusers/admins/{uid}
      let createdCount = 0
      let existsCount = 0
      let skippedCount = 0
      for (const [key, value] of adminEntries) {
        const email = value?.email?.trim()
        const employeeNumber = value?.employeeNumber?.trim()
        const finalEmail = email || (employeeNumber ? `${employeeNumber}@gmail.com` : '')
        if (!finalEmail || !employeeNumber) { skippedCount++; continue }

        // push admin task for secure server-side user creation
        const reqRef = database().ref('/AdminTasks/authUserCreates').push()
        await reqRef.set({ email: finalEmail, password: employeeNumber, userId: key, requestedAt: new Date().toISOString() })

        // wait for mapping to be written back by Cloud Function
        const uid = await waitForAuthUid(key, 25000)
        if (!uid) { skippedCount++; continue }

        // mirror user data under uid path
        const userWithPassword = { ...value, email: finalEmail, password: employeeNumber, sourceKey: key }
        await database().ref(`/Globalusers/admins/${uid}`).update(userWithPassword)

        // read status to decide created vs exists (optional)
        const statusSnap = await database().ref(`/Globalusers/admins/${key}/authCreate/status`).once('value').catch(() => null)
        const status = statusSnap?.val()
        if (status === 'created') createdCount++
        else existsCount++

        setStatusText(`DRC users processed - created: ${createdCount}, existing: ${existsCount}, skipped: ${skippedCount}`)
      }

      setStatusText(`DRC upload done. Realtime saved: ${prepared}. Created: ${createdCount}, existing: ${existsCount}, skipped: ${skippedCount}`)
      Alert.alert('DRC Success', `Realtime saved: ${prepared}. Created: ${createdCount}, existing: ${existsCount}, skipped: ${skippedCount}`)
    } catch (e) {
      console.error('DRC Upload error:', e)
      Alert.alert('DRC Error', e?.message || 'Failed to upload DRC users')
    } finally {
      setIsUploadingDRC(false)
    }
  }

  const updateRealtimePasswords = async () => {
    if (isUpdatingPw) return
    try {
      setIsUpdatingPw(true)
      setStatusText('Updating passwords in Realtime...')
      const admins = userData?.admins || {}
      const adminEntries = Object.entries(admins)
      if (adminEntries.length === 0) {
        Alert.alert('No Data', 'No admins found in JSON to update.')
        setIsUpdatingPw(false)
        return
      }
      const pwUpdates = {}
      let count = 0
      for (const [key, value] of adminEntries) {
        const password = value?.employeeNumber || ''
        pwUpdates[`/Globalusers/admins/${key}/password`] = password
        count++
      }
      await database().ref('/').update(pwUpdates)
      setStatusText(`Updated password field for ${count} admins in Realtime`)
      Alert.alert('Done', `Updated password field for ${count} admins in Realtime`)
    } catch (e) {
      console.error('Update pw error:', e)
      Alert.alert('Error', e?.message || 'Failed to update passwords')
    } finally {
      setIsUpdatingPw(false)
    }
  }

  const bulkDeleteExceptKeepers = async () => {
    if (isBulkDeleting || isUploading || isUpdatingPw || isUploadingDRC) return
    try {
      setIsBulkDeleting(true)
      setStatusText('Queuing bulk delete...')
      const keepEmails = ['2@gmail.com', '5@gmail.com']
      const reqRef = database().ref('/AdminTasks/bulkDeleteUsers').push()
      const reqId = reqRef.key
      await reqRef.set({ keepEmails, requestedAt: new Date().toISOString() })

      setStatusText('Bulk delete queued. Waiting for completion...')
      const ref = database().ref(`/AdminTasks/bulkDeleteUsers/${reqId}`)
      const timeoutId = setTimeout(() => {
        try { ref.off() } catch (e) {}
        Alert.alert('Queued', 'Bulk delete will continue in the background.')
      }, 30000)
      ref.on('value', (snap) => {
        const v = snap.val()
        if (!v || !v.status || v.status === 'queued') return
        clearTimeout(timeoutId)
        try { ref.off() } catch (e) {}
        if (v.status === 'done') {
          const authCount = v.summary?.authDeletedCount || 0
          const rtdbCount = v.summary?.rtdbDeletedCount || 0
          setStatusText(`Bulk delete completed. Auth deleted: ${authCount}, RTDB deleted: ${rtdbCount}`)
          Alert.alert('Completed', `Auth deleted: ${authCount}\nRTDB deleted: ${rtdbCount}`)
        } else {
          setStatusText('Bulk delete failed.')
          Alert.alert('Error', v.error || 'Bulk delete task failed')
        }
      })
    } catch (e) {
      console.error('Bulk delete error:', e)
      Alert.alert('Error', e?.message || 'Failed to queue bulk delete')
    } finally {
      setIsBulkDeleting(false)
    }
  }

  const uploadZambiaUsers = async () => {
    if (isUploadingZambia) return
    try {
      setIsUploadingZambia(true)
      setStatusText('Preparing Zambia data...')

      // Merge admins from all Zambia project files (lazy require)
      const zambiaKalumbila = require('../zambijs/Kalumbila-zambia.json')
      const zambiaKansanshi = require('../zambijs/Kansanshi - zambia.json')
      const zambiaKimiteto = require('../zambijs/kimiteto_employees Zambia.json')
      const zambiaKobold = require('../zambijs/Kobold-zambia.json')
      const zambiaMimosa = require('../zambijs/mimosa_employees.json')

      const sources = [
        zambiaKalumbila?.admins || {},
        zambiaKansanshi?.admins || {},
        zambiaKimiteto?.admins || {},
        zambiaKobold?.admins || {},
        zambiaMimosa?.admins || {},
      ]
      const totalAdmins = sources.reduce((sum, s) => sum + Object.keys(s).length, 0)
      if (totalAdmins === 0) {
        Alert.alert('No Data', 'No Zambia admins found in JSON to upload.')
        setIsUploadingZambia(false)
        return
      }

      // Group path for Zambia bulk source copy
      const ZAMBIA_GROUP_ID = '-ZAMBIA_GROUP_001'

      // 1) Upload each user record under the group path and set countries list incrementally
      let prepared = 0
      const allCountriesSet = new Set()
      for (const src of sources) {
        const entries = Object.entries(src)
        for (let i = 0; i < entries.length; i++) {
          const [key, value] = entries[i]
          const employeeNumber = value?.employeeNumber || ''
          const userWithPassword = { ...value, password: employeeNumber }
          ;(value?.countries || []).forEach((c) => allCountriesSet.add(c))
          await database().ref(`/Globalusers/admins/${ZAMBIA_GROUP_ID}/${key}`).set(userWithPassword)
          prepared += 1
          if (prepared % 25 === 0) setStatusText(`Uploading Zambia admins: ${prepared}/${totalAdmins}...`)
        }
      }
      const allCountries = Array.from(allCountriesSet)
      await database().ref(`/Globalusers/admins/${ZAMBIA_GROUP_ID}/countries`).set(allCountries.length > 0 ? allCountries : ['Zambia'])
      setStatusText(`Uploaded Zambia admins source copy: ${prepared}/${totalAdmins}`)

      // Helper: wait for cloud function to write back authUid mapping
      const waitForAuthUid = async (userId, timeoutMs = 20000) => {
        const start = Date.now()
        const path = `/Globalusers/admins/${userId}/authUid`
        while (Date.now() - start < timeoutMs) {
          const snap = await database().ref(path).once('value')
          const uid = snap.val()
          if (uid) return uid
          await new Promise((r) => setTimeout(r, 600))
        }
        return null
      }

      // 2) Trigger server-side Auth user creation, then mirror data under /Globalusers/admins/{uid}
      let createdCount = 0
      let existsCount = 0
      let skippedCount = 0
      const projectStats = {}
      const getProject = (val) => Array.isArray(val?.projects) && val.projects.length > 0 ? String(val.projects[0]) : 'Unknown'

      let processed = 0
      for (const src of sources) {
        const entries = Object.entries(src)
        for (let i = 0; i < entries.length; i++) {
          const [key, valueRaw] = entries[i]
          // Ensure password equals employeeNumber for all Zambia users
          const value = { ...valueRaw }
          value.password = value.employeeNumber || ''

          const email = value?.email?.trim()
          const employeeNumber = value?.employeeNumber?.trim()
          const finalEmail = email || (employeeNumber ? `${employeeNumber}@gmail.com` : '')
          const project = getProject(value)
          if (!projectStats[project]) projectStats[project] = { processed: 0, created: 0, exists: 0, skipped: 0 }
          projectStats[project].processed += 1
          if (!finalEmail || !employeeNumber) { skippedCount++; projectStats[project].skipped += 1; continue }

          const reqRef = database().ref('/AdminTasks/authUserCreates').push()
          await reqRef.set({ email: finalEmail, password: employeeNumber, userId: key, requestedAt: new Date().toISOString() })

          const uid = await waitForAuthUid(key, 25000)
          if (!uid) { skippedCount++; projectStats[project].skipped += 1; continue }

          const userWithPassword2 = { ...value, email: finalEmail, password: employeeNumber, sourceKey: key }
          await database().ref(`/Globalusers/admins/${uid}`).update(userWithPassword2)

          const statusSnap = await database().ref(`/Globalusers/admins/${key}/authCreate/status`).once('value').catch(() => null)
          const status = statusSnap?.val()
          if (status === 'created') { createdCount++; projectStats[project].created += 1 }
          else { existsCount++; projectStats[project].exists += 1 }

          const summary = Object.entries(projectStats).map(([p, s]) => `${p}: C${s.created}/E${s.exists}/S${s.skipped}/P${s.processed}`).join(' | ')
          processed += 1
          setStatusText(`Zambia ${processed}/${totalAdmins} - created: ${createdCount}, exists: ${existsCount}, skipped: ${skippedCount} | ${summary}`)

          if ((processed) % 10 === 0) {
            await new Promise((r) => setTimeout(r, 10))
          }
        }
      }

      setStatusText(`Zambia upload done. Realtime saved: ${prepared}. Created: ${createdCount}, existing: ${existsCount}, skipped: ${skippedCount}`)
      Alert.alert('Zambia Success', `Realtime saved: ${prepared}. Created: ${createdCount}, existing: ${existsCount}, skipped: ${skippedCount}`)
    } catch (e) {
      console.error('Zambia Upload error:', e)
      Alert.alert('Zambia Error', e?.message || 'Failed to upload Zambia users')
    } finally {
      setIsUploadingZambia(false)
    }
  }

  return (
    <ScrollView contentContainerStyle={styles.container}>
      <Text style={styles.title}>Upload Admin Users</Text>
      <Text style={styles.subtitle}>Passwords will be set to each user's employeeNumber</Text>
      
      <TouchableOpacity style={[styles.button, (isUploading || isUpdatingPw || isUploadingDRC) && styles.buttonDisabled]} onPress={uploadAllUsers} disabled={isUploading || isUpdatingPw || isUploadingDRC}>
        <Text style={styles.buttonText}>{isUploading ? 'Uploading...' : 'Upload (DB + create Auth if missing)'}</Text>
      </TouchableOpacity>

      <TouchableOpacity style={[styles.button, (isUploading || isUpdatingPw || isUploadingDRC) && styles.buttonDisabled, { marginTop: 12, backgroundColor: '#28a745' }]} onPress={uploadDRCUsers} disabled={isUploading || isUpdatingPw || isUploadingDRC}>
        <Text style={styles.buttonText}>{isUploadingDRC ? 'Uploading DRC...' : 'DRC Button Upload'}</Text>
      </TouchableOpacity>

      <TouchableOpacity style={[styles.button, (isUploading || isUpdatingPw || isUploadingDRC) && styles.buttonDisabled, { marginTop: 12 }]} onPress={updateRealtimePasswords} disabled={isUploading || isUpdatingPw || isUploadingDRC}>
        <Text style={styles.buttonText}>{isUpdatingPw ? 'Updating...' : 'Update employee password (Realtime only)'}</Text>
      </TouchableOpacity>

      <TouchableOpacity style={[styles.button, (isUploading || isUpdatingPw || isUploadingDRC || isUploadingZambia) && styles.buttonDisabled, { marginTop: 12, backgroundColor: '#f59e0b' }]} onPress={uploadZambiaUsers} disabled={isUploading || isUpdatingPw || isUploadingDRC || isUploadingZambia}>
        <Text style={styles.buttonText}>{isUploadingZambia ? 'Uploading Zambia...' : 'Zambia Button Upload'}</Text>
      </TouchableOpacity>

      <TouchableOpacity style={[styles.button, (isUploading || isUpdatingPw || isUploadingDRC || isBulkDeleting) && styles.buttonDisabled, { marginTop: 12, backgroundColor: '#B00020' }]} onPress={bulkDeleteExceptKeepers} disabled={isUploading || isUpdatingPw || isUploadingDRC || isBulkDeleting}>
        <Text style={styles.buttonText}>{isBulkDeleting ? 'Deleting...' : 'Delete ALL users except 2@gmail.com, 5@gmail.com'}</Text>
      </TouchableOpacity>

      {!!statusText && <Text style={styles.status}>{statusText}</Text>}
    </ScrollView>
  )
}

const styles = StyleSheet.create({
  container: { flexGrow: 1, justifyContent: 'center', alignItems: 'center', padding: 16, backgroundColor: '#fff' },
  title: { fontSize: 20, fontWeight: '600', color: '#015185', marginBottom: 6 },
  subtitle: { fontSize: 14, color: '#666', marginBottom: 16, textAlign: 'center' },
  button: { backgroundColor: '#0078D4', paddingVertical: 12, paddingHorizontal: 20, borderRadius: 6 },
  buttonDisabled: { backgroundColor: '#7fb6e6' },
  buttonText: { color: '#fff', fontSize: 16, fontWeight: '600' },
  status: { marginTop: 12, fontSize: 14, color: '#333' }
})

export default UserdataUp