
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Modal,
  ScrollView,
  Platform,
  Image,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import DateTimePicker from '@react-native-community/datetimepicker';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import NetInfo from '@react-native-community/netinfo';
import { Linking } from 'react-native';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width, height } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing', progress = 0 }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
          </View>
          <Text style={styles.syncModalTextadd}>{message}</Text>
          {progress > 0 && (
            <View style={styles.progressContainer}>
              <View style={styles.progressBar}>
                <View style={[styles.progressFill, { width: `${progress}%` }]} />
              </View>
              <Text style={styles.progressText}>{progress}%</Text>
            </View>
          )}
        </View>
      </View>
    </Modal>
  );
};

const HSE_124Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    fleetNumber: '',
    dateServiceStarted: '',
    dateServiceEnded: '',
    engineHours: '',
  });
  const [selectedPersonnel, setSelectedPersonnel] = useState([]);
  const [personnelData, setPersonnelData] = useState({});
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [activeDateField, setActiveDateField] = useState('');
  const [showPersonnelModal, setShowPersonnelModal] = useState(false);
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  const [tempDateValue, setTempDateValue] = useState(new Date());
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [reportId, setReportId] = useState(null);
  const isMounted = useRef(true);
  
  // New state variables for step1 and step2 data
  const [step1Data, setStep1Data] = useState({});
  const [step2Data, setStep2Data] = useState({});

  const personnelOptions = [
    { key: 'mechanic', label: 'Mechanic' },
    { key: 'hydraulicTech', label: 'Hydraulic Tech' },
    { key: 'boilerMaker', label: 'Boiler Maker' },
    { key: 'autoElectrician', label: 'Auto Electrician' },
    { key: 'operator', label: 'Operator' },
    { key: 'supervisor', label: 'Supervisor' },
  ];

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  // Load data from step1 and step2
  useEffect(() => {
    isMounted.current = true;
    loadStepData();
    return () => {
      isMounted.current = false;
    };
  }, []);

  const loadStepData = async () => {
    try {
      // Load step1 data
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      if (savedStep1) {
        const parsedStep1 = JSON.parse(savedStep1);
        setStep1Data(parsedStep1);
      }

      // Load step2 data
      const savedStep2 = await AsyncStorage.getItem('hse124Step2');
      if (savedStep2) {
        const parsedStep2 = JSON.parse(savedStep2);
        setStep2Data(parsedStep2);
      }
    } catch (error) {
      console.error('Error loading step data:', error);
    }
  };

  const refreshAllData = async () => {
    await loadStepData();
    // Also refresh any other data that might be needed
    await loadSavedData();
  };

  const isValidSignatureUri = (uri) => uri && typeof uri === 'string' && uri.startsWith('data:image');

  const handleSignaturePress = (personnelKey) => {
    navigation.navigate('HSE41_42_signature', {
      role: personnelData[personnelKey]?.label || personnelKey,
      onSave: (signature) => {
        setPersonnelData(prev => ({
          ...prev,
          [personnelKey]: {
            ...prev[personnelKey],
            signature: signature
          }
        }));
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleClearSignature = (personnelKey) => {
    setPersonnelData(prev => ({
      ...prev,
      [personnelKey]: {
        ...prev[personnelKey],
        signature: null
      }
    }));
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = (personnelKey) => {
    console.error(`${personnelKey} signature image failed to load`);
    Toast.show({ 
      type: 'error', 
      text1: 'Error', 
      text2: `Failed to load ${personnelData[personnelKey]?.label || personnelKey} signature. Please try again.`, 
      position: 'bottom' 
    });
    handleClearSignature(personnelKey);
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate && activeDateField) {
      setSelectedDate(selectedDate);
      setFormData(prev => ({ 
        ...prev, 
        [activeDateField]: selectedDate.toLocaleDateString() 
      }));
    }
  };

  const showDatePickerModal = (field) => {
    setActiveDateField(field);
    setShowDatePicker(true);
  };

  const handlePersonnelSelection = (personnelKey, isSelected) => {
    if (isSelected) {
      setSelectedPersonnel(prev => [...prev, personnelKey]);
      setPersonnelData(prev => ({
        ...prev,
        [personnelKey]: {
          key: personnelKey,
          label: personnelOptions.find(p => p.key === personnelKey)?.label || personnelKey,
          name: '',
          signature: ''
        }
      }));
    } else {
      setSelectedPersonnel(prev => prev.filter(key => key !== personnelKey));
      setPersonnelData(prev => {
        const newData = { ...prev };
        delete newData[personnelKey];
        return newData;
      });
    }
  };

  const handlePersonnelNameChange = (personnelKey, name) => {
    setPersonnelData(prev => ({
      ...prev,
      [personnelKey]: {
        ...prev[personnelKey],
        name: name
      }
    }));
  };

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse124Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData.formData || {});
        setSelectedPersonnel(parsedData.selectedPersonnel || []);
        setPersonnelData(parsedData.personnelData || {});
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        formData,
        selectedPersonnel,
        personnelData,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse124Step3', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData();
  };

  const getAdminUserId = async () => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      return userId;
    } catch (error) {
      console.error('Error getting userId:', error);
      return null;
    }
  };

  const uploadPDFToMongoDB = async (filePath, userId) => {
    try {
      console.log('📤 Uploading PDF to MongoDB backend...');
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_124_${userId}_${Date.now()}.pdf`
      };

      const response = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      console.log('📥 Upload response:', response);

      if (!response.success) {
        throw new Error(response.message || 'Failed to upload PDF to MongoDB');
      }

      console.log('✅ PDF uploaded successfully:', response.downloadUrl);
      return response.downloadUrl;
    } catch (error) {
      console.error('❌ Error uploading PDF to MongoDB:', error);
      throw error;
    }
  };

  const generatePdf = async () => {
    try {
      setSyncModalVisible(true);
      setUploadProgress(20);
      
      const userId = await getAdminUserId();
      if (!userId) {
        throw new Error('User ID not found');
      }

      const newReportId = step1Data.requestNumber || `HSE-124-${Date.now()}`;
      setReportId(newReportId);

      // Generate PDF
      console.log('📄 Generating PDF for HSE-124...');
      const htmlContent = generatePDFHTML();
      
      const options = {
        html: htmlContent,
        fileName: `HSE_124_${formData.fleetNumber || 'Report'}_${new Date().toISOString().split('T')[0]}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      console.log('✅ PDF generated at:', file.filePath);
      setUploadProgress(50);

      // Upload PDF to MongoDB
      console.log('📤 Uploading PDF to MongoDB...');
      const pdfDownloadUrl = await uploadPDFToMongoDB(file.filePath, userId);
      console.log('✅ PDF uploaded:', pdfDownloadUrl);
      setUploadProgress(70);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      const completeReportData = {
        _id: newReportId,
        userId,
        requestNumber: newReportId,
        status: 'completed',
        documentType: 'HSE-124',
        documentName: 'Workshop RC Rig Check Sheet',
        step1: step1Data,
        step2: step2Data,
        step3: {
          formData,
          selectedPersonnel,
          personnelData
        },
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId
      };

      console.log('📋 Report data to save:', JSON.stringify(completeReportData, null, 2));
      
      let saveResponse;
      try {
        saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
        console.log('📥 Save response:', saveResponse);
      } catch (saveError) {
        console.error('❌ Save error details:', saveError);
        throw new Error(saveError.message || 'Failed to save report to MongoDB');
      }

      if (!saveResponse.success) {
        console.error('❌ MongoDB save failed:', saveResponse.message || saveResponse.error);
        throw new Error(saveResponse.message || 'Failed to save report to MongoDB');
      }
      console.log('✅ Report saved to MongoDB');
      setUploadProgress(90);

      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const notificationPayload = {
          hseInspectionData: {
            _id: newReportId,
            id: newReportId,
            requestNumber: newReportId,
            checklistName: 'HSE-124 Workshop RC Rig Check Sheet',
            equipmentName: step1Data.selectedEquipment?.equipmentName || '',
            country: step1Data.country || '',
            project: step1Data.project || '',
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('http://168.231.113.219:5028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }

      // Delete temp PDF file
      await RNFS.unlink(file.filePath).catch(err => console.warn('⚠️ Error deleting temp PDF:', err));

      setUploadProgress(100);
      
      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        setGeneratedPdfPath(pdfDownloadUrl);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Report generated and saved successfully.',
          position: 'bottom',
        });
      }
    } catch (error) {
      console.error('❌ Error in generatePdf:', error);
      console.error('❌ Error message:', error.message);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to generate report. Please try again.',
          position: 'bottom',
        });
      }
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const handleBackToHome = () => {
    setShowSuccessModal(false);
    navigation.navigate('HSEInspectionHome');
  };

  const handleViewPDF = async () => {
    if (!reportId) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Report ID not available.' });
      return;
    }

    try {
      const response = await HSEInspectionAPI.getHSEInspectionById(reportId);

      if (!response.success || !response.data) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'Report not found in the database.' });
        return;
      }

      const pdfUrl = response.data.pdfDownloadUrl;
      if (!pdfUrl) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'PDF URL not available in the report.' });
        return;
      }

      setShowSuccessModal(false);
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
        }
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Could not fetch the PDF URL. Please try again.' });
    }
  };

  const generatePDFHTML = () => {
    // Generate personnel rows for the PDF
    let personnelRows = '';
    selectedPersonnel.forEach(key => {
      const personnel = personnelData[key];
      if (personnel) {
        personnelRows += `
          <tr class="info-row">
            <td><strong>${personnel.label}:</strong></td>
            <td><strong>Signature:</strong></td>
            <td>${personnel.name || ''}</td>
            <td>${personnel.signature ? `<img src="${personnel.signature}" alt="${personnel.label} Signature" style="max-width: 80px; max-height: 40px; border: 1px solid #015185; border-radius: 4px;" />` : ''}</td>
          </tr>
        `;
      }
    });

    // Generate checklist items with their selections and notes
    let checklistRows = '';
    if (step2Data.selections) {
      const sections = [
        {
          key: 'donkey_engine',
          name: 'DONKEY ENGINE',
          items: [
            { key: 'engineOil', name: 'ENGINE OIL (CHANGED/LEVEL)' },
            { key: 'coolant', name: 'COOLANT (LEAKS/LEVEL)' },
            { key: 'airFilter1', name: 'AIR FILTER #1 (CLEANED HOUSING/CHANGED)' },
            { key: 'airFilter2', name: 'AIR FILTER #2 (CLEANED HOUSING/ROTATED)' },
            { key: 'fuelFilter', name: 'FUEL FILTER (CHANGED)' },
            { key: 'fuelSystem', name: 'FUEL SYSTEM (LEAKS)' },
            { key: 'vBelts', name: 'V BELTS (CONDITION/TENSION)' },
            { key: 'radiator', name: 'RADIATOR (CLEANED/ PRESSURE TESTED)' },
            { key: 'radiatorCap', name: 'RADIATOR CAP (INSPECTED)' },
            { key: 'engineMountings', name: 'ENGINE MOUNTINGS (WORN/SECURED)' },
          ]
        },
        {
          key: 'every_200_hrs',
          name: 'EVERY 2000 HRS',
          items: [
            { key: 'valveClearance', name: 'VALVE CLEARANCE (ADJUSTED/CHECKED)' },
          ]
        },
        {
          key: 'electrical',
          name: 'ELECTRICAL',
          items: [
            { key: 'batteries', name: 'BATTERIES (POLE CONDITION/ JELLED)' },
            { key: 'dateEngraved', name: 'DATE ENGRAVED' },
            { 
              key: 'voltageAtCrankingDonkey', 
              name: 'VOLTAGE AT CRANKING (DONKEY)',
              inputType: 'VoltageInput',
              positions: ['V RUNNING', 'V NOT RUNNING']
            },
            { 
              key: 'voltageAtCrankingTruck', 
              name: 'VOLTAGE AT CRANKING (TRUCK)',
              inputType: 'VoltageInput',
              positions: ['1', '2', 'COMBINED']
            },
            { key: 'alternator', name: 'ALTERNATOR (MOUNTING/CHECKED)' },
            { key: 'starter', name: 'STARTER (MOUNTING/TESTED)' },
            { key: 'lights', name: 'LIGHTS (WORKING/WIRING)' },
            { 
              key: 'chargingSystemDonke', 
              name: 'CHARGING SYSTEM DONKEY (CHARGING/WIRES/FUSES)',
              inputType: 'VoltageInput',
              positions: ['V RUNNING', 'V NOT RUNNING']
            },
            { 
              key: 'chargingSystemTruck', 
              name: 'CHARGING SYSTEM TRUCK (CHARGING/WIRES/FUSE)',
              inputType: 'VoltageInput',
              positions: ['V RUNNING', 'V NOT RUNNING']
            },
            { key: 'rpmVoltTempOilGauge', name: 'RPM/VOLT/TEMP/OIL GAUGE (WORKING/CALIBRATED)' },
            { key: 'eStops', name: 'E STOPS (CHECKED / TESTED)' },
            { 
              key: 'engineProtectionCutO', 
              name: 'ENGINE PROTECTION CUT OUT (OIL PRESSURE TEST)' 
            },
            { 
              key: 'engineProtectionTemp', 
              name: 'TEMPERATURE TEST' 
            },
            { key: 'horn', name: 'HORN (CHECKED / TESTED)' },
          ]
        },
        {
          key: 'marst',
          name: 'MARST',
          items: [
            { key: 'keeperSpannerRam', name: 'KEEPER SPANNER/RAM (CONDITION/PINS/LEAKS)' },
            { 
              key: 'breakOutRam', 
              name: 'BREAK OUT RAM (PINS/LEAKS)',
              inputType: 'RamPositions',
              positions: ['RAM 1', 'RAM 2', 'RAM 3']
            },
            { key: 'breakOutRotationMoto', name: 'BREAK OUT ROTATION MOTOR (SECURED /LEAKS)' },
            { key: 'traverseRam', name: 'TRAVERSE RAM (LEAKS/PINS/WHEELS)' },
            { key: 'marstRaisingRams', name: 'MARST RAISING RAMS (LEAKS/PINS/BUSHES/PRES HOLD)' },
            { key: 'headTiltRam', name: 'HEAD TILT RAM (LEAKS/PINS/BUSHES/PRES HOLD)' },
            { key: 'hoseArmSwivel', name: '3" HOSE ARM/SWIVEL (SECURED/INSPECTED)' },
            { key: 'tableHalfMoons', name: 'TABLE HALF MOONS (CHANGED/CONDITION)' },
            { key: 'tableDeckBush', name: 'TABLE DECK BUSH (CHANGED/CONDITION)' },
            { key: 'greaseNipples', name: 'GREASE NIPPLES (GREASED)' },
          ]
        },
        {
          key: 'drill_head',
          name: 'DRILL HEAD',
          items: [
            { key: 'gearbox', name: 'GEARBOX (PACKED WITH GREASE)' },
            { key: 'rotationMotors', name: 'ROTATION MOTORS (SECURE/LEAKS)' },
            { key: 'wearCollar', name: 'WEAR COLLAR (INSPECTED/CHANGED)' },
            { key: 'bullshaft', name: 'BULLSHAFT (PLAY/WEAR)' },
            { key: 'subsaver', name: 'SUBSAVER (CONDITION)' },
            { key: 'headInnertube', name: 'HEAD INNERTUBE (PLAY/WEAR)' },
            { key: 'deflectorBox', name: 'DEFLECTOR BOX (WEAR/BLOCKED)' },
            { key: 'cycloneHoses', name: 'CYCLONE HOSES (CONDITION)' },
            { key: 'airHoses', name: '2" AIR HOSES (CONDITION)' },
            { key: 'safetySocks', name: 'SAFETY SOCKS (CONDITION CLIPS)' },
          ]
        },
        {
          key: 'control_panel',
          name: 'CONTROL PANEL',
          items: [
            { key: 'gauges', name: 'GAUGES (LEAKS/CALIBRATED)' },
            { key: 'levers', name: 'LEVERS (STIFF/PLAY/LEAKS)' },
            { key: 'dripTray', name: 'DRIP TRAY (PRESENT/CLEAN)' },
            { key: 'soakBags', name: 'SOAK BAGS (PRESENT/CLEAN)' },
            { key: 'quickCouplers', name: 'QUICK COUPLERS (CLEAN/LEAKS/INSPECTED)' },
            { key: 'jacklegs', name: 'JACKLEGS (PRESS HOLD/LEAKS)' },
            { key: 'hydraulicHoses', name: 'HYDRAULIC HOSES (LEAKS/INSPECTED)' },
            { key: 'fireExtinguisher', name: 'FIRE EXTINGUISHER (INSPECT/DATE OF EXPIRY)' },
          ]
        }
      ];

      sections.forEach(section => {
        checklistRows += `
          <tr class="section-header">
            <td colspan="4">${section.name}</td>
          </tr>
        `;
        
        section.items.forEach(item => {
          const selection = step2Data.selections[item.key];
          const note = step2Data.notes[item.key];
          let yesChecked = '';
          let noChecked = '';
          let comments = note || '';

          // Handle different input types
          if (item.inputType === 'VoltageInput') {
            // For voltage inputs, show the actual voltage values
            const voltageValues = [];
            if (item.positions) {
              item.positions.forEach(position => {
                const voltageKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                const voltageValue = step2Data.selections[voltageKey];
                if (voltageValue) {
                  voltageValues.push(`${position}: ${voltageValue}V`);
                }
              });
            }
            if (voltageValues.length > 0) {
              comments = voltageValues.join(' | ');
            }
            // For voltage inputs, we still show the main selection
            yesChecked = selection === 'YES/OK' ? '✓' : '';
            noChecked = selection === 'NO/DEF' ? '✓' : '';
          } else if (item.inputType === 'RamPositions') {
            // For RAM positions, show the detailed status for each RAM
            const ramStatuses = [];
            if (item.positions) {
              item.positions.forEach(position => {
                const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                const positionValue = step2Data.selections[positionKey];
                if (positionValue) {
                  ramStatuses.push(`${position}: ${positionValue}`);
                }
              });
            }
            if (ramStatuses.length > 0) {
              comments = ramStatuses.join(' | ');
            }
            // For RAM positions, we still show the main selection
            yesChecked = selection === 'YES/OK' ? '✓' : '';
            noChecked = selection === 'NO/DEF' ? '✓' : '';
          } else {
            // Standard Yes/No items
            yesChecked = selection === 'YES/OK' ? '✓' : '';
            noChecked = selection === 'NO/DEF' ? '✓' : '';
          }
          
          checklistRows += `
            <tr>
              <td>${item.name}</td>
              <td style="text-align: center;">${yesChecked}</td>
              <td style="text-align: center;">${noChecked}</td>
              <td>${comments}</td>
            </tr>
          `;
        });
      });
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE-FOR-8-1.1 Workshop RC Rig Check Sheet</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 5px;
                  padding: 0;
                  font-size: 14px;
              }
              h1 {
                  text-align: center;
                  margin-bottom: 10px;
                  font-size: 1.5em;
              }
              .header-container {
                  display: flex;
                  flex-wrap: wrap;
                  justify-content: space-between;
                  align-items: flex-start;
                  margin-bottom: 15px;
              }
              .header-table {
                  border-collapse: collapse;
                  width: 100%;
                  max-width: 250px;
              }
              .header-table th, .header-table td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .header-table th {
                  background-color: #f2f2f2;
              }
              h2 {
                  text-align: center;
                  font-size: 1.2em;
                  margin: 15px 0;
              }
              table.checklist {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 15px;
              }
              .checklist th, .checklist td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
                  word-wrap: break-word;
              }
              .checklist th {
                  background-color: #f2f2f2;
              }
              .section-header td {
                  background-color: #e0e0e0;
                  font-weight: bold;
                  text-align: center;
              }
              .info-row td {
                  padding: 6px;
              }
              .sub-comment {
                  margin: 4px 0;
                  padding: 4px;
                  border-top: 1px solid #eee;
                  font-size: 0.85em;
              }
              .checklist th:nth-child(1), .checklist td:nth-child(1) {
                  width: 50%;
              }
              .checklist th:nth-child(2), .checklist td:nth-child(2) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(3), .checklist td:nth-child(3) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(4), .checklist td:nth-child(4) {
                  width: 20%;
              }
              .footer-table {
                  width: 100%;
                  border-collapse: collapse;
                  margin-top: 15px;
              }
              .footer-table td {
                  border: 1px solid #ddd;
                  padding: 8px;
                  font-size: 0.9em;
              }
              .step1-info {
                  background-color: #f8f9fa;
                  padding: 10px;
                  border-radius: 5px;
                  margin-bottom: 15px;
                  border: 1px solid #e9ecef;
              }
              .step1-info h3 {
                  margin: 0 0 10px 0;
                  color: #015185;
                  font-size: 16px;
              }
              .step1-info p {
                  margin: 5px 0;
                  font-size: 14px;
              }
              .detailed-data {
                  background-color: #f8f9fa;
                  padding: 8px;
                  border-radius: 4px;
                  margin-top: 5px;
                  border-left: 3px solid #015185;
              }
              .detailed-data-item {
                  margin: 2px 0;
                  font-size: 12px;
                  color: #333;
              }
              .voltage-reading {
                  background-color: #e8f4fd;
                  padding: 4px 8px;
                  border-radius: 3px;
                  margin: 2px;
                  display: inline-block;
                  font-size: 11px;
                  color: #015185;
                  font-weight: 600;
              }
              .ram-status {
                  background-color: #fff3cd;
                  padding: 4px 8px;
                  border-radius: 3px;
                  margin: 2px;
                  display: inline-block;
                  font-size: 11px;
                  color: #856404;
                  font-weight: 600;
              }
              .status-ok {
                  background-color: #d4edda;
                  color: #155724;
              }
              .status-check {
                  background-color: #fff3cd;
                  color: #856404;
              }
              .status-replace {
                  background-color: #f8d7da;
                  color: #721c24;
              }
              .detailed-data h3 {
                  margin: 0 0 10px 0;
                  color: #015185;
                  font-size: 14px;
                  border-bottom: 2px solid #015185;
                  padding-bottom: 5px;
              }
              .detailed-data-item {
                  margin: 8px 0;
                  padding: 8px;
                  background-color: #fff;
                  border-radius: 4px;
                  border-left: 3px solid #015185;
              }
              .detailed-data-item strong {
                  color: #015185;
                  font-weight: 600;
              }
              .project-info-table {
                  margin-bottom: 20px;
              }
              .project-info-table .section-header td {
                  background-color: #015185;
                  color: white;
                  font-weight: bold;
                  text-align: center;
                  font-size: 16px;
              }
              .project-info-table .info-row td {
                  padding: 8px;
                  border: 1px solid #ddd;
                  vertical-align: top;
              }
              .project-info-table .info-row td:first-child {
                  background-color: #f8f9fa;
                  font-weight: 600;
                  color: #015185;
                  width: 25%;
              }
              .project-info-table .info-row td:nth-child(2),
              .project-info-table .info-row td:nth-child(4) {
                  width: 25%;
              }
              .project-info-table .info-row td:nth-child(3) {
                  width: 25%;
              }
              .project-info-table .info-row td[colspan="3"] {
                  width: 75%;
              }

              @media screen and (max-width: 600px) {
                  body {
                      margin: 2px;
                      font-size: 10px;
                  }
                  h1 {
                      font-size: 1em;
                      margin-bottom: 5px;
                  }
                  .header-container {
                      flex-direction: column;
                      align-items: center;
                  }
                  .header-table {
                      max-width: 100%;
                      margin-top: 5px;
                  }
                  .header-table th, .header-table td {
                      padding: 3px;
                      font-size: 0.7em;
                  }
                  h2 {
                      font-size: 0.9em;
                      margin: 10px 0;
                  }
                  table.checklist {
                      font-size: 0.8em;
                  }
                  .checklist th, .checklist td {
                      padding: 3px;
                      font-size: 0.7em;
                      word-wrap: break-word;
                  }
                  .checklist th:nth-child(1), .checklist td:nth-child(1) {
                      width: 50%;
                  }
                  .checklist th:nth-child(2), .checklist td:nth-child(2) {
                      width: 15%;
                  }
                  .checklist th:nth-child(3), .checklist td:nth-child(3) {
                      width: 15%;
                  }
                  .checklist th:nth-child(4), .checklist td:nth-child(4) {
                      width: 20%;
                  }
                  .info-row td {
                      padding: 4px;
                      font-size: 0.7em;
                  }
                  .sub-comment {
                      font-size: 0.6em;
                      padding: 2px;
                  }
                  .footer-table td {
                      padding: 4px;
                      font-size: 0.7em;
                  }
              }
          </style>
      </head>
      <body>
          <div class="header-container">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              <table class="header-table">
                  <tr>
                      <th colspan="2">Header Information</th>
                  </tr>
                  <tr>
                      <td><strong>Document ID:</strong></td>
                      <td>HSE-124</td>
                  </tr>
                  <tr>
                      <td><strong>Version Date:</strong></td>
                      <td>19/01/2024</td>
                  </tr>
                  <tr>
                      <td><strong>Version:</strong></td>
                      <td>HSM</td>
                  </tr>
                  <tr>
                      <td><strong>Approved By:</strong></td>
                      <td></td>
                  </tr>
              </table>
          </div>

          <!-- Step 1 Information -->
          

          <table class="checklist">
              <tr class="info-row">
                  <td><strong>Date Service Started:</strong> ${formData.dateServiceStarted || ''}</td>
                  <td><strong>Fleet Number:</strong> ${formData.fleetNumber || ''}</td>
                  <td><strong>Engine Hours (Donkey):</strong> ${formData.engineHours || ''}</td>
                  <td><strong>Serviced By:</strong></td>
              </tr>
              <tr class="info-row">
                  <td><strong>Date Service Ended:</strong> ${formData.dateServiceEnded || ''}</td>
                  <td><strong>Work Carried Out:</strong></td>
                  <td></td>
                  <td></td>
              </tr>
              ${personnelRows}
          </table>

          <!-- Project Information Table -->
          <h2>Project Information</h2>
          <table class="checklist project-info-table">
              <tr class="section-header">
                  <td colspan="4">Project Details</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Request Number:</strong></td>
                  <td colspan="3">${step1Data.requestNumber || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Country:</strong></td>
                  <td>${step1Data.country || 'N/A'}</td>
                  <td><strong>Project:</strong></td>
                  <td>${step1Data.project || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Equipment Name:</strong></td>
                  <td colspan="3">${step1Data.selectedEquipment?.equipmentName || step1Data.selectedEquipment?.equipmentNumber || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Equipment Number:</strong></td>
                  <td>${step1Data.selectedEquipment?.equipmentNumber || step1Data.selectedEquipment?.entryNumber || 'N/A'}</td>
                  <td><strong>Equipment Category:</strong></td>
                  <td>${step1Data.selectedEquipment?.mainCategory || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Equipment Model:</strong></td>
                  <td>${step1Data.selectedEquipment?.model || 'N/A'}</td>
                  <td><strong>Equipment Status:</strong></td>
                  <td>${step1Data.selectedEquipment?.status || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>GPS Coordinates:</strong></td>
                  <td colspan="3">${step1Data.coordinates || 'N/A'}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>GPS Address:</strong></td>
                  <td colspan="3">${step1Data.gpsAddress || 'N/A'}</td>
              </tr>
          </table>

          <h2>Checklist</h2>

          <table class="checklist">
              <tr>
                  <th>Task</th>
                  <th>Yes</th>
                  <th>No</th>
                  <th>Comments</th>
              </tr>
              ${checklistRows}
          </table>

        

          <table class="footer-table">
              <tr class="info-row">
                  <td><strong>Comments:</strong></td>
                  <td><strong>Next Service Due (Hours):</strong></td>
                  <td><strong>Signature:</strong></td>
              </tr>
          </table>
      </body>
      </html>
    `;
  };

  const generateVoltageSummary = () => {
    let summaryHtml = '';
    const voltageItems = [
      { key: 'voltageAtCrankingDonkey', name: 'VOLTAGE AT CRANKING (DONKEY)', positions: ['V RUNNING', 'V NOT RUNNING'] },
      { key: 'voltageAtCrankingTruck', name: 'VOLTAGE AT CRANKING (TRUCK)', positions: ['1', '2', 'COMBINED'] },
      { key: 'chargingSystemDonke', name: 'CHARGING SYSTEM DONKEY (CHARGING/WIRES/FUSES)', positions: ['V RUNNING', 'V NOT RUNNING'] },
      { key: 'chargingSystemTruck', name: 'CHARGING SYSTEM TRUCK (CHARGING/WIRES/FUSE)', positions: ['V RUNNING', 'V NOT RUNNING'] },
    ];

    voltageItems.forEach(item => {
      const voltageValues = [];
      if (item.positions) {
        item.positions.forEach(position => {
          const voltageKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
          const voltageValue = step2Data.selections[voltageKey];
          if (voltageValue) {
            voltageValues.push(`${position}: ${voltageValue}V`);
          }
        });
      }
      if (voltageValues.length > 0) {
        summaryHtml += `
          <div class="detailed-data-item">
            <strong>${item.name}:</strong> ${voltageValues.join(' | ')}
          </div>
        `;
      }
    });
    return summaryHtml;
  };

  const generateRamPositionsSummary = () => {
    let summaryHtml = '';
    const ramItems = [
      { key: 'breakOutRam', name: 'BREAK OUT RAM (PINS/LEAKS)', positions: ['RAM 1', 'RAM 2', 'RAM 3'] },
      { key: 'traverseRam', name: 'TRAVERSE RAM (LEAKS/PINS/WHEELS)' },
      { key: 'marstRaisingRams', name: 'MARST RAISING RAMS (LEAKS/PINS/BUSHES/PRES HOLD)' },
      { key: 'headTiltRam', name: 'HEAD TILT RAM (LEAKS/PINS/BUSHES/PRES HOLD)' },
    ];

    ramItems.forEach(item => {
      const ramStatuses = [];
      if (item.positions) {
        item.positions.forEach(position => {
          const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
          const positionValue = step2Data.selections[positionKey];
          if (positionValue) {
            ramStatuses.push(`${position}: ${positionValue}`);
          }
        });
      }
      if (ramStatuses.length > 0) {
        summaryHtml += `
          <div class="detailed-data-item">
            <strong>${item.name}:</strong> ${ramStatuses.join(' | ')}
          </div>
        `;
      }
    });
    return summaryHtml;
  };

  // Helper functions for data preview
  const hasVoltageData = () => {
    const voltageKeys = [
      'voltageAtCrankingDonkey_V_RUNNING', 'voltageAtCrankingDonkey_V_NOT_RUNNING',
      'voltageAtCrankingTruck_1', 'voltageAtCrankingTruck_2', 'voltageAtCrankingTruck_COMBINED',
      'chargingSystemDonke_V_RUNNING', 'chargingSystemDonke_V_NOT_RUNNING',
      'chargingSystemTruck_V_RUNNING', 'chargingSystemTruck_V_NOT_RUNNING'
    ];
    return voltageKeys.some(key => step2Data.selections[key]);
  };

  const hasRamPositionData = () => {
    const ramKeys = [
      'breakOutRam_RAM_1', 'breakOutRam_RAM_2', 'breakOutRam_RAM_3'
    ];
    return ramKeys.some(key => step2Data.selections[key]);
  };

  const renderVoltagePreview = () => {
    const voltageItems = [
      { key: 'voltageAtCrankingDonkey', name: 'Voltage at Cranking (Donkey)', positions: ['V RUNNING', 'V NOT RUNNING'] },
      { key: 'voltageAtCrankingTruck', name: 'Voltage at Cranking (Truck)', positions: ['1', '2', 'COMBINED'] },
      { key: 'chargingSystemDonke', name: 'Charging System (Donkey)', positions: ['V RUNNING', 'V NOT RUNNING'] },
      { key: 'chargingSystemTruck', name: 'Charging System (Truck)', positions: ['V RUNNING', 'V NOT RUNNING'] },
    ];

    return voltageItems.map(item => {
      const voltageValues = [];
      if (item.positions) {
        item.positions.forEach(position => {
          const voltageKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
          const voltageValue = step2Data.selections[voltageKey];
          if (voltageValue) {
            voltageValues.push(`${position}: ${voltageValue}V`);
          }
        });
      }
      
      if (voltageValues.length > 0) {
        return (
          <View key={item.key} style={styles.dataPreviewItem}>
            <Text style={styles.dataPreviewItemTitle}>{item.name}</Text>
            <Text style={styles.dataPreviewItemValue}>{voltageValues.join(' | ')}</Text>
          </View>
        );
      }
      return null;
    });
  };

  const renderRamPositionsPreview = () => {
    const ramItems = [
      { key: 'breakOutRam', name: 'Break Out RAM (PINS/LEAKS)', positions: ['RAM 1', 'RAM 2', 'RAM 3'] },
    ];

    return ramItems.map(item => {
      const ramStatuses = [];
      if (item.positions) {
        item.positions.forEach(position => {
          const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
          const positionValue = step2Data.selections[positionKey];
          if (positionValue) {
            ramStatuses.push(`${position}: ${positionValue}`);
          }
        });
      }
      
      if (ramStatuses.length > 0) {
        return (
          <View key={item.key} style={styles.dataPreviewItem}>
            <Text style={styles.dataPreviewItemTitle}>{item.name}</Text>
            <Text style={styles.dataPreviewItemValue}>{ramStatuses.join(' | ')}</Text>
          </View>
        );
      }
      return null;
    });
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  const handleSubmit = async () => {
    // Validate required fields
    if (!formData.fleetNumber || !formData.dateServiceStarted || !formData.dateServiceEnded || !formData.engineHours) {
      Alert.alert('Error', 'Please fill in all required fields');
      return;
    }

    // Validate personnel selection
    if (selectedPersonnel.length === 0) {
      Alert.alert('Error', 'Please select at least one personnel type');
      return;
    }

    // Validate personnel data
    const missingData = selectedPersonnel.filter(key => {
      const personnel = personnelData[key];
      return !personnel?.name || !personnel?.signature;
    });

    if (missingData.length > 0) {
      const missingLabels = missingData.map(key => personnelData[key]?.label || key).join(', ');
      Alert.alert('Error', `Please fill in name and signature for: ${missingLabels}`);
      return;
    }

    try {
      // Refresh all data to ensure we have the latest information
      await refreshAllData();
      
      // Save data before generating PDF
      await saveData();
      // Generate PDF immediately after successful validation
      await generatePdf();
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', 'Failed to submit form. Please try again.');
    }
  };

  const renderPersonnelSection = (personnelKey) => {
    const personnel = personnelData[personnelKey];
    if (!personnel) return null;

    return (
      <View key={personnelKey} style={styles.personnelSection}>
        <Text style={styles.personnelTitle}>{personnel.label}</Text>
        
        <View style={styles.nameField}>
          <Text style={styles.nameLabel}>Name:</Text>
          <TextInput
            style={styles.input}
            value={personnel.name}
            onChangeText={(value) => handlePersonnelNameChange(personnelKey, value)}
            placeholder="Enter name"
            placeholderTextColor="#999"
          />
        </View>

        <View style={styles.signatureField}>
          <Text style={styles.signatureLabel}>Signature:</Text>
          <TouchableOpacity
            style={styles.signatureBoxImage}
            onPress={() => handleSignaturePress(personnelKey)}
            key={signatureRefreshKey}
          >
            {isValidSignatureUri(personnel.signature) ? (
              <Image
                source={{ uri: personnel.signature }}
                style={styles.signatureImage}
                onError={() => handleImageError(personnelKey)}
              />
            ) : (
              <View style={styles.plusIconCircleContainer}>
                <View style={styles.plusCircle}>
                  <MaterialIcons name="add" size={20} color="#015185" />
                </View>
              </View>
            )}
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.pageTitle}>RC Drill Rig Checklist - Review & Submit</Text>

        {/* Refresh Data Button */}
        <View style={styles.refreshContainer}>
          <TouchableOpacity style={styles.refreshButton} onPress={refreshAllData}>
            <MaterialIcons name="refresh" size={20} color="#015185" />
            <Text style={styles.refreshButtonText}>Refresh All Data</Text>
          </TouchableOpacity>
        </View>

        {/* Data Summary */}
        <View style={styles.dataSummaryContainer}>
          <Text style={styles.dataSummaryTitle}>Data Summary</Text>
          <View style={styles.dataSummaryRow}>
            <Text style={styles.dataSummaryLabel}>Step 1 Data:</Text>
            <Text style={styles.dataSummaryValue}>
              {step1Data.country && step1Data.project ? '✓ Loaded' : '✗ Not Loaded'}
            </Text>
          </View>
          <View style={styles.dataSummaryRow}>
            <Text style={styles.dataSummaryLabel}>Step 2 Data:</Text>
            <Text style={styles.dataSummaryValue}>
              {step2Data.selections ? `✓ Loaded (${Object.keys(step2Data.selections).length} items)` : '✗ Not Loaded'}
            </Text>
          </View>
          <View style={styles.dataSummaryRow}>
            <Text style={styles.dataSummaryLabel}>Step 3 Data:</Text>
            <Text style={styles.dataSummaryValue}>
              {formData.fleetNumber ? '✓ Filled' : '✗ Not Filled'}
            </Text>
          </View>
        </View>

        {/* Project Information Preview */}
        {step1Data.country && step1Data.project && (
          <View style={styles.projectInfoPreviewContainer}>
            <Text style={styles.projectInfoPreviewTitle}>Project Information Preview</Text>
            <View style={styles.projectInfoPreviewRow}>
              <Text style={styles.projectInfoPreviewLabel}>Request Number:</Text>
              <Text style={styles.projectInfoPreviewValue}>{step1Data.requestNumber || 'N/A'}</Text>
            </View>
            <View style={styles.projectInfoPreviewRow}>
              <Text style={styles.projectInfoPreviewLabel}>Country:</Text>
              <Text style={styles.projectInfoPreviewValue}>{step1Data.country}</Text>
            </View>
            <View style={styles.projectInfoPreviewRow}>
              <Text style={styles.projectInfoPreviewLabel}>Project:</Text>
              <Text style={styles.projectInfoPreviewValue}>{step1Data.project}</Text>
            </View>
            <View style={styles.projectInfoPreviewRow}>
              <Text style={styles.projectInfoPreviewLabel}>Equipment:</Text>
              <Text style={styles.projectInfoPreviewValue}>
                {step1Data.selectedEquipment?.equipmentName || step1Data.selectedEquipment?.equipmentNumber || 'N/A'}
              </Text>
            </View>
            <View style={styles.projectInfoPreviewRow}>
              <Text style={styles.projectInfoPreviewLabel}>GPS Location:</Text>
              <Text style={styles.projectInfoPreviewValue}>
                {step1Data.coordinates ? `${step1Data.coordinates}` : 'N/A'}
              </Text>
            </View>
          </View>
        )}

        {/* Detailed Data Preview */}
        {step2Data.selections && Object.keys(step2Data.selections).length > 0 && (
          <View style={styles.detailedDataPreviewContainer}>
            <Text style={styles.detailedDataPreviewTitle}>Detailed Data Preview</Text>
            
            {/* Voltage Readings Preview */}
            {hasVoltageData() && (
              <View style={styles.dataPreviewSection}>
                <Text style={styles.dataPreviewSectionTitle}>Voltage Readings</Text>
                {renderVoltagePreview()}
              </View>
            )}
            
            {/* RAM Positions Preview */}
            {hasRamPositionData() && (
              <View style={styles.dataPreviewSection}>
                <Text style={styles.dataPreviewSectionTitle}>RAM Position Status</Text>
                {renderRamPositionsPreview()}
              </View>
            )}
          </View>
        )}

        {isLoading && (
          <View style={styles.loadingContainer}>
            <MaterialIcons name="hourglass-empty" size={40} color="#015185" />
            <Text style={styles.loadingText}>Generating PDF Report...</Text>
            <Text style={styles.loadingSubtext}>Please wait while we compile your data</Text>
          </View>
        )}

        {/* Form Section */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>FORM DETAILS</Text>
          </View>
          
          <View style={styles.formFields}>
            {/* FLEET NUMBER Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>FLEET NUMBER:</Text>
              <TextInput
                style={styles.input}
                value={formData.fleetNumber}
                onChangeText={(value) => handleInputChange('fleetNumber', value)}
                placeholder="Enter fleet number"
                placeholderTextColor="#999"
              />
            </View>

            {/* DATE SERVICE STARTED Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>DATE SERVICE STARTED:</Text>
              <TouchableOpacity style={styles.pickerButton} onPress={() => showDatePickerModal('dateServiceStarted')}>
                <Text style={styles.pickerButtonText}>
                  {formData.dateServiceStarted || 'Select Date'}
                </Text>
                <MaterialIcons name="calendar-today" size={20} color="#015185" />
              </TouchableOpacity>
            </View>

            {/* DATE SERVICE ENDED Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>DATE SERVICE ENDED:</Text>
              <TouchableOpacity style={styles.pickerButton} onPress={() => showDatePickerModal('dateServiceEnded')}>
                <Text style={styles.pickerButtonText}>
                  {formData.dateServiceEnded || 'Select Date'}
                </Text>
                <MaterialIcons name="calendar-today" size={20} color="#015185" />
              </TouchableOpacity>
            </View>

            {/* ENGINE HOURS Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>ENGINE HOURS (DONKEY):</Text>
              <TextInput
                style={styles.input}
                value={formData.engineHours}
                onChangeText={(value) => handleInputChange('engineHours', value)}
                placeholder="Enter engine hours"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>
          </View>
        </View>

        {/* Personnel Selection Section */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>PERSONNEL SELECTION</Text>
          </View>
          
          <View style={styles.formFields}>
            <TouchableOpacity 
              style={styles.personnelSelectButton} 
              onPress={() => setShowPersonnelModal(true)}
            >
              <MaterialIcons name="people" size={20} color="#015185" />
              <Text style={styles.personnelSelectButtonText}>
                Select Personnel Types ({selectedPersonnel.length} selected)
              </Text>
              <MaterialIcons name="chevron-right" size={20} color="#015185" />
            </TouchableOpacity>

            {/* Display selected personnel */}
            {selectedPersonnel.length > 0 && (
              <View style={styles.selectedPersonnelContainer}>
                <Text style={styles.selectedPersonnelTitle}>Selected Personnel:</Text>
                {selectedPersonnel.map(key => renderPersonnelSection(key))}
              </View>
            )}
          </View>
        </View>

        {/* PDF Preview Section */}
        {generatedPdfPath && (
          <View style={styles.section}>
            <View style={styles.sectionHeader}>
              <Text style={styles.sectionTitle}>PDF Generated</Text>
            </View>
            <View style={styles.formFields}>
              <Text style={styles.pdfInfoText}>PDF has been generated successfully!</Text>
              <Text style={styles.pdfPathText}>Path: {generatedPdfPath}</Text>
            </View>
          </View>
        )}
      </ScrollView>

      {/* Personnel Selection Modal */}
      <Modal
        visible={showPersonnelModal}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowPersonnelModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Personnel Types</Text>
            <Text style={styles.modalSubtitle}>Check the personnel types involved in this service:</Text>
            
            <ScrollView style={styles.personnelOptionsContainer}>
              {personnelOptions.map((personnel) => {
                const isSelected = selectedPersonnel.includes(personnel.key);
                return (
                  <TouchableOpacity
                    key={personnel.key}
                    style={styles.personnelOptionRow}
                    onPress={() => handlePersonnelSelection(personnel.key, !isSelected)}
                  >
                    <View style={styles.checkboxContainer}>
                      <View style={[
                        styles.checkbox,
                        isSelected && styles.checkboxSelected
                      ]}>
                        {isSelected && (
                          <MaterialIcons name="check" size={16} color="#fff" />
                        )}
                      </View>
                    </View>
                    <Text style={styles.personnelOptionLabel}>{personnel.label}</Text>
                  </TouchableOpacity>
                );
              })}
            </ScrollView>

            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => setShowPersonnelModal(false)}
              >
                <Text style={styles.modalButtonText}>Done</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* iOS Date Picker Modal */}
      {Platform.OS === 'ios' && showDatePicker && (
        <Modal
          visible={showDatePicker}
          animationType="slide"
          transparent={true}
          onRequestClose={handleIOSDatePickerCancel}
          presentationStyle="pageSheet"
        >
          <View style={styles.iosDatePickerBackground}>
            <View style={styles.iosDatePickerContainer}>
              <View style={styles.iosDatePickerHeader}>
                <TouchableOpacity onPress={handleIOSDatePickerCancel}>
                  <Text style={styles.iosDatePickerButton}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity onPress={handleIOSDatePickerDone}>
                  <Text style={[styles.iosDatePickerButton, styles.iosDatePickerDoneButton]}>Done</Text>
                </TouchableOpacity>
              </View>
              <DateTimePicker
                value={tempDateValue}
                mode="date"
                display="spinner"
                onChange={handleDateChange}
                style={styles.iosDatePicker}
              />
            </View>
          </View>
        </Modal>
      )}

      {/* Android Date Picker */}
      {Platform.OS === 'android' && showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display="default"
          onChange={handleDateChange}
        />
      )}

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={[styles.nextButton, isLoading && styles.disabledButton]} onPress={handleSubmit} disabled={isLoading}>
          <Text style={[styles.nextButtonText, isLoading && styles.disabledButtonText]}>
            {isLoading ? 'Generating PDF...' : 'Submit Form'}
          </Text>
          <MaterialIcons style={styles.bottomicon1} name={isLoading ? "hourglass-empty" : "check"} size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <Toast />

      {/* Sync Modal */}
      <SyncModal 
        visible={syncModalVisible} 
        message={`Processing... ${uploadProgress}%`}
        progress={uploadProgress}
      />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.successIconContainer}>
              <MaterialIcons name="check-circle" size={60} color="#4CAF50" />
            </View>
            <Text style={styles.modalTitle}>PDF Generated Successfully!</Text>
            <Text style={styles.modalSubtitle}>
              Your HSE-124 Workshop RC Rig Check Sheet has been generated and saved.
            </Text>
            
            <View style={styles.successModalButtons}>
              <TouchableOpacity
                style={[styles.successModalButton, styles.backButton]}
                onPress={handleBackToHome}
              >
                <MaterialIcons name="home" size={20} color="#015185" />
                <Text style={styles.backButtonText}>Back to Home</Text>
              </TouchableOpacity>
              
              <TouchableOpacity
                style={[styles.successModalButton, styles.viewPdfButton]}
                onPress={handleViewPDF}
              >
                <MaterialIcons name="picture-as-pdf" size={20} color="#fff" />
                <Text style={styles.viewPdfButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  pageTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 20,
    backgroundColor: '#F2F2F2',
  },
  refreshContainer: {
    alignItems: 'center',
    marginBottom: 10,
    paddingVertical: 10,
  },
  refreshButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingVertical: 10,
    paddingHorizontal: 20,
  },
  refreshButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 10,
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
    margin: 20,
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  formFields: {
    padding: 20,
  },
  fieldContainer: {
    marginBottom: 20,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
    width: '100%',
  },
  pickerButtonText: {
    flex: 1,
    fontSize: 16,
    color: '#333',
  },
  personnelSelectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    padding: 15,
    marginBottom: 20,
  },
  personnelSelectButtonText: {
    flex: 1,
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginLeft: 10,
  },
  selectedPersonnelContainer: {
    marginTop: 10,
  },
  selectedPersonnelTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  personnelSection: {
    marginBottom: 25,
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  personnelTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  nameField: {
    width: '100%',
    marginBottom: 20,
  },
  nameLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  signatureField: {
    width: '100%',
  },
  signatureLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  signatureBoxImage: {
    height: height / 6,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 20,
    width: width * 0.9,
    maxHeight: height * 0.7,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  modalSubtitle: {
    fontSize: 14,
    color: '#666',
    marginBottom: 20,
    textAlign: 'center',
  },
  personnelOptionsContainer: {
    maxHeight: height * 0.4,
  },
  personnelOptionRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  checkboxContainer: {
    marginRight: 15,
  },
  checkbox: {
    width: 24,
    height: 24,
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 4,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  checkboxSelected: {
    backgroundColor: '#015185',
  },
  personnelOptionLabel: {
    fontSize: 16,
    color: '#333',
    flex: 1,
  },
  modalButtons: {
    marginTop: 20,
    alignItems: 'center',
  },
  modalButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 30,
    borderRadius: 6,
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
    backgroundColor: '#fff',
  },
  loadingText: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 10,
  },
  loadingSubtext: {
    fontSize: 14,
    color: '#666',
    marginTop: 5,
  },
  pdfInfoText: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 10,
  },
  pdfPathText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  disabledButton: {
    opacity: 0.7,
  },
  disabledButtonText: {
    color: '#999',
  },
  successIconContainer: {
    alignItems: 'center',
    marginBottom: 15,
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 20,
  },
  successModalButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    paddingHorizontal: 25,
    borderRadius: 6,
    width: '45%',
  },
  backButton: {
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#015185',
  },
  backButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 10,
  },
  viewPdfButton: {
    backgroundColor: '#015185',
  },
  viewPdfButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 10,
  },
  dataSummaryContainer: {
    marginTop: 10,
    marginBottom: 20,
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  dataSummaryTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  dataSummaryRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 5,
  },
  dataSummaryLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  dataSummaryValue: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  detailedDataPreviewContainer: {
    marginTop: 20,
    marginBottom: 20,
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  detailedDataPreviewTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  dataPreviewSection: {
    marginBottom: 15,
  },
  dataPreviewSectionTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  dataPreviewItem: {
    marginBottom: 10,
  },
  dataPreviewItemTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 5,
  },
  dataPreviewItemValue: {
    fontSize: 13,
    color: '#333',
  },
  projectInfoPreviewContainer: {
    marginTop: 20,
    marginBottom: 20,
    padding: 15,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  projectInfoPreviewTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  projectInfoPreviewRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  projectInfoPreviewLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  projectInfoPreviewValue: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    minWidth: 200,
  },
  syncModalText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
    textAlign: 'center',
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  progressContainer: {
    marginTop: 15,
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: 8,
    backgroundColor: '#e0e0e0',
    borderRadius: 4,
    overflow: 'hidden',
    marginBottom: 8,
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 4,
  },
  progressText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  iosDatePickerBackground: {
    flex: 1,
    justifyContent: 'flex-end',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  iosDatePickerContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    paddingBottom: 20,
  },
  iosDatePickerHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  iosDatePickerButton: {
    fontSize: 17,
    color: '#015185',
    fontWeight: '600',
  },
  iosDatePickerDoneButton: {
    fontWeight: 'bold',
  },
  iosDatePicker: {
    height: 200,
    backgroundColor: '#fff',
  },
});

export default HSE_124Step3;