import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DateTimePicker from '@react-native-community/datetimepicker';
// Photos moved to Step 3
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const HSE_034Step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();

  const [formData, setFormData] = useState({
    initiator: '',
    team: '',
    date: new Date(),
    plantOrEquipment: false,
    designOrLayoutOfArea: false,
    operatingProcess: false,
    competencyPeopleRole: false,
    describeProposedChange: '',
    reasonForProposedChange: '',
    canRiskBeManaged: '',
  });

  const [hazardsRisks, setHazardsRisks] = useState([
    { id: 1, hazard: '', risk: '' },
  ]);

  // Impact/Photos moved to Step 3

  const [showDatePicker, setShowDatePicker] = useState(false);

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const saved = await AsyncStorage.getItem('hse034Step2');
      if (saved) {
        const data = JSON.parse(saved);
        if (data.formData) {
          const parseBool = (v) => v === true || v === 'true' || v === 'Yes' || v === 'yes' || v === 1;
          setFormData(prev => ({
            ...prev,
            ...data.formData,
            plantOrEquipment: parseBool(data.formData.plantOrEquipment),
            designOrLayoutOfArea: parseBool(data.formData.designOrLayoutOfArea),
            // Operating process may have been saved under either key
            operatingProcess: parseBool(
              data.formData.operatingProcess !== undefined
                ? data.formData.operatingProcess
                : data.formData.standardSopOrWorkProcedure || data.formData.standardSopWorkProcedure
            ),
            // Competency/People Role consolidated to one flag
            competencyPeopleRole: parseBool(
              data.formData.competencyPeopleRole !== undefined
                ? data.formData.competencyPeopleRole
                : (data.formData.competency || data.formData.peopleRole)
            ),
          }));
        }
        if (Array.isArray(data.hazardsRisks) && data.hazardsRisks.length > 0) {
          setHazardsRisks(data.hazardsRisks);
        }
         // ignore impact/photos (now saved in Step 3)
      }
    } catch (error) {
      console.error('Error loading 034 Step 2 data:', error);
    }
  };

  const saveStep2Data = async () => {
    try {
      const toSave = {
        formData,
        hazardsRisks,
        // only step2 fields persisted
        timestamp: new Date().toISOString(),
      };
      await AsyncStorage.setItem('hse034Step2', JSON.stringify(toSave));
    } catch (error) {
      console.error('Error saving 034 Step 2 data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data.',
        position: 'bottom',
      });
    }
  };

  const handleInputChange = (field, value) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    saveStep2Data();
  };

  const toggleCheckbox = (field) => {
    setFormData(prev => {
      const next = { ...prev, [field]: !prev[field] };
      // Save immediately after toggle
      AsyncStorage.setItem('hse034Step2', JSON.stringify({
        formData: next,
        hazardsRisks,
        timestamp: new Date().toISOString(),
      })).catch(() => {});
      return next;
    });
  };

  // Impact section is text only – no toggles

  const handleHRChange = (id, field, value) => {
    setHazardsRisks(prev => prev.map(row => (row.id === id ? { ...row, [field]: value } : row)));
    saveStep2Data();
  };

  const addHazardRiskRow = () => {
    const newId = Math.max(...hazardsRisks.map(r => r.id)) + 1;
    setHazardsRisks(prev => [...prev, { id: newId, hazard: '', risk: '' }]);
    saveStep2Data();
  };

  const removeHazardRiskRow = (id) => {
    if (hazardsRisks.length <= 1) {
      Toast.show({ type: 'info', text1: 'At least one row is required', position: 'bottom' });
      return;
    }
    Alert.alert('Remove Row', 'Are you sure you want to remove this row?', [
      { text: 'Cancel', style: 'cancel' },
      { text: 'Remove', style: 'destructive', onPress: () => { setHazardsRisks(prev => prev.filter(r => r.id !== id)); saveStep2Data(); } },
    ]);
  };

  const handleBackPress = () => {
    Alert.alert('Go Back', 'Unsaved changes will be saved locally.', [
      { text: 'Cancel', style: 'cancel' },
      { text: 'OK', onPress: async () => { await saveStep2Data(); setActiveStep?.(1); } },
    ], { cancelable: false });
  };

  const handleNext = async () => {
    await saveStep2Data();
    Toast.show({ type: 'success', text1: 'Saved', text2: 'Step 2 saved successfully.', position: 'bottom' });
    setActiveStep?.(3);
  };

  const formatDate = (date) => {
    if (!date) return 'Select Date';
    if (typeof date === 'string') return new Date(date).toLocaleDateString('en-GB');
    return date.toLocaleDateString('en-GB');
  };

  const renderInputField = (label, value, onChangeText, placeholder = '', keyboardType = 'default') => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TextInput
        style={styles.textInput}
        value={value}
        onChangeText={onChangeText}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#888"
      />
    </View>
  );

  const renderTextArea = (label, value, onChangeText, placeholder = '') => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TextInput
        style={[styles.textInput, { minHeight: 80, textAlignVertical: 'top' }]}
        value={value}
        onChangeText={onChangeText}
        placeholder={placeholder}
        placeholderTextColor="#888"
        multiline
      />
    </View>
  );

  const renderDateField = (label, value, onPress) => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>{label}</Text>
      <TouchableOpacity style={styles.dateButton} onPress={onPress}>
        <Text style={styles.dateButtonText}>{formatDate(value)}</Text>
        <MaterialIcons name="calendar-today" size={20} color="#015185" />
      </TouchableOpacity>
    </View>
  );

  const renderCheckboxRow = (label, value, onToggle) => (
    <TouchableOpacity
      style={[styles.inputContainer, styles.checkboxRow]}
      onPress={onToggle}
      activeOpacity={0.7}
      hitSlop={{ top: 8, bottom: 8, left: 8, right: 8 }}
    >
      <View style={styles.checkboxLeft}>
        <View style={[styles.squareBox, value && styles.squareBoxChecked]} />
        <Text style={styles.checkboxLabel}>{label}</Text>
      </View>
    </TouchableOpacity>
  );

  // Photos removed from Step 2

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        <View style={styles.section}>
          {renderInputField('INITIATOR', formData.initiator, (t) => handleInputChange('initiator', t), 'Enter initiator')}
          {renderInputField('TEAM', formData.team, (t) => handleInputChange('team', t), 'Enter team')}
          {renderDateField('DATE', formData.date, () => setShowDatePicker(true))}

          <Text style={styles.sectionHeaderTextBold}>8.1.3 MANAGEMENT OF CHANGE FORM</Text>
          <View style={styles.divider} />

          <Text style={styles.subSectionTitle}>Indicate TYPE OF CHANGE (✓) and provide details below</Text>
          {renderCheckboxRow('Plant or Equipment', formData.plantOrEquipment, () => toggleCheckbox('plantOrEquipment'))}
          {renderCheckboxRow('Design or Layout of Area', formData.designOrLayoutOfArea, () => toggleCheckbox('designOrLayoutOfArea'))}
          {renderCheckboxRow('Operating Process (Standard, SOP, or Work Procedure)', formData.operatingProcess, () => toggleCheckbox('operatingProcess'))}
          {renderCheckboxRow('Competency (People / Role)', formData.competencyPeopleRole, () => toggleCheckbox('competencyPeopleRole'))}

          <Text style={styles.subSectionTitle}>DESCRIBE THE PROPOSED CHANGE</Text>
          {renderTextArea('Description', formData.describeProposedChange, (t) => handleInputChange('describeProposedChange', t), 'Enter description')}

          <Text style={styles.subSectionTitle}>REASON FOR THE PROPOSED CHANGE</Text>
          {renderTextArea('Reason', formData.reasonForProposedChange, (t) => handleInputChange('reasonForProposedChange', t), 'Enter reason')}

          <Text style={styles.subSectionTitle}>What HAZARDS / RISKS are presented by the proposed change?</Text>
          {hazardsRisks.map((row, index) => (
            <View key={row.id} style={styles.deviationRow}>
              <View style={styles.deviationHeader}>
                <Text style={styles.deviationNumber}>Row {index + 1}</Text>
                {hazardsRisks.length > 1 && (
                  <TouchableOpacity style={styles.removeButton} onPress={() => removeHazardRiskRow(row.id)}>
                    <MaterialIcons name="delete" size={20} color="#f44336" />
                  </TouchableOpacity>
                )}
              </View>
              <View style={styles.deviationContent}>
                <View style={styles.deviationField}>
                  <Text style={styles.deviationLabel}>HAZARDS</Text>
                  <TextInput
                    style={styles.deviationTextInput}
                    value={row.hazard}
                    onChangeText={(t) => handleHRChange(row.id, 'hazard', t)}
                    placeholder="Describe hazard"
                    placeholderTextColor="#888"
                  />
                </View>
                <View style={styles.deviationField}>
                  <Text style={styles.deviationLabel}>RISKS</Text>
                  <TextInput
                    style={styles.deviationTextInput}
                    value={row.risk}
                    onChangeText={(t) => handleHRChange(row.id, 'risk', t)}
                    placeholder="Describe risk"
                    placeholderTextColor="#888"
                  />
                </View>
              </View>
            </View>
          ))}

          <TouchableOpacity style={styles.addButton} onPress={addHazardRiskRow}>
            <MaterialIcons name="add" size={18} color="#fff" />
            <Text style={styles.addButtonText}>Add Hazard/Risk</Text>
          </TouchableOpacity>

          {renderInputField('Can the Risk be managed to an acceptable level?', formData.canRiskBeManaged, (t) => handleInputChange('canRiskBeManaged', t), 'Yes/No or details')}
          {/* Sections E, F, and Photos moved to Step 3 */}
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {showDatePicker && (
        <DateTimePicker
          value={formData.date ? (typeof formData.date === 'string' ? new Date(formData.date) : formData.date) : new Date()}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => {
            setShowDatePicker(false);
            if (selectedDate) handleInputChange('date', selectedDate);
          }}
        />
      )}

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 110,
  },
  section: {
    marginBottom: 25,
  },
  inputContainer: {
    marginBottom: 15,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
    paddingVertical: 6,
  },
  inputLabel: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 5,
    fontWeight: '500',
  },
  checkboxButton: {
    paddingHorizontal: 6,
    paddingVertical: 6,
  },
  checkboxLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  checkboxLabel: {
    fontSize: 16,
    color: '#015185',
    marginLeft: 10,
    fontWeight: '500',
    flexShrink: 1,
  },
  squareBox: {
    width: 20,
    height: 20,
    borderWidth: 2,
    borderColor: '#888',
    borderRadius: 3,
    backgroundColor: 'transparent',
  },
  squareBoxChecked: {
    borderColor: '#0b3d91',
    backgroundColor: '#0b3d91',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 12,
    backgroundColor: '#fff',
  },
  dateButtonText: {
    fontSize: 16,
    color: '#333',
  },
  sectionHeaderTextBold: {
    fontSize: 18,
    fontWeight: '700',
    color: '#015185',
    marginTop: 10,
    marginBottom: 8,
  },
  subSectionTitle: {
    fontSize: 16,
    fontWeight: '700',
    color: '#015185',
    marginTop: 8,
    marginBottom: 6,
  },
  divider: {
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    marginBottom: 10,
  },
  impactNote: {
    fontSize: 13,
    color: '#015185',
    fontWeight: '600',
    marginTop: 2,
    marginBottom: 6,
  },
  impactBox: {
    marginTop: 4,
    padding: 12,
    backgroundColor: '#F9FBFD',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#E2ECF5',
  },
  impactLine: {
    fontSize: 14,
    color: '#333',
    lineHeight: 20,
    marginBottom: 4,
  },
  photosLabel: {
    marginTop: 10,
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  photoUploadBox: {
    height: 140,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#F9FBFD',
    marginTop: 8,
    marginBottom: 10,
  },
  photoUploadDisabled: {
    opacity: 0.6,
  },
  photoGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 10,
  },
  photoItem: {
    width: (width * 0.92 - 30) / 3,
    height: (width * 0.92 - 30) / 3,
    borderRadius: 6,
    overflow: 'hidden',
    position: 'relative',
  },
  photoImage: {
    width: '100%',
    height: '100%',
  },
  photoRemove: {
    position: 'absolute',
    top: 6,
    right: 6,
    backgroundColor: 'rgba(0,0,0,0.6)',
    borderRadius: 12,
    padding: 2,
  },
  deviationRow: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 15,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  deviationHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  deviationNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
  },
  removeButton: {
    padding: 5,
  },
  deviationContent: {
    gap: 10,
  },
  deviationField: {
    marginBottom: 10,
  },
  deviationLabel: {
    fontSize: 14,
    color: '#333',
    marginBottom: 5,
    fontWeight: '500',
  },
  deviationTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    paddingHorizontal: 10,
    paddingVertical: 8,
    fontSize: 14,
    backgroundColor: '#fff',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#015185',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderRadius: 5,
    alignSelf: 'flex-start',
    marginTop: 4,
    marginBottom: 10,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginLeft: 5,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginRight: 5,
  },
  bottomicon1: {
    marginLeft: 5,
  },
});

export default HSE_034Step2;