import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Modal,
  ScrollView,
  Platform,
  Image,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import DateTimePicker from '@react-native-community/datetimepicker';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import NetInfo from '@react-native-community/netinfo';
import { Linking } from 'react-native';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';


const { width, height } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing', progress = 0 }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
          </View>
          <Text style={styles.syncModalTextadd}>{message}</Text>
          {progress > 0 && (
            <View style={styles.progressContainer}>
              <View style={styles.progressBar}>
                <View style={[styles.progressFill, { width: `${progress}%` }]} />
              </View>
              <Text style={styles.progressText}>{progress}%</Text>
            </View>
          )}
        </View>
      </View>
    </Modal>
  );
};

const HSE41_42_step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [rig, setRig] = useState('');
  const [location, setLocation] = useState('');
  const [date, setDate] = useState(new Date());
  const [supervisor, setSupervisor] = useState('');
  const [selectedRole, setSelectedRole] = useState(null);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [tempDate, setTempDate] = useState(new Date());

  const [supervisorSignature, setSupervisorSignature] = useState('');
  const [managerSignature, setManagerSignature] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);
  const [hseId, setHseId] = useState(null);
  const [requestNumber, setRequestNumber] = useState('');
  const [uploadProgress, setUploadProgress] = useState(0);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  const roleOptions = [
    { label: 'Site Supervisor', value: 'supervisor' },
    { label: 'Site Manager', value: 'manager' },
    { label: 'Both', value: 'both' },
  ];

  const clearAllHseData = async (showToast = true) => {
    try {
      // Clear all stored data from all steps
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      
      // Also clear any HSE results and notes if they exist
      if (hseId) {
        await AsyncStorage.removeItem(`hseResults_${hseId}`);
        await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      }
      
      if (showToast) {
        Toast.show({
          type: 'success',
          text1: 'Data Cleared',
          text2: 'All inspection data has been cleared successfully.',
          position: 'top',
        });
      }
    } catch (error) {
      console.error('Error clearing HSE data:', error);
      if (showToast) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to clear some data. Please try again.',
          position: 'top',
        });
      }
    }
  };

  // Get current report number from MongoDB
  const getCurrentReportNumber = async () => {
    try {
      const response = await HSEInspectionAPI.getCurrentCounter();
      if (response.success && response.data) {
        return parseInt(response.data.currentNumber) || 0;
      }
      return 0;
    } catch (error) {
      console.error('❌ Error getting current report number from MongoDB:', error);
      return 0;
    }
  };

  // Increment the global report counter in MongoDB
  const incrementReportCounter = async () => {
    try {
      const response = await HSEInspectionAPI.incrementCounter();
      if (response.success && response.data) {
        return parseInt(response.data.currentNumber) || 0;
      }
      return 0;
    } catch (error) {
      console.error('❌ Error incrementing report counter in MongoDB:', error);
      return 0;
    }
  };

  const handleBackPress = () => {
    // Simply go back to step 2, don't clear data
    setActiveStep(2);
  };

  // Load equipment data from step 1
  useEffect(() => {
    const loadEquipmentData = async () => {
      try {
        const savedData = await AsyncStorage.getItem('hseStep1');
        if (savedData) {
          const data = JSON.parse(savedData);
          
          // Set hseId if available
          if (data.id) {
            setHseId(data.id);
          }
          
          if (data.selectedEquipment) {
            setSelectedEquipment(data.selectedEquipment);
            
            // Debug: Log equipment data structure
            console.log('Equipment data structure:', JSON.stringify(data.selectedEquipment, null, 2));
            
            // Auto-fill rig from equipment data - prioritize regNum from the specified path
            let equipmentRig = null;
            
            // Check the specific path you mentioned
            if (data.selectedEquipment.EquipmentsDataJSON?.GlobalEquipmentJsonData?.[1]?.regNum) {
              equipmentRig = data.selectedEquipment.EquipmentsDataJSON.GlobalEquipmentJsonData[1].regNum;
              console.log('Found regNum from nested path:', equipmentRig);
            }
            // Fallback to other possible fields
            else if (data.selectedEquipment.regNum) {
              equipmentRig = data.selectedEquipment.regNum;
              console.log('Found regNum from direct field:', equipmentRig);
            }
            else if (data.selectedEquipment.rig) {
              equipmentRig = data.selectedEquipment.rig;
              console.log('Found rig field:', equipmentRig);
            }
            else if (data.selectedEquipment.equipmentNumber) {
              equipmentRig = data.selectedEquipment.equipmentNumber;
              console.log('Found equipmentNumber field:', equipmentRig);
            }
            else if (data.selectedEquipment.registrationNumber) {
              equipmentRig = data.selectedEquipment.registrationNumber;
              console.log('Found registrationNumber field:', equipmentRig);
            }
            
            console.log('Found equipment rig:', equipmentRig);
            
            if (equipmentRig) {
              setRig(equipmentRig);
              console.log('Auto-filled rig with:', equipmentRig);
            } else {
              console.log('No rig data found in equipment');
            }
          }
          
          // Auto-fill location from project
          if (data.project) {
            setLocation(data.project);
          }
        }
      } catch (error) {
        console.error('Error loading equipment data:', error);
      }
    };
    loadEquipmentData();
    
    // No cleanup needed - data should persist between steps
  }, []);

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setDate(selectedDate);
    }
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  const isValidSignatureUri = (uri) => uri && typeof uri === 'string' && uri.startsWith('data:image');

  const handleSignaturePress = (type) => {
    navigation.navigate('HSE41_42_signature', {
      role: type === 'supervisor' ? 'Supervisor' : 'Manager',
      onSave: (signature) => {
        if (type === 'supervisor') {
          setSupervisorSignature(signature);
        } else if (type === 'manager') {
          setManagerSignature(signature);
        }
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };



  const handleClearSignature = (type) => {
    if (type === 'supervisor') {
      setSupervisorSignature(null);
    } else if (type === 'manager') {
      setManagerSignature(null);
    }
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = (type) => {
    console.error(`${type} signature image failed to load`);
    Toast.show({ type: 'error', text1: 'Error', text2: `Failed to load ${type.toLowerCase()} signature. Please try again.`, position: 'bottom' });
    handleClearSignature(type);
  };

  const generatePDFContent = async () => {
    // Load data from AsyncStorage
    const savedHse1 = await AsyncStorage.getItem('hseStep1');
    const savedHse2 = await AsyncStorage.getItem('hseStep2');
    
    const step1Data = savedHse1 ? JSON.parse(savedHse1) : {};
    const step2Data = savedHse2 ? JSON.parse(savedHse2) : {};
    
    const checklistTitle = step1Data.selectedChecklist?.document_name || 'HSE Inspection Report';
    const documentTitle = step1Data.selectedChecklist?.document_title || checklistTitle;
    const equipmentName = step1Data.selectedEquipment?.equipmentName || 'N/A';
    const equipmentNumber = step1Data.selectedEquipment?.equipmentNumber || 'N/A';
    const country = step1Data.country || 'N/A';
    const project = step1Data.project || 'N/A';
    const coordinates = step1Data.coordinates || 'N/A';
    const gpsAddress = step1Data.gpsAddress || 'N/A';
    
    // Get request number from step1 data
    const requestNumber = step1Data.requestNumber || 'N/A';
    
    const assessmentData = step2Data.assessmentData || {};
    const sections = step2Data.sections || [];
    const notes = step2Data.notes || {};
    
    // Also try to load notes from the hseResults if available
    let additionalNotes = {};
    try {
      const savedHse1 = await AsyncStorage.getItem('hseStep1');
      if (savedHse1) {
        const step1Data = JSON.parse(savedHse1);
        const hseId = step1Data.id;
        if (hseId) {
          const savedNotes = await AsyncStorage.getItem(`hseNotes_${hseId}`);
          if (savedNotes) {
            additionalNotes = JSON.parse(savedNotes);
          }
        }
      }
    } catch (error) {
      console.error('Error loading additional notes:', error);
    }
    
    // Merge notes from both sources
    const allNotes = { ...notes, ...additionalNotes };
    
    // Calculate totals
    let totalYes = 0;
    let totalNo = 0;
    let totalNA = 0;
    
    // Helper functions
    const getCheckbox = (key, assessmentData) => {
      const selection = assessmentData[key] || '';
      if (selection === 'Yes' || selection === 'Okay' || selection === 'Ok') {
        return ' <span class="yes-icon">✓</span>'; // Green checkmark for Yes/Okay/Ok
      } else if (selection === 'No' || selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok') {
        return ' <span class="no-icon">❌</span>'; // Red cross for No/Faulty/Reject/Not Ok
      } else if (selection === 'N/A' || selection === 'Not Applicable') {
        return ' <span class="na-icon">-</span>'; // Orange dash for N/A
      }
      return ''; // No icon if no selection
    };
    
    const getComment = (key, allNotes) => {
      return allNotes[key] || '';
    };
    
    const groupItemsBySubcategory = (items) => {
      const groups = [];
      let currentGroup = null;
      
      items.forEach((item) => {
        // Check if item has a subcategory (e.g., "Compressor:", "Fire extinguishers:")
        if (item.itemName && item.itemName.trim().endsWith(':')) {
          // Start new group
          if (currentGroup) {
            groups.push(currentGroup);
          }
          currentGroup = {
            subcategory: item.itemName.replace(':', ''),
            items: [item]
          };
        } else if (currentGroup) {
          // Add to current group
          currentGroup.items.push(item);
        } else {
          // Single item without subcategory
          groups.push({
            subcategory: null,
            items: [item]
          });
        }
      });
      
      if (currentGroup) {
        groups.push(currentGroup);
      }
      
      return groups;
    };
    
    // Enhanced grouping function that handles repeated aspect values
    const groupItemsByAspect = (items) => {
      const groups = [];
      let currentGroup = null;
      
      items.forEach((item) => {
        // Check if this item should start a new group - only for actual section headers
        // These should be very specific and only include items that are truly section headers
        // Only items ending with a colon (:) should be treated as section headers
        const shouldStartNewGroup = item.itemName && item.itemName.trim().endsWith(':');
        
        if (shouldStartNewGroup) {
          // Start new group
          if (currentGroup) {
            groups.push(currentGroup);
          }
          currentGroup = {
            subcategory: item.itemName.trim().endsWith(':') ? item.itemName.replace(':', '') : item.itemName,
            items: [item]
          };
        } else if (currentGroup) {
          // Add to current group
          currentGroup.items.push(item);
        } else {
          // Single item without subcategory
          groups.push({
            subcategory: null,
            items: [item]
          });
        }
      });
      
      if (currentGroup) {
        groups.push(currentGroup);
      }
      
      return groups;
    };
    
    const groupItemsBySubcategoryOld = (items, documentId, sectionKey) => {
      const groups = [];
      let currentGroup = null;
      
      items.forEach((item, index) => {
        if (typeof item === 'object' && item !== null) {
          const itemKey = Object.keys(item)[0];
          const itemName = itemKey;
          const key = `${documentId}_${sectionKey}_${index}`;
          
          // Check if item has a subcategory - only items ending with colon are section headers
          if (itemName && itemName.trim().endsWith(':')) {
            // Start new group
            if (currentGroup) {
              groups.push(currentGroup);
            }
            currentGroup = {
              subcategory: itemName.replace(':', ''),
              items: [{ itemName, key }]
            };
          } else if (currentGroup) {
            // Add to current group
            currentGroup.items.push({ itemName, key });
          } else {
            // Single item without subcategory
            groups.push({
              subcategory: null,
              items: [{ itemName, key }]
            });
          }
        }
      });
      
      if (currentGroup) {
        groups.push(currentGroup);
      }
      
      return groups;
    };
    
        // Generate checklist rows with proper rowspan structure
    let checklistRows = '';
    
    if (sections && sections.length > 0) {
      sections.forEach((section) => {
        if (section.subItems && section.subItems.length > 0) {
          // Add section header (full width)
          checklistRows += `
            <tr>
              <td colspan="3" class="section-header">${section.name}</td>
            </tr>
          `;

          // Build a flat list of rows for this section while preserving subcategories
          const groupedItems = groupItemsByAspect(section.subItems);
          const rows = [];

          groupedItems.forEach((group) => {
            if (group.subcategory) {
              // Push a subcategory row (rendered in ITEMS column)
              rows.push({ itemHtml: `<strong>${group.subcategory}</strong>`, remarkHtml: '' });
              group.items.forEach((it) => {
                rows.push({ itemHtml: `${it.itemName}${getCheckbox(it.key, assessmentData)}`, remarkHtml: `${getComment(it.key, allNotes)}` });
              });
            } else {
              // No subcategory, each item becomes a row
              group.items.forEach((it) => {
                rows.push({ itemHtml: `${it.itemName}${getCheckbox(it.key, assessmentData)}`, remarkHtml: `${getComment(it.key, allNotes)}` });
              });
            }
          });

          // Render rows with ASPECT in the first column once (rowspan)
          const totalRows = rows.length || 1;
          rows.forEach((r, idx) => {
            if (idx === 0) {
              checklistRows += `
                <tr>
                  <td rowspan="${totalRows}" class="rowspan-cell" style="text-align: center; vertical-align: top;">${section.name}</td>
                  <td>${r.itemHtml}</td>
                  <td>${r.remarkHtml}</td>
                </tr>
              `;
            } else {
              checklistRows += `
                <tr>
                  <td>${r.itemHtml}</td>
                  <td>${r.remarkHtml}</td>
                </tr>
              `;
            }
          });
        }
      });
    } else {
      // Fallback to old format if sections not available
      const step1Data = savedHse1 ? JSON.parse(savedHse1) : {};
      const checklist = step1Data.selectedChecklist;
      
      if (checklist && checklist.checklist) {
        Object.entries(checklist.checklist).forEach(([sectionKey, sectionValue]) => {
          if (Array.isArray(sectionValue)) {
            // Add section header
            checklistRows += `
              <tr>
                <td colspan="3" class="section-header">${sectionKey}</td>
              </tr>
            `;
            
            // Group items by aspect - use enhanced grouping
            const groupedItems = groupItemsByAspect(sectionValue.map((item, index) => {
              if (typeof item === 'object' && item !== null) {
                const itemKey = Object.keys(item)[0];
                // The itemKey is the actual inspection item text (e.g., "Danger tape posted around perimeter of drill site")
                return {
                  itemName: itemKey,
                  key: `${checklist.document_id}_${sectionKey}_${index}`
                };
              }
              return item;
            }));
            
            // Check if we have any grouped items
            const hasGroupedItems = groupedItems.some(group => group.subcategory);
            if (hasGroupedItems) {
              // Compute total rows for the section so ASPECT can span them
              const sectionRowspan = groupedItems.reduce((sum, g) => sum + (g.items ? g.items.length : 0), 0) || 1;
              let isFirstItem = true;

              groupedItems.forEach((group) => {
                if (group.subcategory) {
                  const subgroupRowspan = group.items.length || 1;

                  if (isFirstItem) {
                    checklistRows += `
                      <tr>
                        <td rowspan="${sectionRowspan}" class="rowspan-cell" style="text-align: center; vertical-align: top;">${sectionKey}</td>
                        <td rowspan="${subgroupRowspan}" class="subcategory-cell" style="text-align: center; vertical-align: top;">${group.subcategory}</td>
                        <td>${group.items[0].itemName}${getCheckbox(group.items[0].key, assessmentData)}</td>
                        <td>${getComment(group.items[0].key, allNotes)}</td>
                      </tr>
                    `;
                    isFirstItem = false;
                  } else {
                    checklistRows += `
                      <tr>
                        <td rowspan="${subgroupRowspan}" class="subcategory-cell" style="text-align: center; vertical-align: top;">${group.subcategory}</td>
                        <td>${group.items[0].itemName}${getCheckbox(group.items[0].key, assessmentData)}</td>
                        <td>${getComment(group.items[0].key, allNotes)}</td>
                      </tr>
                    `;
                  }

                  for (let i = 1; i < group.items.length; i++) {
                    const item = group.items[i];
                    checklistRows += `
                      <tr>
                        <td>${item.itemName}${getCheckbox(item.key, assessmentData)}</td>
                        <td>${getComment(item.key, allNotes)}</td>
                      </tr>
                    `;
                  }
                } else {
                  const item = group.items[0];
                  if (isFirstItem) {
                    checklistRows += `
                      <tr>
                        <td rowspan="${sectionRowspan}" class="rowspan-cell" style="text-align: center; vertical-align: top;">${sectionKey}</td>
                        <td>${item.itemName}${getCheckbox(item.key, assessmentData)}</td>
                        <td>${getComment(item.key, allNotes)}</td>
                      </tr>
                    `;
                    isFirstItem = false;
                  } else {
                    checklistRows += `
                      <tr>
                        <td>${item.itemName}${getCheckbox(item.key, assessmentData)}</td>
                        <td>${getComment(item.key, allNotes)}</td>
                      </tr>
                    `;
                  }
                }
              });
            } else {
              // No grouped items, so all items belong to the main section
              sectionValue.forEach((item, index) => {
                if (typeof item === 'object' && item !== null) {
                  const itemKey = Object.keys(item)[0];
                  const key = `${checklist.document_id}_${sectionKey}_${index}`;
                  
                  if (index === 0) {
                    // First item shows section name in ASPECT
                    checklistRows += `
                      <tr>
                        <td rowspan="${sectionValue.length}" class="subcategory-cell" style="text-align: center; vertical-align: top;">${sectionKey}</td>
                        <td>${itemKey}${getCheckbox(key, assessmentData)}</td>
                        <td>${getComment(key, allNotes)}</td>
                      </tr>
                    `;
                  } else {
                    // Subsequent items show empty ASPECT (it's spanned)
                    checklistRows += `
                      <tr>
                        <td>${itemKey}${getCheckbox(key, assessmentData)}</td>
                        <td>${getComment(key, allNotes)}</td>
                      </tr>
                    `;
                  }
                }
              });
            }
          }
        });
      }
    }

    return `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE Inspection Report</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 0;
              box-sizing: border-box;
            }
            .container {
              width: 100%;
              max-width: 800px;
              margin: 20px auto;
              border: 1px solid #000;
              box-sizing: border-box;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              background-color: #f0f0f0;
              padding: 15px;
              border-bottom: 1px solid #000;
            }
            .header .logo {
              width: 120px;
              height: auto;
              margin-right: 20px;
            }
            .header .details {
              flex: 1;
            }
            .header .details table {
              border-collapse: collapse;
              width: 100%;
              border: 1px solid #000;
            }
            .header .details td {
              padding: 8px 12px;
              text-align: left;
              border: 1px solid #000;
              font-size: 12px;
            }
            .header .details td:first-child {
              font-weight: bold;
              background-color: #f8f9fa;
              width: 40%;
            }
            .inspection-details {
              display: grid;
              grid-template-columns: 1fr 1fr;
              padding: 10px;
              border-bottom: 1px solid #000;
              gap: 0;
              background-color: #f8f9fa;
            }
            .inspection-details div {
              padding: 8px 12px;
              border-right: 1px solid #000;
              border-bottom: 1px solid #000;
              font-size: 12px;
            }
            .inspection-details div:nth-child(odd) {
              font-weight: bold;
              background-color: #e9ecef;
            }
            .inspection-details div:nth-child(even) {
              background-color: #ffffff;
            }
            .table {
              width: 100%;
              border-collapse: collapse;
              table-layout: auto;
              min-width: 800px;
            }
            .table th,
            .table td {
              padding: 8px;
              border: 1px solid #000;
              text-align: left;
              word-wrap: break-word;
              overflow-wrap: break-word;
            }
            .table td span {
              display: inline-block;
              margin-left: 5px;
            }
            .table th {
              white-space: nowrap;
              overflow: hidden;
              text-overflow: ellipsis;
            }
            .table th:nth-child(1) {
              width: 25%;
              min-width: 150px;
            }
            .table th:nth-child(2) {
              width: 55%;
              min-width: 350px;
            }
            .table th:nth-child(3) {
              width: 20%;
              min-width: 120px;
            }
            .table th {
              background-color: #d3d3d3;
              color: #333;
              font-weight: bold;
              text-align: center;
              padding: 12px 8px;
              white-space: nowrap;
              overflow: hidden;
              text-overflow: ellipsis;
            }
            .table .section-header {
              background-color: #a9a9a9;
              color: #333;
              font-weight: bold;
              text-align: center;
              padding: 10px 8px;
            }
            .checkbox {
              color: #015185;
              font-weight: bold;
              font-size: 16px;
            }
            .yes-icon {
              color: #4CAF50;
              font-weight: bold;
              font-size: 18px;
            }
            .no-icon {
              color: #F44336;
              font-weight: bold;
              font-size: 18px;
            }
            .na-icon {
              color: #FF9800;
              font-weight: bold;
              font-size: 16px;
            }
            .rowspan-cell {
              vertical-align: top;
              background-color: #f8f9fa;
              font-weight: 600;
              color: #015185;
              border-right: 2px solid #015185;
            }
            .subcategory-cell {
              background-color: #e9ecef;
              font-weight: 600;
              color: #333;
              border-right: 2px solid #6c757d;
              border-bottom: 1px solid #000;
            }
            /* Ensure proper rowspan display */
            .table tr td[rowspan] {
              border-bottom: 1px solid #000;
            }
            .table tr:last-child td[rowspan] {
              border-bottom: none;
            }
            .table tr:nth-child(even) {
              background-color: #fafafa;
            }
            .table tr:nth-child(odd) {
              background-color: #ffffff;
            }
            @media (max-width: 600px) {
              .container {
                margin: 10px;
                border: none;
                width: 100%;
                box-sizing: border-box;
              }
              .header {
                flex-direction: column;
                align-items: center;
                padding: 10px;
                text-align: center;
              }
              .header .logo {
                width: 100px;
                height: auto;
                margin-right: 0;
                margin-bottom: 15px;
              }
              .header .details {
                width: 100%;
              }
              .header .details table {
                font-size: 11px;
              }
              .header .details td {
                padding: 6px 8px;
              }
              
              .inspection-details {
                grid-template-columns: 1fr;
                padding: 8px;
              }
              .inspection-details div {
                width: 100%;
                border-right: none;
              }
              .table {
                width: 100%;
                display: block;
                overflow-x: auto;
                min-width: 800px;
              }
              .table th,
              .table td {
                display: block;
                width: 100% !important;
                box-sizing: border-box;
                text-align: left;
              }
              .table tr {
                display: block;
                margin-bottom: 10px;
                border-bottom: 1px solid #000;
              }
              .table th {
                background-color: #d3d3d3;
                color: #333;
                font-weight: bold;
              }
              .table .section-header {
                background-color: #a9a9a9;
                color: #333;
                font-weight: bold;
                width: 100%;
              }
              .table td:empty {
                display: none;
              }
              .table th:empty {
                display: none;
              }
              /* Mobile rowspan handling */
              .table td[rowspan] {
                border-bottom: 1px solid #000;
                margin-bottom: 10px;
              }
              .table tr:last-child td[rowspan] {
                border-bottom: none;
                margin-bottom: 0;
              }
            }
          </style>
        </head>
        <body>
          <div class="container">
                          <div class="header">
                <div class="logo">
                  <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 120px; height: auto;">
                </div>
                <div class="details">
                  <table>
                    <tr><td>Document name</td><td>${checklistTitle}</td></tr>
                    <tr><td>Document number</td><td>${step1Data.selectedChecklist?.document_id || 'HSE-043'}</td></tr>
                    <tr><td>Request number</td><td>${requestNumber}</td></tr>
                    <tr><td>Version</td><td>${step1Data.selectedChecklist?.version || '1'}</td></tr>
                    <tr><td>Revision date</td><td>${step1Data.selectedChecklist?.version_date || new Date().toLocaleDateString()}</td></tr>
                    <tr><td>Approved by</td><td>${step1Data.selectedChecklist?.approved_by || 'HSM'}</td></tr>
                  </table>
                </div>
              </div>
            <div class="report-title">${documentTitle}</div>
            <div class="inspection-details">
              <div>Rig</div><div>${rig}</div>
              <div>Location</div><div>${location}</div>
              <div>Date</div><div>${date.toLocaleDateString()}</div>
              <div>Supervisor</div><div>${supervisor}</div>
            </div>
            <table class="table">
              <tr>
                <th>ASPECT</th>
                <th>ITEMS TO BE INSPECTED</th>
                <th>REMARKS</th>
              </tr>
              ${checklistRows}
            </table>
              
              <div style="margin-top: 30px; padding: 20px; border-top: 1px solid #000;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 20px;">
                  <div style="text-align: center; width: 45%;">
                    <div style="border-bottom: 1px solid #000; padding-bottom: 5px; margin-bottom: 10px;">
                      <strong>Site Supervisor</strong>
                    </div>
                    <div style="height: 60px; border: 1px solid #000; margin-bottom: 10px; display: flex; align-items: center; justify-content: center;">
                      ${supervisorSignature && isValidSignatureUri(supervisorSignature) ? 
                        `<img src="${supervisorSignature}" style="max-width: 100%; max-height: 100%;" alt="Supervisor Signature"/>` : 
                        'Signature'
                      }
                    </div>
                    <div style="font-weight: bold;">${supervisor}</div>
                    <div style="font-size: 12px; color: #666;">Date: ${date.toLocaleDateString()}</div>
                  </div>
                  
                  <div style="text-align: center; width: 45%;">
                    <div style="border-bottom: 1px solid #000; padding-bottom: 5px; margin-bottom: 10px;">
                      <strong>Site Manager</strong>
                    </div>
                    <div style="height: 60px; border: 1px solid #000; margin-bottom: 10px; display: flex; align-items: center; justify-content: center;">
                      ${managerSignature && isValidSignatureUri(managerSignature) ? 
                        `<img src="${managerSignature}" style="max-width: 100%; max-height: 100%;" alt="Manager Signature"/>` : 
                        'Signature'
                      }
                    </div>
                    <div style="font-weight: bold;">Manager Name</div>
                    <div style="font-size: 12px; color: #666;">Date: ${date.toLocaleDateString()}</div>
                  </div>
                </div>
              </div>
            </div>
          </body>
          </html>
        `;
  };

  const generateAndSavePDF = async () => {
    try {
      setSyncModalVisible(true);
      setUploadProgress(20);
      
      // Check if required data exists
      const savedHse1 = await AsyncStorage.getItem('hseStep1');
      let savedHse2 = await AsyncStorage.getItem('hseStep2');

      if (!savedHse1) {
        throw new Error('Required inspection data is missing. Please complete step 1 first.');
      }

      // If step2 is missing, don't fail - fallback to empty assessment so PDF generation can proceed.
      if (!savedHse2) {
        console.warn('hseStep2 not found in AsyncStorage, proceeding with empty assessment data');
        savedHse2 = JSON.stringify({ assessmentData: {}, sections: [] });
      }

      const step1Data = JSON.parse(savedHse1);
      const step2Data = JSON.parse(savedHse2);
      
      // Use the request number from step1 as the database ID
      const dbId = step1Data.requestNumber;
      if (!dbId) {
        throw new Error('Request number not found in step 1 data');
      }
      
      setRequestNumber(dbId);
      
      // Generate PDF
      console.log('📄 Generating PDF for HSE-041/042...');
      const htmlContent = await generatePDFContent();
      const options = {
        html: htmlContent,
        fileName: `HSE_Report_${dbId}`,
        directory: Platform.OS === 'android' ? 'Documents' : 'DocumentDirectory',
      };
      
      const result = await RNHTMLtoPDF.convert(options);
      if (!result || !result.filePath) {
        throw new Error('Failed to generate PDF file');
      }
      const filePath = result.filePath;
      console.log('✅ PDF generated at:', filePath);
      setUploadProgress(50);

      if (isOnline) {
        // Upload to MongoDB
        const userId = await AsyncStorage.getItem('userUid');
        if (!userId) {
          throw new Error('User ID not found. Please log in again.');
        }

        // Upload PDF to MongoDB
        console.log('📤 Uploading PDF to MongoDB...');
        const pdfFile = {
          uri: `file://${filePath}`,
          type: 'application/pdf',
          name: `hse_041_042_${userId}_${Date.now()}.pdf`
        };
        const uploadResponse = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
        if (!uploadResponse.success) {
          throw new Error('Failed to upload PDF to MongoDB');
        }
        const downloadUrl = uploadResponse.downloadUrl;
        console.log('✅ PDF uploaded:', downloadUrl);
        setUploadProgress(70);
        
        // Save to MongoDB with request number as ID
        console.log('💾 Saving report to MongoDB...');
        const reportData = {
          _id: dbId,
          userId,
          requestNumber: dbId,
          status: 'completed',
          documentType: step1Data.selectedChecklist?.document_id || 'HSE-041/042',
          documentName: step1Data.selectedChecklist?.document_name || 'HSE Inspection Report',
          step1: step1Data,
          step2: step2Data,
          step3: {
            rig,
            location,
            date: date.toISOString(),
            supervisor,
            selectedRole,
            supervisorSignature: isValidSignatureUri(supervisorSignature) ? supervisorSignature : null,
            managerSignature: isValidSignatureUri(managerSignature) ? managerSignature : null,
          },
          pdfDownloadUrl: downloadUrl,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
          createdBy: userId,
          lastModifiedBy: userId
        };
        
        console.log('📋 Report data to save:', JSON.stringify(reportData, null, 2));
        
        let saveResponse;
        try {
          saveResponse = await HSEInspectionAPI.createHSEInspection(userId, reportData);
          console.log('📥 Save response:', saveResponse);
        } catch (saveError) {
          console.error('❌ Save error details:', saveError);
          throw new Error(saveError.message || 'Failed to save report to MongoDB');
        }

        if (!saveResponse.success) {
          console.error('❌ MongoDB save failed:', saveResponse.message || saveResponse.error);
          throw new Error(saveResponse.message || 'Failed to save report to MongoDB');
        }
        console.log('✅ Report saved to MongoDB with ID:', dbId);
        
        setPdfDownloadUrl(downloadUrl);
        setUploadProgress(90);
        
        // Send HSE inspection notifications
        console.log('📧 Sending HSE inspection notifications...');
        try {
          const userData = await AsyncStorage.getItem('userData');
          const user = userData ? JSON.parse(userData) : null;
          const createdByName = user?.name || supervisor || 'Unknown Inspector';
          const inspectorEmail = user?.email || '';
          
          const notificationPayload = {
            hseInspectionData: {
              _id: dbId,
              id: dbId,
              requestNumber: dbId,
              checklistName: step1Data.selectedChecklist?.document_name || 'HSE-041/042 Inspection',
              equipmentName: step1Data.selectedEquipment?.equipmentName || '',
              country: step1Data.country || '',
              project: step1Data.project || '',
              status: 'completed',
            },
            createdByName,
            inspectorEmail,
            pdfUrl: downloadUrl
          };
          
          const notifResponse = await fetch('http://168.231.113.219:5028/api/hse-inspection-notifications/new-inspection', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(notificationPayload),
          });
          
          if (notifResponse.ok) {
            console.log('✅ HSE notifications sent successfully');
          }
        } catch (notifError) {
          console.error('❌ Error sending HSE notifications:', notifError);
        }
        
        // Delete temp PDF file
        await RNFS.unlink(filePath).catch(err => console.warn('⚠️ Failed to delete local PDF:', err));
        
        // Increment the global report counter
        await incrementReportCounter();

        // Clear local storage
        await AsyncStorage.removeItem('hseStep1').catch(err => console.warn('⚠️ Error clearing step1:', err));
        await AsyncStorage.removeItem('hseStep2').catch(err => console.warn('⚠️ Error clearing step2:', err));
        await AsyncStorage.removeItem('hseStep3').catch(err => console.warn('⚠️ Error clearing step3:', err));
        
        setUploadProgress(100);
        
        return downloadUrl;
      } else {
        // Save locally for offline sync
        const offlineData = {
          requestNumber: dbId,
          rig,
          location,
          date: date.toISOString(),
          supervisor,
          selectedRole,
          supervisorSignature: isValidSignatureUri(supervisorSignature) ? supervisorSignature : null,
          managerSignature: isValidSignatureUri(managerSignature) ? managerSignature : null,
          pdfPath: filePath,
          createdAt: new Date().toISOString(),
        };
        
        const offlineReports = JSON.parse(await AsyncStorage.getItem('offlineHSEReports') || '[]');
        offlineReports.push(offlineData);
        await AsyncStorage.setItem('offlineHSEReports', JSON.stringify(offlineReports));
        
        return null;
      }
    } catch (error) {
      console.error('❌ Error generating PDF:', error);
      console.error('❌ Error message:', error.message);
      throw error;
    } finally {
      setSyncModalVisible(false);
      setUploadProgress(0);
    }
  };

    const handleGenerateReport = async () => {
    if (!rig.trim() || !location.trim() || !supervisor.trim() || !selectedRole) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please fill all required fields',
        position: 'bottom',
      });
      return;
    }

    if (selectedRole === 'supervisor' && !supervisorSignature) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please add supervisor signature',
        position: 'bottom',
      });
      return;
    }

    if (selectedRole === 'manager' && !managerSignature) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please add manager signature',
        position: 'bottom',
      });
      return;
    }

    if (selectedRole === 'both' && (!supervisorSignature || !managerSignature)) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Please add both signatures',
        position: 'bottom',
      });
      return;
    }

    try {
      // Save all data to AsyncStorage
      const savedHse = await AsyncStorage.getItem('hseStep1');
      let hseData = savedHse ? JSON.parse(savedHse) : {};
      hseData.rig = rig.trim();
      hseData.location = location.trim();
      hseData.date = date.toISOString();
      hseData.supervisor = supervisor.trim();
      hseData.selectedRole = selectedRole;
      hseData.supervisorSignature = supervisorSignature;
      hseData.managerSignature = managerSignature;
      await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
      
      // Generate PDF and save to database
      const pdfUrl = await generateAndSavePDF();
      
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `HSE inspection report generated and saved successfully!`,
        position: 'bottom',
      });
      
      // Show success modal with PDF view option
      if (pdfUrl) {
        setShowSuccessModal(true);
      } else {
        // Clear stored data after successful generation
        setTimeout(async () => {
          try {
            // Clear all stored data from all steps
            await AsyncStorage.removeItem('hseStep1');
            await AsyncStorage.removeItem('hseStep2');
            await AsyncStorage.removeItem('hseStep3');
            
            // Also clear any HSE results and notes if they exist
            if (hseId) {
              await AsyncStorage.removeItem(`hseResults_${hseId}`);
              await AsyncStorage.removeItem(`hseNotes_${hseId}`);
            }
            
            Toast.show({
              type: 'success',
              text1: 'Data Cleared',
              text2: 'All inspection data has been cleared successfully.',
              position: 'top',
            });
            
            navigation.navigate('HomeScreen');
          } catch (error) {
            console.error('Error clearing data:', error);
            Toast.show({
              type: 'error',
              text1: 'Error',
              text2: 'Failed to clear some data. Please try again.',
              position: 'top',
            });
            navigation.navigate('HomeScreen');
          }
        }, 2000);
      }
    } catch (error) {
      console.error('❌ Error generating report:', error);
      console.error('❌ Error details:', error.message);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to generate report.',
        position: 'bottom',
        visibilityTime: 4000,
      });
    }
  };

  const handleViewPdf = async () => {
    if (!requestNumber) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Report ID not available.' });
      return;
    }

    try {
      const response = await HSEInspectionAPI.getHSEInspectionById(requestNumber);

      if (!response.success || !response.data) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'Report not found in the database.' });
        return;
      }

      const pdfUrl = response.data.pdfDownloadUrl;
      if (!pdfUrl) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'PDF URL not available in the report.' });
        return;
      }

      setShowSuccessModal(false);
      
      // Allow modal close animation to finish before navigating
      setTimeout(() => {
        navigation.navigate('HSEViewPDF', {
          pdfUrl: pdfUrl,
          onGoBack: () => {
            setShowSuccessModal(false);
          }
        });
      }, 300);
    } catch (error) {
      console.error('❌ Error fetching PDF URL:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Could not fetch the PDF URL. Please try again.' });
    }
  };

  const handleBackToHome = async () => {
    setShowSuccessModal(false);
    try {
      // Clear all stored data from all steps
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      
      // Also clear any HSE results and notes if they exist
      if (hseId) {
        await AsyncStorage.removeItem(`hseResults_${hseId}`);
        await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      }
      
      Toast.show({
        type: 'success',
        text1: 'Data Cleared',
        text2: 'All inspection data has been cleared successfully.',
        position: 'top',
      });
      
      navigation.navigate('HomeScreen');
    } catch (error) {
      console.error('Error clearing data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to clear some data. Please try again.',
        position: 'top',
      });
      navigation.navigate('HomeScreen');
    }
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
       

        {/* Basic Information Section */}
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Rig *</Text>
          <TextInput
            style={styles.textInput}
            value={rig}
            onChangeText={setRig}
            placeholder="Enter rig number or registration"
            placeholderTextColor="#888"
          />
          {rig && (
            <Text style={styles.autoFillNote}>
              {selectedEquipment?.regNum || selectedEquipment?.rig || selectedEquipment?.equipmentNumber 
                ? 'Auto-filled from equipment regNum' 
                : 'Manually entered'}
            </Text>
          )}
          {!rig && selectedEquipment && (
            <Text style={styles.autoFillNote}>
              No regNum found in equipment data. Please enter manually.
            </Text>
          )}
        </View>

        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Location *</Text>
          <TextInput
            style={[styles.textInput, styles.disabledInput]}
            value={location}
            editable={false}
            placeholder="Location from project"
            placeholderTextColor="#888"
          />
          <Text style={styles.autoFillNote}>Auto-filled from project selection</Text>
        </View>

        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Date *</Text>
          <TouchableOpacity style={styles.pickerButton} onPress={showDatePickerModal}>
            <Text style={styles.pickerButtonText}>{date.toLocaleDateString()}</Text>
            <MaterialIcons name="calendar-today" size={20} color="#015185" />
          </TouchableOpacity>
        </View>

        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Supervisor *</Text>
          <TextInput
            style={styles.textInput}
            value={supervisor}
            onChangeText={setSupervisor}
            placeholder="Enter supervisor name"
            placeholderTextColor="#888"
          />
        </View>

        {/* Role Selection Section */}
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Approvals *</Text>
          <Dropdown
            style={styles.dropdown}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            data={roleOptions}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select your role"
            value={selectedRole}
            onChange={item => setSelectedRole(item.value)}
          />
        </View>

        {/* Signature Section */}
        {selectedRole && (
          <View style={styles.section}>
            <Text style={styles.sectionTitle}>Signatures</Text>
            
            {(selectedRole === 'supervisor' || selectedRole === 'both') && (
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Supervisor Name</Text>
                <TextInput
                  style={styles.textInput}
                  placeholder="Enter supervisor name"
                  placeholderTextColor="#888"
                  value={supervisor}
                  onChangeText={setSupervisor}
                />
              </View>
            )}

            {(selectedRole === 'supervisor' || selectedRole === 'both') && (
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Supervisor Signature</Text>
                <View style={styles.signatureBoxImage}>
                  {isValidSignatureUri(supervisorSignature) ? (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('supervisor')}>
                      <Image
                        source={{ uri: supervisorSignature }}
                        style={styles.signatureImage}
                        resizeMode="contain"
                        onError={() => handleImageError('Supervisor')}
                      />
                    </TouchableOpacity>
                  ) : (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('supervisor')}>
                      <View style={styles.plusCircle}>
                        <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                      </View>
                    </TouchableOpacity>
                  )}
                </View>
              </View>
            )}

            {(selectedRole === 'manager' || selectedRole === 'both') && (
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Manager Name</Text>
                <TextInput
                  style={styles.textInput}
                  placeholder="Enter manager name"
                  placeholderTextColor="#888"
                  value={supervisor}
                  onChangeText={setSupervisor}
                />
              </View>
            )}

            {(selectedRole === 'manager' || selectedRole === 'both') && (
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Manager Signature</Text>
                <View style={styles.signatureBoxImage}>
                  {isValidSignatureUri(managerSignature) ? (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('manager')}>
                      <Image
                        source={{ uri: managerSignature }}
                        style={styles.signatureImage}
                        resizeMode="contain"
                        onError={() => handleImageError('Manager')}
                      />
                    </TouchableOpacity>
                  ) : (
                    <TouchableOpacity style={styles.plusIconCircleContainer} onPress={() => handleSignaturePress('manager')}>
                      <View style={styles.plusCircle}>
                        <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                      </View>
                    </TouchableOpacity>
                  )}
                </View>
              </View>
            )}
          </View>
        )}

        <View style={styles.infoSection}>
          <MaterialIcons name="info" size={24} color="#015185" />
          <Text style={styles.infoText}>
            All assessment data has been collected. Fill in the details above and add your signature to generate the final report.
          </Text>
        </View>
      </ScrollView>

      {/* Date Picker Modal */}
      {/**
       * Use a modal-based picker on iOS so user can confirm (Done/Cancel).
       * Android continues to use the inline/native picker.
       */}
      {showDatePicker && Platform.OS === 'ios' ? (
        <Modal transparent animationType="slide" visible={showDatePicker} onRequestClose={() => setShowDatePicker(false)}>
          <View style={styles.iosDatePickerOverlay}>
            <View style={styles.iosDatePickerContainer}>
              <DateTimePicker
                value={tempDate}
                mode="date"
                display="spinner"
                onChange={(event, selected) => {
                  // update temporary date while spinning
                  if (selected) setTempDate(selected);
                }}
                style={{ width: '100%' }}
              />
              <View style={styles.iosPickerButtons}>
                <TouchableOpacity onPress={() => { setShowDatePicker(false); }} style={styles.iosPickerButton}>
                  <Text style={{ color: '#015185' }}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity onPress={() => { setDate(tempDate); setShowDatePicker(false); }} style={styles.iosPickerButton}>
                  <Text style={{ color: '#015185', fontWeight: '600' }}>Done</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
      ) : showDatePicker && Platform.OS !== 'ios' ? (
        <DateTimePicker
          value={date}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => {
            setShowDatePicker(false);
            if (selectedDate) setDate(selectedDate);
          }}
        />
      ) : null}



      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        
        {/* <TouchableOpacity
          style={styles.exitButton}
          onPress={() => {
            Alert.alert(
              'Exit Inspection',
              'Do you want to exit the entire HSE inspection? All filled data will be lost.',
              [
                { text: 'Cancel', style: 'cancel' },
                {
                  text: 'Exit',
                  style: 'destructive',
                  onPress: async () => {
                    await clearAllHseData();
                    navigation.navigate('HomeScreen');
                  },
                },
              ],
              { cancelable: false }
            );
          }}
        >
          <MaterialIcons name="close" size={16} color="#dc3545" />
          <Text style={styles.exitButtonText}>Exit</Text>
        </TouchableOpacity> */}
        
        <TouchableOpacity
          style={[styles.nextButton, isLoading && styles.disabledButton]}
          onPress={handleGenerateReport}
          disabled={isLoading}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              HSE Inspection Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* SyncModal for Loading */}
      <SyncModal 
        visible={syncModalVisible || isLoading} 
        message={`Processing... ${uploadProgress}%`}
        progress={uploadProgress}
      />

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingContainer: {
    alignItems: 'center',
    paddingVertical: 15,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '600',
    color: '#015185',
    marginTop: 5,
  },
  subheadingText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 5,
  },
  inputContainer: {
    marginBottom: width * 0.04,
  },
  inputLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
  },
  pickerButtonText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  dropdown: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    width: '100%',
    alignSelf: 'stretch',
  },
  placeholderStyle: {
    fontSize: 14,
    color: '#888',
  },
  dropdownText: {
    fontSize: 14,
    color: '#333',
  },
  section: {
    marginBottom: width * 0.04,
  },
  dynamicSection: {
    marginTop: 20,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  dynamicFieldContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  fieldHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  fieldNumber: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  infoSection: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f4fd',
    padding: 15,
    borderRadius: 8,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: '#015185',
  },
  infoText: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  exitButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingVertical: 10,
    borderRadius: 5,
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#dc3545',
  },
  exitButtonText: {
    color: '#dc3545',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 5,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  disabledInput: {
    backgroundColor: '#f5f5f5',
    color: '#666',
  },
  autoFillNote: {
    fontSize: 12,
    color: '#015185',
    fontStyle: 'italic',
    marginTop: 4,
    marginLeft: 4,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  progressContainer: {
    marginTop: 15,
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: 8,
    backgroundColor: '#e0e0e0',
    borderRadius: 4,
    overflow: 'hidden',
    marginBottom: 8,
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 4,
  },
  progressText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  iosDatePickerOverlay: {
    flex: 1,
    justifyContent: 'flex-end',
    backgroundColor: 'rgba(0,0,0,0.4)'
  },
  iosDatePickerContainer: {
    backgroundColor: '#fff',
    paddingTop: 10,
    borderTopLeftRadius: 12,
    borderTopRightRadius: 12,
  },
  iosPickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderTopWidth: 1,
    borderTopColor: '#eee'
  },
  iosPickerButton: {
    padding: 8,
  },
});

export default HSE41_42_step3;