
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  ScrollView,
  Alert,
  Modal,
  FlatList,
  Image,
  Animated,
  TextInput,
  Platform,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import HSEInspectionAPI from '../ConfigMongo/HSEInspectionMongoConfig';
import { MongoAPI } from '../ConfigMongo/MongoDBConfig';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const DROPDOWN_OPTIONS = [
  { label: 'View PDF Report', value: 'view-pdf', icon: <MaterialIcons name="picture-as-pdf" size={22} color="#015185" /> },
  { label: 'Delete Inspection', value: 'delete', icon: <MaterialIcons name="delete" size={22} color="#e74c3c" /> },
  { label: 'Delete All Reports', value: 'delete-all', icon: <MaterialIcons name="delete-sweep" size={22} color="#e74c3c" /> },
];

const HSEInspectionHome = () => {
  const navigation = useNavigation();
  const [savedInspections, setSavedInspections] = useState([]);
  const [filteredInspections, setFilteredInspections] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedInspection, setSelectedInspection] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [showSearch, setShowSearch] = useState(false);
  
  // Filter states
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [activeFilters, setActiveFilters] = useState({
    locations: [],
    projects: [],
    dateRange: 'all', // all, today, week, month, custom
  });
  const [availableLocations, setAvailableLocations] = useState([]);
  const [availableProjects, setAvailableProjects] = useState([]);
  const [userInfoModalVisible, setUserInfoModalVisible] = useState(false);
  const [userInfo, setUserInfo] = useState({ countries: [], projects: [], inspectionPermissions: [], employeeNumber: '', name: '' });
  const [loadingUserInfo, setLoadingUserInfo] = useState(false);
  const [permissionFilter, setPermissionFilter] = useState({ onlyMine: false, userId: null });
  
  // Pagination states
  const [page, setPage] = useState(1);
  const [hasMore, setHasMore] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const PAGE_SIZE = 8;

  useEffect(() => {
    loadSavedInspections(true);
  }, []);

  // Fetch current user info to determine permissions from MongoDB
  const fetchCurrentUserInfo = async () => {
    try {
      setLoadingUserInfo(true);
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) return;
      
      const response = await MongoAPI.getUserById(userId);
      if (!response || !response.success || !response.data) return;
      
      const user = response.data;
      const hseModule = user.modules?.find(m => m.module === 'hseInspection') || { permissions: [] };
      
      setUserInfo({
        countries: user.countries || [],
        projects: user.projects || [],
        inspectionPermissions: hseModule.permissions || [],
        employeeNumber: user.employeeNumber || '',
        name: user.fullName || user.name || '',
      });
      
      // Default permission filter: if user has onlyMineView, enable the filter
      if (hseModule.permissions.includes('onlyMineView')) {
        setPermissionFilter({ onlyMine: true, userId: userId });
      }
    } catch (e) {
      console.error('Error fetching user info from MongoDB:', e);
    } finally {
      setLoadingUserInfo(false);
    }
  };

  useEffect(() => {
    fetchCurrentUserInfo();
  }, []);

  // Extract unique countries and projects from inspections
  useEffect(() => {
    if (savedInspections.length > 0) {
      const countries = new Set();
      const projects = new Set();

      savedInspections.forEach(inspection => {
        // Prefer step1 fields but fall back to top-level keys
        const country = pickFirstString(inspection?.step1?.country, inspection?.country);
        const project = pickFirstString(inspection?.step1?.project, inspection?.project);

        if (country) countries.add(country);
        if (project) projects.add(project);
      });

      setAvailableLocations(Array.from(countries).sort());
      setAvailableProjects(Array.from(projects).sort());
    } else {
      setAvailableLocations([]);
      setAvailableProjects([]);
    }
  }, [savedInspections]);

  // Apply permission filter when permissionFilter changes
  useEffect(() => {
    if (!permissionFilter.onlyMine) {
      setFilteredInspections(savedInspections);
      return;
    }
    const userId = permissionFilter.userId;
    const filtered = savedInspections.filter(item => item._adminId === userId || item.adminId === userId || item.userId === userId);
    setFilteredInspections(filtered);
  }, [permissionFilter, savedInspections]);

  // Filter inspections based on search query AND active filters
  useEffect(() => {
    let filtered = [...savedInspections];

    // Apply search filter
    if (searchQuery.trim() !== '') {
      const query = searchQuery.toLowerCase();
      filtered = filtered.filter(inspection => {
        // Document ID
        const documentId = pickFirstString(inspection?.step1?.selectedChecklist?.document_id, inspection?.selectedChecklist?.document_id, inspection?.pdfData?.document_id);
        if (documentId.toLowerCase().includes(query)) return true;

        // Request number / id
        const requestNumber = pickFirstString(inspection?.request, inspection?.id, inspection?.step1?.requestNumber, inspection?.pdfData?.request_number);
        if (requestNumber.toLowerCase().includes(query)) return true;

        // Location / Country / Project / Rig
        const location = pickFirstString(inspection?.step1?.country, inspection?.step1?.location, inspection?.location, inspection?.pdfData?.location);
        if (location.toLowerCase().includes(query)) return true;

        const project = pickFirstString(inspection?.step1?.project, inspection?.project, inspection?.pdfData?.project);
        if (project.toLowerCase().includes(query)) return true;

        const rig = pickFirstString(inspection?.rig, inspection?.step1?.rig);
        if (rig.toLowerCase().includes(query)) return true;

        // Date (day/month/year)
        const date = inspection.createdAt ? new Date(inspection.createdAt).toLocaleDateString('en-GB') : '';
        if (date.toLowerCase().includes(query)) return true;

        return false;
      });
    }

    // Apply country filter
    if (activeFilters.locations.length > 0) {
      filtered = filtered.filter(inspection => {
        const country = pickFirstString(inspection?.step1?.country, inspection?.country);
        return activeFilters.locations.includes(country);
      });
    }

    // Apply project filter
    if (activeFilters.projects.length > 0) {
      filtered = filtered.filter(inspection => {
        const project = pickFirstString(inspection?.step1?.project, inspection?.project);
        return activeFilters.projects.includes(project);
      });
    }

    // Apply date range filter
    if (activeFilters.dateRange !== 'all') {
      const now = new Date();
      const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());

      filtered = filtered.filter(inspection => {
        if (!inspection.createdAt) return false;
        const inspectionDate = new Date(inspection.createdAt);

        switch (activeFilters.dateRange) {
          case 'today':
            return inspectionDate >= today;
          case 'week':
            const weekAgo = new Date(today);
            weekAgo.setDate(weekAgo.getDate() - 7);
            return inspectionDate >= weekAgo;
          case 'month':
            const monthAgo = new Date(today);
            monthAgo.setMonth(monthAgo.getMonth() - 1);
            return inspectionDate >= monthAgo;
          default:
            return true;
        }
      });
    }

    setFilteredInspections(filtered);
  }, [searchQuery, savedInspections, activeFilters]);

  const loadSavedInspections = async (isInitial = false) => {
    try {
      if (isInitial) {
        setSyncModalVisible(true);
        setLoading(true);
      } else {
        setLoadingMore(true);
      }

      // Check network connectivity first
      const netState = await NetInfo.fetch();
      const connected = !!netState?.isConnected;

      // If offline, load from AsyncStorage
      if (!connected) {
        const stored = await AsyncStorage.getItem('hseInspections');
        if (stored) {
          const parsed = JSON.parse(stored);
          const paginated = parsed.slice(0, page * PAGE_SIZE);
          setSavedInspections(paginated);
          setHasMore(parsed.length > paginated.length);
          Toast.show({ type: 'info', text1: 'Offline Mode', text2: 'Loaded inspections from local storage.', position: 'top' });
        } else {
          setSavedInspections([]);
          setHasMore(false);
          Toast.show({ type: 'info', text1: 'Offline Mode', text2: 'No local inspections available.', position: 'top' });
        }
        return;
      }

      // When online, fetch from MongoDB
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'User not authenticated. Please log in again.', position: 'top' });
        return;
      }

      console.log('🔄 Fetching HSE inspections from MongoDB for user:', userId);

      // Get user permissions to determine which inspections to fetch
      const userResponse = await MongoAPI.getUserById(userId);
      if (!userResponse || !userResponse.success) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to fetch user permissions.', position: 'top' });
        return;
      }

      const user = userResponse.data;
      const hseModule = user.modules?.find(m => m.module === 'hseInspection') || { permissions: [] };
      const permissions = hseModule.permissions || [];

      let aggregated = [];

      // Fetch based on permissions
      if (permissions.includes('onlyMineView')) {
        // Fetch only user's own inspections
        const response = await HSEInspectionAPI.getHSEInspections(userId, { page: 1, limit: 1000 });
        if (response.success && response.data) {
          aggregated = response.data.map(item => ({
            id: item._id,
            _adminId: item.userId,
            ...item
          }));
        }
      } else if (permissions.includes('view')) {
        // Fetch all inspections and filter by country/project
        const response = await HSEInspectionAPI.getAllHSEInspections({ page: 1, limit: 1000 });
        console.log('📥 getAllHSEInspections response:', response);
        
        if (response.success && response.data) {
          console.log(`📊 Total inspections fetched: ${response.data.length}`);
          
          aggregated = response.data
            .filter(item => {
              console.log('🔍 Filtering item:', {
                id: item._id,
                country: item.step1?.country,
                project: item.step1?.project,
                userCountries: user.countries,
                userProjects: user.projects
              });
              
              // Include if country OR project matches, or if step1 data is missing (backward compatibility)
              return !item.step1 || 
                     user.countries.includes(item.step1?.country) || 
                     user.projects.includes(item.step1?.project);
            })
            .map(item => ({
              id: item._id,
              _adminId: item.userId,
              ...item
            }));
            
          console.log(`✅ Filtered to ${aggregated.length} inspections for user`);
        }
      } else {
        Toast.show({ type: 'error', text1: 'Permission Denied', text2: 'You do not have permission to view HSE inspections.', position: 'top' });
        setSavedInspections([]);
        setHasMore(false);
        return;
      }

      // Sort by date
      if (aggregated.length > 0) {
        aggregated.sort((a, b) => {
          const dateA = a.createdAt ? new Date(a.createdAt).getTime() : 0;
          const dateB = b.createdAt ? new Date(b.createdAt).getTime() : 0;
          return dateB - dateA;
        });
        
        // Save to AsyncStorage for offline use
        await AsyncStorage.setItem('hseInspections', JSON.stringify(aggregated));
        
        // Display paginated results
        const paginated = aggregated.slice(0, page * PAGE_SIZE);
        setSavedInspections(paginated);
        setHasMore(aggregated.length > paginated.length);
      } else {
        setSavedInspections([]);
        setHasMore(false);
        await AsyncStorage.setItem('hseInspections', JSON.stringify([]));
      }

      console.log(`✅ Loaded ${aggregated.length} HSE inspections from MongoDB`);

    } catch (error) {
      console.error('❌ Error loading HSE inspections from MongoDB:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load HSE inspections from MongoDB.', position: 'top' });
      
      // Try to load from cache
      const stored = await AsyncStorage.getItem('hseInspections');
      if (stored) {
        const parsed = JSON.parse(stored);
        setSavedInspections(parsed.slice(0, page * PAGE_SIZE));
        setHasMore(parsed.length > page * PAGE_SIZE);
      }
    } finally {
      setLoading(false);
      setSyncModalVisible(false);
      setLoadingMore(false);
    }
  };

  const handleLoadMore = () => {
    if (!loadingMore && hasMore && !loading) {
      setPage(prev => prev + 1);
    }
  };

  useEffect(() => {
    if (page > 1) {
      loadSavedInspections(false);
    }
  }, [page]);

  const handleStartInspection = () => {
    navigation.navigate('Hsemoduleheader43');
  };

  const handleGoBack = () => {
    navigation.navigate('MainApp');
  };

  const handleViewPDF = (inspection) => {
    if (inspection.pdfDownloadUrl) {
      navigation.navigate('HSEViewPDF', {
        pdfUrl: inspection.pdfDownloadUrl,
        onGoBack: () => {
          // Refresh the list when returning from PDF viewer
          loadSavedInspections();
        }
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not available for this inspection.',
        position: 'top',
      });
    }
  };

  const handleDeleteInspection = (inspection) => {
    Alert.alert(
      'Delete Inspection',
      'Are you sure you want to delete this inspection? This action cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const userId = await AsyncStorage.getItem('userUid');
              if (!userId) {
                Toast.show({
                  type: 'error',
                  text1: 'Error',
                  text2: 'User not authenticated.',
                  position: 'top',
                });
                return;
              }

              const response = await HSEInspectionAPI.deleteHSEInspection(userId, inspection.id);

              if (response.success) {
                Toast.show({
                  type: 'success',
                  text1: 'Success',
                  text2: 'Inspection deleted successfully.',
                  position: 'top',
                });
                
                // Refresh the list
                loadSavedInspections(true);
              } else {
                throw new Error(response.error || 'Failed to delete');
              }
            } catch (error) {
              console.error('❌ Error deleting inspection:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to delete inspection from MongoDB.',
                position: 'top',
              });
            }
          },
        },
      ],
      { cancelable: true }
    );
  };

  const handleDeleteAllReports = () => {
    // Note: Bulk delete is disabled for safety
    Toast.show({
      type: 'info',
      text1: 'Feature Disabled',
      text2: 'Bulk delete is disabled for safety. Please delete reports individually.',
      position: 'top',
    });
  };

  // User Info Modal to show permission details and toggle filter
  const UserInfoModal = ({ visible, onClose, userInfo }) => {
    return (
      <Modal
        visible={visible}
        transparent
        animationType="slide"
        presentationStyle="overFullScreen"
        onRequestClose={onClose}
      >
        <View style={styles.modalOverlayUnderHeader}>
          <View style={styles.modalContentCard}>
            <View style={{ flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center' }}>
              <Text style={styles.modalTitle}>Inspection Permissions</Text>
              <TouchableOpacity onPress={onClose} style={styles.modalCloseButton}>
                <Text style={styles.modalCloseText}>Close</Text>
              </TouchableOpacity>
            </View>

            {loadingUserInfo ? (
              <View style={{ paddingVertical: 20, alignItems: 'center' }}>
                <ActivityIndicator size="large" color="#015185" />
              </View>
            ) : (
              <>
                <View style={styles.modalRow}>
                  <Text style={styles.modalLabel}>Name</Text>
                  <Text style={styles.modalValue}>{userInfo.name || 'N/A'}</Text>
                </View>
                <View style={styles.modalRow}>
                  <Text style={styles.modalLabel}>Countries</Text>
                  <Text style={styles.modalValue}>{(userInfo.countries || []).join(', ') || 'N/A'}</Text>
                </View>
                <View style={styles.modalRow}>
                  <Text style={styles.modalLabel}>Projects</Text>
                  <Text style={styles.modalValue}>{(userInfo.projects || []).join(', ') || 'N/A'}</Text>
                </View>
                <View style={styles.modalRow}>
                  <Text style={styles.modalLabel}>Permissions</Text>
                  <Text style={styles.modalValue}>{(userInfo.inspectionPermissions || []).join(', ') || 'N/A'}</Text>
                </View>

                <View style={{ flexDirection: 'row', marginTop: 14 }}>
                  <TouchableOpacity
                    style={[styles.successBtn, { flex: 1 }]}
                    onPress={() => setPermissionFilter((prev) => ({ ...prev, onlyMine: !prev.onlyMine }))}
                  >
                    <Text style={styles.successBtnText}>{permissionFilter.onlyMine ? 'Show All Reports' : 'Show Only My Reports'}</Text>
                  </TouchableOpacity>
                </View>
              </>
            )}
          </View>
        </View>
      </Modal>
    );
  };

  const handleDropdownAction = async (action, inspection) => {
    try {
      switch (action) {
        case 'view-pdf':
          handleViewPDF(inspection);
          break;
        case 'delete':
          handleDeleteInspection(inspection);
          break;
        case 'delete-all':
          handleDeleteAllReports();
          break;
        default:
          break;
      }
    } catch (error) {
      console.error('Error processing action:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to process action.',
        position: 'top',
      });
    }
  };

  const toggleSearch = () => {
    setShowSearch(!showSearch);
    if (!showSearch) {
      setSearchQuery('');
    }
  };

  const clearSearch = () => {
    setSearchQuery('');
    setShowSearch(false);
  };

  const toggleLocationFilter = (location) => {
    setActiveFilters(prev => ({
      ...prev,
      locations: prev.locations.includes(location)
        ? prev.locations.filter(l => l !== location)
        : [...prev.locations, location]
    }));
  };

  const toggleProjectFilter = (project) => {
    setActiveFilters(prev => ({
      ...prev,
      projects: prev.projects.includes(project)
        ? prev.projects.filter(p => p !== project)
        : [...prev.projects, project]
    }));
  };

  const setDateRangeFilter = (range) => {
    setActiveFilters(prev => ({
      ...prev,
      dateRange: range
    }));
  };

  const clearAllFilters = () => {
    setActiveFilters({
      locations: [],
      projects: [],
      dateRange: 'all',
    });
    setFilterModalVisible(false);
  };

  const applyFilters = () => {
    setFilterModalVisible(false);
  };

  const hasActiveFilters = () => {
    return activeFilters.locations.length > 0 || 
           activeFilters.projects.length > 0 || 
           activeFilters.dateRange !== 'all';
  };

  const toSafeString = (value) => {
    if (value === null || value === undefined) return '';
    if (typeof value === 'string') return value;
    if (typeof value === 'number') return String(value);
    // Any unexpected object/array should not be rendered directly
    return '';
  };

  const pickFirstString = (...candidates) => {
    for (const c of candidates) {
      const s = toSafeString(c);
      if (s && s.trim() !== '') return s.trim();
    }
    return '';
  };

  const renderInspectionCard = ({ item, index }) => {
    const dropdownPosition = index > filteredInspections.length - 4 ? 'top' : 'auto';
    
    // Get request number from item data
    const requestNumber = pickFirstString(
      item?.requestNumber,
      item?.step1?.requestNumber,
      item?.request,
      item?.id
    ) || 'N/A';
    
    // Get fields from step1 data
    const location = pickFirstString(item?.step1?.country, item?.location, item?.country) || 'N/A';
    const project = pickFirstString(item?.step1?.project, item?.project) || 'N/A';
    const equipment = pickFirstString(
      item?.step1?.selectedEquipment?.equipmentName,
      item?.step1?.selectedEquipment?.equipmentNumber,
      item?.step1?.selectedEquipment?.entryNumber,
      item?.selectedEquipment?.equipmentName,
      item?.selectedEquipment?.equipmentNumber
    ) || 'N/A';
    
    const documentId = pickFirstString(
      item?.documentType,
      item?.step1?.selectedChecklist?.document_id,
      item?.selectedChecklist?.document_id
    ) || 'N/A';
    
    const createdAtText = toSafeString(item?.createdAt)
      ? new Date(item.createdAt).toLocaleString('en-GB', { 
          day: '2-digit', 
          month: '2-digit', 
          year: 'numeric', 
          hour: '2-digit', 
          minute: '2-digit' 
        })
      : 'Unknown Date';
    
    return (
      <View style={styles.inspectionCard}>
        <View style={styles.cardHeader}>
          <View style={styles.cardLeft}>
            <View style={styles.cardInfo}>
              <View style={styles.titleRow}>
                <Text style={styles.titleText}>Request: {requestNumber}</Text>
              </View>
            </View>
          </View>
          <View style={styles.dropdownContainer}>
            <Dropdown
              style={styles.dropdown}
              data={DROPDOWN_OPTIONS}
              labelField="label"
              valueField="value"
              value={null}
              onChange={async (option) => {
                await new Promise(resolve => setTimeout(resolve, 50));
                handleDropdownAction(option.value, item);
              }}
              renderRightIcon={() => <MaterialIcons name="more-vert" size={22} color="#333" />}
              renderItem={(option) => (
                <View style={styles.dropdownItem}>
                  {option.icon}
                  <Text
                    style={[
                      styles.dropdownText,
                      { color: (option.value === 'delete' || option.value === 'delete-all') ? '#e74c3c' : '#015185' },
                    ]}
                  >
                    {option.label}
                  </Text>
                </View>
              )}
              placeholder=""
              showsVerticalScrollIndicator={false}
              dropdownPosition={dropdownPosition}
              containerStyle={styles.dropdownContainerStyle}
              itemTextStyle={styles.dropdownItemText}
              selectedTextStyle={{ display: 'none' }}
            />
          </View>
        </View>
        {/* Details Rows */}
        <View style={styles.detailsList}>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Document ID</Text>
            <Text style={styles.detailValuePrimary}>{documentId}</Text>
          </View>
          <View style={styles.detailRow}> 
            <Text style={styles.detailLabel}>Location</Text>
            <Text style={styles.detailValuePrimary}>{location}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Project</Text>
            <Text style={styles.detailValuePrimary}>{project}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValuePrimary}>{equipment}</Text>
          </View>
          <View style={styles.detailRow}>
            <Text style={styles.detailLabel}>Date</Text>
            <Text style={styles.detailValuePrimary}>{createdAtText}</Text>
          </View>
        </View>

        {item.pdfDownloadUrl && (
          <View style={styles.pdfIndicator}>
            <MaterialIcons name="picture-as-pdf" size={16} color="#3481BC" />
            <Text style={styles.pdfText}>PDF Available</Text>
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Updated Header to match Team Employment Home */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={handleGoBack}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>HSE Inspection</Text>
          </View>
        <View style={styles.headerRight}>
          <TouchableOpacity onPress={loadSavedInspections} style={styles.refreshButton}>
            <Image
              source={require('../../Images/adminhome/equp/cloud.png')}
              style={styles.refreshIcon}
            />
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.iconButton, { marginLeft: 8 }]}
            onPress={() => setUserInfoModalVisible(true)}
          >
            <MaterialIcons name="lock" size={26} color="#fff" />
          </TouchableOpacity>
        </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by document ID, request number, location, rig, or date..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={[styles.filterButton, hasActiveFilters() && styles.filterButtonActive]}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons 
              name="filter-list" 
              size={width * 0.06} 
              color={hasActiveFilters() ? "#fff" : "#333"} 
            />
            {hasActiveFilters() && (
              <View style={styles.filterBadge}>
                <Text style={styles.filterBadgeText}>
                  {(activeFilters.locations.length + activeFilters.projects.length + (activeFilters.dateRange !== 'all' ? 1 : 0))}
                </Text>
              </View>
            )}
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <ScrollView style={styles.scrollView} showsVerticalScrollIndicator={false}>
        <View style={styles.content}>
          {/* Saved Inspections Section */}
          <View style={styles.sectionContainer}>
            <View style={styles.sectionHeader}>
              <View>
                <Text style={styles.sectionTitle}>
                  {searchQuery.length > 0 ? 'Search Results' : hasActiveFilters() ? 'Filtered Inspections' : 'Saved Inspections'}
                </Text>
                {(searchQuery.length > 0 || hasActiveFilters()) && (
                  <Text style={styles.resultCount}>
                    {filteredInspections.length} of {savedInspections.length} inspections
                  </Text>
                )}
              </View>
              {(searchQuery.length > 0 || hasActiveFilters()) && (
                <TouchableOpacity 
                  onPress={() => {
                    setSearchQuery('');
                    clearAllFilters();
                  }} 
                  style={styles.clearSearchButton}
                >
                  <MaterialIcons name="close" size={16} color="#666" />
                  <Text style={styles.clearSearchText}>Clear All</Text>
                </TouchableOpacity>
              )}
            </View>
            
            {/* Active Filters Display */}
            {hasActiveFilters() && (
              <View style={styles.activeFiltersContainer}>
                <Text style={styles.activeFiltersLabel}>Active Filters:</Text>
                <View style={styles.activeFiltersChips}>
                  {activeFilters.dateRange !== 'all' && (
                    <View style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>
                        {activeFilters.dateRange === 'today' ? 'Today' : 
                         activeFilters.dateRange === 'week' ? 'Last 7 Days' : 'Last 30 Days'}
                      </Text>
                    </View>
                  )}
                  {activeFilters.locations.map(location => (
                    <View key={location} style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>{location}</Text>
                    </View>
                  ))}
                  {activeFilters.projects.map(project => (
                    <View key={project} style={styles.activeFilterChip}>
                      <Text style={styles.activeFilterChipText}>{project}</Text>
                    </View>
                  ))}
                </View>
              </View>
            )}
            
            {loading ? (
              <View style={styles.loadingContainer}>
                <MaterialIcons name="hourglass-empty" size={48} color="#ccc" />
                <Text style={styles.loadingText}>Loading inspections...</Text>
              </View>
            ) : filteredInspections.length > 0 ? (
              <>
                <FlatList
                  data={filteredInspections}
                  keyExtractor={(item) => item.id}
                  renderItem={renderInspectionCard}
                  showsVerticalScrollIndicator={false}
                  scrollEnabled={false}
                  onEndReached={handleLoadMore}
                  onEndReachedThreshold={0.5}
                  ListFooterComponent={
                    loadingMore && hasMore ? (
                      <View style={styles.loadMoreContainer}>
                        <ActivityIndicator size="small" color="#3481BC" />
                        <Text style={styles.loadMoreText}>Loading more...</Text>
                      </View>
                    ) : null
                  }
                />
                {!hasMore && filteredInspections.length > 0 && (
                  <Text style={styles.endOfListText}>No more inspections to load</Text>
                )}
              </>
            ) : (
              <View style={styles.emptyContainer}>
                <View style={styles.emptyIconContainer}>
                  <MaterialIcons name={searchQuery.length > 0 ? "search" : "assignment"} size={64} color="#ccc" />
                </View>
                <Text style={styles.emptyTitle}>
                  {searchQuery.length > 0 ? 'No Search Results' : 'No Saved Inspections'}
                </Text>
                <Text style={styles.emptySubtext}>
                  {searchQuery.length > 0 
                    ? `No inspections found matching "${searchQuery}"`
                    : 'Start your first HSE inspection to see it here'
                  }
                </Text>
                {!searchQuery && (
                  <TouchableOpacity style={styles.emptyButton} onPress={handleStartInspection}>
                    <MaterialIcons name="add" size={20} color="#3481BC" />
                    <Text style={styles.emptyButtonText}>Start First Inspection</Text>
                  </TouchableOpacity>
                )}
              </View>
            )}
          </View>
        </View>
      </ScrollView>

      {/* Enhanced Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.buttonGradient}
        >
          <TouchableOpacity style={styles.startButton} onPress={handleStartInspection}>
            <MaterialIcons name="play-arrow" size={24} color="#fff" />
            <Text style={styles.buttonText}>Start New HSE Inspection</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <UserInfoModal
        visible={userInfoModalVisible}
        onClose={() => setUserInfoModalVisible(false)}
        userInfo={userInfo}
      />
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Loading Data" />
      
      {/* Filter Modal */}
      <Modal
        visible={filterModalVisible}
        animationType="slide"
        transparent={true}
        onRequestClose={() => setFilterModalVisible(false)}
      >
        <View style={styles.filterModalOverlay}>
          <View style={styles.filterModalContainer}>
            {/* Header */}
            <View style={styles.filterModalHeader}>
              <Text style={styles.filterModalTitle}>Filter Inspections</Text>
              <TouchableOpacity onPress={() => setFilterModalVisible(false)}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>

            <ScrollView style={styles.filterModalContent} showsVerticalScrollIndicator={false}>
              {/* Date Range Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Date Range</Text>
                <View style={styles.filterChipsContainer}>
                  {[
                    { label: 'All Time', value: 'all' },
                    { label: 'Today', value: 'today' },
                    { label: 'Last 7 Days', value: 'week' },
                    { label: 'Last 30 Days', value: 'month' },
                  ].map(option => (
                    <TouchableOpacity
                      key={option.value}
                      style={[
                        styles.filterChip,
                        activeFilters.dateRange === option.value && styles.filterChipActive
                      ]}
                      onPress={() => setDateRangeFilter(option.value)}
                    >
                      <Text style={[
                        styles.filterChipText,
                        activeFilters.dateRange === option.value && styles.filterChipTextActive
                      ]}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>

              {/* Country Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>
                  Countries {activeFilters.locations.length > 0 && `(${activeFilters.locations.length})`}
                </Text>
                {availableLocations.length > 0 ? (
                  <View style={styles.filterChipsContainer}>
                    {availableLocations.map(location => (
                      <TouchableOpacity
                        key={location}
                        style={[
                          styles.filterChip,
                          activeFilters.locations.includes(location) && styles.filterChipActive
                        ]}
                        onPress={() => toggleLocationFilter(location)}
                      >
                        <Text style={[
                          styles.filterChipText,
                          activeFilters.locations.includes(location) && styles.filterChipTextActive
                        ]}>
                          {location}
                        </Text>
                        {activeFilters.locations.includes(location) && (
                          <MaterialIcons name="check" size={16} color="#fff" style={{ marginLeft: 4 }} />
                        )}
                      </TouchableOpacity>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noFilterOptions}>No countries available</Text>
                )}
              </View>

              {/* Project Filter */}
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>
                  Projects {activeFilters.projects.length > 0 && `(${activeFilters.projects.length})`}
                </Text>
                {availableProjects.length > 0 ? (
                  <View style={styles.filterChipsContainer}>
                    {availableProjects.map(project => (
                      <TouchableOpacity
                        key={project}
                        style={[
                          styles.filterChip,
                          activeFilters.projects.includes(project) && styles.filterChipActive
                        ]}
                        onPress={() => toggleProjectFilter(project)}
                      >
                        <Text style={[
                          styles.filterChipText,
                          activeFilters.projects.includes(project) && styles.filterChipTextActive
                        ]}>
                          {project}
                        </Text>
                        {activeFilters.projects.includes(project) && (
                          <MaterialIcons name="check" size={16} color="#fff" style={{ marginLeft: 4 }} />
                        )}
                      </TouchableOpacity>
                    ))}
                  </View>
                ) : (
                  <Text style={styles.noFilterOptions}>No projects available</Text>
                )}
              </View>
            </ScrollView>

            {/* Footer Buttons */}
            <View style={styles.filterModalFooter}>
              <TouchableOpacity 
                style={styles.filterClearButton}
                onPress={clearAllFilters}
              >
                <Text style={styles.filterClearButtonText}>Clear All</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.filterApplyButton}
                onPress={applyFilters}
              >
                <LinearGradient
                  colors={['#3481BC', '#025383']}
                  style={styles.filterApplyGradient}
                >
                  <Text style={styles.filterApplyButtonText}>Apply Filters</Text>
                </LinearGradient>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#fff',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginHorizontal: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
    position: 'relative',
  },
  filterButtonActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  filterBadge: {
    position: 'absolute',
    top: -6,
    right: -6,
    backgroundColor: '#e74c3c',
    borderRadius: 10,
    minWidth: 20,
    height: 20,
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 4,
  },
  filterBadgeText: {
    color: '#fff',
    fontSize: 10,
    fontWeight: 'bold',
  },
  scrollView: {
    flex: 1,
  },
  content: {
    padding: 16,
  },
  statsContainer: {
    flexDirection: 'row',
    marginBottom: 24,
    gap: 12,
  },
  statCard: {
    flex: 1,
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    flexDirection: 'row',
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  statIconContainer: {
    width: 48,
    height: 48,
    borderRadius: 24,
    backgroundColor: '#f0f8ff',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 12,
  },
  statContent: {
    flex: 1,
  },
  statNumber: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: 2,
  },
  statLabel: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
  },
  sectionContainer: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 16,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  sectionHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  resultCount: {
    fontSize: 12,
    color: '#666',
    marginTop: 4,
    fontWeight: '500',
  },
  clearSearchButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 12,
    paddingVertical: 6,
    backgroundColor: '#f0f0f0',
    borderRadius: 16,
  },
  clearSearchText: {
    fontSize: 12,
    color: '#666',
    marginLeft: 4,
    fontWeight: '500',
  },
  activeFiltersContainer: {
    marginBottom: 16,
    paddingTop: 8,
  },
  activeFiltersLabel: {
    fontSize: 12,
    color: '#666',
    fontWeight: '600',
    marginBottom: 8,
  },
  activeFiltersChips: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 6,
  },
  activeFilterChip: {
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 12,
    backgroundColor: '#e3f2fd',
    borderWidth: 1,
    borderColor: '#015185',
  },
  activeFilterChipText: {
    fontSize: 11,
    color: '#015185',
    fontWeight: '600',
  },
  loadingContainer: {
    alignItems: 'center',
    padding: 40,
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
    marginTop: 12,
  },
  emptyContainer: {
    alignItems: 'center',
    padding: 40,
  },
  emptyIconContainer: {
    width: 120,
    height: 120,
    borderRadius: 60,
    backgroundColor: '#f8f9fa',
    alignItems: 'center',
    justifyContent: 'center',
    marginBottom: 16,
  },
  emptyTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 8,
  },
  emptySubtext: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginBottom: 24,
    lineHeight: 20,
  },
  emptyButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 20,
    paddingVertical: 12,
    borderRadius: 24,
    borderWidth: 1,
    borderColor: '#3481BC',
  },
  emptyButtonText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#3481BC',
    marginLeft: 8,
  },
  inspectionCard: {
    backgroundColor: '#f8f9fa',
    borderRadius: 12,
    padding: 20,
    marginBottom: 12,
    borderWidth: 1,
    borderColor: '#e9ecef',
    marginHorizontal: -4,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
  },
  cardLeft: {
    flexDirection: 'row',
    flex: 1,
  },
  cardIconContainer: {
    display: 'none',
  },
  cardInfo: {
    flex: 1,
  },
  cardTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 4,
  },
  requestSubtitle: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
    marginBottom: 8,
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 6,
    alignSelf: 'flex-start',
    borderWidth: 1,
    borderColor: '#e3f2fd',
  },
  cardDate: {
    fontSize: 15,
    color: '#666',
    marginBottom: 10,
  },
  cardDetails: {
    flexDirection: 'row',
    gap: 20,
  },
  detailItem: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  detailText: {
    fontSize: 13,
    color: '#666',
    marginLeft: 6,
  },
  dropdownContainer: {
    justifyContent: 'center',
    alignItems: 'center',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainerStyle: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  pdfIndicator: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    alignSelf: 'flex-start',
    marginTop: 12,
  },
  pdfText: {
    fontSize: 12,
    color: '#3481BC',
    fontWeight: '500',
    marginLeft: 4,
  },
  bottomContainer: {
    padding: 20,
    paddingBottom: Platform.OS === 'ios' ? 40 : 20,
    backgroundColor: '#fff',
    borderTopWidth: 1,
    borderTopColor: '#eee',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 5,
  },
  startButton: {
    borderRadius: 12,
    overflow: 'hidden',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  gradientButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    padding: 18,
  },
  startButtonText: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#fff',
    marginLeft: 10,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },

  // New styles for details list
  detailsList: {
    marginTop: 12,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#f1f1f1',
  },
  detailLabel: {
    color: '#777',
    fontSize: 13,
  },
  detailValuePrimary: {
    color: '#015185',
    fontSize: 13,
    fontWeight: '600',
  },

  // compact title row
  titleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    gap: 8,
  },
  titleText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    flexShrink: 1,
  },
  titleDate: {
    fontSize: 12,
    color: '#777',
    marginLeft: 8,
  },
  searchHeaderContainer: {
    flex: 1,
    alignItems: 'center',
  },
  searchHeaderTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#fff',
  },
  closeSearchButton: {
    padding: 8,
    borderRadius: 6,
    backgroundColor: 'rgba(255, 255, 255, 0.1)',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },

  // Filter Modal Styles
  filterModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'flex-end',
  },
  filterModalContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    maxHeight: height * 0.75,
  },
  filterModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  filterModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  filterModalContent: {
    maxHeight: height * 0.5,
    paddingHorizontal: 20,
  },
  filterSection: {
    marginTop: 20,
    marginBottom: 16,
  },
  filterSectionTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 12,
  },
  filterChipsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 8,
  },
  filterChip: {
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 20,
    backgroundColor: '#f8f9fa',
    borderWidth: 1,
    borderColor: '#dee2e6',
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  filterChipActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  filterChipText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  filterChipTextActive: {
    color: '#fff',
    fontWeight: '600',
  },
  noFilterOptions: {
    fontSize: 14,
    color: '#999',
    fontStyle: 'italic',
    textAlign: 'center',
    paddingVertical: 12,
  },
  filterModalFooter: {
    flexDirection: 'row',
    padding: 20,
    borderTopWidth: 1,
    borderTopColor: '#e9ecef',
    gap: 12,
  },
  filterClearButton: {
    flex: 1,
    paddingVertical: 14,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#dee2e6',
    alignItems: 'center',
    justifyContent: 'center',
  },
  filterClearButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#666',
  },
  filterApplyButton: {
    flex: 1,
    borderRadius: 8,
    overflow: 'hidden',
  },
  filterApplyGradient: {
    paddingVertical: 14,
    alignItems: 'center',
    justifyContent: 'center',
  },
  filterApplyButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#fff',
  },
  modalOverlayUnderHeader: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.25)',
    justifyContent: 'flex-start',
    paddingTop: TOTAL_HEADER_HEIGHT + 10,
    alignItems: 'center',
  },
  modalContentCard: {
    width: '92%',
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
  },
  modalRow: { flexDirection: 'row', justifyContent: 'space-between', paddingVertical: 8 },
  modalLabel: { color: '#666', fontWeight: '600' },
  modalValue: { color: '#015185', width: '60%', textAlign: 'right' },
  loadMoreContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 16,
    gap: 8,
  },
  loadMoreText: {
    fontSize: 14,
    color: '#666',
    marginLeft: 8,
  },
  endOfListText: {
    textAlign: 'center',
    color: '#999',
    fontSize: 13,
    paddingVertical: 16,
    fontStyle: 'italic',
  },

});

export default HSEInspectionHome;
