import { Platform } from 'react-native';

// MongoDB API base URL
const BASE_URL = Platform.select({
  ios: 'http://168.231.113.219:5017',
  android: 'http://168.231.113.219:5017',
  default: 'http://168.231.113.219:5017',
});

// Request timeout configuration
const TIMEOUT_MS = 3000; // 3 seconds for fast offline fallback

// Helper function to make API calls with timeout and better error handling
const apiCallWithTimeout = async (url, options = {}, timeoutMs = TIMEOUT_MS) => {
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), timeoutMs);

  try {
    const response = await fetch(url, {
      ...options,
      signal: controller.signal,
      headers: {
        'Accept': 'application/json',
        'Content-Type': 'application/json',
        ...options.headers,
      },
    });

    clearTimeout(timeoutId);

    // Check if response is ok
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`HTTP ${response.status}: ${errorText}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    clearTimeout(timeoutId);
    
    // Enhanced error messages for iOS
    if (error.name === 'AbortError') {
      throw new Error('Request timeout - please check your internet connection');
    } else if (error.message.includes('Network request failed')) {
      throw new Error('Network connection failed - please check your internet connection');
    } else {
      throw error;
    }
  }
};

const NotificationAPI = {
  /**
   * Get all notifications for a user
   */
  getAllNotifications: async (userId) => {
    try {
      console.log('📱 Fetching notifications for user:', userId);
      
      const result = await apiCallWithTimeout(`${BASE_URL}/api/notifications/user/${userId}`, {
        method: 'GET',
      });

      console.log(`✅ Fetched ${result.data?.length || 0} notifications`);
      return result;
    } catch (error) {
      console.error('❌ Error fetching notifications:', error);
      return { success: false, error: error.message, data: [] };
    }
  },

  /**
   * Get unread notification count
   */
  getUnreadCount: async (userId) => {
    try {
      const result = await apiCallWithTimeout(`${BASE_URL}/api/notifications/user/${userId}/unread-count`, {
        method: 'GET',
      });

      return result;
    } catch (error) {
      console.error('❌ Error fetching unread count:', error);
      return { success: false, unreadCount: 0 };
    }
  },

  /**
   * Mark notification as read
   */
  markAsRead: async (notificationId, userId) => {
    try {
      console.log('✅ Marking notification as read:', notificationId);
      
      const result = await apiCallWithTimeout(`${BASE_URL}/api/notifications/${notificationId}/read`, {
        method: 'PUT',
        body: JSON.stringify({ userId }),
      });

      return result;
    } catch (error) {
      console.error('❌ Error marking notification as read:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Mark all notifications as read
   */
  markAllAsRead: async (userId) => {
    try {
      console.log('✅ Marking all notifications as read for user:', userId);
      
      const result = await apiCallWithTimeout(`${BASE_URL}/api/notifications/user/${userId}/mark-all-read`, {
        method: 'PUT',
      });

      return result;
    } catch (error) {
      console.error('❌ Error marking all notifications as read:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Delete a notification
   */
  deleteNotification: async (notificationId, userId) => {
    try {
      console.log('🗑️ Deleting notification:', notificationId);
      
      const result = await apiCallWithTimeout(`${BASE_URL}/api/notifications/${notificationId}`, {
        method: 'DELETE',
        body: JSON.stringify({ userId }),
      });

      return result;
    } catch (error) {
      console.error('❌ Error deleting notification:', error);
      return { success: false, error: error.message };
    }
  },

  /**
   * Create a new notification (used by backend services)
   */
  createNotification: async (notificationData) => {
    try {
      console.log('📝 Creating notification:', notificationData.title);
      
      const result = await apiCallWithTimeout(`${BASE_URL}/api/notifications`, {
        method: 'POST',
        body: JSON.stringify(notificationData),
      });

      console.log('✅ Notification created successfully');
      return result;
    } catch (error) {
      console.error('❌ Error creating notification:', error);
      return { success: false, error: error.message };
    }
  },
};

export default NotificationAPI;

