// Notice Board MongoDB API Configuration
import { Platform } from 'react-native';

const BASE_URL = Platform.select({
  ios: 'http://168.231.113.219:5015',
  android: 'http://168.231.113.219:5015',
  default: 'http://168.231.113.219:5015',
});

// Request timeout configuration - 3 seconds for fast offline fallback
const TIMEOUT_MS = 3000; // 3 seconds

// Helper function to make API calls with timeout and better error handling
const apiCallWithTimeout = async (url, options = {}, timeoutMs = TIMEOUT_MS) => {
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), timeoutMs);

  try {
    const response = await fetch(url, {
      ...options,
      signal: controller.signal,
      headers: {
        'Accept': 'application/json',
        ...options.headers,
      },
    });

    clearTimeout(timeoutId);

    // Check if response is ok
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`HTTP ${response.status}: ${errorText}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    clearTimeout(timeoutId);
    
    // Enhanced error messages for iOS
    if (error.name === 'AbortError') {
      throw new Error('Request timeout - please check your internet connection');
    } else if (error.message.includes('Network request failed')) {
      throw new Error('Network connection failed - please check your internet connection');
    } else {
      throw error;
    }
  }
};

const NoticeAPI = {
  // Upload Image
  uploadImage: async (userId, imageFile) => {
    try {
      console.log('📤 Uploading image to MongoDB...');
      
      const formData = new FormData();
      formData.append('image', {
        uri: imageFile.uri,
        type: imageFile.type || 'image/jpeg',
        name: imageFile.name || 'notice-image.jpg'
      });

      const data = await apiCallWithTimeout(`${BASE_URL}/notices/upload-image/${userId}`, {
        method: 'POST',
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      }, 10000); // 10 seconds for image upload (shorter for faster feedback)

      console.log('📥 Upload image response:', data);
      return data;
    } catch (error) {
      console.error('❌ Error uploading image:', error);
      return { success: false, error: error.message };
    }
  },

  // Create Notice
  createNotice: async (noticeData) => {
    try {
      console.log('📝 Creating notice in MongoDB:', noticeData);
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(noticeData),
      });

      console.log('📥 Create notice response:', data);
      return data;
    } catch (error) {
      console.error('❌ Error creating notice:', error);
      return { success: false, error: error.message };
    }
  },

  // Get All Notices for User
  getAllNotices: async (userId) => {
    try {
      console.log('📋 Fetching all notices for user:', userId);
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices/all/${userId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      console.log(`📥 Found ${data.data?.length || 0} notices`);
      return data;
    } catch (error) {
      console.error('❌ Error fetching notices:', error);
      return { success: false, error: error.message, data: [] };
    }
  },

  // Get Unread Count
  getUnreadCount: async (userId) => {
    try {
      console.log('📊 Getting unread count for user:', userId);
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices/unread-count/${userId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      console.log(`📥 Unread count: ${data.unreadCount}`);
      return data;
    } catch (error) {
      console.error('❌ Error getting unread count:', error);
      return { success: false, error: error.message, unreadCount: 0 };
    }
  },

  // Mark Notice as Read
  markAsRead: async (noticeId, userId) => {
    try {
      console.log('👁️ Marking notice as read:', { noticeId, userId });
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices/mark-read/${noticeId}/${userId}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      console.log('📥 Mark as read response:', data);
      return data;
    } catch (error) {
      console.error('❌ Error marking notice as read:', error);
      return { success: false, error: error.message };
    }
  },

  // Get Single Notice
  getNotice: async (noticeId, userId) => {
    try {
      console.log('📄 Fetching notice:', noticeId);
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices/${noticeId}?userId=${userId}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      console.log('📥 Notice data:', data);
      return data;
    } catch (error) {
      console.error('❌ Error fetching notice:', error);
      return { success: false, error: error.message };
    }
  },

  // Delete Notice
  deleteNotice: async (noticeId, userId) => {
    try {
      console.log('🗑️ Deleting notice:', { noticeId, userId });
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices/${noticeId}/${userId}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      console.log('📥 Delete notice response:', data);
      return data;
    } catch (error) {
      console.error('❌ Error deleting notice:', error);
      return { success: false, error: error.message };
    }
  },

  // Update Notice
  updateNotice: async (noticeId, userId, noticeData) => {
    try {
      console.log('✏️ Updating notice:', { noticeId, userId });
      
      const data = await apiCallWithTimeout(`${BASE_URL}/notices/${noticeId}/${userId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(noticeData),
      });

      console.log('📥 Update notice response:', data);
      return data;
    } catch (error) {
      console.error('❌ Error updating notice:', error);
      return { success: false, error: error.message };
    }
  },
};

export default NoticeAPI;

