const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const bcrypt = require('bcryptjs');
const { MongoClient } = require('mongodb');

const app = express();
const PORT = process.env.WEBPROADMIN_PORT || 5050;

// MongoDB connection
const MONGO_URI = 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@api.titandrillingzm.com:27017/titan_drilling?authSource=admin';
const DB_NAME = 'titan_drilling';
const COLLECTION_NAME = 'WebProadmin';

let db;
let webProAdminCollection;

// Middleware
app.use(cors());
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));

// Connect to MongoDB
async function connectDB() {
    try {
        const client = await MongoClient.connect(MONGO_URI, {
            useNewUrlParser: true,
            useUnifiedTopology: true
        });
        db = client.db(DB_NAME);
        webProAdminCollection = db.collection(COLLECTION_NAME);
        console.log(`✅ Connected to MongoDB: ${DB_NAME}.${COLLECTION_NAME}`);
        
        // Create initial admin user if collection is empty
        await createInitialUser();
    } catch (error) {
        console.error('❌ MongoDB connection error:', error);
        process.exit(1);
    }
}

// Create initial admin user from ProWebAdmindot.json
async function createInitialUser() {
    try {
        const count = await webProAdminCollection.countDocuments();
        
        if (count === 0) {
            console.log('📝 Creating initial Web Pro Admin user...');
            
            const initialUser = {
                "_id": "user_1761833101603_9zsp1be58",
                "uid": "user_1761833101603_9zsp1be58",
                "name": "Amit.coder",
                "email": "admin@titandrillingzm.com",
                "role": "WebProAdmin",
                "employeeNumber": "TYU7777989",
                "department": "UIN87979",
                "countries": ["DRC", "Zambia"],
                "modules": [
                    { "module": "Equipments", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "Documents", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "teamEmploy", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "inspectionReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "dvirReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "checklist", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "incidentReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "dailyReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "requestMaintenance", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "jobcard", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "serviceSchedule", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "fuelLog", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "logbook", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "hrForm", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "hrApplications", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "teamKpi", "permissions": ["create", "edit", "delete", "view", "update"] }
                ],
                "loginStatus": false,
                "lastLogin": new Date(),
                "password": "admin@titandrillingzm.comA1",
                "hashedPassword": "$2b$10$CUWpUf3by/2GiP5Lx0GMB.qKOZT4AgHj44evqypFceEU/sE3KN2ge",
                "customUID": "user_1761833101603_9zsp1be58",
                "nrcIdNumber": "YIOO888089",
                "mobile": "86787678678",
                "projects": ["Lubumbashi", "Musompo", "IME", "Kamoa", "Kansanshi", "Kalumbila", "Kimteto", "Kobold", "FQM Exploration", "Mimosa"],
                "createdAt": new Date(),
                "updatedAt": new Date()
            };
            
            await webProAdminCollection.insertOne(initialUser);
            console.log('✅ Initial Web Pro Admin user created successfully!');
            console.log('📧 Email: admin@titandrillingzm.com');
            console.log('🔑 Password: admin@titandrillingzm.comA1');
        } else {
            console.log('✅ Web Pro Admin users already exist');
        }
    } catch (error) {
        console.error('❌ Error creating initial user:', error);
    }
}

// API: Login
app.post('/api/webproadmin/login', async (req, res) => {
    try {
        const { email, password } = req.body;
        
        if (!email || !password) {
            return res.status(400).json({
                success: false,
                message: 'Email and password are required'
            });
        }
        
        // Find user by email
        const user = await webProAdminCollection.findOne({ email: email });
        
        if (!user) {
            return res.status(401).json({
                success: false,
                message: 'Invalid email or password'
            });
        }
        
        // Compare password
        const isPasswordValid = await bcrypt.compare(password, user.hashedPassword);
        
        if (!isPasswordValid) {
            return res.status(401).json({
                success: false,
                message: 'Invalid email or password'
            });
        }
        
        // Update login status and last login
        await webProAdminCollection.updateOne(
            { _id: user._id },
            { 
                $set: { 
                    loginStatus: true, 
                    lastLogin: new Date() 
                } 
            }
        );
        
        // Remove sensitive data before sending response
        delete user.password;
        delete user.hashedPassword;
        
        res.json({
            success: true,
            message: 'Login successful',
            user: user
        });
        
    } catch (error) {
        console.error('❌ Login error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Logout
app.post('/api/webproadmin/logout', async (req, res) => {
    try {
        const { uid } = req.body;
        
        if (!uid) {
            return res.status(400).json({
                success: false,
                message: 'User ID is required'
            });
        }
        
        await webProAdminCollection.updateOne(
            { uid: uid },
            { $set: { loginStatus: false } }
        );
        
        res.json({
            success: true,
            message: 'Logout successful'
        });
        
    } catch (error) {
        console.error('❌ Logout error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Get user profile
app.get('/api/webproadmin/profile/:uid', async (req, res) => {
    try {
        const { uid } = req.params;
        
        const user = await webProAdminCollection.findOne({ uid: uid });
        
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }
        
        // Remove sensitive data
        delete user.password;
        delete user.hashedPassword;
        
        res.json({
            success: true,
            user: user
        });
        
    } catch (error) {
        console.error('❌ Get profile error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Get all Web Pro Admins
app.get('/api/webproadmin/all', async (req, res) => {
    try {
        const users = await webProAdminCollection.find({}).toArray();
        
        // Remove sensitive data
        users.forEach(user => {
            delete user.password;
            delete user.hashedPassword;
        });
        
        res.json({
            success: true,
            count: users.length,
            users: users
        });
        
    } catch (error) {
        console.error('❌ Get all users error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Create new Web Pro Admin
app.post('/api/webproadmin/create', async (req, res) => {
    try {
        const userData = req.body;
        
        // Check if email already exists
        const existingUser = await webProAdminCollection.findOne({ email: userData.email });
        
        if (existingUser) {
            return res.status(400).json({
                success: false,
                message: 'Email already exists'
            });
        }
        
        // Hash password
        const hashedPassword = await bcrypt.hash(userData.password, 10);
        
        const newUser = {
            ...userData,
            hashedPassword: hashedPassword,
            loginStatus: false,
            createdAt: new Date(),
            updatedAt: new Date()
        };
        
        await webProAdminCollection.insertOne(newUser);
        
        // Remove sensitive data
        delete newUser.password;
        delete newUser.hashedPassword;
        
        res.json({
            success: true,
            message: 'Web Pro Admin created successfully',
            user: newUser
        });
        
    } catch (error) {
        console.error('❌ Create user error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// Health check
app.get('/health', (req, res) => {
    res.json({
        status: 'OK',
        service: 'Web Pro Admin Login',
        port: PORT,
        timestamp: new Date()
    });
});

// Start server
connectDB().then(() => {
    app.listen(PORT, () => {
        console.log(`🔐 Web Pro Admin Login API running on port ${PORT}`);
        console.log(`📡 http://api.titandrillingzm.com:${PORT}`);
    });
});

