const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require('cors');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
require('dotenv').config();

const app = express();
const PORT = process.env.INCIDENT_REPORT_PORT || 5007;
// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadPath = '/root/node-mongo-api/uploads/documents/IncidentReportsdocx';
    // Create directory if it doesn't exist
    if (!fs.existsSync(uploadPath)) {
      fs.mkdirSync(uploadPath, { recursive: true });
    }
    cb(null, uploadPath);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ 
  storage: storage,
  limits: {
    fileSize: 50 * 1024 * 1024 // 50MB limit
  },
  fileFilter: function (req, file, cb) {
    // Allow PDF and image files
    const allowedTypes = /jpeg|jpg|png|gif|pdf/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    
    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only PDF and image files are allowed!'));
    }
  }
});

// Middleware
app.use(cors({
  origin: '*',
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Serve uploaded files statically
app.use('/uploads', express.static('/root/node-mongo-api/uploads'));

// Request logging middleware
app.use((req, res, next) => {
  console.log(`📡 ${req.method} ${req.url} - ${new Date().toISOString()}`);
  next();
});

// Connect to MongoDB with better error handling
const connectDB = async () => {
  try {
    // Using shared MongoDB connection from dbConnection.js
    console.log('✅ MongoDB connected successfully for Incident Report Handler!');
  } catch (err) {
    console.error('❌ MongoDB connection error:', err);
    console.error('⚠️ Will retry connection...');
    setTimeout(() => {
      connectDB();
    }, 5000);
  }
};

connectDB();

// Handle MongoDB connection events
mongoose.connection.on('error', (err) => {
  console.error('❌ MongoDB connection error:', err);
});

mongoose.connection.on('disconnected', () => {
  console.log('⚠️ MongoDB disconnected');
});

mongoose.connection.on('connected', () => {
  console.log('✅ MongoDB connected');
});

// Define Incident Report Schema
const IncidentReportSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  id: { type: String, required: true },
  adminUid: { type: String, required: true },
  incidentNumber: { type: String, required: true },
  incidentDate: { type: Date },
  incidentArea: String,
  severity: String,
  incidentCategory: String,
  country: String,
  project: String,
  selectedIncidents: Object,
  involvesEquipment: { type: Boolean, default: false },
  vehicleId: String,
  vehicleName: String,
  personInvolved: String,
  coordinates: String,
  gpsAddress: String, // Added GPS address field
  otherIncidentNote: String,
  createdAt: { type: Date, default: Date.now },
  isArchived: { type: Boolean, default: false },
  pdfDownloadUrl: String,
  step1: Object,
  step2: Object,
  uploadTimestamp: { type: Date, default: Date.now }
}, {
  strict: false,
  collection: 'IncidentReports'
});

// Indexes for fast searching
IncidentReportSchema.index({ adminUid: 1 });
IncidentReportSchema.index({ incidentNumber: 1 });
IncidentReportSchema.index({ country: 1 });
IncidentReportSchema.index({ project: 1 });
IncidentReportSchema.index({ isArchived: 1 });
IncidentReportSchema.index({ createdAt: -1 });

const IncidentReport = mongoose.model('IncidentReport', IncidentReportSchema);

// Health Check Route
app.get('/', (req, res) => {
  res.json({
    status: 'Incident Report Server is running',
    port: PORT,
    collection: 'IncidentReports'
  });
});

// Utility: sanitize incoming report payload so Mongo never stores local device paths
const isWebUrl = (v) => typeof v === 'string' && (v.startsWith('http://') || v.startsWith('https://'));
const isLocalPath = (v) => typeof v === 'string' && (v.startsWith('file://') || v.startsWith('/storage') || v.startsWith('/data'));

const sanitizeReportData = (data) => {
  if (!data || typeof data !== 'object') return data;

  const clone = JSON.parse(JSON.stringify(data));

  // Remove any locally stored PDF paths
  if (clone.step2 && clone.step2.pdfLocalPath) {
    delete clone.step2.pdfLocalPath;
  }
  if (clone.pdfLocalPath) delete clone.pdfLocalPath;

  // If pdfDownloadUrl is present but is a local path, drop it (it should be set by upload API)
  if (clone.pdfDownloadUrl && !isWebUrl(clone.pdfDownloadUrl)) delete clone.pdfDownloadUrl;
  if (clone.step2 && clone.step2.pdfDownloadUrl && !isWebUrl(clone.step2.pdfDownloadUrl)) delete clone.step2.pdfDownloadUrl;

  // Clean incidentImages: keep only web URLs
  if (clone.step2 && Array.isArray(clone.step2.incidentImages)) {
    clone.step2.incidentImages = clone.step2.incidentImages.map((u) => (isWebUrl(u) ? u : null));
  }

  return clone;
};

// Get all incident reports for a specific user (respects permissions)
app.get('/incident-reports/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    console.log(`📊 Fetching incident reports for admin: ${adminUid}`);
    
    // Fetch user permissions from AdminUsers collection
    let userPermissions = null;
    try {
      const AdminUser = mongoose.connection.db.collection('AdminUsers');
      const user = await AdminUser.findOne({ uid: adminUid });
      
      if (user && user.modules) {
        const incidentModule = user.modules.find(m => m.module === 'incidentReport');
        userPermissions = incidentModule?.permissions || [];
        console.log(`👤 User permissions:`, userPermissions);
      }
    } catch (permErr) {
      console.warn('⚠️ Could not fetch user permissions:', permErr.message);
    }
    
    // Build query based on permissions
    let query = {};
    
    // If user has "onlyMineView" permission, show only their reports
    if (userPermissions && userPermissions.includes('onlyMineView')) {
      console.log('🔒 onlyMineView enabled - filtering by adminUid');
      query.adminUid = adminUid;
    } else {
      console.log('✅ Full access - returning all reports (frontend will filter by country/project)');
      // No filter - return all reports (frontend handles country/project filtering)
    }
    
    const reports = await IncidentReport.find(query)
      .sort({ createdAt: -1 })
      .lean();

    console.log(`✅ Successfully fetched ${reports.length} incident reports`);
    
    res.json({
      success: true,
      count: reports.length,
      data: reports
    });
  } catch (err) {
    console.error('❌ Error fetching incident reports:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get all incident reports (for admin with full access)
app.get('/incident-reports', async (req, res) => {
  try {
    console.log('📊 Fetching all incident reports...');
    
    const reports = await IncidentReport.find({})
      .sort({ createdAt: -1 })
      .lean();

    console.log(`✅ Successfully fetched ${reports.length} incident reports`);
    
    res.json({
      success: true,
      count: reports.length,
      data: reports
    });
  } catch (err) {
    console.error('❌ Error fetching incident reports:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get single incident report by ID
app.get('/incident-reports/:adminUid/:reportId', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    console.log(`📄 Fetching incident report ${reportId} (requested by admin ${adminUid})`);
    
    // Get ANY report regardless of who created it (no adminUid check)
    const report = await IncidentReport.findOne({ 
      _id: reportId
    }).lean();

    if (!report) {
      return res.status(404).json({ 
        success: false,
        error: 'Incident report not found' 
      });
    }

    console.log(`✅ Successfully fetched incident report ${reportId}`);
    
    res.json({
      success: true,
      data: report
    });
  } catch (err) {
    console.error('❌ Error fetching incident report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Create or update incident report (idempotent)
app.post('/incident-reports', async (req, res) => {
  try {
    const reportData = sanitizeReportData(req.body);
    console.log(`📝 Upsert incident report: ${reportData.incidentNumber}`);

    // Normalize IDs
    if (!reportData._id) {
      reportData._id = reportData.id || `IR_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    }
    if (!reportData.id) {
      reportData.id = reportData._id;
    }

    // Upsert to avoid E11000 duplicate errors on retry
    const upserted = await IncidentReport.findOneAndUpdate(
      { _id: reportData._id },
      { $set: reportData },
      { new: true, upsert: true, setDefaultsOnInsert: true }
    );

    console.log(`✅ Upserted incident report: ${reportData.incidentNumber}`);

    res.status(201).json({
      success: true,
      message: 'Incident report created/updated successfully',
      data: upserted
    });
  } catch (err) {
    console.error('❌ Error creating/updating incident report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Update incident report
app.put('/incident-reports/:adminUid/:reportId', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    const updateData = sanitizeReportData(req.body);
    console.log(`✏️ Updating incident report ${reportId} (requested by admin ${adminUid})`);
    
    // Update ANY report regardless of who created it (no adminUid check)
    const updatedReport = await IncidentReport.findOneAndUpdate(
      { _id: reportId },
      { $set: updateData },
      { new: true, runValidators: true }
    );

    if (!updatedReport) {
      return res.status(404).json({ 
        success: false,
        error: 'Incident report not found' 
      });
    }

    console.log(`✅ Successfully updated incident report ${reportId}`);
    
    res.json({
      success: true,
      message: 'Incident report updated successfully',
      data: updatedReport
    });
  } catch (err) {
    console.error('❌ Error updating incident report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Archive/Unarchive incident report
app.patch('/incident-reports/:adminUid/:reportId/archive', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    const { isArchived } = req.body;
    console.log(`📦 ${isArchived ? 'Archiving' : 'Unarchiving'} incident report ${reportId} (requested by admin ${adminUid})`);
    
    // Update ANY report regardless of who created it (no adminUid check)
    const updatedReport = await IncidentReport.findOneAndUpdate(
      { _id: reportId },
      { $set: { isArchived } },
      { new: true }
    );

    if (!updatedReport) {
      return res.status(404).json({ 
        success: false,
        error: 'Incident report not found' 
      });
    }

    console.log(`✅ Successfully ${isArchived ? 'archived' : 'unarchived'} incident report ${reportId}`);
    
    res.json({
      success: true,
      message: `Incident report ${isArchived ? 'archived' : 'unarchived'} successfully`,
      data: updatedReport
    });
  } catch (err) {
    console.error(`❌ Error ${req.body.isArchived ? 'archiving' : 'unarchiving'} incident report:`, err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Delete incident report
app.delete('/incident-reports/:adminUid/:reportId', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    console.log(`🗑️ Deleting incident report ${reportId} (requested by admin ${adminUid})`);
    
    // Delete ANY report regardless of who created it (no adminUid check)
    const deletedReport = await IncidentReport.findOneAndDelete({ 
      _id: reportId
    });

    if (!deletedReport) {
      return res.status(404).json({ 
        success: false,
        error: 'Incident report not found' 
      });
    }

    console.log(`✅ Successfully deleted incident report ${reportId} (was created by ${deletedReport.adminUid})`);
    
    res.json({
      success: true,
      message: 'Incident report deleted successfully',
      data: deletedReport
    });
  } catch (err) {
    console.error('❌ Error deleting incident report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Bulk upload incident reports
app.post('/incident-reports/bulk-upload', async (req, res) => {
  try {
    const reports = req.body;
    console.log(`📦 Bulk uploading ${reports.length} incident reports`);
    
    if (!Array.isArray(reports) || reports.length === 0) {
      return res.status(400).json({ 
        success: false,
        error: 'Invalid data format. Expected array of reports.' 
      });
    }

    // Ensure each report has required IDs
    const reportsToInsert = reports.map(report => {
      if (!report._id) {
        report._id = report.id || `IR_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      }
      if (!report.id) {
        report.id = report._id;
      }
      return report;
    });

    const result = await IncidentReport.bulkWrite(
      reportsToInsert.map(report => ({
        updateOne: {
          filter: { _id: report._id },
          update: { $set: report },
          upsert: true
        }
      }))
    );

    console.log(`✅ Bulk upload completed: ${result.upsertedCount} inserted, ${result.modifiedCount} updated`);
    
    res.json({
      success: true,
      message: 'Bulk upload completed successfully',
      inserted: result.upsertedCount,
      updated: result.modifiedCount,
      total: reports.length
    });
  } catch (err) {
    console.error('❌ Error in bulk upload:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Upload PDF for incident report
app.post('/incident-reports/:adminUid/:reportId/upload-pdf', upload.single('pdf'), async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const pdfUrl = `http://api.titandrillingzm.com:5007/uploads/documents/IncidentReportsdocx/${req.file.filename}`;
    
    // Update the report with PDF URL
    const updatedReport = await IncidentReport.findOneAndUpdate(
      { _id: reportId, adminUid },
      { $set: { pdfDownloadUrl: pdfUrl } },
      { new: true }
    );

    if (!updatedReport) {
      return res.status(404).json({
        success: false,
        error: 'Incident report not found'
      });
    }

    console.log(`✅ PDF uploaded for incident report ${reportId}`);
    
    res.json({
      success: true,
      message: 'PDF uploaded successfully',
      pdfUrl: pdfUrl,
      data: updatedReport
    });
  } catch (err) {
    console.error('❌ Error uploading PDF:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Upload images for incident report
app.post('/incident-reports/:adminUid/:reportId/upload-images', upload.array('images', 9), async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    
    if (!req.files || req.files.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'No image files uploaded'
      });
    }

    const imageUrls = req.files.map(file => 
      `http://api.titandrillingzm.com:5007/uploads/documents/IncidentReportsdocx/${file.filename}`
    );
    
    // Update the report with image URLs
    const updatedReport = await IncidentReport.findOneAndUpdate(
      { _id: reportId, adminUid },
      { $set: { 'step2.incidentImages': imageUrls } },
      { new: true }
    );

    if (!updatedReport) {
      return res.status(404).json({
        success: false,
        error: 'Incident report not found'
      });
    }

    console.log(`✅ ${imageUrls.length} images uploaded for incident report ${reportId}`);
    
    res.json({
      success: true,
      message: 'Images uploaded successfully',
      imageUrls: imageUrls,
      data: updatedReport
    });
  } catch (err) {
    console.error('❌ Error uploading images:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Get incident report statistics
app.get('/incident-reports/stats/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    console.log(`📊 Fetching statistics for admin ${adminUid}`);
    
    const stats = await IncidentReport.aggregate([
      { $match: { adminUid } },
      {
        $group: {
          _id: null,
          total: { $sum: 1 },
          archived: { $sum: { $cond: ['$isArchived', 1, 0] } },
          active: { $sum: { $cond: ['$isArchived', 0, 1] } }
        }
      }
    ]);

    const result = stats.length > 0 ? stats[0] : { total: 0, archived: 0, active: 0 };
    
    console.log(`✅ Statistics fetched successfully`);
    
    res.json({
      success: true,
      data: result
    });
  } catch (err) {
    console.error('❌ Error fetching statistics:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Start server
app.listen(PORT, () => {
  console.log(`🚀 Incident Report Handler server running on port ${PORT}`);
});

module.exports = app;

