const express = require("express");
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require("cors");
const http = require("http");
const https = require("https");
const { Server } = require("socket.io");
const nodemailer = require('nodemailer');
const path = require('path');
require('dotenv').config();

// MongoDB connection string from environment variables
const MONGO_HOST = process.env.MONGO_HOST || 'api.titandrillingzm.com';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'mrdThsSnATwD231hbOPmnhj';
const MONGO_DATABASE = process.env.MONGO_DATABASE || 'titan_drilling';
const MONGO_AUTH_SOURCE = process.env.MONGO_AUTH_SOURCE || 'admin';

const MONGO_URI = `mongodb://${MONGO_USER}:${MONGO_PASSWORD}@${MONGO_HOST}:${MONGO_PORT}/${MONGO_DATABASE}?authSource=${MONGO_AUTH_SOURCE}`;

// Server configuration
const PORT = process.env.EQUIPMENT_PORT || 5001;
const SERVER_IP = process.env.SERVER_IP || 'api.titandrillingzm.com';
const CORS_ORIGIN = process.env.CORS_ORIGIN || '*';
const BODY_PARSER_LIMIT = process.env.BODY_PARSER_LIMIT || '100mb';

// Create Express app and HTTP server
const app = express();
const server = http.createServer(app);

// Create Socket.IO server
const io = new Server(server, {
  cors: {
    origin: CORS_ORIGIN,
    methods: ['GET', 'POST', 'PUT', 'DELETE'],
    credentials: true,
  },
  transports: ['websocket', 'polling'],
});

// Middleware
app.use(express.json({ limit: BODY_PARSER_LIMIT }));
app.use(express.urlencoded({ limit: BODY_PARSER_LIMIT, extended: true }));
app.use(cors({
  origin: CORS_ORIGIN,
  methods: ['GET', 'POST', 'PUT', 'DELETE'],
  allowedHeaders: ['Content-Type'],
}));

// Socket.IO Connection Handler
io.on('connection', (socket) => {
  console.log('✅ Client connected:', socket.id);
  
  socket.on('disconnect', () => {
    console.log('❌ Client disconnected:', socket.id);
  });
});

// Connect to MongoDB
// Using shared MongoDB connection from dbConnection.js
console.log('✅ Using shared MongoDB connection');

// Equipment Schema
const EquipmentSchema = new mongoose.Schema({
  id: String,
  equipmentName: String,
  mainCategory: String,
  category: String,
  equipmentType: String,
  status: String,
  country: String,
  firebaseKey: String,
  make: String,
  model: String,
  year: String,
  serialNumber: String,
  location: String,
  description: String,
  specifications: mongoose.Schema.Types.Mixed,
  customName: String,
  databaseName: String,
  uploadedAt: { type: Date, default: Date.now },
  uploadedFrom: { type: String, default: "React Native App" },
  mongoUploadVersion: { type: String, default: "1.0.0" },
  firebaseData: mongoose.Schema.Types.Mixed,
}, { 
  strict: false,
  timestamps: true 
});

const Equipment = mongoose.model("Equipment", EquipmentSchema);

// GlobalUser Model - reuse if already exists
let GlobalUser;
try {
  // Try to get existing model
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ Using existing GlobalUser model');
} catch (error) {
  // If model doesn't exist, create it
  const GlobalUserSchema = new mongoose.Schema({
    _id: String,
    uid: String,
    name: String,
    email: String,
    role: String,
    fcmTokens: [{
      token: String,
      deviceId: String,
      platform: String,
      updatedAt: Date
    }],
    fcmToken: String,
  }, { strict: false, collection: 'GlobalUsers' });
  
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
});

// Helper function to send push notifications to admins and supervisors
const sendEquipmentNotifications = async (equipmentData, createdByName) => {
  try {
    console.log('🔔 Sending equipment notifications to admins and supervisors...');
    
    // Get all users with role ProAdmin or Supervisor
    const users = await GlobalUser.find({
      role: { $in: ['ProAdmin', 'Supervisor'] },
      $or: [
        { fcmToken: { $exists: true, $ne: null } },
        { 'fcmTokens.0': { $exists: true } }
      ]
    }).select('name email role fcmToken fcmTokens');
    
    if (users.length === 0) {
      console.log('⚠️ No admins/supervisors with FCM tokens found');
      return { success: false, error: 'No recipients found' };
    }
    
    console.log(`📱 Found ${users.length} admin(s)/supervisor(s) with FCM tokens`);
    
    // Collect all FCM tokens
    const tokens = [];
    users.forEach(user => {
      if (user.fcmTokens && user.fcmTokens.length > 0) {
        user.fcmTokens.forEach(tokenObj => {
          if (tokenObj.token && !tokens.includes(tokenObj.token)) {
            tokens.push(tokenObj.token);
          }
        });
      }
      if (user.fcmToken && !tokens.includes(user.fcmToken)) {
        tokens.push(user.fcmToken);
      }
    });
    
    if (tokens.length === 0) {
      console.log('⚠️ No FCM tokens found');
      return { success: false, error: 'No FCM tokens' };
    }
    
    // Send push notification via MongoDB Push Notification Service
    const notificationData = {
      roles: ['ProAdmin', 'Supervisor'],
      title: '🚜 New Equipment Added',
      body: `${equipmentData.equipmentName || equipmentData.mainCategory} has been added\n\nAdded by: ${createdByName}`,
      data: {
        type: 'equipment_added',
        action: 'view_equipment',
        equipmentId: equipmentData.id,
        equipmentName: equipmentData.equipmentName || 'New Equipment',
        mainCategory: equipmentData.mainCategory,
        country: equipmentData.country,
        createdBy: createdByName,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '🚜 New Equipment Added',
              body: `${equipmentData.equipmentName || equipmentData.mainCategory} has been added\n\nAdded by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'equipment_notifications',
        }
      }
    };
    
    // Make HTTP request using native http module
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5016,
        path: '/push-notifications/send-to-roles',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let responseData = '';
        
        res.on('data', (chunk) => {
          responseData += chunk;
        });
        
        res.on('end', () => {
          try {
            const parsedData = JSON.parse(responseData);
            resolve(parsedData);
          } catch (parseError) {
            console.error('❌ Error parsing response:', parseError);
            resolve({ success: false, error: 'Invalid JSON response' });
          }
        });
      });
      
      req.on('error', (error) => {
        console.error('❌ Request error:', error);
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notification result:`, result);
    
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save equipment notifications to MongoDB
const saveNotificationsToMongoDB = async (equipmentData, createdByName, adminUsers) => {
  try {
    console.log('💾 Saving equipment notifications to MongoDB for each admin/supervisor...');
    
    // Create notification objects for each admin/supervisor
    const notifications = adminUsers.map(user => ({
      notificationId: `equipment_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${user._id}`,
      userId: user._id,
      title: '🚜 New Equipment Added',
      body: `${equipmentData.equipmentName || equipmentData.mainCategory} has been added\n\nAdded by: ${createdByName}`,
      type: 'equipment_added',
      timestamp: Date.now(),
      isRead: false,
      equipmentName: equipmentData.equipmentName || 'New Equipment',
      mainCategory: equipmentData.mainCategory,
      country: equipmentData.country,
      data: {
        type: 'equipment_added',
        action: 'view_equipment',
        equipmentId: equipmentData.id,
        equipmentName: equipmentData.equipmentName || 'New Equipment',
        mainCategory: equipmentData.mainCategory,
        country: equipmentData.country,
        createdBy: createdByName,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save notifications to MongoDB via HTTP request
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send email notifications to admins and supervisors
const sendEquipmentEmails = async (equipmentData, createdByName) => {
  try {
    console.log('📧 Sending email notifications to admins and supervisors...');
    
    // Get all users with role ProAdmin or Supervisor
    const users = await GlobalUser.find({
      role: { $in: ['ProAdmin', 'Supervisor'] },
      email: { $exists: true, $ne: null, $ne: '' }
    }).select('name email role');
    
    if (users.length === 0) {
      console.log('⚠️ No admins/supervisors with emails found');
      return { success: false, error: 'No recipients found' };
    }
    
    console.log(`📧 Found ${users.length} admin(s)/supervisor(s) with emails`);
    
    // Prepare email content
    const emailSubject = `New Equipment Added: ${equipmentData.equipmentName || equipmentData.mainCategory}`;
    const emailHtml = `
      <!DOCTYPE html>
      <html>
      <head>
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background: linear-gradient(135deg, #3481BC, #025383); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
          .header h1 { margin: 0; font-size: 24px; }
          .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
          .info-box { background: white; padding: 20px; border-radius: 8px; margin: 15px 0; border-left: 4px solid #3481BC; }
          .info-row { display: flex; margin: 10px 0; }
          .info-label { font-weight: bold; color: #025383; width: 150px; }
          .info-value { color: #555; flex: 1; }
          .footer { text-align: center; margin-top: 20px; color: #999; font-size: 12px; }
          .icon { font-size: 48px; margin-bottom: 10px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <div class="icon">🚜</div>
            <h1>New Equipment Added</h1>
          </div>
          <div class="content">
            <p>Hello,</p>
            <p>A new equipment has been added to the Titan Drilling system.</p>
            
            <div class="info-box">
              <h3 style="margin-top: 0; color: #025383;">Equipment Details</h3>
              <div class="info-row">
                <span class="info-label">Equipment ID:</span>
                <span class="info-value">${equipmentData.id}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Equipment Name:</span>
                <span class="info-value">${equipmentData.equipmentName || 'N/A'}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Main Category:</span>
                <span class="info-value">${equipmentData.mainCategory}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Country:</span>
                <span class="info-value">${equipmentData.country}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Model:</span>
                <span class="info-value">${equipmentData.model || 'N/A'}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Status:</span>
                <span class="info-value">${equipmentData.status || 'Available'}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Added By:</span>
                <span class="info-value">${createdByName}</span>
              </div>
              <div class="info-row">
                <span class="info-label">Date & Time:</span>
                <span class="info-value">${new Date().toLocaleString('en-GB', { 
                  day: '2-digit', 
                  month: 'short', 
                  year: 'numeric',
                  hour: '2-digit',
                  minute: '2-digit'
                })}</span>
              </div>
            </div>
            
            <p>Please review the equipment details in the system.</p>
          </div>
          <div class="footer">
            <p>This is an automated notification from Titan Drilling Equipment Management System.</p>
            <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
          </div>
        </div>
      </body>
      </html>
    `;
    
    // Send emails to all admins and supervisors
    let successCount = 0;
    let failCount = 0;
    
    for (const user of users) {
      try {
        await emailTransporter.sendMail({
          from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
          to: user.email,
          subject: emailSubject,
          html: emailHtml,
        });
        console.log(`✅ Email sent to ${user.name} (${user.email})`);
        successCount++;
      } catch (emailError) {
        console.error(`❌ Failed to send email to ${user.email}:`, emailError);
        failCount++;
      }
    }
    
    console.log(`📧 Email summary: ${successCount} sent, ${failCount} failed`);
    return { 
      success: true, 
      sent: successCount, 
      failed: failCount,
      total: users.length 
    };
  } catch (error) {
    console.error('❌ Error sending email notifications:', error);
    return { success: false, error: error.message };
  }
};

// Sample equipment data (you can replace this with your actual data)
const sampleEquipmentData = [
  {
    id: "1",
    equipmentName: "Sample Equipment 1",
    mainCategory: "Heavy Equipment",
    category: "Excavator",
    equipmentType: "CAT 320",
    status: "Active",
    country: "DRC",
    make: "Caterpillar",
    model: "320",
    year: "2020",
    serialNumber: "CAT320001",
    location: "Site A",
    description: "Heavy duty excavator",
    databaseName: "titan_drilling",
    uploadedAt: new Date(),
    uploadedFrom: "Auto-upload on server start",
    mongoUploadVersion: "1.0.0"
  },
  {
    id: "2", 
    equipmentName: "Sample Equipment 2",
    mainCategory: "Light Vehicle",
    category: "Truck",
    equipmentType: "Toyota Hilux",
    status: "Active",
    country: "DRC",
    make: "Toyota",
    model: "Hilux",
    year: "2021",
    serialNumber: "HILUX001",
    location: "Site B",
    description: "Light utility vehicle",
    databaseName: "titan_drilling",
    uploadedAt: new Date(),
    uploadedFrom: "Auto-upload on server start",
    mongoUploadVersion: "1.0.0"
  }
];

// Function to auto-populate equipment data
const autoPopulateEquipment = async () => {
  try {
    console.log("🔄 Starting auto-population of equipment data...");
    
    // Check if data already exists
    const existingCount = await Equipment.countDocuments();
    if (existingCount > 0) {
      console.log(`✅ Equipment data already exists (${existingCount} items). Skipping auto-population.`);
      return;
    }

    // Insert sample data
    const result = await Equipment.insertMany(sampleEquipmentData);
    console.log(`✅ Auto-populated ${result.length} equipment items successfully!`);
    
  } catch (error) {
    console.error("❌ Error auto-populating equipment data:", error);
  }
};

// Function to get equipment data
const getEquipmentData = async (req, res) => {
  try {
    const equipment = await Equipment.find().sort({ uploadedAt: -1 });
    res.json({
      success: true,
      message: "Equipment data retrieved successfully",
      data: equipment,
      count: equipment.length
    });
  } catch (error) {
    console.error("❌ Error fetching equipment data:", error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
};

// Function to add equipment data
const addEquipmentData = async (req, res) => {
  try {
    const { equipmentData } = req.body;
    
    if (!equipmentData || !Array.isArray(equipmentData)) {
      return res.status(400).json({
        success: false,
        error: "equipmentData must be an array"
      });
    }

    const equipmentWithMetadata = equipmentData.map(item => ({
      ...item,
      databaseName: "titan_drilling",
      uploadedAt: new Date(),
      uploadedFrom: "React Native App",
      mongoUploadVersion: "1.0.0"
    }));

    const result = await Equipment.insertMany(equipmentWithMetadata, { ordered: false });
    
    res.json({
      success: true,
      message: `Successfully added ${result.length} equipment items`,
      count: result.length,
      data: result
    });
  } catch (error) {
    console.error("❌ Error adding equipment data:", error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
};

// Function to clear all equipment data
const clearEquipmentData = async (req, res) => {
  try {
    const result = await Equipment.deleteMany({});
    res.json({
      success: true,
      message: `Cleared ${result.deletedCount} equipment items`,
      deletedCount: result.deletedCount
    });
  } catch (error) {
    console.error("❌ Error clearing equipment data:", error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
};

// Get equipment count
const getEquipmentCount = async (req, res) => {
  try {
    const count = await Equipment.countDocuments();
    res.json({ count });
  } catch (err) {
    console.error("❌ Error fetching count:", err);
    res.status(500).json({ error: err.message });
  }
};

// Get all equipment (old path)
const getAllEquipment = async (req, res) => {
  try {
    const equipment = await Equipment.find().sort({ uploadedAt: -1 });
    res.json(equipment);
  } catch (err) {
    console.error("❌ Error fetching equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Get equipment by database name
const getEquipmentByDatabase = async (req, res) => {
  try {
    const { databaseName } = req.params;
    const equipment = await Equipment.find({ databaseName }).sort({ uploadedAt: -1 });
    res.json(equipment);
  } catch (err) {
    console.error("❌ Error fetching equipment by database:", err);
    res.status(500).json({ error: err.message });
  }
};

// Get equipment by ID
const getEquipmentById = async (req, res) => {
  try {
    const equipment = await Equipment.findOne({ id: req.params.id });
    if (!equipment) {
      return res.status(404).json({ error: "Equipment not found" });
    }
    res.json(equipment);
  } catch (err) {
    console.error("❌ Error fetching equipment by ID:", err);
    res.status(500).json({ error: err.message });
  }
};

// Add single equipment
const addSingleEquipment = async (req, res) => {
  try {
    const equipmentData = {
      ...req.body,
      uploadedAt: new Date(),
      uploadedFrom: req.body.uploadedFrom || "React Native App",
      mongoUploadVersion: "1.0.0",
      databaseName: req.body.databaseName || "titan_drilling",
    };
    
    const newEquipment = new Equipment(equipmentData);
    await newEquipment.save();
    
    console.log(`✅ Equipment added: ${equipmentData.equipmentName || equipmentData.id}`);
    
    // Emit real-time update via Socket.IO
    io.emit('equipment:added', newEquipment);
    
    // Get creator name from request or default to 'Admin'
    let createdByName = req.body.createdByName || 'Admin';
    
    // If we have a user ID, try to get the user's name from MongoDB
    if (req.body.createdBy) {
      try {
        const creator = await GlobalUser.findById(req.body.createdBy).select('name');
        if (creator && creator.name) {
          createdByName = creator.name;
        }
      } catch (userError) {
        console.warn('⚠️ Could not fetch creator name:', userError);
      }
    }
    
    // Send notifications in background (don't block response)
    setTimeout(async () => {
      try {
        console.log('🔔 Sending notifications for new equipment...');
        
        // Get all admins and supervisors for notifications
        const adminUsers = await GlobalUser.find({
          role: { $in: ['ProAdmin', 'Supervisor'] }
        }).select('_id name email role fcmToken fcmTokens');
        
        if (adminUsers.length === 0) {
          console.log('⚠️ No admins or supervisors found for notifications');
          return;
        }
        
        console.log(`📋 Found ${adminUsers.length} admin(s)/supervisor(s) for notifications`);
        
        // Save notifications to MongoDB (for notification screen)
        const mongoResult = await saveNotificationsToMongoDB(equipmentData, createdByName, adminUsers);
        if (mongoResult.success) {
          console.log(`✅ Notifications saved to MongoDB for ${mongoResult.count || adminUsers.length} user(s)`);
        } else {
          console.error('❌ MongoDB notification save failed:', mongoResult.error);
        }
        
        // Send push notifications (Firebase FCM only for push, not storage)
        const pushResult = await sendEquipmentNotifications(equipmentData, createdByName);
        if (pushResult.success) {
          console.log(`✅ Push notifications sent to ${pushResult.successCount || 0} device(s)`);
        } else {
          console.error('❌ Push notification failed:', pushResult.error);
        }
        
        // Send email notifications
        const emailResult = await sendEquipmentEmails(equipmentData, createdByName);
        if (emailResult.success) {
          console.log(`✅ Emails sent to ${emailResult.sent} recipient(s)`);
        } else {
          console.error('❌ Email notification failed:', emailResult.error);
        }
      } catch (notifError) {
        console.error('❌ Error sending notifications:', notifError);
      }
    }, 500);
    
    res.json({ 
      message: "✅ Equipment added successfully!", 
      data: newEquipment 
    });
  } catch (err) {
    console.error("❌ Error adding equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Update equipment
const updateEquipment = async (req, res) => {
  try {
    const equipment = await Equipment.findOneAndUpdate(
      { id: req.params.id },
      { ...req.body, updatedAt: new Date() },
      { new: true }
    );
    if (!equipment) {
      return res.status(404).json({ error: "Equipment not found" });
    }
    
    // Emit real-time update via Socket.IO
    io.emit('equipment:updated', equipment);
    
    res.json({ message: "✅ Equipment updated!", data: equipment });
  } catch (err) {
    console.error("❌ Error updating equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Delete equipment by ID
const deleteEquipmentById = async (req, res) => {
  try {
    const equipment = await Equipment.findOneAndDelete({ id: req.params.id });
    if (!equipment) {
      return res.status(404).json({ error: "Equipment not found" });
    }
    
    // Emit real-time delete via Socket.IO
    io.emit('equipment:deleted', { id: req.params.id });
    
    res.json({ message: "✅ Equipment deleted!" });
  } catch (err) {
    console.error("❌ Error deleting equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Clear all equipment (old path)
const clearAllEquipment = async (req, res) => {
  try {
    const result = await Equipment.deleteMany({});
    res.json({ 
      message: `✅ Cleared ${result.deletedCount} equipment items`,
      deletedCount: result.deletedCount 
    });
  } catch (err) {
    console.error("❌ Error clearing equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Bulk upload equipment data (old path)
const bulkUploadEquipment = async (req, res) => {
  try {
    console.log("📥 Received bulk upload request with body size:", Buffer.byteLength(JSON.stringify(req.body)));
    const { equipmentData, databaseName } = req.body;
    
    if (!equipmentData || !Array.isArray(equipmentData)) {
      return res.status(400).json({ error: "equipmentData must be an array" });
    }

    const equipmentWithMetadata = equipmentData.map(item => ({
      ...item,
      databaseName: databaseName || "titan_drilling",
      uploadedAt: new Date(),
      uploadedFrom: "React Native App",
      mongoUploadVersion: "1.0.0",
    }));

    console.log(`📦 Processing ${equipmentWithMetadata.length} items for upload`);
    const result = await Equipment.insertMany(equipmentWithMetadata, { ordered: false });
    
    console.log(`✅ Bulk uploaded ${result.length} equipment items`);
    res.json({ 
      message: `✅ Successfully uploaded ${result.length} equipment items`,
      count: result.length,
      data: result 
    });
  } catch (err) {
    console.error("❌ Error bulk uploading equipment:", err);
    
    if (err.name === 'PayloadTooLargeError') {
      res.status(413).json({ error: "Request entity too large. Please reduce the data size or increase the server limit." });
    } else if (err.writeErrors) {
      const successCount = err.insertedDocs?.length || 0;
      console.log(`⚠️ Partial success: ${successCount} items uploaded, ${err.writeErrors.length} duplicates skipped`);
      res.json({
        message: `⚠️ Uploaded ${successCount} items, ${err.writeErrors.length} duplicates skipped`,
        successCount,
        errorCount: err.writeErrors.length,
      });
    } else {
      res.status(500).json({ error: err.message });
    }
  }
};

// Get list of databases
const getDatabases = async (req, res) => {
  try {
    if (mongoose.connection.readyState !== 1) {
      throw new Error("MongoDB connection not ready");
    }
    const admin = mongoose.connection.db.admin();
    try {
      const { databases } = await admin.listDatabases();
      const dbNames = databases.map(db => db.name);
      console.log("✅ Databases retrieved:", dbNames);
      res.json({ databases: dbNames });
    } catch (dbErr) {
      console.error("❌ Error in listDatabases:", dbErr);
      throw new Error("Insufficient permissions to list databases");
    }
  } catch (err) {
    console.error("❌ Error fetching databases:", err);
    res.status(500).json({ error: `Failed to fetch databases: ${err.message}` });
  }
};

// Delete equipment from specific database
const clearDatabaseEquipment = async (req, res) => {
  try {
    const { databaseName } = req.params;
    const result = await Equipment.deleteMany({ databaseName });
    res.json({ 
      message: `✅ Cleared ${result.deletedCount} equipment items from ${databaseName}`,
      deletedCount: result.deletedCount 
    });
  } catch (err) {
    console.error("❌ Error clearing database:", err);
    res.status(500).json({ error: err.message });
  }
};

// ===== ROUTES SETUP =====

// Health check
app.get("/", (req, res) => {
  res.json({ message: "✅ Server is running and MongoDB connected!" });
});

// NEW PATHS (GlobalEquipmentJsonData)
app.get("/EquipmentsDataJSON/GlobalEquipmentJsonData", getEquipmentData);
app.post("/EquipmentsDataJSON/GlobalEquipmentJsonData", addEquipmentData);
app.delete("/EquipmentsDataJSON/GlobalEquipmentJsonData", clearEquipmentData);

// OLD PATHS (for compatibility)
app.get("/data", getAllEquipment);
app.get("/data/:databaseName", getEquipmentByDatabase);
app.get("/count", getEquipmentCount);
app.get("/equipment/:id", getEquipmentById);
app.post("/add", addSingleEquipment);
app.put("/equipment/:id", updateEquipment);
app.delete("/equipment/:id", deleteEquipmentById);
app.delete("/clear-all", clearAllEquipment);
app.post("/bulk-upload", bulkUploadEquipment);
app.get("/databases", getDatabases);
app.delete("/clear-database/:databaseName", clearDatabaseEquipment);

// Handle undefined routes
app.use((req, res) => {
  res.status(404).json({ error: `Route ${req.method} ${req.originalUrl} not found` });
});

// ===== SERVER STARTUP =====
server.listen(PORT, "0.0.0.0", () => {
  const serverUrl = `http://${SERVER_IP}:${PORT}`;
  
  console.log(`🚀 Server running on port ${PORT}`);
  console.log(`📡 Server URL: ${serverUrl}`);
  console.log(`🔗 Health check: ${serverUrl}/`);
  console.log(`⚡ Socket.IO: Enabled for real-time updates`);
  console.log(`\n🆕 NEW PATHS:`);
  console.log(`📊 Get equipment: GET ${serverUrl}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
  console.log(`📤 Add equipment: POST ${serverUrl}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
  console.log(`🗑️ Clear equipment: DELETE ${serverUrl}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
  console.log(`\n📋 OLD PATHS (compatibility):`);
  console.log(`📊 Get equipment: GET ${serverUrl}/data`);
  console.log(`📊 Count: GET ${serverUrl}/count`);
  console.log(`📤 Bulk upload: POST ${serverUrl}/bulk-upload`);
  console.log(`🗄️ List databases: GET ${serverUrl}/databases`);
  console.log(`\n🔒 Environment: ${process.env.NODE_ENV || 'development'}`);
});

// Auto-populate equipment data when server starts
setTimeout(async () => {
  await autoPopulateEquipment();
}, 2000); // Wait 2 seconds for MongoDB connection

module.exports = {
  app,
  io,
  server,
  autoPopulateEquipment,
  getEquipmentData,
  addEquipmentData,
  clearEquipmentData,
  getEquipmentCount,
  getAllEquipment,
  getEquipmentByDatabase,
  getEquipmentById,
  addSingleEquipment,
  updateEquipment,
  deleteEquipmentById,
  clearAllEquipment,
  bulkUploadEquipment,
  getDatabases,
  clearDatabaseEquipment,
  Equipment
};
